﻿Imports System.IO
Imports System.Media
Imports System.Text.Json

Public Class Form1
    Inherits Form

    ' --- Structures ---
    Private Structure TreasureInfo
        Public Name As String
        Public Rarity As String
        Public Points As Integer
        Public Image As Image
    End Structure

    ' --- Save Data ---
    Private Class SaveData
        Public Property Level As Integer
        Public Property Score As Integer
    End Class

    ' UI controls
    Private lblTitle As Label
    Private pnlSandbox As Panel
    Private btnNewDig As Button
    Private lblTimer As Label
    Private lblFound As Label
    Private lblScore As Label
    Private cmbDifficulty As ComboBox
    Private lblInstructions As Label
    Private btnSaveLog As Button
    Private btnSoundToggle As Button
    Private btnHint As Button ' New hint button
    Private btnManualSave As Button ' New hint button

    ' Grid
    Private gridCols As Integer = 10
    Private gridRows As Integer = 10
    Private cellSize As Integer = 48
    Private treasureCount As Integer = 8

    ' Game state
    Private treasures(,) As TreasureInfo
    Private revealed(,) As Boolean
    Private foundCount As Integer = 0
    Private totalScore As Integer = 0
    Private startTime As DateTime
    Private running As Boolean = False
    Private digLog As New List(Of String)
    Private soundEnabled As Boolean = True

    ' Energy system
    Private brushStrokes As Integer = 25
    Private lblEnergy As Label
    Private Const ENERGY_BONUS_MULTIPLIER As Integer = 5

    ' Timer
    Private gameTimer As Timer

    ' Sounds
    Private sndBrush As SoundPlayer
    Private sndFound As SoundPlayer
    Private sndComplete As SoundPlayer

    ' All treasure definitions
    Private allTreasures As List(Of TreasureInfo)

    ' Hint usage
    Private hintsUsed As Integer = 0
    Private Const MAX_HINTS As Integer = 3

    ' --- Progressive Levels ---
    Private currentLevel As Integer = 1
    Private Const MAX_LEVEL As Integer = 5
    Private Const GRID_INCREASE As Integer = 2
    Private Const TREASURE_INCREASE_FACTOR As Double = 1.2

    ' --- Hint visual aid ---
    Private highlightCell As (x As Integer, y As Integer)?

    Private saveFile As String = Path.Combine(
    Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
    "DigAdventureSave.json")

    Public Sub New()
        InitializeComponent()
        LoadProgress()
        Me.Font = New Font("Segoe UI", 16, FontStyle.Bold)
        SetupUI()
        LoadResources()
        InitializeGame()
    End Sub

    Private Sub LoadResources()
        ' --- Define treasure types ---
        allTreasures = New List(Of TreasureInfo) From {
            New TreasureInfo With {.Name = "Coin", .Rarity = "Common", .Points = 10, .Image = My.Resources.coin},
            New TreasureInfo With {.Name = "Bead", .Rarity = "Common", .Points = 15, .Image = My.Resources.bead},
            New TreasureInfo With {.Name = "Amulet", .Rarity = "Rare", .Points = 40, .Image = My.Resources.trinket},
            New TreasureInfo With {.Name = "Crown", .Rarity = "Legendary", .Points = 100, .Image = My.Resources.trinket},
            New TreasureInfo With {.Name = "Golden Idol", .Rarity = "Legendary", .Points = 120, .Image = My.Resources.coin}
        }

        ' Sounds
        sndBrush = New SoundPlayer(My.Resources.Brush)
        sndFound = New SoundPlayer(My.Resources.found)
        sndComplete = New SoundPlayer(My.Resources.COMPLETE)
    End Sub

    Private Sub SetupUI()
        ' --- Base Form ---
        Me.Text = $"🏺 Archaeological Dig Adventure - Level {currentLevel}"
        Me.ClientSize = New Size(1180, 860)
        Me.FormBorderStyle = FormBorderStyle.FixedDialog
        Me.MaximizeBox = False
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.BackColor = Color.AntiqueWhite
        Me.Font = New Font("Segoe UI", 14, FontStyle.Regular)

        ' === TITLE BAR ===
        lblTitle = New Label() With {
        .Text = "🏺 Archaeological Dig Adventure!",
        .Font = New Font("Segoe UI", 26, FontStyle.Bold),
        .ForeColor = Color.DarkGoldenrod,
        .BackColor = Color.Bisque,
        .Dock = DockStyle.Top,
        .Height = 80,
        .TextAlign = ContentAlignment.MiddleCenter
    }
        Me.Controls.Add(lblTitle)

        ' === MAIN LAYOUT PANEL ===
        Dim mainPanel As New TableLayoutPanel With {
        .Dock = DockStyle.Fill,
        .ColumnCount = 2,
        .RowCount = 1
    }
        mainPanel.ColumnStyles.Add(New ColumnStyle(SizeType.AutoSize))
        mainPanel.ColumnStyles.Add(New ColumnStyle(SizeType.Percent, 100))
        Me.Controls.Add(mainPanel)

        ' === SANDBOX PANEL ===
        pnlSandbox = New Panel() With {
        .BackColor = Color.SandyBrown,
        .BorderStyle = BorderStyle.FixedSingle,
        .Margin = New Padding(40, 100, 0, 0),
        .Width = gridCols * cellSize + 4,
        .Height = gridRows * cellSize + 4
    }
        AddHandler pnlSandbox.Paint, AddressOf pnlSandbox_Paint
        AddHandler pnlSandbox.MouseClick, AddressOf pnlSandbox_MouseClick
        mainPanel.Controls.Add(pnlSandbox)

        ' === SIDEBAR PANEL ===
        Dim sidePanel As New FlowLayoutPanel With {
        .FlowDirection = FlowDirection.TopDown,
        .Padding = New Padding(80, 85, 0, 0),
        .AutoScroll = True,
        .WrapContents = False,
        .Width = 400,
        .Height = 700
    }
        mainPanel.Controls.Add(sidePanel)

        ' === BUTTONS & LABELS ===
        btnNewDig = MakeBtn("🪓 New Dig")
        AddHandler btnNewDig.Click, AddressOf btnNewDig_Click
        sidePanel.Controls.Add(btnNewDig)

    cmbDifficulty = New ComboBox() With {
        .Width = 250,
        .DropDownStyle = ComboBoxStyle.DropDownList
    }
    cmbDifficulty.Items.AddRange({"Easy (few treasures)", "Medium", "Hard (many treasures)"})
    cmbDifficulty.SelectedIndex = 1
        ''cmbDifficulty.Font = New Font("Segoe UI", 8, FontStyle.Regular)
        AddHandler cmbDifficulty.SelectedIndexChanged, AddressOf cmbDifficulty_SelectedIndexChanged
        sidePanel.Controls.Add(cmbDifficulty)

        lblTimer = New Label() With {.AutoSize = True, .Text = "⏱ Time: 00:00", .Font = New Font("Segoe UI", 18, FontStyle.Bold)}
        lblTimer.Padding = New Padding(0, 30, 0, 0)
        sidePanel.Controls.Add(lblTimer)

        lblFound = New Label() With {.AutoSize = True, .Text = "💎 Found: 0 / 0", .Font = New Font("Segoe UI", 18, FontStyle.Bold)}
    sidePanel.Controls.Add(lblFound)

    lblScore = New Label() With {.AutoSize = True, .Text = "🏆 Score: 0", .Font = New Font("Segoe UI", 18, FontStyle.Bold)}
    sidePanel.Controls.Add(lblScore)

    lblEnergy = New Label() With {.AutoSize = True, .Text = "🖌️ Strokes: 25", .Font = New Font("Segoe UI", 18, FontStyle.Bold), .ForeColor = Color.DarkSlateBlue}
    sidePanel.Controls.Add(lblEnergy)

        lblInstructions = New Label() With {
        .AutoSize = True,
        .MaximumSize = New Size(260, 300),
        .Text = "Click sand squares to uncover treasures. Rarity = more points!" + vbCrLf + vbCrLf,
        .Font = New Font("Segoe UI", 14)
    }
        lblInstructions.Padding = New Padding(0, 20, 0, 0)
        sidePanel.Controls.Add(lblInstructions)

        btnSoundToggle = MakeBtn("🔈 Sound: ON")
        AddHandler btnSoundToggle.Click, AddressOf btnSoundToggle_Click
        sidePanel.Controls.Add(btnSoundToggle)

        btnSaveLog = MakeBtn("💾 Save Log")
        AddHandler btnSaveLog.Click, AddressOf btnSaveLog_Click
        sidePanel.Controls.Add(btnSaveLog)

    btnHint = MakeBtn("💡 Give a Hint")
        AddHandler btnHint.Click, AddressOf btnHint_Click
        sidePanel.Controls.Add(btnHint)

    btnManualSave = MakeBtn("📘 Save Progress")
        AddHandler btnManualSave.Click, AddressOf btnManualSave_Click
        sidePanel.Controls.Add(btnManualSave)

    ' === GAME TIMER ===
    gameTimer = New Timer() With {.Interval = 500}
        AddHandler gameTimer.Tick, AddressOf gameTimer_Tick
    End Sub

    ' Helper for button styling
    Function MakeBtn(text As String) As Button
        Return New Button With {
            .Text = text,
            .Width = 250,
            .Height = 55,
            .BackColor = Color.Bisque,
            .FlatStyle = FlatStyle.Flat
        }
    End Function

    Private Sub SetupUI2()
        ' === Base form ===
        Me.Text = $"🏺 Archaeological Dig Adventure - Level {currentLevel}"
        Me.ClientSize = New Size(1000, 730)
        Me.FormBorderStyle = FormBorderStyle.FixedDialog
        Me.MaximizeBox = False
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.BackColor = Color.AntiqueWhite

        ' === Title ===
        lblTitle = New Label() With {
            .Text = "🏺 Archaeological Dig Adventure!",
            .Font = New Font("Segoe UI", 24, FontStyle.Bold),
            .ForeColor = Color.DarkGoldenrod,
            .BackColor = Color.Bisque,
            .TextAlign = ContentAlignment.MiddleCenter,
            .AutoSize = False,
            .Size = New Size(Me.ClientSize.Width - 40, 70),
            .Location = New Point(20, 10)
        }
        Me.Controls.Add(lblTitle)

        ' Sandbox
        pnlSandbox = New Panel() With {
            .Location = New Point(20, lblTitle.Bottom + 20),
            .Size = New Size(gridCols * cellSize + 2, gridRows * cellSize + 2),
            .BackColor = Color.SandyBrown,
            .BorderStyle = BorderStyle.FixedSingle
        }
        AddHandler pnlSandbox.Paint, AddressOf pnlSandbox_Paint
        AddHandler pnlSandbox.MouseClick, AddressOf pnlSandbox_MouseClick
        Me.Controls.Add(pnlSandbox)

        ' === Sidebar ===
        Dim rightX As Integer = pnlSandbox.Right + 30
        Dim nextY As Integer = pnlSandbox.Top
        Dim buttonWidth As Integer = 220
        Dim buttonHeight As Integer = 60
        Dim spacing As Integer = 14

        btnNewDig = New Button() With {.Text = "🪓 New Dig", .Location = New Point(rightX, nextY), .Size = New Size(buttonWidth, buttonHeight)}
        AddHandler btnNewDig.Click, AddressOf btnNewDig_Click
        Me.Controls.Add(btnNewDig)
        nextY += buttonHeight + spacing

        cmbDifficulty = New ComboBox() With {.DropDownStyle = ComboBoxStyle.DropDownList, .Location = New Point(rightX, nextY), .Size = New Size(buttonWidth, 40)}
        cmbDifficulty.Items.AddRange(New String() {"Easy (few treasures)", "Medium", "Hard (many treasures)"})
        cmbDifficulty.SelectedIndex = 1
        AddHandler cmbDifficulty.SelectedIndexChanged, AddressOf cmbDifficulty_SelectedIndexChanged
        Me.Controls.Add(cmbDifficulty)
        nextY += cmbDifficulty.Height + spacing + 10

        lblTimer = New Label() With {.Text = "⏱ Time: 00:00", .Location = New Point(rightX, nextY), .AutoSize = True, .Font = New Font(Me.Font.FontFamily, 18, FontStyle.Bold)}
        Me.Controls.Add(lblTimer)
        nextY += lblTimer.Height + spacing

        lblFound = New Label() With {.Text = "💎 Found: 0 / 0", .Location = New Point(rightX, nextY), .AutoSize = True, .Font = New Font(Me.Font.FontFamily, 18, FontStyle.Bold)}
        Me.Controls.Add(lblFound)
        nextY += lblFound.Height + spacing

        lblScore = New Label() With {.Text = "🏆 Score: 0", .Location = New Point(rightX, nextY), .AutoSize = True, .Font = New Font(Me.Font.FontFamily, 18, FontStyle.Bold)}
        Me.Controls.Add(lblScore)
        nextY += lblScore.Height + spacing

        lblEnergy = New Label() With {
            .Text = "🖌️ Strokes: 25",
            .Location = New Point(rightX, nextY),
            .AutoSize = True,
            .Font = New Font(Me.Font.FontFamily, 18, FontStyle.Bold),
            .ForeColor = Color.DarkSlateBlue
        }
        Me.Controls.Add(lblEnergy)
        nextY += lblEnergy.Height + spacing

        lblInstructions = New Label() With {.Text = "Click on the sand squares to uncover hidden treasures!" & vbCrLf & "Each rarity gives more points!",
                                            .Location = New Point(rightX, nextY), .Size = New Size(260, 100), .Font = New Font(Me.Font.FontFamily, 14, FontStyle.Regular)}
        Me.Controls.Add(lblInstructions)
        nextY += lblInstructions.Height + spacing

        btnSoundToggle = New Button() With {.Text = "🔈 Sound: ON", .Location = New Point(rightX, nextY - 10), .Size = New Size(buttonWidth, buttonHeight)}
        AddHandler btnSoundToggle.Click, AddressOf btnSoundToggle_Click
        Me.Controls.Add(btnSoundToggle)
        nextY += buttonHeight + spacing

        btnSaveLog = New Button() With {.Text = "💾 Save Log", .Location = New Point(rightX, nextY - 10), .Size = New Size(buttonWidth, buttonHeight)}
        AddHandler btnSaveLog.Click, AddressOf btnSaveLog_Click
        Me.Controls.Add(btnSaveLog)
        nextY += buttonHeight + spacing

        ' --- Hint button ---
        btnHint = New Button() With {
            .Text = "💡 Give a Hint",
            .Location = New Point(rightX, nextY - 10),
            .Size = New Size(buttonWidth, buttonHeight)
        }
        AddHandler btnHint.Click, AddressOf btnHint_Click
        Me.Controls.Add(btnHint)

        btnManualSave = New Button() With {
    .Text = "💾 Save Progress",
    .Location = New Point(rightX + 300, nextY),
    .Size = New Size(buttonWidth, buttonHeight)
}
        AddHandler btnManualSave.Click, AddressOf btnManualSave_Click
        Me.Controls.Add(btnManualSave)

        ' Timer
        gameTimer = New Timer() With {.Interval = 500}
        AddHandler gameTimer.Tick, AddressOf gameTimer_Tick

    End Sub

    Private Sub InitializeGame(Optional keepScore As Boolean = False)
        ' Adjust grid and treasure count based on current level
        gridCols = 10 + (currentLevel - 1) * GRID_INCREASE
        gridRows = 10 + (currentLevel - 1) * GRID_INCREASE

        ' Resize panel dynamically
        pnlSandbox.Size = New Size(gridCols * cellSize + 2, gridRows * cellSize + 2)

        ' Adaptive treasure density by difficulty
        Dim totalCells As Integer = gridCols * gridRows
        Select Case cmbDifficulty.SelectedIndex
            Case 0 : treasureCount = Math.Max(1, CInt(totalCells * 0.15))
            Case 1 : treasureCount = Math.Max(1, CInt(totalCells * 0.2))
            Case 2 : treasureCount = Math.Max(1, CInt(totalCells * 0.25))
        End Select

        ' Scale up treasure count per level
        treasureCount = CInt(treasureCount * Math.Pow(TREASURE_INCREASE_FACTOR, currentLevel - 1))

        ReDim treasures(gridCols - 1, gridRows - 1)
        ReDim revealed(gridCols - 1, gridRows - 1)

        Dim rnd As New Random()
        Dim placed As Integer = 0
        While placed < treasureCount
            Dim cx = rnd.Next(0, gridCols)
            Dim cy = rnd.Next(0, gridRows)
            If treasures(cx, cy).Name Is Nothing Then
                treasures(cx, cy) = GetRandomTreasure(rnd)
                placed += 1
            End If
        End While

        foundCount = 0
        If Not keepScore Then totalScore = 0
        brushStrokes = 25
        lblFound.Text = $"💎 Found: {foundCount} / {treasureCount}"
        lblScore.Text = $"🏆 Score: {totalScore}"
        lblEnergy.Text = $"🖌️ Strokes: {brushStrokes}"
        digLog.Clear()
        lblTimer.Text = "⏱ Time: 00:00"
        running = False
        hintsUsed = 0
        btnHint.Enabled = True
        pnlSandbox.Invalidate()
        Me.Text = $"🏺 Archaeological Dig Adventure - Level {currentLevel}"
    End Sub

    ' === GAME INIT ===
    Private Sub InitializeGame2()
        ' Adaptive treasure density by difficulty
        Dim totalCells As Integer = gridCols * gridRows
        Select Case cmbDifficulty.SelectedIndex
            Case 0 : treasureCount = Math.Max(1, CInt(totalCells * 0.15))
            Case 1 : treasureCount = Math.Max(1, CInt(totalCells * 0.2))
            Case 2 : treasureCount = Math.Max(1, CInt(totalCells * 0.25))
        End Select

        ReDim treasures(gridCols - 1, gridRows - 1)
        ReDim revealed(gridCols - 1, gridRows - 1)

        Dim rnd As New Random()
        Dim placed As Integer = 0
        While placed < treasureCount
            Dim cx = rnd.Next(0, gridCols)
            Dim cy = rnd.Next(0, gridRows)
            If treasures(cx, cy).Name Is Nothing Then
                treasures(cx, cy) = GetRandomTreasure(rnd)
                placed += 1
            End If
        End While

        foundCount = 0
        totalScore = 0
        brushStrokes = 25
        lblFound.Text = $"💎 Found: {foundCount} / {treasureCount}"
        lblScore.Text = $"🏆 Score: {totalScore}"
        lblEnergy.Text = $"🖌️ Strokes: {brushStrokes}"
        digLog.Clear()
        lblTimer.Text = "⏱ Time: 00:00"
        running = False
        hintsUsed = 0
        btnHint.Enabled = True
        pnlSandbox.Invalidate()
    End Sub

    Private Function GetRandomTreasure(rnd As Random) As TreasureInfo
        Dim roll As Integer = rnd.Next(100)
        Dim rarity As String
        If roll < 70 Then
            rarity = "Common"
        ElseIf roll < 95 Then
            rarity = "Rare"
        Else
            rarity = "Legendary"
        End If
        Dim pool = allTreasures.Where(Function(t) t.Rarity = rarity).ToList()
        Return pool(rnd.Next(pool.Count))
    End Function

    Private Sub pnlSandbox_Paint(sender As Object, e As PaintEventArgs)
        Dim g As Graphics = e.Graphics
        g.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias

        For x As Integer = 0 To gridCols - 1
            For y As Integer = 0 To gridRows - 1
                Dim rect As New Rectangle(x * cellSize + 1, y * cellSize + 1, cellSize - 2, cellSize - 2)

                If revealed(x, y) Then
                    g.FillRectangle(Brushes.BurlyWood, rect)
                    If treasures(x, y).Name IsNot Nothing Then
                        Dim img As Image = treasures(x, y).Image
                        Dim scale As Single = Math.Min(rect.Width / img.Width, rect.Height / img.Height)
                        g.DrawImage(img, rect.Left + (rect.Width - img.Width * scale) \ 2, rect.Top + (rect.Height - img.Height * scale) \ 2, img.Width * scale, img.Height * scale)
                        Using pen As New Pen(Color.Gold, 3)
                            g.DrawRectangle(pen, rect)
                        End Using
                    Else
                        g.DrawRectangle(Pens.DarkGoldenrod, rect)
                    End If
                Else
                    g.FillRectangle(Brushes.SandyBrown, rect)
                    g.DrawRectangle(Pens.DarkGoldenrod, rect)
                End If
            Next
        Next

        ' Highlight hint reference cell if one exists
        If highlightCell.HasValue Then
            Dim hx = highlightCell.Value.x
            Dim hy = highlightCell.Value.y
            If hx >= 0 AndAlso hx < gridCols AndAlso hy >= 0 AndAlso hy < gridRows Then
                Dim rect As New Rectangle(hx * cellSize + 1, hy * cellSize + 1, cellSize - 2, cellSize - 2)
                Using pen As New Pen(Color.Red, 4)
                    e.Graphics.DrawRectangle(pen, rect)
                End Using
            End If
        End If
    End Sub

    ' === CLICK DIG ===
    Private Sub pnlSandbox_MouseClick(sender As Object, e As MouseEventArgs)
        If e.Button <> MouseButtons.Left Then Return
        Dim gx As Integer = e.X \ cellSize
        Dim gy As Integer = e.Y \ cellSize
        If gx < 0 OrElse gx >= gridCols OrElse gy < 0 OrElse gy >= gridRows Then Return

        If Not running Then
            startTime = DateTime.Now
            running = True
            gameTimer.Start()
        End If

        If revealed(gx, gy) Then
            MessageBox.Show("Already brushed here!", "Try another spot", MessageBoxButtons.OK, MessageBoxIcon.Information)
            Return
        End If

        If brushStrokes <= 0 Then
            MessageBox.Show("You're out of brush strokes! Start a new dig.", "No Energy", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            Return
        End If

        brushStrokes -= 1
        lblEnergy.Text = $"🖌️ Strokes: {brushStrokes}"

        revealed(gx, gy) = True
        pnlSandbox.Invalidate(New Rectangle(gx * cellSize, gy * cellSize, cellSize, cellSize))

        Dim treasure = treasures(gx, gy)
        If treasure.Name IsNot Nothing Then
            foundCount += 1
            totalScore += treasure.Points
            lblFound.Text = $"💎 Found: {foundCount} / {treasureCount}"
            lblScore.Text = $"🏆 Score: {totalScore}"
            digLog.Add($"Found a {treasure.Rarity} {treasure.Name} ({treasure.Points} pts) at ({gx},{gy}) at {DateTime.Now:T}")
            If soundEnabled Then sndFound.Play()
        End If

        If foundCount >= treasureCount Then
            running = False
            gameTimer.Stop()
            If soundEnabled Then sndComplete.Play()
            Dim bonus As Integer = brushStrokes * ENERGY_BONUS_MULTIPLIER
            totalScore += bonus
            lblScore.Text = $"🏆 Score: {totalScore}"
            Dim elapsed = DateTime.Now - startTime
            digLog.Add($"Level {currentLevel} complete! Time: {elapsed.Minutes:00}:{elapsed.Seconds:00} | Total Score: {totalScore} | Bonus: {bonus}")

            ' Check for next level
            If currentLevel < MAX_LEVEL Then
                MessageBox.Show($"🎉 Level {currentLevel} complete! Moving to Level {currentLevel + 1}..." & vbCrLf &
                        $"Time: {elapsed.Minutes:00}:{elapsed.Seconds:00}" & vbCrLf &
                        $"Energy Bonus: +{bonus} pts" & vbCrLf &
                        $"Total Score: {totalScore}", "Level Complete")

                SaveProgress()
                currentLevel += 1
                InitializeGame(keepScore:=True)
            Else
                MessageBox.Show($"🏆 Congratulations! You completed all {MAX_LEVEL} levels!" & vbCrLf &
                        $"Final Score: {totalScore}", "Adventure Complete")
                SaveProgress()
                currentLevel = 1
                InitializeGame()
            End If
        End If

    End Sub

    Private Sub btnNewDig_Click(sender As Object, e As EventArgs)
        InitializeGame()
    End Sub

    Private Sub cmbDifficulty_SelectedIndexChanged(sender As Object, e As EventArgs)
        InitializeGame()
    End Sub

    Private Sub btnSoundToggle_Click(sender As Object, e As EventArgs)
        soundEnabled = Not soundEnabled
        btnSoundToggle.Text = If(soundEnabled, "🔈 Sound: ON", "🔇 Sound: OFF")
    End Sub

    Private Sub gameTimer_Tick(sender As Object, e As EventArgs)
        If running Then
            Dim elapsed = DateTime.Now - startTime
            lblTimer.Text = $"⏱ Time: {elapsed.Minutes:00}:{elapsed.Seconds:00}"
        End If
    End Sub

    Private Sub btnSaveLog_Click(sender As Object, e As EventArgs)
        Using sfd As New SaveFileDialog()
            sfd.Filter = "Text files (*.txt)|*.txt|All files (*.*)|*.*"
            sfd.FileName = $"dig-log-{DateTime.Now:yyyyMMdd-HHmmss}.txt"
            If sfd.ShowDialog() = DialogResult.OK Then
                File.WriteAllLines(sfd.FileName, digLog)
                MessageBox.Show("Log saved!", "Saved", MessageBoxButtons.OK, MessageBoxIcon.Information)
            End If
        End Using
    End Sub

    Private Sub btnHint_Click(sender As Object, e As EventArgs)
        If hintsUsed >= MAX_HINTS Then
            MessageBox.Show("You've used all your hints!", "Hint", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            Return
        End If

        ' Find all hidden treasures
        Dim hiddenTreasures As New List(Of (x As Integer, y As Integer, name As String))
        For x As Integer = 0 To gridCols - 1
            For y As Integer = 0 To gridRows - 1
                If Not revealed(x, y) AndAlso treasures(x, y).Name IsNot Nothing Then
                    hiddenTreasures.Add((x, y, treasures(x, y).Name))
                End If
            Next
        Next

        If hiddenTreasures.Count = 0 Then
            MessageBox.Show("No more hidden treasures!", "Hint", MessageBoxButtons.OK, MessageBoxIcon.Information)
            Return
        End If

        ' Pick a random reference point on the grid
        Dim rnd As New Random()
        Dim refX As Integer = rnd.Next(0, gridCols)
        Dim refY As Integer = rnd.Next(0, gridRows)

        ' Find the closest hidden treasure to that point
        Dim nearest = hiddenTreasures.OrderBy(Function(t) Math.Sqrt((t.x - refX) ^ 2 + (t.y - refY) ^ 2)).First()
        Dim distance As Integer = CInt(Math.Round(Math.Sqrt((nearest.x - refX) ^ 2 + (nearest.y - refY) ^ 2)))

        ' Store the reference cell to highlight
        highlightCell = (refX, refY)
        pnlSandbox.Invalidate()

        ' Start a short timer to remove the highlight after 1 second
        Dim flashTimer As New Timer() With {.Interval = 1000}
        AddHandler flashTimer.Tick,
        Sub()
            highlightCell = Nothing
            pnlSandbox.Invalidate()
            flashTimer.Stop()
            flashTimer.Dispose()
        End Sub
        flashTimer.Start()

        ' Display the hint message
        MessageBox.Show($"💡 From ({refX},{refY}), the nearest hidden treasure is about {distance} blocks away!",
                    "Hint", MessageBoxButtons.OK, MessageBoxIcon.Information)

        hintsUsed += 1
    End Sub

    ' === TEXT HINT SYSTEM ===
    Private Sub btnHint2_Click(sender As Object, e As EventArgs)
        If hintsUsed >= MAX_HINTS Then
            MessageBox.Show("You've used all your hints!", "Hint", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            Return
        End If

        ' Find all hidden treasures
        Dim hiddenTreasures As New List(Of (x As Integer, y As Integer, name As String))
        For x As Integer = 0 To gridCols - 1
            For y As Integer = 0 To gridRows - 1
                If Not revealed(x, y) AndAlso treasures(x, y).Name IsNot Nothing Then
                    hiddenTreasures.Add((x, y, treasures(x, y).Name))
                End If
            Next
        Next

        If hiddenTreasures.Count = 0 Then
            MessageBox.Show("No more hidden treasures!", "Hint", MessageBoxButtons.OK, MessageBoxIcon.Information)
            Return
        End If

        ' Pick a random hidden treasure to hint
        Dim rnd As New Random()
        Dim hintTreasure = hiddenTreasures(rnd.Next(hiddenTreasures.Count))

        ' Give quadrant-based hint
        Dim quadrant As String = ""
        quadrant &= If(hintTreasure.y < gridRows / 2, "north", "south")
        quadrant &= "-"
        quadrant &= If(hintTreasure.x < gridCols / 2, "west", "east")
        quadrant = quadrant.Replace("-", " ")

        MessageBox.Show($"A hidden treasure lies to the {quadrant} quadrant!", "Hint", MessageBoxButtons.OK, MessageBoxIcon.Information)
        hintsUsed += 1
    End Sub

    Private Sub SaveProgress()
        Dim data As New SaveData() With {
        .Level = currentLevel,
        .Score = totalScore
    }

        Dim json As String = JsonSerializer.Serialize(data)
        File.WriteAllText(saveFile, json)
    End Sub

    Private Sub LoadProgress()
        If File.Exists(saveFile) Then
            Try
                Dim json As String = File.ReadAllText(saveFile)
                Dim data As SaveData = JsonSerializer.Deserialize(Of SaveData)(json)

                If data IsNot Nothing Then
                    currentLevel = Math.Max(1, Math.Min(MAX_LEVEL, data.Level))
                    totalScore = Math.Max(0, data.Score)
                End If

            Catch ex As Exception
                MessageBox.Show("Save file corrupted — starting fresh.")
                currentLevel = 1
                totalScore = 0
            End Try
        Else
            currentLevel = 1
            totalScore = 0
        End If
    End Sub

    Private Sub btnManualSave_Click(sender As Object, e As EventArgs)
        SaveProgress()
        MessageBox.Show("Progress saved!", "Saved", MessageBoxButtons.OK, MessageBoxIcon.Information)
    End Sub

End Class

