' F-22 Enhanced (VB.NET) - parallax starfield, new enemy types, multi-phase bosses
' Single-file Windows Forms game. Replace the project's default code with this file.

''Option Strict On
Option Explicit On

Imports System.Drawing
Imports System.Windows.Forms
Imports System.Media

Module F22Game

    <STAThread()>
    Public Sub Main()
        Application.EnableVisualStyles()
        Application.SetCompatibleTextRenderingDefault(False)
        Application.Run(New GameForm())
    End Sub

    Public Class GameForm
        Inherits Form

        Private ReadOnly gameTimer As New Timer()
        Private ReadOnly rnd As New Random()

        ' Player
        Private playerX As Integer
        Private playerY As Integer
        Private playerSpeed As Integer = 6
        Private playerSize As Integer = 40
        Private playerLives As Integer = 3
        Private playerShield As Integer = 0

        ' Game state
        Private bullets As New List(Of Bullet)()
        Private enemyBullets As New List(Of Bullet)()
        Private enemies As New List(Of Enemy)()
        Private powerups As New List(Of PowerUp)()
        Private boss As Boss = Nothing

        Private level As Integer = 1
        Private score As Integer = 0
        Private paused As Boolean = False
        Private gameOver As Boolean = False

        ' Spawning / difficulty
        Private enemySpawnTimer As Integer = 0
        Private enemySpawnInterval As Integer = 60

        ' Player fire control
        Private canShootCooldown As Integer = 0
        Private baseFireDelay As Integer = 12
        Private rapidFireTicksLeft As Integer = 0

        ' Input
        Private isLeft As Boolean = False
        Private isRight As Boolean = False
        Private isUp As Boolean = False
        Private isDown As Boolean = False
        Private isShooting As Boolean = False

        ' Parallax stars
        Private stars As New List(Of Star)()

        ' Sounds (optional)
        Private soundShoot As SoundPlayer = Nothing
        Private soundExplosion As SoundPlayer = Nothing
        Private soundPowerup As SoundPlayer = Nothing
        Private soundBossEnter As SoundPlayer = Nothing
        Private soundLevelUp As SoundPlayer = Nothing

        Public Sub New()
            Me.Text = "F-22 - Enhanced VB.NET"
            Me.ClientSize = New Size(540, 720)
            Me.DoubleBuffered = True
            Me.FormBorderStyle = FormBorderStyle.FixedSingle
            Me.MaximizeBox = False
            Me.StartPosition = FormStartPosition.CenterScreen

            playerX = Me.ClientSize.Width \ 2 - playerSize \ 2
            playerY = Me.ClientSize.Height - playerSize - 30

            gameTimer.Interval = 16 ' ~60 FPS
            AddHandler gameTimer.Tick, AddressOf GameTick
            gameTimer.Start()

            Me.SetStyle(ControlStyles.AllPaintingInWmPaint Or ControlStyles.UserPaint Or ControlStyles.OptimizedDoubleBuffer, True)

            AddHandler Me.KeyDown, AddressOf GameForm_KeyDown
            AddHandler Me.KeyUp, AddressOf GameForm_KeyUp
            AddHandler Me.Paint, AddressOf GameForm_Paint

            LoadSounds()
            InitStars()
            ResetGame()
        End Sub

        Private Sub LoadSounds()
            Try
                Dim base = System.IO.Path.Combine(Application.StartupPath, "Sounds")
                If System.IO.Directory.Exists(base) Then
                    If System.IO.File.Exists(System.IO.Path.Combine(base, "shoot.wav")) Then soundShoot = New SoundPlayer(System.IO.Path.Combine(base, "shoot.wav"))
                    If System.IO.File.Exists(System.IO.Path.Combine(base, "explosion.wav")) Then soundExplosion = New SoundPlayer(System.IO.Path.Combine(base, "explosion.wav"))
                    If System.IO.File.Exists(System.IO.Path.Combine(base, "powerup.wav")) Then soundPowerup = New SoundPlayer(System.IO.Path.Combine(base, "powerup.wav"))
                    If System.IO.File.Exists(System.IO.Path.Combine(base, "boss_enter.wav")) Then soundBossEnter = New SoundPlayer(System.IO.Path.Combine(base, "boss_enter.wav"))
                    If System.IO.File.Exists(System.IO.Path.Combine(base, "levelup.wav")) Then soundLevelUp = New SoundPlayer(System.IO.Path.Combine(base, "levelup.wav"))
                End If
            Catch ex As Exception
                ' Ignore sound load errors; game still runs
            End Try
        End Sub

        Private Sub InitStars()
            stars.Clear()
            ' Three layers: far (slow), mid, near (fast)
            For i As Integer = 0 To 60
                stars.Add(New Star(rnd.Next(0, Me.ClientSize.Width), rnd.Next(0, Me.ClientSize.Height), rnd.Next(1, 2), 0.4F))
            Next
            For i As Integer = 0 To 40
                stars.Add(New Star(rnd.Next(0, Me.ClientSize.Width), rnd.Next(0, Me.ClientSize.Height), rnd.Next(1, 3), 1.0F))
            Next
            For i As Integer = 0 To 25
                stars.Add(New Star(rnd.Next(0, Me.ClientSize.Width), rnd.Next(0, Me.ClientSize.Height), rnd.Next(2, 4), 2.2F))
            Next
        End Sub

        Private Sub ResetGame()
            bullets.Clear()
            enemyBullets.Clear()
            enemies.Clear()
            powerups.Clear()
            boss = Nothing

            level = 1
            score = 0
            playerLives = 3
            playerShield = 0
            paused = False
            gameOver = False

            enemySpawnInterval = 60
            enemySpawnTimer = 0
            canShootCooldown = 0
            rapidFireTicksLeft = 0

            playerX = Me.ClientSize.Width \ 2 - playerSize \ 2
            playerY = Me.ClientSize.Height - playerSize - 30

            InitStars()
        End Sub

        Private Sub StartNextLevel()
            level += 1
            ' Increase difficulty
            enemySpawnInterval = Math.Max(12, 60 - (level - 1) * 5)
            ' Heal a bit
            playerLives = Math.Min(6, playerLives + 1)
            If soundLevelUp IsNot Nothing Then TryCast(soundLevelUp, SoundPlayer).Play()
        End Sub

        Private Sub GameForm_KeyDown(sender As Object, e As KeyEventArgs)
            Select Case e.KeyCode
                Case Keys.Left, Keys.A
                    isLeft = True
                Case Keys.Right, Keys.D
                    isRight = True
                Case Keys.Up, Keys.W
                    isUp = True
                Case Keys.Down, Keys.S
                    isDown = True
                Case Keys.Space
                    isShooting = True
                Case Keys.P
                    paused = Not paused
                Case Keys.R
                    ResetGame()
                Case Keys.Escape
                    Me.Close()
            End Select
        End Sub

        Private Sub GameForm_KeyUp(sender As Object, e As KeyEventArgs)
            Select Case e.KeyCode
                Case Keys.Left, Keys.A
                    isLeft = False
                Case Keys.Right, Keys.D
                    isRight = False
                Case Keys.Up, Keys.W
                    isUp = False
                Case Keys.Down, Keys.S
                    isDown = False
                Case Keys.Space
                    isShooting = False
            End Select
        End Sub

        Private Sub GameTick(sender As Object, e As EventArgs)
            If paused OrElse gameOver Then
                Me.Invalidate()
                Return
            End If

            ' Update stars (parallax scrolling downward)
            For Each s In stars
                s.Y += s.Speed
                If s.Y > Me.ClientSize.Height Then
                    s.Y = -2
                    s.X = rnd.Next(0, Me.ClientSize.Width)
                End If
            Next

            ' Player movement
            Dim moveX = 0
            Dim moveY = 0
            If isLeft Then moveX -= playerSpeed
            If isRight Then moveX += playerSpeed
            If isUp Then moveY -= playerSpeed
            If isDown Then moveY += playerSpeed

            playerX += moveX
            playerY += moveY

            ' Keep in bounds
            playerX = Math.Max(0, Math.Min(Me.ClientSize.Width - playerSize, playerX))
            playerY = Math.Max(0, Math.Min(Me.ClientSize.Height - playerSize, playerY))

            ' Shooting cooldown
            If canShootCooldown > 0 Then canShootCooldown -= 1
            If rapidFireTicksLeft > 0 Then rapidFireTicksLeft -= 1

            If isShooting AndAlso canShootCooldown <= 0 Then
                FirePlayerBullet()
                canShootCooldown = If(rapidFireTicksLeft > 0, Math.Max(2, baseFireDelay \ 3), baseFireDelay)
            End If

            ' Update bullets
            For i As Integer = bullets.Count - 1 To 0 Step -1
                bullets(i).Y += bullets(i).Vy
                bullets(i).X += bullets(i).Vx
                If bullets(i).Y < -40 OrElse bullets(i).X < -60 OrElse bullets(i).X > Me.ClientSize.Width + 60 Then bullets.RemoveAt(i)
            Next

            For i As Integer = enemyBullets.Count - 1 To 0 Step -1
                enemyBullets(i).Y += enemyBullets(i).Vy
                enemyBullets(i).X += enemyBullets(i).Vx
                If enemyBullets(i).Y > Me.ClientSize.Height + 40 OrElse enemyBullets(i).X < -80 OrElse enemyBullets(i).X > Me.ClientSize.Width + 80 Then enemyBullets.RemoveAt(i)
            Next

            ' Spawn enemies unless boss active
            If boss Is Nothing Then
                If enemySpawnTimer <= 0 Then
                    SpawnEnemyForLevel()
                    enemySpawnTimer = Math.Max(6, enemySpawnInterval - rnd.Next(0, Math.Max(1, level)))
                Else
                    enemySpawnTimer -= 1
                End If
            End If

            ' Update enemies
            For i As Integer = enemies.Count - 1 To 0 Step -1
                Dim en = enemies(i)
                en.Update(Me.ClientSize, playerX, playerY, rnd)

                ' enemy shooting
                If en.CanShoot AndAlso rnd.NextDouble() < en.ShootProbability Then
                    Dim eb = New Bullet(en.X + en.Size \ 2 - 4, en.Y + en.Size, 6)
                    ' aim towards player for snipers, slight for others
                    Dim dx = (playerX + playerSize \ 2) - eb.X
                    Dim dy = (playerY + playerSize \ 2) - eb.Y
                    Dim dist = Math.Sqrt(dx * dx + dy * dy)
                    If dist <> 0 Then
                        eb.Vx = CInt(Math.Round(dx / dist * en.ShotSpeed))
                    Else
                        eb.Vx = 0
                    End If
                    eb.Vy = CInt(Math.Round(dy / dist * en.ShotSpeed))
                    enemyBullets.Add(eb)
                End If

                ' Remove if off-screen
                If en.Y > Me.ClientSize.Height + 120 Then
                    enemies.RemoveAt(i)
                ElseIf en.X + en.Size < -120 OrElse en.X > Me.ClientSize.Width + 120 Then
                    enemies.RemoveAt(i)
                End If
            Next

            ' Update boss
            If boss IsNot Nothing Then
                boss.Update(Me.ClientSize, playerX, playerY, rnd)

                ' Check phase changes
                Dim prevPhase = boss.Phase
                boss.CheckPhase()
                If boss.Phase <> prevPhase Then
                    ' spawn some minions on phase change
                    For k As Integer = 0 To 3 + boss.Phase
                        enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), boss.Y + boss.Size + rnd.Next(10, 60), 28, 2 + rnd.Next(0, 2), EnemyType.Kamikaze, rnd))
                    Next
                End If

                ' Boss fires bullets in patterns depending on phase
                If boss.CanFire Then
                    Select Case boss.Phase
                        Case 1
                            ' spread frontal shots
                            For bI As Integer = 0 To boss.FireCount - 1
                                Dim bx = boss.X + boss.Size \ 2 - 6 + (bI - boss.FireCount \ 2) * 14
                                Dim by = boss.Y + boss.Size
                                Dim eb = New Bullet(bx, by, 6)
                                eb.Vx = (bI - boss.FireCount \ 2) * 2
                                enemyBullets.Add(eb)
                            Next
                        Case 2
                            ' homing salvo: shoot several aimed bullets
                            For bI As Integer = 0 To boss.FireCount + 1
                                Dim bx = boss.X + rnd.Next(10, boss.Size - 10)
                                Dim by = boss.Y + boss.Size
                                Dim eb = New Bullet(bx, by, 6)
                                Dim dx = (playerX + playerSize \ 2) - eb.X
                                Dim dy = (playerY + playerSize \ 2) - eb.Y
                                Dim dist = Math.Sqrt(dx * dx + dy * dy)
                                If dist <> 0 Then
                                    eb.Vx = CInt(Math.Round(dx / dist * 4.0))
                                    eb.Vy = CInt(Math.Round(dy / dist * 4.0))
                                End If
                                enemyBullets.Add(eb)
                            Next
                        Case 3
                            ' bullet-hell ring + random lasers
                            Dim rings = 2
                            For rI As Integer = 0 To rings - 1
                                Dim shots = 18
                                For sI As Integer = 0 To shots - 1
                                    Dim angle = sI * (2 * Math.PI / shots) + rI * 0.1
                                    Dim vx = CInt(Math.Round(Math.Cos(angle) * (4 + rI)))
                                    Dim vy = CInt(Math.Round(Math.Sin(angle) * (4 + rI)))
                                    Dim eb = New Bullet(boss.X + boss.Size \ 2, boss.Y + boss.Size \ 2, vy)
                                    eb.Vx = vx
                                    enemyBullets.Add(eb)
                                Next
                            Next
                            ' occasional laser (fast downward wide bullets)
                            For l As Integer = -2 To 2
                                Dim eb2 = New Bullet(boss.X + boss.Size \ 2 + l * 18, boss.Y + boss.Size, 10)
                                eb2.Vx = l * 1
                                eb2.W = 6
                                eb2.H = 18
                                eb2.Damage = 2
                                enemyBullets.Add(eb2)
                            Next
                    End Select
                    boss.TriggerFireCooldown()
                End If

            End If

            ' Collisions: player bullets vs enemies
            For bi As Integer = bullets.Count - 1 To 0 Step -1
                Dim b = bullets(bi)
                Dim removedBullet As Boolean = False
                For ei As Integer = enemies.Count - 1 To 0 Step -1
                    Dim en = enemies(ei)
                    If RectIntersect(b.X, b.Y, b.W, b.H, en.X, en.Y, en.Size, en.Size) Then
                        ' Hit
                        en.Health -= b.Damage
                        bullets.RemoveAt(bi)
                        removedBullet = True
                        If en.Health <= 0 Then
                            score += en.ScoreValue
                            TrySpawnPowerup(en.X + en.Size \ 2, en.Y + en.Size \ 2)
                            If soundExplosion IsNot Nothing Then soundExplosion.Play()
                            enemies.RemoveAt(ei)
                        End If
                        Exit For
                    End If
                Next
                If removedBullet Then Continue For

                ' Boss collision
                If boss IsNot Nothing AndAlso RectIntersect(b.X, b.Y, b.W, b.H, boss.X, boss.Y, boss.Size, boss.Size) Then
                    boss.Health -= b.Damage
                    bullets.RemoveAt(bi)
                    If boss.Health <= 0 Then
                        score += boss.ScoreValue
                        boss = Nothing
                        StartNextLevel()
                        If soundExplosion IsNot Nothing Then soundExplosion.Play()
                    End If
                End If
            Next

            ' Collisions: enemy bullets vs player
            For i As Integer = enemyBullets.Count - 1 To 0 Step -1
                Dim eb = enemyBullets(i)
                If RectIntersect(eb.X, eb.Y, eb.W, eb.H, playerX, playerY, playerSize, playerSize) Then
                    enemyBullets.RemoveAt(i)
                    If playerShield > 0 Then
                        playerShield -= 1
                    Else
                        playerLives -= 1
                        If soundExplosion IsNot Nothing Then soundExplosion.Play()
                    End If
                    If playerLives <= 0 Then gameOver = True
                End If
            Next

            ' Collisions: enemies vs player
            For i As Integer = enemies.Count - 1 To 0 Step -1
                Dim en = enemies(i)
                If RectIntersect(en.X, en.Y, en.Size, en.Size, playerX, playerY, playerSize, playerSize) Then
                    enemies.RemoveAt(i)
                    If playerShield > 0 Then playerShield -= 1 Else playerLives -= 1
                    If soundExplosion IsNot Nothing Then soundExplosion.Play()
                    If playerLives <= 0 Then gameOver = True
                End If
            Next

            ' Collisions: player vs boss
            If boss IsNot Nothing AndAlso RectIntersect(boss.X, boss.Y, boss.Size, boss.Size, playerX, playerY, playerSize, playerSize) Then
                ' Collision hurts player, damages boss too
                boss.Health -= 10
                If playerShield > 0 Then playerShield -= 1 Else playerLives -= 1
                If soundExplosion IsNot Nothing Then soundExplosion.Play()
                If boss.Health <= 0 Then
                    score += boss.ScoreValue
                    boss = Nothing
                    StartNextLevel()
                End If
                If playerLives <= 0 Then gameOver = True
            End If

            ' Powerup pickups
            For i As Integer = powerups.Count - 1 To 0 Step -1
                Dim p = powerups(i)
                p.Y += 2
                If RectIntersect(playerX, playerY, playerSize, playerSize, p.X, p.Y, p.W, p.H) Then
                    ApplyPowerup(p.Type)
                    If soundPowerup IsNot Nothing Then soundPowerup.Play()
                    powerups.RemoveAt(i)
                ElseIf p.Y > Me.ClientSize.Height + 40 Then
                    powerups.RemoveAt(i)
                End If
            Next

            ' Check level boss spawn condition
            If boss Is Nothing AndAlso score >= level * 350 Then
                ' Spawn boss centered
                boss = New Boss(Me.ClientSize.Width \ 2 - 140, -240, 280, 160, 900 + level * 220, rnd)
                If soundBossEnter IsNot Nothing Then soundBossEnter.Play()
            End If

            ' Occasionally spawn a bonus health powerup
            If rnd.NextDouble() < 0.001 + level * 0.0009 Then
                powerups.Add(New PowerUp(rnd.Next(40, Me.ClientSize.Width - 40), -20, PowerUpType.Health))
            End If

            ' Occasionally spawn small enemy waves
            If rnd.NextDouble() < 0.002 + level * 0.001 Then
                Dim sx = rnd.Next(40, Me.ClientSize.Width - 40)
                For k As Integer = 0 To 2 + level \ 3
                    enemies.Add(New Enemy(sx + (k - 1) * 36, -40 - k * 30, 32, 2 + rnd.Next(0, 2), EnemyType.Flocker, rnd))
                Next
            End If

            Me.Invalidate()
        End Sub

        Private Sub FirePlayerBullet()
            ' Multi-shot when powerups active
            If rapidFireTicksLeft > 0 Then
                ' three-way spread
                bullets.Add(New Bullet(playerX + playerSize \ 2 - 4, playerY - 12, -14) With {.Vx = -2, .Damage = 6})
                bullets.Add(New Bullet(playerX + playerSize \ 2 - 4, playerY - 12, -16) With {.Vx = 0, .Damage = 6})
                bullets.Add(New Bullet(playerX + playerSize \ 2 - 4, playerY - 12, -14) With {.Vx = 2, .Damage = 6})
            Else
                bullets.Add(New Bullet(playerX + playerSize \ 2 - 4, playerY - 12, -16) With {.Damage = 5})
            End If
            If soundShoot IsNot Nothing Then soundShoot.Play()
        End Sub

        Private Sub SpawnEnemyForLevel()
            ' Choose enemy type based on level
            Dim t As Integer = rnd.Next(0, 100)
            If level < 3 Then
                ' mostly simple enemies
                enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 36, 2 + rnd.Next(0, 2), EnemyType.Flocker, rnd))
            ElseIf level < 6 Then
                If t < 50 Then
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 36, 2 + rnd.Next(0, 3), EnemyType.Flocker, rnd))
                ElseIf t < 80 Then
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 28, 3 + rnd.Next(0, 2), EnemyType.Shooter, rnd))
                Else
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 32, 2 + rnd.Next(0, 2), EnemyType.Kamikaze, rnd))
                End If
            Else
                If t < 35 Then
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 36, 3 + rnd.Next(0, 3), EnemyType.Flocker, rnd))
                ElseIf t < 65 Then
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 28, 3 + rnd.Next(0, 3), EnemyType.Shooter, rnd))
                ElseIf t < 85 Then
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 32, 3 + rnd.Next(0, 2), EnemyType.Kamikaze, rnd))
                Else
                    enemies.Add(New Enemy(rnd.Next(16, Me.ClientSize.Width - 48), -40, 40, 1 + rnd.Next(0, 1), EnemyType.Sniper, rnd))
                End If
            End If
        End Sub

        Private Sub TrySpawnPowerup(cx As Integer, cy As Integer)
            ' Chance to spawn
            Dim chance = 0.14 ' 14%
            If rnd.NextDouble() < chance Then
                Dim t = rnd.Next(0, 100)
                Dim typ As PowerUpType = PowerUpType.RapidFire
                If t < 45 Then
                    typ = PowerUpType.RapidFire
                ElseIf t < 75 Then
                    typ = PowerUpType.Shield
                Else
                    typ = PowerUpType.Health
                End If
                powerups.Add(New PowerUp(cx - 10, cy - 10, typ))
            End If
        End Sub

        Private Sub ApplyPowerup(t As PowerUpType)
            Select Case t
                Case PowerUpType.RapidFire
                    rapidFireTicksLeft += 600 ' ~10 seconds
                Case PowerUpType.Shield
                    playerShield = Math.Min(4, playerShield + 2)
                Case PowerUpType.Health
                    playerLives = Math.Min(6, playerLives + 1)
            End Select
        End Sub

        Private Sub GameForm_Paint(sender As Object, e As PaintEventArgs)
            Dim g = e.Graphics
            g.Clear(Color.Black)

            ' Draw stars (parallax)
            For Each s In stars
                Dim alpha = CInt(Math.Min(255, 50 + s.Speed * 60))
                Using b As New SolidBrush(Color.FromArgb(alpha, 255, 255, 255))
                    g.FillRectangle(b, s.X, s.Y, s.Size, s.Size)
                End Using
            Next

            ' Draw player
            DrawPlayer(g, playerX, playerY, playerSize, playerShield)

            ' Draw bullets
            For Each b In bullets
                g.FillRectangle(Brushes.LightYellow, b.X, b.Y, b.W, b.H)
            Next
            For Each eb In enemyBullets
                g.FillRectangle(Brushes.OrangeRed, eb.X, eb.Y, eb.W, eb.H)
            Next

            ' Draw enemies
            For Each en In enemies
                DrawEnemy(g, en)
            Next

            ' Draw boss
            If boss IsNot Nothing Then DrawBoss(g, boss)

            ' Draw powerups
            For Each p In powerups
                DrawPowerup(g, p)
            Next

            ' HUD
            Using f As New Font("Consolas", 12)
                g.DrawString(String.Format("Score: {0}   Level: {1}   Lives: {2}   Shield: {3}", score, level, playerLives, playerShield), f, Brushes.White, New PointF(6, 6))
            End Using

            If paused Then
                Using f As New Font("Arial", 28, FontStyle.Bold)
                    Dim s = "PAUSED"
                    Dim sz = g.MeasureString(s, f)
                    g.DrawString(s, f, Brushes.Orange, (Me.ClientSize.Width - sz.Width) / 2, (Me.ClientSize.Height - sz.Height) / 2)
                End Using
            End If

            If gameOver Then
                Using f As New Font("Arial", 24, FontStyle.Bold)
                    Dim s = "GAME OVER - Press R to restart"
                    Dim sz = g.MeasureString(s, f)
                    g.DrawString(s, f, Brushes.Red, (Me.ClientSize.Width - sz.Width) / 2, (Me.ClientSize.Height - sz.Height) / 2)
                End Using
            End If
        End Sub

        Private Sub DrawPlayer(g As Graphics, x As Integer, y As Integer, size As Integer, shield As Integer)
            ' Nose triangle
            Dim noseH = size \ 2
            Dim pts() As Point = {New Point(x + size \ 2, y), New Point(x + size, y + noseH), New Point(x, y + noseH)}
            g.FillPolygon(Brushes.LightBlue, pts)

            ' Body
            g.FillRectangle(Brushes.SteelBlue, x + size \ 4, y + noseH, size - (size \ 2), size \ 3)

            ' Wings
            Dim wingPts() As Point = {New Point(x + size \ 2, y + noseH), New Point(x + size + size \ 2 - 6, y + size \ 2), New Point(x + size \ 2, y + size \ 2)}
            g.FillPolygon(Brushes.SlateGray, wingPts)

            ' Shield indicator
            If shield > 0 Then
                Using p As New Pen(Color.Cyan, 3)
                    g.DrawEllipse(p, x - 6, y - 6, size + 12, size + 12)
                End Using
            End If
        End Sub

        Private Sub DrawEnemy(g As Graphics, en As Enemy)
            Dim rect = New Rectangle(en.X, en.Y, en.Size, en.Size)
            Select Case en.Type
                Case EnemyType.Flocker
                    g.FillEllipse(Brushes.Crimson, rect)
                    g.DrawEllipse(Pens.DarkRed, rect)
                Case EnemyType.Shooter
                    g.FillRectangle(Brushes.Gold, rect)
                    g.DrawRectangle(Pens.Orange, rect)
                Case EnemyType.Tanker
                    g.FillEllipse(Brushes.DarkMagenta, rect)
                    g.DrawEllipse(Pens.Purple, rect)
                Case EnemyType.Kamikaze
                    g.FillPolygon(Brushes.Orange, New Point() {New Point(en.X + en.Size \ 2, en.Y), New Point(en.X + en.Size, en.Y + en.Size), New Point(en.X, en.Y + en.Size)})
                    g.DrawPolygon(Pens.DarkOrange, New Point() {New Point(en.X + en.Size \ 2, en.Y), New Point(en.X + en.Size, en.Y + en.Size), New Point(en.X, en.Y + en.Size)})
                Case EnemyType.Sniper
                    g.FillRectangle(Brushes.LightGreen, rect)
                    g.DrawRectangle(Pens.DarkGreen, rect)
                Case EnemyType.Orbiter
                    g.FillEllipse(Brushes.CadetBlue, rect)
                    g.DrawEllipse(Pens.Teal, rect)
            End Select

            ' Health bar
            If en.MaxHealth > 0 Then
                Dim barW = en.Size
                Dim pct = CSng(en.Health) / en.MaxHealth
                g.FillRectangle(Brushes.Red, en.X, en.Y - 6, CInt(barW * pct), 4)
                g.DrawRectangle(Pens.White, en.X, en.Y - 6, barW, 4)
            End If
        End Sub

        Private Sub DrawBoss(g As Graphics, b As Boss)
            Dim rect = New Rectangle(b.X, b.Y, b.Size, b.Size)
            g.FillRectangle(Brushes.DimGray, rect)
            g.DrawRectangle(Pens.Black, rect)
            Using f As New Font("Consolas", 12, FontStyle.Bold)
                g.DrawString("BOSS", f, Brushes.Red, b.X + 8, b.Y + 8)
                g.DrawString("Phase: " & b.Phase.ToString(), SystemFonts.DefaultFont, Brushes.White, b.X + 8, b.Y + 28)
            End Using
            ' Boss health
            Dim barW = b.Size
            Dim pct = CSng(b.Health) / b.MaxHealth
            g.FillRectangle(Brushes.Red, b.X, b.Y - 12, CInt(barW * pct), 10)
            g.DrawRectangle(Pens.White, b.X, b.Y - 12, barW, 10)
        End Sub

        Private Sub DrawPowerup(g As Graphics, p As PowerUp)
            Dim r = New Rectangle(p.X, p.Y, p.W, p.H)
            Select Case p.Type
                Case PowerUpType.RapidFire
                    g.FillEllipse(Brushes.LimeGreen, r)
                    g.DrawEllipse(Pens.Green, r)
                    g.DrawString("R", SystemFonts.DefaultFont, Brushes.Black, p.X + 6, p.Y + 4)
                Case PowerUpType.Shield
                    g.FillEllipse(Brushes.DeepSkyBlue, r)
                    g.DrawEllipse(Pens.Blue, r)
                    g.DrawString("S", SystemFonts.DefaultFont, Brushes.Black, p.X + 6, p.Y + 4)
                Case PowerUpType.Health
                    g.FillEllipse(Brushes.HotPink, r)
                    g.DrawEllipse(Pens.DarkRed, r)
                    g.DrawString("H", SystemFonts.DefaultFont, Brushes.Black, p.X + 6, p.Y + 4)
            End Select
        End Sub

        Private Function RectIntersect(ax As Integer, ay As Integer, aw As Integer, ah As Integer, bx As Integer, by As Integer, bw As Integer, bh As Integer) As Boolean
            Return Not (ax + aw < bx OrElse ax > bx + bw OrElse ay + ah < by OrElse ay > by + bh)
        End Function

    End Class

    ' --- Game Entities ---
    Public Class Star
        Public Property X As Integer
        Public Property Y As Integer
        Public Property Size As Integer
        Public Property Speed As Single
        Public Sub New(x As Integer, y As Integer, size As Integer, speed As Single)
            Me.X = x
            Me.Y = y
            Me.Size = size
            Me.Speed = speed
        End Sub
    End Class

    Public Class Bullet
        Public Property X As Integer
        Public Property Y As Integer
        Public Property Vx As Integer
        Public Property Vy As Integer
        Public Property W As Integer = 8
        Public Property H As Integer = 12
        Public Property Damage As Integer = 4

        Public Sub New(x As Integer, y As Integer, vy As Integer)
            Me.X = x
            Me.Y = y
            Me.Vy = vy
            Me.Vx = 0
        End Sub
    End Class

    Public Enum EnemyType
        Flocker
        Shooter
        Tanker
        Kamikaze
        Sniper
        Orbiter
    End Enum

    Public Class Enemy
        Public Property X As Integer
        Public Property Y As Integer
        Public Property Size As Integer
        Public Property Speed As Integer
        Public Property Type As EnemyType
        Public Property Seed As Integer

        Public Property Health As Integer
        Public Property MaxHealth As Integer
        Public Property ScoreValue As Integer

        Public Property CanShoot As Boolean = False
        Public Property ShootProbability As Double = 0.01
        Public Property ShotSpeed As Double = 3.5

        ' Orbiter specific
        Private angle As Double = 0
        Private orbitCenterX As Integer = 0
        Private orbitCenterY As Integer = 0
        Private orbitRadius As Integer = 40

        Public Sub New(x As Integer, y As Integer, size As Integer, speed As Integer, t As EnemyType, r As Random)
            Me.X = x
            Me.Y = y
            Me.Size = size
            Me.Speed = speed
            Me.Type = t
            Me.Seed = If(r IsNot Nothing, r.Next(0, 1000), Environment.TickCount And &HFFFF)

            Select Case t
                Case EnemyType.Flocker
                    Me.MaxHealth = 8 * 2 ''Easy Mode
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 15
                    Me.CanShoot = False
                    Me.ShootProbability = 0
                    Me.ShotSpeed = 3
                Case EnemyType.Shooter
                    Me.MaxHealth = 6 * 2
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 25
                    Me.CanShoot = True
                    Me.ShootProbability = 0.02
                    Me.ShotSpeed = 3.8
                Case EnemyType.Tanker
                    Me.MaxHealth = 18 * 2
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 50
                    Me.CanShoot = False
                    Me.ShootProbability = 0
                    Me.ShotSpeed = 2.5
                Case EnemyType.Kamikaze
                    Me.MaxHealth = 10 * 2
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 30
                    Me.CanShoot = False
                    Me.ShootProbability = 0
                    Me.ShotSpeed = 3
                Case EnemyType.Sniper
                    Me.MaxHealth = 8 * 2
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 40
                    Me.CanShoot = True
                    Me.ShootProbability = 0.008
                    Me.ShotSpeed = 5.5
                Case EnemyType.Orbiter
                    Me.MaxHealth = 10 * 2
                    Me.Health = Me.MaxHealth
                    Me.ScoreValue = 35
                    Me.CanShoot = False
                    Me.ShootProbability = 0
                    Me.ShotSpeed = 3
                    ' initialize orbit center loosely around spawn
                    Me.orbitCenterX = x
                    Me.orbitCenterY = y + 40
                    Me.orbitRadius = 24 + (r.Next(0, 3) * 8)
                    Me.angle = r.NextDouble() * Math.PI * 2
            End Select
        End Sub

        Public Sub Update(bounds As Size, playerX As Integer, playerY As Integer, r As Random)
            Select Case Me.Type
                Case EnemyType.Flocker
                    ' gently steer towards player's X
                    Dim targetX = playerX
                    Dim dx = targetX - (Me.X + Me.Size \ 2)
                    Me.X += CInt(Math.Sign(dx) * Math.Min(3, Math.Abs(dx) \ 10 + 1))
                    ' sine wave vertical movement
                    Me.Y += Me.Speed + CInt(Math.Sin((Me.Seed + Me.Y) / 18.0) * 2)
                Case EnemyType.Shooter
                    ' zigzag while keeping downward pace
                    Me.X += CInt(Math.Sin((Me.Y + Me.Seed) / 14.0) * 4)
                    Me.Y += Me.Speed
                Case EnemyType.Tanker
                    ' slow straight down
                    Me.Y += Me.Speed
                Case EnemyType.Kamikaze
                    ' homes in on player
                    Dim dxk = (playerX + 20) - (Me.X + Me.Size \ 2)
                    Dim dyk = (playerY + 20) - (Me.Y + Me.Size \ 2)
                    Dim distk = Math.Sqrt(dxk * dxk + dyk * dyk)
                    If distk <> 0 Then
                        Me.X += CInt(Math.Round(dxk / distk * (Me.Speed + 3)))
                        Me.Y += CInt(Math.Round(dyk / distk * (Me.Speed + 3)))
                    Else
                        Me.Y += Me.Speed
                    End If
                Case EnemyType.Sniper
                    ' stays higher, slowly aligns then shoots
                    Me.Y += Math.Max(1, Me.Speed - 1)
                    Me.X += CInt(Math.Sign(playerX - Me.X) * Math.Min(2, Math.Abs(playerX - Me.X) \ 20 + 1))
                Case EnemyType.Orbiter
                    Me.angle += 0.06
                    Me.X = orbitCenterX + CInt(Math.Cos(angle) * orbitRadius)
                    Me.Y = orbitCenterY + CInt(Math.Sin(angle) * orbitRadius)
                    ' slowly drift downward
                    orbitCenterY += 1
            End Select

            ' Bound X
            If Me.X < -200 Then Me.X = -200
            If Me.X + Me.Size > bounds.Width + 200 Then Me.X = bounds.Width + 200 - Me.Size
        End Sub
    End Class

    Public Class Boss
        Public Property X As Integer
        Public Property Y As Integer
        Public Property Size As Integer
        Public Property MaxHealth As Integer
        Public Property Health As Integer
        Public Property Speed As Integer
        Public Property ScoreValue As Integer

        Private fireCooldown As Integer = 0
        Private fireRate As Integer = 90
        Public Property FireCount As Integer = 5

        Private rnd As Random

        Public Property Phase As Integer = 1

        Public Sub New(x As Integer, y As Integer, sizeX As Integer, sizeY As Integer, maxHealth As Integer, r As Random)
            Me.X = x
            Me.Y = y
            Me.Size = sizeX
            Me.MaxHealth = maxHealth
            Me.Health = maxHealth
            Me.Speed = 2
            Me.ScoreValue = 1000
            Me.fireCooldown = fireRate
            Me.FireCount = 5
            Me.rnd = If(r, New Random())
        End Sub

        Public Sub Update(bounds As Size, playerX As Integer, playerY As Integer, r As Random)
            ' Boss moves slowly left-right, occasionally dive towards player
            Dim centerX = bounds.Width \ 2 - Me.Size \ 2
            Dim targetX = centerX + CInt(Math.Sin((Me.Y + Environment.TickCount) / 700.0) * 140)
            Dim dx = targetX - Me.X
            Me.X += CInt(Math.Sign(dx) * Math.Min(4, Math.Abs(dx) \ 6 + 1))

            If Me.Y < 30 Then
                Me.Y += 3
            ElseIf r.NextDouble() < 0.004 + Phase * 0.002 Then
                Me.Y += 12
            Else
                ' small float
                Me.Y += CInt(Math.Sin(Environment.TickCount / 400.0) * 1)
            End If

            ' fire control
            If fireCooldown > 0 Then fireCooldown -= 1

            ' adjust fire rate by phase
            Select Case Phase
                Case 1
                    fireRate = 90
                    FireCount = 5
                Case 2
                    fireRate = 60
                    FireCount = 7
                Case 3
                    fireRate = 36
                    FireCount = 10
            End Select
        End Sub

        Public ReadOnly Property CanFire As Boolean
            Get
                Return fireCooldown <= 0
            End Get
        End Property

        Public Sub TriggerFireCooldown()
            fireCooldown = fireRate
        End Sub

        Public Sub CheckPhase()
            Dim newPhase = 1
            Dim pct = CSng(Me.Health) / Me.MaxHealth
            If pct < 0.35 Then
                newPhase = 3
            ElseIf pct < 0.7 Then
                newPhase = 2
            Else
                newPhase = 1
            End If
            If newPhase <> Me.Phase Then
                Me.Phase = newPhase
                ' small visual reaction (could be enhanced)
                ' also slightly heal or change behavior if desired
            End If
        End Sub
    End Class

    Public Enum PowerUpType
        RapidFire
        Shield
        Health
    End Enum

    Public Class PowerUp
        Public Property X As Integer
        Public Property Y As Integer
        Public Property W As Integer = 24
        Public Property H As Integer = 24
        Public Property Type As PowerUpType

        Public Sub New(x As Integer, y As Integer, Optional t As PowerUpType = PowerUpType.RapidFire)
            Me.X = x
            Me.Y = y
            Me.Type = t
        End Sub
    End Class

End Module

