﻿Imports System.Drawing
Imports System.Drawing.Imaging
Imports System.Collections.Generic

Public Class SilhouetteGenerator

    ' -------------------------------------------------------
    ' Main function to generate a high-quality silhouette
    ' -------------------------------------------------------
    Public Shared Function GenerateSilhouette(input As Bitmap) As Bitmap
        Dim bgColor As Color = DetectBackgroundColor(input)

        Dim mask = CreateForegroundMask(input, bgColor)
        mask = ApplySobelEdgeBoost(mask)
        mask = MorphologicalSmooth(mask)
        mask = FloodFillLargestShape(mask)

        Dim silhouette = CreateSilhouetteBitmap(mask)
        Return silhouette
    End Function

    ' -------------------------------------------------------
    ' Detect background using corner averaging
    ' -------------------------------------------------------
    Private Shared Function DetectBackgroundColor(bmp As Bitmap) As Color
        Dim corners As New List(Of Color) From {
            bmp.GetPixel(0, 0),
            bmp.GetPixel(bmp.Width - 1, 0),
            bmp.GetPixel(0, bmp.Height - 1),
            bmp.GetPixel(bmp.Width - 1, bmp.Height - 1)
        }

        Dim r = CInt(corners.Average(Function(c) c.R))
        Dim g = CInt(corners.Average(Function(c) c.G))
        Dim b = CInt(corners.Average(Function(c) c.B))

        Return Color.FromArgb(r, g, b)
    End Function

    ' -------------------------------------------------------
    ' Create binary mask: 255 = foreground, 0 = background
    ' -------------------------------------------------------
    Private Shared Function CreateForegroundMask(bmp As Bitmap, bg As Color) As Byte(,)
        Dim w = bmp.Width
        Dim h = bmp.Height

        Dim mask(h - 1, w - 1) As Byte

        For y = 0 To h - 1
            For x = 0 To w - 1
                Dim c = bmp.GetPixel(x, y)
                Dim diff = Math.Abs(CInt(c.R) - CInt(bg.R)) + Math.Abs(CInt(c.G) - CInt(bg.G)) + Math.Abs(CInt(c.B) - CInt(bg.B))

                mask(y, x) = If(diff > 35, CByte(255), CByte(0))
            Next
        Next

        Return mask
    End Function

    ' -------------------------------------------------------
    ' Sobel edge detection to strengthen contours
    ' -------------------------------------------------------
    Private Shared Function ApplySobelEdgeBoost(mask As Byte(,)) As Byte(,)
        Dim h = mask.GetLength(0)
        Dim w = mask.GetLength(1)
        Dim result(h - 1, w - 1) As Byte

        Dim gx(,) As Integer = {{-1, 0, 1}, {-2, 0, 2}, {-1, 0, 1}}
        Dim gy(,) As Integer = {{1, 2, 1}, {0, 0, 0}, {-1, -2, -1}}

        For y = 1 To h - 2
            For x = 1 To w - 2
                Dim sumX As Integer = 0
                Dim sumY As Integer = 0

                For ky = -1 To 1
                    For kx = -1 To 1
                        sumX += gx(ky + 1, kx + 1) * mask(y + ky, x + kx)
                        sumY += gy(ky + 1, kx + 1) * mask(y + ky, x + kx)
                    Next
                Next

                Dim mag = Math.Min(255, Math.Sqrt(sumX * sumX + sumY * sumY))
                result(y, x) = If(mag > 40, CByte(255), mask(y, x))
            Next
        Next

        Return result
    End Function

    ' -------------------------------------------------------
    ' Morphological smoothing (closing)
    ' -------------------------------------------------------
    Private Shared Function MorphologicalSmooth(mask As Byte(,)) As Byte(,)
        Dim h = mask.GetLength(0)
        Dim w = mask.GetLength(1)
        Dim result(h - 1, w - 1) As Byte

        For y = 1 To h - 2
            For x = 1 To w - 2
                Dim count As Integer =
                    CInt(mask(y - 1, x - 1)) + CInt(mask(y - 1, x)) + CInt(mask(y - 1, x + 1)) +
                    CInt(mask(y, x - 1)) + CInt(mask(y, x)) + CInt(mask(y, x + 1)) +
                    CInt(mask(y + 1, x - 1)) + CInt(mask(y + 1, x)) + CInt(mask(y + 1, x + 1))

                result(y, x) = If(count >= 3 * 255, CByte(255), mask(y, x))
            Next
        Next

        Return result
    End Function

    ' -------------------------------------------------------
    ' Keep only the largest connected region (the person)
    ' -------------------------------------------------------
    Private Shared Function FloodFillLargestShape(mask As Byte(,)) As Byte(,)
        Dim h = mask.GetLength(0)
        Dim w = mask.GetLength(1)

        Dim visited(h - 1, w - 1) As Boolean
        Dim bestRegion As New List(Of Point)
        Dim q As New Queue(Of Point)

        For y = 0 To h - 1
            For x = 0 To w - 1
                If mask(y, x) = 255 AndAlso Not visited(y, x) Then
                    Dim region As New List(Of Point)
                    q.Enqueue(New Point(x, y))
                    visited(y, x) = True

                    While q.Count > 0
                        Dim p = q.Dequeue()
                        region.Add(p)

                        For Each d In {New Point(1, 0), New Point(-1, 0),
                                       New Point(0, 1), New Point(0, -1)}
                            Dim nx = p.X + d.X
                            Dim ny = p.Y + d.Y

                            If nx >= 0 And nx < w And ny >= 0 And ny < h Then
                                If mask(ny, nx) = 255 AndAlso Not visited(ny, nx) Then
                                    visited(ny, nx) = True
                                    q.Enqueue(New Point(nx, ny))
                                End If
                            End If
                        Next
                    End While

                    If region.Count > bestRegion.Count Then bestRegion = region
                End If
            Next
        Next

        Dim cleaned(h - 1, w - 1) As Byte
        For Each p In bestRegion
            cleaned(p.Y, p.X) = 255
        Next

        Return cleaned
    End Function

    ' -------------------------------------------------------
    ' Render silhouette bitmap
    ' -------------------------------------------------------
    Private Shared Function CreateSilhouetteBitmap(mask As Byte(,)) As Bitmap
        Dim h = mask.GetLength(0)
        Dim w = mask.GetLength(1)

        Dim result As New Bitmap(w, h, PixelFormat.Format32bppArgb)

        For y = 0 To h - 1
            For x = 0 To w - 1
                If mask(y, x) = 255 Then
                    result.SetPixel(x, y, Color.Black)
                Else
                    result.SetPixel(x, y, Color.Transparent)
                End If
            Next
        Next

        Return result
    End Function

End Class
