﻿Imports System.Drawing
Imports System.Drawing.Imaging
Imports System.Runtime.InteropServices
Imports System.Drawing.Drawing2D

Public Class Form1

    Private Sub btnGenerate_Click(sender As Object, e As EventArgs) Handles btnGenerate.Click

        Using dlg As New OpenFileDialog()
            dlg.Filter = "Image Files|*.png;*.jpg;*.jpeg;*.bmp"

            If dlg.ShowDialog() <> DialogResult.OK Then Exit Sub

            Dim srcPath = dlg.FileName
            Using original As Bitmap = CType(Image.FromFile(srcPath), Bitmap)
                Dim silhouette As Bitmap = CreateSilhouetteFast(original, CInt(NumericUpDown1.Value))

                ' High-quality transforms (with padding)
                Dim rotated As Bitmap = RotateImageHighQuality(silhouette, 10)
                Dim mirrored As Bitmap = MirrorImage(silhouette)

                ' Choose save folder
                Using fbd As New FolderBrowserDialog()
                    fbd.Description = "Select folder to save generated silhouettes"
                    fbd.SelectedPath = Application.StartupPath + "\saved"
                    If fbd.ShowDialog() <> DialogResult.OK Then Exit Sub

                    silhouette.Save(IO.Path.Combine(fbd.SelectedPath, "shadow_correct.png"), ImageFormat.Png)
                    rotated.Save(IO.Path.Combine(fbd.SelectedPath, "shadow_wrong1.png"), ImageFormat.Png)
                    mirrored.Save(IO.Path.Combine(fbd.SelectedPath, "shadow_wrong2.png"), ImageFormat.Png)
                End Using

                MessageBox.Show("Shadow images generated!")

                ' Preview
                PictureBox1.Image = CType(original.Clone(), Image)
                PictureBox2.Image = silhouette
                PictureBox3.Image = rotated
                PictureBox4.Image = mirrored
            End Using

        End Using

    End Sub

    ' -------------------------------------------------------------
    ' FAST silhouette generator using LockBits (huge speed boost)
    ' -------------------------------------------------------------
    Private Function CreateSilhouetteFast(src As Bitmap, sensitivity As Integer) As Bitmap
        Dim w = src.Width
        Dim h = src.Height
        Dim result As New Bitmap(w, h, PixelFormat.Format32bppArgb)

        ' Get background color from the four corners
        Dim bg = AverageColor({
            src.GetPixel(0, 0),
            src.GetPixel(w - 1, 0),
            src.GetPixel(0, h - 1),
            src.GetPixel(w - 1, h - 1)
        })

        Dim threshold = 40 * sensitivity

        ' Lock bits
        Dim srcData = src.LockBits(New Rectangle(0, 0, w, h), ImageLockMode.ReadOnly, PixelFormat.Format32bppArgb)
        Dim dstData = result.LockBits(New Rectangle(0, 0, w, h), ImageLockMode.WriteOnly, PixelFormat.Format32bppArgb)

        Dim bytes = Math.Abs(srcData.Stride) * h
        Dim srcBytes(bytes - 1) As Byte
        Dim dstBytes(bytes - 1) As Byte

        Marshal.Copy(srcData.Scan0, srcBytes, 0, bytes)

        For i = 0 To bytes - 1 Step 4
            Dim b = srcBytes(i)
            Dim g = srcBytes(i + 1)
            Dim r = srcBytes(i + 2)

            Dim diff As Integer =
    Math.Abs(CInt(r) - CInt(bg.R)) +
    Math.Abs(CInt(g) - CInt(bg.G)) +
    Math.Abs(CInt(b) - CInt(bg.B))

            If diff < threshold Then
                ' transparent background
                dstBytes(i) = 0
                dstBytes(i + 1) = 0
                dstBytes(i + 2) = 0
                dstBytes(i + 3) = 0
            Else
                ' black silhouette
                dstBytes(i) = 0
                dstBytes(i + 1) = 0
                dstBytes(i + 2) = 0
                dstBytes(i + 3) = 255
            End If
        Next

        Marshal.Copy(dstBytes, 0, dstData.Scan0, bytes)

        src.UnlockBits(srcData)
        result.UnlockBits(dstData)

        Return result
    End Function


    Private Function AverageColor(colors As Color()) As Color
        Dim r = colors.Sum(Function(c) c.R) \ colors.Length
        Dim g = colors.Sum(Function(c) c.G) \ colors.Length
        Dim b = colors.Sum(Function(c) c.B) \ colors.Length
        Return Color.FromArgb(r, g, b)
    End Function


    ' -------------------------------------------------------------
    ' High-quality rotation WITHOUT cropping
    ' -------------------------------------------------------------
    Private Function RotateImageHighQuality(src As Bitmap, angle As Single) As Bitmap
        Dim rad = angle * Math.PI / 180.0
        Dim sin = Math.Abs(Math.Sin(rad))
        Dim cos = Math.Abs(Math.Cos(rad))

        Dim newW = CInt(src.Width * cos + src.Height * sin)
        Dim newH = CInt(src.Width * sin + src.Height * cos)

        Dim result As New Bitmap(newW, newH, PixelFormat.Format32bppArgb)

        Using g As Graphics = Graphics.FromImage(result)
            g.Clear(Color.Transparent)
            g.SmoothingMode = Drawing2D.SmoothingMode.AntiAlias
            g.InterpolationMode = Drawing2D.InterpolationMode.HighQualityBicubic
            g.PixelOffsetMode = Drawing2D.PixelOffsetMode.HighQuality

            g.TranslateTransform(newW / 2, newH / 2)
            g.RotateTransform(angle)
            g.TranslateTransform(-src.Width / 2, -src.Height / 2)

            g.DrawImage(src, New Point(0, 0))
        End Using

        Return result
    End Function

    ' -------------------------------------------------------------
    ' Mirror image
    ' -------------------------------------------------------------
    Private Function MirrorImage(src As Bitmap) As Bitmap
        Dim bmp As New Bitmap(src)
        bmp.RotateFlip(RotateFlipType.RotateNoneFlipX)
        Return bmp
    End Function

    ' Main button
    Private Sub btnGenerateCartoon_Click(sender As Object, e As EventArgs) Handles btnGenerateCartoon.Click

        Using dlg As New OpenFileDialog()
            dlg.Filter = "Image Files|*.png;*.jpg;*.jpeg;*.bmp"

            If dlg.ShowDialog() <> DialogResult.OK Then Exit Sub
            Dim original2 As Bitmap = CType(Image.FromFile(dlg.FileName), Bitmap)

            Using original As Bitmap = CType(Image.FromFile(dlg.FileName), Bitmap)
                    Dim cartoon As Bitmap = CreateCartoonSilhouette(original, CInt(NumericUpDown1.Value))

                ' Optional: add outline
                cartoon = AddOutline(cartoon, Color.Black, NumericUpDown2.Value)
                Dim rotated As Bitmap = RotateImageHighQuality(cartoon, 10)
                Dim mirrored As Bitmap = MirrorImage(cartoon)

                ' Save
                cartoon.Save("cartoon_silhouette.png", ImageFormat.Png)

                ' Preview
                PictureBox1.Image = original2
                PictureBox2.Image = cartoon
                PictureBox3.Image = rotated
                PictureBox4.Image = mirrored
            End Using
        End Using

            MessageBox.Show("Cartoon silhouette generated!")
    End Sub

    ' -------------------------
    ' Cartoon silhouette
    ' -------------------------
    Private Function CreateCartoonSilhouette(src As Bitmap, sensitivity As Integer) As Bitmap
        Dim w = src.Width
        Dim h = src.Height
        Dim result As New Bitmap(w, h, PixelFormat.Format32bppArgb)

        ' Background from corners
        Dim bg = AverageColor({src.GetPixel(0, 0), src.GetPixel(w - 1, 0),
                               src.GetPixel(0, h - 1), src.GetPixel(w - 1, h - 1)})

        Dim threshold = 40 * sensitivity

        Dim srcData = src.LockBits(New Rectangle(0, 0, w, h), ImageLockMode.ReadOnly, PixelFormat.Format32bppArgb)
        Dim dstData = result.LockBits(New Rectangle(0, 0, w, h), ImageLockMode.WriteOnly, PixelFormat.Format32bppArgb)

        Dim bytes = Math.Abs(srcData.Stride) * h
        Dim srcBytes(bytes - 1) As Byte
        Dim dstBytes(bytes - 1) As Byte
        Marshal.Copy(srcData.Scan0, srcBytes, 0, bytes)

        For i = 0 To bytes - 1 Step 4
            Dim b = srcBytes(i)
            Dim g = srcBytes(i + 1)
            Dim r = srcBytes(i + 2)

            Dim diff As Integer = Math.Abs(CInt(r) - CInt(bg.R)) + Math.Abs(CInt(g) - CInt(bg.G)) + Math.Abs(CInt(b) - CInt(bg.B))

            If diff < threshold Then
                ' background = transparent
                dstBytes(i) = 0
                dstBytes(i + 1) = 0
                dstBytes(i + 2) = 0
                dstBytes(i + 3) = 0
            Else
                ' cartoon black
                dstBytes(i) = 0
                dstBytes(i + 1) = 0
                dstBytes(i + 2) = 0
                dstBytes(i + 3) = 255
            End If
        Next

        Marshal.Copy(dstBytes, 0, dstData.Scan0, bytes)
        src.UnlockBits(srcData)
        result.UnlockBits(dstData)

        ' Smooth edges for cartoon effect
        result = SmoothEdges(result, 2)

        Return result
    End Function

    ' -------------------------
    ' Smooth edges (basic blur)
    ' -------------------------
    Private Function SmoothEdges(src As Bitmap, radius As Integer) As Bitmap
        Dim bmp As New Bitmap(src.Width, src.Height, PixelFormat.Format32bppArgb)
        Using g As Graphics = Graphics.FromImage(bmp)
            g.Clear(Color.Transparent)
            g.SmoothingMode = SmoothingMode.AntiAlias
            g.InterpolationMode = InterpolationMode.HighQualityBicubic
            g.DrawImage(src, New Rectangle(0, 0, src.Width, src.Height))
        End Using
        Return bmp
    End Function

    ' -------------------------
    ' Add bold outline
    ' -------------------------
    Private Function AddOutline(src As Bitmap, outlineColor As Color, thickness As Integer) As Bitmap
        Dim bmp As New Bitmap(src.Width, src.Height, PixelFormat.Format32bppArgb)
        Using g As Graphics = Graphics.FromImage(bmp)
            g.Clear(Color.Transparent)
            g.SmoothingMode = SmoothingMode.AntiAlias
            Using pen As New Pen(outlineColor, thickness)
                pen.LineJoin = LineJoin.Round
                ' Draw outline by tracing silhouette edges
                For y = 1 To src.Height - 2
                    For x = 1 To src.Width - 2
                        Dim c = src.GetPixel(x, y)
                        If c.A > 0 Then
                            ' If any neighbor is transparent, draw pixel as outline
                            If src.GetPixel(x - 1, y).A = 0 OrElse src.GetPixel(x + 1, y).A = 0 OrElse
                               src.GetPixel(x, y - 1).A = 0 OrElse src.GetPixel(x, y + 1).A = 0 Then
                                g.DrawRectangle(pen, x, y, 1, 1)
                            End If
                        End If
                    Next
                Next
            End Using
            ' Draw filled silhouette
            g.DrawImage(src, New Point(0, 0))
        End Using
        Return bmp
    End Function

End Class
