﻿Imports System.Drawing
Imports System.Windows.Forms

Public Class Form1
    Inherits Form

    ' --- Settings ---
    Private Const TICK_MS As Integer = 20
    Private Const PLAYER_SPEED As Integer = 6
    Private Const BULLET_SPEED As Integer = 10
    Private Const ENEMY_SPEED As Integer = 2

    Private explosions As New List(Of Explosion)

    ' --- Game state ---
    Private gameTimer As Timer
    Private spawnTimer As Timer
    Private rand As New Random()

    Private playerPos As PointF
    Private playerSize As SizeF = New SizeF(30, 20)
    Private playerLives As Integer = 5
    Private score As Integer = 0

    Private bullets As List(Of Bullet)
    Private enemies As List(Of Enemy)
    Private zones As List(Of ZoneSegment)

    Private keysDown As New HashSet(Of Keys)

    Private boss As Enemy = Nothing
    Private bossBullets As New List(Of Bullet)

    ' --------------------
    ' INTERNAL CLASSES
    ' --------------------
    Private Class Bullet
        Public Property Pos As PointF
        Public Property Dir As PointF
        Public Property Alive As Boolean = True
        Public Property Rect As RectangleF
        Public Sub New(p As PointF, d As PointF)
            Pos = p
            Dir = d
        End Sub
    End Class

    Private Class Enemy
        Public Property Pos As PointF
        Public Property Vel As PointF
        Public Property HP As Integer = 1
        Public Property Alive As Boolean = True
        Public Sub New(p As PointF, v As PointF)
            Pos = p
            Vel = v
        End Sub
    End Class

    Private Class ZoneSegment
        Public Property Offset As Integer
        Public Property ColorA As Color
        Public Property ColorB As Color
        Public Sub New(off As Integer, ca As Color, cb As Color)
            Offset = off
            ColorA = ca
            ColorB = cb
        End Sub
    End Class

    Private Class ExplosionParticle
        Public Property Pos As PointF
        Public Property Vel As PointF
        Public Property Life As Integer
        Public Property Color As Color
    End Class

    Private Class Explosion
        Public Property Particles As List(Of ExplosionParticle)
        Public Property Alive As Boolean = True
    End Class

    ' --------------------
    ' CONSTRUCTOR
    ' --------------------
    Public Sub New()
        Me.Text = "Vanguard (VB.NET)"
        Me.ClientSize = New Size(800, 600)
        Me.DoubleBuffered = True
        Me.BackColor = Color.Black
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()

        InitGameData()

        gameTimer = New Timer With {.Interval = TICK_MS}
        AddHandler gameTimer.Tick, AddressOf GameTick
        gameTimer.Start()

        spawnTimer = New Timer With {.Interval = 1000}
        AddHandler spawnTimer.Tick, AddressOf SpawnEnemyTick
        spawnTimer.Start()

        AddHandler Me.KeyDown, AddressOf OnKeyDown
        AddHandler Me.KeyUp, AddressOf OnKeyUp
    End Sub

    Private Sub InitGameData()
        bullets = New List(Of Bullet)
        enemies = New List(Of Enemy)

        zones = New List(Of ZoneSegment) From {
            New ZoneSegment(0, Color.DarkSlateBlue, Color.MidnightBlue),
            New ZoneSegment(0, Color.DarkGreen, Color.FromArgb(0, 64, 0))
        }

        playerPos = New PointF(100, ClientSize.Height / 2 - playerSize.Height / 2)
        playerLives = 3
        score = 0
    End Sub

    ' --------------------
    ' INPUT
    ' --------------------
    Private Sub OnKeyDown(sender As Object, e As KeyEventArgs)
        If Not keysDown.Contains(e.KeyCode) Then keysDown.Add(e.KeyCode)

        ' Fire
        Select Case e.KeyCode
            Case Keys.Up, Keys.Down, Keys.Left, Keys.Right
                FireBullet(e.KeyCode)
        End Select

        If e.KeyCode = Keys.R Then ResetGame()
    End Sub

    Private Sub OnKeyUp(sender As Object, e As KeyEventArgs)
        If keysDown.Contains(e.KeyCode) Then keysDown.Remove(e.KeyCode)
    End Sub

    ' --------------------
    ' GAME TICK
    ' --------------------
    Private Sub GameTick(sender As Object, e As EventArgs)
        UpdatePlayer()
        UpdateBullets()
        UpdateEnemies()
        UpdateExplosions()
        CheckCollisions()
        If boss Is Nothing AndAlso score >= 3000 Then
            SpawnBoss()
        End If

        If boss IsNot Nothing Then
            UpdateBoss()
            UpdateBossBullets()
        End If
        Invalidate()
    End Sub

    ' --------------------
    ' PLAYER
    ' --------------------
    Private Sub UpdatePlayer()
        Dim dx As Single = 0
        Dim dy As Single = 0

        If keysDown.Contains(Keys.A) OrElse keysDown.Contains(Keys.Left) Then dx -= 1
        If keysDown.Contains(Keys.D) OrElse keysDown.Contains(Keys.Right) Then dx += 1
        If keysDown.Contains(Keys.W) OrElse keysDown.Contains(Keys.Up) Then dy -= 1
        If keysDown.Contains(Keys.S) OrElse keysDown.Contains(Keys.Down) Then dy += 1

        If dx <> 0 OrElse dy <> 0 Then
            Dim len As Single = CSng(Math.Sqrt(dx * dx + dy * dy))
            dx = dx / len * PLAYER_SPEED
            dy = dy / len * PLAYER_SPEED

            playerPos = New PointF(
                Math.Max(0, Math.Min(playerPos.X + dx, ClientSize.Width - playerSize.Width)),
                Math.Max(0, Math.Min(playerPos.Y + dy, ClientSize.Height - playerSize.Height))
            )
        End If
    End Sub

    ' --------------------
    ' BULLETS
    ' --------------------
    Private Sub FireBullet(k As Keys)
        Dim dir As PointF

        Select Case k
            Case Keys.Up : dir = New PointF(0, -1)
            Case Keys.Down : dir = New PointF(0, 1)
            Case Keys.Left : dir = New PointF(-1, 0)
            Case Keys.Right : dir = New PointF(1, 0)
            Case Else : Exit Sub
        End Select

        Dim origin As New PointF(
            playerPos.X + playerSize.Width / 2,
            playerPos.Y + playerSize.Height / 2
        )

        bullets.Add(New Bullet(origin, dir))
    End Sub

    Private Sub UpdateBullets()
        For Each b In bullets
            If Not b.Alive Then Continue For

            b.Pos = New PointF(
                b.Pos.X + b.Dir.X * BULLET_SPEED,
                b.Pos.Y + b.Dir.Y * BULLET_SPEED
            )

            b.Rect = New RectangleF(b.Pos.X - 3, b.Pos.Y - 3, 6, 6)

            If b.Pos.X < -20 OrElse b.Pos.X > ClientSize.Width + 20 _
            OrElse b.Pos.Y < -20 OrElse b.Pos.Y > ClientSize.Height + 20 Then
                b.Alive = False
            End If
        Next

        bullets.RemoveAll(Function(x) Not x.Alive)
    End Sub

    ' --------------------
    ' ENEMIES
    ' --------------------
    Private Sub SpawnEnemyTick(sender As Object, e As EventArgs)
        SpawnEnemy()
    End Sub

    Private Sub SpawnEnemy()
        Dim x As Single = ClientSize.Width + 40
        Dim y As Single = rand.Next(40, ClientSize.Height - 40)

        Dim vx As Single = -ENEMY_SPEED - CSng(rand.NextDouble() * 2)
        Dim vy As Single = CSng((rand.NextDouble() - 0.5) * 3)

        Dim hp As Integer =
            If(rand.NextDouble() < 0.1, 3,
            If(rand.NextDouble() < 0.2, 2, 1))

        Dim eObj As New Enemy(New PointF(x, y), New PointF(vx, vy))
        eObj.HP = hp
        enemies.Add(eObj)
    End Sub

    Private Sub UpdateEnemies()
        For Each en In enemies
            If Not en.Alive Then Continue For

            en.Pos = New PointF(en.Pos.X + en.Vel.X, en.Pos.Y + en.Vel.Y)

            If en.Pos.Y < 10 OrElse en.Pos.Y > ClientSize.Height - 30 Then
                en.Vel = New PointF(en.Vel.X, -en.Vel.Y)
            End If

            If en.Pos.X < -40 Then
                en.Alive = False
                playerLives -= 1
                If playerLives <= 0 Then GameOver()
            End If
        Next

        enemies.RemoveAll(Function(x) Not x.Alive)
    End Sub

    ' --------------------
    ' COLLISIONS
    ' --------------------
    Private Sub CheckCollisions()
        Dim pRect As New RectangleF(playerPos.X, playerPos.Y, playerSize.Width, playerSize.Height)

        ' Bullet vs enemy
        For Each b In bullets
            If Not b.Alive Then Continue For
            For Each en In enemies
                If Not en.Alive Then Continue For

                Dim er As New RectangleF(en.Pos.X - 12, en.Pos.Y - 12, 24, 24)

                If er.IntersectsWith(b.Rect) Then
                    en.HP -= 1
                    b.Alive = False

                    If en.HP <= 0 Then
                        en.Alive = False
                        score += 100

                        Dim center As New PointF(en.Pos.X, en.Pos.Y)
                        CreateExplosion(center, 25)  ' <<< NEW
                    Else
                        score += 25
                    End If

                    Exit For
                End If

                If boss IsNot Nothing AndAlso boss.Alive Then
                    Dim br As New RectangleF(boss.Pos.X - 60, boss.Pos.Y - 60, 120, 120)

                    If br.IntersectsWith(b.Rect) Then
                        boss.HP -= 10
                        b.Alive = False

                        If boss.HP <= 0 Then
                            boss.Alive = False
                            CreateExplosion(New PointF(boss.Pos.X, boss.Pos.Y), 80)
                            score += 5000
                        End If
                    End If
                End If

                For Each bb In bossBullets
                    If bb.Rect.IntersectsWith(pRect) Then
                        bb.Alive = False
                        playerLives -= 1
                        CreateExplosion(New PointF(playerPos.X, playerPos.Y), 50)
                        If playerLives <= 0 Then GameOver()
                    End If
                Next

            Next
        Next

        ' Enemy vs player
        For Each en In enemies
            Dim er As New RectangleF(en.Pos.X - 12, en.Pos.Y - 12, 24, 24)
            If er.IntersectsWith(pRect) Then
                CreateExplosion(New PointF(en.Pos.X, en.Pos.Y), 20)
                en.Alive = False
                playerLives -= 1
                If playerLives <= 0 Then GameOver()
            End If
        Next
    End Sub

    ' --------------------
    ' RESET / GAME OVER
    ' --------------------
    Private Sub ResetGame()
        InitGameData()
    End Sub

    Private Sub GameOver()
        gameTimer.Stop()
        spawnTimer.Stop()

        Dim r = MessageBox.Show(
            $"GAME OVER" & vbCrLf &
            $"Score: {score}" & vbCrLf &
            "Retry?",
            "Vanguard",
            MessageBoxButtons.YesNo
        )

        If r = DialogResult.Yes Then
            InitGameData()
            gameTimer.Start()
            spawnTimer.Start()
        Else
            Close()
        End If
    End Sub

    ' --------------------
    ' DRAWING
    ' --------------------
    Protected Overrides Sub OnPaint(e As PaintEventArgs)
        Dim g = e.Graphics

        DrawZones(g)
        DrawPlayer(g)

        For Each b In bullets
            g.FillEllipse(Brushes.Yellow, b.Rect)
        Next

        For Each en In enemies
            Dim col As Color = If(en.HP = 3, Color.OrangeRed, If(en.HP = 2, Color.Orange, Color.LightCoral))
            Dim rect As New RectangleF(en.Pos.X - 12, en.Pos.Y - 12, 24, 24)
            Using br As New SolidBrush(col)
                g.FillEllipse(br, rect)
            End Using
        Next

        ' draw explosions
        For Each ex In explosions
            For Each p In ex.Particles
                If p.Life > 0 Then
                    Dim alpha As Integer = Math.Min(255, Math.Max(0, p.Life * 10))
                    Using br As New SolidBrush(Color.FromArgb(alpha, p.Color.R, p.Color.G, p.Color.B))
                        g.FillEllipse(br, p.Pos.X, p.Pos.Y, 4, 4)
                    End Using
                End If
            Next
        Next

        ' Draw boss
        If boss IsNot Nothing AndAlso boss.Alive Then
            Dim rect As New RectangleF(boss.Pos.X - 60, boss.Pos.Y - 60, 120, 120)
            Using br As New SolidBrush(Color.DarkRed)
                g.FillEllipse(br, rect)
            End Using

            ' HP bar
            Dim hpPerc = boss.HP / 500.0!
            g.FillRectangle(Brushes.DarkRed, 200, 20, 400, 12)
            g.FillRectangle(Brushes.Red, 200, 20, 400 * hpPerc, 12)
        End If

        ' Boss bullets
        For Each bb In bossBullets
            g.FillEllipse(Brushes.Orange, bb.Rect)
        Next

        ' HUD
        Using f As New Font("Consolas", 12, FontStyle.Bold)
            g.DrawString($"Score: {score}", f, Brushes.White, 8, 8)
            g.DrawString($"Lives: {playerLives}", f, Brushes.White, 8, 28)
        End Using
    End Sub

    Private Sub DrawZones(g As Graphics)
        Dim w = ClientSize.Width
        Dim h = ClientSize.Height

        For z = 0 To zones.Count - 1
            zones(z).Offset += 2
            If zones(z).Offset > 200 Then zones(z).Offset = 0

            Dim stripe = 80
            Dim off = zones(z).Offset
            Using br1 As New SolidBrush(Color.FromArgb(40, zones(z).ColorA)),
                  br2 As New SolidBrush(Color.FromArgb(30, zones(z).ColorB))

                For x = -160 To w Step 160
                    g.FillRectangle(br1, New Rectangle(x + off, 0, stripe, h))
                    g.FillRectangle(br2, New Rectangle(x + off + stripe, 0, stripe, h))
                Next
            End Using
        Next
    End Sub

    Private Sub DrawPlayer(g As Graphics)
        Dim cx = playerPos.X + playerSize.Width / 2
        Dim cy = playerPos.Y + playerSize.Height / 2

        Dim tip As New PointF(cx + 18, cy)
        Dim left As New PointF(cx - 8, cy - 10)
        Dim right As New PointF(cx - 8, cy + 10)

        Using br As New SolidBrush(Color.LightBlue)
            g.FillPolygon(br, {tip, left, right})
        End Using
    End Sub

    Private Sub CreateExplosion(center As PointF, Optional count As Integer = 20)
        Dim ex As New Explosion With {.Particles = New List(Of ExplosionParticle)}

        For i As Integer = 1 To count
            Dim angle = rand.NextDouble() * Math.PI * 2
            Dim speed = 1 + rand.NextDouble() * 4
            Dim dx = Math.Cos(angle) * speed
            Dim dy = Math.Sin(angle) * speed

            ex.Particles.Add(New ExplosionParticle With {
            .Pos = center,
            .Vel = New PointF(CSng(dx), CSng(dy)),
            .Life = 20 + rand.Next(20),
            .Color = Color.FromArgb(255, rand.Next(180, 255), rand.Next(80), 0)
        })
        Next

        explosions.Add(ex)
    End Sub

    Private Sub UpdateExplosions()
        For Each ex In explosions
            For Each p In ex.Particles
                If p.Life > 0 Then
                    p.Pos = New PointF(p.Pos.X + p.Vel.X, p.Pos.Y + p.Vel.Y)
                    p.Vel = New PointF(p.Vel.X * 0.95F, p.Vel.Y * 0.95F) ' slow down
                    p.Life -= 1
                End If
            Next
            If ex.Particles.TrueForAll(Function(pp) pp.Life <= 0) Then
                ex.Alive = False
            End If
        Next

        explosions.RemoveAll(Function(e) Not e.Alive)
    End Sub

    Private Sub SpawnBoss()
        boss = New Enemy(
        New PointF(ClientSize.Width + 150, ClientSize.Height / 2),
        New PointF(-1.5F, 0)
    )
        boss.HP = 500
    End Sub

    Private Sub UpdateBoss()
        If boss Is Nothing OrElse Not boss.Alive Then Return

        ' Move onto screen first
        If boss.Pos.X > ClientSize.Width - 200 Then
            boss.Pos = New PointF(boss.Pos.X - 2, boss.Pos.Y)
            Return
        End If

        ' Move up/down
        boss.Pos = New PointF(boss.Pos.X + boss.Vel.X, boss.Pos.Y + boss.Vel.Y)

        If boss.Pos.Y < 50 OrElse boss.Pos.Y > ClientSize.Height - 50 Then
            boss.Vel = New PointF(boss.Vel.X, -boss.Vel.Y)
        End If

        ' Fire every ~40 ticks
        If rand.Next(40) = 0 Then
            Dim dx = (playerPos.X - boss.Pos.X)
            Dim dy = (playerPos.Y - boss.Pos.Y)
            Dim len = Math.Sqrt(dx * dx + dy * dy)
            Dim dir As New PointF(dx / CSng(len), dy / CSng(len))
            bossBullets.Add(New Bullet(boss.Pos, dir))
        End If
    End Sub

    Private Sub UpdateBossBullets()
        For Each b In bossBullets
            b.Pos = New PointF(
            b.Pos.X + b.Dir.X * 6,
            b.Pos.Y + b.Dir.Y * 6
        )
            b.Rect = New RectangleF(b.Pos.X - 5, b.Pos.Y - 5, 10, 10)

            If b.Pos.X < -20 OrElse b.Pos.X > ClientSize.Width + 20 OrElse
           b.Pos.Y < -20 OrElse b.Pos.Y > ClientSize.Height + 20 Then
                b.Alive = False
            End If
        Next
        bossBullets.RemoveAll(Function(bb) Not bb.Alive)
    End Sub

    ' --------------------
    ' ENTRY POINT
    ' --------------------
    <STAThread>
    Public Shared Sub Main()
        Application.EnableVisualStyles()
        Application.Run(New Form1())
    End Sub

End Class
