﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Text

Module NativeOpenFileDialog

    Private Const MAX_PATH As Integer = 260

    <StructLayout(LayoutKind.Sequential, CharSet:=CharSet.Unicode)>
    Private Structure OPENFILENAME
        Public lStructSize As Integer
        Public hwndOwner As IntPtr
        Public hInstance As IntPtr
        Public lpstrFilter As IntPtr
        Public lpstrCustomFilter As IntPtr
        Public nMaxCustFilter As Integer
        Public nFilterIndex As Integer
        Public lpstrFile As IntPtr
        Public nMaxFile As Integer
        Public lpstrFileTitle As IntPtr
        Public nMaxFileTitle As Integer
        Public lpstrInitialDir As IntPtr
        Public lpstrTitle As IntPtr
        Public Flags As Integer
        Public nFileOffset As Short
        Public nFileExtension As Short
        Public lpstrDefExt As IntPtr
        Public lCustData As IntPtr
        Public lpfnHook As IntPtr
        Public lpTemplateName As IntPtr
        Public pvReserved As IntPtr
        Public dwReserved As Integer
        Public FlagsEx As Integer
    End Structure

    <DllImport("comdlg32.dll", CharSet:=CharSet.Unicode, EntryPoint:="GetOpenFileNameW", SetLastError:=True)>
    Private Function GetOpenFileName(ByRef ofn As OPENFILENAME) As Boolean
    End Function

    <DllImport("comdlg32.dll", SetLastError:=True)>
    Private Function CommDlgExtendedError() As Integer
    End Function

    ''' <summary>
    ''' Shows the native Open File dialog.
    ''' Returns selected file path or Nothing if cancelled.
    ''' </summary>
    Public Function ShowNativeOpenFileDialog(Optional ownerHandle As IntPtr = Nothing) As String
        Dim ofn As New OPENFILENAME()

        ' Allocate unmanaged memory for all strings and buffers
        Dim filterPtr As IntPtr = IntPtr.Zero
        Dim filePtr As IntPtr = IntPtr.Zero
        Dim titlePtr As IntPtr = IntPtr.Zero
        Dim initialDirPtr As IntPtr = IntPtr.Zero
        Dim titlePtrLocal As IntPtr = IntPtr.Zero

        Try
            ' Filter string (double-null terminated)
            filterPtr = Marshal.StringToHGlobalUni("All Files (*.*)" & ChrW(0) & "*.*" & ChrW(0) & ChrW(0))
            ' File buffer
            filePtr = Marshal.AllocHGlobal((MAX_PATH + 1) * 2)
            titlePtr = Marshal.AllocHGlobal((MAX_PATH + 1) * 2)
            ' Zero buffers
            For i As Integer = 0 To (MAX_PATH + 1) * 2 - 1
                Marshal.WriteByte(filePtr, i, 0)
                Marshal.WriteByte(titlePtr, i, 0)
            Next
            ' Initial directory and dialog title
            initialDirPtr = Marshal.StringToHGlobalUni(Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments))
            titlePtrLocal = Marshal.StringToHGlobalUni("Select a file")

            ' Fill OPENFILENAME struct
            ofn.lStructSize = Marshal.SizeOf(Of OPENFILENAME)()
            ofn.hwndOwner = ownerHandle
            ofn.hInstance = IntPtr.Zero
            ofn.lpstrFilter = filterPtr
            ofn.lpstrCustomFilter = IntPtr.Zero
            ofn.nMaxCustFilter = 0
            ofn.nFilterIndex = 1
            ofn.lpstrFile = filePtr
            ofn.nMaxFile = MAX_PATH
            ofn.lpstrFileTitle = titlePtr
            ofn.nMaxFileTitle = MAX_PATH
            ofn.lpstrInitialDir = initialDirPtr
            ofn.lpstrTitle = titlePtrLocal
            ofn.Flags = &H80000 Or &H4 Or &H8 ' OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST
            ofn.nFileOffset = 0
            ofn.nFileExtension = 0
            ofn.lpstrDefExt = IntPtr.Zero
            ofn.lCustData = IntPtr.Zero
            ofn.lpfnHook = IntPtr.Zero
            ofn.lpTemplateName = IntPtr.Zero
            ofn.pvReserved = IntPtr.Zero
            ofn.dwReserved = 0
            ofn.FlagsEx = 0

            ' Show dialog
            Dim ok As Boolean = GetOpenFileName(ofn)
            If ok Then
                Dim result As String = Marshal.PtrToStringUni(ofn.lpstrFile)
                If result IsNot Nothing Then
                    result = result.Split(ChrW(0))(0)
                End If
                Return result
            Else
                Dim err As Integer = CommDlgExtendedError()
                If err <> 0 Then
                    Throw New InvalidOperationException($"GetOpenFileName failed. CommDlgExtendedError = 0x{err:X}")
                End If
                Return Nothing ' Cancelled
            End If

        Finally
            ' Free all unmanaged memory safely
            If filterPtr <> IntPtr.Zero Then Marshal.FreeHGlobal(filterPtr)
            If filePtr <> IntPtr.Zero Then Marshal.FreeHGlobal(filePtr)
            If titlePtr <> IntPtr.Zero Then Marshal.FreeHGlobal(titlePtr)
            If initialDirPtr <> IntPtr.Zero Then Marshal.FreeHGlobal(initialDirPtr)
            If titlePtrLocal <> IntPtr.Zero Then Marshal.FreeHGlobal(titlePtrLocal)
        End Try
    End Function

End Module

