﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Text

Module NativeSaveFileDialog

    Private Const MAX_PATH As Integer = 260

    <StructLayout(LayoutKind.Sequential, CharSet:=CharSet.Unicode)>
    Private Structure OPENFILENAME
        Public lStructSize As Integer
        Public hwndOwner As IntPtr
        Public hInstance As IntPtr
        Public lpstrFilter As IntPtr
        Public lpstrCustomFilter As IntPtr
        Public nMaxCustFilter As Integer
        Public nFilterIndex As Integer
        Public lpstrFile As IntPtr
        Public nMaxFile As Integer
        Public lpstrFileTitle As IntPtr
        Public nMaxFileTitle As Integer
        Public lpstrInitialDir As IntPtr
        Public lpstrTitle As IntPtr
        Public Flags As Integer
        Public nFileOffset As Short
        Public nFileExtension As Short
        Public lpstrDefExt As IntPtr
        Public lCustData As IntPtr
        Public lpfnHook As IntPtr
        Public lpTemplateName As IntPtr
        Public pvReserved As IntPtr
        Public dwReserved As Integer
        Public FlagsEx As Integer
    End Structure

    <DllImport("comdlg32.dll", CharSet:=CharSet.Unicode, EntryPoint:="GetSaveFileNameW", SetLastError:=True)>
    Private Function GetSaveFileName(ByRef ofn As OPENFILENAME) As Boolean
    End Function

    <DllImport("comdlg32.dll", SetLastError:=True)>
    Private Function CommDlgExtendedError() As Integer
    End Function

    ''' <summary>
    ''' Shows the native Windows Save File dialog.
    ''' Returns selected file path, or Nothing if cancelled.
    ''' </summary>
    Public Function ShowNativeSaveFileDialog(Optional ownerHandle As IntPtr = Nothing,
                                             Optional defaultExtension As String = "txt",
                                             Optional dialogTitle As String = "Save As...",
                                             Optional defaultFileName As String = "",
                                             Optional filter As String = "All Files (*.*)" & vbNullChar & "*.*" & vbNullChar) As String
        Dim ofn As New OPENFILENAME()

        ' Allocate unmanaged memory
        Dim filterPtr As IntPtr = IntPtr.Zero
        Dim filePtr As IntPtr = IntPtr.Zero
        Dim titlePtr As IntPtr = IntPtr.Zero
        Dim initialDirPtr As IntPtr = IntPtr.Zero
        Dim defExtPtr As IntPtr = IntPtr.Zero

        Try
            ' Prepare double-null-terminated filter
            If Not filter.EndsWith(vbNullChar & vbNullChar) Then filter &= vbNullChar & vbNullChar
            filterPtr = Marshal.StringToHGlobalUni(filter)

            ' Allocate file buffer
            filePtr = Marshal.AllocHGlobal((MAX_PATH + 1) * 2)
            For i As Integer = 0 To (MAX_PATH + 1) * 2 - 1
                Marshal.WriteByte(filePtr, i, 0)
            Next

            ' If default filename provided, copy to buffer
            If Not String.IsNullOrEmpty(defaultFileName) Then
                Dim bytes() As Byte = Encoding.Unicode.GetBytes(defaultFileName)
                Marshal.Copy(bytes, 0, filePtr, bytes.Length)
            End If

            titlePtr = Marshal.StringToHGlobalUni(dialogTitle)
            initialDirPtr = Marshal.StringToHGlobalUni(Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments))
            defExtPtr = Marshal.StringToHGlobalUni(defaultExtension)

            ' Fill structure
            ofn.lStructSize = Marshal.SizeOf(Of OPENFILENAME)()
            ofn.hwndOwner = ownerHandle
            ofn.lpstrFilter = filterPtr
            ofn.lpstrFile = filePtr
            ofn.nMaxFile = MAX_PATH
            ofn.lpstrFileTitle = IntPtr.Zero
            ofn.nMaxFileTitle = 0
            ofn.lpstrInitialDir = initialDirPtr
            ofn.lpstrTitle = titlePtr
            ofn.Flags = &H80000 Or &H2 Or &H4 Or &H8 Or &H100000 ' OFN_EXPLORER | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_NOREADONLYRETURN
            ofn.lpstrDefExt = defExtPtr

            ' Show dialog
            Dim ok As Boolean = GetSaveFileName(ofn)
            If ok Then
                Dim result As String = Marshal.PtrToStringUni(ofn.lpstrFile)
                If result IsNot Nothing Then
                    result = result.Split(ChrW(0))(0)
                End If
                Return result
            Else
                Dim err = CommDlgExtendedError()
                If err <> 0 Then
                    Throw New InvalidOperationException($"GetSaveFileName failed. CommDlgExtendedError = 0x{err:X}")
                End If
                Return Nothing
            End If

        Finally
            ' Always free unmanaged memory
            If filterPtr <> IntPtr.Zero Then Marshal.FreeHGlobal(filterPtr)
            If filePtr <> IntPtr.Zero Then Marshal.FreeHGlobal(filePtr)
            If titlePtr <> IntPtr.Zero Then Marshal.FreeHGlobal(titlePtr)
            If initialDirPtr <> IntPtr.Zero Then Marshal.FreeHGlobal(initialDirPtr)
            If defExtPtr <> IntPtr.Zero Then Marshal.FreeHGlobal(defExtPtr)
        End Try
    End Function

End Module
