﻿Imports System
Imports System.Collections.Generic
Imports System.IO
Imports System.Text
Imports System.Windows.Forms
Imports System.Windows.Forms.DataVisualization.Charting

Public Class Form1
    Inherits Form

    Private lblInstructions As Label
    Private lstObjects As ListBox
    Private cmbMaterial As ComboBox
    Private txtName As TextBox
    Private btnAdd As Button
    Private btnRemove As Button
    Private btnStart As Button
    Private btnPause As Button
    Private nudSpeed As NumericUpDown
    Private chkSimulate As CheckBox
    Private chart As Chart
    Private txtLog As TextBox
    Private btnLogObservation As Button
    Private btnExport As Button
    Private lblElapsed As Label
    Private tip As ToolTip
    Private timerTick As Timer
    Private lblEnhance As Label
    Private btnTheme As Button
    Private pnlLeft As Panel

    Private isDarkTheme As Boolean = False

    Private Class ShipObject
        Public Property Id As Guid = Guid.NewGuid()
        Public Property Name As String
        Public Property Material As String
        Public Property StartSimHours As Double
        Public Property Series As Series
    End Class

    Private Class Observation
        Public Property TimeSimHours As Double
        Public Property RealTime As DateTime
        Public Property ObjectName As String
        Public Property Note As String
    End Class

    Private objects As New List(Of ShipObject)()
    Private observations As New List(Of Observation)()
    Private simHoursElapsed As Double = 0.0
    Private isRunning As Boolean = False

    Public Sub New()
        Text = "Shipwreck Experiment — Lab Dashboard"
        Width = 1200
        Height = 750
        StartPosition = FormStartPosition.CenterScreen
        Font = New Drawing.Font("Segoe UI", 11, Drawing.FontStyle.Regular)

        InitializeControls()
        InitializeChart()

        timerTick = New Timer() With {.Interval = 1000}
        AddHandler timerTick.Tick, AddressOf TimerTick_Tick
    End Sub

    Private Sub InitializeControls()
        pnlLeft = New Panel With {
            .Left = 10, .Top = 10, .Width = 380, .Height = 690,
            .BorderStyle = BorderStyle.FixedSingle
        }
        Controls.Add(pnlLeft)

        lblInstructions = New Label With {
            .Text = "Shipwreck Lab Controls",
            .Font = New Drawing.Font("Segoe UI", 13, Drawing.FontStyle.Bold),
            .Left = 10, .Top = 10, .AutoSize = True
        }
        pnlLeft.Controls.Add(lblInstructions)

        lstObjects = New ListBox With {.Left = 10, .Top = 50, .Width = 350, .Height = 200}
        pnlLeft.Controls.Add(lstObjects)

        Dim lblName As New Label With {.Text = "Object name:", .Left = 10, .Top = 260}
        pnlLeft.Controls.Add(lblName)
        txtName = New TextBox With {.Left = 10, .Top = 285, .Width = 250}
        pnlLeft.Controls.Add(txtName)

        Dim lblMat As New Label With {.Text = "Material:", .Left = 10, .Top = 320}
        pnlLeft.Controls.Add(lblMat)
        cmbMaterial = New ComboBox With {
            .Left = 10, .Top = 345, .Width = 250,
            .DropDownStyle = ComboBoxStyle.DropDownList
        }
        cmbMaterial.Items.AddRange(New String() {
            "Paper", "Plastic", "Metal - Iron", "Metal - Aluminum", "Metal - Stainless Steel"
        })
        cmbMaterial.SelectedIndex = 0
        pnlLeft.Controls.Add(cmbMaterial)

        btnAdd = New Button With {.Text = "Add", .Left = 270, .Top = 283, .Width = 90, .AutoSize = True}
        AddHandler btnAdd.Click, AddressOf BtnAdd_Click
        pnlLeft.Controls.Add(btnAdd)

        btnRemove = New Button With {.Text = "Remove", .Left = 270, .Top = 345, .Width = 90, .AutoSize = True}
        AddHandler btnRemove.Click, AddressOf BtnRemove_Click
        pnlLeft.Controls.Add(btnRemove)

        Dim lblSpeed As New Label With {.Text = "Simulation speed (sec per tick):", .Left = 10, .Top = 390}
        pnlLeft.Controls.Add(lblSpeed)
        nudSpeed = New NumericUpDown With {.Left = 10, .Top = 415, .Width = 120, .Minimum = 1, .Maximum = 3600, .Value = 1}
        pnlLeft.Controls.Add(nudSpeed)
        chkSimulate = New CheckBox With {.Text = "Accelerate simulation", .Left = 150, .Top = 415, .Checked = True}
        pnlLeft.Controls.Add(chkSimulate)

        btnStart = New Button With {.Text = "▶ Start", .Left = 10, .Top = 460, .Width = 160, .AutoSize = True}
        AddHandler btnStart.Click, AddressOf BtnStart_Click
        pnlLeft.Controls.Add(btnStart)
        btnPause = New Button With {.Text = "⏸ Pause", .Left = 190, .Top = 460, .Width = 160, .Enabled = False, .AutoSize = True}
        AddHandler btnPause.Click, AddressOf BtnPause_Click
        pnlLeft.Controls.Add(btnPause)

        lblElapsed = New Label With {.Text = "Simulated time: 0.0 h", .Left = 10, .Top = 505, .AutoSize = True}
        pnlLeft.Controls.Add(lblElapsed)

        Dim lblObs As New Label With {.Text = "Observation note:", .Left = 10, .Top = 535}
        pnlLeft.Controls.Add(lblObs)
        txtLog = New TextBox With {.Left = 10, .Top = 560, .Width = 350, .Height = 45, .Multiline = True}
        pnlLeft.Controls.Add(txtLog)
        btnLogObservation = New Button With {.Text = "Add Observation", .Left = 10, .Top = 615, .Width = 160, .AutoSize = True}
        AddHandler btnLogObservation.Click, AddressOf BtnLogObservation_Click
        pnlLeft.Controls.Add(btnLogObservation)
        btnExport = New Button With {.Text = "Export CSV", .Left = 190, .Top = 615, .Width = 160, .AutoSize = True}
        AddHandler btnExport.Click, AddressOf BtnExport_Click
        pnlLeft.Controls.Add(btnExport)

        btnTheme = New Button With {.Text = "🌗 Toggle Theme", .Left = 10, .Top = 650, .Width = 160, .AutoSize = True}
        AddHandler btnTheme.Click, AddressOf BtnTheme_Click
        pnlLeft.Controls.Add(btnTheme)

        tip = New ToolTip()
        tip.SetToolTip(txtName, "Enter a name for the object (e.g. 'Iron Nail').")
        tip.SetToolTip(cmbMaterial, "Select the material type for realistic corrosion.")
        tip.SetToolTip(btnStart, "Start simulation and track degradation.")
    End Sub

    Private Sub InitializeChart()
        chart = New Chart With {.Left = 400, .Top = 10, .Width = 760, .Height = 690}
        Dim area As New ChartArea("MainArea")

        area.AxisX.Title = "Simulated Time (hours)"
        area.AxisY.Title = "Degradation (%)"
        area.AxisX.TitleFont = New Drawing.Font("Segoe UI", 12, Drawing.FontStyle.Bold)
        area.AxisY.TitleFont = New Drawing.Font("Segoe UI", 12, Drawing.FontStyle.Bold)
        area.AxisX.LabelStyle.Font = New Drawing.Font("Segoe UI", 10)
        area.AxisY.LabelStyle.Font = New Drawing.Font("Segoe UI", 10)
        area.AxisX.MajorGrid.LineColor = Drawing.Color.LightGray
        area.AxisY.MajorGrid.LineColor = Drawing.Color.LightGray
        area.AxisX.Minimum = 0
        area.AxisY.Minimum = 0
        area.AxisY.Maximum = 100
        area.InnerPlotPosition = New ElementPosition(10, 5, 85, 85)

        chart.ChartAreas.Add(area)
        chart.Legends.Add(New Legend("Legend") With {
            .Docking = Docking.Bottom,
            .Font = New Drawing.Font("Segoe UI", 10)
        })

        AddHandler chart.MouseClick, AddressOf Chart_MouseClick
        Controls.Add(chart)

        tip.SetToolTip(chart, "Click any point to add annotations directly.")
    End Sub

    ' === THEME TOGGLE ===
    Private Sub BtnTheme_Click(sender As Object, e As EventArgs)
        isDarkTheme = Not isDarkTheme
        ApplyTheme()
    End Sub

    Private Sub ApplyTheme()
        If isDarkTheme Then
            BackColor = Drawing.Color.FromArgb(25, 25, 30)
            ForeColor = Drawing.Color.WhiteSmoke
            pnlLeft.BackColor = Drawing.Color.FromArgb(35, 35, 40)
            lstObjects.BackColor = Drawing.Color.FromArgb(45, 45, 50)
            lstObjects.ForeColor = Drawing.Color.White
            txtLog.BackColor = Drawing.Color.FromArgb(45, 45, 50)
            txtLog.ForeColor = Drawing.Color.White
            chart.BackColor = Drawing.Color.FromArgb(30, 30, 35)
            chart.ChartAreas(0).BackColor = Drawing.Color.FromArgb(20, 20, 25)
            chart.ChartAreas(0).AxisX.LabelStyle.ForeColor = Drawing.Color.WhiteSmoke
            chart.ChartAreas(0).AxisY.LabelStyle.ForeColor = Drawing.Color.WhiteSmoke
            chart.ChartAreas(0).AxisX.TitleForeColor = Drawing.Color.Gainsboro
            chart.ChartAreas(0).AxisY.TitleForeColor = Drawing.Color.Gainsboro
            btnTheme.Text = "☀ Light Mode"
        Else
            BackColor = Drawing.Color.WhiteSmoke
            ForeColor = Drawing.Color.Black
            pnlLeft.BackColor = Drawing.Color.White
            lstObjects.BackColor = Drawing.Color.White
            lstObjects.ForeColor = Drawing.Color.Black
            txtLog.BackColor = Drawing.Color.White
            txtLog.ForeColor = Drawing.Color.Black
            chart.BackColor = Drawing.Color.WhiteSmoke
            chart.ChartAreas(0).BackColor = Drawing.Color.White
            chart.ChartAreas(0).AxisX.LabelStyle.ForeColor = Drawing.Color.Black
            chart.ChartAreas(0).AxisY.LabelStyle.ForeColor = Drawing.Color.Black
            chart.ChartAreas(0).AxisX.TitleForeColor = Drawing.Color.Black
            chart.ChartAreas(0).AxisY.TitleForeColor = Drawing.Color.Black
            btnTheme.Text = "🌗 Dark Mode"
        End If
    End Sub

    ' === Simulation Logic ===
    Private Sub BtnAdd_Click(sender As Object, e As EventArgs)
        Dim name = txtName.Text.Trim()
        If name = "" Then
            MessageBox.Show("Enter a name for the object.")
            Return
        End If
        Dim mat = cmbMaterial.SelectedItem.ToString()
        Dim obj As New ShipObject With {.Name = name, .Material = mat, .StartSimHours = simHoursElapsed}
        objects.Add(obj)
        lstObjects.Items.Add($"{name} ({mat})")

        Dim s As New Series(obj.Name) With {
            .ChartType = SeriesChartType.Line,
            .ChartArea = "MainArea",
            .BorderWidth = 3,
            .Color = GetMaterialColor(mat),
            .Legend = "Legend",
            .MarkerStyle = MarkerStyle.Circle,
            .MarkerSize = 7
        }
        obj.Series = s
        chart.Series.Add(s)
        txtName.Clear()
    End Sub

    Private Function GetMaterialColor(material As String) As Drawing.Color
        Select Case material.ToLower()
            Case "paper" : Return Drawing.Color.SaddleBrown
            Case "plastic" : Return Drawing.Color.SeaGreen
            Case "metal - iron" : Return Drawing.Color.IndianRed
            Case "metal - aluminum" : Return Drawing.Color.Gray
            Case "metal - stainless steel" : Return Drawing.Color.SteelBlue
            Case Else : Return Drawing.Color.Black
        End Select
    End Function

    Private Sub BtnRemove_Click(sender As Object, e As EventArgs)
        Dim idx = lstObjects.SelectedIndex
        If idx < 0 Then Return
        chart.Series.Remove(objects(idx).Series)
        objects.RemoveAt(idx)
        lstObjects.Items.RemoveAt(idx)
    End Sub

    Private Sub BtnStart_Click(sender As Object, e As EventArgs)
        isRunning = True
        btnStart.Enabled = False
        btnPause.Enabled = True
        timerTick.Start()
    End Sub

    Private Sub BtnPause_Click(sender As Object, e As EventArgs)
        isRunning = False
        btnStart.Enabled = True
        btnPause.Enabled = False
        timerTick.Stop()
    End Sub

    Private Sub TimerTick_Tick(sender As Object, e As EventArgs)
        If Not isRunning Then Return
        Dim hoursThisTick As Double = If(chkSimulate.Checked, (timerTick.Interval / 1000.0) / nudSpeed.Value, (timerTick.Interval / 1000.0) / 3600.0)
        simHoursElapsed += hoursThisTick
        lblElapsed.Text = $"Simulated time: {simHoursElapsed:F1} h"

        For Each obj In objects
            Dim lifeSpan = MaterialLifespan(obj.Material)
            Dim age = simHoursElapsed - obj.StartSimHours
            Dim degradation = ComputeDegradation(obj.Material, age, lifeSpan)
            Dim dp As New DataPoint(simHoursElapsed, degradation)
            dp.ToolTip = $"{obj.Name} ({obj.Material})" & vbCrLf &
                         $"Time: {simHoursElapsed:F1} h" & vbCrLf &
                         $"Degradation: {degradation:F1}%"
            obj.Series.Points.Add(dp)
        Next
    End Sub

    Private Function MaterialLifespan(material As String) As Double
        Select Case material.ToLower()
            Case "paper" : Return 48.0
            Case "plastic" : Return 1000.0
            Case "metal - iron" : Return 300.0
            Case "metal - aluminum" : Return 800.0
            Case "metal - stainless steel" : Return 2000.0
            Case Else : Return 200.0
        End Select
    End Function

    Private Function ComputeDegradation(material As String, ageHours As Double, lifeSpan As Double) As Double
        Dim exponent As Double = 1.0
        Select Case material.ToLower()
            Case "paper" : exponent = 1.6
            Case "plastic" : exponent = 0.5
            Case "metal - iron" : exponent = 1.2
            Case "metal - aluminum" : exponent = 0.8
            Case "metal - stainless steel" : exponent = 0.5
        End Select
        Dim p = 100 * Math.Pow(ageHours / lifeSpan, exponent)
        Return Math.Min(100, Math.Max(0, p))
    End Function

    Private Sub Chart_MouseClick(sender As Object, e As MouseEventArgs)
        Dim hit As HitTestResult = chart.HitTest(e.X, e.Y)
        If hit.ChartElementType = ChartElementType.DataPoint Then
            Dim s As Series = hit.Series
            Dim dp As DataPoint = s.Points(hit.PointIndex)
            Dim note As String = InputBox("Add a note:", "Annotation", dp.Tag?.ToString())
            If note IsNot Nothing AndAlso note.Trim() <> "" Then
                dp.Tag = note
                dp.MarkerColor = Drawing.Color.Gold
                dp.Label = "★ " & note
                dp.LabelForeColor = Drawing.Color.DarkGoldenrod
                dp.LabelAngle = -20
                dp.LabelBackColor = Drawing.Color.FromArgb(200, Drawing.Color.LightYellow)
            End If
        End If
    End Sub

    Private Sub BtnLogObservation_Click(sender As Object, e As EventArgs)
        If txtLog.Text.Trim() = "" Then Return
        observations.Add(New Observation With {
            .RealTime = DateTime.Now,
            .TimeSimHours = simHoursElapsed,
            .ObjectName = If(lstObjects.SelectedIndex >= 0, objects(lstObjects.SelectedIndex).Name, ""),
            .Note = txtLog.Text
        })
        txtLog.Clear()
    End Sub

    Private Sub BtnExport_Click(sender As Object, e As EventArgs)
        Dim sfd As New SaveFileDialog With {.Filter = "CSV|*.csv", .FileName = "ShipwreckObservations.csv"}
        If sfd.ShowDialog() <> DialogResult.OK Then Return
        Using sw As New StreamWriter(sfd.FileName)
            sw.WriteLine("Time,SimHours,Object,Note")
            For Each o In observations
                sw.WriteLine($"{o.RealTime},{o.TimeSimHours:F2},{o.ObjectName},{o.Note}")
            Next
        End Using
        MessageBox.Show("Export complete.")
    End Sub

    <STAThread>
    Public Shared Sub Main()
        Application.EnableVisualStyles()
        Application.SetCompatibleTextRenderingDefault(False)
        Application.Run(New Form1())
    End Sub
End Class

