﻿Imports System.Media
Imports System.IO

Public Class Form1
    Private rand As New Random()
    Private treasures As New List(Of Point)
    Private foundTreasures As New List(Of Point)
    Private Const treasureRadius As Integer = 10
    Private Const hitRange As Integer = 20
    Private treasureCount As Integer = 5

    Private mapWindow As MapForm
    Private digSound As SoundPlayer
    Private foundSound As SoundPlayer

    Private gameTimer As New Timer()
    Private elapsedSeconds As Integer = 0
    Private score As Integer = 0

    Private lblTime As Label
    Private lblScore As Label
    Private lblLeft As Label

    Private highScoreFile As String = "highscores.txt"

    Private lblBonus As Label
    Private bonusFlashTimer As Timer
    Private bonusFlashVisible As Boolean = True
    Private criticalBonusThreshold As Integer = 1000
    Private bonusPulseTimer As Timer
    Private bonusPulseStep As Integer = 0
    Private pulseIncreasing As Boolean = True
    Private currentPulseInterval As Integer = 80 ' default smooth pulse

    ' --- Consistent UI font for Form1 ---
    Private uiFont As New Font("Segoe UI", 12, FontStyle.Regular)
    Private uiLabelFont As New Font("Segoe UI", 13, FontStyle.Bold) ' slightly larger for labels like Bonus

    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        Me.Text = "Treasure Hunt!"

        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()
        Me.MaximizeBox = False
        Me.FormBorderStyle = FormBorderStyle.FixedSingle

        'Apply consistent font to the whole form as fallback
        Me.Font = uiFont

        Me.DoubleBuffered = True
        Me.BackColor = Color.SandyBrown
        Me.Size = New Size(840, 560)

        ' Buttons
        Dim btnBury As New Button() With {.Text = "Bury Treasure", .Location = New Point(10, 10), .AutoSize = True}
        AddHandler btnBury.Click, AddressOf BuryTreasure
        Controls.Add(btnBury)

        Dim btnReset As New Button() With {.Text = "Reset", .Location = New Point(140, 10), .AutoSize = True}
        AddHandler btnReset.Click, AddressOf ResetGame
        Controls.Add(btnReset)

        Dim btnMap As New Button() With {.Text = "Show Map", .Location = New Point(230, 10), .AutoSize = True}
        AddHandler btnMap.Click, AddressOf ShowMap
        Controls.Add(btnMap)

        Dim btnScores As New Button() With {.Text = "View High Scores", .Location = New Point(340, 10), .AutoSize = True}
        AddHandler btnScores.Click, AddressOf ShowHighScores
        Controls.Add(btnScores)

        ' Labels
        lblTime = New Label() With {
    .Text = "Time: 0s",
    .Location = New Point(500, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblTime)

        lblScore = New Label() With {
    .Text = "Score: 0",
    .Location = New Point(600, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblScore)

        lblLeft = New Label() With {
    .Text = "Treasures: 0",
    .Location = New Point(720, 14),
    .AutoSize = True,
    .Font = uiFont
}
        Controls.Add(lblLeft)

        ' Bonus label uses the slightly larger bold label font
        lblBonus = New Label() With {
    .Text = "Bonus: 0",
    .Location = New Point(500, 34),
    .AutoSize = True,
    .Font = uiLabelFont
}
        lblBonus.ForeColor = Color.DarkGreen
        Controls.Add(lblBonus)

        gameTimer.Interval = 1000
        AddHandler gameTimer.Tick, AddressOf GameTimer_Tick

        bonusFlashTimer = New Timer() With {.Interval = 300} ' 0.3 seconds flash
        AddHandler bonusFlashTimer.Tick, AddressOf BonusFlash_Tick

        bonusPulseTimer = New Timer() With {.Interval = 80} ' update every 80ms for smoothness
        AddHandler bonusPulseTimer.Tick, AddressOf BonusPulse_Tick

        LoadSounds()
    End Sub

    Private Sub LoadSounds()
        Try
            If File.Exists("dig.wav") Then
                digSound = New SoundPlayer("dig.wav")
                digSound.Load()
            End If
            If File.Exists("found.wav") Then
                foundSound = New SoundPlayer("found.wav")
                foundSound.Load()
            End If
        Catch
            MessageBox.Show("Sound loading failed.")
        End Try
    End Sub

    Private Sub PlayDigSound()
        Try
            If digSound IsNot Nothing Then digSound.Play() Else SystemSounds.Beep.Play()
        Catch
        End Try
    End Sub

    Private Sub PlayFoundSound()
        Try
            If foundSound IsNot Nothing Then foundSound.Play() Else SystemSounds.Hand.Play()
        Catch
        End Try
    End Sub

    Private Sub BuryTreasure(sender As Object, e As EventArgs)
        treasures.Clear()
        foundTreasures.Clear()
        score = 0
        elapsedSeconds = 0
        UpdateScoreDisplay()
        UpdateTimeDisplay()
        UpdateBonusDisplay()
        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen

        lblBonus.Visible = True

        Dim areaWidth As Integer = Me.ClientSize.Width
        Dim areaHeight As Integer = Me.ClientSize.Height - 50
        For i As Integer = 1 To treasureCount
            treasures.Add(New Point(rand.Next(50, areaWidth - 50), rand.Next(60, areaHeight)))
        Next

        PlayFoundSound()
        MessageBox.Show("Treasures buried! Timer started. Now dig and find them!", "Ready")

        UpdateMap()
        UpdateLeftDisplay()
        Me.Invalidate()
        gameTimer.Start()
    End Sub

    Private Sub ResetGame(sender As Object, e As EventArgs)
        treasures.Clear()
        foundTreasures.Clear()
        score = 0
        elapsedSeconds = 0
        UpdateBonusDisplay()
        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen
        lblBonus.Visible = True
        gameTimer.Stop()
        UpdateScoreDisplay()
        UpdateTimeDisplay()
        UpdateLeftDisplay()
        Me.Invalidate()
        UpdateMap()
    End Sub

    Private Sub ShowMap(sender As Object, e As EventArgs)
        If mapWindow Is Nothing OrElse mapWindow.IsDisposed Then
            mapWindow = New MapForm()
        End If
        UpdateMap()
        mapWindow.Show()
        mapWindow.BringToFront()
    End Sub

    Private Sub UpdateMap()
        If mapWindow IsNot Nothing AndAlso Not mapWindow.IsDisposed Then
            mapWindow.UpdateMap(treasures.Except(foundTreasures).ToList())
        End If
    End Sub

    Private Sub GameTimer_Tick(sender As Object, e As EventArgs)
        elapsedSeconds += 1
        UpdateTimeDisplay()
        UpdateBonusDisplay()
    End Sub

    Private Sub UpdateTimeDisplay()
        lblTime.Text = $"Time: {elapsedSeconds}s"
    End Sub

    Private Sub UpdateScoreDisplay()
        lblScore.Text = $"Score: {score}"
    End Sub

    Private Sub UpdateLeftDisplay()
        lblLeft.Text = $"Treasures: {treasures.Count - foundTreasures.Count}"
    End Sub

    Private Sub Form1_MouseClick(sender As Object, e As MouseEventArgs) Handles Me.MouseClick
        If treasures.Count = 0 Then
            PlayDigSound()
            Return
        End If

        PlayDigSound()

        For Each t In treasures
            Dim dx As Integer = e.X - t.X
            Dim dy As Integer = e.Y - t.Y
            Dim distance As Double = Math.Sqrt(dx * dx + dy * dy)
            If distance < hitRange AndAlso Not foundTreasures.Contains(t) Then
                foundTreasures.Add(t)

                Dim basePoints As Integer = 1000
                Dim decay As Integer = 10
                Dim pts As Integer = basePoints - (elapsedSeconds * decay)
                If pts < 50 Then pts = 50
                score += pts

                PlayFoundSound()
                MessageBox.Show($"You found a treasure! +{pts} points", "Success")

                Me.Invalidate()
                UpdateMap()
                UpdateScoreDisplay()
                UpdateLeftDisplay()

                If foundTreasures.Count >= treasures.Count Then
                    EndGame()
                    ''gameTimer.Stop()
                    ''MessageBox.Show($"All treasures found! Time: {elapsedSeconds}s | Score: {score}", "Complete")
                    ''SaveHighScore()
                End If
                Exit For
            End If
        Next
    End Sub

    Private Sub SaveHighScore()
        Dim name As String = InputBox("Enter your name for the leaderboard:", "High Score")
        If String.IsNullOrWhiteSpace(name) Then name = "Anonymous"
        Try
            File.AppendAllLines(highScoreFile, {$"{name},{score},{elapsedSeconds},{DateTime.Now}"})
        Catch
            MessageBox.Show("Error saving score.")
        End Try
    End Sub

    Private Sub EndGame()
        gameTimer.Stop()

        Dim playerName As String = InputBox("You found all the treasures! Enter your name:", "Game Over", "Player")
        If String.IsNullOrWhiteSpace(playerName) Then playerName = "Anonymous"

        ' --- Base score accumulated during play ---
        Dim baseScore As Integer = score

        ' --- Nonlinear (exponential) time bonus ---
        ' Formula: timeBonus = maxBonus * e^(-elapsedSeconds / timeConstant)
        ' Fast finishes get large bonuses; slower times drop off smoothly.
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0 ' Larger = slower decay (adjust to taste)
        Dim timeBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        ' --- Total score ---
        Dim finalScore As Integer = baseScore + timeBonus

        ' --- Save high score ---
        Dim recordLine As String = $"{playerName},{finalScore},{elapsedSeconds},{DateTime.Now}"

        Try
            File.AppendAllText(highScoreFile, recordLine & Environment.NewLine)
        Catch ex As Exception
            MessageBox.Show("Could not save high score: " & ex.Message)
        End Try

        ' --- Compute player's rank for display ---
        Dim rank As Integer = 0
        Dim totalPlayers As Integer = 0

        Try
            If File.Exists(highScoreFile) Then
                Dim scores = File.ReadAllLines(highScoreFile)
                Dim scoreList = New List(Of (Player As String, Score As Integer))

                For Each line In scores
                    Dim parts = line.Split(","c)
                    If parts.Length >= 2 Then
                        Dim s As Integer
                        If Integer.TryParse(parts(1), s) Then
                            scoreList.Add((parts(0), s))
                        End If
                    End If
                Next

                totalPlayers = scoreList.Count
                Dim ordered = scoreList.OrderByDescending(Function(x) x.Score).ToList()
                For i As Integer = 0 To ordered.Count - 1
                    If String.Equals(ordered(i).Player, playerName, StringComparison.OrdinalIgnoreCase) AndAlso ordered(i).Score = finalScore Then
                        rank = i + 1
                        Exit For
                    End If
                Next
            End If
        Catch
            rank = 0
        End Try

        ' ✅ Auto-open leaderboard and highlight this player
        Dim hs As New HighScoresForm(highScoreFile, playerName, finalScore)
        hs.ShowDialog()

        ' --- Show detailed results ---
        Dim msg As String = $"Your Score: {finalScore}" & vbCrLf &
                        $"Base Score: {baseScore}" & vbCrLf &
                        $"Time Bonus: +{timeBonus}" & vbCrLf &
                        $"Time: {elapsedSeconds} seconds"

        If rank > 0 Then
            msg &= vbCrLf & $"🏅 You ranked #{rank} out of {totalPlayers}!"
        End If

        MessageBox.Show(msg, "Congratulations!", MessageBoxButtons.OK, MessageBoxIcon.Information)

        ResetGame(Nothing, Nothing)
    End Sub

    Private Sub ShowHighScores(sender As Object, e As EventArgs)
        Dim hs As New HighScoresForm(highScoreFile)
        hs.ShowDialog()
    End Sub

    Private Sub Form1_Paint(sender As Object, e As PaintEventArgs) Handles Me.Paint
        Dim g As Graphics = e.Graphics
        For Each t In foundTreasures
            g.FillEllipse(Brushes.Gold, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
            g.DrawEllipse(Pens.Black, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
        Next
        g.DrawRectangle(Pens.Brown, 0, 50, Me.ClientSize.Width - 1, Me.ClientSize.Height - 60)
    End Sub

    Private Sub UpdateBonusDisplay()
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0
        Dim currentBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        lblBonus.Text = $"Bonus: {currentBonus}"

        ' --- Color + pulse trigger ---
        If currentBonus < criticalBonusThreshold Then
            If Not bonusPulseTimer.Enabled Then
                pulseIncreasing = True
                bonusPulseStep = 150
                bonusPulseTimer.Start()
            End If
        Else
            lblBonus.ForeColor = Color.DarkGreen
            If bonusPulseTimer.Enabled Then bonusPulseTimer.Stop()
        End If
    End Sub

    Private Sub BonusFlash_Tick(sender As Object, e As EventArgs)
        ' Toggle the label’s visibility for blinking effect
        bonusFlashVisible = Not bonusFlashVisible
        lblBonus.Visible = bonusFlashVisible
    End Sub

    Private Sub BonusPulse_Tick(sender As Object, e As EventArgs)
        ' Pulse intensity oscillates between 100–255
        If pulseIncreasing Then
            bonusPulseStep += 15
            If bonusPulseStep >= 255 Then
                bonusPulseStep = 255
                pulseIncreasing = False
            End If
        Else
            bonusPulseStep -= 15
            If bonusPulseStep <= 100 Then
                bonusPulseStep = 100
                pulseIncreasing = True
            End If
        End If

        ' Apply animated red tone (smooth fade between bright and dim)
        lblBonus.ForeColor = Color.FromArgb(bonusPulseStep, 255, 60, 60)
    End Sub

End Class

