using System;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Web;
using System.Web.Util;
using Microsoft.ContentManagement.Publishing; 
using Utilities;

namespace cmsconvlog
{
	/// <summary>
	//
	//CMSConvLog
	//
	//	Description:
	//
	//	CMSConvLog parses an IIS log file and converts
	//	- /NR/rdonlyres/<GUID> into the Resource.Path equivalent for ResourceGalleryItems
	//	- /NR/rdonlyres/<GUID>/image.jpg into the Posting.Path/image.jpg for local attachments
	//	- /NR/exeres/<GUID> into the Channel.Path or Posting.Path for ChannelItems.
	//
	//	CMSConvLog saves the new log file under the same directory as
	//	the input log, with a _CmsConvLog suffix. If the file already
	//	exists, new lines are appened.
	//
	//	CMSConvLog requires the logged on user to have proper access to the
	//	CMS repository in order to use GetByGUID() function in Upublished mode.
	//
	//	CMSConvLog has been tested with:
	//	- Content Management Server 2002 with .Net Framework 1.0
	//	- Content Management Server 2002 SP1/SP1a with .Net Framework 1.0 or 1.1
	//	- NCSA or W3C log file formats.
	//
	//	Usage
	//
	//	cmsconvlog.exe <log_file> [/roots<+|-> /progress<+|->]
	//
	//	<log_file>         Filename of the log file to convert.
	//
	//	/roots[+|-]        Short form /r (ON by default)
	//	Toggles root nodes to be printed as part of the path
	//	+       /Channels/WoodgroveNet/Default
	//	-       /WoodgroveNet/Default
	//
	//	/progress[+|-]     Short form /p (ON by default)
	//	Display cycles while program is running
	//
	//	/errout[+|-]       Short form /e (OFF by default)
	//	Display log of GUID not matched in the CMS repository
	//
	//	Examples:
	//
	//	CMSConvLog.exe c:\windows\system32\logfiles\w3svc1\ex030101.log
	//	CMSConvLog.exe ..\ex030101.log /roots- /progress-
	//	CMSConvLog.exe ..\ex030101.log /r- /p-	
	//
	/// </summary>

	class WCArguments
	{
		[Utilities.DefaultCommandLineArgument(Utilities.CommandLineArgumentType.Required)]
		public string logfile = "";
		public bool roots = true;
		public bool progress = true;
		public bool errout = false;
	}

	class convlog
	{
		static CmsApplicationContext g_cmsContext = new CmsApplicationContext(); 
		static int g_rdonlyresReplaced = 0;
		static int g_exeresReplaced = 0;
		static int g_rdonlyresFailed = 0;
		static int g_UrdonlyresFailed = 0;
		static int g_exeresFailed = 0;
		static int g_UexeresFailed = 0;
		static string version = "\r\nCMSConvLog Version 2.0.1 (25-Feb-2004)\r\n";
		static bool g_PrintRootObjets = true;
		static bool g_errout = false;
		static StringBuilder Err = new StringBuilder();
		
		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		/// <param name="args">Arguments as described in the help</param>
		/// <returns>0 if success, 1 if error</returns>
		[STAThread]
		static int Main(string[] args)
		{
			Console.WriteLine(version); 
			WCArguments parsedArgs = new WCArguments();
	
			if (!Utilities.Utility.ParseCommandLineArguments(args, parsedArgs)) 
			{
				Usage();
				return 1;
			}
			else
			{
				g_errout = parsedArgs.errout;
				g_PrintRootObjets = parsedArgs.roots;
				Utility.g_ShowStars = parsedArgs.progress;
				return ProcessLogFile(parsedArgs.logfile);
			}
		}
		
		/// <summary>
		/// ProcessLogFile handles:
		///		- log file opening 
		///		- calls to Transform() to convert each line one by one
		///		- new log file creation and writing
		/// </summary>
		/// <param name="lg">Logfile to convert</param>
		/// <returns>0 if success, 1 if error</returns>
		static int ProcessLogFile(string lg)
		{
			Utility.OutputDebug("ProcessLogFile(), Start");
			string line, newlg;
			int i = 1;
			
			// Creating new logfile name
			try 
			{
				int dot = lg.LastIndexOf(".");
				newlg = lg.Insert(dot,"_CmsConvLog");
			}
			catch (Exception e)
			{
				Console.Write(" Warning: the log filename passed does not have an extension (Code " + e.GetHashCode() + ").\r\n");
				newlg = lg + "_CmsConvLog";
			}
			
			// Authentication
			try 
			{
				Utility.OutputDebug("ProcessLogFile(), CmsApplicationContext.AuthenticateAsCurrentUser()");
				g_cmsContext.AuthenticateAsCurrentUser(PublishingMode.Unpublished); 
				Utility.OutputDebug("ProcessLogFile(), CmsApplicationContext.AuthenticateAsCurrentUser() done.");
			}
			catch(Exception e)
			{
				Console.WriteLine(" Authentication Failure");
				Console.WriteLine(" " + e.Message);
				Utility.OutputDebug("ProcesslogFile(), Authentication Failure");
				Utility.OutputDebug(Utility.NewLine + " " + e.GetBaseException() + Utility.NewLine);
				return 1;
			}
			
			ResourceGallery g_ResourceGallery = g_cmsContext.Searches.GetByPath("/Resources") as ResourceGallery;

			// Process
			try
			{
				StreamReader sr = new StreamReader(lg);
				StreamWriter sw = new StreamWriter(newlg, true, System.Text.Encoding.ASCII);
				Console.Write(" Processing logfile : " + lg + "\r\n");
				Console.Write(" Writing new logfile: " + newlg + "\r\n");
				
				if ( g_PrintRootObjets ) Console.Write(" Root objects will be printed in path.\r\n");
				else  Console.Write(" /NoRoots option detected: Root objects will NOT be printed in path.\r\n");
		
				ThreadStart worker = new ThreadStart ( Utility.ShowStars );
				Thread t = new Thread (worker);
				if (Utility.g_ShowStars) t.Start();
			
				while ((line = sr.ReadLine()) != null) 
				{
					try 
					{ 
						sw.WriteLine(Transform(line));
					}
					catch(Exception e)
					{
						Console.WriteLine(Utility.NewLine + " An error occured '" + e.Message + "')");
						Console.WriteLine(" Processing log line number: " + i + Utility.NewLine);
						Utility.OutputDebug(" ProcesslogFile(), Error calling Transform()" + i + Utility.NewLine);
						Utility.OutputDebug(" ProcesslogFile(), Processing log line number: " + i + Utility.NewLine);
						Utility.OutputDebug(Utility.NewLine + " " + e.GetBaseException() + Utility.NewLine);
						i++;
					}
					i++;
				}
				sr.Close();
				sw.Close();
					
				// Stop stars thread
				if (Utility.g_ShowStars) 
				{
					Utility.g_ShowStars = false;
					Console.Write("\r  ");
					t.Abort();
				}

				// Output Results
				Console.WriteLine("\r\n Successfully processed " + i + " lines of IIS logs." + Utility.NewLine);
				if ( g_rdonlyresReplaced > 0 ) 
				{
					Console.WriteLine(" /NR/rdonlyres:"); 
					Console.WriteLine("  - " + g_rdonlyresReplaced + " GUID converted");
				}
				if ( g_rdonlyresFailed > 0 ) 
				{
					Console.WriteLine("  - " + g_rdonlyresFailed + " GUID unresolved");
					Console.WriteLine("  - " + g_UrdonlyresFailed + " unique GUID unresolved");
				}
				if ( g_exeresReplaced > 0 ) 
				{
					Console.WriteLine(Utility.NewLine + " /NR/exeres:"); 
					Console.WriteLine("  - " + g_exeresReplaced + " GUID converted");
				}
				if ( g_exeresFailed > 0 ) 
				{
					Console.WriteLine("  - " + g_exeresFailed + " GUID unresolved");
					Console.WriteLine("  - " + g_UexeresFailed + " unique GUID unresolved");
				}
								
				if ( (Err.Length > 0) & (g_errout == true) ) 
				{
					Console.WriteLine(Utility.NewLine + "  List of unique GUID unresolved:");
					Console.WriteLine(Err + Utility.NewLine );
				}
				return 0;
			}
			catch (Exception e) 
			{
				Console.WriteLine(Utility.NewLine + " An error occured '" + e.Message + "')");
				Console.WriteLine(" Processing log line number: " + i + Utility.NewLine);
				Utility.OutputDebug(" ProcesslogFile(), General error " + i + Utility.NewLine);
				Utility.OutputDebug(" ProcesslogFile(), Processing log line number: " + i + Utility.NewLine);
				Utility.OutputDebug(Utility.NewLine + " " + e.GetBaseException() + Utility.NewLine);
				return 1;
			}
		}

		/// <summary>
		/// Transform() searches for /NR patterns in a string and converts
		/// the associated GUID to a corresponding node in the CMS repository.
		/// </summary>
		/// <param name="line">String or line of log to convert</param>
		/// <returns>log line with GUID transformed</returns>
		static string Transform(string line)
		{
			bool fail = false;
			string lineret = line;
			string tmp;
			int iPos = 0;
			Match mat;

			Regex rdonlyres = new Regex(@"(/NR/rdonlyres/\w{8}-\w{4}-\w{4}-\w{4}-\w{12}[^ ]*)");
			Regex exeres = new Regex(@"(/NR/exeres/\w{8}-\w{4}-\w{4}-\w{4}-\w{12}[^ ]*)");
			Regex rd2exe = new Regex(@"(/NR/rdonlyres/\w{8}-\w{4}-\w{4}-\w{4}-\w{12}/\d/*)");
			Regex guid = new Regex(@"(\w{8}-\w{4}-\w{4}-\w{4}-\w{12})");
			
			mat = rdonlyres.Match(lineret);
			if ( mat.Success)
			{
				iPos = lineret.IndexOf(@"/NR/rdonlyres/");
				tmp = lineret.Substring(iPos+14, 36);    
				Utility.OutputDebug("Transform(), Identified rdonlyres node: " + tmp);
			
				try
				{
					ResourceGalleryItem pResourceGalleryItem = g_cmsContext.Searches.GetByGuid("{"+tmp+"}") as ResourceGalleryItem;
					if (pResourceGalleryItem  != null) 
					{
						if ( g_PrintRootObjets ) 
							lineret = rdonlyres.Replace(lineret, System.Web.HttpUtility.UrlEncode(pResourceGalleryItem.Path).Replace("%2f","/"));
						else 
							lineret = rdonlyres.Replace(lineret, System.Web.HttpUtility.UrlEncode(pResourceGalleryItem .Path.Substring(10,pResourceGalleryItem.Path.Length-10)).Replace("%2f","/")); 
						g_rdonlyresReplaced++;
					}
					else 
					{
						ChannelItem pChannelItem = g_cmsContext.Searches.GetByGuid("{"+tmp+"}") as ChannelItem;
						if (pChannelItem != null) 
						{
							if ( g_PrintRootObjets ) 
								lineret = rd2exe.Replace(lineret, HttpUtility.UrlEncode(pChannelItem.Path).Replace("%2f","/"));
							else 
								lineret = rd2exe.Replace(lineret, HttpUtility.UrlEncode(pChannelItem.Path.Substring(9,pChannelItem.Path.Length-9)).Replace("%2f","/"));
							g_rdonlyresReplaced++;
						}
						else 
						{
							fail = true;	
						}	
					}
				}					
				catch (Exception e)
				{
					Utility.OutputDebug(" Transform(), rdonlyres failed: " + tmp + "\r\n");
					Utility.OutputDebug(" Transform(), line: " + line + Utility.NewLine);
					Utility.OutputDebug(Utility.NewLine + " " + e.GetBaseException() + Utility.NewLine);
					fail = true;
				}
				finally
				{
					if (fail)
					{
						if ( Err.ToString().IndexOf(tmp) == -1 )
						{
							Err.Append("   . ResourceGalleryItem: " + tmp + "\r\n");
							g_rdonlyresFailed++;
							g_UrdonlyresFailed++;
						}
						else
							g_rdonlyresFailed++;
					}
				}
				mat = null;
				fail = false;
			}

			mat = exeres.Match(lineret);
			if ( mat.Success )
			{
				iPos = lineret.IndexOf(@"/NR/exeres/");
				tmp = lineret.Substring(iPos+11, 36);
				mat = guid.Match(tmp);
				if ( ! mat.Success )
				{
					iPos = lineret.IndexOf(@"/NR/exeres/", iPos + 1 );
					tmp = lineret.Substring(iPos+11, 36);
				}
				Utility.OutputDebug("Transform(), Identified exeres node: " + tmp);

				try 
				{
					ChannelItem pChannelItem = g_cmsContext.Searches.GetByGuid("{"+tmp+"}") as ChannelItem;
					if (pChannelItem != null) 
					{
						if ( g_PrintRootObjets ) 
							lineret = exeres.Replace(lineret, HttpUtility.UrlEncode(pChannelItem.Path).Replace("%2f","/"));
						else 
							lineret = exeres.Replace(lineret, HttpUtility.UrlEncode(pChannelItem.Path.Substring(9,pChannelItem.Path.Length-9)).Replace("%2f","/"));
						g_exeresReplaced++;
					}
					else 
						fail = true;
				}
				catch (Exception)
				{
					try 
					{
						TemplateGalleryItem pTemplateGalleryItem = g_cmsContext.Searches.GetByGuid("{"+tmp+"}") as TemplateGalleryItem;
						if (pTemplateGalleryItem != null) 
						{
							if ( g_PrintRootObjets ) 
								lineret = exeres.Replace(lineret, HttpUtility.UrlEncode(pTemplateGalleryItem.Path).Replace("%2f","/"));
							else 
								lineret = exeres.Replace(lineret, HttpUtility.UrlEncode(pTemplateGalleryItem.Path.Substring(9,pTemplateGalleryItem.Path.Length-9)).Replace("%2f","/"));
							g_exeresReplaced++;
						}
						else
							fail = true;
					}
					catch (Exception e)
					{
						Utility.OutputDebug(" Transform(), ChannelItem/TemplateGalleryItem failed: " + tmp + Utility.NewLine);
						Utility.OutputDebug(" line: " + line + Utility.NewLine);
						Utility.OutputDebug(Utility.NewLine + " " + e.GetBaseException() + Utility.NewLine);
						fail = true;
					}
				}
				finally
				{
					if (fail)
					{
						if ( Err.ToString().IndexOf(tmp) == -1 )
						{
							Err.Append("   . ChannelItem: " + tmp + "\r\n");
							g_exeresFailed++;
							g_UexeresFailed++;
						}
						else
							g_exeresFailed++;
					}
				}
				fail = false;
				mat = null;
			}
			return lineret;
		}

		/// <summary>
		/// Usage() displays help for CMSConvLog program.
		/// </summary>
		static void Usage()
		{
			StringBuilder Output = new StringBuilder();

			Output.Append(Utility.NewLine);
			Output.Append(" Description:\r\n\r\n");
			Output.Append("   CMSConvLog parses an IIS log file and converts \r\n");
			Output.Append("    - /NR/rdonlyres/<GUID> into the Resource.Path equivalent for ResourceGalleryItems\r\n");
			Output.Append("    - /NR/rdonlyres/<GUID>/image.jpg into the Posting.Path/image.jpg for local attachments\r\n");
			Output.Append("    - /NR/exeres/<GUID> into the Channel.Path or Posting.Path for ChannelItems\r\n\r\n");
			Output.Append("   CMSConvLog saves the new log file under the same directory as\r\n");
			Output.Append("   the input log, with a _CmsConvLog suffix. If the file already\r\n");
			Output.Append("   exists, new lines are appened.\r\n\r\n");
			Output.Append("   CMSConvLog requires the logged on user to have proper access to the \r\n");
			Output.Append("   CMS repository in order to use GetByGUID() function in Upublished mode. \r\n\r\n");
			Output.Append("   CMSConvLog has been tested with:\r\n");
			Output.Append("    - Content Management Server 2002 with .Net Framework 1.0\r\n");
			Output.Append("    - Content Management Server 2002 SP1/SP1a with .Net Framework 1.0 or 1.1\r\n");
			Output.Append("    - NCSA or W3C log file formats.\r\n\r\n");
			Output.Append(" Usage" + Utility.NewLine+ Utility.NewLine);
			Output.Append("   cmsconvlog.exe <log_file> [/roots<+|-> /progress<+|->]\r\n\r\n");
			Output.Append("   <log_file>         Filename of the log file to convert.\r\n\r\n");
			Output.Append("   /roots[+|-]        Short form /r (ON by default)\r\n");
			Output.Append("                      Toggles root nodes to be printed as part of the path\r\n");
			Output.Append("                        +\t/Channels/WoodgroveNet/Default\r\n");
			Output.Append("                        -\t/WoodgroveNet/Default\r\n\r\n");
			Output.Append("   /progress[+|-]     Short form /p (ON by default)\r\n");
			Output.Append("                      Display cycles while program is running \r\n\r\n");
			Output.Append("   /errout[+|-]       Short form /e (OFF by default)\r\n");
			Output.Append("                      Display log of GUID not matched in the CMS repository\r\n\r\n");
			Output.Append(" Examples:\r\n\r\n");
			Output.Append("   CMSConvLog.exe " + @"c:\windows\system32\logfiles\w3svc1\ex030101.log" + "\r\n");
			Output.Append("   CMSConvLog.exe " + @"..\ex030101.log /roots- /progress-" + " \r\n");
			Output.Append("   CMSConvLog.exe " + @"..\ex030101.log /r- /p-" + " \r\n");

			Console.Write(Output);
		}
	}
}
