///
///  This source code is freeware and is provided on an "as is" basis without warranties of any kind
///  All risk is with the end user.
///  
///  Created by: Murray Thompson
///  
///  Last Updated: April 30, 2005
///  

using System;
using Microsoft.ContentManagement.Publishing;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using System.Text.RegularExpressions;

namespace CommonCode.Helpers
{
	/// <summary>
	/// Set of methods to assist with CMS objects
	/// </summary>
	namespace Enum
	{
		public enum TextSearchType 
			//used for defining type of match to perform for a text-based value
			//	None- no specified text to search against
			//	Prefix- values prefixed with specified text
			//	Suffix- values suffixed with specified text
			//	SuffixBeforeExt- values suffixed with specified text (before file extension)
			//	Any- values contain specified text at any position
		{
			None, Prefix, Suffix, SuffixBeforeExt, Any
		}

		public enum ResourceTextPropertyType
			//type of text-based resource gallery property to search for specified text
		{
			Name, DisplayName, Description
		}

		public enum CaseSensitive
		{
			No, Yes
		}
		public enum SortDirection
		{
			None, Asc, Desc
		}
	}

	
	public class CmsObjectHelper
	{
		public CmsObjectHelper()
		{
		}

		#region IsPosting()
		private static bool IsPosting(HierarchyItem pItem)
		{
			Posting i;
			try 
			{
				i = (Posting) pItem;
			}
			catch
			{
				return false;
			}

			return true;
		}
		#endregion

		#region IsChannel()
		private static bool IsChannel(HierarchyItem pItem)
		{
			Channel i;
			try 
			{
				i = (Channel) pItem;
			}
			catch
			{
				return false;
			}

			return true;
		}
		#endregion
		
		#region IsResource()
		private static bool IsResource(HierarchyItem pItem)
		{
			Resource i;
			try 
			{
				i = (Resource) pItem;
			}
			catch
			{
				return false;
			}

			return true;
		}
		#endregion
		
		#region IsResourceGallery()
		private static bool IsResourceGallery(HierarchyItem pItem)
		{
			ResourceGallery i;
			try 
			{
				i = (ResourceGallery) pItem;
			}
			catch
			{
				return false;
			}

			return true;
		}
		#endregion
	
		public static string GetItemDisplayPath(string strGUID)
			//This method will check if the string given matches the GUID of
			//a CMS object: channel, posting, resource gallery, or resource
			//If it matches one of those items, it returns the path of the item in CMS,
			//first stripping off the root CMS gallery or channel from the beginning of the path.
			//If no matching item's GUID is found by through the string value,
			//the original string is returned
		{
				
			string textValue="";

			try
			{
				CmsHttpContext cmsHttpContext= CmsHttpContext.Current;
				HierarchyItem item= cmsHttpContext.Searches.GetByGuid(strGUID);

				string strC= cmsHttpContext.RootChannel.Path;
				string strRG= cmsHttpContext.RootResourceGallery.Path;

				if (IsChannel(item))
				{
					Channel c= (Channel)item;
					textValue= c.Path.Substring(strC.Length);
				}
				if (IsPosting(item))
				{
					Posting p= (Posting)item;
					textValue= p.Path.Substring(strC.Length);
				}
				if (IsResourceGallery(item))
				{
					ResourceGallery rg= (ResourceGallery)item;
					textValue =rg.Path;
				}
				if (IsResource(item))
				{
					Resource r= (Resource)item;
					textValue= r.Path.Substring(strRG.Length);
				}
			
			}
			catch
			{
				textValue=strGUID;
			}

				
			return textValue;

		}


		public static bool CheckForContent(Placeholder ph)
			//this method will check to see if any non-whitespace content exists in a HTML placeholder object
			//if not, it will hide the given WebControl associated with the placeholder
		{
			String phContent= ph.Datasource.RawContent;
			string str_regex_tags= "<(?!(?>\\s*img))[^>]*>"; //expression for finding any HTML tag, except <img>
			string str_regex_wspace= "\\s|&nbsp;"; //expression for finding any whitespace characters (including html non-breaking space)

			//strip out all tag elements
			Regex re_tags= new Regex(str_regex_tags,RegexOptions.IgnoreCase | RegexOptions.Compiled);
			phContent=re_tags.Replace(phContent,"");

			//strip out all whitespace elements
			Regex re_wspace= new Regex(str_regex_wspace, RegexOptions.Compiled);
			phContent=re_wspace.Replace(phContent,"");

			//if any text remaining, keep object visible
			if (phContent.Length>0)
			{ 
				return true; 
			}
			else
			{ 
				return false; 
			}	
		}

        
		#region ShowObject
		public static void ShowObject(WebControl obj, bool showobj)
		{
			obj.Visible=showobj;
		}

		public static void ShowObject(HtmlControl obj, bool showobj)
		{
			obj.Visible=showobj;
		}
		#endregion

		public static bool TextPropertyMatches(Resource res, Enum.ResourceTextPropertyType propType, string searchValue, Enum.TextSearchType searchType, Enum.CaseSensitive caseSens)
		//this method will determine if a resource's text-based resource property matches a value being sought
		//	res: resource gallery item being examined
		//	propType: type of property to examine
		//  caseSens: indicates if the search should be case sensitive or not
		//	fileName
		
		{
			string resPropVal= "";
			if (caseSens== Enum.CaseSensitive.No)
			{
				searchValue=searchValue.ToLower();
			}

			//set which type of resource property to examine based on propType value
			switch (propType)
			{
				case Enum.ResourceTextPropertyType.Name:
					resPropVal=res.Name;
					break;
				case Enum.ResourceTextPropertyType.DisplayName:
					resPropVal=res.DisplayName;
					break;
				case Enum.ResourceTextPropertyType.Description:
					resPropVal=res.Description;
					break;
			}
			
			if (caseSens== Enum.CaseSensitive.No)
			{
				resPropVal=resPropVal.ToLower();
			}

			//determine if value matches search criteria, using prefix/suffix options (fileNameSearchType and fileNameSearchValue)
			switch (searchType)
			{
				case Enum.TextSearchType.None: 
					//no prefix or suffix- use name as-is
					return true;
				case Enum.TextSearchType.Prefix:
					//prefix: if resource property value begins with prefix text
					if (resPropVal.StartsWith(searchValue))
					{
						return true;
					}
					else
					{
						return false;
					}
				case Enum.TextSearchType.Suffix:
					//extension suffix: if resource property value ends with suffix text (after the file extension part)
					if (resPropVal.EndsWith(searchValue))
					{
						return true;
					}
					else
					{
						return false;
					}
				case Enum.TextSearchType.SuffixBeforeExt:
					//suffix: if resource property value (before the file extension part) ends with suffix text (before the file extension)
					//	If the resource name is being examined, the suffix will be applied between the main name and file extension
					//	Note: if the file extension in the resource name was changed manually through 
					//			the Resource Manager, the stored file extension value is not updated in MCMS
					//			and a match will not be found
					if (propType==Enum.ResourceTextPropertyType.Name)
					{
						string thExt="."+res.FileExtension.ToLower();
						resPropVal=resPropVal.ToLower().Replace(thExt, "");
					}
					if (resPropVal.EndsWith(searchValue))
					{
						return true;
					}
					else
					{
						return false;
					}
				case Enum.TextSearchType.Any:
					//match any part of name: if resource property value contains search text
					if (resPropVal.IndexOf(searchValue)>0)
					{
						return true;
					}
					else
					{
						return false;
					}

			}		
			return false;
		}


	}
}
