using System;
using System.Collections;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Web;
using System.Web.SessionState;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using Microsoft.ContentManagement.Publishing;

namespace ContosoNet.Console
{
	/// <summary>
	/// Summary description for ChannelMan.
	/// </summary>
	public class ChannelMan : System.Web.UI.Page
	{
		protected System.Web.UI.WebControls.TextBox NewDisplayNameTextBox;
		protected System.Web.UI.WebControls.Button AddChannelButton;
		protected System.Web.UI.WebControls.Label l;
		protected System.Web.UI.WebControls.Label NewNameLabel;
		protected System.Web.UI.WebControls.Button MoveUpButton;
		protected System.Web.UI.WebControls.Button MoveDownButton;
		protected System.Web.UI.WebControls.Label ErrorLabel;
		protected System.Web.UI.WebControls.Button DeleteButton;
		protected System.Web.UI.WebControls.Label ExistingSectionsLabel;
		protected System.Web.UI.WebControls.Label ParentChannelNameLabel;
		protected System.Web.UI.WebControls.TextBox NewNameTextBox;
		protected System.Web.UI.WebControls.Label AddChannelLabel;
		protected System.Web.UI.WebControls.ListBox ChannelListBox;

		private CmsApplicationContext context;
		private Channel currentChannel;
		protected System.Web.UI.WebControls.Label NewPostingAdviceLabel;
		protected System.Web.UI.WebControls.Button HideShowButton;
		private string currentChannelID;

		private void Page_Load(object sender, System.EventArgs e)
		{
			ClearError();

			// Get context and current channel for use throughout this class...
			// Note that since this isn't a posting, we can't use the CmsHttpContext,
			// we need to create our own application context in the "Update" publishing
			// mode.
			context = new CmsApplicationContext();
			System.Security.Principal.WindowsIdentity id 
				= (System.Security.Principal.WindowsIdentity) HttpContext.Current.User.Identity;
			if (id != null)
			{
				context.AuthenticateUsingUserHandle( id.Token, PublishingMode.Update);
			}
			
			currentChannelID = Request.QueryString["channelID"];
			currentChannel = (Channel) context.Searches.GetByGuid (currentChannelID);

			// If we're starting fresh, fill in the channel name and list box
			if (!IsPostBack)
			{
				try
				{
					// Display the channel name and channel list
					ParentChannelNameLabel.Text = "Managing channel: "
												+ currentChannel.DisplayName;
					UpdateChannelListBox (null);
				}
				catch (Exception ex)
				{
					SetError (ex.Message);
				}
			}
		}

		// UpdateChannelListBox (selectedItems)
		//
		// Updates the channel list box with the current channel items
		// selectedItems is a hash table containing the names of items selected in the UI
		private void UpdateChannelListBox (Hashtable selectedItems)
		{
			// Grab the channel again since the old channel object will not reflect changes
			Channel updatedChannel = (Channel) context.Searches.GetByGuid (currentChannelID);

			// Now clear out the list box
			ChannelListBox.Items.Clear();

			// OK, finally we can paint the new values in the list box
			ChannelItemCollection children = updatedChannel.AllChildren;
			children.SortByOrdinal (false);	// Same sort order the Site Manager uses...
			foreach (ChannelItem ci in children)
			{
				// Formulate the display text for this channel item
				string liText;
				string hiddenText = "";
				if (ci.IsHiddenModePublished)
				{
					hiddenText = " (hidden)";
				}
				if ((ci as Channel) != null)
				{
					// It's a channel - label it so
					liText = ci.DisplayName + " (channel)" + hiddenText;
				}
				else
				{
					// It's not a channel - must be a posting
					liText = ci.DisplayName + " (posting)" + hiddenText;
				}

				// Add the list item for this channel item
				ListItem li = new ListItem (liText, ci.Name);
				if (selectedItems != null)
				{
					if (selectedItems.ContainsKey(ci.Name))
					{
						li.Selected = true;
					}
				}
				ChannelListBox.Items.Add (li);
			}
		}

		private void ClearError()
		{
			ErrorLabel.Text = "";
		}

		private void SetError (string message)
		{
			ErrorLabel.Text = message;
		}

		#region Web Form Designer generated code
		override protected void OnInit(EventArgs e)
		{
			//
			// CODEGEN: This call is required by the ASP.NET Web Form Designer.
			//
			InitializeComponent();
			base.OnInit(e);
		}
		
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{    
			this.AddChannelButton.Click += new System.EventHandler(this.AddChannelButton_Click);
			this.MoveUpButton.Click += new System.EventHandler(this.MoveUpButton_Click);
			this.MoveDownButton.Click += new System.EventHandler(this.MoveDownButton_Click);
			this.DeleteButton.Click += new System.EventHandler(this.DeleteButton_Click);
			this.HideShowButton.Click += new System.EventHandler(this.HideShowButton_Click);
			this.Load += new System.EventHandler(this.Page_Load);

		}
		#endregion

		// AddChannelButton_Click (sender, e)
		// Adds a new subchannel
		private void AddChannelButton_Click(object sender, System.EventArgs e)
		{
			try
			{
				// Add the channel - hidden at first!
				Channel newChannel = currentChannel.CreateChannel ();
				newChannel.Name = NewNameTextBox.Text;
				newChannel.DisplayName = NewDisplayNameTextBox.Text;
				newChannel.IsHiddenModePublished = true;

				context.CommitAll();

				// Update the UI
				Hashtable selectedItems = new Hashtable();
				selectedItems.Add (NewNameTextBox.Text, true);
				UpdateChannelListBox (selectedItems);

                NewDisplayNameTextBox.Text = "";
				NewNameTextBox.Text = "";

			}
			catch (Exception ex)
			{
				SetError (ex.Message);
			}
		}

		// MoveUpButton_Click (sender, e)
		// Move the selected items up a notch in the sort order
		private void MoveUpButton_Click(object sender, System.EventArgs e)
		{
			try
			{
				// If the top item is selected, we do nothing since you can't move it
				// up any more than that!
				// If the top item is not selected, then we do our magic...
				if (!ChannelListBox.Items[0].Selected)
				{
					Hashtable selectedItems = new Hashtable();	// List of all selected channels

					int ordinal = 1;	// Ordinal for each channel item
					int offset = 0;		// Offset of ordinal for each channel item

					// Loop through the listbox from bottom to top
					// We recalculate the ordinal for each item as we go...
					for (int i=ChannelListBox.Items.Count-1; i>=0; i--)
					{
						string name = ChannelListBox.Items[i].Value;
						ChannelItem ci = currentChannel.AllChildren[name];
						if (!ChannelListBox.Items[i].Selected)
						{
							// This item is not selected - move it down past any items
							// which already moved up
							ci.SortOrdinal = ordinal-offset;
							offset = 0;
						}
						else
						{
							// This item is selected - move it up and remember the offset
							// for future nonselected items
							ci.SortOrdinal = ordinal+1;
							offset++;
							selectedItems.Add (ci.Name, true);
						}
						ordinal++;
					}

					context.CommitAll();
					UpdateChannelListBox (selectedItems);
				}
			}
			catch (Exception ex)
			{
				SetError (ex.Message);
			}
		}

		// MoveDownButton_Click (sender, e)
		private void MoveDownButton_Click(object sender, System.EventArgs e)
		{
			try
			{
				// If the bottom item is selected, we do nothing since you can't move it
				// down any more than that!
				// If the bottom item is not selected, then we do our magic...
				if (!ChannelListBox.Items[ChannelListBox.Items.Count-1].Selected)
				{
					Hashtable selectedItems = new Hashtable();	// List of all selected channels

					int ordinal = ChannelListBox.Items.Count;	// Ordinal for each channel item
					int offset = 0;		// Offset of ordinal for each channel item

					// Loop through the listbox from top to bottom
					// We recalculate the ordinal for each item as we go...
					for (int i=0; i<ChannelListBox.Items.Count; i++)
					{
						string name = ChannelListBox.Items[i].Value;
						ChannelItem ci = currentChannel.AllChildren[name];
						if (!ChannelListBox.Items[i].Selected)
						{
							// This item is not selected - move it up past any items
							// which already moved down
							ci.SortOrdinal = ordinal+offset;
							offset = 0;
						}
						else
						{
							// This item is selected - move it down and remember the offset
							// for future nonselected items
							ci.SortOrdinal = ordinal-1;
							offset++;
							selectedItems.Add (ci.Name, true);
						}
						ordinal--;
					}

					context.CommitAll();
					UpdateChannelListBox (selectedItems);
				}
			}
			catch (Exception ex)
			{
				SetError (ex.Message);
			}
		}

		// HideShowButton_Click (sender, e)
		// Toggles the "Hide When Published" flag for each selected item
		private void HideShowButton_Click(object sender, System.EventArgs e)
		{
			try
			{
				for (int i=0; i<ChannelListBox.Items.Count; i++)
				{
					if (ChannelListBox.Items[i].Selected)
					{
						// OK, this one was selected - toggle its hidden/show flag
						string name = ChannelListBox.Items[i].Value;
						ChannelItem ci = currentChannel.AllChildren[name];
						
						ci.IsHiddenModePublished = !ci.IsHiddenModePublished;
					}
				}

				context.CommitAll();
				UpdateChannelListBox (null);
			}
			catch (Exception ex)
			{
				SetError (ex.Message);
			}
		}

		// DeleteButton_Click (sender, e)
		// Deletes all selected postings and channels, unless there are subchannels...
		private void DeleteButton_Click(object sender, System.EventArgs e)
		{
			try
			{
				for (int i=0; i<ChannelListBox.Items.Count; i++)
				{
					if (ChannelListBox.Items[i].Selected)
					{
						// OK, we have a candidate for deletion
						string name = ChannelListBox.Items[i].Value;
						ChannelItem ci = currentChannel.AllChildren[name];
						
						// If it's a channel, delete its postings and then the channel
						Channel c = ci as Channel;
						if (c != null)
						{
							if (c.Channels.Count > 0)
							{
								throw (new Exception ("Unable to delete channel which has sub-channels"));
							}
							foreach (Posting cp in c.Postings)
							{
								cp.Delete();
							}
							c.Delete();
						}
						
						// If it's a posting, just delete it
						Posting p = ci as Posting;
						if (p != null)
						{
							p.Delete();
						}
					}
				}

				context.CommitAll();
				UpdateChannelListBox (null);
			}
			catch (Exception ex)
			{
				SetError (ex.Message);
			}
		}
	}
}
