using System;
using Microsoft.ContentManagement.Publishing;
using System.Resources;
using System.Globalization;
using System.Threading;
using System.Web;

namespace ContosoNet
{
	/// <summary>
	/// Summary description for LocalizationManager.
	/// </summary>
	public class LocalizationManager : ResourceManager
	{
		public LocalizationManager () : 
			base ("ContosoNet.localization.strings",
			System.Reflection.Assembly.GetExecutingAssembly())
		{
			m_currentLanguage = GetLanguageCookie ();

			// Set the thread culture and get the resource manager
			Thread.CurrentThread.CurrentCulture =
				CultureInfo.CreateSpecificCulture (m_currentLanguage);
			Thread.CurrentThread.CurrentUICulture =
				CultureInfo.CreateSpecificCulture (m_currentLanguage);
		}

		// Current language property
		private string m_currentLanguage;
		public string CurrentLanguage 
		{
			get
			{
				return (m_currentLanguage);
			}
		}

		public void CheckForLocalizedPosting ()
		{
			CheckForLocalizedPosting (m_currentLanguage);
		}

		public void CheckForLocalizedPosting (string chosenLanguage)
		{
			string myPostingPath = CmsHttpContext.Current.Posting.Path;

			// If the channel path doesn't have the localized content path, 
			// then there is no localized channel...
			int i;
			if ( (i=myPostingPath.ToLower().IndexOf 
				(SiteValues.ExternalLanguageParentUrl.ToLower())) <= 0)
			{
				// This is not a localized page - do nothing.
				return;
			}

			// OK, this page is localized. Let's see if the language of choice
			// is already on this page. Parse the page language out of the URL.
			int j = i + SiteValues.ExternalLanguageParentUrl.Length + 1;
			int k = myPostingPath.IndexOf ("/", j);
			string pageLanguage = myPostingPath.Substring (j,k-j);

			if (pageLanguage.ToLower() == chosenLanguage.ToLower())
			{
				// This page is already localized to the chosen page - do nothing
				return;
			}

			// If here, the page is localized to some language other than the
			// language of choice. Check to see if there is a corresponding
			// page.

			string newUrl = myPostingPath.Substring (0, i) +
				chosenLanguage +
				myPostingPath.Substring (k);

			ChannelItem newChannelItem =
				CmsHttpContext.Current.Searches.GetByUrl (newUrl);
			if (newChannelItem != null)
			{
				HttpContext.Current.Response.Redirect (newChannelItem.Url);
			}
		}
		
		// TranslateImageUrl (url, localizationManager)
		//
		// Attempts to translate an image URL into a localized image
		//
		public string TranslateImageUrl (string url)
		{
			return (url.Replace ("en-us", m_currentLanguage));
		}

		// TranslateDisplayName (name, localizationManager)
		//
		// Attempts to translate a name into a localized display name
		// Names will include:
		//    - Channel names
		//    - Custom property names
		//    - Custom property allowed values
		public string TranslateDisplayName (string name)
		{
			string result = base.GetString ("display" + name);
			if (result == null)
			{
				return (name);
			}
			else
			{
				return (result);
			}
		}

		private const string LANG_COOKIE_NAME = "language";
		
		public void SetLanguageCookie (string language)
		{
			HttpContext context = HttpContext.Current;

			HttpCookie cookie = context.Request.Cookies [LANG_COOKIE_NAME];
			if (cookie == null)
			{
				cookie = new HttpCookie(LANG_COOKIE_NAME);
			}
			cookie.Value = language;
			cookie.Expires = DateTime.MaxValue;

			context.Response.AppendCookie (cookie);
		}

		public string GetLanguageCookie ()
		{
			HttpContext context = HttpContext.Current;

			// Try to get the language from the cookie
			HttpCookie cookie = context.Request.Cookies [LANG_COOKIE_NAME];
			if (cookie != null)
			{
				string cookieValue = cookie.Value;
				if (cookieValue != null)
				{
					return (cookieValue);
				}
			}

			// If here, either the cookie or the value was missing.
			// Determine default value return that, also setting the cookie
			// to that value
			string defaultValue = "en-us";
			SetLanguageCookie (defaultValue);
			return (defaultValue);
		}

	}
}
