using System;
using System.Web;
using System.Xml;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.Publishing.Events;

namespace ContosoNet
{
	/// <summary>
	/// Summary description for PublishingEvents.
	/// </summary>
	public class PublishingEvents
	{
		public PublishingEvents()
		{
		}

		public static void SyncCustomProperties (Object sender, ChangingEventArgs e)
		{
			Posting posting = e.Target as Posting;

			if (posting != null)
			{
				if (posting.Template.Path == SiteValues.ExternalJobPostingTemplatePath)
				{
					ContosoNetControls.CustomPropertiesEventHandler.SyncCustomProperties
						(sender, e, SiteValues.PlaceholderForCustomProperties);
				}
			}
		}

		public static bool IsApproved (Object sender, ChangedEventArgs e)
		{
			bool isApproved = false;

			Posting posting = e.Target as Posting;
			if (posting != null)
			{
				if (posting.State == PostingState.Published)
				{
					isApproved = true;
				}
			}

			return (isApproved);
		}

		public static void SyndicateJobPostings (Object sender, ChangedEventArgs e)
		{
			Posting posting = e.Target as Posting;

			if (posting != null &&
				posting.Template.Path == SiteValues.ExternalJobPostingTemplatePath &&
				posting.Parent.Path == SiteValues.ExternalJobPostingsChannelPath)
			{
				try
				{
					// OK, we have a bonafide job posting.
					// Now we build an XML representation
					XmlDocument postingXmlDoc = new XmlDocument ();
					postingXmlDoc.LoadXml ("<?xml version='1.0' ?><JobPosting />");
					XmlElement postingXmlRootElement = postingXmlDoc.DocumentElement;

					postingXmlRootElement.SetAttribute ("name",
						HttpUtility.HtmlEncode (posting.Name));
					postingXmlRootElement.SetAttribute ("displayName",
						HttpUtility.HtmlEncode (posting.DisplayName));
					postingXmlRootElement.SetAttribute ("postingDate",
						HttpUtility.HtmlEncode (posting.StartDate.ToString()));
					postingXmlRootElement.SetAttribute ("name",
						HttpUtility.HtmlEncode (posting.Name));

					XmlElement divisionElement =
						postingXmlDoc.CreateElement ("division");
					divisionElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Division"].Value);
					postingXmlRootElement.AppendChild (divisionElement);

					XmlElement geographyElement =
						postingXmlDoc.CreateElement ("geography");
					geographyElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Geography"].Value);
					postingXmlRootElement.AppendChild (geographyElement);

					XmlElement hiringManagerElement =
						postingXmlDoc.CreateElement ("hiringManager");
					hiringManagerElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Hiring Manager"].Value);
					postingXmlRootElement.AppendChild (hiringManagerElement);

					XmlElement jobCategoryElement =
						postingXmlDoc.CreateElement ("jobCategory");
					jobCategoryElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Job Category"].Value);
					postingXmlRootElement.AppendChild (jobCategoryElement);

					XmlElement jobLevelElement =
						postingXmlDoc.CreateElement ("jobLevel");
					jobLevelElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Job Level"].Value);
					postingXmlRootElement.AppendChild (jobLevelElement);

					XmlElement jobTitleElement =
						postingXmlDoc.CreateElement ("jobTitle");
					jobTitleElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Job Title"].Value);
					postingXmlRootElement.AppendChild (jobTitleElement);

					XmlElement locationElement =
						postingXmlDoc.CreateElement ("location");
					locationElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Location"].Value);
					postingXmlRootElement.AppendChild (locationElement);

					XmlElement requisitionElement =
						postingXmlDoc.CreateElement ("requisitionID");
					requisitionElement.InnerText = 
						HttpUtility.HtmlEncode (posting.CustomProperties["Requisition"].Value);
					postingXmlRootElement.AppendChild (requisitionElement);

					HtmlPlaceholder descriptionPlaceholder =
						posting.Placeholders ["Job Description"] as HtmlPlaceholder;
					if (descriptionPlaceholder != null)
					{
						XmlElement descriptionElement =
							postingXmlDoc.CreateElement ("description");
						descriptionElement.InnerText = 
							HttpUtility.HtmlEncode (descriptionPlaceholder.Html);
						postingXmlRootElement.AppendChild (descriptionElement);
					}

					// The XML is built - export for syndication
					postingXmlDoc.Save ("c:\\JobPostingsForSyndication\\" +
						posting.Guid + ".xml");
				}
				catch (Exception ex)
				{
					try
					{
						System.Diagnostics.EventLog.WriteEntry
							("ContosoNet", "Job Posting Syndication Failed: " + ex.Message,
							System.Diagnostics.EventLogEntryType.Warning);
					}
					catch {}
				}
			}
		}

		public static void CrossPostToIntranet (Object sender, ChangedEventArgs e)
		{
			Posting posting = e.Target as Posting;

			if (posting != null)
			{
				// Try to cross-post
				AttemptCrossPost (posting,
					SiteValues.ExternalJobPostingTemplatePath,
					SiteValues.ExternalJobPostingsChannelPath,
					SiteValues.InternalJobPostingsTemplatePath,
					SiteValues.InternalJobPostingsChannelPath);
				AttemptCrossPost (posting,
					SiteValues.ExternalPressReleasesTemplatePath,
					SiteValues.ExternalPressReleasesChannelPath,
					SiteValues.InternalPressReleasesTemplatePath,
					SiteValues.InternalPressReleasesChannelPath);
				AttemptCrossPost (posting,
					SiteValues.ExternalPressReleaseDocumentTemplatePath,
					SiteValues.ExternalPressReleaseDocumentChannelPath,
					SiteValues.InternalPressReleaseDocumentTemplatePath,
					SiteValues.InternalPressReleaseDocumentChannelPath);
			}
		}

		private static void AttemptCrossPost (Posting posting,
			string externalTemplatePath, string externalChannelPath, 
			string internalTemplatePath, string internalChannelPath)
		{
			if (posting.Template.Path == externalTemplatePath &&
				posting.Parent.Path == externalChannelPath)
			{
				// This is an external job posting, so let's try to cross-post it
				// internally.
				//
				// Note that the internal site may or may not be installed, so
				// we'll check to be sure... IN PRODUCTION it is not efficient 
				// to use exception handling in this way.

				Channel internalChannel = null;
				Template internalTemplate = null;
				Posting internalPosting = null;
				try
				{
					// First, commit the original posting...
					CmsHttpContext.Current.CommitAll();
					// Now try to get the intranet channel and template
					internalChannel = 
						SiteUtils.GetChannel (internalChannelPath) 
						as Channel;
					internalTemplate =
						SiteUtils.GetTemplate (internalTemplatePath)
						as Template;
					internalPosting =
						internalChannel.Postings[posting.Name];

				}
				catch (Exception ex)
				{
					try
					{
						System.Diagnostics.EventLog.WriteEntry
							("ContosoNet", "Cross-posting failed: " + ex.Message,
							System.Diagnostics.EventLogEntryType.Warning);
					}
					catch {}
				}

				// Check to see if we found the intranet
				if (internalChannel != null &&
					internalTemplate != null)
				{
					// OK, there is an intranet site.
					// This might be an update, so first we need to delete
					// any existing posting
					if (internalPosting != null)
					{
						internalPosting.Delete();
					}
					// Now cross-post
					internalPosting =
						internalChannel.CreateConnectedPosting 
						(internalTemplate, posting);
					internalPosting.DisplayName = posting.DisplayName;
					internalPosting.Approve();
					CmsHttpContext.Current.CommitAll();
				}
			}
		}
	}
}
