using System;
using System.Collections;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Web;
using System.Web.Services;

using System.Security.Principal;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.SharePoint.Placeholders;

namespace ContosoNet.WS
{
	/// <summary>
	/// Summary description for News.
	/// </summary>
	public class News : System.Web.Services.WebService
	{
		public News()
		{
			//CODEGEN: This call is required by the ASP.NET Web Services Designer
			InitializeComponent();
		}

		#region Component Designer generated code
		
		//Required by the Web Services Designer 
		private IContainer components = null;
				
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if(disposing && components != null)
			{
				components.Dispose();
			}
			base.Dispose(disposing);		
		}
		
		#endregion

		[WebMethod]
		public string Add (string name, string title, string summary, string body)
		{
			try
			{
				// Get application context for this web service
				CmsApplicationContext context = new CmsApplicationContext();
				WindowsIdentity id = HttpContext.Current.User.Identity as WindowsIdentity;
				if (id != null)
				{
					context.AuthenticateUsingUserHandle( id.Token, PublishingMode.Update );
				}

				// To create a new posting we first need the channel and template
				Channel newsChannel = SiteUtils.GetChannel (SiteValues.ExternalPressReleasesChannelPath, context);
				if (newsChannel == null)
				{
					throw new Exception ("Unable to locate channel " + SiteValues.ExternalPressReleasesChannelPath);
				}

				Template newsTemplate =
					SiteUtils.GetTemplate (SiteValues.ExternalPressReleasesTemplatePath, context);
				if (newsTemplate == null)
				{
					throw new Exception ("Unable to locate template " + SiteValues.ExternalPressReleasesTemplatePath);
				}

				// Now make the new posting
				Posting newsPosting = newsChannel.CreatePosting (newsTemplate);
				newsPosting.Name = name;
				newsPosting.DisplayName = title;

				// Get the placeholders for this posting
				HtmlPlaceholder titlePlaceholder =
					newsPosting.Placeholders [SiteValues.PlaceholderForNewsTitle]
					as HtmlPlaceholder;
				if (titlePlaceholder == null)
				{
					throw new Exception ("Placeholder not found: " + SiteValues.PlaceholderForNewsTitle);
				}

				HtmlPlaceholder summaryPlaceholder =
					newsPosting.Placeholders [SiteValues.PlaceholderForNewsSummary]
					as HtmlPlaceholder;
				if (summaryPlaceholder == null)
				{
					throw new Exception ("Placeholder not found: " + SiteValues.PlaceholderForNewsSummary);
				}

				HtmlPlaceholder bodyPlaceholder =
					newsPosting.Placeholders [SiteValues.PlaceholderForNewsBody]
					as HtmlPlaceholder;
				if (bodyPlaceholder == null)
				{
					throw new Exception ("Placeholder not found: " + SiteValues.PlaceholderForNewsBody);
				}

				// Fill in the placeholders
				titlePlaceholder.Html = Server.HtmlDecode (title);
				summaryPlaceholder.Html = Server.HtmlDecode (summary);
				bodyPlaceholder.Html = Server.HtmlDecode (body);

				// Submit the posting
				newsPosting.Submit();

				// Nothing will happen unless we commit the transaction
				CmsHttpContext.Current.CommitAll();

				// Return the good news
				return ("News submitted");
			}
			catch (Exception ex)
			{
				return ("ERROR: " + ex.Message);
			}
		}

		// Structure for returning data
		public struct NewsItem
		{
			public string requestStatus;
			public string title;
			public string publicationDate;
			public string summary;
			public string body;
			public string url;
		}
		
		[WebMethod]
		public NewsItem[] GetRecentNews (int maxItemCount)
		{
			try
			{
				if (maxItemCount <= 0)
				{
					throw (new Exception ("Invalid item count - must specify a positive integer"));
				}

				// Get application context for this web service
				CmsApplicationContext context = new CmsApplicationContext();
				WindowsIdentity id = HttpContext.Current.User.Identity as WindowsIdentity;
				if (id != null)
				{
					context.AuthenticateUsingUserHandle( id.Token, PublishingMode.Published );
				}

				// Get the channel of news postings
				Channel newsChannel = SiteUtils.GetChannel (SiteValues.ExternalPressReleasesChannelPath, context);
				if (newsChannel == null)
				{
					throw new Exception ("Unable to locate channel " + SiteValues.ExternalPressReleasesChannelPath);
				}

				// Get the news postings and sort them by date, descending
				PostingCollection pc = newsChannel.Postings;
				pc.SortByStartDate (false);

				// We return the # of items requested, or the # of items available,
				// which ever is less
				int postingCount = Math.Min (pc.Count, maxItemCount);
				if (postingCount == 0)
				{
					throw new Exception ("Zero postings available");
				}

				// Declare an array of NewsItems to hold the result
				NewsItem [] resultItems = new NewsItem [postingCount];

				// Now loop through the postings
				for (int i=0; i<postingCount; i++)
				{
					Posting newsPosting = pc[i];

					// Get the placeholders so we can extract the content from them
					HtmlPlaceholder titlePlaceholder =
						newsPosting.Placeholders [SiteValues.PlaceholderForNewsTitle]
						as HtmlPlaceholder;
					if (titlePlaceholder == null)
					{
						throw new Exception ("Placeholder not found: " + SiteValues.PlaceholderForNewsTitle);
					}

					HtmlPlaceholder summaryPlaceholder =
						newsPosting.Placeholders [SiteValues.PlaceholderForNewsSummary]
						as HtmlPlaceholder;
					if (summaryPlaceholder == null)
					{
						throw new Exception ("Placeholder not found: " + SiteValues.PlaceholderForNewsSummary);
					}

					// Now copy the posting data into our structure
					resultItems [i].requestStatus = "OK";
					resultItems [i].title = Server.HtmlEncode (titlePlaceholder.Html);
					DateTime pubDate = pc[i].StartDate.ToUniversalTime();
					resultItems [i].publicationDate = pubDate.ToString ("ddd, dd MMM yyyy hh:mm:ss") + " GMT";
					resultItems [i].summary = Server.HtmlEncode (summaryPlaceholder.Html);
					resultItems [i].url = pc[i].Url;

					// We need a bit of special coding to copy the body, since it might
					// be either an HTML placeholder or a SharePoint Document placeholder.
					HtmlPlaceholder bodyHtmlPlaceholder =
						newsPosting.Placeholders [SiteValues.PlaceholderForNewsBody]
						as HtmlPlaceholder;
					if (bodyHtmlPlaceholder != null)
					{
						resultItems [i].body = Server.HtmlEncode (bodyHtmlPlaceholder.Html);
					}
					SharePointDocumentPlaceholder bodySPDocPlaceholder =
						newsPosting.Placeholders [SiteValues.PlaceholderForNewsBody]
						as SharePointDocumentPlaceholder;
					if (bodySPDocPlaceholder != null)
					{
						resultItems [i].body = Server.HtmlEncode (bodySPDocPlaceholder.ContentHtml);
					}
				
				}

				return (resultItems);
			}
			catch (Exception ex)
			{
				// Make a 1-item structure just so we can return the bad news...
				NewsItem[] resultItem = new NewsItem [1];
				resultItem [0].requestStatus = ex.Message;
				resultItem [0].title = "";
				resultItem [0].publicationDate = "";
				resultItem [0].summary = "";
				resultItem [0].body = "";
				resultItem [0].url = "";

				return (resultItem);
			}
		}
	}
}
