/* LightFTPServer.FTPServerControl
 * Author: joel@collude.com.au
 * Date: 17/11/2003
 * Description: 
 *	This class takes care of each valid authenticated user. There is a simple switch statement that is performed
 *	on each of the commands a client user emits. One could see this being tidied up somehow.
 *
 * Last update: 12/2/04
 *	Updated according to some FxCop holes, general cleanup after performance analysis
 */

using System;
using System.IO;
using System.Net.Sockets;
using System.Net;
using System.Reflection;
using System.Threading;

namespace LightFTPServer
{
	public class FTPServerControl
	{
		public enum FtpType
		{
			Ascii,
			Binary,
			Image
		};

		TcpClient client = null;
		string currentDirectory = "";
		string clientIP;
		int clientPort;
		FtpType type;
		TcpListener passiveSocket;
		bool passive = false;
		static int passivePort = 3500;
		Authentication clientAuthenticationToken;
	
		LightFTPServerLogInterface.LogOutput logOutput;

		public FTPServerControl(TcpClient client, LightFTPServerLogInterface.LogOutput loggingAssembly)
		{
			this.client = client;
			logOutput = loggingAssembly;
		}

		public bool UserLoginAndAuthentication()
		{
			string username;
			string password;

			string input;
			input = ReadControl();
			if (!(input.ToUpper().IndexOf("USER")>=0))
			{
				WriteControl("530 Login incorrect");
				return false;
			}
	
			// get the username
			username = input.Substring(input.IndexOf(" ")+1, input.Length-input.IndexOf(" ")-1);
			
			WriteControl("331 User " + username + " okay, need password.");

			// get the password
			input = ReadControl();
			if (!(input.ToUpper().IndexOf("PASS")>=0))
			{
				WriteControl("530 Login incorrect");
				return false;
			}

			password = input.Substring(input.IndexOf(" ")+1, input.Length-input.IndexOf(" ")-1);

			clientAuthenticationToken = LightFTPServer.Authentication.Authenticate(username, password);
			if (clientAuthenticationToken==null)
			{
				WriteControl("530 Login incorrect");
				logOutput.WriteLine("SECURITY:login incorrect:username=" + username + ";password=" + password);
				return false;
			}
			if (!ChangeDirectory(clientAuthenticationToken.HomeDirectory))
			{
				WriteControl("530 Unable to change to specified home directory");
				logOutput.WriteLine("SECURITY:login incorrect:username=" + username + ";password=" + password);
				return false;
			}

			WriteControl("230 Logged in.");
			// at this point, the user is authenticated;
			return true;
		}

		public void StartProcessing()
		{
			string input;

			// we need to send out the welcome message
			// TODO: clean this guy up, we could possibly use a StringBuilder here to improve speed
			string welcomeMessage = "220 " + Dns.GetHostByName(Dns.GetHostName()).AddressList[0].ToString() + " C# LightFTP Server ready.";
			WriteControl(welcomeMessage);
			
			// allow the user to try and log in 3 times, if it fails, bailout.
			int userAuthCounter = 3;
			while ((userAuthCounter>0) && (!UserLoginAndAuthentication()))
				userAuthCounter--;
			if (userAuthCounter<=0)
			{
				QuitAndKillThread(false);
				return;
			}
	
			input = ReadControl();
			while (!(input.ToUpper().IndexOf("QUIT")>=0) && !(input.Length==0))
			{
				string command;
				string parameters;
				if (input.IndexOf(" ")>0)
				{
					command = input.Substring(0,input.IndexOf(" ")).ToUpper();
					parameters = input.Substring(input.IndexOf(" ")+1);
				}
				else
				{
					command = input.ToUpper();
					parameters = null;
				}

				switch (command.ToUpper())
				{
					case "SYST":
						WriteControl("215 Windows_NT");
						break;
					case "PWD":
						WriteControl("257 \"" + currentDirectory + "\" is cwd.");
						break;
					case "NOOP":
						WriteControl("200 NOOP okay.");
						break;
					case "PORT":
						// figure out what we're connecting to
						string[] splitTest = input.Split(',');
						if (splitTest.Length<6)
							WriteControl("550 Invalid arguments.");
						else
						{
							string portParse = input.Substring(input.IndexOf(" ")+1, input.Length-input.IndexOf(" ")-1);
							portParse = portParse.Replace(',','.');
							clientIP = PortParseIP(portParse);
							clientPort = PortParsePort(portParse);
							WriteControl("200 Port command okay. " + clientIP.ToString() + ":" + clientPort.ToString());
						}
						break;		
					case "TYPE":
						if (input.IndexOf("A")>0)
							type = FtpType.Ascii;
						else if (input.IndexOf("B")>0)
							type = FtpType.Binary;
						else if (input.IndexOf("I")>0)
							type = FtpType.Image;
						else
							type = FtpType.Binary;
						
						WriteControl("200 Type set to " + GetDataType());
						break;
					case "LIST":
						// the directory already exists so we just have to remap it
						ListDirectory(clientIP, clientPort, currentDirectory.Replace('/','\\'));
						WriteControl("226 Listing completed.");
						break;
					case "NLST":
						ListDirectory(clientIP, clientPort, currentDirectory.Replace('/','\\'));
						WriteControl("226 Name list completed.");
						break;
					case "PASV":
						bool attachToPortSuccessful = false;
						// create a passive socket
						while (!attachToPortSuccessful)
						{
							try
							{
								System.Threading.Interlocked.Increment(ref passivePort);
								passiveSocket = new TcpListener(Dns.GetHostByName(Dns.GetHostName()).AddressList[0],passivePort);
								passive = true;
								passiveSocket.Start();
								attachToPortSuccessful = true;
							}
							catch (Exception ex)
							{
#if DEBUG
								logOutput.DebugWriteLine(ex.Message);
#endif
								logOutput.WriteLine("Could not attach passive socket to port \"" + passivePort);
								attachToPortSuccessful = false;
								System.Threading.Interlocked.Increment(ref passivePort);
							}
						}
						WriteControl("227 Entering Passive Mode (" + GetPasvConnectionString(Dns.GetHostByName(Dns.GetHostName()).AddressList[0].ToString(), passivePort) + ")");
						break;
					case "CWD":
						if (!(parameters==null))
						{
							if (ChangeDirectoryCwd(parameters))
								WriteControl("250 \"" + currentDirectory + "\" is new cwd.");
							else
								WriteControl("550 No such directory");
						}
						else
							WriteControl("550 No such directory");
						break;
					case "CDUP":
						if (ChangeDirectory(currentDirectory.Substring(0,currentDirectory.LastIndexOf("/"))))
							WriteControl("250 \"" + currentDirectory + "\" is new cwd.");
						else
							WriteControl("550 No such directory");
						break;
					case "RETR":
						if (RetrieveFile(parameters, type))
							WriteControl("226 Transfer completed.");
						else
							WriteControl("550 No such file or directory");
						break;
					case "STOR":
						if (UploadFile(parameters, type))
							WriteControl("226 Transfer completed.");
						else
							WriteControl("550 No such directory");
						break;
					case "DELE":
						if (DeleteFile(parameters))
							WriteControl("250 Deleted.");
						else
							WriteControl("550 No such file or directory");
						break;
					case "RMD":
						if (DeleteDirectory(parameters))
							WriteControl("250 Deleted.");
						else
							WriteControl("550 No such file or directory");
						break;
					case "MKD":
						if (CreateDirectory(parameters))
							WriteControl("257 \"" + parameters + "\" directory created.");
						else
							WriteControl("521 A file by that pathname already exists");
						break;
					default:
						string clientCommand;
						if (input.IndexOf(" ")>=0)
							clientCommand = input.Substring(0,input.IndexOf(" "));
						else
							clientCommand = input;
						WriteControl("502 " + command + " not implemented.");
#if DEBUG
						logOutput.DebugWriteLine("502 command not implemented: " + command);
#endif
						break;
				}
				input = ReadControl();
				if (input==null)
					input = "";
			}
			QuitAndKillThread(true);		
		}


		private void QuitAndKillThread(bool writeGoodbye)
		{
			if ((client!=null) && (writeGoodbye))
			{
				// we're using Reflection to look at the Socket privates of TcpClient. Rotor source code helps here
				FieldInfo field = client.GetType().GetField("m_ClientSocket", BindingFlags.Instance | BindingFlags.NonPublic);
				Socket socket = (System.Net.Sockets.Socket)field.GetValue(client);

				if (socket.Connected)
				{
					WriteControl("221 Goodbye!");
					client.Close();
				}
			}
			if ((client!=null) && !(writeGoodbye))
			{
				FieldInfo field = client.GetType().GetField("m_ClientSocket", BindingFlags.Instance | BindingFlags.NonPublic);
				Socket socket = (System.Net.Sockets.Socket)field.GetValue(client);

				if (socket.Connected)
				{
					client.Close();
				}
			}
			try
			{
				if (passiveSocket!=null)
					passiveSocket.Stop();
				logOutput.WriteLine("Connection closed.");
				logOutput.Close();
				// TODO: not sure how to kill this thread here - have to figure this one out.
				// I suspect it's a particular pattern I should be applying.
				//Thread.CurrentThread.Abort();
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				logOutput.WriteLine("An exception occured.");
				logOutput.WriteLine("Client details:");
				logOutput.WriteLine(" username: " + clientAuthenticationToken.Username);
				logOutput.WriteLine(" currentDirectory: " + currentDirectory);
			}
		}

		private bool CreateDirectory(string directory)
		{
			if ((directory.IndexOf("/")>=0) || (directory.IndexOf("..")>=0))
			{
				if (!(AllowAccessToDirectory(directory)))
					return false;
			}

			try
			{
				if (directory.IndexOf("/")>=0)
				{
					Directory.CreateDirectory(directory.Replace("/","\\"));
					return true;
				}
				else
				{
					string deleteDirectory = currentDirectory + "/" + directory;
					Directory.CreateDirectory(deleteDirectory.Replace("/","\\"));
					return true;
				}			
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.WriteLine(ex.Message);
				logOutput.DebugWriteLine("CreateDirectory failed on directory \"" + directory);
#endif
				return false;
			}
		}

		private bool DeleteDirectory(string directory)
		{
			if ((directory.IndexOf("/")>=0) || (directory.IndexOf("..")>=0))
			{
				if (!(AllowAccessToDirectory(directory)))
					return false;
			}
	
			try
			{
				if (directory.IndexOf("/")>=0)
				{
					Directory.Delete(directory.Replace("/","\\"), true);
					return true;
				}
				else
				{
					string deleteDirectory = currentDirectory + "/" + directory;
					Directory.Delete(deleteDirectory.Replace("/","\\"));
					return true;
				}
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return false;
			}

		}

		private bool DeleteFile(string file)
		{
			if ((file.IndexOf("/")>=0) || (file.IndexOf("..")>=0))
			{
				if (!(AllowAccessToDirectory(file.Substring(0,file.LastIndexOf("/")))))
					return false;
			}

			try
			{
				if (file.IndexOf("/")>=0)
				{
					File.Delete(file.Replace("/","\\"));
					return true;
				}
				else
				{
					string deleteFile = currentDirectory + "/" + file;
					deleteFile = deleteFile.Replace("/","\\");
					File.Delete(deleteFile);
					return true;
				}
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return false;
			}

		}

		private bool UploadFile(string file, FtpType fileType)
		{
			Socket socket = GetClientConnection();
			try
			{
				if (!(file.IndexOf("/")>=0))
					file = currentDirectory + "/" + file;
	
				if (!(AllowAccessToDirectory(file.Substring(0,file.LastIndexOf("/")))))
					return false;

				file = file.Replace("/","\\");
                
				if ((fileType == FtpType.Binary) || (fileType == FtpType.Image))
				{
					BinaryWriter writer = new BinaryWriter(new StreamWriter(file, false).BaseStream);
					byte[] buffer = new byte[32768];
					int bytes;
					while ((bytes = socket.Receive(buffer))>0)
					{
						writer.Write(buffer,0,bytes);
						// TODO: gc optimize
						// buffer = new byte[32768];
					}
					writer.Close();
					socket.Shutdown(SocketShutdown.Both);
					socket.Close();
					return true;
				}
				else
				{
					StreamWriter writer = new StreamWriter(file, false);
					byte[] buffer = new byte[32768];
					int bytes;
					while ((bytes = socket.Receive(buffer))>0)
					{
						writer.Write(System.Text.Encoding.ASCII.GetChars(buffer, 0, bytes));
						// TODO: gc optimize
						//buffer = new byte[32768];
					}
					writer.Close();
					socket.Shutdown(SocketShutdown.Both);
					socket.Close();
					return true;
				}
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				logOutput.WriteLine("UploadFile failed on file \"" + file);
				return false;
			}
		}


		private bool RetrieveFile(string file, FtpType fileType)
		{
			Socket socket = GetClientConnection();
			try
			{
				string fileName;


				// if the file contains a path, then we need to check that path
				if (file.IndexOf("/")>=0)
				{
					if (!AllowAccessToDirectory(file.Substring(0,file.LastIndexOf("/"))))
					{
						return false;
					}
					else
						fileName = file.Replace('/','\\');
				}
				else
				{

					if (currentDirectory[currentDirectory.Length-1]=='/')
						fileName = currentDirectory.Replace('/','\\') + file;
					else
						fileName = currentDirectory.Replace('/','\\') + '\\' + file;
				}

				if (File.Exists(fileName))
				{
					if ((fileType == FtpType.Binary) || (fileType == FtpType.Image))
					{
						BinaryReader reader = new BinaryReader(new StreamReader(fileName).BaseStream);
						byte[] buffer = new byte[32768];
						int bytes;
						while ((bytes = reader.Read(buffer,0,16384))>0)
						{
							socket.Send(buffer,bytes,SocketFlags.None);
							// TODO: gc optimize
							//buffer = new byte[32768];
						}
					}
					else
					{
						StreamReader reader = new StreamReader(fileName);
						char[] buffer = new char[32768];
						int bytes;
						while ((bytes = reader.Read(buffer,0,16384))>0)
						{
							socket.Send(System.Text.Encoding.ASCII.GetBytes(buffer),bytes,SocketFlags.None);
							// TODO: gc optimize
							//buffer = new char[32768];
						}
					}
					socket.Shutdown(SocketShutdown.Both);
					socket.Close();
					return true;
				}
				else
				{
					socket.Close();
					return false;
				}
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return false;
			}
		}

		private bool AllowAccessToDirectory(string directory)
		{
			if (directory.LastIndexOf("..")>=1)
				return false;

			string tempCurrentDirectory = directory;

				// do the security check
			if (clientAuthenticationToken.RestrictHomeDirectory)
			{
				if (tempCurrentDirectory.Length < clientAuthenticationToken.HomeDirectory.Length)
					return false;
				string partialRestrictHomeDirectory = tempCurrentDirectory.Substring(0, clientAuthenticationToken.HomeDirectory.Length);
				if (!(partialRestrictHomeDirectory==clientAuthenticationToken.HomeDirectory))
					return false;
			}

			return true;
		}

		private bool ChangeDirectory(string absoluteDirectory)
		{
			// FxCop rule change
			//if (absoluteDirectory=="")
			if (absoluteDirectory.Length==0)
				absoluteDirectory = "/";

			// do the security check
			if (clientAuthenticationToken.RestrictHomeDirectory)
			{
				if (absoluteDirectory.Length < clientAuthenticationToken.HomeDirectory.Length)
					return false;
				string partialRestrictHomeDirectory = absoluteDirectory.Substring(0, clientAuthenticationToken.HomeDirectory.Length);
				if (!(partialRestrictHomeDirectory==clientAuthenticationToken.HomeDirectory))
					return false;
			}

			try
			{
				DirectoryInfo dirInfo = new DirectoryInfo(absoluteDirectory.Replace("/",@"\"));
				if (dirInfo.Exists)
				{
					currentDirectory = absoluteDirectory;
					return true;
				}
				else
					return false;
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return false;
			}
		}


		private bool ChangeDirectoryCwd(string directory)
		{            
			if (directory.LastIndexOf("..")>=1)
				return false;

			string tempCurrentDirectory;

			if (directory.Length==1 && directory[0]=='/')
			{
				if (clientAuthenticationToken.RestrictHomeDirectory)
					tempCurrentDirectory = clientAuthenticationToken.HomeDirectory;
				else
					tempCurrentDirectory = "/";
			}
			else if (directory.IndexOf("/")>=0)
			{
//				tempCurrentDirectory = currentDirectory;
//				if (directory[0]=='/')
//					tempCurrentDirectory += directory;
//				else
//					tempCurrentDirectory += "/" + directory;
				if ((directory[directory.Length-1]=='/') && (directory.Length>1))
					directory = directory.Substring(0,directory.Length-1);
				tempCurrentDirectory = directory;
			}
			else
			{
				if ((directory.Length>=2) && ((directory[0]=='.') && (directory[1]=='.')) && (currentDirectory.Length>1))
				{
					tempCurrentDirectory = currentDirectory.Substring(0,currentDirectory.LastIndexOf("/"));
				}
				else
				{
					tempCurrentDirectory = currentDirectory;
					tempCurrentDirectory += "/" + directory;
				}
			}

			// FxCop rule change
			//if (tempCurrentDirectory=="")
			if (tempCurrentDirectory.Length==0)
				tempCurrentDirectory = "/";
			if (tempCurrentDirectory.IndexOf(".")>=0)
				return false;

			// do the security check
			if (clientAuthenticationToken.RestrictHomeDirectory)
			{
				if (tempCurrentDirectory.Length < clientAuthenticationToken.HomeDirectory.Length)
					return false;
				string partialRestrictHomeDirectory = tempCurrentDirectory.Substring(0, clientAuthenticationToken.HomeDirectory.Length);
				if (!(partialRestrictHomeDirectory==clientAuthenticationToken.HomeDirectory))
					return false;
			}

			try
			{
				if (tempCurrentDirectory.IndexOf(@"//")>=0)
					tempCurrentDirectory = tempCurrentDirectory.Replace(@"//","/");
				DirectoryInfo dirInfo = new DirectoryInfo(tempCurrentDirectory.Replace("/",@"\"));
				if (dirInfo.Exists)
				{
					currentDirectory = tempCurrentDirectory;
					return true;
				}
				else
					return false;
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return false;
			}

		}

		private void ListDirectory(string hostname, int port, string directory)
		{
			DirectoryInfo dirInfo = new DirectoryInfo(directory);
			Socket socket = GetClientConnection();

			foreach (DirectoryInfo dir in dirInfo.GetDirectories())
			{
				string date = Convert.ToDateTime(dir.CreationTime.ToString()).ToString("MMM dd hh:mm");
				string name = dir.Name;
				socket.Send(System.Text.Encoding.Default.GetBytes("drwxrwxrwx 2 nobody nobody 0 " + date + " " + name + "\r\n"));
			}
			foreach (FileInfo file in dirInfo.GetFiles())
			{
				string date = Convert.ToDateTime(file.CreationTime.ToString()).ToString("MMM dd hh:mm");
				string fileLength = file.Length.ToString();
				string fileName = file.Name;
				socket.Send(System.Text.Encoding.Default.GetBytes("-rwxrwxrwx 1 nobody nobody " + fileLength + " " + date + " " + fileName + "\r\n"));
			}

			socket.Shutdown(SocketShutdown.Both);
		}	

		private Socket GetClientConnection()
		{
			if (passive)
			{
				Socket socket =  passiveSocket.AcceptSocket();
				WriteControl("150 Data connection accepted.");
				return socket;
			}
			else
			{
				WriteControl("150 Opening data connection.");
				Socket socket;
				socket = new Socket(AddressFamily.InterNetwork,SocketType.Stream,ProtocolType.Tcp);
				socket.Connect(new IPEndPoint(IPAddress.Parse(clientIP),clientPort));
				return socket;
			}
		}

		private string GetPasvConnectionString(string ip, int port)
		{
			string returnString = ip.Replace('.',',');
			returnString += ",";
			// figure out the port assignment
			string binaryRepresentation = Convert.ToString(port,2);
			int binaryRepresentationLength = binaryRepresentation.Length;
			for (int i=0;i<16-binaryRepresentationLength;i++)
				binaryRepresentation = "0" + binaryRepresentation;
			// split the binaryRepresentation in half.
			string last = binaryRepresentation.Substring(binaryRepresentation.Length-8);
			string first = binaryRepresentation.Substring(0,binaryRepresentation.Length-8);

			returnString += Convert.ToInt32(first, 2).ToString() + ",";
			returnString += Convert.ToInt32(last, 2).ToString();
			return returnString;
		}

		private string GetDataType()
		{
			if (type == FtpType.Ascii)
				return "ASCII";
			else if (type == FtpType.Binary)
				return "BINARY";
			else if (type == FtpType.Image)
				return "IMAGE";
			else
				return "BINARY";
		}

		private string PortParseIP(string s)
		{
			string ip = "";
			int count = 0;
			for (int i=0;i<s.Length;i++)
			{
				if (s[i]=='.')
					count++;
				if (count==4)
					return ip;

				ip += s[i].ToString();
			}
			return ip;
		}
	
		private int PortParsePort(string s)
		{
			string port = "";
			int count = 0;
			for (int i=0;i<s.Length;i++)
			{
				if (s[i]=='.')
					count++;

				if (count>=4)
					port += s[i].ToString();
			}

			port = port.Substring(1,port.Length-1);

			// figure out the port, convert to binary.
			string first = Convert.ToString(int.Parse(port.Substring(0,port.IndexOf("."))), 2);
			int firstLength = first.Length;
			for (int i=0;i<8-firstLength;i++)
				first = "0" + first;
			string second = Convert.ToString(int.Parse(port.Substring(port.IndexOf(".")+1, port.Length-port.IndexOf(".")-1)),2);
			int secondLength = second.Length;
			for (int i=0;i<8-secondLength;i++)
				second = "0" + second;

			string binary = first + second;
			return Convert.ToInt32(binary, 2);
		}

		private string ReadControl()
		{
			string returnString = "";
			byte[] buffer = new byte[64];

			try
			{
				// this removes the 100% cpu bug
				if ((client.GetStream().CanRead) && (client.GetStream().Read(buffer, 0, buffer.Length)>=0))
				{
					returnString = System.Text.Encoding.ASCII.GetString(buffer);
					while (!(returnString.IndexOf('\n')>=0) && (returnString[0]!='\0'))
					{
						buffer = new byte[64];
						client.GetStream().Read(buffer, 0, buffer.Length);
						returnString += System.Text.Encoding.ASCII.GetString(buffer);
					}
					// if the return string contains nothing (\0's because the client disconnected return
					if ((returnString.Length==64) && (returnString[0]=='\0'))
						return "";
					returnString = returnString.Substring(0,returnString.IndexOf("\n")).Trim();
					if ((logOutput!=null) && (clientAuthenticationToken!=null))
					{
						logOutput.WriteLogOutput(clientAuthenticationToken.Username, returnString);
						logOutput.Flush();
					}
					return returnString;
				}
				else 
					return "";
			}
			catch (ObjectDisposedException ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				// we assume that something went seriously wrong here, and we should bail out
				QuitAndKillThread(false);
				return "";
			}
			catch (IOException ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				// we assume that something went seriously wrong here, and we should bail out
				QuitAndKillThread(false);
				return "";
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				return "";
			}
		}

		private void WriteControl(string s)
		{
			try
			{
				if ((logOutput!=null) && (clientAuthenticationToken!=null))
				{
					logOutput.WriteLogOutput(clientAuthenticationToken.Username, s);
					logOutput.Flush();
				}
				s = s+="\r\n";
				if (client.GetStream().CanWrite)
				{
					client.GetStream().Write(System.Text.ASCIIEncoding.ASCII.GetBytes(s), 0, s.Length);
				}
				else
				{
					logOutput.WriteLine("WriteControl unable to write to socket");
				}
			}
			catch (ObjectDisposedException ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				// we assume that something went seriously wrong here, and we should bail out
				QuitAndKillThread(false);
			}
			catch (IOException ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				// we assume that something went seriously wrong here, and we should bail out
				QuitAndKillThread(false);
			}
			catch (SocketException ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
				// we assume that something went seriously wrong here, and we should bail out
				QuitAndKillThread(false);
			}
			catch (Exception ex)
			{
#if DEBUG
				logOutput.DebugWriteLine(ex.Message);
#endif
			}
		}
	}
}
