///
///  This source code is freeware and is provided on an "as is" basis without warranties of any kind, 
///  whether express or implied, including without limitation warranties that the code is free of defect, 
///  fit for a particular purpose or non-infringing.  The entire risk as to the quality and performance of 
///  the code is with the end user.
///

using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using Microsoft.ContentManagement.Publishing;

namespace StefanG.SiteMapProviders
{
    public class MCMSSiteMapProvider : SiteMapProvider
    {

        // Here we copy the relevant information from the channel item into the SiteMapNode 
        // object. We are using the Display Name as the Title and the GUID of the channel item 
        // as the unique key for the SiteMapNode object
        // This allows easy lookup of the channel item in the MCMS repository.
        protected SiteMapNode GetSiteMapNodeFromChannelItem(ChannelItem ci)
        {
            SiteMapNode smn = null;
            if (ci != null)
            {
                smn = new SiteMapNode(this, ci.Guid);
                smn.Url = ci.Url;
                smn.Title = ci.DisplayName;
                smn.Description = ci.Description;
            }
            return smn;
        }

	// Return a SiteMapNode for a given MCMS channel item identified by GUID or Url
        public override SiteMapNode FindSiteMapNode(string GuidOrUrl)
        {
            ChannelItem ci = null;
            if (GuidOrUrl.StartsWith("{"))
                ci = CmsHttpContext.Current.Searches.GetByGuid(GuidOrUrl) as ChannelItem;
            else
                ci = EnhancedGetByUrl(CmsHttpContext.Current, GuidOrUrl);
            return GetSiteMapNodeFromChannelItem(ci);
        }

	// Return a SiteMapNode for the given HttpContext.  
        // As this context is always the current context it is fine to return the current MCMS channel item. 
        public override SiteMapNode FindSiteMapNode(HttpContext context)
        {
            ChannelItem ci = CmsHttpContext.Current.ChannelItem;
            return GetSiteMapNodeFromChannelItem(ci);
        }

        // Return a collection of SiteMapNodes for all child elements of the current channel.  
        // Here both Channels and Postings are returned. This method can be tailored to your needs. 
        public override SiteMapNodeCollection GetChildNodes(SiteMapNode node)
        {
            SiteMapNodeCollection smnc = new SiteMapNodeCollection();

            Channel channel = CmsHttpContext.Current.Searches.GetByGuid(node.Key) as Channel;
            if (channel != null)
            {
                ChannelCollection cc = channel.Channels;
                cc.SortByDisplayName();
                foreach (Channel c in cc)
                {
                    smnc.Add(GetSiteMapNodeFromChannelItem(c));
                }

                PostingCollection pc = channel.Postings;
                pc.SortByDisplayName();
                foreach (Posting p in pc)
                {
                    smnc.Add(GetSiteMapNodeFromChannelItem(p));
                }
            }
            return smnc;
        }

        // Return the SiteMapNode for the parent object of the current Node. 
        // This method is called by the SiteMapPath control (the one that can be used as Breadcrumb controls) 
        // Here we go up to the root channel. This method can be tailored to your needs  
        // if you would to return only parts of the path 
        public override SiteMapNode GetParentNode(SiteMapNode node)
        {
            ChannelItem ci = CmsHttpContext.Current.Searches.GetByGuid(node.Key) as ChannelItem;
            return GetSiteMapNodeFromChannelItem(ci.Parent);
        }

        // Return the SiteMapNode for the root object of the tree. Here we return the root channel 
        // This method can be tailored to your needs if you would like to start at a different level. 
        protected override SiteMapNode GetRootNodeCore()
        {
            Channel root = CmsHttpContext.Current.RootChannel;
            return GetSiteMapNodeFromChannelItem(root);
        }

	// Helper function to check if the "Map Channel name to Host Header name" 
        // feature is enabled or not
        private bool MapChannelToHostHeaderEnabled(CmsContext ctx) 
        { 
            return (ctx.RootChannel.UrlModePublished == "http://Channels/"); 
        } 
 
        // Replacement for the Searches.GetByUrl method as the original one
        // does not work correct with host header mapping enabled.
        // details: http://support.microsoft.com/?id=887530
        private ChannelItem EnhancedGetByUrl(CmsContext ctx, string Url) 
        { 
            if (MapChannelToHostHeaderEnabled(ctx)) 
            { 
                string Path = HttpUtility.UrlDecode(Url); 
                Path = Path.Replace("http://", "/Channels/"); 
                if (!Path.StartsWith("/Channels/")) 
                    Path = "/Channels/" + 
                           HttpContext.Current.Request.Url.Host + Path; 
                if (Path.EndsWith(".htm")) 
                    Path = Path.Substring(0, Path.Length - 4); 
                if (Path.EndsWith("/")) 
                    Path = Path.Substring(0, Path.Length - 1); 
                return (ChannelItem)(ctx.Searches.GetByPath(Path)); 
            } 
            else 
                return ctx.Searches.GetByUrl(Url); 
        }


    }
}
