using System;
using System.Collections;
using System.ComponentModel;
using System.IO;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml;
using System.Xml.Xsl;
using System.Xml.XPath;
using Microsoft.ContentManagement.Common;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.WebControls.Design;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.WebControls;
using ioko.ComponentModel.LicenseProvider;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// The Affinity Items component provides a system to select links to appear on the page, and an administration tool to configure these links.
	/// </summary>
	/// <remarks>
	/// In authoring mode, the control displays a list of categorised links supplied by an XML file,
	/// that the user can select to appear in presentation mode. There is also a control panel to add new
	/// affinity items or to edit existing links. 
	/// </remarks>	
	[ SupportedPlaceholderDefinitionType( typeof(XmlPlaceholderDefinition) ),
	ToolboxData("<{0}:AffinitySelector runat=server></{0}:AffinitySelector>"),
	Designer(typeof(MSIBPlusPack.ContentManagement.Publishing.Placeholders.AffinitySelectorDesigner)),
	LicenseProviderAttribute(typeof(PlusPackLicenseProvider))]
	public class AffinitySelector : BasePlaceholderControl
	{
		#region Licensing static fields

		private static PlusPackLicense license = null;

		private static DateTime lastDate;

		#endregion

		#region Licensing Test

		private void ValidateLicense()

		{

			try

			{

				bool updateLicense = false;

 

				if (license == null)

					updateLicense = true;

				else if (lastDate != DateTime.Today)

					updateLicense = true;

 

				if (updateLicense)

				{

					license = (PlusPackLicense)LicenseManager.Validate(typeof(AffinitySelector),this);

					lastDate = DateTime.Today;

				}

 

				switch(license.Validity)

				{

					case MSIBLicenseValidator.LicenseState.Full:

						return;

 

					case MSIBLicenseValidator.LicenseState.Trial_Active:

						if (updateLicense)

						{

							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);

							int daysRemaining = span.Days + 1;

							if (daysRemaining <= 7)

								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",

									daysRemaining));

						}

						return;

 

					case MSIBLicenseValidator.LicenseState.Invalid:

					case MSIBLicenseValidator.LicenseState.None:

						break;

 

					case MSIBLicenseValidator.LicenseState.Trial_Expired:

						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");

				}           

			} 

			catch {}

 

			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");

		}



		#endregion

		#region OnInit
		protected override void OnInit(EventArgs e)
		{
			ValidateLicense();  
			base.OnInit (e);
		}
		#endregion
		
		#region Dispose
		/// <summary>
		/// Frees resources used by the control
		/// </summary>
		public override void Dispose()
		{
			if (license != null) 
			{
				license.Dispose();
				license = null;
			}
			base.Dispose();	
		}

		#endregion

		#region enums
		/// <summary>
		/// The display mode of the control
		/// </summary>
		public enum Mode
		{
			/// <summary>
			/// In this mode the control will never return an error, and in the event of complete failure will simply not render
			/// </summary>
			Production = 1,

			/// <summary>
			/// In this mode the control will display error messages
			/// </summary>
			Debug = 2,

			/// <summary>
			/// The Resultant XML will be displayed in the browser, typically used when creating/debugging stylesheets
			/// </summary>
			OutputXml = 3
		}
		#endregion

		#region Controls
		
		ListSelector ListSelect;
		Literal litPresentation;
		AffinityEditor affinityEdit;

		#endregion

		#region Constants

		//XML constants
		private const string mcstrXML_SELECT_ALL_ITEMS = "/items/affinity_item";
		private const string mcstrXML_TITLE_ATTRIBUTE = "title";
		private const string mcstrXML_CATEGORY_ATTRIBUTE = "category";		
		private const string mcstrXML_VISIBLE_ATTRIBUTE = "visible";
		private const string mcstrXML_KEY_ATTRIBUTE = "key";
		private const string mcstrXML_ROOT_NODE = "items";
		private const string mcstrXML_AFFINITY_NODE = "affinity_item";
		private const string mcstrXML_DESCRIPTION_NODE = "description";
		private const string mcstrXML_DISPLAY_TEXT_NODE = "display_text";
		private const string mcstrXML_URL_NODE = "url";
		private const string mcstrXML_TARGET_NODE = "target";

		//property defaults
		private const string mcstrTITLE = "Useful Links";
		private const string mcstrCONTROL_DISPLAY_TITLE = "Affinity Items";
		private const int mcintMAX_ITEMS = 5;
		private const bool mcblnCATEGORY_LIST_LOCKED = false;
		

		#endregion

		#region Private Members

		private string _title = mcstrTITLE;
		private string _controlDisplayTitle = mcstrCONTROL_DISPLAY_TITLE;
		private string _xmlDataFile = "";
		private string _xsltStylesheet = "";
		private bool _categoryListLocked = mcblnCATEGORY_LIST_LOCKED;
		private int _maxItems = mcintMAX_ITEMS;
		private string _categoryList = "";
		private ListSelectionMode _selectFromMode = ListSelectionMode.Multiple;
		private ListSelectionMode _selectToMode = ListSelectionMode.Multiple;
		private Mode _presentationMode = Mode.Production;


		//CSS
		string _CSSTitle = "";
		string _CSSPanel = "";
		string _CSSAvailableList = "";
		string _CSSSelectedList = "";
		string _CSSCategoryList = "";
		string _CSSMessage = "";

		// Errors Collection
		private StringBuilder _Errors = new StringBuilder();

		#endregion

		#region Public Properties

		#region Title
		/// <summary>
		/// The initial title of the affinity items when displayed in presentation mode.
		/// </summary>
		/// <value>
		/// The title to be displayed.
		/// </value>
		/// <remarks>
		/// This is the title that will be shown in the title textbox in authoring mode, and then used for the
		/// title of the affinity items in presentation mode. This property sets the default title - if the user changes
		/// it the new title will be saved in the placeholder XML and used thereafter.
		/// </remarks>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(mcstrTITLE),
		Description("The initial title of the affinity items when displayed in presentation")]
		public string Title
		{
			get
			{
				return _title;
			}
			set
			{
				_title = value;
			}
		}
		#endregion

		#region ControlDisplayName
		/// <summary>
		/// The name displayed on the control in authoring mode.
		/// </summary>
		/// <value>
		/// The name to be displayed.
		/// </value>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(mcstrCONTROL_DISPLAY_TITLE),
		Description("This sets the control display name in authoring mode")]
		public string ControlDisplayTitle
		{
			get
			{
				return _controlDisplayTitle;
			}
			set
			{
				_controlDisplayTitle = value;
			}
		}
		#endregion

		#region XmlDataFile
		/// <summary>
		/// The XML file used to populate the control.
		/// </summary>
		/// /// <value>
		/// An absolute path to the XML file. 
		/// </value>
		/// <remarks>
		/// The file contains the affinity items that the user can select from. This data file is mandatory 
		/// and the control will not function without it.
		/// </remarks>
		/// <example>
		/// The structure of the XML document should be as follows.
		/// <code>
		/// &lt;items&gt;
		///		&lt;affinity_item category="Internal Links"&gt;
		///			&lt;description&gt;Job Vacancies&lt;/description&gt;
		///			&lt;display_text&gt;Vacancies&lt;/display_text&gt;
		///			&lt;url&gt;/mysite/vacancies&lt;/url&gt;
		///			&lt;target&gt;_SELF&lt;/target&gt;
		///		&lt;/affinity_item&gt;
		///		&lt;affinity_item category="External Links"&gt;
		///			&lt;description&gt;Microsoft website&lt;/description&gt;
		///			&lt;display_text&gt;Microsoft&lt;/display_text&gt;
		///			&lt;url&gt;http://www.microsoft.com &lt;/url&gt;
		///			&lt;target&gt;_SELF&lt;/target&gt;
		///		&lt;/affinity_item&gt;
		///		...
		///	 &lt;/items&gt;
		/// </code>
		/// Each affinity item has a category, which is supplied as an attribute of the "affinity_item" node.
		/// The child nodes of each affinity item are as follows:
		/// <list type="bullet">
		/// <item>description - A description of the affinity item. This is used for the alt tag of the link in presentation
		/// mode.</item>
		/// <item>display_text - The text to be displayed on the link. This text is also used for selection of items
		/// in authoring mode.</item>
		/// <item>url - The url that will be used for the href attribute of the link.</item>
		/// <item>target - The target of the link.</item>
		/// </list>
		/// </example>
		/// 
		[
		Editor(typeof(System.Windows.Forms.Design.FileNameEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("The XML file used to populate the control")]
		public string XmlDataFile
		{
			set
			{
				_xmlDataFile = value;
			}
			get
			{
				return _xmlDataFile;
			}
		}
		#endregion

		#region XsltStylesheet
		/// <summary>
		/// The XSLT file used in presentation mode.
		/// </summary>
		/// /// <value>
		/// An absolute or relative path to the XSLT file. 
		/// </value>
		/// <remarks>
		/// In Production and Debug modes, this XSLT file will transform the XML output by the control into html.
		/// </remarks>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("The XSLT file used in presentation mode")]
		public string XsltStylesheet
		{
			set
			{
				_xsltStylesheet = value;
			}
			get
			{
				return _xsltStylesheet;
			}
		}
		#endregion

		#region CategoryListLocked
		/// <summary>
		/// Locks the category listbox.
		/// </summary>
		/// <value>
		/// True if the listbox should be locked. False if it should not.
		/// </value>
		/// <remarks>
		/// When the category list is locked, if a category is supplied in the category list property,
		/// this will be the displayed category. If no category is selected, all categories will be displayed.
		/// </remarks>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(mcblnCATEGORY_LIST_LOCKED),
		Description("Locks the category listbox")]
		public bool CategoryListLocked
		{
			get
			{
				return _categoryListLocked;
			}
			set
			{
				_categoryListLocked = value;
			}
		}
		#endregion

		#region CategoryList
		/// <summary>
		/// A comma delimited list of categories to be made available for selection.
		/// </summary>
		/// <value>
		/// A comma delimited list of categories.
		/// </value>
		/// <remarks>
		/// A list of categories can be provided to restrict the user to a single or multiple categories. 
		/// This should take the form of a comma delimited list, for example "Internal Links,External Links".
		/// If the property is left empty, all categories will be avaliable.As the comma is used to separate categories, it should not be used within a category name.
		/// </remarks>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("A comma delimited list of categories to be made available for selection")]
		public string CategoryList
		{
			get
			{
				return _categoryList;
			}
			set
			{
				_categoryList = value;
			}
		}
		#endregion

		#region MaxItems
		/// <summary>
		/// The maximum number of affinity items allowed.
		/// </summary>
		/// <value>
		/// The number of items allowed.
		/// </value>
		/// <remarks>
		///	The number of affinity items to be selected can be restricted with this property. 
		///</remarks>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(mcintMAX_ITEMS),
		Description("The maximum number of affinity items allowed")]
		public int MaxItems
		{
			get
			{
				return _maxItems;
			}
			set
			{
				_maxItems = value;
			}
		}
		#endregion

		#region CssTitle
		/// <summary>
		/// The CSS class of the control's display name in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the display name.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the control's title in authoring mode")]
		public string CssTitle
		{
			get
			{
				return _CSSTitle;
			}
			set
			{
				_CSSTitle = value;
			}
		}
		#endregion

		#region CssPanel
		/// <summary>
		/// The CSS class of the control panel in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the control panel.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the control panel in authoring mode")]
		public string CssPanel
		{
			get
			{
				return _CSSPanel;
			}
			set
			{
				_CSSPanel = value;
			}
		}
		#endregion

		#region CssSourceList
		/// <summary>
		/// The CSS class of the available items listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the available items listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the available items listbox in authoring mode")]
		public string CssAvailableList
		{
			get
			{
				return _CSSAvailableList;
			}
			set
			{
				_CSSAvailableList = value;
			}
		}
		#endregion

		#region CssSelectedList
		/// <summary>
		/// The CSS class of the selected items listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the selected items listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the selected items listbox in authoring mode")]
		public string CssSelectedList
		{
			get
			{
				return _CSSSelectedList;
			}
			set
			{
				_CSSSelectedList = value;
			}
		}
		#endregion

		#region CssCategoryList
		/// <summary>
		/// The CSS class of the category listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the category listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the category listbox in authoring mode")]
		public string CssCategoryList
		{
			get
			{
				return _CSSCategoryList;
			}
			set
			{
				_CSSCategoryList = value;
			}
		}
		#endregion

		#region CssMessage
		/// <summary>
		///The CSS class of the controls message text in authoring mode.
		/// </summary>
		/// <value>
		/// The message CSS class.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the controls message text in authoring mode")]
		public string CssMessage
		{
			get
			{
				return _CSSMessage;
			}
			set
			{
				_CSSMessage = value;
			}
		}
		#endregion

		#region SelectFromMode
		/// <summary>
		/// Gets or sets the selection mode of the Available items listbox.
		/// </summary>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(ListSelectionMode.Multiple),
		Description("Gets or sets the selection mode of the Available items listbox.")]
		public ListSelectionMode SelectFromMode
		{
			set
			{
				_selectFromMode = value;
			}
			get
			{
				return _selectFromMode;
			}

		}
		#endregion

		#region SelectToMode
		/// <summary>
		/// Gets or sets the selection mode of the Selected items listbox.
		/// </summary>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(ListSelectionMode.Multiple),
		Description("Gets or sets the selection mode of the Selected items listbox.")]
		public ListSelectionMode SelectToMode
		{
			set
			{
				_selectToMode = value;
			}
			get
			{
				return _selectToMode;
			}

		}
		#endregion

		#region PresentationMode
		/// <summary>
		/// The display mode of the control when viewed in CMS Presentation mode
		/// </summary>
		/// <value>
		///	The display mode.
		/// </value>
		/// <remarks>
		/// This property only affects the control in CMS presentation mode.
		/// The mode of the control can be set to one of the following:
		/// <list type="bullet">
		/// <item>Production - In this mode the control will never return an error, 
		/// and in the event of complete failure will simply not render</item>
		/// <item>Debug - In this mode the control will display error messages</item>
		/// <item>OutputXml - The resultant XML will be displayed in the browser.
		///  This is typically used when creating/debugging stylesheets</item>
		/// </list>
		/// </remarks>
		[Bindable(true),
		Category("Settings"),
		DefaultValue(Mode.Production),
		Description("The display mode of the control when viewed in CMS Presentation mode")]
		public Mode PresentationMode
		{
			get
			{
				return _presentationMode;
			}
			set
			{
				_presentationMode = value;
			}
		}
		#endregion

		#endregion

		#region BasePlaceholder Overrides

		#region CreateAuthoringChildControls(BaseModeContainer authoringContainer)

		/// <summary>
		/// This creates all the child controls for authoring mode
		/// </summary>
		/// <param name="authoringContainer">BaseModeContainer</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			this.ListSelect = new ListSelector();
			this.ListSelect.ID = "lstSelector";
			this.ListSelect.XmlDataFile = XmlDataFile;
			this.ListSelect.ControlDisplayTitle = ControlDisplayTitle;
			this.ListSelect.Title = Title;
			this.ListSelect.MaxItems = MaxItems;
			this.ListSelect.CategoryList = CategoryList;
			this.ListSelect.CategoryListLocked = CategoryListLocked;
			this.ListSelect.CssCategoryList = CssCategoryList;
			this.ListSelect.CssSelectedList = CssSelectedList;
			this.ListSelect.CssMessage = CssMessage;
			this.ListSelect.CssPanel = CssPanel;
			this.ListSelect.CssAvailableList = CssAvailableList;
			this.ListSelect.CssTitle = CssTitle;

			this.affinityEdit = new AffinityEditor();
			this.affinityEdit.ID = "affinityEdit";
			this.affinityEdit.XmlDataFile = _xmlDataFile;
			this.affinityEdit.CssError = _CSSMessage;
			this.affinityEdit.CssPanel = _CSSPanel;
			this.affinityEdit.CssTitle = _CSSTitle;

			
			//use a try - catch to avoid ViewState error when switching from presentation mode
			//to edit mode
			try
			{
				authoringContainer.Controls.Add(this.ListSelect);

			}
			catch{}
			try
			{
				authoringContainer.Controls.Add(this.affinityEdit);
			}
			catch{}
			
			}

		#endregion
			
		#region CreatePresentationChildControls(BaseModeContainer presentationContainer)

		/// <summary>
		/// Creates the presentation control
		/// </summary>
		/// <param name="presentationContainer">BaseModeContainer</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{
			litPresentation = new Literal();

			try
			{
				presentationContainer.Controls.Add(litPresentation);
			}
			catch
			{}


		}

		#endregion

		#region  LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)

		/// <summary>
		/// Loads the XML stored inside the placeholder into our authoring mode controls
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			EnsureChildControls();

			try
			{

				XmlDocument xmlData = GetXmlData();
				
				WebAuthorContext webAuthorContext = WebAuthorContext.Current;
				if(webAuthorContext.Mode == WebAuthorContextMode.AuthoringReedit)
				{
					string strPlaceholderXml = (((XmlPlaceholder)this.BoundPlaceholder).XmlAsString);

					//if we have placeholder content - load it
					if(strPlaceholderXml.Length > 0)
					{
						try
						{
						

							//remove items in placeholder XML that are no longer in the XML file
							string strCleanedPhXML = RemoveDeletedItems(strPlaceholderXml, xmlData);

							ListSelect.LoadForAuthoring(strCleanedPhXML);
						}
						catch(XmlException exc)
						{
							//if loading errors, load the default instead
							ListSelect.PopulateList(GetXPath(), null);

						}
					}
					else
					{
						//if there is no content - load the default
						ListSelect.PopulateList(GetXPath(), null);
					}
				}
			}
			catch(Exception ex)
			{
				string strErr = "LoadPlaceholderContentForAuthoring Error";
				if(_xmlDataFile.Length == 0)
				{
					strErr  += ": Unable to locate XML data file";
				}
				LogError(ex,strErr);
				ListSelect.ReportError(_Errors.ToString());

			}
			

		}

		#endregion

		#region LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)

		/// <summary>
		/// Loads the Xml stored in the placeholder into the presentation control
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			try
			{
				string strPlaceholderXml = "";

				//if in authoring preview mode, generate the preview XML.
				// this must be done as the selected listbox items are unavailable in authoring preview mode
				WebAuthorContext wContext = WebAuthorContext.Current;
				if(wContext.Mode == WebAuthorContextMode.AuthoringPreview)
				{
					try
					{
						ArrayList arrSelected =  ListSelect.GetPreviewItems();
						//generate preview placeholder xml
						strPlaceholderXml = ListSelect.GetPreviewXml(arrSelected);					
						
					}
					catch(Exception exc)
					{
						string test = exc.Message;
					}
				}
				else
				{
					strPlaceholderXml = (((XmlPlaceholder)this.BoundPlaceholder).XmlAsString);
				}

				litPresentation.Text = "";
				
				if(_presentationMode == Mode.OutputXml)
				{
					if(strPlaceholderXml.Length > 0)
					{
						litPresentation.Text = DisplayAsHtml(strPlaceholderXml);
					}

				}
				else
				{
					if(strPlaceholderXml.Length > 0)
					{
						//check if the control is set to appear in presentation mode before transformation
						if(IsControlVisible(strPlaceholderXml))
						{
							
							// return string 
							StringBuilder sbReturn = new StringBuilder();

							//Get a mapped path to the style sheet
							string strXSLFile = HttpContext.Current.Server.MapPath(_xsltStylesheet);

							//transform xml
							StringReader sReader = new StringReader(strPlaceholderXml);
							StringWriter sWriter = new StringWriter(sbReturn);
							XmlTextWriter xmlWriter = new XmlTextWriter(sWriter);
							XPathDocument xpthResults = new XPathDocument(sReader);
			
							XslTransform xslTrans = new XslTransform();
						
							xslTrans.Load(strXSLFile);

							xslTrans.Transform(xpthResults, null, xmlWriter);
					
						
							litPresentation.Text = sbReturn.ToString();
						}
					}
				}
			}
			catch (Exception ex)
			{
				switch(PresentationMode)
				{
					case Mode.Debug:
					{
						string strErr = "LoadPlaceholderContentForPresentation Error";
						if(_xsltStylesheet.Length == 0)
						{
							strErr += ": Unable to locate XSLT file";

						}

						LogError(ex,strErr);
						litPresentation.Text = _Errors.ToString();
						break;
					}
					case Mode.OutputXml:
					{
						LogError(ex,"LoadPlaceholderContentForPresentation Error:");
						litPresentation.Text = _Errors.ToString()  + DisplayAsHtml(((XmlPlaceholder)this.BoundPlaceholder).XmlAsString);
						break;
					}
					default:
					{
						litPresentation.Text = "";
						break;
					}
				}
			}

		}

		#endregion

		#region SavePlaceholderContent(PlaceholderControlSaveEventArgs e)

		/// <summary>
		/// Saves selected items into the placeholder
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			EnsureChildControls();

			try
			{				
				string strPlaceholderXml = ListSelect.GetPlaceholderXml();
				(((XmlPlaceholder)this.BoundPlaceholder).XmlAsString) = strPlaceholderXml;
				
			}
			catch
			{
			}
			

		}

		#endregion

		#region OnPopulatingDefaultContent(PlaceholderControlCancelEventArgs e)

		/// <summary>
		/// Loads default content into the placeholder
		/// </summary>
		/// <param name="e">PlaceholderControlCancelEventArgs</param>
		/// <remarks>
		/// Overridden from BasePlaceholderControl
		/// </remarks>
		protected override void OnPopulatingDefaultContent(PlaceholderControlCancelEventArgs e)
		{
		
				//get xpath for category
				ListSelect.PopulateList(GetXPath(), null);
			
		}

		#endregion

		#endregion

		#region Internal Functions

		#region GetXPath()
		/// <summary>
		/// Check the value of the CategoryList property to determine which category should be shown.
		/// If a category list is given, use the first category.
		/// </summary>
		/// <returns>An XPath to the selected category</returns>
		private string GetXPath()
		{
			string strReturn = mcstrXML_SELECT_ALL_ITEMS;
			string strCategory = _categoryList;
			if(strCategory != "")
			{
				if(strCategory.IndexOf(",") > 0)
				{
					strCategory = strCategory.Substring(0, strCategory.IndexOf(","));
				}
				if(IsValidCategory(strCategory))
				{
					strReturn = mcstrXML_SELECT_ALL_ITEMS  + "[" + "@" + mcstrXML_CATEGORY_ATTRIBUTE + "='" + strCategory + "']";
				}

			}
			return strReturn;
		}
		#endregion

		#region IsValidCategory
		/// <summary>
		/// Checks if the category given in the category list property is
		/// present in the xml file
		/// </summary>
		/// <param name="sCategory">The category name to test</param>
		/// <returns>True if the given catgory exists</returns>
		private bool IsValidCategory(string sCategory)
		{
			bool blnReturn = true;
			
			//load XML from data file
			XmlDocument xmlDoc = GetXmlData();

			//loop through XML document to get a list of all categories
			XmlNodeList xmlList = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS);
				
			string sCategoryName = "";
			StringBuilder sbCatList = new StringBuilder();
			foreach(XmlNode ndeItem in xmlList)
			{
				try
				{
					sCategoryName = ndeItem.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].Value;
					sbCatList.Append(sCategoryName);
					sbCatList.Append(",");
				}
				catch(Exception e)
				{
					string test = e.Message;
				}

			}
			string sCategoryList = sbCatList.ToString(); 

			//check that the category given in the property is in our category list from the XML 
			string sText = "," + sCategory.ToUpper() + ",";
			string sTest = "," + sCategoryList.ToUpper() + ",";
				
			blnReturn = (sTest.IndexOf(sText) > -1);

			return blnReturn;
		}
		#endregion

		#region IsControlVisible()
		/// <summary>
		/// Checks the state of the visible checkbox to decide if the control should
		/// be rendered in presentation mode
		/// </summary>
		/// <param name="sXml">The placeholder XML</param>
		/// <returns>True if the control should be displayed in presentation mode</returns>
		private bool IsControlVisible(string sXml)
		{
			XmlDocument xmlDoc = new XmlDocument();
			xmlDoc.LoadXml(sXml);
			bool bReturn = false;
			try
			{
				XmlNode xmlRoot = xmlDoc.SelectSingleNode(mcstrXML_ROOT_NODE);
				string sChecked = xmlRoot.Attributes[mcstrXML_VISIBLE_ATTRIBUTE].Value;
				bReturn = bool.Parse(sChecked);
			}
			catch
			{
				bReturn = false;
			}
			return bReturn;
		}
		#endregion

		#region RemoveDeletedItems()

		/// <summary>
		/// On loading content for authoring, checks each affinity item in the placeholder XML to determine if it is still
		/// in the XML data file. If the item has been deleted from the data file, it is removed from the
		/// placeholder XML
		/// </summary>
		/// <param name="placeholderXml">The placeholder XML</param>
		/// <param name="xmlData">The XML data document</param>
		/// <returns></returns>
		private string RemoveDeletedItems(string placeholderXml, XmlDocument xmlData)
		{
			string strReturn = "";

			try
			{
				//load placeholder content into an XML document
				XmlDocument xmlDoc = new XmlDocument();
				xmlDoc.LoadXml(placeholderXml);

				//get a nodelist of all items
				XmlNodeList xnlItems = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS);
				foreach(XmlNode xnItem in xnlItems)
				{
					//get key for item
					string strKey = xnItem.Attributes[mcstrXML_KEY_ATTRIBUTE].InnerText;
					string strCategory = strKey.Substring(strKey.IndexOf("*") + 1);
					string strDisplayText = strKey.Substring(0,strKey.IndexOf("*"));

					//select all items for the category from the XML data file
					string strXPath = mcstrXML_SELECT_ALL_ITEMS  + "[" + "@" + mcstrXML_CATEGORY_ATTRIBUTE + "='" + strCategory + "']";
					XmlNodeList xnlDataItems = xmlData.SelectNodes(strXPath);
					//loop through all items of this category
					bool bIsInDataFile = false;
					foreach(XmlNode xnDataNode in xnlDataItems)
					{
						//if there is no node with this display_text in the data file - remove it from
						//the ph xml
						string strDataDisplayText = xnDataNode.SelectSingleNode(mcstrXML_DISPLAY_TEXT_NODE).InnerText;
						//if we match the display text switch the flag to true
						if(strDisplayText == strDataDisplayText)
						{
							bIsInDataFile = true;
						}
					
					}
					if(!bIsInDataFile)
					{
						XmlNode xnRoot = xmlDoc.SelectSingleNode(mcstrXML_ROOT_NODE);
						xnRoot.RemoveChild(xnItem);
						
					}

				}
				strReturn = xmlDoc.InnerXml;
				return strReturn;
			}
			catch(Exception ex)
			{
				strReturn = placeholderXml;
				return strReturn;

			}

		}

		#endregion

		#region Display As HTML
		/// <summary>
		/// Converts XML to display in a browser. Used for OutputXML mode.
		/// </summary>
		/// <param name="strDisplay">The string to convert</param>
		/// <returns>The converted string</returns>
		private string DisplayAsHtml(string strDisplay)
		{

			string strReturn = strDisplay;

			strReturn = strReturn.Replace("><", "&gt;*BE**BR**B*&lt;");

			strReturn = strReturn.Replace("<","*B*&lt;");

			strReturn = strReturn.Replace(">", "&gt;*BE*");

			strReturn = strReturn.Replace("*BR*", "<br>");

			strReturn = strReturn.Replace("*BE*", "</b>");

			strReturn = strReturn.Replace("*B*", "<b>");

			return strReturn;

		}
		#endregion		

		#region GetXmlData()

		/// <summary>
		/// Opens the file specified in the XmlDataFile property and returns its contents
		/// </summary>
		/// <returns>The Xml data file</returns>
		private XmlDocument GetXmlData()
		{
			XmlDocument xmlDoc = new XmlDocument();
			XmlTextReader rdrXml = new XmlTextReader(XmlDataFile);
			try
			{
				xmlDoc.Load(rdrXml);
				rdrXml.Close();
			}
			catch(Exception e)
			{
				throw(e);
			}

			return xmlDoc;

		}
		#endregion

		#endregion

		#region Error Handling LogError(Exception e, string sCustomMessage, string sOrigin) + 2 overloads
		/// <summary>
		/// Logs errors to the internal error variable, if mode is correct
		/// </summary>
		/// <param name="e">Exception - the error object generated</param>
		private void LogError(Exception e){LogError(e, null, null);}
		/// <summary>
		/// Logs errors to the internal error variable, if mode is correct
		/// </summary>
		/// <param name="e">Exception - the error object generated</param>
		/// <param name="sCustomText">string - custom message</param>
		private void LogError(Exception e, string sCustomText){LogError(e, sCustomText, null);}
		/// <summary>
		/// Logs errors to the internal error variable, if mode is correct
		/// </summary>
		/// <param name="e">Exception - the error object generated</param>
		/// <param name="sCustomText">string - custom message</param>
		/// <param name="sOrigin">string - the origin of the error</param>
		private void LogError(Exception e, string sCustomText, string sOrigin)
		{
			StringBuilder sbError = new StringBuilder();
			WebAuthorContext wContext = WebAuthorContext.Current;
			if((PresentationMode != Mode.Production)||(wContext.Mode == WebAuthorContextMode.AuthoringNew) || (wContext.Mode == WebAuthorContextMode.AuthoringReedit))
			{
				const string cstrBR = "<br>";

				if(e != null)
				{
					sbError.Append(e.Message);
					sbError.Append(cstrBR);
					if(e.InnerException != null)
					{
						sbError.Append(e.InnerException.Message);
						sbError.Append(cstrBR);
					}
				}

				if(sCustomText != null)
				{
					sbError.Append(sCustomText);
					sbError.Append(cstrBR);

				}


				if(sOrigin != null)
				{
					sbError.Append(sOrigin);
					sbError.Append(cstrBR);
				}

				_Errors.Append(sbError.ToString());
			}
			
		}
		#endregion

		/// <summary>
		/// Renders the control
		/// </summary>
		/// <param name="writer">The writer to render the control to</param>
		protected override void Render(HtmlTextWriter writer)
		{
			// Check to see whether we are in presentation mode
			if (this.PresentationChildControlsAreAvailable)	
			{
				// catch the output of the original PlaceholderControl
				TextWriter tempWriter = new StringWriter();
				base.Render(new System.Web.UI.HtmlTextWriter(tempWriter));

				string sHtml= tempWriter.ToString();

				// strip out the span tags here.
				string sCleanedHtml = CleanHtml(sHtml);

				writer.Write(sCleanedHtml);
			}
			else
			{
					
				base.Render(writer);
			}
			
			
		}

		/// <summary>
		/// Removes the wrapping span tags from the placeholder in Presentation mode to enable the placeholder to
		/// conform to XHTML and accessibility standards
		/// </summary>
		/// <param name="html">The HTML to clean</param>
		/// <returns>The placeholder content with the wrapping span tags removed</returns>
		private string CleanHtml(string html)
		{
			string sReturn = "";
			//remove opening tag
			sReturn = html.Remove(0, html.IndexOf(">") + 1);
			//remove closing tag
			sReturn = sReturn.Remove(sReturn.LastIndexOf("</span>"),7);

			return sReturn;
		}

		
	}
}
