using System;
using System.Collections;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.IO;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// Summary description for ListSelector.
	/// </summary>
	internal class ListSelector : WebControl, INamingContainer
	{
        
		public ListSelector()
		{
			
			_controlDisplayTitle = "Affinity Items";
			_categoryListLocked = false;
			_maxItems = 5;
			_categoryList = "";
			_selectFromMode  = ListSelectionMode.Multiple;
			_selectToMode = ListSelectionMode.Multiple;

			_CSSTitle = "";
			_CSSPanel = "";
			_CSSAvailableList = "";
			_CSSSelectedList = "";
			_CSSCategoryList = "";
			_CSSMessage = "";

		}

		#region Constants

		//captions
		private const string mcstrCategoryCaption = "Category";
		private const string mcstrAvailableCaption = "Available";
		private const string mcstrSelectedCaption = "Selected";
		private const string mcstrAddCaption = "Add";
		private const string mcstrRemoveCaption = "Remove";
		private const string mcstrClearCaption = "Clear";
		private const string mcstrUpCaption = "+";
		private const string mcstrDownCaption = "-";
		private const string mcstrMESSAGE_CAPTION = "Maximum * Items";
		private const string mcstrCHECKBOX_CAPTION = "Visible";

		//XML constants
		private const string mcstrXML_SELECT_ALL_ITEMS = "/items/affinity_item";
		private const string mcstrXML_TITLE_ATTRIBUTE = "title";
		private const string mcstrXML_CATEGORY_ATTRIBUTE = "category";		
		private const string mcstrXML_VISIBLE_ATTRIBUTE = "visible";
		private const string mcstrXML_KEY_ATTRIBUTE = "key";
		private const string mcstrXML_ROOT_NODE = "items";
		private const string mcstrXML_AFFINITY_NODE = "affinity_item";
		private const string mcstrXML_DESCRIPTION_NODE = "description";
		private const string mcstrXML_DISPLAY_TEXT_NODE = "display_text";
		private const string mcstrXML_URL_NODE = "url";
		private const string mcstrXML_TARGET_NODE = "target";
		


		#endregion

		#region Controls

		private Label lblDisplayName;
		private Label lblError;
		private CheckBox chkVisible;
		private TextBox txtTitle;
		private Label lblCategoryCaption;
		private DropDownList lstCategories;
		private Label lblAvailableCaption;
		private ListBox lstAvailable;
		private Button cmdAdd;
		private Button cmdRemove;
		private Button cmdClear;
		private Label lblMessage;
		private Label lblSelectedCaption;
		private ListBox lstSelected;
		private Button cmdUp;
		private Button cmdDown;

		#endregion

		#region Private Members

		private string _title;
		private string _controlDisplayTitle;
		private bool _categoryListLocked;
		private int _maxItems;
		private string _categoryList;
		private ListSelectionMode _selectFromMode;
		private ListSelectionMode _selectToMode;
		private string _xmlDataFile;

		//CSS
		string _CSSTitle;
		string _CSSPanel;
		string _CSSAvailableList;
		string _CSSSelectedList;
		string _CSSCategoryList;
		string _CSSMessage;

		//Array containing selected items - for use in preview mode
		static ArrayList arrSelectedItems = new ArrayList();

		#endregion

		#region Public Properties

		public string Title
		{
			get
			{
				return _title;
			}
			set
			{
				_title = value;
			}
		}

		public string XmlDataFile
		{
			get
			{
				return _xmlDataFile;
			}
			set
			{
				_xmlDataFile = value;
			}
		}

		public string ControlDisplayTitle
		{
			get
			{
				return _controlDisplayTitle;
			}
			set
			{
				_controlDisplayTitle = value;
			}
		}

		public string CategoryList
		{
			get
			{
				return _categoryList;
			}
			set
			{
				_categoryList = value;
			}

		}
	

		public bool CategoryListLocked
		{
			get
			{
				return _categoryListLocked;
			}
			set
			{
				_categoryListLocked = value;
			}
		}

		public int MaxItems
		{
			get
			{
				return _maxItems;
			}
			set
			{
				_maxItems = value;
			}
		}

		#region CssTitle
		/// <summary>
		/// The CSS class of the control's display name in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the display name.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the control's title in authoring mode")]
		public string CssTitle
		{
			get
			{
				return _CSSTitle;
			}
			set
			{
				_CSSTitle = value;
			}
		}
		#endregion

		#region CssPanel
		/// <summary>
		/// The CSS class of the control panel in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the control panel.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the control panel in authoring mode")]
		public string CssPanel
		{
			get
			{
				return _CSSPanel;
			}
			set
			{
				_CSSPanel = value;
			}
		}
		#endregion

		#region CssSourceList
		/// <summary>
		/// This sets the CSS class of the available items listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the available items listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the available items listbox in authoring mode")]
		public string CssAvailableList
		{
			get
			{
				return _CSSAvailableList;
			}
			set
			{
				_CSSAvailableList = value;
			}
		}
		#endregion

		#region CssSelectedList
		/// <summary>
		/// This sets the CSS class of the selected items listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the selected items listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the selected items listbox in authoring mode")]
		public string CssSelectedList
		{
			get
			{
				return _CSSSelectedList;
			}
			set
			{
				_CSSSelectedList = value;
			}
		}
		#endregion

		#region CssCategoryList
		/// <summary>
		/// The CSS class of the category listbox in authoring mode.
		/// </summary>
		/// <value>
		/// The CSS class of the category listbox.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the category listbox in authoring mode")]
		public string CssCategoryList
		{
			get
			{
				return _CSSCategoryList;
			}
			set
			{
				_CSSCategoryList = value;
			}
		}
		#endregion

		#region CssMessage
		/// <summary>
		///The CSS class of the controls message text in authoring mode.
		/// </summary>
		/// <value>
		/// The message CSS class.
		/// </value>
		[
		Category("CSS"),
		DefaultValue(""),
		Description("This sets the CSS class of the controls message text in authoring mode")]
		public string CssMessage
		{
			get
			{
				return _CSSMessage;
			}
			set
			{
				_CSSMessage = value;
			}
		}
		#endregion

		

		#region SelectFromMode
		/// <summary>
		/// Use this to adjust the list select mode between Multi and Single
		/// </summary>
		[Bindable(true),
		Category("Custom"),
		DefaultValue(ListSelectionMode.Multiple),
		Description("Switch the From Lists selection mide between Multi and single")]
		public ListSelectionMode SelectFromMode
		{
			set
			{
				_selectFromMode = value;
			}
			get
			{
				return _selectFromMode;
			}

		}
		#endregion

		#region SelectToMode
		/// <summary>
		/// Use this to adjust the list select mode between Multi and Single
		/// </summary>
		[Bindable(true),
		Category("Custom"),
		DefaultValue(ListSelectionMode.Multiple),
		Description("Switch the SelectToMode Lists selection mide between Multi and single")]
		public ListSelectionMode SelectToMode
		{
			set
			{
				_selectToMode = value;
			}
			get
			{
				return _selectToMode;
			}

		}
		#endregion

		#endregion

		#region Overridden properties

		public override ControlCollection Controls
		{
			get
			{
				EnsureChildControls();
				return base.Controls;
			}
		}
		#endregion

		#region Overridden Methods

		#region CreateChildControls()

		protected override void CreateChildControls()
		{
			Controls.Clear();

			lblDisplayName = new Label();
			lblDisplayName.Text = _controlDisplayTitle;

			lblError = new Label();
			lblError.Text = "";
			lblError.CssClass = _CSSMessage;

			chkVisible = new CheckBox();
			chkVisible.Text = mcstrCHECKBOX_CAPTION;

			txtTitle = new TextBox();
			txtTitle.Text = _title;

			lblCategoryCaption = new Label();
			lblCategoryCaption.Text = mcstrCategoryCaption;

			lstCategories = new DropDownList();
			lstCategories.CssClass = _CSSCategoryList;
			lstCategories.AutoPostBack = true;
			lstCategories.SelectedIndexChanged += new EventHandler(this.categoryChanged);
			lstCategories.Enabled = (!_categoryListLocked);

			lblAvailableCaption = new Label();
			lblAvailableCaption.Text = mcstrAvailableCaption;

			lstAvailable = new ListBox();
			lstAvailable.CssClass = _CSSAvailableList;
			lstAvailable.SelectionMode = _selectFromMode;

			cmdAdd = new Button();
			cmdAdd.Click += new EventHandler(this.addClick);
			cmdAdd.Text = mcstrAddCaption;
			cmdAdd.Width = Unit.Pixel(60);

			cmdRemove = new Button();
			cmdRemove.Click += new EventHandler(this.removeClick);
			cmdRemove.Text = mcstrRemoveCaption;
			cmdRemove.Width = Unit.Pixel(60);

			cmdClear = new Button();
			cmdClear.Click += new EventHandler(this.clearClick);
			cmdClear.Text = mcstrClearCaption;
			cmdClear.Width = Unit.Pixel(60);

			lblMessage = new Label();
			lblMessage.CssClass = _CSSMessage;
			lblMessage.Text = "";

			lblSelectedCaption = new Label();
			lblSelectedCaption.Text = mcstrSelectedCaption + " (max " + MaxItems +")";

			lstSelected = new ListBox();
			lstSelected.ID = "lstSelected";
			lstSelected.CssClass = _CSSSelectedList;
			lstSelected.SelectionMode = _selectToMode;

			cmdUp = new Button();
			cmdUp.Click += new EventHandler(this.upClick);
			cmdUp.Text = mcstrUpCaption;
			cmdUp.Width = Unit.Pixel(25);

			cmdDown = new Button();
			cmdDown.Click += new EventHandler(this.downClick);
			cmdDown.Text = mcstrDownCaption;
			cmdDown.Width = Unit.Pixel(25);

			this.Controls.Add(lblDisplayName);
			this.Controls.Add(lblError);
			this.Controls.Add(chkVisible);
			this.Controls.Add(txtTitle);
			this.Controls.Add(lblCategoryCaption);
			this.Controls.Add(lstCategories);
			this.Controls.Add(lblAvailableCaption);
			this.Controls.Add(lstAvailable);
			this.Controls.Add(cmdAdd);
			this.Controls.Add(cmdRemove);
			this.Controls.Add(cmdClear);
			this.Controls.Add(lblSelectedCaption);
			this.Controls.Add(lstSelected);
			this.Controls.Add(cmdUp);
			this.Controls.Add(cmdDown);
		}

		#endregion

		#region Render

		protected override void Render(HtmlTextWriter writer)
		{
			AddAttributesToRender(writer);

			writer.AddAttribute(HtmlTextWriterAttribute.Class, _CSSPanel);
			writer.RenderBeginTag(HtmlTextWriterTag.Table);

			//DisplayName Row
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.AddAttribute(HtmlTextWriterAttribute.Colspan,"2");
			writer.AddAttribute(HtmlTextWriterAttribute.Class, _CSSTitle);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblDisplayName.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Error label
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblError.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Visible checkbox
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			chkVisible.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Title textbox
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			txtTitle.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Category List Caption
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblCategoryCaption.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Category ListBox
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lstCategories.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Available ListBox Caption
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblAvailableCaption.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Available ListBox
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lstAvailable.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Add button
			writer.AddAttribute(HtmlTextWriterAttribute.Align,"center");
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			cmdAdd.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Remove button
			writer.AddAttribute(HtmlTextWriterAttribute.Align,"center");
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			cmdRemove.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Clear button
			writer.AddAttribute(HtmlTextWriterAttribute.Align,"center");
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			cmdClear.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Message Label
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblMessage.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Selected ListBox Caption
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lblSelectedCaption.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			//Selected Listbox
			writer.RenderBeginTag(HtmlTextWriterTag.Tr);
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			lstSelected.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderBeginTag(HtmlTextWriterTag.Td);
			cmdUp.RenderControl(writer);
			writer.WriteFullBeginTag("br/");
			cmdDown.RenderControl(writer);
			writer.RenderEndTag();
			writer.RenderEndTag();

			writer.RenderEndTag();// end table

			
		
		}

		#endregion

		#endregion

		#region Event Handlers

		#region addClick

		private void addClick(Object sender, EventArgs e)
		{
			lblMessage.Text = "";
			//check that selected items list contains no more than MaxItems
			int iSelected = lstSelected.Items.Count;
			if(!(iSelected >= _maxItems))
			{
				//check if add is allowed
				if(IsAddAllowed())
				{
					ArrayList arrItems = new ArrayList();
					for(int i = 0; i < lstAvailable.Items.Count; i++)
					{
						if(lstAvailable.Items[i].Selected == true)
						{
							ListItem li = lstAvailable.Items[i];
							li.Selected = false;
							
							lstSelected.Items.Add(li);
							arrItems.Add(li);
							AddSelectedItem(li);
						}
					}
					foreach(ListItem liSelected in arrItems)
					{
						lstAvailable.Items.Remove(liSelected);
					}
					iSelected = lstSelected.Items.Count;
					if(iSelected >= _maxItems)
					{
						lblMessage.Text = "";
						cmdAdd.Enabled = false;
					}
				}
				else
				{
					//more items are selected than can be added to lstSelected
					lblMessage.Text = mcstrMESSAGE_CAPTION.Replace("*", _maxItems.ToString());
					
				}
			}
			else
			{
				//lstSelected is already full
				lblMessage.Text = mcstrMESSAGE_CAPTION.Replace("*", _maxItems.ToString());
					
				cmdAdd.Enabled = false;
			}
		}

		#endregion

		#region removeClick

		private void removeClick(Object sender, EventArgs e)
		{
			ArrayList arrItems = new ArrayList();
			for(int i = 0; i < lstSelected.Items.Count; i++)
			{
				if(lstSelected.Items[i].Selected == true)
				{
					ListItem li = lstSelected.Items[i];
					li.Selected = false;
					//only add to available list if it is in the currently selected category
					if(IsInCategory(li.Value))
					{
						lstAvailable.Items.Add(li);
					}
					arrItems.Add(li);
				}
			}
			foreach(ListItem liSelected in arrItems)
			{
				lstSelected.Items.Remove(liSelected);
				RemoveSelectedItem(liSelected);
			}
			//if selected items list now has less than MaxItems, re-enable the add button
			int iSelected = lstSelected.Items.Count;
			if(iSelected < _maxItems)
			{
				cmdAdd.Enabled = true;
				lblMessage.Text = "";
			}
			//sort available listbox
			Sort(lstAvailable);

		}

		#endregion

		#region clearClick

		private void clearClick(Object sender, EventArgs e)
		{
			int iCount = lstSelected.Items.Count;
			for(int i = 0; i < iCount; i++)
			{
				ListItem li = lstSelected.Items[i];
				li.Selected = false;
				//only add to available list if it is in the currently selected category
				if(IsInCategory(li.Value))
				{
					lstAvailable.Items.Add(li);
				}
			}
			lstSelected.Items.Clear();
			ClearSelectedItems();
			//ensure our add button is enabled and error message is cleared
			cmdAdd.Enabled = true;
			lblMessage.Text = "";
			//sort available listbox
			Sort(lstAvailable);
		}

		#endregion

		#region upClick

		private void upClick(Object sender, EventArgs e)
		{
			if(lstSelected.SelectedIndex > 0)
			{
				ArrayList arrList = new ArrayList();
			
				try
				{
					//read list items into an array - using value as key
					foreach(ListItem li in lstSelected.Items)
					{
						arrList.Add(li.Value);
					}
					//move item in ArrayList
					string strSelected = lstSelected.SelectedItem.Value;
					int i = lstSelected.SelectedIndex;
					arrList.RemoveAt(i);
					arrList.Insert(i-1,strSelected);

					//reorder our module level selected items array
					try
					{
						arrSelectedItems.RemoveAt(i);
						arrSelectedItems.Insert(i-1,strSelected);
					}
					catch{}

					lstSelected.Items.Clear();
					
					//add re-ordered items
					foreach(string strItem in arrList)
					{
						ListItem li  = new ListItem(strItem.Substring(0,strItem.IndexOf("*")),strItem);
						lstSelected.Items.Add(li);
					}
				}
				catch(Exception ex)
				{
					string err = ex.Message;
				}
			}
			else
			{
				foreach(ListItem li in lstSelected.Items)
				{
					li.Selected = false;
				}
			}

			
		}

		#endregion

		#region downClick

		private void downClick(Object sender, EventArgs e)
		{
			if(lstSelected.SelectedIndex < (lstSelected.Items.Count - 1))
			{
				ArrayList arrList = new ArrayList();
				try
				{
					//read list items into an array - using value as key
					foreach(ListItem li in lstSelected.Items)
					{
						arrList.Add(li.Value);
					}
					//move item in ArrayList
					string strSelected = lstSelected.SelectedItem.Value;
					int i = lstSelected.SelectedIndex;
					arrList.RemoveAt(i);
					arrList.Insert(i+1,strSelected);

					//reorder selected items array
					try
					{
						arrSelectedItems.RemoveAt(i);
						arrSelectedItems.Insert(i+1,strSelected);
					}
					catch{}

					lstSelected.Items.Clear();

					//add re-ordered items
					foreach(string strItem in arrList)
					{
						ListItem li  = new ListItem(strItem.Substring(0,strItem.IndexOf("*")),strItem);
						lstSelected.Items.Add(li);
					}
				}
				catch(Exception ex)
				{
					string err = ex.Message;
				}
			}
			else
			{
				foreach(ListItem li in lstSelected.Items)
				{
					li.Selected = false;
				}
			}
			
		}

		#endregion

		#region categoryChanged

		private void categoryChanged(Object sender, EventArgs e)
		{
			UpdateItems();
		}

		#endregion


		#endregion

		#region PopulateList

		internal void PopulateList(string xPath, string selectedCategory)
		{
			EnsureChildControls();

			string strDisplayText = "";
			string strCategory = "";
			lstAvailable.Items.Clear();
			
			try
			{
				//load XML from data file
				XmlDocument xmlDoc = GetXmlData();

				XmlNodeList xmlItemList = xmlDoc.SelectNodes(xPath);

				//first check items exist for this category - to make sure we have not been passed a deleted category
				//if there are no items in this category, default to 'All'
				if(xmlItemList.Count == 0)
				{
					xPath = mcstrXML_SELECT_ALL_ITEMS;
					xmlItemList = xmlDoc.SelectNodes(xPath);
				}

				//add each affinity item to the Available ListBox - 
				//Use  the display_text for the text of each list item
				//use format display_text*category for value of each item,
				//to ensure items with same display_text but in different categories can be distinguished

				foreach(XmlNode xmlItemNode in xmlItemList)
				{
					strDisplayText = xmlItemNode.SelectSingleNode(mcstrXML_DISPLAY_TEXT_NODE).InnerText;
					strCategory = xmlItemNode.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].Value;
					ListItem li = new ListItem(strDisplayText, strDisplayText + "*" + strCategory);
					//only add to available list if item has not been selected
					if(!IsItemSelected(li))
					{
						lstAvailable.Items.Add(li);
					}
				}
			
				//load categories
				PopulateCategoryList();


				//if we have been passed a selected category, select it
				if(selectedCategory != null)
				{
					foreach(ListItem li in lstCategories.Items)
					{
						li.Selected = false;
						if(li.Text == selectedCategory)
						{
							li.Selected = true;
						}
					}
				}
				//sort list
				Sort(lstAvailable);
			}
			catch
			{}
		}

		#endregion

		#region IsItemSelected()


		/// <summary>
		/// Checks the keys of all items in the selected listbox against the key
		/// of the list item to be added
		/// </summary>
		/// <param name="li">The ListItem to be tested</param>
		/// <returns>true if the List Item appears in the selected items listbox</returns>
		private bool IsItemSelected(ListItem li)
		{
			bool bReturn = false;
			string strItemKey = li.Value;
			string strSelectedItemKey = "";
			foreach(ListItem liSelected in lstSelected.Items)
			{
				strSelectedItemKey = liSelected.Value;
				if(strItemKey == strSelectedItemKey)
				{
					bReturn = true;
					return bReturn;
				}
			}
			return bReturn;
		}


		#endregion

		#region UpdateItems()

		/// <summary>
		/// Updates the source listbox when the category is changed or when a new item is added with the affinity editor
		/// </summary>
		public void UpdateItems()
		{
			string strSelect = GetXpath();
			string strSelectedItem = lstCategories.SelectedItem.Text;
			PopulateList(strSelect, null);

			foreach(ListItem li in lstCategories.Items)
			{
				li.Selected = false;
				if (li.Text == strSelectedItem)
				{
					li.Selected = true;
					return;
				
				}
				
			}
			
		}

		#endregion

		#region IsInCategory
		/// <summary>
		/// Checks if a list item is in the currently selected category
		/// </summary>
		/// <param name="key">The key of the list item to test</param>
		/// <returns>True if the list item is in the currently selected category</returns>
		private bool IsInCategory(string key)
		{
			
			bool bReturn = true;
			
			//retrieve category name
			string sCategory = key.Substring(key.IndexOf("*") + 1);

			//check this category against the currently selected item
			string sSelectedCategory = lstCategories.SelectedItem.Text;
			if(sCategory == sSelectedCategory)
			{
				bReturn = true;
			}
			else
			{
				bReturn = false;
			}
			
			return bReturn;

		}

		#endregion

		#region GetXpath()
		/// <summary>
		/// Returns an XPath to all affinity items of the selected category
		/// </summary>
		/// <returns></returns>
		internal string GetXpath()
		{
			string sCategory = lstCategories.SelectedItem.Text;
			string sSelect = "";
			if(sCategory == "All")
			{
				sSelect = mcstrXML_SELECT_ALL_ITEMS;
			}
			else
			{
				sSelect = mcstrXML_SELECT_ALL_ITEMS + "[" + "@" + mcstrXML_CATEGORY_ATTRIBUTE + "='" + sCategory + "']";
			}
			return sSelect;
		}
		#endregion

		#region IsAddAllowed
		/// <summary>
		/// Checks how many items are selected - returns false if number is more than MaxItems + number already in listbox
		/// </summary>
		/// <returns>bool</returns>
		private bool IsAddAllowed()
		{
			int iTotalCount = lstAvailable.Items.Count -1;
			int iSelectedItemsCount = 0;
			for (int iCount=iTotalCount; iCount >= 0; iCount--)
			{
				ListItem itmSelect = lstAvailable.Items[iCount];
				if (itmSelect.Selected)
				{
					iSelectedItemsCount ++;
				}
			}
			int iTotalSelected = iSelectedItemsCount + lstSelected.Items.Count;
			if(iTotalSelected > _maxItems)
			{
				return false;
			}
			else
			{
				return true;
			}

		}
		#endregion

		#region Category Functions
		
		#region GetCategoryList
		/// <summary>
		/// Retrieves a comma delimited list of categories from the xml document
		/// </summary>
		/// <param name="xmlDoc">The xml data file</param>
		/// <returns>Lit of categories</returns>
		private string GetCategoryList(XmlDocument xmlDoc)
		{
			StringBuilder sbCatList = new StringBuilder();
			string sAllCategories = "";
			//check if a list has been set using the category list property
			if(_categoryList != "")
			{
				
				string[] sArray = _categoryList.Split(',');
				string sCat = "";
				for(int iCount = 0; iCount < sArray.Length; iCount++)
				{
					sCat = sArray[iCount];
					//only add valid categories
					if(isValidCategory(sCat, xmlDoc))
					{
						sbCatList.Append(sCat);
						sbCatList.Append(",");
					}
				}
				if (sbCatList.ToString() == "")
				{
					//if none of the values are valid, return all categories
					sAllCategories = GetAllCategories(xmlDoc);
					sbCatList.Append(sAllCategories);
				}
			}
			else
			{
				sAllCategories = GetAllCategories(xmlDoc);
				sbCatList.Append(sAllCategories);
			}
			string sTemp = sbCatList.ToString();
			//remove trailing comma
			sTemp = sTemp.Trim(',');
			//filter out duplicate values
			string sReturn = FilterValues(sTemp);

			return sReturn;


		}
		#endregion

		#region GetAllCategories()
		/// <summary>
		/// Returns all categories from the xml document
		/// </summary>
		/// <param name="xmlDoc">The xml data file</param>
		/// <returns>A List of all categories in the document</returns>
		private string GetAllCategories(XmlDocument xmlDoc)
		{
			//we have no list so return all categories

			//loop through each affinity item node and retrieve the category attribute
			XmlNodeList xmlList = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS);
			StringBuilder sbCatList = new StringBuilder();
			string sCategory = "";
			foreach(XmlNode ndeItem in xmlList)
			{
				try
				{
					sCategory = ndeItem.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].Value;
					sbCatList.Append(sCategory);
					sbCatList.Append(",");
				}
				catch(Exception e)
				{
					string test = e.Message;
				}
			}
			string sReturn = sbCatList.ToString();
			return sReturn;


		}
		#endregion

		#region isValidCategory
		/// <summary>
		/// Checks if the category given in the category list property is
		/// present in the XML file
		/// </summary>
		/// <param name="sCategory">The category name to test</param>
		/// <param name="xmlDoc">The XML data file</param>
		/// <returns>True if the given category exists</returns>
		private bool isValidCategory(string sCategory, XmlDocument xmlDoc)
		{
			bool blnReturn = true;
			
			//loop through XML document to get a list of all categories
			XmlNodeList xmlList = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS);
				
			string sCategoryName = "";
			StringBuilder sbCatList = new StringBuilder();
			foreach(XmlNode ndeItem in xmlList)
			{
				try
				{
					sCategoryName = ndeItem.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].Value;
					sbCatList.Append(sCategoryName);
					sbCatList.Append(",");
				}
				catch(Exception e)
				{
					string test = e.Message;
				}

			}
			string sCategoryList = sbCatList.ToString(); 

			//check that the category given in the property is in our category list from the XML
			string sText = "," + sCategory.ToUpper() + ",";
			string sTest = "," + sCategoryList.ToUpper() + ",";
				
			blnReturn = (sTest.IndexOf(sText) > -1);

		

			return blnReturn;
		}
		#endregion
		
		#region FilterValues
		/// <summary>
		/// Takes a comma delimited list of values and removes dupilcate values
		/// </summary>
		/// <param name="sList">The list to filter</param>
		/// <returns>The filtered list</returns>
		private string FilterValues(string sList)
		{
			string[] sArray = sList.Split(',');
			StringBuilder sbFiltered = new StringBuilder();
			ArrayList arrTemp = new ArrayList();
			string sCategory = "";
			
			for(int iCount = 0; iCount < sArray.Length; iCount++)
			{
				sCategory = sArray[iCount].ToString();
				//add to array if it is not already there
				if(! arrTemp.Contains(sCategory))
				{
					arrTemp.Add(sCategory);
				}
			}

			//only sort if the category list is not locked - otherwise use the first category
			if(!CategoryListLocked)
			{
				arrTemp.Sort();
			}

			foreach(string sItem in arrTemp)
			{
				sbFiltered.Append(sItem);
				sbFiltered.Append(",");

			}
			string sReturn = sbFiltered.ToString();
			sReturn = sReturn.Trim(',');
			return sReturn;


		}
		#endregion

		#region PopulateCategoryList
		/// <summary>
		/// populates the category listbox using the categories from the XML file
		/// </summary>
		private void PopulateCategoryList()
		{
			lstCategories.Items.Clear();
			if(_categoryList == "")
			{
				lstCategories.Items.Add ("All");
			}
			// Open the XML data file
			XmlDocument xmlDoc = GetXmlData();
			//first check if it is allowed in our category list property

			//get all the categories in the document
			string strCatList = GetCategoryList(xmlDoc);
			string[] sArray = strCatList.Split(',');
			//loop through array and add each item to listbox
			for(int iCount = 0; iCount < sArray.Length; iCount++)
			{

				lstCategories.Items.Add (sArray[iCount].ToString());

			}
			
	
		}
		#endregion


		#endregion

		#region Sort()
		/// <summary>
		/// Sorts a listbox alphabetically
		/// </summary>
		/// <param name="lstItems">The Listbox to sort</param>
		/// <returns> The sorted Listbox</returns>
		private ListBox Sort(ListBox lstItems)
		{
			ArrayList arrList = new ArrayList();
			//add value of each list item to an ArrayList
			foreach(ListItem li in lstItems.Items)
			{
				arrList.Add(li.Value);

			}
			//sort ArrayList
			arrList.Sort();

			//put sorted items back into listbox
			lstItems.Items.Clear();
			foreach(string strKey in arrList)
			{
				ListItem li = new ListItem(strKey.Substring(0,strKey.IndexOf("*")), strKey);
				lstItems.Items.Add(li);
			}
			return lstItems;

		}

		#endregion

		#region GetPlaceholderXml()

		/// <summary>
		/// Serializes the state of the listSelector control to XML
		/// </summary>
		/// <returns>The XML to be stored in the placeholder</returns>
		internal string GetPlaceholderXml()
		{
			EnsureChildControls();

			string strReturn = "";

			XmlDocument xmlDoc = new XmlDocument();

			XmlNode xmlRoot = xmlDoc.CreateElement("", mcstrXML_ROOT_NODE,"");
			//title attribute
			XmlAttribute attTitle = xmlDoc.CreateAttribute("",mcstrXML_TITLE_ATTRIBUTE,"");
			attTitle.Value = txtTitle.Text;
			xmlRoot.Attributes.Append(attTitle);
			//category attribute
			XmlAttribute attCategory = xmlDoc.CreateAttribute("",mcstrXML_CATEGORY_ATTRIBUTE,"");
			attCategory.Value = lstCategories.SelectedItem.Text;
			
			
			xmlRoot.Attributes.Append(attCategory);

			//visible attribute
			XmlAttribute attVisible = xmlDoc.CreateAttribute("",mcstrXML_VISIBLE_ATTRIBUTE,"");
			attVisible.Value = (chkVisible.Checked).ToString();
			xmlRoot.Attributes.Append(attVisible);

			//add each selected item

			foreach(ListItem li in lstSelected.Items)
			{
				XmlNode xmlAffinity = xmlDoc.CreateElement("",mcstrXML_AFFINITY_NODE,"");
				//key attribute
				XmlAttribute attItemKey = xmlDoc.CreateAttribute("", mcstrXML_KEY_ATTRIBUTE ,"");
				attItemKey.Value = li.Value;
				xmlAffinity.Attributes.Append(attItemKey);

				//category attribute
				XmlAttribute attItemCategory = xmlDoc.CreateAttribute("", mcstrXML_CATEGORY_ATTRIBUTE ,"");
				attItemCategory.Value = (li.Value).Substring((li.Value).IndexOf("*") + 1);
				xmlAffinity.Attributes.Append(attItemCategory);


				//add display_text node
				XmlNode xmlDisplayText = xmlDoc.CreateElement("",mcstrXML_DISPLAY_TEXT_NODE,"");
				xmlDisplayText.InnerText = li.Text;
				xmlAffinity.AppendChild(xmlDisplayText);

				//add description node
				XmlNode xmlDescription = xmlDoc.CreateElement("",mcstrXML_DESCRIPTION_NODE,"");
				xmlDescription.InnerText = GetNodeValue(li.Value, mcstrXML_DESCRIPTION_NODE);
				xmlAffinity.AppendChild(xmlDescription);

				//add url node
				XmlNode xmlUrl = xmlDoc.CreateElement("",mcstrXML_URL_NODE,"");
				xmlUrl.InnerText = GetNodeValue(li.Value, mcstrXML_URL_NODE);
				xmlAffinity.AppendChild(xmlUrl);

				//add target node
				XmlNode xmlTarget = xmlDoc.CreateElement("",mcstrXML_TARGET_NODE,"");
				xmlTarget.InnerText = GetNodeValue(li.Value, mcstrXML_TARGET_NODE);
				xmlAffinity.AppendChild(xmlTarget);

				//add to root
				xmlRoot.AppendChild(xmlAffinity);

			}
            
			xmlDoc.AppendChild(xmlRoot);

			strReturn = xmlDoc.InnerXml;

			return strReturn;

		}
		#endregion

		#region GetNodeValue()

		/// <summary>
		/// Performs a lookup in the XML data file to retrieve the child nodes of a specified item
		/// </summary>
		/// <param name="key">The key of the item to test</param>
		/// <param name="node">The node to return</param>
		/// <returns>The value of the return node</returns>
		internal string GetNodeValue(string key, string node)
		{
			string strReturn = "";
			//load XML from data file
			XmlDocument xmlDoc = GetXmlData();

			//extract category and display text
			string strCategory = key.Substring(key.IndexOf("*") + 1);
			string strDisplayText = key.Substring(0,key.IndexOf("*"));

			//return only the nodes of the given category
			XmlNodeList xmlList = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS + "[" + "@" + mcstrXML_CATEGORY_ATTRIBUTE + "='" + strCategory + "']");
			//find the node by matching the display text
			foreach(XmlNode xmlItem in xmlList)
			{
				if(xmlItem.SelectSingleNode(mcstrXML_DISPLAY_TEXT_NODE).InnerText == strDisplayText)
				{
					strReturn = xmlItem.SelectSingleNode(node).InnerText;

				}
			}
			return strReturn;

		}
		#endregion

		#region LoadForAuthoring()

		/// <summary>
		/// Uses the placeholder XML to set the state of the listselector control. Called from 
		/// LoadPlaceholderContentForAuthoring if we have XML saved in the placeholder.
		/// </summary>
		/// <param name="placeholderXml">The placeholder XML</param>
		internal void LoadForAuthoring(string placeholderXml)
		{
			EnsureChildControls();

			XmlDocument xmlDoc = new XmlDocument();
			xmlDoc.LoadXml(placeholderXml);
			
			XmlNode xmlRoot = xmlDoc.SelectSingleNode(mcstrXML_ROOT_NODE);

			//set visible checkbox
			chkVisible.Checked = bool.Parse(xmlRoot.Attributes[mcstrXML_VISIBLE_ATTRIBUTE].InnerText);
		
			//set title
			txtTitle.Text = xmlRoot.Attributes[mcstrXML_TITLE_ATTRIBUTE].InnerText;

			//add each item to selected listbox
			XmlNodeList xmlItemList = xmlDoc.SelectNodes(mcstrXML_SELECT_ALL_ITEMS);
			
			string strDisplayText = "";
			string strCategory = "";
			//clear the selected items listbox
			lstSelected.Items.Clear();
			//clear our selected items array
			ClearSelectedItems();
			foreach(XmlNode xmlItemNode in xmlItemList)
			{
				strDisplayText = xmlItemNode.SelectSingleNode(mcstrXML_DISPLAY_TEXT_NODE).InnerText;
				strCategory = xmlItemNode.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].InnerText;
				ListItem li = new ListItem(strDisplayText, strDisplayText + "*" + strCategory);
				lstSelected.Items.Add(li);
				AddSelectedItem(li);
				

			}

			//get selected category
			string strSelectedCategory = xmlRoot.Attributes[mcstrXML_CATEGORY_ATTRIBUTE].InnerText;
			//build XPath
			string strXPath = "";
			if (strSelectedCategory == "All")
			{
				strXPath = mcstrXML_SELECT_ALL_ITEMS;
			}
			else
			{			
				strXPath = mcstrXML_SELECT_ALL_ITEMS  + "[" + "@" + mcstrXML_CATEGORY_ATTRIBUTE + "='" + strSelectedCategory + "']";
			}
			//populate
			PopulateList(strXPath, strSelectedCategory);

		}

		#endregion

		#region GetPreviewXml()

		/// <summary>
		/// Returns an XML string containing the selected items in Authoring Preview mode.
		/// The selected items must be taken from the arrSelectedItems ArrayList as the actual listbox items
		/// are not available in authoring preview mode
		/// </summary>
		/// <param name="items">The ArrayList containing the selected items</param>
		/// <returns>The XML to be stored in the placeholder</returns>
		internal string GetPreviewXml(ArrayList items)
		{
			EnsureChildControls();

			string strReturn = "";

			XmlDocument xmlDoc = new XmlDocument();

			XmlNode xmlRoot = xmlDoc.CreateElement("", mcstrXML_ROOT_NODE,"");
			//title attribute
			XmlAttribute attTitle = xmlDoc.CreateAttribute("",mcstrXML_TITLE_ATTRIBUTE,"");
			attTitle.Value =  txtTitle.Text;
			xmlRoot.Attributes.Append(attTitle);

			//visible attribute
			XmlAttribute attVisible = xmlDoc.CreateAttribute("",mcstrXML_VISIBLE_ATTRIBUTE,"");
			attVisible.Value = (chkVisible.Checked).ToString();
			xmlRoot.Attributes.Append(attVisible);

			//add each selected item

			foreach(string sItem in items)
			{
				XmlNode xmlAffinity = xmlDoc.CreateElement("",mcstrXML_AFFINITY_NODE,"");
				//key attribute
				XmlAttribute attItemKey = xmlDoc.CreateAttribute("", mcstrXML_KEY_ATTRIBUTE ,"");
				attItemKey.Value = sItem;
				xmlAffinity.Attributes.Append(attItemKey);

				//category attribute
				XmlAttribute attItemCategory = xmlDoc.CreateAttribute("", mcstrXML_CATEGORY_ATTRIBUTE ,"");
				attItemCategory.Value = (sItem).Substring((sItem).IndexOf("*") + 1);
				xmlAffinity.Attributes.Append(attItemCategory);


				//add display_text node
				XmlNode xmlDisplayText = xmlDoc.CreateElement("",mcstrXML_DISPLAY_TEXT_NODE,"");
				xmlDisplayText.InnerText = sItem.Substring(0,sItem.IndexOf("*"));
				xmlAffinity.AppendChild(xmlDisplayText);

				//add description node
				XmlNode xmlDescription = xmlDoc.CreateElement("",mcstrXML_DESCRIPTION_NODE,"");
				xmlDescription.InnerText = GetNodeValue(sItem, mcstrXML_DESCRIPTION_NODE);
				xmlAffinity.AppendChild(xmlDescription);

				//add url node
				XmlNode xmlUrl = xmlDoc.CreateElement("",mcstrXML_URL_NODE,"");
				xmlUrl.InnerText = GetNodeValue(sItem, mcstrXML_URL_NODE);
				xmlAffinity.AppendChild(xmlUrl);

				//add target node
				XmlNode xmlTarget = xmlDoc.CreateElement("",mcstrXML_TARGET_NODE,"");
				xmlTarget.InnerText = GetNodeValue(sItem, mcstrXML_TARGET_NODE);
				xmlAffinity.AppendChild(xmlTarget);

				//add to root
				xmlRoot.AppendChild(xmlAffinity);

			}
            
			xmlDoc.AppendChild(xmlRoot);

			strReturn = xmlDoc.InnerXml;

			return strReturn;

		}
		#endregion

		#region ArrayList routines

		/// <summary>
		/// Adds a list item's value to the module level ArrayList when the list item
		/// is added to the selected items box
		/// </summary>
		/// <param name="item">The list item added</param>
		private void AddSelectedItem(ListItem item)
		{
			try
			{
			arrSelectedItems.Add(item.Value);
			}
			catch
			{
			}

		}

		/// <summary>
		/// Removes a list item's value from the module level ArrayList when the list item
		/// is removed from the selected items box
		/// </summary>
		/// <param name="item">The list item removed</param>
		private void RemoveSelectedItem(ListItem item)
		{
			try
			{
				arrSelectedItems.Remove(item.Value);
			}
			catch
			{
			}

		}

		/// <summary>
		/// Clears the module level ArrayList
		/// </summary>
		private void ClearSelectedItems()
		{
			try
			{
				arrSelectedItems.Clear();
			}
			catch
			{
			}

		}

		#endregion

		#region GetPreviewItems

		/// <summary>
		/// Called from the placeholder to retrieve the selected items ArrayList
		/// </summary>
		/// <returns></returns>
		internal ArrayList GetPreviewItems()
		{
			return arrSelectedItems;
		}

		#endregion

		#region GetXmlData()

		/// <summary>
		/// Opens the file specified in the XmlDataFile property and returns its contents
		/// </summary>
		/// <returns>The XML data document</returns>
		private XmlDocument GetXmlData()
		{
			XmlDocument xmlDoc = new XmlDocument();
			XmlTextReader rdrXml = new XmlTextReader(XmlDataFile);
			try
			{
				xmlDoc.Load(rdrXml);
				rdrXml.Close();
			}
			catch(Exception e)
			{
				throw(e);
			}

			return xmlDoc;

		}
		#endregion

		#region ReportError(string error)
		/// <summary>
		/// Outputs errors raised from the placeholder control to the error label
		/// </summary>
		/// <param name="error">The error to report</param>
		internal void ReportError(string error)
		{
			EnsureChildControls();
			lblError.Text = error;

		}

		#endregion

		

		

	}
}
