using System;
using System.Text;
using System.IO;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Collections;
using System.Xml;
using System.Xml.XPath;
using System.Xml.Xsl;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.WebControls;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using System.ComponentModel;
using ioko.ComponentModel.LicenseProvider;


namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
		#region Enums

	/// <summary>
	/// Enumerates modes of the control.
	/// </summary>
	public enum EnmContactMode :int
	{
		/// <summary>
		/// Production mode. No errors will be output in presentation mode.
		/// </summary>
		production = 1,
		/// <summary>
		/// Debug mode. Errors will be displayed if they occur.
		/// </summary>
		debug = 2,
		/// <summary>
		/// Debug mode. Errors will be displayed if they occur, and XML files will be 
		/// generated in the location specified in the XMLDebugVirtualPath property to assist debugging.
		/// </summary>
		debugXML = 3
	}
	;	
	
	#endregion

	/// <summary>
	/// The Contact Selector control allows you to display content from another CMS page 
	/// on an existing page. Specifically, it is designed to display contact 
	/// information on a CMS page. 
	/// </summary>
	[DefaultProperty("StartChannelPath"),
	ToolboxData("<{0}:ContactSelector runat=server></{0}:ContactSelector>")]
	[LicenseProviderAttribute(typeof(PlusPackLicenseProvider))]
	public class ContactSelector : BasePlaceholderControl
	{
		/// <summary>
		/// Initialises a new instance of the ContactSelector class.
		/// </summary>
		public ContactSelector()
		{	 
		}

		/// <summary>
		/// Called when the control is initialised.
		/// </summary>
		/// <param name="e">EventArgs class.</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense(); 
			base.OnInit (e);
		}

		#region Licensing 

		#region Licensing function
		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(ContactSelector),this);
					lastDate = DateTime.Today;
				}
				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;
					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}
						return;
					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;
					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}           
			} 
			catch {}
			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion Licensing function

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#endregion Licensing 

		#region Class-level enums

		/// <summary>
		/// Enumerates the properties of a CMS Posting.
		/// </summary>
		private enum cmsPostProperties : int
		{
			/// <summary>
			/// Name of the Posting.
			/// </summary>
			name = 1,
			/// <summary>
			/// DisplayName of the Posting.
			/// </summary>
			displayname = 2,
			/// <summary>
			/// URL of the Posting.
			/// </summary>
			url = 3,
			/// <summary>
			/// GUID of the Posting.
			/// </summary>
			guid = 4,
			/// <summary>
			/// Internally-assigned number of the Posting.
			/// </summary>
			number = 5
		}

		#endregion

		#region Constants
		private const bool m_cbln_VISIBLE_DEFAULT_PRESENTATION = false;
		// set our defaults for new pages..
		private const string m_cstr_DEFAULT_CHECKED_CHECKBOXES = "Name, Mail, MoreInfoURL";
		private const string m_cstr_PLACEHOLDER_FOR_AUTH_DISPLAY = "Name";
		private const string m_cstr_NODE_NAME = "contact";
		private const string m_cstr_AUTH_CONTROL_TITLE = "Select Contact";
		private const string m_cstr_PRES_CONTROL_TITLE_NODE_NAME = "controlTitle";
		private const string m_cstr_PRES_LINK_TEXT = "More";
		private const string m_cstr_PRES_HYPERLINK_SYMBOL = ">>";
		private const string m_cstr_PRES_CONTROL_TITLE_VALUE = "Contact";
		private const string m_cstr_AUTH_CONTROL_TITLE_CLASS = "ContactTitle";
		private const string m_cstr_AUTH_SELECTED_CONTACT_TEXT = "Current page contact:<br>";
		private const string m_cstr_AUTH_NO_CONTACT_SELECTED_TEXT = "None defined<br>";
		private const string m_cstr_AUTH_CONTACT_NOT_PUBLISHED_TEXT = " (page not published)";
		private const string m_cstr_REPLACE_LIST = "";
		private const string m_cstr_REPLACE_VALUES = "";
		private const string m_cstr_EXCLUDE_PLACEHOLDERS = "metadata";
		private const string m_cstr_CHECKBOX_TEXT = "Visible";
		private const string m_cstr_DROPDOWN_DEFAULT_TEXT = "Select...";
		private const string m_cstr_XML_NODE_CHANNELS = "channels";
		private const string m_cstr_XML_NODE_CHANNEL = "channel";
		private const string m_cstr_XML_ENABLED_ATTRIBUTE = "enabled";
		private const string m_cstr_XML_DISPLAY_FIELD_ATTRIBUTE = "displayField";
		private const EnmContactMode m_cint_CONTROL_MODE = EnmContactMode.production;
		private const string m_cstr_XML_DEBUG_VIRTUAL_PATH = "/debug/";
		private const string m_cstr_XML_ATTRIBUTE_PRES_TABLE_CLASS = "presTableClass";
		private const string m_cstr_XML_ATTRIBUTE_PRES_TITLE_CLASS = "presTitleClass";
		private const string m_cstr_XML_ATTRIBUTE_PRES_DETAILS_CLASS = "presDetailsClass";
		private const string m_cstr_XML_ATTRIBUTE_MAIL_IMAGE_PATH = "mailImgPath";
		private const string m_cstr_XML_ATTRIBUTE_PHONE_IMAGE_PATH = "phoneImgPath";
		private const string m_cstr_XML_ATTRIBUTE_LINE_IMAGE_PATH = "lineImgPath";
		private const string m_cstr_XML_ATTRIBUTE_LINK_TEXT = "linkText";
		private const string m_cstr_XML_ATTRIBUTE_LINK_SYMBOL = "linkSymbol";
		private const string m_cstr_XML_CONTACT_GUID_ELEMENT = "contactGuid";
		private const string m_cstr_XML_SETTINGS_ELEMENT = "presentationSettings";
		private const string m_cstr_XML_SELECTED_PH_ELEMENT = "selectedPlaceholder";
        // Post Node
		private const string m_cstrXML_NODE_POST = "post";
		// Special 'department-level' posting node - currently this is hardcoded in XSLT,
		// any change needs to be implemented here also..
		private const string m_cstrXML_NODE_DEPARTMENT_POST = "departmentPost";
		// Placeholder Node
		private const string m_cstrXML_NODE_PLACEHOLDER = "placeholder";
		// Name
		private const string m_cstrXML_NODE_NAME = "name";
		// Value
		private const string m_cstrXML_NODE_VALUE = "value";
		private const bool m_blnPUBLISHED_CONTACTS = true;
		// XSL file location constants..		
		private const string m_cstrXSL_AUTH_TREEVIEW_PATH = "auth_treeview.xslt";
		private const string m_cstrXSL_CONTACT_PRESENTATION = "contact_presentation.xslt";
		private const string m_cstrEDIT_BUTTON_TEXT = "Edit contact";
		private const string m_cstrDEPARTMENT_POSTING_IDENTIFIER = "department";
		
		#endregion

		#region Private members
			private string _authTitleClass = "";
			private string _authDetailsClass = "";
			private string _authTableClass = "";
			private string _presTitleClass = "";
			private string _presDetailsClass = "";
			private string _presTableClass = "";
			private string _authContactNameClass = "";
			private string _startChannelPath = "";
			private string _replaceList = m_cstr_REPLACE_LIST;
			private string _replaceValues = m_cstr_REPLACE_VALUES;
			private string _authDisplayField = m_cstr_PLACEHOLDER_FOR_AUTH_DISPLAY;
			private EnmContactMode _mode = EnmContactMode.production;
			private string _nodeName = m_cstr_NODE_NAME;
			private string _controlTitle = m_cstr_AUTH_CONTROL_TITLE;
			private string _checkBoxText = m_cstr_CHECKBOX_TEXT;
			private string _contactNotPublishedText = m_cstr_AUTH_CONTACT_NOT_PUBLISHED_TEXT;
			// Must a channel contain Postings
			private bool _bPublishedPostingsOnly = m_blnPUBLISHED_CONTACTS;
			private string _mailImgPath = "";
			private string _phoneImgPath = "";
			private string _lineImgPath = "";
			private string _presentationLinkText = m_cstr_PRES_LINK_TEXT;
			private string _presentationHyperlinkSymbol = m_cstr_PRES_HYPERLINK_SYMBOL;
			private string _listPlusImgPath = "";
			private string _listMinusImgPath = "";
			private string _currContactText = m_cstr_AUTH_SELECTED_CONTACT_TEXT;
			private string _noContactSelectedText = m_cstr_AUTH_NO_CONTACT_SELECTED_TEXT;
			private bool _bDefaultVisibilityInPresentation = m_cbln_VISIBLE_DEFAULT_PRESENTATION;
			private bool _bPageHasContact = false;
			private string _defaultCheckedCheckboxes = m_cstr_DEFAULT_CHECKED_CHECKBOXES;
			private string _xslAuthTreeviewPath = m_cstrXSL_AUTH_TREEVIEW_PATH;
			private string _xslContactPresentation = m_cstrXSL_CONTACT_PRESENTATION;
			private string _excludePlaceholders = m_cstr_EXCLUDE_PLACEHOLDERS;
			private string _XmlDebugVirtualPath = m_cstr_XML_DEBUG_VIRTUAL_PATH;
			private string _DepartmentPostingName = m_cstrDEPARTMENT_POSTING_IDENTIFIER;
			// control array for checkboxes
			private ArrayList _chkBoxes = new ArrayList();
		#endregion

		#region Interface Controls
		private CheckBox chkEnable;
		private Literal contactInfo;
		private Literal litTree;
		private Label lblSelectedPostPath;
		private Label lblCurrContactText;
		private Button btnEdit;
		#endregion

		#region Property definitions
		/// <summary>
		/// Sets whether the control is visible (presentation mode only) for new pages 
		/// which use this control.
		/// </summary>
		/// <value>
		/// True or False.
		/// </value>
		/// <remarks>
		/// This property sets the default value of the 'Visible' checkbox in authoring mode 
		/// of new pages.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Sets default control visibility (presentation mode only) for new pages using this control")]
		public bool VisibleByDefaultInPresentation
		{
			get
			{	
				return _bDefaultVisibilityInPresentation;
			}
			set
			{
				_bDefaultVisibilityInPresentation = value;
			}
		}
		/// <summary>
		/// A comma-separated list of fields which should be made visible by default for new 
		/// pages.
		/// </summary>
		/// <value>
		/// String containing names of placeholders used in the contact Posting's template, 
		/// separated by commas.
		/// </value>
		/// <remarks>
		/// The checkboxes for each specified placeholder will be checked for new pages when in 
		/// authoring mode. 
		/// These fields will then be displayed in 'presentation' mode.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Comma-separated list of placeholders, the associated checkboxes of which should be checked by default (for new pages) in authoring mode")]
		public string DefaultCheckedCheckboxes
		{
			get
			{	
				return _defaultCheckedCheckboxes;
			}
			set
			{
				_defaultCheckedCheckboxes = value;
			}
		}
		/// <summary>
		/// Text displayed before the name of the current contact.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Note that HTML can also be used here. If no value is specified, the default value 
		/// of "Current page contact:&lt;br&gt;" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Label text preceeding value of current contact in authoring mode")]
		public string CurrentContactText
		{
			get
			{	
				return _currContactText;
			}
			set
			{
				_currContactText = value;
			}
		}
		/// <summary>
		/// Text displayed in authoring mode if no contact is defined.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Note that HTML can also be used here. If no value is specified, the default value of 
		/// "None defined&lt;br&gt;" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Text displayed in authoring mode if no contact is defined")]
		public string NoContactDefinedText
		{
			get
			{	
				return _noContactSelectedText;
			}
			set
			{
				_noContactSelectedText = value;
			}
		}
		/// <summary>
		/// Text displayed adjacent to contact in authoring mode if selected contact's Posting 
		/// is not in 'published' state.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Note that HTML can also be used here. If no value is specified, the default value of 
		/// "  (page not published)" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Text displayed adjacent to contact in authoring mode if selected contact's posting is not in 'published' state")]
		public string ContactNotPublishedText
		{
			get
			{	
				return _contactNotPublishedText;
			}
			set
			{
				_contactNotPublishedText = value;
			}
		}
		/// <summary>
		/// Used in conjunction with ReplaceValues property. A comma-separated list of values (case-sensitive) to filter from placeholder 
		/// HTML of selected contact posting.
		/// </summary>
		/// <value>
		/// String containing values to replace, separated with commas.
		/// </value>
		/// <remarks>
		/// This property is used in conjunction with the ReplaceValues property. Any value 
		/// specified here is replaced with the corresponding value in the ReplaceValues property. 
		/// Hence, the string before the first comma will be replaced with the string before the 
		/// first comma in the ReplaceValues property, and so on with each position.<br/><br/>
		/// Note that any spaces are assumed to be part of the string value rather than the 
		/// delimiter, which is the comma character only. 
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Comma-separated list of string values (case-sensitive) to filter from placeholder HTML of selected contact posting. Any spaces are assumed to be part of string rather than delimiter. Used with ReplaceValues property.")]
		public string ReplaceList
		{
			get
			{	
				return _replaceList;
			}
			set
			{
				_replaceList = value;
			}
		}
		/// <summary>
		/// Used in conjunction with ReplaceList property. A comma-separated list of values (case-sensitive) to replace values in placeholder 
		/// HTML of selected contact posting.
		/// </summary>
		/// <value>
		/// String containing replacement values, separated with commas.
		/// </value>
		/// <remarks>
		/// This property is used in conjunction with the ReplaceList property. Any value 
		/// specified here is used to replace the corresponding value in the ReplaceList property. 
		/// Hence, the string before the first comma will used to replace the string before the 
		/// first comma in the ReplaceList property, and so on with each position.<br/><br/>
		/// Note that any spaces are assumed to be part of the string value rather than the 
		/// delimiter, which is the comma character only. 
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Comma-separated list of string values (case-sensitive) to replace values specified in ReplaceList. Any spaces are assumed to be part of string rather than delimiter. Used with ReplaceList property.")]
		public string ReplaceValues
		{
			get
			{	
				return _replaceValues;
			}
			set
			{
				_replaceValues = value;
			}
		}
		/// <summary>
		/// Path to location of 'plus' image used in treeview in authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. 
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.ImageUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to 'plus' image used in treeview in authoring mode")]
		public string PlusImagePath 
		{
			get
			{	
				return _listPlusImgPath;
			}
			set
			{
				_listPlusImgPath = value;
			}
		}
		/// <summary>
		/// Path to location of 'minus' image used in treeview in authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. 
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.ImageUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to 'minus' image used in treeview in authoring mode")]
		public string MinusImagePath 
		{
			get
			{	
				return _listMinusImgPath;
			}
			set
			{
				_listMinusImgPath = value;
			}
		}
		/// <summary>
		/// Path to location of thin line image used in treeview in authoring mode. When specified 
		/// a line with a height of 1 pixel will be displayed under the control header in 
		/// presentation mode. 
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. 
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.ImageUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to line image used in presentation mode (optional)")]
		public string LineImagePath 
		{
			get
			{	
				return _lineImgPath;
			}
			set
			{
				_lineImgPath = value;
			}
		}
		/// <summary>
		/// Path to location of image displayed next to e-mail field value in  
		/// presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. 
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.ImageUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to image displayed next to e-mail in presentation mode")]
		public string MailImagePath 
		{
			get
			{	
				return _mailImgPath;
			}
			set
			{
				_mailImgPath = value;
			}
		}
		/// <summary>
		/// Path to location of image displayed next to phone number field value in  
		/// presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. 
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.ImageUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to image displayed next to phone number in presentation mode")]
		public string PhoneImagePath 
		{
			get
			{	
				return _phoneImgPath;
			}
			set
			{
				_phoneImgPath = value;
			}
		}
		/// <summary>
		/// Text displayed next to 'more info' link in  
		/// presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Note that HTML can also be used here. If no value is specified, the default value of 
		/// "More" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Text displayed next to optional hyperlink in presentation mode")]
		public string PresentationLinkText 
		{
			get
			{	
				return _presentationLinkText;
			}
			set
			{
				_presentationLinkText = value;
			}
		}
		/// <summary>
		/// Text displayed as hyperlink for 'more info' link in  
		/// presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Note that HTML can also be used here. If no value is specified, the default value of 
		/// "&gt;&gt;" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Text displayed as actual 'more info' hyperlink in presentation mode")]
		public string PresentationLinkSymbol 
		{
			get
			{	
				return _presentationHyperlinkSymbol;
			}
			set
			{
				_presentationHyperlinkSymbol = value;
			}
		}
		/// <summary>
		/// CSS class of control title in Authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of control title in Authoring mode")]
		public string AuthoringTitleClass 
		{
			get
			{
				return _authTitleClass;
			}
			set
			{
				_authTitleClass = value;
			}
		}
		/// <summary>
		/// CSS class of control content in Authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of control content in Authoring mode")]
		public string AuthoringDetailsClass 
		{
			get
			{
				return _authDetailsClass;
			}
			set
			{
				_authDetailsClass = value;
			}
		}
		/// <summary>
		/// CSS class of table in Authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of table in Authoring mode")]
		public string AuthoringTableClass 
		{
			get
			{
				return _authTableClass;
			}
			set
			{
				_authTableClass = value;
			}
		}
		/// <summary>
		/// CSS class of control title in Presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of control title in Presentation mode")]
		public string PresentationTitleClass 
		{
			get
			{
				return _presTitleClass;
			}
			set
			{
				_presTitleClass = value;
			}
		}
		/// <summary>
		/// CSS class of control content in Presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of control content in Presentation mode")]
		public string PresentationDetailsClass 
		{
			get
			{
				return _presDetailsClass;
			}
			set
			{
				_presDetailsClass = value;
			}
		}
		/// <summary>
		/// CSS class of table in Presentation mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of table in Presentation mode")]
		public string PresentationTableClass 
		{
			get
			{
				return _presTableClass;
			}
			set
			{
				_presTableClass = value;
			}
		}
		/// <summary>
		/// CSS class of label which displays name of page contact in Authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		[
		Browsable(true),
		Category("CSS Settings"),
		DefaultValue(""),
		Description("CSS class of label which displays name of page contact in Authoring mode")]
		public string AuthoringContactNameLabelClass 
		{
			get
			{
				return _authContactNameClass;
			}
			set
			{
				_authContactNameClass = value;
			}
		}
		/// <summary>
		/// Path to top-level channel where the contact channels/postings are stored. 
		/// This value must be specified before the control can be used.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// This value will usually begin with "/channels/". The actual location of the contact 
		/// channel/posting structure can be anywhere on your site as long as this reference 
		/// points to the correct location.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to top-level channel hosting contact channels/postings")]
		public string StartChannelPath 
		{
			get
			{
				return _startChannelPath;
			}
			set
			{
				_startChannelPath = value;
			}
		}
		/// <summary>
		/// Name of the HTML placeholder which stores the name (or alternative identification field) of contact  
		/// in the contact template.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// This value is used in the treeview to identify contact postings.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Name of HTML placeholder which stores name (or alternative identification field) of contact")]
		public string AuthoringDisplayField 
		{
			get
			{
				return _authDisplayField;
			}
			set
			{
				_authDisplayField = value;
			}
		}
		/// <summary>
		/// Path to the XSL file used to generate presentation mode HTML.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. If no value is specified the default value 
		/// of "contact_presentation.xslt" is used.
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to XSL file for presentation mode transformation")]
		public string XSLPathPresentation
		{
			get
			{
				return _xslContactPresentation;
			}
			set
			{
				_xslContactPresentation = value;
			}
		}
		/// <summary>
		/// Path to the XSL file used to generate the treeview HTML in authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Filepath can be absolute or relative. If no value is specified the default value 
		/// of "auth_treeview.xslt" is used.
		/// </remarks>
		[
		Browsable(true),
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to XSL file for transformation of treeview used in authoring mode")]
		public string XSLPathAuthoringTreeview
		{
			get
			{
				return _xslAuthTreeviewPath;
			}
			set
			{
				_xslAuthTreeviewPath = value;
			}
		}
		/// <summary>
		/// Path to a folder where XML output files are created in debug mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Folder path should be a virtual path. To assist debugging, several XML files are 
		/// generated when the 'Mode' property is set to 'debugXML'. The following files 
		/// are generated:-
		/// <list type="table">
		/// <listheader>
		/// <term>File</term>
		/// <description>Details</description>
		/// </listheader>
		/// <item>
		/// <term>loadph_auth.xml</term>
		/// <description>Generated when the page is loaded in authoring mode and contact placeholder 
		/// contains data. File contains XML placeholder contents.</description>
		/// </item>	
		/// <item>
		/// <term>tree.xml</term>
		/// <description>Generated when the page is loaded in authoring mode. File contains 
		/// the XML representing the channel/posting hierarchy of contact postings, which 
		/// will be transformed into the treeview displayed in authoring mode.</description>
		/// </item>
		/// <item>
		/// <term>loadph_present.xml</term>
		/// <description>Generated when the page is loaded in presentation mode. 
		/// File contains XML placeholder contents at page load.</description>
		/// </item>
		/// <item>
		/// <term>xml_for_trans.xml</term>
		/// <description>Generated when the page is loaded in presentation mode. File contains 
		/// the XML which will be transformed into the presentation mode HTML.</description>
		/// </item>
		/// <item>
		/// <term>savePH.xml</term>
		/// <description>Generated when the page is saved in authoring mode. File contains 
		/// the XML which will be saved into the contact XML placeholder.</description>
		/// </item>
		/// </list>
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Path to folder where XML output files are created in debug mode")]
		public string XMLDebugVirtualPath
		{
			get
			{	
				return _XmlDebugVirtualPath;
			}
			set
			{
				_XmlDebugVirtualPath = value;
			}
		}
		/// <summary>
		/// String used to identify special Postings which represent a department, rather 
		/// than an individual. This feature is optional, and can be used when you wish 
		/// the contact details for a department to be displayed, rather than those of an 
		/// individual.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// In order to make the contact details for a department available, a special Posting can be 
		/// created which will contain these details and sit next to any Postings for the 
		/// individuals of that department. If the name of the Posting contains the string 
		/// specified in this property, it will be recognised as the department Posting 
		/// and the department name (which will correspond to a Channel containing contact Postings) 
		/// will link to this Posting in the treeview used to select a contact. <br/><br/>
		/// If no value is specified, the default value of "department" is used.<br/><br/>
		/// Modify this property value if you wish to use a string other than "department" 
		/// to identify these Postings.
		/// </remarks>
		/// <example>
		/// Contact Postings are stored in a Channel location of 
		/// "/Channels/mySite/Contacts/" with "Finance" and "HR" subchannels. There is a 
		/// Posting within the "Finance" Channel with the string "department" (assuming 
		/// this property is not modified) in the name. This Posting will be recognised as a department-level contact. It 
		/// will then be possible to select "Finance" as a page contact in the treeview 
		/// displayed in authoring mode.<br/><br/>
		/// </example>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("String used to identify special Postings which represent a department.	The Name of any such Posting should contain this string to be recognised.")]
		public string DepartmentPostingName
		{
			get
			{	
				return _DepartmentPostingName;
			}
			set
			{
				_DepartmentPostingName = value;
			}
		}
		/// <summary>
		/// Determines the behaviour of the control. 
		/// </summary>
		/// <value>
		/// Only values from the EnmContactMode enumerator are valid. These can easily
		/// be selected using the Properties dialog in Visual Studio .Net.
		/// </value>
		/// <remarks>
		/// Whilst in development, the 'debug' and 'debugXML' modes can be used to help 
		/// identify problems. In Production mode, errors will never be output whilst 
		/// in presentation mode.
		/// <seealso cref="ContactSelector.XMLDebugVirtualPath"/> 
		/// <seealso cref="EnmContactMode"/>
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(EnmContactMode.production),
		Description("Presentation mode of control")]
		public EnmContactMode Mode 
		{
			get
			{
				return _mode;
			}
			set
			{
				_mode = value;
			}
		}
		/// <summary>
		/// Name of root node in XML file.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// Optionally, an alternative name can be given to the root node used in XML generated 
		/// by the control. If no value is specified, the default value of "contact" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("XML_Settings"),
		DefaultValue(m_cstr_NODE_NAME),
		Description("Name of root node in XML file")]
		public string NodeName 
		{
			get
			{
				return _nodeName;
			}
			set
			{
				_nodeName = value;
			}
		}
		/// <summary>
		/// Title for control in authoring mode.
		/// </summary>
		/// <value>
		/// String.
		/// </value>
		/// <remarks>
		/// This value is displayed as the control's header text in authoring mode. If no 
		/// value is specified, the default value of "Select Contact" is used.
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(m_cstr_AUTH_CONTROL_TITLE),
		Description("Title for control in authoring mode")]
		public string ControlTitle
		{
			get
			{
				return _controlTitle;
			}
			set
			{
				_controlTitle = value;
			}
		}
		/// <summary>
		/// Determines whether contact Postings not in 'Published' state will be available 
		/// for selection.
		/// </summary>
		/// <value>
		/// True or False.
		/// </value>
		/// <remarks>
		/// If true only contact Postings which are published will be available.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(m_blnPUBLISHED_CONTACTS),
		Description("If true only contact postings which are approved will be available for selection.")]
		public bool PublishedContactsOnly
		{
			get
			{
				return _bPublishedPostingsOnly;
			}
			set
			{
				_bPublishedPostingsOnly = value;
			}
		}
		/// <summary>
		/// Read-only property which specifies if a contact has been 
		/// specified (and stored) for the current page.
		/// </summary>
		/// <value>
		/// True or False.
		/// </value>
		/// <remarks>
		/// If true, a contact has been stored for the current page.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(false),
		Description("If true, a contact has been specified (and stored) for the current page.")]
		public bool PageHasContact
		{
			get
			{
				return _bPageHasContact;
			}
		}
		/// <summary>
		/// A comma-separated list of placeholder names in the contact template 
		/// which are to be excluded from list available for display in presentation mode.
		/// </summary>
		/// <value>
		/// String containing placeholder names, separated by commas.
		/// </value>
		/// <remarks>
		/// Use of this property enables additional placeholders to be present in the contact 
		/// template, which are not available for display in presentation mode. By default, any 
		/// placeholder with the name "metadata" is excluded to avoid conflict with metadata mechanisims. 
		/// </remarks>
		[
		Browsable(true),
		Category("Settings"),
		DefaultValue(""),
		Description("Comma-separated list of placeholders in contact template which are to be excluded from list available for display in presentation mode.")]
		public string ExcludePlaceholders
		{
			get
			{
				return _excludePlaceholders;
			}
			set
			{
				_excludePlaceholders = value;
			}
		}

		#endregion

		#region Custom Functions - core helper functions

		#region BuildTreeHTML()

		/// <summary>
		/// This function builds the HTML for the treeview used in authoring mode, 
		/// and can be overridden.
		/// </summary>
		/// <returns>String.</returns>
		protected virtual string BuildTreeHTML()
		{
			HttpRequest httpRequest = HttpContext.Current.Request;

			// Create the xml
			string htmlList = "";
			XmlDocument xmlDoc = new XmlDocument();
			try
			{
				XmlElement xmlRoot = xmlDoc.CreateElement("",m_cstr_XML_NODE_CHANNELS,"");
				XmlElement xmlChannels = getChannelPostOnlyXML(xmlDoc);
				xmlRoot.AppendChild(xmlChannels);
				xmlDoc.AppendChild(xmlRoot);
				if (Mode == EnmContactMode.debugXML) 
				{
					try
					{
						xmlDoc.Save(System.Web.HttpContext.Current.Server.MapPath(XMLDebugVirtualPath + "tree.xml"));
					}
					catch (Exception exp)
					{
						HttpContext.Current.Trace.Warn("MSIB Contact control", "Error writing to XML debug file. Check permissions and that directory exists.", exp);
					}
				}
			}
			catch (Exception exp)
			{
				HttpContext.Current.Trace.Warn("MSIB Contact control", "Error building treeview from XML structure. Check 'tree.xml' debug file to verify XML structure.", exp);
			}
			htmlList = xmlDoc.OuterXml;
			htmlList = transformXML(htmlList, _xslAuthTreeviewPath);			
			return htmlList;
		}

		#endregion

		#region writeTreeHTML()

		/// <summary>
		/// Emits the result of <code>BuildTreeHTML()</code> to a literal control.
		/// </summary>
		private void writeTreeHTML()
		{
			litTree.Text = BuildTreeHTML();
		}

		#endregion
		
		#region getChannel
		/// <summary>
		/// Returns the current channel
		/// </summary>
		/// <returns>Channel object</returns>
		private Channel getChannel()
		{
			return CmsHttpContext.Current.Channel;
		}

		/// <summary>
		/// Returns the requested channel using Searches
		/// </summary>
		/// <remarks>
		/// If sChannel starts with '/' then it is considered to be pathed from Root and will have '/Channels' appended.
		/// If sChannel does not start with '/' it will be considered to follow on from current channel
		/// </remarks>
		/// <param name="sChannel">Path to channel [NOTE: prefix '/channels/' is not required</param>
		/// <returns>Channel or NULL if not found</returns>
		private Channel getChannel(string sChannel)
		{
			string sSearch;

			int iTest = sChannel.IndexOf("/");

			if(sChannel.IndexOf("..") == 0)
			{
				return getChannelRelative(sChannel);
			}
			
			// First case [Most likely] we are going from route
			if(sChannel.IndexOf("/") == 0)
			{
				if(sChannel.ToLower().IndexOf("/channels") != 0)
				{
					sSearch = "/channels" + sChannel;
				}
				else
				{
					sSearch = sChannel;
				}
			}
			else
			{
				// We are following on from current channel
				sSearch = getChannel().Path + "/" + sChannel;
			}
				
			sSearch = sSearch.Replace("//","/");

			Channel cmsReturnChannel = (Channel)CmsHttpContext.Current.Searches.GetByPath(sSearch);

			return cmsReturnChannel;

		}

		private Channel getChannelRelative(string sPath)
		{
			Channel cmsReturnChannel = (Channel)CmsHttpContext.Current.Channel.GetByRelativePath(sPath);

			return cmsReturnChannel;
		}
		#endregion

		#region hasRealData

		/// <summary>
		/// Determines if a placeholder contains real data other than formatting.
		/// </summary>
		/// <param name="sValue">Input string to test.</param>
		/// <returns>Boolean.</returns>
		private bool hasRealData(string sValue)
		{
			sValue = sValue.Replace("&nbsp;", "");
			return (sValue.Length > 3) ? true : false;
		}


		#endregion

		#region BoundXmlPlaceholder

		/// <summary>
		/// Casts the placeholder bound to this to XmlPlaceholder
		/// </summary>
		/// <returns>XmlPlaceholder object</returns>
		private XmlPlaceholder BoundXmlPlaceholder
		{
			get
			{
				return (XmlPlaceholder)this.BoundPlaceholder;
			}
		}

		#endregion

		#region RetrieveGuid(XmlDocument xmlData)
		/// <summary>
		/// This function retrieves the stored GUID from an XmlDocument, and can be overridden.
		/// </summary>
		/// <remarks>Used in CreateAuthoringChildControls() and 
		/// LoadPlaceholderContentForPresentation. </remarks>
		/// <param name="xmlData">XmlDocument object in which GUID is stored.</param>
		/// <returns>String.</returns>
		private string RetrieveGuid(XmlDocument xmlData)
		{
			try
			{
				// get posting GUID..
				XPathNavigator xPath = xmlData.CreateNavigator();
				XPathNodeIterator Iterator = xPath.Select("/" + m_cstr_NODE_NAME + "/" + m_cstr_XML_CONTACT_GUID_ELEMENT);
				while (Iterator.MoveNext())
				{
					if (Iterator.Current.LocalName.ToLower().Trim() == m_cstr_XML_CONTACT_GUID_ELEMENT.ToLower().Trim())
					{
						return Iterator.Current.Value;
					}
				}
				return "";
			}
			catch
			{
				return "";
			}
		}

		#endregion
	
		#region saveContactGUID(string strGuid)
		/// <summary>
		/// Saves the GUID selected via the treeview to the placeholder.
		/// </summary>
		/// <remarks>Also stores display settings required in authoring mode.</remarks>
		/// <param name="strGuid">GUID to be stored.</param>
		private void saveContactGUID(string strGuid)
		{
			XmlDocument XmlSavedDoc = new XmlDocument();
			XmlElement XmlRoot = XmlSavedDoc.CreateElement("",_nodeName,"");
			XmlRoot.Attributes.Append(getAttribute(XmlSavedDoc, m_cstr_XML_ENABLED_ATTRIBUTE, chkEnable.Checked.ToString()));
			XmlRoot.Attributes.Append(getAttribute(XmlSavedDoc, m_cstr_PRES_CONTROL_TITLE_NODE_NAME, m_cstr_PRES_CONTROL_TITLE_VALUE));
			XmlRoot.Attributes.Append(getAttribute(XmlSavedDoc, m_cstr_XML_ATTRIBUTE_PRES_TITLE_CLASS, _presTitleClass));
			XmlRoot.Attributes.Append(getAttribute(XmlSavedDoc, m_cstr_XML_ATTRIBUTE_PRES_DETAILS_CLASS, _presDetailsClass));
			XmlRoot.AppendChild(getElement(XmlSavedDoc, m_cstr_XML_CONTACT_GUID_ELEMENT, strGuid));
			XmlElement XmlDisplayParams = XmlSavedDoc.CreateElement("", m_cstr_XML_SETTINGS_ELEMENT, "");
			CheckBox chkTemp;
			for (int iLoop = 0; iLoop<_chkBoxes.Count; iLoop++)
			{
				chkTemp = (CheckBox)_chkBoxes[iLoop];
				XmlDisplayParams.Attributes.Append(getAttribute(XmlSavedDoc, chkTemp.ID.Replace("chk_", ""), chkTemp.Checked.ToString()));
			}
			XmlRoot.AppendChild(XmlDisplayParams);
			XmlSavedDoc.AppendChild(XmlRoot);
			if (Mode == EnmContactMode.debugXML) 
			{
				try
				{
					XmlSavedDoc.Save(System.Web.HttpContext.Current.Server.MapPath(XMLDebugVirtualPath + "savePH.xml"));
				}
				catch (Exception exp)
				{
					HttpContext.Current.Trace.Warn("MSIB Contact control", "Error writing to XML debug file. Check permissions and that directory exists.", exp);
				}
			}
			this.BoundXmlPlaceholder.XmlAsString = XmlSavedDoc.OuterXml;		
		}

		#endregion
			
		#region getTemplatePlaceholders(Posting oPosting)

		/// <summary>
		/// Returns collection of placeholders of specified posting
		/// </summary>
		/// <param name="oPosting">Posting to query</param>
		/// <returns>PlaceholderCollection</returns>
		private PlaceholderCollection getTemplatePlaceholders(Posting oPosting)
		{
			return oPosting.Placeholders;
		}

		#endregion

		#region getFirstPosting functions - recursively finds first posting in a hierarchy
		
		#region getFirstPosting(string sChannelPath)

		/// <summary>
		/// Get the first posting in a channel hierarchy (recurses down until a posting is found)
		/// </summary>
		/// <param name="sChannelPath">The channel path to start from</param>
		/// <returns>First posting object found in hierarchy</returns>
		///<remarks>Returns null if error or unable to find a posting in the specified location</remarks>
		private Posting getFirstPosting(string sChannelPath)
		{
			try
			{
				Channel oStartChannel = (Channel)CmsHttpContext.Current.Searches.GetByPath(sChannelPath);
				return getFirstPosting(oStartChannel);				
			}
			catch
			{
				return null;
			}
		}


		
		#endregion

		#region getFirstPosting(Channel oChannel)
		
		/// <summary>
		/// Get the first posting in a channel hierarchy (recurses down until a posting is found)
		/// </summary>
		/// <param name="oChannel">The channel object to start from</param>
		/// <returns>First posting object found in hierarchy</returns>
		///<remarks>Returns null if error or unable to find a posting in the specified location</remarks>
		private Posting getFirstPosting(Channel oChannel)
		{
			try
			{
				Posting oPost = getFirstPostingFromChannel(oChannel);
				if (oPost == null)
				{
					foreach(Channel oSubChannel in oChannel.Channels)
					{					
						if (oPost != null)
						{
							return oPost;
						}
						else
						{
							oPost = getFirstPosting(oSubChannel);
						}
					}
				}
				return oPost;
			}
			catch
			{
				return null;
			}
		}

		#endregion

		#region getFirstPostingFromChannel(Channel oChannel)

		/// <summary>
		/// Get the first available posting in a channel, otherwise returns null
		/// </summary>
		/// <param name="oChannel">The channel to search</param>
		/// <returns>First posting object found in channel</returns>
		///<remarks>Returns null if error or unable to find a posting in the specified location</remarks>
		private Posting getFirstPostingFromChannel(Channel oChannel)
		{
			return (oChannel.Postings.Count > 0) ? oChannel.Postings[0] : null;
		}

#endregion

		#endregion
		
		#region Edit contact button click handler

		/// <summary>
		/// Event handler for click event of 'Edit contact' button.
		/// </summary>
		/// <param name="sender">Implemented in standard event delegate.</param>
		/// <param name="e">Implemented in standard event delegate.</param>
		private void btnEdit_Click(object sender, EventArgs e)
		{
			// get GUID of contact from placeholder..
			XmlDocument xmlData = new XmlDocument();
			string sPhXml = this.BoundXmlPlaceholder.XmlAsString;
			if (sPhXml.Length > 1)
			{
				xmlData.InnerXml = sPhXml;
				string sGuid = RetrieveGuid(xmlData);
				Posting oPosting = (Posting)CmsHttpContext.Current.Searches.GetByGuid(sGuid);
				if (oPosting != null)
				{
					HttpContext.Current.Response.Redirect(oPosting.UrlModeUnpublished);
				}
			}
		}

		#endregion
		
		#endregion

		#region Placeholder Overrides

		/// <summary>
		/// Creates child controls used in presentation mode. Override the 
		/// CreateContactAuthoringChildControls method to modify implementation.
		/// </summary>
		/// <param name="authoringContainer">Container to which child controls are added.</param>
		/// <remarks>Adds a Checkbox control for each placeholder in the template used for contacts
		/// to a class-scoped control array, in addition to Button and Label controls.</remarks>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			CreateContactAuthoringChildControls(authoringContainer);
		}

		/// <summary>
		/// Creates child controls used in presentation mode. Override the 
		/// CreateContactPresentationChildControls method to modify implementation.
		/// </summary>
		/// <param name="authoringContainer">Container to which child controls are added.</param>
		/// <remarks>Adds a Literal control.</remarks>
		protected override void CreatePresentationChildControls(BaseModeContainer authoringContainer)
		{
			CreateContactPresentationChildControls(authoringContainer);
		}

		/// <summary>
		/// Loads content from the placeholder in authoring mode. Override the 
		/// LoadContactPlaceholderContentForAuthoring method to modify implementation.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			LoadContactPlaceholderContentForAuthoring(e);
		}

		/// <summary>
		/// Loads content from the placeholder in presentation mode. Override the 
		/// LoadContactPlaceholderContentForPresentation method to modify implementation.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			LoadContactPlaceholderContentForPresentation(e);
		}

		/// <summary>
		/// Saves content to the underlying placeholder. Override the 
		/// SaveContactPlaceholderContent method to modify implementation.
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs class.</param>
		/// <remarks>This method is fired when a page hosting the control is saved.</remarks>
		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			SaveContactPlaceholderContent(e);
		}

		#endregion Placeholder Overrides

		#region Placeholder Override implementations

		#region CreateContactAuthoringChildControls

		/// <summary>
		/// This method is for use in derived classes. 
		/// Creates child controls used in Authoring mode. 
		/// </summary>
		/// <param name="authoringContainer">Container to add controls to.</param>
		/// <remarks>This method can be used to modify the implementation of adding 
		/// child controls in authoring mode.</remarks>
		protected virtual void CreateContactAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			Table tblAuthCtrls = new Table();
			tblAuthCtrls.BorderWidth = 0;
			tblAuthCtrls.Width = Unit.Percentage(100);
			tblAuthCtrls.CssClass = _authTableClass;
			TableRow tr = new TableRow(); // row 1
			
			TableCell tc = new TableCell(); // cell 1
			tc.ColumnSpan = 3;
			tc.CssClass = _authTitleClass;
			tc.Text = ControlTitle;
			tr.Cells.Add(tc);
			tblAuthCtrls.Rows.Add(tr);
			tr = new TableRow();
			tc = new TableCell();
			// if page currently has a contact assigned, populate our hidden textbox
			// with the GUID..
            XmlDocument xmlDoc = new XmlDocument();
			string sGuid = "";
			if (WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringReedit)
			{
				try
				{
					xmlDoc.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
					// get posting GUID..
					sGuid = RetrieveGuid(xmlDoc);
				}
				catch (Exception exp)
				{
					HttpContext.Current.Trace.Warn("MSIB Contact control", "Error retrieving GUID of stored contact Posting from placeholder.", exp);
				}
			}
			this.Page.RegisterHiddenField("txtSelectedPostGuid", sGuid);
			tr.Cells.Add(tc);
		
			tblAuthCtrls.Rows.Add(tr);

			tr = new TableRow();
			tc = new TableCell();
			tc.ColumnSpan = 2;
			tc.CssClass = _authDetailsClass;
			lblCurrContactText = new Label();
			lblCurrContactText.Text = _currContactText;
			tc.Controls.Add(lblCurrContactText);
			tr.Cells.Add(tc);
			tblAuthCtrls.Rows.Add(tr);
			tr = new TableRow();
			tc = new TableCell();
			tc.ColumnSpan = 2;
			Table tblInner = new Table();
			TableRow trInner = new TableRow();
			TableCell tcInner = new TableCell();
			tcInner.Attributes.Add("width", "70%");
			//tc.ColumnSpan = 1;
			lblSelectedPostPath = new Label();
			lblSelectedPostPath.Attributes.Clear();
			lblSelectedPostPath.CssClass = _authContactNameClass;
			lblSelectedPostPath.Attributes["id"] = "lblSelectedPostPath";
			lblSelectedPostPath.Attributes["name"] = "lblSelectedPostPath";
			tcInner.Controls.Add(lblSelectedPostPath);
			trInner.Cells.Add(tcInner);
			// new
			tcInner = new TableCell();
			btnEdit = new Button();
			btnEdit.Enabled = false;
			btnEdit.ID = "btnEdit";
			btnEdit.Text = m_cstrEDIT_BUTTON_TEXT;
			btnEdit.Click += new EventHandler(btnEdit_Click);
			tcInner.Controls.Add(btnEdit);
			trInner.Cells.Add(tcInner);
			tblInner.Rows.Add(trInner);
			tc.Controls.Add(tblInner);
			//end..
			tr.Cells.Add(tc);
			tblAuthCtrls.Rows.Add(tr);
			// add literal for treeview..
			tr = new TableRow();
			tc = new TableCell();
			tc.ColumnSpan = 2;
			litTree = new Literal();
			tc.Controls.Add(litTree);
			tr.Cells.Add(tc);
            tblAuthCtrls.Rows.Add(tr);
			tr = new TableRow();
			tc = new TableCell();	
			tc.ColumnSpan = 2;
			tc.CssClass = _authDetailsClass;
			chkEnable = new CheckBox();
			// add client-side event handler for toggle checkboxes..
			chkEnable.Attributes.Add("onclick", "toggleCheckBoxes()");
			chkEnable.Text = _checkBoxText;
			tc.Controls.Add(chkEnable);
			tr.Cells.Add(tc);
			tblAuthCtrls.Rows.Add(tr);
			// now add checkboxes for which fields should be visible in presentation..
			// these are generated from a template used by a contact posting..
			Posting oFirstPost = getFirstPosting(_startChannelPath);
			if (oFirstPost != null)
			{
				CheckBox chkTemp;
				Table tblDisplayElms = new Table();
				tblDisplayElms.BorderWidth = 0;
				tblDisplayElms.Width = Unit.Percentage(100);
				PlaceholderCollection oPHCol = getTemplatePlaceholders(oFirstPost);
				for (int iLoop=0; iLoop<oPHCol.Count; iLoop++)
				{
					_excludePlaceholders = _excludePlaceholders.Replace(" ", "").ToLower();
					string[] sExcludePlaceholders = _excludePlaceholders.Split(',');
					string sTempPHName = "";
					bool bUsePlaceholder = true;
					for (int iLoop2=0; iLoop2<sExcludePlaceholders.Length; iLoop2++)
					{
						sTempPHName = oPHCol[iLoop].Name;
						if (sTempPHName.ToLower() == sExcludePlaceholders[iLoop2])
						{
							bUsePlaceholder = false;
						}
					}
					if (bUsePlaceholder)
					{
						tr = new TableRow();
						tc = new TableCell();
						chkTemp = new CheckBox();
						chkTemp.Text = sTempPHName;
						chkTemp.ID = "chk_" + sTempPHName;
						// now add to arrayList
						_chkBoxes.Add(chkTemp);
						tc.Controls.Add(chkTemp);
						tr.Cells.Add(tc);
						tblDisplayElms.Rows.Add(tr);
					}
				}
				tr = new TableRow();
				tc = new TableCell();	
				tc.Width = Unit.Percentage(5);
				tr.Cells.Add(tc);
				tc = new TableCell();
				tc.Controls.Add(tblDisplayElms);
				tr.Cells.Add(tc);
				tblAuthCtrls.Rows.Add(tr);
			}
			else
			{
				tr = new TableRow();
				tc = new TableCell();	
				Literal litError = new Literal();
				litError.Text = "<br>Error: no postings found - check StartChannel value";
				tc.Controls.Add(litError);
				tr.Cells.Add(tc);
				tblAuthCtrls.Rows.Add(tr);
			}
			try
			{
				authoringContainer.Controls.Add(tblAuthCtrls);
			}
			catch (Exception exp)
			{
				HttpContext.Current.Trace.Write("MSIB Contact control", "Error adding table to control tree.", exp);
			}
		}

		#endregion CreateContactAuthoringChildControls

		#region CreateContactPresentationChildControls()
		
		/// <summary>
		/// This method is for use in derived classes. 
		/// Creates child controls used in presentation mode.
		/// </summary>
		/// <param name="presentationContainer">Container to add controls to.</param>
		/// <remarks>This method can be used to modify the implementation of adding 
		/// child controls in presentation mode.</remarks>
		protected virtual void CreateContactPresentationChildControls(BaseModeContainer presentationContainer)
		{
			EnsureChildControls();
			contactInfo = new Literal();
			try
			{
				presentationContainer.Controls.Add(contactInfo);
			}
			catch (Exception exp)
			{
				HttpContext.Current.Trace.Write("MSIB Contact control", "Error adding 'contactInfo' literal to control tree.", exp);
			}
		}

		#endregion

		#region LoadContactPlaceholderContentForAuthoring()
		/// <summary>
		/// This method is for use in derived classes. 
		/// Builds placeholder content when page is loaded in Authoring mode.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		/// <remarks>This method can be used to modify the implementation of loading placeholder 
		/// content in authoring mode. Currently emits client-side script code e.g. to control treeview.</remarks>
		protected virtual void LoadContactPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			EnsureChildControls();
			writeTreeHTML();
			// insert JavaScript functions to control treeview/population of hidden text field..
			StringBuilder sbJSListFunction = new StringBuilder("", 600);
			sbJSListFunction.Append("<script language=JavaScript>function Expand(strSpan) { ");
			sbJSListFunction.Append("var strPlusImgSrc = '" + _listPlusImgPath + "';"); 
			sbJSListFunction.Append("var strMinusImgSrc = '" + _listMinusImgPath + "';"); 
			sbJSListFunction.Append("if (document.all[strSpan].style.display == 'inline' || document.all[strSpan].style.display == '' || document.all[strSpan].style.display == 'block' ) { ");
			sbJSListFunction.Append("document.all[strSpan].style.display = 'none'; document.all['img' + strSpan].src = strPlusImgSrc; ");
			sbJSListFunction.Append("} else { ");
			sbJSListFunction.Append("document.all(strSpan).style.display = 'inline'; document.all['img' + strSpan].src = strMinusImgSrc; }} <");
			sbJSListFunction.Append("/script>");

			if(!this.Page.IsClientScriptBlockRegistered("listController"))
			{
				this.Page.RegisterClientScriptBlock("listController", sbJSListFunction.ToString());
			}

			StringBuilder sbJSInsertFunction = new StringBuilder("", 400);
			sbJSInsertFunction.Append("<script language=JavaScript> function insertGuid(strGuid, strDisplayName) {");
			sbJSInsertFunction.Append("document.forms[0].txtSelectedPostGuid.value = strGuid;");
			sbJSInsertFunction.Append("document.all.lblSelectedPostPath.innerText = strDisplayName; }<");
			
			sbJSInsertFunction.Append("/script>");

			if(!this.Page.IsClientScriptBlockRegistered("populateHidden"))
			{
				this.Page.RegisterClientScriptBlock("populateHidden", sbJSInsertFunction.ToString());
			}
			
			// build string of all checkbox client IDs to pass to JS..
			string sAllCheckBoxes = "";
			CheckBox chkTemp2;
			string sTempID;
			for(int iLoop=0; iLoop<_chkBoxes.Count; iLoop++)
			{
				chkTemp2 = (CheckBox)_chkBoxes[iLoop];
				sTempID = chkTemp2.ClientID;
				sAllCheckBoxes += sTempID + ",";
			}
			// all checkbox client IDs (comma-separated) now stored in server side sAllCheckBoxes..
			// -- now pass to client-side --
			StringBuilder sbJSCheckboxDefaults = new StringBuilder("", 600);
			sbJSCheckboxDefaults.Append("<script language=JavaScript> ");
			sbJSCheckboxDefaults.Append("var strAllCheckBoxClientIDs = '" + sAllCheckBoxes + "';");
			sbJSCheckboxDefaults.Append("var re = new RegExp (' ', 'gi');");
			sbJSCheckboxDefaults.Append("strAllCheckBoxClientIDs = strAllCheckBoxClientIDs.replace(re, ''); ");
			sbJSCheckboxDefaults.Append("strAllCheckBoxClientIDs = strAllCheckBoxClientIDs.split(','); ");
			sbJSCheckboxDefaults.Append("<");
			sbJSCheckboxDefaults.Append("/script>");

			if(!this.Page.IsClientScriptBlockRegistered("defaultChecks"))
			{
				this.Page.RegisterClientScriptBlock("defaultChecks", sbJSCheckboxDefaults.ToString());
			}
			
			// -- now emit function..
			StringBuilder sbJSCheckboxFunction = new StringBuilder("", 600);
			sbJSCheckboxFunction.Append("<script language=JavaScript>function toggleCheckBoxes() {");
			sbJSCheckboxFunction.Append("for (i=0; i<strAllCheckBoxClientIDs.length-1; i++) { ");
			sbJSCheckboxFunction.Append("if (document.forms[0]." + chkEnable.ClientID + ".checked == true) {");
			sbJSCheckboxFunction.Append("document.forms[0][strAllCheckBoxClientIDs[i]].disabled = false; ");
			sbJSCheckboxFunction.Append("} else {");
			sbJSCheckboxFunction.Append("document.forms[0][strAllCheckBoxClientIDs[i]].disabled = true; ");
			sbJSCheckboxFunction.Append("} } }<");
			sbJSCheckboxFunction.Append("/script>");

			if(!this.Page.IsClientScriptBlockRegistered("toggleChecks"))
			{
				this.Page.RegisterClientScriptBlock("toggleChecks", sbJSCheckboxFunction.ToString());
			}

			string strJSCallToggleFunction = "<script language=JavaScript>toggleCheckBoxes(); <";
			strJSCallToggleFunction += "/script>";
			if(!this.Page.IsClientScriptBlockRegistered("callToggle"))
			{
				this.Page.RegisterStartupScript("callToggle", strJSCallToggleFunction);
			}

			// now perform CMS mode-specific actions
			WebAuthorContext webAuthorContext = WebAuthorContext.Current;
			try
			{
				switch (webAuthorContext.Mode)
				{
					case WebAuthorContextMode.AuthoringReedit:
						bool bSetVisibleFields = false;
						XmlDocument xmlData = new XmlDocument();
						string sPhXml = this.BoundXmlPlaceholder.XmlAsString;
						if (sPhXml.Length>0)
						{
							xmlData.LoadXml(sPhXml);
							if (Mode == EnmContactMode.debugXML) 
							{
								try
								{
									xmlData.Save(System.Web.HttpContext.Current.Server.MapPath(XMLDebugVirtualPath + "loadph_auth.xml"));
								}
								catch (Exception exp)
								{
									HttpContext.Current.Trace.Warn("MSIB Contact control", "Error writing to XML debug file. Check permissions and that directory exists.", exp);
								}
							}
							XPathNavigator xPath = xmlData.CreateNavigator();
							// deal with root element first..
							xPath.MoveToFirstChild();
							if (xPath.LocalName == m_cstr_NODE_NAME)
							{
								if (xPath.GetAttribute(m_cstr_XML_ENABLED_ATTRIBUTE, "").ToString().ToLower() == "true")
								{
									chkEnable.Checked = true;
									bSetVisibleFields = true;
								}
							}
							// now loop through child nodes..
							xPath.MoveToFirstChild();
							do 
							{
								try
								{
									switch(xPath.LocalName)
									{
										case m_cstr_XML_CONTACT_GUID_ELEMENT:
											if (xPath.Value.Length>0)
											{
												Posting oCurrentContact = (Posting)CmsHttpContext.Current.Searches.GetByGuid(xPath.Value);
												if (oCurrentContact != null)
												{
													try
													{
														_bPageHasContact = true;
														btnEdit.Enabled = true;
														HtmlPlaceholder tempPH = (HtmlPlaceholder)oCurrentContact.Placeholders[_authDisplayField];
														lblSelectedPostPath.Text = tempPH.Text;
													}
													catch (Exception exp)
													{
														lblSelectedPostPath.Text = oCurrentContact.DisplayName;
														HttpContext.Current.Trace.Warn("MSIB Contact control", "Unable to find placeholder with name '" + _authDisplayField + "' in contact template. Using DisplayName instead.", exp);
													}
													if (oCurrentContact.State != PostingState.Published)
													{
														lblSelectedPostPath.Text += " " + _contactNotPublishedText;
													}
												}
												else
												{
													lblSelectedPostPath.Text = "Unable to find page for selected contact";
												}
											}
											else
											{
												lblSelectedPostPath.Text = m_cstr_AUTH_NO_CONTACT_SELECTED_TEXT;
											}
											break;
										case m_cstr_XML_SETTINGS_ELEMENT:
											if (bSetVisibleFields) 
											{
												CheckBox chkTemp;
												string sCleanedChkID;
												xPath.MoveToFirstAttribute();
												do 
												{
													for(int iLoop = 0; iLoop<_chkBoxes.Count; iLoop++)
													{
														chkTemp = (CheckBox)_chkBoxes[iLoop];
														// remove checkbox naming prefix..
														sCleanedChkID = chkTemp.ID.Replace("chk_", "");
														if (xPath.LocalName.ToLower() == sCleanedChkID.ToLower())
														{
															if (xPath.Value.ToLower() == "true")
															{
																chkTemp.Checked = true;
															}
															break;
														}
													}
												}
												while (xPath.MoveToNextAttribute());
											}
											break;
									}
								}
								catch (Exception exp)
								{
									HttpContext.Current.Trace.Warn("MSIB Contact control", "Error reading XML structure.", exp);
								}
							}
							while (xPath.MoveToNext());
						}
						else
						{
							lblSelectedPostPath.Text = m_cstr_AUTH_NO_CONTACT_SELECTED_TEXT;
						}
						break;
					case WebAuthorContextMode.AuthoringNew:
						//set defaults for new pages..
						lblSelectedPostPath.Text = m_cstr_AUTH_NO_CONTACT_SELECTED_TEXT;
						if (_bDefaultVisibilityInPresentation)
						{
							chkEnable.Checked = true;
							string sDefaultChecks = _defaultCheckedCheckboxes.Replace(" ", "");
							string[] sArrDefaultChecks = sDefaultChecks.Split(',');
							for (int iLoop=0; iLoop<sArrDefaultChecks.Length; iLoop++)
							{
								// do we have a corresponding checkbox?
								for (int iChkLoop=0; iChkLoop<_chkBoxes.Count; iChkLoop++)
								{
									CheckBox chkTemp = (CheckBox)_chkBoxes[iChkLoop];
									// remove checkbox naming prefix..
									String sCleanedChkID = chkTemp.ID.Replace("chk_", "");
									if (sCleanedChkID.ToLower() == sArrDefaultChecks[iLoop].ToLower())
									{
										chkTemp.Checked = true;
									}
								}
							}
						}
						break;
					default:

						//xmlData.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
						break;
				}
			}
			catch (Exception exp)
			{
				if ((Mode == EnmContactMode.debug) || (Mode == EnmContactMode.debugXML))
				{
					litTree.Text += "<br><br>Error loading placeholder content<br><br>" + exp.Message;
					HttpContext.Current.Trace.Warn("MSIB Contact control", "Error retrieving data from contact Posting.", exp);
				}
				else
				{
					HttpContext.Current.Trace.Warn("MSIB Contact control", "Error loading placeholder content.", exp);
				}
			}
		}

		#endregion LoadContactPlaceholderContentForAuthoring()

		#region LoadContactPlaceholderContentForPresentation()
		/// <summary>
		/// This method is for use in derived classes. 
		/// Builds placeholder content when page is loaded in presentation mode.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		/// <remarks>This method can be used to modify the implementation of loading placeholder 
		/// content in presentation mode. This method performs a posting look-up using Searches on the GUID stored in the 
		/// bound XML placeholder. Outputs details based on display fields selected in authoring mode.</remarks>
		private void LoadContactPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			string sEnabled = "";
			EnsureChildControls();
			contactInfo.Visible = false;
			XmlDocument xmlData = new XmlDocument();
			xmlData.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
			if (Mode == EnmContactMode.debugXML) 
			{
				try
				{
					xmlData.Save(System.Web.HttpContext.Current.Server.MapPath(XMLDebugVirtualPath + "loadph_present.xml"));
				}
				catch (Exception exp) 
				{
					HttpContext.Current.Trace.Warn("MSIB Contact control", "Error writing to XML debug file. Check permissions and that directory exists.", exp);
				}
			}
			XmlNode xmlNode = xmlData.SelectSingleNode(m_cstr_NODE_NAME);
			sEnabled = xmlNode.Attributes.Item(0).Value;
			if (sEnabled.ToLower() == "true")
			{
				//need to test that at least one field has been chosen for display
				//i.e. we don't want any output in presentation mode if author has selected
				//control visibility to true, but not selected any fields for display..
				bool bFieldChosen = false;
				xmlNode = xmlData.SelectSingleNode("//" + m_cstr_XML_SETTINGS_ELEMENT);
				if (xmlNode.Attributes.Count>0)
				{
					XPathNavigator xNav = xmlNode.CreateNavigator();
					xNav.MoveToFirstAttribute();
					while (xNav.MoveToNextAttribute())
					{
						if (xNav.Value.ToLower() == "true")
						{
							bFieldChosen = true;
							break;
						}
					}
				}
				
				if (bFieldChosen)
				{
					try
					{
						// get posting GUID..
						string sGUID = RetrieveGuid(xmlData);
						if (sGUID.Length > 0)
						{
							Posting oContactPost = (Posting)CmsHttpContext.Current.Searches.GetByGuid(sGUID);
							if (oContactPost != null)
							{
								_bPageHasContact = true;
								string sTitleClass = "";
								string sDetailsClass = "";
								string sTableClass = "";
								
								sTitleClass = _presTitleClass;
								sDetailsClass = _presDetailsClass;
								sTableClass = _presTableClass;
								
								XmlDocument xmlDoc = new XmlDocument();
								XmlElement xmlRoot = xmlDoc.CreateElement("",_nodeName,"");
								// insert settings node from previous XmlDocument - represents fields to
								// be displayed in presentation mode
								XmlNode xmlSettings = xmlDoc.ImportNode(xmlData.SelectSingleNode("//" + m_cstr_XML_SETTINGS_ELEMENT), true);
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_PRES_CONTROL_TITLE_NODE_NAME, m_cstr_PRES_CONTROL_TITLE_VALUE));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_PRES_TABLE_CLASS, sTableClass));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_PRES_TITLE_CLASS, sTitleClass));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_PRES_DETAILS_CLASS, sDetailsClass));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_PHONE_IMAGE_PATH, _phoneImgPath));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_MAIL_IMAGE_PATH, _mailImgPath));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_LINE_IMAGE_PATH, _lineImgPath));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_LINK_TEXT, HttpContext.Current.Server.HtmlEncode(_presentationLinkText)));
								xmlSettings.Attributes.Append(getAttribute(xmlDoc, m_cstr_XML_ATTRIBUTE_LINK_SYMBOL, HttpContext.Current.Server.HtmlEncode(_presentationHyperlinkSymbol)));
						
								xmlRoot.AppendChild(xmlSettings);
								xmlRoot.AppendChild((searchPost(xmlDoc, oContactPost)));
								xmlDoc.AppendChild(xmlRoot);
								if (Mode == EnmContactMode.debugXML) 
								{
									try
									{
										xmlDoc.Save(System.Web.HttpContext.Current.Server.MapPath(XMLDebugVirtualPath + "xml_for_trans.xml"));
									}
									catch (Exception exp)
									{
										HttpContext.Current.Trace.Warn("MSIB Contact control", "Error writing to XML debug file. Check permissions and that directory exists.", exp);
									}
								}
								string strHTML = transformXML(xmlDoc.OuterXml, _xslContactPresentation);
								// decode necessary here..
								string strCleanedHTML = HttpContext.Current.Server.HtmlDecode(strHTML);
								contactInfo.Text = strCleanedHTML;
								contactInfo.Visible = true;
							}
						}
					
					}
					catch (Exception except)
					{
						if (this.Mode == EnmContactMode.production)
						{
							HttpContext.Current.Trace.Warn("MSIB Contact control", "Error retrieving data from contact Posting.", except);
						}
						else
						{
							contactInfo.Text = "Error - <br>" + except.Message + "<br>" + except.Source;
							HttpContext.Current.Trace.Warn("MSIB Contact control", "Error retrieving data from contact Posting.", except);
							contactInfo.Visible = true;
						}
					}
				}
			}
		}
		#endregion LoadContactPlaceholderContentForPresentation

		#region SaveContactPlaceholderContent()
	
		/// <summary>
		/// This method is for use in derived classes. 
		/// Saves placeholder content in authoring mode.
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs class.</param>
		/// <remarks>This method can be used to modify the implementation of saving content 
		/// to the underlying placeholder.</remarks>
		protected virtual void SaveContactPlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			// Save content back into placeholder
			EnsureChildControls();  
			string strSelectedGuid = this.Page.Request.Form["txtSelectedPostGuid"];
         	if (strSelectedGuid.Length>0)
			{
				// postings have been successfully retrieved and one is selected
				Posting oContactPost = (Posting)CmsHttpContext.Current.Searches.GetByGuid(strSelectedGuid);
				if (oContactPost != null)
				{
					saveContactGUID(strSelectedGuid);
				}
			}
		}

		#endregion SaveContactPlaceholderContent()
	
		#endregion Placeholder Override implementations

		#region CMS-related XML Builder Functions (XML built from CMS object properties)

		#region searchChannel(XmlDocument xmlDoc, Channel cmsChannel)
		/// <summary>
		/// This method calls the searchPost() method for each posting in the supplied 
		/// channel.
		/// </summary>
		/// <param name="xmlDoc">XmlDocument to add XML to.</param>
		/// <param name="cmsChannel">Channel to search.</param>
		/// <returns>XmlElement.</returns>
		private XmlElement searchChannel(XmlDocument xmlDoc, Channel cmsChannel)
		{

			XmlElement xmlChannel = getChannelData(xmlDoc, cmsChannel);
			
			//Now we loop the posts in this channel and add them
			foreach(Posting cmsPost in cmsChannel.Postings)
			{
				xmlChannel.AppendChild(searchPost(xmlDoc, cmsPost));
			}
			
			return xmlChannel;

		}
		#endregion

		#region searchPost(XmlDocument xmlDoc, Posting cmsPost)
		/// <summary>
		/// This function build the XmlElement that represents a posting. 
		/// Each posting element has the following attributes:-<br/><br/>
		///	  
		///	name			= Posting.Name
		///	displayname		= Posting.DisplayName
		///	url				= Posting.Url
		///	guid			= Posting.GUID
		///	<br/><br/>  
		///	It also contains a collection of placeholder nodes.
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create elements.</param>
		/// <param name="cmsPost">Posting to be searched.</param>
		/// <returns>XmlElement representing the posting.</returns>
		private XmlElement searchPost(XmlDocument xmlDoc, Posting cmsPost)
		{
			
			XmlElement xmlReturn = getPostData(xmlDoc, cmsPost);

			if(_startChannelPath.Length > 0)
			{
				//Now we loop the placeholders
				foreach(Placeholder cmsPH in cmsPost.Placeholders)
				{
					XmlElement xElm = getPlaceholderData(xmlDoc, cmsPH);
					if (xElm != null)
					{
						xmlReturn.AppendChild(xElm);
					}
				}
			}
			return xmlReturn;
		}
		#endregion

		#region getPostData

		/// <summary>
		/// This function is responsible for gathering post property values.
		/// </summary>
		/// <param name="xmlDoc">XmlDocument object used to create XmlElements.</param>
		/// <param name="cmsPost">Posting object.</param>
		/// <returns>XmlElement.</returns>
		private XmlElement getPostData(XmlDocument xmlDoc, Posting cmsPost)
		{
			XmlElement xmlReturn;
			if ((cmsPost.Name.ToLower().IndexOf(DepartmentPostingName.ToLower()) > -1) && (WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringNew
					|| WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringReedit))
			{
				xmlReturn = xmlDoc.CreateElement("",m_cstrXML_NODE_DEPARTMENT_POST,"");
			}
			else
			{
				xmlReturn = xmlDoc.CreateElement("",m_cstrXML_NODE_POST,"");
			}

			#region Gather the default post values and assign them as attributes
			//Posting Name
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.name.ToString(), cmsPost.Name));
			//Display Name
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.displayname.ToString(), cmsPost.DisplayName));
			//URL
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.url.ToString(), cmsPost.Url));
			//GUID
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.guid.ToString(), cmsPost.Guid));
			#endregion

			return xmlReturn;

		}
		#endregion
		
		#region getPlaceholderData(XmlDocument xmlDoc, Placeholder cmsPH)
		/// <summary>
		/// Takes the placeholder name and raw data and creates an element.
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create the Element.</param>
		/// <param name="cmsPH">The placeholder to query.</param>
		/// <returns>XmlElement.</returns>
		private XmlElement getPlaceholderData(XmlDocument xmlDoc, Placeholder cmsPH)
		{
			XmlElement xmlPH = xmlDoc.CreateElement("", m_cstrXML_NODE_PLACEHOLDER, "");
			xmlPH.SetAttribute(m_cstrXML_NODE_NAME, cmsPH.Name);
			
			#region Add the attributes

			Type phType = cmsPH.GetType();
			
			string sValue;
			if (phType.UnderlyingSystemType.ToString().IndexOf("Image") > 0)
			{
				sValue = ((ImagePlaceholder)cmsPH).Src;
			}
			else if (phType.UnderlyingSystemType.ToString().IndexOf("Html") > 0)
			{
				// use HTML rather than RawContent so that placeholder content is HtmlEncoded etc..
				// also need to deal with different ways a page author could enter data
				// into placeholders:
				// 1. E-mail addresses could be mailto: link or plain text. Since our XSLT is
				//    responsible for supplying mailto: wrapping, we strip it out here 
				//	  ready for this.
				// 2. URLs could be in <a> tag or plain text. Again, wrapping in <a> tag is 
				//	  performed by XSLT so we strip out here.
				
				HtmlPlaceholder cmsHtmlPH = (HtmlPlaceholder)cmsPH;
				sValue = cmsHtmlPH.Html;
				string sTextValue = cmsHtmlPH.Text;
				if (!hasRealData(sTextValue))
				{
					return (null);
				}
				// deal with e-mail addresses..
				int iPosMailToString = sValue.IndexOf("mailto:") > 0 ? sValue.IndexOf("mailto:") : 0;
				if (iPosMailToString > 0) 
				{
					int iPosLastQuote = sValue.LastIndexOf("\"");
					// 7 refers to length of "mailto:" string..
					sValue = sValue.Substring(iPosMailToString + 7, iPosLastQuote - (iPosMailToString + 7));
					
					if (sValue.LastIndexOf("\\") > 0)
					{
						sValue = sValue.Replace("\\", "");
					}
				}
				// deal with hyperlinks etc..
				if ((sValue.ToLower().IndexOf("href", 0, sValue.Length) > -1) || 
					(sValue.ToLower().IndexOf("www.", 0, sValue.Length) > -1))
				{
					int iPosHrefString = sValue.IndexOf("href") > 0 ? sValue.IndexOf("href") : 0;
					if (iPosHrefString > 0)
					{
						// link is wrapped in anchor tag..
						int iPosFirstQuote = sValue.IndexOf("href=\"") + 5;
						int iPosLastQuote = sValue.IndexOf("\"", iPosFirstQuote + 1);
						sValue = sValue.Substring(iPosFirstQuote + 1, iPosLastQuote - iPosFirstQuote - 1);
					}
					else
					{
						// first let's clean any <p> tags which are present..
						sValue = sValue.ToLower().Replace("<p>", "");
						sValue = sValue.ToLower().Replace("</p>", "");

						// we will manually add 'http://' prefix here to make absolute link. This 
						// avoids problems if a base href is set on the page..
						if ((sValue.ToLower().IndexOf("http://", 0, sValue.Length) == -1) && 
						(sValue.ToLower().IndexOf("https://", 0, sValue.Length) == -1))
						{
							// assume author would have specified https:// prefix if it was required..
							sValue = "http://" + sValue;
						}
					}
				}
			}
			else
			{
				sValue = cmsPH.Datasource.RawContent;
			}
			//perform replace operations if required..
			if (_replaceList.Length>0)
			{	
				string[] sReplaceList = _replaceList.Split(',');
				string[] sReplaceValues = _replaceValues.Split(',');
				for (int iLoop=0; iLoop<sReplaceList.Length; iLoop++)
				{
					sValue = sValue.Replace(sReplaceList[iLoop], sReplaceValues[iLoop]);
				}
			}
			
			xmlPH.AppendChild(getElement(xmlDoc, m_cstrXML_NODE_VALUE, sValue));
			
			if (sValue.Length <= 1)
			{
				return (null);
			}

			#endregion

			return xmlPH;
		}
		#endregion

		#region getChannelData(XmlDocument xmlDoc, Channel cmsChannel)
		/// <summary>
		/// Takes the channel name and raw data and creates an element.
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create the Element.</param>
		/// <param name="cmsChannel">The channel to query.</param>
		/// <returns>XmlElement.</returns>
		private XmlElement getChannelData(XmlDocument xmlDoc, Channel cmsChannel)
		{

			#region Create Channel Fragment, and name attribute

			XmlElement xmlReturn = xmlDoc.CreateElement("", m_cstr_XML_NODE_CHANNEL, "");

			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "name", cmsChannel.DisplayName.Replace("&", "&amp;")));

			//guid
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "guid", cmsChannel.Guid));

			// Url
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "url", cmsChannel.UrlModePublished));

			// Posting Count
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "postcount", cmsChannel.Postings.Count.ToString()));
			
			#endregion	

			return xmlReturn;

		}

#endregion

		#region getChannelPostOnlyXML(XmlDocument xmlDoc)

		/// <summary>
		/// If <code>_startChannelPath</code> is valid channel, builds XML to represent 
		/// hierarchy of contact postings. 
		/// </summary>
		/// <param name="xmlDoc">XmlDocument to add XML to.</param>
		/// <remarks>Error message will be output in Authoring mode 
		/// if <code>_startChannelPath</code> is not valid.</remarks>
		/// <returns>XmlElement.</returns>
		private XmlElement getChannelPostOnlyXML(XmlDocument xmlDoc)
		{
			Channel oStartChannel = getChannel(_startChannelPath);
			XmlElement xmlReturn;
			if (oStartChannel != null)
			{
				xmlReturn = buildChannelPostOnlyXML(xmlDoc, oStartChannel, 0);
			}
			else
			{
				if (WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringNew
					|| WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringReedit)
				{
					litTree.Text = "Error: Specified location for contacts is not a channel";
				}
				xmlReturn = xmlDoc.CreateElement("error");
				xmlReturn.Value = "Specified location for contacts is not a channel";
			}
			return xmlReturn;
		}

		#endregion

		/// <summary>
		/// Builds XML representing posting data.
		/// </summary>
		/// <remarks><code>&lt;p&gt;</code> and <code>&lt;br&gt;</code> tags are cleaned from 
		/// placeholder content ready for transformation.</remarks>
		/// <returns>XmlElement.</returns>
		private XmlElement buildPostXML(XmlDocument xmlDoc, Posting oPosting)
		{
			XmlElement xmlPostElement = getPostData(xmlDoc, oPosting);
			XmlElement xmlPHContent;
			try
			{
				HtmlPlaceholder oSelPH = (HtmlPlaceholder)oPosting.Placeholders[_authDisplayField];
				// here we want to use the Text property of the placeholder so that our
				// treeview can't be corrupted. However, we need to still allow some formatting 
				// in presentation mode, so we replace the "\r\n" inserted by a break in the 
				// placeholder with a ", "
				string sPHText = oSelPH.Text;

				sPHText = sPHText.Replace("\r", ", ");
				sPHText = sPHText.Replace("\n", "");
				// add element for selected placeholder content..
				xmlPHContent = getElement(xmlDoc, m_cstr_XML_SELECTED_PH_ELEMENT, sPHText);
				xmlPostElement.AppendChild(xmlPHContent);
			}
			catch
			{
				xmlPHContent = getElement(xmlDoc, m_cstr_XML_SELECTED_PH_ELEMENT, oPosting.DisplayName);
				xmlPostElement.AppendChild(xmlPHContent);
			}
			return xmlPostElement;
		}

		/// <summary>
		/// Builds XML representing channel data.
		/// </summary>
		/// <remarks>A 'level' attribute is adding to represent depth of channel from start channel.</remarks>
		/// <returns>XmlElement.</returns>
		private XmlElement buildChannelXML(XmlDocument xmlDoc, Channel oChannel, int iLevel)
		{
			XmlElement xmlReturn = getChannelData(xmlDoc, oChannel);
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "level", iLevel.ToString()));
			return xmlReturn;
		}

		#region buildChannelPostOnlyXML(XmlDocument xmlDoc, Channel oChannel, int iLevel)

		/// <summary>
		/// Builds XML to represent hierarchy of contact postings.
		/// </summary>
		/// <remarks>XML contains data on channels and postings.</remarks>
		/// <returns>XmlElement.</returns>
		private XmlElement buildChannelPostOnlyXML(XmlDocument xmlDoc, Channel oChannel, int iLevel)
		{
			// add channel data..
			XmlElement xmlReturn = buildChannelXML(xmlDoc, oChannel, iLevel);
			// add posting data..
			foreach(Posting oPosting in oChannel.Postings)
			{
				// filter unapproved postings if required..
				if (_bPublishedPostingsOnly) 
				{
					if (oPosting.State == PostingState.Published)
					{
						xmlReturn.AppendChild(buildPostXML(xmlDoc, oPosting));
					}
				}
				else
				{
					xmlReturn.AppendChild(buildPostXML(xmlDoc, oPosting));
				}
			}
			//recurse through sub-channels..
			if (oChannel.Channels.Count > 0)
			{
				iLevel++;
				foreach(Channel oSubchannel in oChannel.Channels)
				{
					xmlReturn.AppendChild(buildChannelPostOnlyXML(xmlDoc, oSubchannel, iLevel));
				}	
			}
			return xmlReturn;
		}

		#endregion
		
		#endregion

		#region Generic XML builder functions

		#region getAttribute(XmlDocument xmlDoc, string sName, string sValue)
		/// <summary>
		/// Create and returns and XmlAttribute from the passed parameters
		/// </summary>
		/// <param name="xmlDoc">XmlDocument.</param>
		/// <param name="sName">String - then name of the attribute.</param>
		/// <param name="sValue">String - the value.</param>
		/// <returns>XmlAttribute.</returns>
		private XmlAttribute getAttribute(XmlDocument xmlDoc, string sName, string sValue)
		{
			XmlAttribute xmlReturnAttribute = xmlDoc.CreateAttribute(sName);

			xmlReturnAttribute.Value = sValue;

			return xmlReturnAttribute;
		}
		#endregion

		#region XmlElement getElement(XmlDocument xmlDoc, string sName, string sValue)
		/// <summary>
		/// Returns an XmlElement with the node name as requested and the value. 
		/// </summary>
		/// <param name="xmlDoc">XmlDocument - used to create the element.</param>
		/// <param name="sName">String - name of the element node.</param>
		/// <param name="sValue">String - the node content.</param>
		/// <returns>XmlElement.</returns>
		private XmlElement getElement(XmlDocument xmlDoc, string sName, string sValue)
		{
			XmlElement xmlReturn = xmlDoc.CreateElement("", sName, "");

		

			// sValue may be placeholder content so should be HTMLEncoded but 
			// also need to deal with single quotes for JavaScript..
			sValue = HttpUtility.HtmlEncode(sValue);
			

			try
			{
				xmlReturn.InnerXml = sValue;
			}
			catch
			{
				xmlReturn.InnerText = sValue;
			}

			return xmlReturn;

		}
		#endregion

		#endregion

		#region Generic XSL functions

		#region XSL Transform of result 
		/// <summary>
		/// This method transforms the supplied XML string using the XSL stylesheet at the path supplied.
		/// </summary>
		/// <param name="strXML">A string of XML data to be transformed.</param>
		/// <param name="strXSLFile">A string containing the virtual path 
		/// to the XSL stylesheet to use.</param>
		/// <returns>A string of HTML representing the transformed result set.</returns>
		/// <remarks>In the event of an error the function returns a string representing an XML node 
		/// containing details of the error.</remarks>
		private static string transformXML(string strXML, string strXSLFile)
		{

			StringBuilder sbReturn = new StringBuilder();

			//Get a mapped path to the style sheet
			strXSLFile = HttpContext.Current.Server.MapPath(strXSLFile);

			StringReader sReader = new StringReader(strXML);

			StringWriter sWriter = new StringWriter(sbReturn);

			XmlTextWriter xmlWriter = new XmlTextWriter(sWriter);

			//Load XML into an XPath
			XPathDocument xpthResults = new XPathDocument(sReader);
			
			XslTransform xslTrans = new XslTransform();

			try
			{
			
				//Load our XSL file
				xslTrans.Load(strXSLFile);

				//And transform
				xslTrans.Transform(xpthResults, null, xmlWriter);

			}
			catch(Exception e)
			{
				
				sbReturn.Append("<Error>");
				sbReturn.Append(e.Message);
				sbReturn.Append(" - ");
				sbReturn.Append(e.InnerException);
				sbReturn.Append("</Error>");

			}

			return sbReturn.ToString();

		}
		/// <summary>
		/// This method transforms the supplied XML string using the XSL stylesheet accessed by the supplied 
		/// XmlTextReader.
		/// </summary>
		/// <param name="strXML">A string of XML data to be transformed.</param>
		/// <param name="xmlXSL">An XmlTextReader accessing the XSL stylesheet to use.</param>
		/// <returns>A string of HTML representing the transformed result set.</returns>
		/// <remarks>In the event of an error the function returns a string representing an XML node 
		/// containing details of the error.</remarks>
		private static string transformXML(string strXML, XmlTextReader xmlXSL)
		{

			//Dim a return string 
			StringBuilder sbReturn = new StringBuilder();

			//Ok, we need to read our string in
			StringReader sReader = new StringReader(strXML);

			//And we need to write back out as well
			StringWriter sWriter = new StringWriter(sbReturn);

			//And of course it will be in XML
			XmlTextWriter xmlWriter = new XmlTextWriter(sWriter);

			//Load XML into an XPath
			XPathDocument xpthResults = new XPathDocument(sReader);

			//XmlTextReader xmlStyleSheet = new XmlTextReader(xmlXSL);
			
			//Something has to do the actual work, and that is the job of this little beasty here
			XslTransform xslTrans = new XslTransform();

			try
			{
			
				//Load our XSL file
				xslTrans.Load(xmlXSL);

				//And tranform
				xslTrans.Transform(xpthResults, null, xmlWriter);

			}
			catch(Exception e)
			{
				
				//sbReturn.Append("<Error>");
				//sbReturn.Append(e.Message);
				//sbReturn.Append(" - ");
				//sbReturn.Append(e.InnerException);
				//sbReturn.Append("</Error>");

				throw(e);

			}

			return sbReturn.ToString();

		}

		/// <summary>
		/// This method transforms the supplied XML string using the XSL stylesheet accessed by the supplied 
		/// XmlReader.
		/// </summary>
		/// <param name="xmlNav">An XmlDocument object representing the XML data to be transformed.</param>
		/// <param name="xmlXSL">An XmlReader accessing the XSL stylesheet to use.</param>
		/// <param name="xslArgs">An XsltArgumentList containing the arguments to be used by the transformation.</param>
		/// <returns>A string of HTML representing the transformed result set.</returns>
		/// <remarks>In the event of an error the function returns a string representing an XML node 
		/// containing details of the error.</remarks>
		private static string transformXML(XmlDocument xmlNav, XmlReader xmlXSL, XsltArgumentList xslArgs)
		{

			StringBuilder sbReturn = new StringBuilder();

			StringWriter sWriter = new StringWriter(sbReturn);

			XmlTextWriter xmlWriter = new XmlTextWriter(sWriter);
			
			XslTransform xslTrans = new XslTransform();


			try
			{
			
				//Load our XSL file
				xslTrans.Load(xmlXSL);

				//And transform
				xslTrans.Transform(xmlNav, xslArgs, xmlWriter);

			}
			catch(Exception e)
			{
				
				
				throw(e);

			}

			return sbReturn.ToString();

		}



		#endregion

		#endregion

	

		public override void Dispose()
		{	
			if (license != null)
			{
				license.Dispose();
				license = null;
			}
			base.Dispose ();
		}

	}
}

