using System;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Web;
using System.Xml;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Events;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;

namespace MSIBPlusPack.Web
{
	/// <summary>
	/// Summary description for EventApproved.
	/// </summary>
	public class EventApproved : IHttpModule
	{
		// Root node for the XML document
		private const string xmlRootNode = "MSIBPlusPackEventPlaceHolderData";

		// Gets the connection string
		private string sDBConn = "";

		// Local variables relative to the event
		/// <summary>
		/// This property exposes the title for the Event.
		/// </summary>
		private string EventTitle;

		/// <summary>
		/// This property exposes the date and time for the
		/// start of the Event.
		/// </summary>
		private DateTime StartDate;

		/// <summary>
		/// This property exposes the date and time for the
		/// end of the Event.
		/// </summary>
		private DateTime EndDate;

		/// <summary>
		/// This property exposes the category of the Event.
		/// </summary>
		private string Category;

		/// <summary>
		/// This property exposes the location of the Event.
		/// </summary>
		private string Location;

		/// <summary>
		/// This property exposes the description of the Event.
		/// </summary>
		private string Description;

		/// <summary>
		/// Constructor - creates a new instance of the object.
		/// </summary>
		public EventApproved()
		{
		}

		/// <summary>
		/// Method called when the object is released.
		/// </summary>
		public void Dispose()
		{
			// Removes the handler for the event passed when a posting is approved
			// PostingEvents.Current.Approved -= ChangedEventHandler(Current_Approved);
		}
	
		/// <summary>
		/// This method is only responsible for initialising the connection string property
		/// to the Event database and to add an event handler to the Posting approved event
		/// (Current_Approved).
		/// </summary>
		/// <param name="context">Current HTTP Application context</param>
		public void Init(HttpApplication context)
		{
			// Gets the conenction string to the database
			this.sDBConn = ConfigurationSettings.AppSettings["msibEventDB"];

			// Adds a handler for the event passed when a posting is approved
			PostingEvents.Current.Approved += new ChangedEventHandler(Current_Approved);
		}

		/// <summary>
		/// This method stores the content of an Event in the MSIB event database.
		/// </summary>
		/// <param name="sender">sender object</param>
		/// <param name="e">ChangedEventArgs</param>
		private void Current_Approved(object sender, ChangedEventArgs e)
		{
			if(e.Target is Posting)
			{
				// Gets teh reference to the posting
				Posting p = e.Target as Posting;

				// Checks that the posting being approved contains the Event place holder
				foreach(Placeholder ph in p.Placeholders)
				{
					if(ph is XmlPlaceholder)
					{
						// Gets the data from the placeholder
						if(this.LoadPHDataXmlDocument(ph.Datasource.RawContent))
						{
							// If so, gets the data to save in the database
							SqlConnection oDBConn = new SqlConnection();
							oDBConn.ConnectionString = this.sDBConn;
							try
							{
								SqlCommand oCommand = new SqlCommand();
								oCommand.CommandText = "sp_UpdateEvent";
								oCommand.CommandType = CommandType.StoredProcedure;
								oCommand.Connection = oDBConn;

								// Passing the posting guid
								SqlParameter oParamPGuid = new SqlParameter("@PGUID", SqlDbType.NVarChar, 50);
								oParamPGuid.Value = p.Guid;
								oCommand.Parameters.Add(oParamPGuid);

								// Passing the Title
								SqlParameter oParamTitle = new SqlParameter("@EventTitle", SqlDbType.NVarChar, 100);
								oParamTitle.Value = this.EventTitle;
								oCommand.Parameters.Add(oParamTitle);

								// Passing the start date time
								SqlParameter oParamStartDT = new SqlParameter("@StartDT", SqlDbType.DateTime);
								oParamStartDT.Value = this.StartDate;
								oCommand.Parameters.Add(oParamStartDT);

								// Passing the end date time
								SqlParameter oParamEndDT = new SqlParameter("@EndDT", SqlDbType.DateTime);
								oParamEndDT.Value = this.EndDate;
								oCommand.Parameters.Add(oParamEndDT);

								// Passing the location
								SqlParameter oParamLocation = new SqlParameter("@Location", SqlDbType.NVarChar, 100);
								oParamLocation.Value = this.Location;
								oCommand.Parameters.Add(oParamLocation);

								// Passing the category
								SqlParameter oParamCategory = new SqlParameter("@Category", SqlDbType.NVarChar, 100);
								oParamCategory.Value = this.Category;
								oCommand.Parameters.Add(oParamCategory);

								// Passing the description
								SqlParameter oParamDesc = new SqlParameter("@Description", SqlDbType.NVarChar, 3200);
								oParamDesc.Value = this.Description;
								oCommand.Parameters.Add(oParamDesc);

								oDBConn.Open();

								oCommand.ExecuteNonQuery();
							}
							catch (Exception ex)
							{
								// Does nothing
								string sMessage = ex.Message;
							}
							finally
							{
								oDBConn.Close();
							}
						}
					}		
				}
			}
		}

		/// <summary>
		///  This function loads the content from the XmlPlaceHolder onto the placeholder itself.
		///  It then populates the controls with the values found
		/// </summary>
		/// <param name="xmlStream">XML stream containing the data containing the definition of an Event</param>
		private bool LoadPHDataXmlDocument(string xmlStream)
		{
			// Boolean value returned
			bool bReturn = false;

			try
			{
				// Only if the XML placeholders contains some data.
				if(xmlStream.Length > 0)
				{
					// Creates the XML document
					XmlDataDocument phData  = new XmlDataDocument();

					// Loads the content of the 
					phData.LoadXml(xmlStream);
					XmlElement xmlRoot = phData.DocumentElement;
					if(xmlRoot.Name.Equals(xmlRootNode))
					{

						// Data for the element "Title"
						XmlNode xmlTitle;
						xmlTitle = xmlRoot.SelectSingleNode("Title");
						this.EventTitle = xmlTitle.InnerXml;

						// Data for the element relating to the Start of the event
						XmlNode xmlStartDay, xmlStartMonth, xmlStartYear;
						XmlNode xmlStartHour, xmlStartMinute, xmlStartSecond;
						xmlStartDay = xmlRoot.SelectSingleNode("StartDay");
						xmlStartMonth = xmlRoot.SelectSingleNode("StartMonth");
						xmlStartYear = xmlRoot.SelectSingleNode("StartYear");
						xmlStartHour = xmlRoot.SelectSingleNode("StartHour");
						xmlStartMinute = xmlRoot.SelectSingleNode("StartMinute");
						xmlStartSecond = xmlRoot.SelectSingleNode("StartSecond");
						this.StartDate = new DateTime(
							Convert.ToInt32(xmlStartYear.InnerXml),
							Convert.ToInt32(xmlStartMonth.InnerXml),
							Convert.ToInt32(xmlStartDay.InnerXml),
							Convert.ToInt32(xmlStartHour.InnerXml),
							Convert.ToInt32(xmlStartMinute.InnerXml),
							Convert.ToInt32(xmlStartSecond.InnerXml));

						// Data for the element relating to the End of the event
						XmlNode xmlEndDay, xmlEndMonth, xmlEndYear;
						XmlNode xmlEndHour, xmlEndMinute, xmlEndSecond;
						xmlEndDay = xmlRoot.SelectSingleNode("EndDay");
						xmlEndMonth = xmlRoot.SelectSingleNode("EndMonth");
						xmlEndYear = xmlRoot.SelectSingleNode("EndYear");
						xmlEndHour = xmlRoot.SelectSingleNode("EndHour");
						xmlEndMinute = xmlRoot.SelectSingleNode("EndMinute");
						xmlEndSecond = xmlRoot.SelectSingleNode("EndSecond");
						this.EndDate = new DateTime(
							Convert.ToInt32(xmlEndYear.InnerXml),
							Convert.ToInt32(xmlEndMonth.InnerXml),
							Convert.ToInt32(xmlEndDay.InnerXml),
							Convert.ToInt32(xmlEndHour.InnerXml),
							Convert.ToInt32(xmlEndMinute.InnerXml),
							Convert.ToInt32(xmlEndSecond.InnerXml));

						// Data for the element "Location"
						XmlNode xmlLocation;
						xmlLocation = xmlRoot.SelectSingleNode("Location");
						this.Location = xmlLocation.InnerXml;

						// Data for the element "Category"
						XmlNode xmlCategory;
						xmlCategory = xmlRoot.SelectSingleNode("Category");
						this.Category = xmlCategory.InnerXml;

						// Data for the element "Description"
						XmlNode xmlDescription;
						xmlDescription = xmlRoot.SelectSingleNode("Description");
						this.Description = xmlDescription.InnerXml;

						bReturn = true;
					}
					else
						bReturn = false;
				}
				else
				{
					bReturn = false;
				}
			}
			catch (Exception ex)
			{
				string sMessage = String.Format("\n\nAn error has occured in the private function 'LoadPHDataXmlDocument'.\nThe error message is: '{0}'.", ex.Message);
				bReturn = false;
			}
			return bReturn;
		}
	}
}
