using System;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml;
using ioko.ComponentModel.LicenseProvider;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.WebControls;
using Microsoft.ContentManagement.WebControls.Design;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// This custom XML placeholder allows the authoring of Event specific information.
	/// </summary>
	/// <remarks>
	/// <para>
	/// Different Event types can be stored on multiple templates and/or in seperate channels.
	/// The following event data is serialised into the underlying CMS XML placeholder definition:
	/// <list type="bullet">
	/// <item>A title of type string - see <see cref="EventPlaceHolder.EventTitle"/>.</item>
	/// <item>A start date of type DateTime - see <see cref="EventPlaceHolder.StartDate"/>.</item>
	/// <item>An end date of type DateTime - see <see cref="EventPlaceHolder.EndDate"/>.</item>
	/// <item>A category of type string - see <see cref="EventPlaceHolder.Category"/>.</item>
	/// <item>A location of type string - see <see cref="EventPlaceHolder.Location"/>.</item>
	/// <item>A description of type string - see <see cref="EventPlaceHolder.Description"/>.</item>
	/// </list>
	/// </para>
	/// <para>
	/// All string properties of an Event should only contain plain text, as HTML characters may break the XML document
	/// created to store the Event data.
	/// </para>
	/// <para>
	/// The definition of an Event cannot currently be extended or modified.
	/// </para>
	/// <para>
	/// This custom placeholder provides default controls for presentation mode
	/// which are implemented by setting the property <see cref="EventPlaceHolder.DefaultControls"/> to true at design time.
	/// </para>
	/// <para>
	/// This placeholder can be extended to provide a custom presentation design for presentation mode.
	/// This is performed by overriding the following methods:
	/// <list type="bullet">
	/// <item>Creation of the controls at presentation time - see <see cref="EventPlaceHolder.CreatePresentationEventChildControls"/></item>
	/// <item>Association of the values with the Event properties at presentation time - see <see cref="EventPlaceHolder.LoadEventPlaceholderContentForPresentation"/></item>
	/// </list>
	/// </para>
	/// <para>
	/// The Event placeholder cleanly handles any errors that may occur.
	/// The string property <see cref="EventPlaceHolder.ErrorMessage"/> exposes the details of any errors that have been trapped. This property is an empty string if no errors have occurred.
	/// The property <see cref="EventPlaceHolder.Debug"/> can be used to enable debug mode, whereby Literal controls will be created displaying the detail of any errors encountered.
	/// </para>
	/// </remarks>
	[
		ToolboxData("<{0}:EventPlaceHolder runat=server></{0}:EventPlaceHolder>"),
			ToolboxBitmap(typeof(EventPlaceHolder), "Events.BMP"),
			ParseChildren(true), 
		SupportedPlaceholderDefinitionType(typeof(XmlPlaceholderDefinition))
	]
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public class EventPlaceHolder : BasePlaceholderControl
	{
		#region Constants
		/// <summary>
		/// This constant defined the xml Root Node for the XML document
		/// stored in the placeholder
		/// </summary>
		private const string xmlRootNode = "MSIBPlusPackEventPlaceHolderData";
		#endregion

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false; 
				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;
				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(EventPlaceHolder),this);
					lastDate = DateTime.Today;
				}
				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;
					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}
						return;
					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;
					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}           
			} 
			catch {}
			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion

		#region Constructor
		/// <summary>
		/// This method is the default constructor for the placeholder.
		/// No actions are carried out during construction of the object.
		/// </summary>
		public EventPlaceHolder() : base()
		{
		}
		#endregion

		#region Placeholder Control Variables
		/// <summary>
		/// This property is the internal version of the property DefaultControls.
		/// </summary>
		private bool bDefaultControls = true;
		/// <summary>
		/// This property allows a developer to set whether to use the default controls for
		/// the Event placeholder.
		/// </summary>
		/// <remarks>
		/// Setting this property to true overrides any custom display provided by the methods:
		/// <list type="bullet">
		/// <item>Construction of the controls for at presentation time - see <see cref="EventPlaceHolder.CreatePresentationEventChildControls"/></item>
		/// <item>Association of the values with the Event properties at presentation time - see <see cref="EventPlaceHolder.LoadEventPlaceholderContentForPresentation"/></item>
		/// <item>Sets the Event placeholder variables to be saved in CMS - see <see cref="EventPlaceHolder.SaveEventPlaceholderContent"/></item>
		/// </list>
		/// </remarks>
		[
			Category("MSIB Plus Pack - Event placeholder Settings"), 
				DefaultValue("false"),
				Description("This property allows to select the default presentation controls when this Event placeholder control is used.")
		]
		public bool DefaultControls
		{
			get { return bDefaultControls; }
			set { bDefaultControls = value; }
		}

		/// <summary>
		/// This property is the internal verion of the property Debug
		/// </summary>
		private bool bDebug = false;
		/// <summary>
		/// This property is used to enable debug mode,
		/// whereby details of any errors encountered are displayed on screen instead of being supressed.
		/// This property should only be set to true during implementation time.
		/// </summary>
		[
			Category("MSIB Plus Pack - Event placeholder Settings"), 
				DefaultValue("false"),
				Description("This property allows to display some error messages otherwise hidden during normal execution. This property should only be set to true during implementation time.")
		]
		public bool Debug
		{
			get { return bDebug; }
			set { bDebug = value; }
		}

		/// <summary>
		/// Internal version of the property holding the link to the calendar pop up.
		/// </summary>
		private string sCalendarPopUp = "/Woodgrovenet/Calendar.aspx";
		/// <summary>
		/// <para>
		/// This property defines where the calendar pop up window is hosted.
		/// A calendar pop up window is used to select he start and end date of an evetn to avoid
		/// cultural difference in displaying a date.
		/// </para>
		/// <para>
		/// This property must point at the URL where the file "Calendar.aspx" is held within the CMS site.
		/// </para>
		/// </summary>
		[
			Category("MSIB Plus Pack - Event placeholder Settings"), 
				DefaultValue("/Woodgrovenet/Cms/WebAuthor/Dialogs/CalendarPopUp/Calendar.aspx"),
				Description("This property sets where the calendar pop up window is located on the server.")
		]		
		public string CalendarPopUp
		{
			get { return sCalendarPopUp; }
			set { sCalendarPopUp = value; }
		}

		/// <summary>
		/// Internal version of the property ErrorMessage
		/// </summary>
		private string sErrorMessage;

		/// <summary>
		/// This property exposes details of any errors generated during execution of the server control.
		/// </summary>
		/// <remarks>
		/// This property is read-only.
		/// </remarks>
		protected string ErrorMessage
		{
			get { return sErrorMessage; }
		}

		/// <summary>
		/// This property is used to display error messages during
		/// authoring time.
		/// </summary>
		private Literal ltErrorAuth;

		/// <summary>
		/// This property is used to display error messages at
		/// presentation time.
		/// </summary>
		private Literal ltErrorPres;

		/// <summary>
		/// This property exposes the title for the Event. It is limited to 100 characters.
		/// </summary>
		public string EventTitle;

		/// <summary>
		/// This property exposes the date and time for the
		/// start of the Event.
		/// </summary>
		public DateTime StartDate;

		/// <summary>
		/// This property exposes the date and time for the
		/// end of the Event.
		/// </summary>
		public DateTime EndDate;

		/// <summary>
		/// This property exposes the category of the Event. It is limited to 100 characters.
		/// </summary>
		public string Category;

		/// <summary>
		/// This property exposes the location of the Event. It is limited to 100 characters.
		/// </summary>
		public string Location;

		/// <summary>
		/// This property exposes the description of the Event. It is limited to 3200 characters.
		/// </summary>
		public string Description;
		#endregion

		#region Create Authoring Controls
		/// <summary>
		/// These controls are only used when the default controls are set to be created
		/// </summary>
		private TextBox tbAuthEventTitleText;
		private TextBox tbAuthEventStartText;
		private ListBox lbAuthEventStartTextHour;
		private ListBox lbAuthEventStartTextMinute;
		private TextBox tbAuthEventEndText;
		private ListBox lbAuthEventEndTextHour;
		private ListBox lbAuthEventEndTextMinute;
		private TextBox tbAuthEventLocationText;
		private TextBox tbAuthEventCategoryText;
		private TextBox tbAuthEventDescriptionText;
		private Literal ltAuthEventErrorControl;

		/// <summary>
		/// This method is used internally to create the default controls for the Event placeholder
		/// </summary>
		/// <param name="authoringContainer">Authoring container on which controls can be attached</param>
		private void CreateAuthoringEventDefaultChildControls(BaseModeContainer authoringContainer)
		{
			try
			{
				// Table to host the cobntrols
				Table tbAuthEventControl;

				// AuthEvent title
				TableRow trAuthEventTitle;
				TableCell tcAuthEventTitleName;
				TableCell tcAuthEventTitleText;
				Literal ltAuthEventTitleName;

				// AuthEvent start date and time
				TableRow trAuthEventStartDate;
				TableCell tcAuthEventStartDateName;
				TableCell tcAuthEventStartDateText;
				Literal ltAuthEventStartDateName;
				TableRow trAuthEventStartTime;
				TableCell tcAuthEventStartTimeName;
				TableCell tcAuthEventStartTimeText;
				Literal ltAuthEventStartTimeName;

				// AuthEvent end date and time
				TableRow trAuthEventEndDate;
				TableCell tcAuthEventEndDateName;
				TableCell tcAuthEventEndDateText;
				Literal ltAuthEventEndDateName;
				TableRow trAuthEventEndTime;
				TableCell tcAuthEventEndTimeName;
				TableCell tcAuthEventEndTimeText;
				Literal ltAuthEventEndTimeName;

				// AuthEvent location
				TableRow trAuthEventLocation;
				TableCell tcAuthEventLocationName;
				TableCell tcAuthEventLocationText;
				Literal ltAuthEventLocationName;

				// AuthEvent category
				TableRow trAuthEventCategory;
				TableCell tcAuthEventCategoryName;
				TableCell tcAuthEventCategoryText;
				Literal ltAuthEventCategoryName;

				// AuthEvent description
				TableRow trAuthEventDescription;
				TableCell tcAuthEventDescriptionName;
				TableCell tcAuthEventDescriptionText;
				Literal ltAuthEventDescriptionName;

				// Error control when in authoring mode
				TableRow trAuthEventErrorControl;
				TableCell tcAuthEventErrorControl;

				// Creation of the default controls in authoring mode
				tbAuthEventControl = new Table();
				tbAuthEventControl.ID = "tbAuthEventControl";
				tbAuthEventControl.BorderWidth = 1;
				tbAuthEventControl.CellPadding = 4;

				// Creates a row for entering the title of the AuthEvent
				trAuthEventTitle = new TableRow();
				trAuthEventTitle.ID = "trAuthEventTitle";

				tcAuthEventTitleName = new TableCell();
				tcAuthEventTitleName.ID = "tcAuthEventTitleName";
				tcAuthEventTitleName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventTitleName = new Literal();
				ltAuthEventTitleName.ID = "ltAuthEventTitleName";
				ltAuthEventTitleName.Text = "Event Title:";
				tcAuthEventTitleName.Controls.Add(ltAuthEventTitleName);

				tcAuthEventTitleText = new TableCell();
				tcAuthEventTitleText.ID = "tcAuthEventTitleText";
				tcAuthEventTitleText.HorizontalAlign = HorizontalAlign.Left;
				tbAuthEventTitleText = new TextBox();
				tbAuthEventTitleText.ID = "tbAuthEventTitleText";
				tbAuthEventTitleText.MaxLength = 100;
				tbAuthEventTitleText.Width = 250;
				tcAuthEventTitleText.Controls.Add(tbAuthEventTitleText);

				trAuthEventTitle.Controls.Add(tcAuthEventTitleName);
				trAuthEventTitle.Controls.Add(tcAuthEventTitleText);

				tbAuthEventControl.Controls.Add(trAuthEventTitle);
			
				// Creates the row to enter the start date
				trAuthEventStartDate = new TableRow();
				trAuthEventStartDate.ID = "trAuthEventStartDate";

				tcAuthEventStartDateName = new TableCell();
				tcAuthEventStartDateName.ID = "tcAuthEventStartDateName";
				tcAuthEventStartDateName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventStartDateName = new Literal();
				ltAuthEventStartDateName.ID = "ltAuthEventStartDateName";
				ltAuthEventStartDateName.Text = "Start Date:";
				tcAuthEventStartDateName.Controls.Add(ltAuthEventStartDateName);

				tcAuthEventStartDateText = new TableCell();
				tcAuthEventStartDateText.ID = "tcAuthEventStartDateText";
				tcAuthEventStartDateText.HorizontalAlign = HorizontalAlign.Left;

				tbAuthEventStartText = new TextBox();
				tbAuthEventStartText.ID = "tbAuthEventStartText";
				tbAuthEventStartText.Width = 100;
				tbAuthEventStartText.Text = DateTime.Now.ToShortDateString();
				tbAuthEventStartText.ReadOnly = true;
				tcAuthEventStartDateText.Controls.Add(tbAuthEventStartText);

				tcAuthEventStartDateText.Controls.Add(new LiteralControl("&nbsp;"));

				string StartDateContainerName = String.Format("{0}_AuthoringModeControlsContainer_{1}", this.ClientID, this.tbAuthEventStartText.ClientID);
				tcAuthEventStartDateText.Controls.Add(new LiteralControl("<input type='button' value='Set start date' onclick=\"window.open('" + sCalendarPopUp + "?src=" + StartDateContainerName + "', '_blank', 'height=240, width=250, left=100, top=100, location=no, menubar=no, resizable=no,  scrollbars=no, titlebar=no, toolbar=no', true);\">"));

				trAuthEventStartDate.Controls.Add(tcAuthEventStartDateName);
				trAuthEventStartDate.Controls.Add(tcAuthEventStartDateText);

				tbAuthEventControl.Controls.Add(trAuthEventStartDate);

				// Creates the row to enter the start Time
				trAuthEventStartTime = new TableRow();
				trAuthEventStartTime.ID = "trAuthEventStartTime";

				tcAuthEventStartTimeName = new TableCell();
				tcAuthEventStartTimeName.ID = "tcAuthEventStartTimeName";
				tcAuthEventStartTimeName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventStartTimeName = new Literal();
				ltAuthEventStartTimeName.ID = "ltAuthEventStartTimeName";
				ltAuthEventStartTimeName.Text = "Start Time:";
				tcAuthEventStartTimeName.Controls.Add(ltAuthEventStartTimeName);

				tcAuthEventStartTimeText = new TableCell();
				tcAuthEventStartTimeText.ID = "tcAuthEventStartTimeText";
				tcAuthEventStartTimeText.HorizontalAlign = HorizontalAlign.Left;

				lbAuthEventStartTextHour = new ListBox();
				lbAuthEventStartTextHour.ID = "lbAuthEventStartTextHour";
				lbAuthEventStartTextHour.Rows = 1;
				lbAuthEventStartTextHour.Items.Add(new ListItem("00", "00"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("01", "01"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("02", "02"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("03", "03"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("04", "04"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("05", "05"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("06", "06"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("07", "07"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("08", "08"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("09", "09"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("10", "10"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("11", "11"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("12", "12"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("13", "13"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("14", "14"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("15", "15"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("16", "16"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("17", "17"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("18", "18"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("19", "19"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("20", "20"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("21", "21"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("22", "22"));
				lbAuthEventStartTextHour.Items.Add(new ListItem("23", "23"));
				tcAuthEventStartTimeText.Controls.Add(lbAuthEventStartTextHour);

				tcAuthEventStartTimeText.Controls.Add(new LiteralControl(":"));

				lbAuthEventStartTextMinute = new ListBox();
				lbAuthEventStartTextMinute.ID = "lbAuthEventStartTextMinute";
				lbAuthEventStartTextMinute.Rows = 1;
				lbAuthEventStartTextMinute.Items.Add(new ListItem("00", "00"));
				lbAuthEventStartTextMinute.Items.Add(new ListItem("15", "1"));
				lbAuthEventStartTextMinute.Items.Add(new ListItem("30", "2"));
				lbAuthEventStartTextMinute.Items.Add(new ListItem("45", "3"));
				tcAuthEventStartTimeText.Controls.Add(lbAuthEventStartTextMinute);

				trAuthEventStartTime.Controls.Add(tcAuthEventStartTimeName);
				trAuthEventStartTime.Controls.Add(tcAuthEventStartTimeText);

				tbAuthEventControl.Controls.Add(trAuthEventStartTime);

				// Creates the row to enter the end date
				trAuthEventEndDate = new TableRow();
				trAuthEventEndDate.ID = "trAuthEventEndDate";

				tcAuthEventEndDateName = new TableCell();
				tcAuthEventEndDateName.ID = "tcAuthEventEndDateName";
				tcAuthEventEndDateName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventEndDateName = new Literal();
				ltAuthEventEndDateName.ID = "ltAuthEventEndDateName";
				ltAuthEventEndDateName.Text = "End Date:";
				tcAuthEventEndDateName.Controls.Add(ltAuthEventEndDateName);

				tcAuthEventEndDateText = new TableCell();
				tcAuthEventEndDateText.ID = "tcAuthEventEndDateText";
				tcAuthEventEndDateText.HorizontalAlign = HorizontalAlign.Left;

				tbAuthEventEndText = new TextBox();
				tbAuthEventEndText.ID = "tbAuthEventEndTextDay";
				tbAuthEventEndText.Width = 100;
				tbAuthEventEndText.ReadOnly = true;
				tbAuthEventEndText.Text = DateTime.Now.AddDays(1).Day.ToString();
				tcAuthEventEndDateText.Controls.Add(tbAuthEventEndText);

				tcAuthEventEndDateText.Controls.Add(new LiteralControl("&nbsp;"));
				string EndDateContainerName = String.Format("{0}_AuthoringModeControlsContainer_{1}", this.ClientID, this.tbAuthEventEndText.ClientID);
				tcAuthEventEndDateText.Controls.Add(new LiteralControl("<input type='button' value='Set end date' onclick=\"window.open('" + sCalendarPopUp + "?src=" + EndDateContainerName + "', '_blank', 'height=240, width=250, left=100, top=100, location=no, menubar=no, resizable=no,  scrollbars=no, titlebar=no, toolbar=no', true);\">"));

				trAuthEventEndDate.Controls.Add(tcAuthEventEndDateName);
				trAuthEventEndDate.Controls.Add(tcAuthEventEndDateText);

				tbAuthEventControl.Controls.Add(trAuthEventEndDate);

				// Creates the row to enter the End Time
				trAuthEventEndTime = new TableRow();
				trAuthEventEndTime.ID = "trAuthEventEndTime";

				tcAuthEventEndTimeName = new TableCell();
				tcAuthEventEndTimeName.ID = "tcAuthEventEndTimeName";
				tcAuthEventEndTimeName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventEndTimeName = new Literal();
				ltAuthEventEndTimeName.ID = "ltAuthEventEndTimeName";
				ltAuthEventEndTimeName.Text = "End Time:";
				tcAuthEventEndTimeName.Controls.Add(ltAuthEventEndTimeName);

				tcAuthEventEndTimeText = new TableCell();
				tcAuthEventEndTimeText.ID = "tcAuthEventEndTimeText";
				tcAuthEventEndTimeText.HorizontalAlign = HorizontalAlign.Left;

				lbAuthEventEndTextHour = new ListBox();
				lbAuthEventEndTextHour.ID = "lbAuthEventEndTextHour";
				lbAuthEventEndTextHour.Rows = 1;
				lbAuthEventEndTextHour.Items.Add(new ListItem("00", "00"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("01", "01"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("02", "02"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("03", "03"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("04", "04"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("05", "05"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("06", "06"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("07", "07"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("08", "08"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("09", "09"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("10", "10"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("11", "11"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("12", "12"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("13", "13"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("14", "14"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("15", "15"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("16", "16"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("17", "17"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("18", "18"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("19", "19"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("20", "20"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("21", "21"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("22", "22"));
				lbAuthEventEndTextHour.Items.Add(new ListItem("23", "23"));
				tcAuthEventEndTimeText.Controls.Add(lbAuthEventEndTextHour);

				tcAuthEventEndTimeText.Controls.Add(new LiteralControl(":"));

				lbAuthEventEndTextMinute = new ListBox();
				lbAuthEventEndTextMinute.ID = "lbAuthEventEndTextMinute";
				lbAuthEventEndTextMinute.Rows = 1;
				lbAuthEventEndTextMinute.Items.Add(new ListItem("00", "0"));
				lbAuthEventEndTextMinute.Items.Add(new ListItem("15", "1"));
				lbAuthEventEndTextMinute.Items.Add(new ListItem("30", "2"));
				lbAuthEventEndTextMinute.Items.Add(new ListItem("45", "3"));
				tcAuthEventEndTimeText.Controls.Add(lbAuthEventEndTextMinute);

				trAuthEventEndTime.Controls.Add(tcAuthEventEndTimeName);
				trAuthEventEndTime.Controls.Add(tcAuthEventEndTimeText);

				tbAuthEventControl.Controls.Add(trAuthEventEndTime);

				// Creates the row to enter the location
				trAuthEventLocation = new TableRow();
				trAuthEventLocation.ID = "trAuthEventLocation";

				tcAuthEventLocationName = new TableCell();
				tcAuthEventLocationName.ID = "tcAuthEventLocationName";
				tcAuthEventLocationName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventLocationName = new Literal();
				ltAuthEventLocationName.ID = "ltAuthEventLocationName";
				ltAuthEventLocationName.Text = "Location:";
				tcAuthEventLocationName.Controls.Add(ltAuthEventLocationName);

				tcAuthEventLocationText = new TableCell();
				tcAuthEventLocationText.ID = "tcAuthEventLocationText";
				tcAuthEventLocationText.HorizontalAlign = HorizontalAlign.Left;
				tbAuthEventLocationText = new TextBox();
				tbAuthEventLocationText.ID = "tbAuthEventLocationText";
				tbAuthEventLocationText.MaxLength = 100;
				tbAuthEventLocationText.Width = 250;
				tcAuthEventLocationText.Controls.Add(tbAuthEventLocationText);

				trAuthEventLocation.Controls.Add(tcAuthEventLocationName);
				trAuthEventLocation.Controls.Add(tcAuthEventLocationText);

				tbAuthEventControl.Controls.Add(trAuthEventLocation);
			
				// Creates the row to enter the category
				trAuthEventCategory = new TableRow();
				trAuthEventCategory.ID = "trAuthEventCategory";

				tcAuthEventCategoryName = new TableCell();
				tcAuthEventCategoryName.ID = "tcAuthEventCategoryName";
				tcAuthEventCategoryName.HorizontalAlign = HorizontalAlign.Left;
				ltAuthEventCategoryName = new Literal();
				ltAuthEventCategoryName.ID = "ltAuthEventCategoryName";
				ltAuthEventCategoryName.Text = "Category:";
				tcAuthEventCategoryName.Controls.Add(ltAuthEventCategoryName);

				tcAuthEventCategoryText = new TableCell();
				tcAuthEventCategoryText.ID = "tcAuthEventCategoryText";
				tcAuthEventCategoryText.HorizontalAlign = HorizontalAlign.Left;
				tbAuthEventCategoryText = new TextBox();
				tbAuthEventCategoryText.ID = "tbAuthEventCategoryText";
				tbAuthEventCategoryText.MaxLength = 100;
				tbAuthEventCategoryText.Width = 250;
				tcAuthEventCategoryText.Controls.Add(tbAuthEventCategoryText);

				trAuthEventCategory.Controls.Add(tcAuthEventCategoryName);
				trAuthEventCategory.Controls.Add(tcAuthEventCategoryText);

				tbAuthEventControl.Controls.Add(trAuthEventCategory);
			
				// Creates the row to enter the description
				trAuthEventDescription = new TableRow();
				trAuthEventDescription.ID = "trAuthEventDescription";

				tcAuthEventDescriptionName = new TableCell();
				tcAuthEventDescriptionName.ID = "tcAuthEventDescriptionName";
				tcAuthEventDescriptionName.HorizontalAlign = HorizontalAlign.Left;
				tcAuthEventDescriptionName.VerticalAlign = VerticalAlign.Top;
				ltAuthEventDescriptionName = new Literal();
				ltAuthEventDescriptionName.ID = "ltAuthEventDescriptionName";
				ltAuthEventDescriptionName.Text = "Description:";
				tcAuthEventDescriptionName.Controls.Add(ltAuthEventDescriptionName);

				tcAuthEventDescriptionText = new TableCell();
				tcAuthEventDescriptionText.ID = "tcAuthEventDescriptionText";
				tcAuthEventDescriptionText.HorizontalAlign = HorizontalAlign.Left;
				tcAuthEventDescriptionText.VerticalAlign = VerticalAlign.Top;
				tbAuthEventDescriptionText = new TextBox();
				tbAuthEventDescriptionText.ID = "tbAuthEventDescriptionText";
				tbAuthEventDescriptionText.MaxLength = 3200;
				tbAuthEventDescriptionText.Width = 250;
				tbAuthEventDescriptionText.TextMode = TextBoxMode.MultiLine;
				tbAuthEventDescriptionText.Rows = 5;
				tcAuthEventDescriptionText.Controls.Add(tbAuthEventDescriptionText);

				trAuthEventDescription.Controls.Add(tcAuthEventDescriptionName);
				trAuthEventDescription.Controls.Add(tcAuthEventDescriptionText);

				tbAuthEventControl.Controls.Add(trAuthEventDescription);

				// Error control
				trAuthEventErrorControl = new TableRow();
				trAuthEventErrorControl.ID = "trAuthEventErrorControl";

				tcAuthEventErrorControl = new TableCell();
				tcAuthEventErrorControl.ID = "tcAuthEventErrorControl";
				tcAuthEventErrorControl.HorizontalAlign = HorizontalAlign.Left;
				tcAuthEventErrorControl.ColumnSpan = 2;
				ltAuthEventErrorControl = new Literal();
				ltAuthEventErrorControl.ID = "ltAuthEventErrorControl";
				ltAuthEventErrorControl.Visible = true;
				ltAuthEventErrorControl.Text = "";
				tcAuthEventErrorControl.Controls.Add(ltAuthEventErrorControl);
				trAuthEventErrorControl.Controls.Add(tcAuthEventErrorControl);
				tbAuthEventControl.Controls.Add(trAuthEventErrorControl);

				authoringContainer.Controls.Add(tbAuthEventControl);
			}
			catch(Exception ex)
			{
				this.sErrorMessage = String.Format("\n\nAn error occured when creating the default controls in authoring mode in the private method 'CreateAuthoringEventDefaultChildControls'.\nThe error message is: '{0}'.", ex.Message);
			}
		}

		/// <summary>
		/// <para>
		/// This method is inherited from the base placeholder control class.
		/// It provides a mean of creating the default controls for the place holder.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>
		/// The method initialises the protected properties needed to manage an Event data
		/// with the following values:
		/// <list type="bullet">
		/// <item><see cref="EventPlaceHolder.EventTitle"/>: string "Title"</item>
		/// <item><see cref="EventPlaceHolder.StartDate"/>: date and time of "DateTime.Now"</item>
		/// <item><see cref="EventPlaceHolder.EndDate"/>: date and time of "DateTime.Now.AddDays(1)"</item>
		/// <item><see cref="EventPlaceHolder.Location"/>: string "Location"</item>
		/// <item><see cref="EventPlaceHolder.Category"/>: string "Category"</item>
		/// <item><see cref="EventPlaceHolder.Description"/>: string "Description"</item>
		/// </list>
		/// </para>
		/// <para>
		/// If the control is set to display debugging information using the property <see cref="EventPlaceHolder.Debug"/>
		/// then a specific Literal control is also created.
		/// </para>
		/// </remarks>
		/// <param name="authoringContainer">Authoring container on which controls can be attached.</param>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			// Sets the default values
			this.EventTitle = String.Format("Title");
			this.StartDate = DateTime.Now;
			this.EndDate = DateTime.Now.AddDays(1);
			this.Location = String.Format("Location");
			this.Category = String.Format("Category");
			this.Description = String.Format("Description");

			try
			{
				// Creates the default controls.
				this.CreateAuthoringEventDefaultChildControls(authoringContainer);

				// Creates the debugging control
				if(this.bDebug)
				{
					// Creates the debug control
					this.ltErrorAuth = new Literal();
					this.ltErrorAuth.ID = "ltErrorAuth";
					this.ltErrorAuth.Text = "";
					this.ltErrorAuth.Visible = this.bDebug;
					authoringContainer.Controls.Add(ltErrorAuth);
				}
			}
			catch(Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error occurred in the overloaded method 'CreateAuthoringChildControls'.\nThe error message is: '{0}'.", ex.Message);

				// Creates the debugging control
				if(this.bDebug)
				{
					// Creates the debug control
					this.ltErrorAuth = new Literal();
					this.ltErrorAuth.ID = "ltErrorAuth";
					this.ltErrorAuth.Text = this.sErrorMessage;
					this.ltErrorAuth.Visible = this.bDebug;
					authoringContainer.Controls.Add(ltErrorAuth);
				}
			}
		}
		#endregion

		#region Create Presentation Controls
		/// <summary>
		/// These controls are only used when the default controls are set to be created
		/// </summary>
		private Literal ltPresEventTitleText;
		private	Literal ltPresEventStartDateText;
		private Literal ltPresEventStartTimeText;
		private Literal ltPresEventEndDateText;
		private Literal ltPresEventEndTimeText;
		private Literal ltPresEventLocationText;
		private Literal ltPresEventCategoryText;
		private Literal ltPresEventDescriptionText;
		private Literal ltPresEventErrorControl;

		/// <summary>
		/// This method is used internally to create the default controls for the placeholder
		/// </summary>
		/// <param name="presentationContainer">Presentation container on which controls can be attached</param>
		private void CreatePresentationEventDefaultChildControls(BaseModeContainer presentationContainer)
		{
			try
			{
				// Table holding the default controls
				Table tbPresEventControl;

				// PresEvent title
				TableRow trPresEventTitle;
				TableCell tcPresEventTitleName;
				TableCell tcPresEventTitleText;
				Literal ltPresEventTitleName;

				// PresEvent start date and time
				TableRow trPresEventStartDate;
				TableCell tcPresEventStartDateName;
				TableCell tcPresEventStartDateText;
				Literal ltPresEventStartDateName;

				TableRow trPresEventStartTime;
				TableCell tcPresEventStartTimeName;
				TableCell tcPresEventStartTimeText;
				Literal ltPresEventStartTimeName;

				// PresEvent end date and time
				TableRow trPresEventEndDate;
				TableCell tcPresEventEndDateName;
				TableCell tcPresEventEndDateText;
				Literal ltPresEventEndDateName;

				TableRow trPresEventEndTime;
				TableCell tcPresEventEndTimeName;
				TableCell tcPresEventEndTimeText;
				Literal ltPresEventEndTimeName;

				// PresEvent location
				TableRow trPresEventLocation;
				TableCell tcPresEventLocationName;
				TableCell tcPresEventLocationText;
				Literal ltPresEventLocationName;

				// PresEvent category
				TableRow trPresEventCategory;
				TableCell tcPresEventCategoryName;
				TableCell tcPresEventCategoryText;
				Literal ltPresEventCategoryName;

				// PresEvent description
				TableRow trPresEventDescription;
				TableCell tcPresEventDescriptionName;
				TableCell tcPresEventDescriptionText;
				Literal ltPresEventDescriptionName;

				// Error control when in presentation mode
				TableRow trPresEventErrorControl;
				TableCell tcPresEventErrorControl;

				// Creation of the default controls in presentation mode
				tbPresEventControl = new Table();
				tbPresEventControl.ID = "tbPresEventControl";
				tbPresEventControl.BorderWidth = 0;
				tbPresEventControl.CellPadding = 4;

				// Creates a row for entering the title of the PresEvent
				trPresEventTitle = new TableRow();
				trPresEventTitle.ID = "trPresEventTitle";

				tcPresEventTitleName = new TableCell();
				tcPresEventTitleName.ID = "tcPresEventTitleName";
				tcPresEventTitleName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventTitleName = new Literal();
				ltPresEventTitleName.ID = "ltPresEventTitleName";
				ltPresEventTitleName.Text = "Title:";
				tcPresEventTitleName.Controls.Add(ltPresEventTitleName);

				tcPresEventTitleText = new TableCell();
				tcPresEventTitleText.ID = "tcPresEventTitleText";
				tcPresEventTitleText.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventTitleText = new Literal();
				ltPresEventTitleText.ID = "ltPresEventTitleText";
				ltPresEventTitleText.Text = "<b>TITLE</b>";
				tcPresEventTitleText.Controls.Add(ltPresEventTitleText);

				trPresEventTitle.Controls.Add(tcPresEventTitleName);
				trPresEventTitle.Controls.Add(tcPresEventTitleText);

				tbPresEventControl.Controls.Add(trPresEventTitle);
			
				// Creates the row to enter the start date
				trPresEventStartDate = new TableRow();
				trPresEventStartDate.ID = "trPresEventStartDate";

				tcPresEventStartDateName = new TableCell();
				tcPresEventStartDateName.ID = "tcPresEventStartDateName";
				tcPresEventStartDateName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventStartDateName = new Literal();
				ltPresEventStartDateName.ID = "ltPresEventStartDateName";
				ltPresEventStartDateName.Text = "Start Date:";
				tcPresEventStartDateName.Controls.Add(ltPresEventStartDateName);

				tcPresEventStartDateText = new TableCell();
				tcPresEventStartDateText.ID = "tcPresEventStartDateText";
				tcPresEventStartDateText.HorizontalAlign = HorizontalAlign.Left;

				ltPresEventStartDateText = new Literal();
				ltPresEventStartDateText.ID = "ltPresEventStartTimeText";
				ltPresEventStartDateText.Text = "<b>STARTDATE</b>";
				tcPresEventStartDateText.Controls.Add(ltPresEventStartDateText);

				trPresEventStartDate.Controls.Add(tcPresEventStartDateName);
				trPresEventStartDate.Controls.Add(tcPresEventStartDateText);

				tbPresEventControl.Controls.Add(trPresEventStartDate);

				// Creates the row to enter the start Time
				trPresEventStartTime = new TableRow();
				trPresEventStartTime.ID = "trPresEventStartTime";

				tcPresEventStartTimeName = new TableCell();
				tcPresEventStartTimeName.ID = "tcPresEventStartTimeName";
				tcPresEventStartTimeName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventStartTimeName = new Literal();
				ltPresEventStartTimeName.ID = "ltPresEventStartTimeName";
				ltPresEventStartTimeName.Text = "Start Time:";
				tcPresEventStartTimeName.Controls.Add(ltPresEventStartTimeName);

				tcPresEventStartTimeText = new TableCell();
				tcPresEventStartTimeText.ID = "tcPresEventStartTimeText";
				tcPresEventStartTimeText.HorizontalAlign = HorizontalAlign.Left;

				ltPresEventStartTimeText = new Literal();
				ltPresEventStartTimeText.ID = "ltPresEventStartTimeText";
				ltPresEventStartTimeText.Text = "<b>STARTTIME</b>";
				tcPresEventStartTimeText.Controls.Add(ltPresEventStartTimeText);

				trPresEventStartTime.Controls.Add(tcPresEventStartTimeName);
				trPresEventStartTime.Controls.Add(tcPresEventStartTimeText);

				tbPresEventControl.Controls.Add(trPresEventStartTime);

				// Creates the row to enter the end date
				trPresEventEndDate = new TableRow();
				trPresEventEndDate.ID = "trPresEventEndDate";

				tcPresEventEndDateName = new TableCell();
				tcPresEventEndDateName.ID = "tcPresEventEndDateName";
				tcPresEventEndDateName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventEndDateName = new Literal();
				ltPresEventEndDateName.ID = "ltPresEventEndDateName";
				ltPresEventEndDateName.Text = "End Date:";
				tcPresEventEndDateName.Controls.Add(ltPresEventEndDateName);

				tcPresEventEndDateText = new TableCell();
				tcPresEventEndDateText.ID = "tcPresEventEndDateText";
				tcPresEventEndDateText.HorizontalAlign = HorizontalAlign.Left;

				ltPresEventEndDateText = new Literal();
				ltPresEventEndDateText.ID = "ltPresEventEndDateText";
				ltPresEventEndDateText.Text = "<b>ENDDATE</b>";
				tcPresEventEndDateText.Controls.Add(ltPresEventEndDateText);

				trPresEventEndDate.Controls.Add(tcPresEventEndDateName);
				trPresEventEndDate.Controls.Add(tcPresEventEndDateText);

				tbPresEventControl.Controls.Add(trPresEventEndDate);

				// Creates the row to enter the end Time
				trPresEventEndTime = new TableRow();
				trPresEventEndTime.ID = "trPresEventEndTime";

				tcPresEventEndTimeName = new TableCell();
				tcPresEventEndTimeName.ID = "tcPresEventEndTimeName";
				tcPresEventEndTimeName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventEndTimeName = new Literal();
				ltPresEventEndTimeName.ID = "ltPresEventEndTimeName";
				ltPresEventEndTimeName.Text = "End Time:";
				tcPresEventEndTimeName.Controls.Add(ltPresEventEndTimeName);

				tcPresEventEndTimeText = new TableCell();
				tcPresEventEndTimeText.ID = "tcPresEventEndTimeText";
				tcPresEventEndTimeText.HorizontalAlign = HorizontalAlign.Left;

				ltPresEventEndTimeText = new Literal();
				ltPresEventEndTimeText.ID = "ltPresEventEndTextHour";
				ltPresEventEndTimeText.Text = "<b>ENDTIME</b>";
				tcPresEventEndTimeText.Controls.Add(ltPresEventEndTimeText);

				trPresEventEndTime.Controls.Add(tcPresEventEndTimeName);
				trPresEventEndTime.Controls.Add(tcPresEventEndTimeText);

				tbPresEventControl.Controls.Add(trPresEventEndTime);

				// Creates the row to enter the location
				trPresEventLocation = new TableRow();
				trPresEventLocation.ID = "trPresEventLocation";

				tcPresEventLocationName = new TableCell();
				tcPresEventLocationName.ID = "tcPresEventLocationName";
				tcPresEventLocationName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventLocationName = new Literal();
				ltPresEventLocationName.ID = "ltPresEventLocationName";
				ltPresEventLocationName.Text = "Location:";
				tcPresEventLocationName.Controls.Add(ltPresEventLocationName);

				tcPresEventLocationText = new TableCell();
				tcPresEventLocationText.ID = "tcPresEventLocationText";
				tcPresEventLocationText.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventLocationText = new Literal();
				ltPresEventLocationText.ID = "ltPresEventLocationText";
				ltPresEventLocationText.Text = "<b>LOCATION</b>";
				tcPresEventLocationText.Controls.Add(ltPresEventLocationText);

				trPresEventLocation.Controls.Add(tcPresEventLocationName);
				trPresEventLocation.Controls.Add(tcPresEventLocationText);

				tbPresEventControl.Controls.Add(trPresEventLocation);
			
				// Creates the row to enter the category
				trPresEventCategory = new TableRow();
				trPresEventCategory.ID = "trPresEventCategory";

				tcPresEventCategoryName = new TableCell();
				tcPresEventCategoryName.ID = "tcPresEventCategoryName";
				tcPresEventCategoryName.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventCategoryName = new Literal();
				ltPresEventCategoryName.ID = "ltPresEventCategoryName";
				ltPresEventCategoryName.Text = "Category:";
				tcPresEventCategoryName.Controls.Add(ltPresEventCategoryName);

				tcPresEventCategoryText = new TableCell();
				tcPresEventCategoryText.ID = "tcPresEventCategoryText";
				tcPresEventCategoryText.HorizontalAlign = HorizontalAlign.Left;
				ltPresEventCategoryText = new Literal();
				ltPresEventCategoryText.ID = "ltPresEventCategoryText";
				ltPresEventCategoryText.Text = "<b>CATEGORY</b>";
				tcPresEventCategoryText.Controls.Add(ltPresEventCategoryText);

				trPresEventCategory.Controls.Add(tcPresEventCategoryName);
				trPresEventCategory.Controls.Add(tcPresEventCategoryText);

				tbPresEventControl.Controls.Add(trPresEventCategory);
			
				// Creates the row to enter the description
				trPresEventDescription = new TableRow();
				trPresEventDescription.ID = "trPresEventDescription";

				tcPresEventDescriptionName = new TableCell();
				tcPresEventDescriptionName.ID = "tcPresEventDescriptionName";
				tcPresEventDescriptionName.HorizontalAlign = HorizontalAlign.Left;
				tcPresEventDescriptionName.VerticalAlign = VerticalAlign.Top;
				ltPresEventDescriptionName = new Literal();
				ltPresEventDescriptionName.ID = "ltPresEventDescriptionName";
				ltPresEventDescriptionName.Text = "Description:";
				tcPresEventDescriptionName.Controls.Add(ltPresEventDescriptionName);

				tcPresEventDescriptionText = new TableCell();
				tcPresEventDescriptionText.ID = "tcPresEventDescriptionText";
				tcPresEventDescriptionText.HorizontalAlign = HorizontalAlign.Left;
				tcPresEventDescriptionText.VerticalAlign = VerticalAlign.Top;
				tcPresEventDescriptionText.Width = 400;
				ltPresEventDescriptionText = new Literal();
				ltPresEventDescriptionText.ID = "ltPresEventDescriptionText";
				ltPresEventDescriptionText.Text = "<b>DESCRIPTION<br>DESCRIPTION</b>";
				tcPresEventDescriptionText.Controls.Add(ltPresEventDescriptionText);

				trPresEventDescription.Controls.Add(tcPresEventDescriptionName);
				trPresEventDescription.Controls.Add(tcPresEventDescriptionText);

				tbPresEventControl.Controls.Add(trPresEventDescription);

				// Error control
				trPresEventErrorControl = new TableRow();
				trPresEventErrorControl.ID = "trPresEventErrorControl";

				tcPresEventErrorControl = new TableCell();
				tcPresEventErrorControl.ID = "tcPresEventErrorControl";
				tcPresEventErrorControl.HorizontalAlign = HorizontalAlign.Left;
				tcPresEventErrorControl.ColumnSpan = 2;
				ltPresEventErrorControl = new Literal();
				ltPresEventErrorControl.ID = "ltPresEventErrorControl";
				ltPresEventErrorControl.Visible = true;
				ltPresEventErrorControl.Text = "";
				tcPresEventErrorControl.Controls.Add(ltPresEventErrorControl);
				trPresEventErrorControl.Controls.Add(tcPresEventErrorControl);
				tbPresEventControl.Controls.Add(trPresEventErrorControl);

				presentationContainer.Controls.Add(tbPresEventControl);
			}
			catch(Exception ex)
			{
				this.sErrorMessage = String.Format("\n\nAn error occured when creating the default controls in presentation mode in the private method 'CreatePresentationEventDefaultChildControls'.\nThe error message is: '{0}'.", ex.Message);
			}
		}

		/// <summary>
		/// This method can be overriden to provide a custom presentation layer at presentation time.
		/// </summary>
		/// <example>
		/// <para>
		/// This example shows how to create two literals to render the title,
		/// the start and end date of an Event at presentation time.
		/// </para>
		/// <code>
		/// Literal ltPresEventTitle;
		/// Literal ltPresEventTitleValue;
		/// protected override void CreatePresentationEventChildControls(BaseModeContainer presentationContainer)
		/// {
		///		ltPresEventTitle = new Literal();
		///		ltPresEventTitle.ID = "ltPresEventTitle";
		///		ltPresEventTitle.Text = "Event details: ";
		///		presentationContainer.Controls.Add(ltPresEventTitle);
		/// 	ltPresEventTitleValue = new Literal();
		/// 	ltPresEventTitleValue.ID = "ltPresEventTitleValue";
		/// 	ltPresEventTitleValue.Text = "[EVENTDETAILS]";
		/// 	presentationContainer.Controls.Add(ltPresEventTitleValue);
		/// }
		/// </code>
		/// </example>
		/// <param name="presentationContainer">Presentation container on which controls can be attached</param>
		virtual protected void CreatePresentationEventChildControls(BaseModeContainer presentationContainer)
		{
		}

		/// <summary>
		/// <para>
		/// This method is inherited from the base placeholder control class.
		/// This method cannot be overriden to extend the control, it has been sealed.
		/// Instead, override the method <see cref="EventPlaceHolder.CreatePresentationEventChildControls"/>.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>
		/// The method initialises the protected properties needed to manage an Event data
		/// with the following values:
		/// <list type="bullet">
		/// <item><see cref="EventPlaceHolder.EventTitle"/>: string "Title"</item>
		/// <item><see cref="EventPlaceHolder.StartDate"/>: date and time of "DateTime.Now"</item>
		/// <item><see cref="EventPlaceHolder.EndDate"/>: date and time of "DateTime.Now.AddDays(1)"</item>
		/// <item><see cref="EventPlaceHolder.Location"/>: string "Location"</item>
		/// <item><see cref="EventPlaceHolder.Category"/>: string "Category"</item>
		/// <item><see cref="EventPlaceHolder.Description"/>: string "Description"</item>
		/// </list>
		/// </para>
		/// <para>
		/// If the control is set to display debugging information using the property <see cref="EventPlaceHolder.Debug"/>
		/// then a specific Literal control is also created.
		/// </para>
		/// <para>
		/// If the control is set to create the default controls at presentation time using the property <see cref="EventPlaceHolder.DefaultControls"/>
		/// then this method adds the default controls to the presentation container.
		/// </para>
		/// <para>
		/// The method <see cref="EventPlaceHolder.CreatePresentationEventChildControls"/> is called after all initialisation have occured.
		/// </para>
		/// </remarks>
		/// <param name="presentationContainer">Presentation container on which controls can be attached</param>
		protected sealed override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{
			try
			{
				// Sets the default values
				this.EventTitle = String.Format("Title");
				this.StartDate = DateTime.Now;
				this.EndDate = DateTime.Now.AddDays(1);
				this.Location = String.Format("Location");
				this.Category = String.Format("Category");
				this.Description = String.Format("Description");

				// If the default controls need to be created, calls teh associated method
				if(this.bDefaultControls)
					this.CreatePresentationEventDefaultChildControls(presentationContainer);

				// Creates the debugging control
				if(this.bDebug)
				{
					// Creates the debug control
					this.ltErrorPres = new Literal();
					this.ltErrorPres.ID = "ltErrorPres";
					this.ltErrorPres.Visible = this.bDebug;
					presentationContainer.Controls.Add(ltErrorPres);
				}
			}
			catch(Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error occurred in the overloaded method 'CreatePresentationChildControls'.\nThe error message is: '{0}'.", ex.Message);

				// Creates the debugging control
				if(this.bDebug)
				{
					// Creates the debug control
					this.ltErrorPres = new Literal();
					this.ltErrorPres.ID = "ltErrorPres";
					this.ltErrorPres.Text = this.sErrorMessage;
					this.ltErrorPres.Visible = this.bDebug;
					presentationContainer.Controls.Add(ltErrorPres);
				}
			}
			// Calls the method that can be overriden for development purposes
			this.CreatePresentationEventChildControls(presentationContainer);
		}
		#endregion

		#region Load Content for Authoring Mode
		/// <summary>
		/// This method is used internally to provide the default controls with the correct value at authoring time.
		/// </summary>
		/// <param name="e">Argument passed when the placeholder loads the content for authoring.</param>
		private void LoadPlaceholderDefaultContentForAuthoring(PlaceholderControlEventArgs e)
		{
			try
			{
				tbAuthEventTitleText.Text = this.EventTitle;
				tbAuthEventStartText.Text = this.StartDate.ToShortDateString();
				lbAuthEventStartTextHour.SelectedIndex = this.StartDate.Hour;
				lbAuthEventStartTextMinute.SelectedIndex = this.StartDate.Minute / 15;
				tbAuthEventEndText.Text = this.EndDate.ToShortDateString();
				lbAuthEventEndTextHour.SelectedIndex = this.EndDate.Hour;
				lbAuthEventEndTextMinute.SelectedIndex = this.EndDate.Minute / 15;
				tbAuthEventLocationText.Text = this.Location;
				tbAuthEventCategoryText.Text = this.Category;
				tbAuthEventDescriptionText.Text = this.Description;
			}
			catch(Exception ex)
			{
				this.sErrorMessage = String.Format("\n\nAn error occured when loading the values in the default controls in authoring mode in the private method 'LoadPlaceholderDefaultContentForAuthoring'.\nThe error message is: '{0}'.", ex.Message);
			}
		}

		/// <summary>
		/// <para>
		/// This method overrides the default placeholder method to load the information
		/// from the XML placeholder definition, and populate the properties of this placeholder instance.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>
		/// This method should not be overriden, as it contains some specific code to manage the XML document from the
		/// placeholder definition, and how to parse it.
		/// </para>
		/// <para>
		/// If the control is set to create the default controls at authoring time (see <see cref="EventPlaceHolder.DefaultControls"/>)
		/// then the default controls are populated by the XML placeholder Definition document data.
		/// </para>
		/// </remarks>
		/// <param name="e">Placeholder control event argument object.</param>
		protected sealed override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			try
			{
				this.LoadPHDataXmlDocument(((XmlPlaceholder)this.BoundPlaceholder).XmlAsString);
				// Calls the default population if it is set to be
				this.LoadPlaceholderDefaultContentForAuthoring(e);
			}
			catch (Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error has occured in the overloaded function 'LoadPlaceholderContentForAuthoring'.\nThe error message is: '{0}'.", ex.Message);
				// Sets the text for debugging information
				if(this.bDebug)
					this.ltErrorAuth.Text = this.sErrorMessage;
			}
		}
		#endregion

		#region Load Content for Presentation Mode
		/// <summary>
		/// This method is used internally to provide the default controls with the correct value at presentation time.
		/// </summary>
		/// <param name="e">Argument passed when the placeholder loads the content for authoring.</param>
		private void LoadPlaceholderDefaultContentForPresentation(PlaceholderControlEventArgs e)
		{
			try
			{
				ltPresEventTitleText.Text = this.EventTitle;
				ltPresEventStartDateText.Text = this.StartDate.ToShortDateString();
				ltPresEventStartTimeText.Text = this.StartDate.ToShortTimeString();
				ltPresEventEndDateText.Text = this.EndDate.ToShortDateString();
				ltPresEventEndTimeText.Text = this.EndDate.ToShortTimeString();
				ltPresEventLocationText.Text = this.Location;
				ltPresEventCategoryText.Text = this.Category;
				ltPresEventDescriptionText.Text = this.Description;
			}
			catch(Exception ex)
			{
				this.sErrorMessage = String.Format("\n\nAn error occured when loading the values in the default controls in presentation mode in the private method 'LoadPlaceholderDefaultContentForPresentation'.\nThe error message is: '{0}'.", ex.Message);
			}
		}

		/// <summary>
		/// This method can be overriden to provide a specific implementation on the way the Event placeholder
		/// properties populate the custom presentation layer at presentation time - see
		/// <see cref="EventPlaceHolder.CreatePresentationEventChildControls"/> for more information
		/// on how to create a custom presentation layer at presentation time.
		/// </summary>
		/// <example>
		/// <code>
		/// protected override void LoadEventPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		/// {
		/// 	ltPresEventTitleValue.Text = String.Format("The event <font color='red'>{0}</font> will take place between {1} and {2} at {3}",
		/// 	this.EventTitle, this.StartDate.ToShortDateString(), this.EndDate.ToShortDateString(), this.Location);
		/// }
		/// </code>
		/// </example>
		/// <param name="e">Placeholder control event argument object.</param>
		virtual protected void LoadEventPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
		}

		/// <summary>
		/// <para>
		/// This method overrides the default placeholder method to load the information
		/// from the XML placeholder definition, and populate the properties of this placeholder instance.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>
		/// This method cannot be overriden, as it contains some specific code to manage the XML document from the
		/// placeholder definition, and how to parse it.
		/// In order to extend the implementation of this method, the method
		/// <see cref="EventPlaceHolder.LoadEventPlaceholderContentForPresentation"/> can be overriden.
		/// This last method is called atfer processing the XML document from the placeholder definition.
		/// </para>
		/// <para>
		/// If the control is set to create the default controls at presentation time (see <see cref="EventPlaceHolder.DefaultControls"/>)
		/// then the default controls are populated by the XML placeholder Definition document data.
		/// </para>
		/// </remarks>
		/// <param name="e">Placeholder control event argument object.</param>
		protected sealed override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			try
			{
				this.LoadPHDataXmlDocument(((XmlPlaceholder)this.BoundPlaceholder).XmlAsString);
				// Populates the default controls if required
				if(this.bDefaultControls)
					this.LoadPlaceholderDefaultContentForPresentation(e);
			}
			catch (Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error has occured in the overloaded function 'LoadPlaceholderContentForPresentation'.\nThe error message is: '{0}'.", ex.Message);
				// Sets the text for debugging information
				if(this.bDebug)
					this.ltErrorPres.Text = this.sErrorMessage;
			}
			// Calls the method allowing a developer to customise what is done with the values
			// stored in the placeholder
			this.LoadEventPlaceholderContentForPresentation(e);
		}
		#endregion

		#region Save placeholder data
		/// <summary>
		/// This method is used internally to set the control properties properly to be saved
		/// in CMS.
		/// </summary>
		/// <param name="e">Argument passed when the placeholder saves the content</param>
		private void SaveEventPlaceholderDefaultContent(PlaceholderControlSaveEventArgs e)
		{
			try
			{
				// Gets the globalisation information for the date entered
				CultureInfo ci = new CultureInfo(Context.Request.UserLanguages[0]);

				// Gets hte title
				this.EventTitle = this.tbAuthEventTitleText.Text;

				// Gets the start date and time
				this.StartDate = DateTime.Parse(this.tbAuthEventStartText.Text, ci);
				this.StartDate = this.StartDate.AddHours(Convert.ToDouble(this.lbAuthEventStartTextHour.SelectedIndex));
				this.StartDate = this.StartDate.AddMinutes(Convert.ToDouble(this.lbAuthEventStartTextMinute.SelectedIndex) * 15);

				// Gets the end date and time
				this.EndDate = DateTime.Parse(this.tbAuthEventEndText.Text, ci);
				this.EndDate = this.EndDate.AddHours(Convert.ToDouble(this.lbAuthEventEndTextHour.SelectedIndex));
				this.EndDate = this.EndDate.AddMinutes(Convert.ToDouble(this.lbAuthEventEndTextMinute.SelectedIndex) * 15);

				// Get the location
				this.Location = this.tbAuthEventLocationText.Text;

				// Get the category
				this.Category = this.tbAuthEventCategoryText.Text;

				// Get the description
				this.Description = this.tbAuthEventDescriptionText.Text;
			}
			catch(Exception ex)
			{
				this.sErrorMessage = String.Format("\n\nAn error occured when setting the properties' value in the private method 'SaveEventPlaceholderDefaultContent'.\nThe error message is: '{0}'.", ex.Message);
			}
		}

		/// <summary>
		/// This method allows a developer to provide a custom implementation of the placeholder
		/// Save Event during authoring mode. This may proove useful if the data from the presentation
		/// layer at authoring time requires validation or transformation.
		/// </summary>
		/// <example>
		/// <para>
		/// This example shows how to set the Event placeholder properties, which will then be stored
		/// in the XML placeholder definition.
		/// </para>
		/// <para>
		/// It sets the title of the Event to the content of a TextBox control,
		/// the start date and time to [Now], the end date and time to [Now + 1 day]
		/// and sets the values of the other properties.
		/// </para>
		/// <code>
		/// protected override void SaveEventPlaceholderContent(PlaceholderControlSaveEventArgs e)
		/// {
		/// 	this.EventTitle = this.tbAuthEventTitle.Text;
		/// 	this.StartDate = DateTime.Now;
		/// 	this.EndDate = DateTime.Now.AddDays(1);
		/// 	this.Location = "London";
		/// 	this.Category = "Financial Forecast";
		/// 	this.Description = "This is the description for the event to come.";
		/// }
		/// </code>
		/// </example>
		/// <param name="e">Placeholder control save event argument object.</param>
		virtual protected void SaveEventPlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
		}

		/// <summary>
		/// <para>
		/// This method saves the content of the placeholder controls into a XML Document, which is
		/// stored in the XML placeholder defintion.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>
		/// This method cannot be overriden and has been sealed. However, to provide a custom implementation of
		/// this method, the method <see cref="EventPlaceHolder.SaveEventPlaceholderContent"/>
		/// can be overriden.
		/// </para>
		/// <para>
		/// If the default controls are used for this implementation (see <see cref="EventPlaceHolder.DefaultControls"/>),
		/// the content of the default controls will be first loaded in the placeholder properties,
		/// before calling the custom method for saving the content into these properties.
		/// </para>
		/// </remarks>
		/// <param name="e">Placeholder control save event arguments object.</param>
		protected sealed override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			// Verify is the default controls are set, and if so, applies the values
			// to the properties first
			// if(this.bDefaultControls)
				this.SaveEventPlaceholderDefaultContent(e);

			// Allows the user to set the data for himself
			this.SaveEventPlaceholderContent(e);
			try
			{
				// Creating an XML document to save data in the XML PlaceHolder
				XmlDocument phXMLData = CreatePHDataXmlDocument();

				// Saves the content in the CMS
				((XmlPlaceholder)this.BoundPlaceholder).XmlAsString  = phXMLData.InnerXml;
			}
			catch (Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error has occured in the overloaded function 'SavePlaceholderContent'.\nThe error message is: '{0}'", ex.Message);
			}
		}
		#endregion

		#region Helper functions
		/// <summary>
		///  This function loads the content from the XmlPlaceHolder onto the placeholder itself.
		///  It then populates the controls with the values found
		/// </summary>
		/// <param name="xmlStream">XML stream containing the data containing the definition of an Event</param>
		private void LoadPHDataXmlDocument(string xmlStream)
		{
			try
			{
				// Only if the XML placeholders contains some data.
				if(xmlStream.Length > 0)
				{
					// Creates the XML document
					XmlDataDocument phData  = new XmlDataDocument();

					// Loads the content of the 
					phData.LoadXml(xmlStream);
					XmlElement xmlRoot = phData.DocumentElement;

					// Data for the element "Title"
					XmlNode xmlTitle;
					xmlTitle = xmlRoot.SelectSingleNode("Title");
					this.EventTitle = xmlTitle.InnerXml;

					// Data for the element relating to the Start of the event
					XmlNode xmlStartDay, xmlStartMonth, xmlStartYear;
					XmlNode xmlStartHour, xmlStartMinute, xmlStartSecond;
					xmlStartDay = xmlRoot.SelectSingleNode("StartDay");
					xmlStartMonth = xmlRoot.SelectSingleNode("StartMonth");
					xmlStartYear = xmlRoot.SelectSingleNode("StartYear");
					xmlStartHour = xmlRoot.SelectSingleNode("StartHour");
					xmlStartMinute = xmlRoot.SelectSingleNode("StartMinute");
					xmlStartSecond = xmlRoot.SelectSingleNode("StartSecond");
					this.StartDate = new DateTime(
						Convert.ToInt32(xmlStartYear.InnerXml),
						Convert.ToInt32(xmlStartMonth.InnerXml),
						Convert.ToInt32(xmlStartDay.InnerXml),
						Convert.ToInt32(xmlStartHour.InnerXml),
						Convert.ToInt32(xmlStartMinute.InnerXml),
						Convert.ToInt32(xmlStartSecond.InnerXml));

					// Data for the element relating to the End of the event
					XmlNode xmlEndDay, xmlEndMonth, xmlEndYear;
					XmlNode xmlEndHour, xmlEndMinute, xmlEndSecond;
					xmlEndDay = xmlRoot.SelectSingleNode("EndDay");
					xmlEndMonth = xmlRoot.SelectSingleNode("EndMonth");
					xmlEndYear = xmlRoot.SelectSingleNode("EndYear");
					xmlEndHour = xmlRoot.SelectSingleNode("EndHour");
					xmlEndMinute = xmlRoot.SelectSingleNode("EndMinute");
					xmlEndSecond = xmlRoot.SelectSingleNode("EndSecond");
					this.EndDate = new DateTime(
						Convert.ToInt32(xmlEndYear.InnerXml),
						Convert.ToInt32(xmlEndMonth.InnerXml),
						Convert.ToInt32(xmlEndDay.InnerXml),
						Convert.ToInt32(xmlEndHour.InnerXml),
						Convert.ToInt32(xmlEndMinute.InnerXml),
						Convert.ToInt32(xmlEndSecond.InnerXml));

					// Data for the element "Location"
					XmlNode xmlLocation;
					xmlLocation = xmlRoot.SelectSingleNode("Location");
					this.Location = xmlLocation.InnerXml;

					// Data for the element "Category"
					XmlNode xmlCategory;
					xmlCategory = xmlRoot.SelectSingleNode("Category");
					this.Category = xmlCategory.InnerXml;

					// Data for the element "Description"
					XmlNode xmlDescription;
					xmlDescription = xmlRoot.SelectSingleNode("Description");
					this.Description = xmlDescription.InnerXml;
				}
				else
				{
					this.EventTitle = String.Format("Title");
					this.StartDate = DateTime.Now;
					this.EndDate = DateTime.Now.AddDays(1);
					this.Location = String.Format("Location");
					this.Category = String.Format("Category");
					this.Description = String.Format("Description");
				}
			}
			catch (Exception ex)
			{
				this.sErrorMessage += String.Format("\n\nAn error has occured in the private function 'LoadPHDataXmlDocument'.\nThe error message is: '{0}'.", ex.Message);
			}
		}
		/// <summary>
		/// This function creates and returns the XML Document associated with
		/// the content of the placeholder
		/// </summary>
		/// <returns>XmlDocument containing the content of the Event variables</returns>
		private XmlDocument CreatePHDataXmlDocument()
		{
			// Variable returned by the function contianing the XML Document
			XmlDocument phData = new XmlDocument();

			try
			{
				// Loads up the top level document and gets the root node
				phData.LoadXml(String.Format("<{0}></{0}>", xmlRootNode));
				XmlNode phRoot = phData.DocumentElement;

				// Creates the child element "Title"
				XmlElement phElemTitle = phData.CreateElement("Title");
				phElemTitle.InnerText = this.EventTitle;
				phRoot.AppendChild(phElemTitle);

				// Creates the child element "StartDay"
				XmlElement phElemStartDay = phData.CreateElement("StartDay");
				phElemStartDay.InnerText = this.StartDate.Day.ToString();
				phRoot.AppendChild(phElemStartDay);

				// Creates the child element "StartMonth"
				XmlElement phElemStartMonth = phData.CreateElement("StartMonth");
				phElemStartMonth.InnerText = this.StartDate.Month.ToString();
				phRoot.AppendChild(phElemStartMonth);

				// Creates the child element "StartYear"
				XmlElement phElemStartYear = phData.CreateElement("StartYear");
				phElemStartYear.InnerText = this.StartDate.Year.ToString();
				phRoot.AppendChild(phElemStartYear);

				// Creates the child element "StartHour"
				XmlElement phElemStartHour = phData.CreateElement("StartHour");
				phElemStartHour.InnerText = this.StartDate.Hour.ToString();
				phRoot.AppendChild(phElemStartHour);

				// Creates the child element "StartMinute"
				XmlElement phElemStartMinute = phData.CreateElement("StartMinute");
				phElemStartMinute.InnerText = this.StartDate.Minute.ToString();
				phRoot.AppendChild(phElemStartMinute);

				// Creates the child element "StartSecond"
				XmlElement phElemStartSecond = phData.CreateElement("StartSecond");
				phElemStartSecond.InnerText = this.StartDate.Second.ToString();
				phRoot.AppendChild(phElemStartSecond);

				// Creates the child element "EndDay"
				XmlElement phElemEndDay = phData.CreateElement("EndDay");
				phElemEndDay.InnerText = this.EndDate.Day.ToString();
				phRoot.AppendChild(phElemEndDay);

				// Creates the child element "EndMonth"
				XmlElement phElemEndMonth = phData.CreateElement("EndMonth");
				phElemEndMonth.InnerText = this.EndDate.Month.ToString();
				phRoot.AppendChild(phElemEndMonth);

				// Creates the child element "EndYear"
				XmlElement phElemEndYear = phData.CreateElement("EndYear");
				phElemEndYear.InnerText = this.EndDate.Year.ToString();
				phRoot.AppendChild(phElemEndYear);

				// Creates the child element "EndHour"
				XmlElement phElemEndHour = phData.CreateElement("EndHour");
				phElemEndHour.InnerText = this.EndDate.Hour.ToString();
				phRoot.AppendChild(phElemEndHour);

				// Creates the child element "EndMinute"
				XmlElement phElemEndMinute = phData.CreateElement("EndMinute");
				phElemEndMinute.InnerText = this.EndDate.Minute.ToString();
				phRoot.AppendChild(phElemEndMinute);

				// Creates the child element "EndSecond"
				XmlElement phElemEndSecond = phData.CreateElement("EndSecond");
				phElemEndSecond.InnerText = this.EndDate.Second.ToString();
				phRoot.AppendChild(phElemEndSecond);

				// Creates the child element "Location"
				XmlElement phElemLocation = phData.CreateElement("Location");
				phElemLocation.InnerText = this.Location;
				phRoot.AppendChild(phElemLocation);

				// Creates the child element "Category"
				XmlElement phElemCategory = phData.CreateElement("Category");
				phElemCategory.InnerText = this.Category;
				phRoot.AppendChild(phElemCategory);

				// Creates the child element "Description"
				XmlElement phElemDescription = phData.CreateElement("Description");
				phElemDescription.InnerText = this.Description;
				phRoot.AppendChild(phElemDescription);
			}
			catch (Exception e)
			{
				this.sErrorMessage += String.Format("\n\nAn error has occured in the private function 'CreatePHDataXmlDocument'.\nThe error message is: '{0}'.", e.Message);
			}

			return phData;
		}

		/// <summary>
		/// This method validates the data entry for the Event placeholder.
		/// This method needs to be overriden to ensure that the data passed for the event is
		/// correct, in particular, that the dates are valid dates, and that the end date
		/// is after the start date. by default, this method returns true.
		/// </summary>
		/// <example>
		/// This example shows how to ensure that the title is at least one character in length,
		/// and that the start date is before the end date.
		/// <code>
		/// protected override bool ValidateCustomControls()
		/// {
		/// 	bool bReturn;
		/// 	if(this.EventTitle.Length == 0)
		/// 		bReturn = false;
		/// 	if(this.StartDate > this.EndDate)
		/// 		bReturn = false;
		/// 		
		/// 	return bReturn;
		/// }
		/// </code>
		/// </example>
		/// <returns>Must return true if the data is validated</returns>
		virtual protected bool ValidateCustomControls()
		{
			return true;
		}

		/// <summary>
		/// This private method veridies that the controls are correct, in aprticular the default ones,
		/// and cancels savign the dat if not validated.
		/// </summary>
		/// <param name="oEvent">Placeholder control saving event argument object.</param>
		private void ValidateControl(PlaceholderControlSavingEventArgs oEvent)
		{
			bool bReturn = true;

			this.EnsureChildControls();

			// If the default controls have been selected, then validates them
			//if(this.bDefaultControls)
				bReturn = this.ValidateDefaultControl(oEvent);

			// In any case, verifies the custom controls, if any
			bReturn = (bReturn && this.ValidateCustomControls());

			// Cancel the saving of data if the controls are not validated.
			oEvent.Cancel = !bReturn;
		}

		/// <summary>
		/// This private method is to provide some information on to what can go wrong.
		/// when using the default controls
		/// </summary>
		/// <param name="sError">Error message to display</param>
		/// <param name="oEvent">Saving content argument</param>
		private void DisplayUserError(string sError, PlaceholderControlSavingEventArgs oEvent)
		{
//			if(this.bDefaultControls)
//			{
				if(oEvent.IsTemporarySaveForPreview)
					ltPresEventErrorControl.Text = String.Format("<span><font size=1 color='red'><b>{0}</b></font></span>", sError);
				else
				{
					ltAuthEventErrorControl.Text = String.Format("<span><font size=1 color='red'><b>{0}</b></font></span>", sError);
					oEvent.Cancel = true;
				}
//			}
		}

		/// <summary>
		/// This method validates the data entry for the placeholder Control
		/// </summary>
		/// <returns>Returns true if the data is valid</returns>

		private bool ValidateDefaultControl(PlaceholderControlSavingEventArgs oEvent)
		{
			bool bReturn = false;

//			if(this.bDefaultControls)
//			{
				//Resets the variables for the error messages
				if(oEvent.IsTemporarySaveForPreview)
					this.ltPresEventErrorControl.Text = "";
				else
					this.ltAuthEventErrorControl.Text = "";

				try
				{
					bReturn = true;

					if(bReturn)
					{
						// Gets the globalisation information for the date entered
						CultureInfo ci = new CultureInfo(Context.Request.UserLanguages[0]);

						// Gets the start date and time
						DateTime dtStart = DateTime.Parse(this.tbAuthEventStartText.Text, ci);
						DateTime dtEnd = DateTime.Parse(this.tbAuthEventEndText.Text, ci);

						if(dtStart > dtEnd)
						{
							bReturn = false;
							DisplayUserError("Please specify a start date before the end date.", oEvent);
						}
					}

					// Checks the length of the entry Title
					if(bReturn)
					{
						if(this.tbAuthEventTitleText.Text.Trim().Length == 0)
						{
							bReturn = false;
							DisplayUserError("Please specify an event title.", oEvent);
						}
					}
					// Checks the length of the entry Title
					if(bReturn)
					{
						if(this.tbAuthEventTitleText.Text.Trim().Length > 100)
						{
							bReturn = false;
							DisplayUserError("The event title cannot be more than 100 characters.", oEvent);
						}
					}
					// Checks the length of the entry Location
					if(bReturn)
					{
						if(this.tbAuthEventLocationText.Text.Trim().Length == 0)
						{
							bReturn = false;
							DisplayUserError("Please specify an event location.", oEvent);
						}
					}
					// Checks the length of the entry Location
					if(bReturn)
					{
						if(this.tbAuthEventLocationText.Text.Trim().Length > 100)
						{
							bReturn = false;
							DisplayUserError("The event location cannot be more than 100 characters.", oEvent);
						}
					}
					// Checks the length of the entry Category
					if(bReturn)
					{
						if(this.tbAuthEventCategoryText.Text.Trim().Length == 0)
						{
							bReturn = false;
							DisplayUserError("Please specify an event category.", oEvent);
						}
					}
					// Checks the length of the entry Category
					if(bReturn)
					{
						if(this.tbAuthEventCategoryText.Text.Trim().Length > 100)
						{
							bReturn = false;
							DisplayUserError("The event category cannot be more than 100 characters.", oEvent);
						}
					}
					// Checks the length of the entry description
					if(bReturn)
					{
						if(this.tbAuthEventDescriptionText.Text.Trim().Length > 3200)
						{
							bReturn = false;
							DisplayUserError("The event description cannot be more than 3200 characters.", oEvent);
						}
					}
				}
				catch
				{
					DisplayUserError("An error occured whilst validating the Event data.", oEvent);
				}

//			}
//			else
//				bReturn = true;

			return bReturn;
		}
		#endregion

		#region Overiden functions needed for the component
		/// <summary>
		/// This method ensures the view state is set properly for the Event placeholder.
		/// This member handles mandatory component initialization. It has been sealed.
		/// </summary>
		/// <param name="e">EventArgs parameter</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();

			if (WebAuthorContextMode.PresentationUnpublished == WebAuthorContext.Current.Mode)
				this.EnableViewState = false;
			base.OnInit (e);
		}

		/// <summary>
		/// This method validates the content of the Event placeholder.
		/// It will cancel saving the Event if the method <see cref="EventPlaceHolder.ValidateCustomControls"/>
		/// returns false.
		/// </summary>
		/// <param name="e">Placeholder control saving event argument object.</param>
		protected override void OnSavingContent(PlaceholderControlSavingEventArgs e)
		{
			base.OnSavingContent (e);
			this.ValidateControl(e);
			if(e.Cancel)
			{
				string sError;
				if(e.IsTemporarySaveForPreview)
					sError = ltPresEventErrorControl.Text;
				else
					sError = ltAuthEventErrorControl.Text;

				WebAuthorContext.Current.RaiseErrorEvent(new WebAuthorErrorEventArgs(ErrorModeContainer.FailedSavePlaceholder, new FormatException(sError)));
			}
		}
		#endregion
	}
}
