using System;
using System.ComponentModel;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Web.UI;
using System.Web.UI.WebControls;
using ioko.ComponentModel.LicenseProvider;
using Microsoft.ContentManagement.Publishing;

namespace MSIBPlusPack.ContentManagement.Publishing.ServerControl
{
	/// <summary>
	/// This server control allows the user to view a list of the events currently
	/// available on the site.
	/// <remarks>
	/// <para>
	/// This Event server control works in conjunction with the Event placeholder within the
	/// MSIBPlusPack.ContentManagement.Publishing.Placeholders namespace.
	/// </para>
	/// <para>
	/// The Event server control can be used with some default controls. In order to use this
	/// functionality, the property <see cref="EventServerControl.DefaultControls"/>
	/// must be set to true. By setting this property to true, all events within the next calendar
	/// month will be displayed in a table.
	/// </para>
	/// <para>
	/// Only postings with the Event placeholder which have been approved 
	/// are stored in the MSIB Event database and can be displayed in the list of events.
	/// </para>
	/// <para>
	/// The server control does a look up in the MSIB Event database. It is possible to filter the
	/// results from the database by setting the properties of the public properties:
	/// <list type="bullet">
	/// <item>Event title <see cref='EventServerControl.m_EventTitle'></see></item>
	/// <item>Event start date and time <see cref='EventServerControl.m_EventStartDate'></see></item>
	/// <item>Event end date and time <see cref='EventServerControl.m_EventEndDate'></see></item>
	/// <item>Event location <see cref='EventServerControl.m_EventLocation'></see></item>
	/// <item>Event category <see cref='EventServerControl.m_EventCategory'></see></item>
	/// </list>
	/// When values are set on these public properties, a look up is done in database to retrieve only the
	/// Events matching the search criteria.
	/// </para>
	/// </remarks>
	/// </summary>
	[
		DefaultProperty("Text"), 
			ToolboxData("<{0}:EventServerControl runat=server></{0}:EventServerControl>")
	]
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public class EventServerControl : WebControl
	{
		#region Class properties
		/// <summary>
		/// Internal version of the property Text
		/// </summary>
		private string text = "MSIB Event List Server Control";
		/// <summary>
		/// This property can be used to set the appearance name of the server control.
		/// </summary>
		[
			Category("Appearance"), 
				DefaultValue("MSIB Event List Server Control")
		] 
		public string Text 
		{
			get { return text; }
			set { text = value; }
		}

		/// <summary>
		/// This property is the internal version of the property DefaultControls
		/// </summary>
		private bool bDefaultControls = true;
		/// <summary>
		/// <para>
		/// This property allows a developer to set whether to use the default controls for
		/// the Event server control. This provides a basic implementation for the rendering
		/// of the list of events within CMS.
		/// </para>
		/// <para>When set to true, the server control will create a table containing all the Events available
		/// within the next calendar month, i.e. all Events which have a start and end date both included within the
		/// next month.
		/// </para>
		/// <para>
		/// The results will be displayed as a table containing:
		/// <list type="bullet">
		/// <item>The Event title and a link to the Event posting</item>
		/// <item>The Event start date</item>
		/// <item>The Event end date</item>
		/// <item>The Event location</item>
		/// <item>The Event category</item>
		/// </list>
		/// </para>
		/// </summary>
		[
			Category("MSIB Plus Pack - Events"), 
				DefaultValue("true"),
				Description("This property allows to select the default controls when this Event server control is implemented.")
		]
		public bool DefaultControls
		{
			get { return bDefaultControls; }
			set { bDefaultControls = value; }
		}

		/// <summary>
		/// <para>
		/// This public property allows a developer to set a filter on the query passed in the database
		/// to retrieve a list of Events. When set, the value is passed to the stored procedure
		/// "sp_ListEvents", and a look up is done on the exact value for the Event title.
		/// </para>
		/// <para>
		/// If the property is not set, its default value is "null" and the stored procedure will not
		/// filter on the title.
		/// </para>
		/// </summary>
		public string m_EventTitle = null;

		/// <summary>
		/// <para>
		/// This public property allows a developer to set a filter on the query passed in the database
		/// to retrieve a list of Events. When set, the value is passed to the stored procedure
		/// "sp_ListEvents", and a look up is to select Events which start date is greater than the date
		/// and time set.
		/// </para>
		/// <para>
		/// If the property is not set, its default value is set to the default DateTime 
		/// and the stored procedure will not filter on the start date of the Event.
		/// </para>
		/// </summary>
		public DateTime m_EventStartDate = new DateTime();

		/// <summary>
		/// <para>
		/// This public property allows a developer to set a filter on the query passed in the database
		/// to retrieve a list of Events. When set, the value is passed to the stored procedure
		/// "sp_ListEvents", and a look up is to select Events which end date is before the date
		/// and time set.
		/// </para>
		/// <para>
		/// If the property is not set, its default value is set to the default DateTime 
		/// and the stored procedure will not filter on the end date of the Event.
		/// </para>
		/// </summary>
		public DateTime m_EventEndDate = new DateTime();

		/// <summary>
		/// <para>
		/// This public property allows a developer to set a filter on the query passed in the database
		/// to retrieve a list of Events. When set, the value is passed to the stored procedure
		/// "sp_ListEvents", and a look up is done on the exact value for the Event location.
		/// </para>
		/// <para>
		/// If the property is not set, its default value is "null" and the stored procedure will not
		/// filter on the location.
		/// </para>
		/// </summary>
		public string m_EventLocation = null;

		/// <summary>
		/// <para>
		/// This public property allows a developer to set a filter on the query passed in the database
		/// to retrieve a list of Events. When set, the value is passed to the stored procedure
		/// "sp_ListEvents", and a look up is done on the exact value for the Event category.
		/// </para>
		/// <para>
		/// If the property is not set, its default value is "null" and the stored procedure will not
		/// filter on the category.
		/// </para>
		/// </summary>
		public string m_EventCategory = null;
		#endregion

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		/// <summary>
		/// This member handles mandatory component initialization. It has been sealed.
		/// </summary>
		/// <param name="e">EventArgs parameter</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();

			base.OnInit (e);
		}

		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false; 
				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;
				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(EventServerControl),this);
					lastDate = DateTime.Today;
				}
				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;
					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}
						return;
					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;
					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}           
			} 
			catch {}
				throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion

		#region Class methods
		/// <summary>
		/// <para>
		/// This method allows a user to retrieve a DataSet containing all information about the Event postings
		/// held in MSIB Event database.
		/// </para>
		/// <para>
		/// The DataSet returned contains one DataTable called "Events". This data table contains the
		/// following columns:
		/// </para>
		/// <list type="bullet">
		/// <item>Title: event title - type: string</item>
		/// <item>StartDateTime: event start date and time - type: DateTime</item>
		/// <item>EndDateTime: event end date and time - type: DateTime</item>
		/// <item>Category: category of the event - type: string</item>
		/// <item>Location: location of the event - type: string</item>
		/// <item>Description: description of the event - type: string</item>
		/// <item>PostingID: GUID of the associated posting - type: string</item>
		/// <item>CMS_URL: internal CMS URL for the posting - type: string</item>
		/// <item>CMS_PageName: posting name - type: string</item>
		/// <item>CMS_PageDisplayName: posting display name - type: string</item>
		/// <item>CMS_StartDate: date and time when the posting needs to be displayed - type: DateTime</item>
		/// <item>CMS_ExpiryDate: date and time when the posting will stopped to be displayed - type: DateTime</item>
		/// <item>CMS_RevisionDate: date and time for the posting revision - type: DateTime</item>
		/// </list>
		/// <para>
		/// The content of the DataSet returns will only return Events which have been approved.
		/// </para>
		/// <para>
		/// In order to build the DataSet, the stored procedure "sp_ListEvents" is called
		/// with the parameters set with the public properties:
		/// <list type="bullet">
		/// <item>Event title <see cref='EventServerControl.m_EventTitle'></see></item>
		/// <item>Event start date and time <see cref='EventServerControl.m_EventStartDate'></see></item>
		/// <item>Event end date and time <see cref='EventServerControl.m_EventEndDate'></see></item>
		/// <item>Event location <see cref='EventServerControl.m_EventLocation'></see></item>
		/// <item>Event category <see cref='EventServerControl.m_EventCategory'></see></item>
		/// </list>
		/// It is possible to modify this stored procedure to provide better filtering of the Events,
		/// provided that the same parameters and the same columns are being returned.
		/// </para>
		/// </summary>
		/// <returns>DataSet containing the list of Events in CMS.</returns>
		public DataSet GetEventList()
		{
			// Gets the conenction string to the database
			string sDBConn = ConfigurationSettings.AppSettings["msibEventDB"];

			// Data set to be returned
			DataSet dsEvent = new DataSet();

			// Create a new DataTable
			DataTable dtEvent = dsEvent.Tables.Add("Events");

			// Sets the columns for the table
			DataColumn dcTitle = dtEvent.Columns.Add("Title", Type.GetType("System.String"));
			DataColumn dcStartDateTime = dtEvent.Columns.Add("StartDateTime", Type.GetType("System.DateTime"));
			DataColumn dcEndDateTime = dtEvent.Columns.Add("EndDateTime", Type.GetType("System.DateTime"));
			DataColumn dcCategory = dtEvent.Columns.Add("Category", Type.GetType("System.String"));
			DataColumn dcLocation = dtEvent.Columns.Add("Location", Type.GetType("System.String"));
			DataColumn dcDescription = dtEvent.Columns.Add("Description", Type.GetType("System.String"));
			DataColumn dcPostingID = dtEvent.Columns.Add("PostingID", Type.GetType("System.String"));
			DataColumn dcCMSURL = dtEvent.Columns.Add("CMS_URL", Type.GetType("System.String"));
			DataColumn dcCMSPageName = dtEvent.Columns.Add("CMS_PageName", Type.GetType("System.String"));
			DataColumn dcCMSPageDisplayName = dtEvent.Columns.Add("CMS_PageDisplayName", Type.GetType("System.String"));
			DataColumn dcCMSStartDate = dtEvent.Columns.Add("CMS_StartDate", Type.GetType("System.DateTime"));
			DataColumn dcCMSExpiryDate = dtEvent.Columns.Add("CMS_ExpiryDate", Type.GetType("System.DateTime"));
			DataColumn dcCMSRevisionDate = dtEvent.Columns.Add("CMS_RevisionDate", Type.GetType("System.DateTime"));

			// Gets the data filtered from the database
			SqlConnection oDBConn = new SqlConnection();
			oDBConn.ConnectionString = sDBConn;
			try
			{
				SqlCommand oCommand = new SqlCommand();
				oCommand.CommandText = "sp_ListEvents";
				oCommand.CommandType = CommandType.StoredProcedure;
				oCommand.Connection = oDBConn;

				// Passing the title
				if(this.m_EventTitle != null)
				{
					SqlParameter oParamTitle = new SqlParameter("@EventTitle", SqlDbType.NVarChar, 100);
					oParamTitle.Value = this.m_EventTitle;
					oCommand.Parameters.Add(oParamTitle);
				}

				// Passing the start date
				DateTime dt = new DateTime();
				if(this.m_EventStartDate != dt)
				{
					SqlParameter oParamStartDT = new SqlParameter("@StartDT", SqlDbType.DateTime);
					oParamStartDT.Value = this.m_EventStartDate;
					oCommand.Parameters.Add(oParamStartDT);
				}

				// Passing the end date
				if(this.m_EventEndDate != dt)
				{
					SqlParameter oParamEndDT = new SqlParameter("@EndDT", SqlDbType.DateTime);
					oParamEndDT.Value = this.m_EventEndDate;
					oCommand.Parameters.Add(oParamEndDT);
				}

				// Passing the location
				if(this.m_EventLocation != null)
				{
					SqlParameter oParamLocation = new SqlParameter("@Location", SqlDbType.NVarChar, 100);
					oParamLocation.Value = this.m_EventLocation;
					oCommand.Parameters.Add(oParamLocation);
				}

				// Passing the category
				if(this.m_EventCategory != null)
				{
					SqlParameter oParamCategory = new SqlParameter("@Category", SqlDbType.NVarChar, 100);
					oParamCategory.Value = this.m_EventCategory;
					oCommand.Parameters.Add(oParamCategory);
				}

				oDBConn.Open();

				SqlDataReader drReader = oCommand.ExecuteReader();
				while (drReader.Read())
				{
					// Creates a new row in the DataSet
					DataRow drEvent = dtEvent.NewRow();

					// Gets the basic information
					drEvent["Title"] = drReader.GetString(0);
					drEvent["StartDateTime"] = drReader.GetDateTime(1);
					drEvent["EndDateTime"] = drReader.GetDateTime(2);
					drEvent["Location"] = drReader.GetString(3);
					drEvent["Category"] = drReader.GetString(4);
					drEvent["Description"] = drReader.GetString(5);

					// Gets the posting information
					string sPostingID = drReader.GetString(6);
					drEvent["PostingID"] = sPostingID;

					HierarchyItem oPosting = CmsHttpContext.Current.Searches.GetByGuid(sPostingID);
					if(oPosting is Posting)
					{
						// Gets the information on the posting
						Posting p = (Posting)oPosting;
						drEvent["CMS_URL"] = p.Url;
						drEvent["CMS_PageName"] = p.Name;
						drEvent["CMS_PageDisplayName"] = p.DisplayName;
						drEvent["CMS_StartDate"] = p.StartDate;
						drEvent["CMS_ExpiryDate"] = p.ExpiryDate;
						drEvent["CMS_RevisionDate"] = p.RevisionDate;
					}

					// Adds the row to the data table
					dtEvent.Rows.Add(drEvent);
				}
			}
			catch(Exception ex)
			{
				string sMessage = String.Format("\n\nAn error has occured getting the list of valid Event postings.\nThe error message is: '{0}'.", ex.Message);
			}
			finally
			{
				oDBConn.Close();
			}

			return dsEvent;
		}
		#endregion

		#region Helper functions
		/// <summary>
		/// This method creates the default controls, a table, containing all the postings containing
		/// valid events for the  month to come.
		/// </summary>
		private void CreateDefaultControls()
		{
			try
			{
				Table tbEvents = new Table();
				tbEvents.ID = "tbEvents";
				tbEvents.CellPadding = 4;
				tbEvents.CellSpacing = 0;
				tbEvents.Width = 500;

				TableRow trTitle = new TableRow();
				trTitle.ID = "trTitle";

				TableCell tcTitle = new TableCell();
				tcTitle.ID = "tcTitle";
				tcTitle.ColumnSpan = 4;
				tcTitle.HorizontalAlign = HorizontalAlign.Center;
				Literal ltTitle = new Literal();
				ltTitle.ID = "ltTitle";
				ltTitle.Text = String.Format("List of Events from {0} to {1}", DateTime.Now.ToShortDateString(), DateTime.Now.AddMonths(1).ToShortDateString());
				tcTitle.Controls.Add(ltTitle);
				trTitle.Controls.Add(tcTitle);
				tbEvents.Controls.Add(trTitle);

				TableRow trHeader = new TableRow();
				trHeader.ID = "trHeader";

				// EventTitle
				TableCell tbEventTitle = new TableCell();
				tbEventTitle.ID = "tbEventTitle";
				Literal ltEventTitle = new Literal();
				ltEventTitle.ID = "ltEventTitle";
				ltEventTitle.Text = "<b>Title</b>";
				tbEventTitle.Controls.Add(ltEventTitle);
				trHeader.Controls.Add(tbEventTitle);
				// Start
				TableCell tbStart = new TableCell();
				tbStart.ID = "tbStart";
				Literal ltStart = new Literal();
				ltStart.ID = "ltStart";
				ltStart.Text = "<b>Start</b>";
				tbStart.Controls.Add(ltStart);
				trHeader.Controls.Add(tbStart);
				// End
				TableCell tbEnd = new TableCell();
				tbEnd.ID = "tbEnd";
				Literal ltEnd = new Literal();
				ltEnd.ID = "ltEnd";
				ltEnd.Text = "<b>End</b>";
				tbEnd.Controls.Add(ltEnd);
				trHeader.Controls.Add(tbEnd);
				// Location
				TableCell tbLocation = new TableCell();
				tbLocation.ID = "tbLocation";
				Literal ltLocation = new Literal();
				ltLocation.ID = "ltLocation";
				ltLocation.Text = "<b>Location</b>";
				tbLocation.Controls.Add(ltLocation);
				trHeader.Controls.Add(tbLocation);
				// Category
				TableCell tbCategory = new TableCell();
				tbCategory.ID = "tbCategory";
				Literal ltCategory = new Literal();
				ltCategory.ID = "ltCategory";
				ltCategory.Text = "<b>Category</b>";
				tbCategory.Controls.Add(ltCategory);
				trHeader.Controls.Add(tbCategory);

				tbEvents.Controls.Add(trHeader);

				// Sets the default values for the public properties
				this.m_EventTitle = null;
				this.m_EventStartDate = DateTime.Now;
				this.m_EventEndDate = DateTime.Now.AddMonths(1);
				this.m_EventLocation = null;
				this.m_EventCategory = null;

				// Creates the list of items in the table
				DataSet ds = this.GetEventList();
				DataTable dt = ds.Tables["Events"];
				foreach(DataRow dr in dt.Rows)
				{
					// Creates the new Table row
					TableRow tr = new TableRow();
					TableCell tc_Title = new TableCell();
					TableCell tc_Start = new TableCell();
					TableCell tc_End = new TableCell();
					TableCell tc_Loc = new TableCell();
					TableCell tc_Cat = new TableCell();

					tc_Title.Controls.Add(new LiteralControl(String.Format("<a href='{0}'>{1}</a>", (string)dr.ItemArray[7], (string)dr.ItemArray[0])));
					tc_Start.Controls.Add(new LiteralControl(String.Format("{0}", ((DateTime)dr.ItemArray[1]).ToShortDateString())));
					tc_End.Controls.Add(new LiteralControl(String.Format("{0}", ((DateTime)dr.ItemArray[2]).ToShortDateString())));
					tc_Loc.Controls.Add(new LiteralControl(String.Format("{0}", (string)dr.ItemArray[4])));
					tc_Cat.Controls.Add(new LiteralControl(String.Format("{0}", (string)dr.ItemArray[3])));

					tr.Controls.Add(tc_Title);
					tr.Controls.Add(tc_Start);
					tr.Controls.Add(tc_End);
					tr.Controls.Add(tc_Loc);
					tr.Controls.Add(tc_Cat);
					
					tbEvents.Controls.Add(tr);
				}

				// Adds the table to the list of server controls
				this.Controls.Add(tbEvents);
			}
			catch(Exception ex)
			{
				string sMessage = String.Format("\n\nAn error has occured in the private function 'CreateDefaultControls'. The error message is: '{0}'.", ex.Message);
			}
		}
		#endregion
	
		#region Overriden Methods
		/// <summary>
		/// <para>
		/// This method is used to create the children controls needed for the Event server control
		/// when the default controls are selected (see the property <see cref="EventServerControl.DefaultControls"/>).
		/// </para>
		/// <para>
		/// This method can be overriden to provide an implementation to display some specific controls.
		/// </para>
		/// </summary>
		protected override void CreateChildControls()
		{
			// If the user has selected the default controls to be created
			if(this.bDefaultControls)
				this.CreateDefaultControls();

			base.CreateChildControls ();
		}

		/// <summary>
		/// <para>
		/// This overriden method renders this control to the output parameter specified.
		/// </para>
		/// <para>
		/// This method can be overriden to use the dataset 
		/// </para>
		/// </summary>
		/// <param name="output">HTML Text Writer object.</param>
		protected override void Render(HtmlTextWriter output)
		{
			this.RenderChildren(output);
		}
		#endregion
	}
}
