using System.Collections.Specialized;

namespace MSIBPlusPack.CommerceServer.Runtime.Configuration
{
	/// <summary>
	/// An enum of the possible local stores for Tickets issued by the external authentication system
	/// Possible values are:
	///		CommerceProfile - in a Commerce Server UserObject Profile
	///		Session - temporary session cookie
	///		Cookie - permanent cookie
	/// </summary>
	public enum TicketStoreType
	{
		/// <summary>
		/// Store in a Commerce Server UserObject Profile
		/// </summary>
		CommerceProfile,
		/// <summary>
		/// Store in a temporary session cookie
		/// </summary>
		Session,
		/// <summary>
		/// Do not store tickets
		/// </summary>
		None
	}

	/// <summary>
	/// This is a helper class for storing the configuration between the web.config and the implementation of the external authentication
	/// It is only ever used internally.
	/// </summary>
	public class ExternalAuthenticationConfiguration
	{		
		NameValueCollection colPluginConfig;
		string strTypeName;
		string strAssemblyName;
		TicketStoreType ticketStoreType;
		string strTicketStoreTicketName;
		string strTicketStoreTicketExpiryName;

		/// <summary>
		/// Constructor for ExternalAuthenticationConfiguration. Requires a typename and the assembly where the type can be found
		/// </summary>
		/// <param name="typeName">A type that exists within the assembly</param>
		/// <param name="assemblyName">An assembly that is referenced within the web.config assemblies section</param>
		public ExternalAuthenticationConfiguration(string typeName, string assemblyName)
		{
			colPluginConfig = new NameValueCollection();
			strTypeName = typeName;
			strAssemblyName = assemblyName;
			ticketStoreType = TicketStoreType.None;
		}

		/// <summary>
		/// Constructuor for ExternalAuthenticationConfiguration. This is an overloaded version of the constructor for use when ticketing information is to be stored
		/// </summary>
		/// <param name="typeName">A type that exists within the assembly</param>
		/// <param name="assemblyName">An assembly that is referenced within the web.config assemblies section</param>
		/// <param name="ticketStore">The type of ticket store to use. See the enum TicketStoreType for more information</param>
		/// <param name="ticketStoreTicketName">Name of the item within the ticketstore that should be used for the ticket itself</param>
		/// <param name="ticketStoreTicketExpiryName">Name of the item within the ticketstore that should be used for the ticket expiry date</param>
		public ExternalAuthenticationConfiguration(string typeName, string assemblyName, TicketStoreType ticketStore, string ticketStoreTicketName, string ticketStoreTicketExpiryName)
		{
			colPluginConfig = new NameValueCollection();
			strTypeName = typeName;
			strAssemblyName = assemblyName;
			this.ticketStoreType = ticketStore;
			this.strTicketStoreTicketName = ticketStoreTicketName;
			if(ticketStoreTicketExpiryName != null)
				this.strTicketStoreTicketExpiryName = ticketStoreTicketExpiryName;
		}

		/// <summary>
		/// Adds a parameter to the collection to be passed through to tbe plugin
		/// </summary>
		/// <param name="name">Name of the parameter</param>
		/// <param name="itemValue">Value of the parameter</param>
		public void AddPlugInConfigItem(string name, string itemValue)
		{
			colPluginConfig.Add(name, itemValue);
		}

		/// <summary>
		/// Property for accessing the type of the IExternalAuthenticationPlugin
		/// </summary>
		public string TypeName
		{
			get { return strTypeName;}
		}

		/// <summary>
		/// Property for accessing the assembly where the IExternalAuthenticationPlugin resides
		/// </summary>
		public string AssemblyName
		{
			get {return strAssemblyName;}
		}

		/// <summary>
		/// Property for accessing the parameter collection for the IExternalAuthenticationPlugin. This will be passed to the plugin when it is initialised using the Init method on the interface
		/// </summary>
		public NameValueCollection PlugInConfiguration
		{
			get {return colPluginConfig;}
		}

		/// <summary>
		/// The type of storage to use for tickets and ticket expiry dates
		/// </summary>
		public TicketStoreType TicketStore
		{
			get { return ticketStoreType;}
		}

		/// <summary>
		/// Property for accessing the property name within the ticket store where tickets will be stored. Will only return a value if the TicketStore has been specified.
		/// </summary>
		public string TicketStoreTicketName
		{
			get 
			{
				if(!ticketStoreType.Equals(TicketStoreType.None)) 
					return strTicketStoreTicketName;
				else
					return null;
			}
		}

		/// <summary>
		/// Property for accessing the property name within the ticket store where the expiry date of tickets will be stored. Will only return a value if the TicketStore has been specified.
		/// </summary>
		public string TicketStoreExpiryName
		{
			get 
			{
				if(!ticketStoreType.Equals(TicketStoreType.None))
					return strTicketStoreTicketExpiryName;
				else
					return null;
			}
		}

		/// <summary>
		/// Override of the ToString method - shows the current Type and Assembly for the configuration
		/// </summary>
		/// <returns>A string for debug purposes</returns>
		public override string ToString()
		{
			return string.Format("External Authentication Configuration: Type ({0}) Assembly ({1})", TypeName, AssemblyName);
		}
	}
}
