using System;
using System.Web;
using MSIBPlusPack.CommerceServer.Runtime.Configuration;

namespace MSIBPlusPack.CommerceServer.Runtime
{
	/// <summary>
	/// Summary description for ExternalAuthentication.
	/// </summary>
	public class ExternalAuthentication
	{
		IExternalAuthenticationPlugIn iExternalAuthenticationPlugin;
		
		TicketStoreType ticketStore;
		string strTicketStoreTicketName;
		string strTicketStoreTicketExpiryName;


		#region Internal Constructors and Properties
		/// <summary>
		/// Internal Constructor. This is only ever called by the external authentication module when it is starting up.
		/// </summary>
		/// <param name="plugin">The plugin to use for external authentication</param>
		/// <param name="eac">The configuration to pass through for ticketing information</param>
		internal ExternalAuthentication(IExternalAuthenticationPlugIn plugin, ExternalAuthenticationConfiguration eac)
		{
			iExternalAuthenticationPlugin = plugin;
			ticketStore = eac.TicketStore;
			strTicketStoreTicketExpiryName = eac.TicketStoreExpiryName;
			strTicketStoreTicketName = eac.TicketStoreTicketName;
		}

		/// <summary>
		/// Internal property for the external authentication plugin. This is used by the external authentication module when it needs to get a handle on the plugin after initialization.
		/// </summary>
		internal IExternalAuthenticationPlugIn Plugin
		{
			get 
			{
				return iExternalAuthenticationPlugin;
			}
		}

		#endregion Internal Constructors and Properties

		#region Public Methods
		/// <summary>
		/// Authenticates a user against the currently loaded plugin and if successful retrieves the users profile. 
		/// It will attempt to create a profile for the user if one does not already exist. However, this will be a shell of a profile, with just the username set - the site developer should determine whether the profile has enough information to satify their site requirements and add additional details if required. 
		/// </summary>
		/// <param name="username">Username of the user to authenticate</param>
		/// <param name="password">Password of the user to authenticate</param>
		/// <param name="ticketExpiry">If the external authentication provider uses tickets and provides timestamps, they may set this to the expiration date for the ticket. </param>
		/// <param name="ticket">If the external authentication provider uses tickets, they will set a ticket for the user at this point. The site developer can store this ticket locally for reference purposes. </param>
		/// <returns>A Commerce Server UserObject Profile for the user if one could be found or was created, otherwise null</returns>
		public Profile AuthenticateUser (string username, string password, string ticket, DateTime ticketExpiry)
		{
			return AuthenticateUser(username, password, ticket, ticketExpiry, true);
		}

		/// <summary>
		/// Authenticates a user against the currently loaded plugin and if successful retrieves the users profile. 
		/// If the CreateIfNotExisting parameter is true, it will attempt to create a profile for the user if one does not already exist. However, this will be a shell of a profile, with just the username set - the site developer should determine whether the profile has enough information to satify their site requirements and add additional details if required. 
		/// </summary>
		/// <param name="username">Username of the user to authenticate</param>
		/// <param name="password">Password of the user to authenticate</param>
		/// <param name="createProfileIfNotExisting">If true, a profile will be created for the user if one does not already exist. </param>
		/// <returns>A Commerce Server UserObject Profile for the user if one could be found or was created, otherwise null</returns>
		public Profile AuthenticateUser(string username, string password, bool createProfileIfNotExisting)
		{
			return AuthenticateUser(username, password, "", DateTime.Now, createProfileIfNotExisting);
		}

		/// <summary>
		/// Authenticates a user against the currently loaded plugin and if successful retrieves the users profile. 
		/// It will attempt to create a profile for the user if one does not already exist. However, this will be a shell of a profile, with just the username set - the site developer should determine whether the profile has enough information to satify their site requirements and add additional details if required. 
		/// </summary>
		/// <param name="username">Username of the user to authenticate</param>
		/// <param name="password">Password of the user to authenticate</param>		
		/// <returns>A Commerce Server UserObject Profile for the user if one could be found or was created, otherwise null</returns>
		public Profile AuthenticateUser(string username, string password)
		{
			return AuthenticateUser(username, password, true);
		}
		

		/// <summary>
		/// Authenticates a user against the currently loaded plugin and if successful retrieves the users profile. 
		/// If the CreateIfNotExisting parameter is true, it will attempt to create a profile for the user if one does not already exist. However, this will be a shell of a profile, with just the username set - the site developer should determine whether the profile has enough information to satify their site requirements and add additional details if required. 
		/// </summary>
		/// <param name="username">Username of the user to authenticate</param>
		/// <param name="password">Password of the user to authenticate</param>
		/// <param name="ticket">If the external authentication provider uses tickets, they will set a ticket for the user at this point. The site developer can store this ticket locally for reference purposes. </param>
		/// <param name="ticketExpiry">If the external authentication provider uses tickets and provides timestamps, they may set this to the expiration date for the ticket. </param>
		/// <param name="createProfileIfNotExisting">If true, a profile will be created for the user if one does not already exist. </param>
		/// <returns>A Commerce Server UserObject Profile for the user if one could be found or was created, otherwise null</returns>
		public Profile AuthenticateUser (string username, string password, string ticket, DateTime ticketExpiry, bool createProfileIfNotExisting)
		{
			CommerceContext cc = CommerceContext.Current;
			ProfileContext pc = cc.ProfileSystem;
			AuthenticationInfo ai = cc.AuthenticationInfo;
			bool bAuthenticated = iExternalAuthenticationPlugin.AuthenticateUser(username, password, ref ticket, ref ticketExpiry);
			if(!bAuthenticated)
				return null;

			Profile p;
			if(createProfileIfNotExisting)
			{
				try 
				{
					p = pc.CreateProfile(username, "UserObject");
					p.Update();				
				} 
				catch(CommerceProfileAlreadyExistsException cpaee)
				{
					CommerceProfileAlreadyExistsException newE = cpaee;
					p = pc.GetProfile("logon_name", username, "UserObject");
				}

				
			} else
				p = pc.GetProfile("logon_name", username, "UserObject");
			ai.SetAuthTicket(username, ai.SupportsSessionCookies);
			if(!ticketStore.Equals(TicketStoreType.None))
				StoreTicket(ticket, ticketExpiry, p);
			return p;
		}


		/// <summary>
		/// Retrieve the current ticket for the user. 
		/// </summary>
		/// <returns>The authentication ticket </returns>
		public string GetTicket()
		{
			string strTicket = null;
			
			if(!ticketStore.Equals(TicketStoreType.None))
			{
				// check that the ticket is valid
				if(ValidateTicket())
				{
					switch(ticketStore)
					{
						case TicketStoreType.CommerceProfile:
							CommerceContext cc = CommerceContext.Current;
							ProfileContext pc = cc.ProfileSystem;
							Profile profile = pc.GetProfile("logon_name", cc.AuthenticationInfo.AuthTicket.UserID, "UserObject");

							strTicket = profile.Properties[strTicketStoreTicketName].Value.ToString();
							break;
						case TicketStoreType.Session:
							strTicket = HttpContext.Current.Session[strTicketStoreTicketName].ToString();

							break;
						default:
							break;
					}
				}

			} 

			return strTicket;
			
		}
		/// <summary>
		/// Determines whether the current ticket a user has is valid
		/// </summary>
		/// <returns>true if the ticket is valid, false otherwise</returns>
		public bool ValidateTicket()
		{
			if(!ticketStore.Equals(TicketStoreType.None))
			{
				switch(ticketStore)
				{
					case TicketStoreType.CommerceProfile:
						CommerceContext cc = CommerceContext.Current;
						ProfileContext pc = cc.ProfileSystem;
						if(!cc.AuthenticationInfo.IsAuthenticated())
							return false;
						Profile profile = pc.GetProfile("logon_name", cc.AuthenticationInfo.AuthTicket.UserID, "UserObject");

						DateTime dtProfileExpire = (DateTime)profile.Properties[strTicketStoreTicketExpiryName].Value;
						if(dtProfileExpire > DateTime.Now)
							return true;						
						break;

					case TicketStoreType.Session:
						if(DateTime.Parse(HttpContext.Current.Session[strTicketStoreTicketExpiryName].ToString())> DateTime.Now)
							return true;						
						break;
					default:
						break;
				}

			} 
			return false;
		}


		/// <summary>
		/// Allows an existing ticket to be renewed.
		/// </summary>
		/// <param name="ticketExpiry">Will be set to the renewed expiry date if renewal was successful</param>
		/// <returns>true if the renewal was successful, false otherwise. The method will also return false if the ticket store was not specified</returns>
		public bool RenewTicket(DateTime ticketExpiry)
		{
			if(!ticketStore.Equals(TicketStoreType.None))
			{
				switch(ticketStore)
				{
					case TicketStoreType.CommerceProfile:
						CommerceContext cc = CommerceContext.Current;
						ProfileContext pc = cc.ProfileSystem;
						Profile profile = pc.GetProfile("logon_name", cc.AuthenticationInfo.AuthTicket.UserID, "UserObject");

						DateTime dtProfileExpire = (DateTime)profile.Properties[strTicketStoreTicketExpiryName].Value;
						string strProfileName = profile.Properties[strTicketStoreTicketName].Value.ToString();

						if(iExternalAuthenticationPlugin.RenewTicket(ref strProfileName, ref dtProfileExpire))
						{
							profile.Properties[strTicketStoreTicketExpiryName].Value = dtProfileExpire;
							profile.Update();
							return true;
						}
													
						break;
					case TicketStoreType.Session:
						string strSessionName= HttpContext.Current.Session[strTicketStoreTicketName].ToString();
						DateTime dtSessionExpire  = DateTime.Parse(HttpContext.Current.Session[strTicketStoreTicketExpiryName].ToString());
						if(iExternalAuthenticationPlugin.RenewTicket(ref strSessionName, ref dtSessionExpire))
						{
							HttpContext.Current.Session[strTicketStoreTicketExpiryName] = dtSessionExpire;
							return true;
						}
						
						break;
					default:
						break;
				}
			}
			return false;
		}

		#endregion Public Methods

		#region Private Methods

		private void StoreTicket(string ticket, DateTime ticketExpiry, Profile profile)
		{
			if(!ticketStore.Equals(TicketStoreType.None))
			{
				switch(ticketStore)
				{
					case TicketStoreType.CommerceProfile:
						StoreTicketCommerceProfile(ticket, ticketExpiry, profile);
						break;

					case TicketStoreType.Session:
						StoreTicketSession(ticket, ticketExpiry);
						break;
					default:
						break;
				}
			}
		}

		private void StoreTicketSession(string ticket, DateTime ticketExpiry)
		{
			HttpContext.Current.Session[strTicketStoreTicketName] = ticket;
			if(strTicketStoreTicketExpiryName != null)
				HttpContext.Current.Session[strTicketStoreTicketExpiryName] = ticketExpiry;
		}

		private void StoreTicketCommerceProfile(string ticket, DateTime ticketExpiry, Profile profile)
		{
			profile.Properties[strTicketStoreTicketName].Value = ticket;
			if(strTicketStoreTicketExpiryName != null)
				profile.Properties[strTicketStoreTicketExpiryName].Value = ticketExpiry;
			profile.Update();

		}
		#endregion Private Methods
	}
}
