using System;
using System.ComponentModel;
using System.Configuration;
using System.Web;
using ioko.ComponentModel.LicenseProvider;
using MSIBPlusPack.CommerceServer.Runtime.Configuration;

namespace MSIBPlusPack.CommerceServer.Runtime
{
	/// <summary>
	/// This module is responsible for loading in the configuration and initializing the Plugin that will perform the actual authentication
	/// </summary>
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public class ExternalAuthenticationModule : CommerceModule
	{		
		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(ExternalAuthenticationModule),this);
					lastDate = DateTime.Today;
				}

				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;

					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}
						return;

					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;

					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}		
			} 
			catch {}

			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion


		/// <summary>
		/// Private helper method to create an instance of the plugin as specified in the web.config. 
		/// </summary>
		/// <param name="eac">The plugin configuration set</param>
		/// <returns>An initialized plugin.</returns>
		private IExternalAuthenticationPlugIn InitializePlugin(ExternalAuthenticationConfiguration eac)
		{
			IExternalAuthenticationPlugIn iExternalAuthenticationPlugin = Activator.CreateInstance(eac.AssemblyName, eac.TypeName).Unwrap() as IExternalAuthenticationPlugIn;
			iExternalAuthenticationPlugin.Init(eac.PlugInConfiguration);
			return iExternalAuthenticationPlugin;
		}

		/// <summary>
		/// Initializes the ExternalAuthenticationModule. This performs the following tasks:
		/// - declares this modules dependency on the Commerce Server Profile and Authentication Modules - if these modules cannot be loaded then this module will fail to load
		/// - reads in the configuration for the module from web.config
		/// - initializes a global copy of the plugin and stores it in application space
		/// </summary>
		/// <param name="appInstance">An instance of the currently booting HttpApplication</param>
		public override void Init(HttpApplication appInstance)
		{
			// Declare dependency on both the authentication and profile modules. 
			// This will by their dependencies make us dependent the CommerceApplicationModule
			DeclareModuleDependency(typeof(Microsoft.CommerceServer.Runtime.CommerceAuthenticationModule), this.GetType());
			DeclareModuleDependency(typeof(Microsoft.CommerceServer.Runtime.Profiles.CommerceProfileModule), this.GetType());
								
			try 
			{
				ExternalAuthenticationConfiguration eac = ConfigurationSettings.GetConfig("ExternalAuthentication") as ExternalAuthenticationConfiguration;
				appInstance.Application[typeof(ExternalAuthentication).FullName] = new ExternalAuthentication(InitializePlugin(eac), eac);
			} 
			catch (InvalidCastException icex)
			{
				appInstance.Context.Trace.Write("ExternalAuthenticationCommerceModule:Constructor", "Failed to read the configuration settings: " + icex.Message, icex);
				throw icex;
			} 
			catch (Exception ex)
			{
				appInstance.Context.Trace.Write("ExternalAuthenticationCommerceModule:Constructor", "General initialization error: " + ex.Message, ex);
				throw ex;
			}

			ValidateLicense();
		}

		/// <summary>
		/// This passes the disposal responsibility onto the plugin
		/// </summary>
		/// <param name="disposing">True if this was called from the Dispose method, False if called from the finalizer. </param>
		protected override void Dispose(bool disposing)
		{
			((ExternalAuthentication) HttpContext.Current.Application[typeof(ExternalAuthentication).FullName]).Plugin.Dispose(disposing);
			base.Dispose (disposing);
		}

		/// <summary>
		/// Gets a copy of the current ExternalAuthentication, so a site developer can simply call the AuthenticateAsUser method
		/// </summary>
		public static ExternalAuthentication Current
		{
			get 
			{
				return HttpContext.Current.Application[typeof(ExternalAuthentication).FullName] as ExternalAuthentication;
			}
		}

	}
}
