using System;
using System.Collections.Specialized;

namespace MSIBPlusPack.CommerceServer.Runtime
{
	/// <summary>
	/// The interface that all plugin that provide external authentication to Commerce Server should meet. 
	/// Plugin implementors should not maintain any state within the plugin as a single instance of the plugin is created for the entire HttpApplication.
	/// 
	/// </summary>
	public interface IExternalAuthenticationPlugIn
	{
		/// <summary>
		/// Authenticate a user and optionally set a ticket
		/// </summary>
		/// <param name="username">Username of the user to authenticate. This can be null, but it is the responsibility of the plugin implementor to provide another mechanism by which the username can be communicated to the plugin.</param>
		/// <param name="password">Password of the user to authenticate. This can be null, but it is the responsibility of the plugin implementor to provide another mechanism by which the password can be communicated to the plugin.</param>
		/// <param name="ticket">A ticket that can be set by the plugin. This will be communicated back to the site developer</param>
		/// <param name="ticketExpirationDate">Will be set to the date/time the ticket expires on if the authentication was successful.</param>
		/// <returns>true if the user was successfully authenticated, false otherwise.</returns>
		bool AuthenticateUser (string username, string password, ref string ticket, ref DateTime ticketExpirationDate);

		/// <summary>
		/// Initializes the plugin. This method is only called at startup. The plugin implementor should initialize any resources that can be shared between multiple authentication requests. A set of configuration parameters defined in web.config are passed to the plugin.
		/// </summary>
		/// <param name="plugInConfig">A collection of name/value configuration parameters for the plugin.</param>
		void Init(NameValueCollection plugInConfig);

		/// <summary>
		/// Finalizes the plugin. The plugin implementor should close any resources opened in the Init method. 
		/// </summary>
		/// <param name="disposing">True if this was called from the Dispose method, False if called from the finalizer. </param>
		void Dispose(bool disposing);

		/// <summary>
		/// Ensures that a ticket is still valid.
		/// </summary>
		/// <param name="ticket">A ticket issued by the external authentication system</param>
		/// <returns>true if the ticket is still valid, false otherwise</returns>
		bool ValidateTicket(ref string ticket);

		/// <summary>
		/// Renews a ticket for external authentication systems that support ticket renewal
		/// </summary>
		/// <param name="ticket">A ticket issued by the external authentication system</param>
		/// <param name="ticketExpirationDate">Will be set to the date/time the ticket expires until is renewal was successful</param>
		/// <returns>true if the ticket was successfully renew, false otherwise</returns>
		bool RenewTicket(ref string ticket, ref DateTime ticketExpirationDate);
 
	}
}
