using System;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.ComponentModel.Design;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.WebControls.Design;
using Microsoft.ContentManagement.WebControls;
using System.Xml;
using System.Xml.Xsl;
using System.Text;
using System.Drawing;
using MSIBPlusPack.Utilities;
using ioko.ComponentModel.LicenseProvider;


namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/*
	 * *******************************************
	 *				FAQ XmlPlaceHolder  		 *
	 *				  Created By				 *
	 *				  cScape Ltd.				 *
	 * *******************************************
	*/

	#region Enums

	/// <summary>
	/// Enumerates modes of the control.
	/// </summary>
	public enum EnmMode :int
	{
		/// <summary>
		/// Production mode. Standard mode of operation - the output of the control is HTML 
		/// transformed by the XSLT stylesheet specified in XsltStylesheet.
		/// </summary>
		Production = 1,
		/// <summary>
		/// XML Debug mode. Used for debugging - output of the control is XML displayed on 
		/// the web page. This enables the developer to examine the XML which is generated by the 
		/// control.
		/// </summary>
		DebugXML = 2
	}
	;	
	
	#endregion
	
	/// <summary>
	/// The FAQ control uses information pulled from postings in the website to provide a list
	/// of FAQs. This is done, for example, by having FAQ postings with a question placeholder and a detail
	/// placeholder. The control can be set to retrieve the information in the question placeholder for each 
	/// posting to produce an FAQ list.
	/// These FAQs can be retrieved from anywhere in the site or from a specific FAQ channel. 
	/// 
	/// </summary>
	/// <remarks>
	/// The FAQ control will search channels specified in the <see cref="ChannelFilter"/> property and
	///  bring back information about the postings in these channel(s), including 
	/// the content of any placeholders specified in the <see cref="PlaceholderFilter"/> property. 
	/// The search can be filtered by template type using the <see cref="TemplateFilter"/> property.
	/// An XML document is generated from this search, which can be transformed into HTML output, 
	/// in the form of an FAQ list.
	/// Properties of the control can be set at design time or in authoring mode, and there are design time
	///  properties to restrict the amount of control authors have. 
	/// </remarks>
	[Designer(typeof(MSIBPlusPack.ContentManagement.Publishing.Placeholders.FaqDesigner)),
	    DefaultProperty("Text"),
		ToolboxData("<{0}:FaqControl runat=server></{0}:FaqControl>"),
	    LicenseProviderAttribute(typeof(PlusPackLicenseProvider))]
	public class FaqControl : BasePlaceholderControl
	{
		#region Event ReprocessReady
		/// <summary>
		/// This event is raised to indicate that the XSLT transformation has occured 
		/// and the control is ready to transform again 
		/// </summary>
		/// <remarks>Used in a situation where we want to transform the output twice. For example, 
		/// it may be a requirement to have a listing page with a set of # links at the top of the page that link to 
		/// a more detailed list below. This can be achieved by transforming the XML twice using 
		/// different stylesheets. </remarks>
		internal event EventHandler ReprocessReady;
		
		/// <summary>
		/// Raises the ReprocessReady event.
		/// </summary>
		/// <param name="sender">Source object.</param>
		/// <param name="e">EventArgs class.</param>
		internal void OnReprocessReady(Object sender, EventArgs e)
		{
			if(ReprocessReady!=null)
			{
				ReprocessReady(this, new EventArgs());
			}
		}
		#endregion

		#region Event PreTransform
		/// <summary>
		/// This event is raised before the XML generated by the control is transformed by 
		/// the specified XSL stylesheet.
		/// </summary>
		/// <remarks>The OutputXml property can be used to retrieve the XML at this stage.</remarks>
		public event EventHandler PreTransform;

		/// <summary>
		/// Raises the PreTransform event.
		/// </summary>
		/// <param name="sender">Source object.</param>
		/// <param name="e">EventArgs class.</param>
		protected void preTransform(Object sender, EventArgs e)
		{
			if(PreTransform!=null)
			{
				// also need to populate our _outputXml variable here so that any 
				// calls to OutputXml property retrieve XML correctly..
				if ((_xmlOutput == null) || (_xmlOutput == ""))
				{
					XmlDocument postListerOutput = phLister.getXML();
					_xmlOutput = postListerOutput.InnerXml;
				}
				PreTransform(this, new EventArgs());
			}
		}
		#endregion

		#region Controls



		#endregion Controls

		#region Private Constants

		//Root Node Name
		private const string m_cstrROOT_NODE = "root";
		private const string mcstrXML_ROOT = "settings";
		private const string mcstrXML_TAG = "setting";
		private const string mcstrXML_ATTRIBUTE_NAME = "name";
		private const string mcstrXML_XPATH = mcstrXML_ROOT + "/" + mcstrXML_TAG + "[@" + mcstrXML_ATTRIBUTE_NAME + "=\"*N*\"]";

		private const string mcstrXML_NODE_SEARCH_CHANNELS = "search_channels";
		private const string mcstrXML_NODE_SEARCH_PLACEHOLDERS = "search_placeholders";
		private const string mcstrXML_NODE_XSLT_STYLESHEET = "xslt_stylesheet";
		private const string mcstrXML_NODE_TEMPLATENAME = "templatename";
		private const string mcstrXML_NODE_ROOTNODENAME = "rootnodename";
		private const string mcstrXML_NODE_MAXRETURN = "maxreturn";
		private const string mcstrXML_NODE_MAXPERCHANNEL = "maxperchannel";
		private const string mcstrXML_NODE_ENABLELISTER = "active";

		private const string mcstrID_SEARCH_CHANNELS = "txtsearch_channels";
		private const string mcstrID_SEARCH_PLACEHOLDERS = "txtsearch_placeholders";
		private const string mcstrID_XSLT_STYLESHEET = "txtxslt_stylesheet";
		private const string mcstrID_TEMPLATENAME = "txttemplatename";
		private const string mcstrID_ROOTNODENAME = "txtrootnodename";
		private const string mcstrID_MAXRETURN = "txtmaxreturn";
		private const string mcstrID_MAXPERCHANNEL = "chkmaxperchannel";
		private const string mcstrID_ENABLELISTER = "chkEnable";

		private const string mcstrLABEL_SEARCH_CHANNELS = "Use Channels";
		private const string mcstrLABEL_SEARCH_PLACEHOLDERS = "Retrieve Placeholders";
		private const string mcstrLABEL_XSLT_STYLESHEET = "XSL Stylesheet path";
		private const string mcstrLABEL_TEMPLATENAME = "Use templates";
		private const string mcstrLABEL_ROOTNODENAME = "Root node";
		private const string mcstrLABEL_MAXRETURN = "Max return";
		private const string mcstrLABEL_MAXPERCHANNEL = "Per channel";
		private const string mcstrLABEL_ENABLELISTER = "Visible";

		private const string mcstrCONTROL_DISPLAY_NAME = "FAQ Control";
		private const string mcstrHELP_TEXT = "Select options to filter and change presentation of returned FAQ links.";
		private const bool mblnDISPLAY_HELP = true;
		private const EnmMode mDEFAULT_MODE = EnmMode.Production;

		#endregion Private Constants

		#region Private Variables

		// UI Settings
		private  string _Control_Display_Name = mcstrCONTROL_DISPLAY_NAME;
		private  string _Control_CSS_Panel = "";
		private  string _Control_CSS_title = "";
		private  string _Control_CSS_Help = "";
		private  string _Control_Help_Text = mcstrHELP_TEXT;
		private  bool	_Display_Help = mblnDISPLAY_HELP;
		private EnmMode _Mode = mDEFAULT_MODE;

		// Is the control Active
		private bool _active = true;
		private bool _enable_active = true;

		//This is used to store the Channels the control will search (Comma Delimited)
		private string _search_Channels = "";
		private bool _enable_search_Channels = true;

		/// <summary>
		/// This contains a comma delimited list of placeholders to retrieve
		/// </summary>
		private string _search_Placeholders = "";
		private bool _enable_search_Placeholders = true;
		/// <summary>
		/// The Path to the XSLT Style sheet to use for the transform
		/// </summary>
		private string _XSLT_Stylesheet = "";
		private bool _enable_XSLT_Stylesheet = true;

		private string _TemplateName = "";
		private bool _enable_TemplateName = true;

		private string _RootNodeName = m_cstrROOT_NODE;
		private bool _enable_RootNodeName = true;


		//If this is left at Zero, all items found are returned
		private int _MaxReturn = 0;
		private bool _enable_MaxReturn = true;

		//Should the Max Return be on a per channel basis
		private bool _MaxPerChannel = false;
		private bool _enable_MaxPerChannel = true;

		// The Placeholder XML Doc
		private XmlDocument _xmlSettings = new XmlDocument();

		//XML output as string
		private string _xmlOutput;

		#endregion Private Variables

		#region Interface Controls

		private TextBox txtsearch_channels;
		private TextBox txtsearch_placeholders;
		private TextBox txtxslt_stylesheet;
		private TextBox txttemplatename;
		private TextBox txtrootnodename;
		private TextBox txtmaxreturn;
		private CheckBox chkmaxperchannel;
		private CheckBox chkEnableLister;

		private postList phLister;

		#endregion

		#region Public Properties

		#region Mode

		/// <summary>
		/// Determines the behaviour of the control. 
		/// </summary>
		/// <value>
		/// Only values from the EnmMode enumerator are valid. These can easily
		/// be selected using the Properties dialog in Visual Studio .NET.
		/// </value>
		/// <remarks>
		/// Whilst in development, the 'debugXML' mode can be used to help 
		/// identify problems. The output of the control is XML rather than the transformed 
		/// HTML produced by the XSLT stylesheet.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(EnmMode.Production),
		Description("The presentation mode of the control.")]
		public EnmMode Mode
		{
			set
			{
				_Mode = value;
			}
			get
			{
				return _Mode;
			}
		}	
		
		#endregion Mode

		#region XmlOutput

		/// <summary>
		/// Enables the XML generated by the control to be retrieved programatically. The value 
		/// returned is the same as the control output when Mode = EnmMode.debugXML. 
		/// </summary>
		/// <value>
		/// String value representing the XML.
		/// </value>
		/// <remarks>
		/// The value returned is the output of the control before the XSL transformation is 
		/// applied.</remarks>
		[
		Browsable(false),
		Category("Custom"),
		DefaultValue(""),
		Description("Enables the XML generated by the control to be retrieved programatically")]
		public string XmlOuput
		{
			get
			{
				return _xmlOutput;
			}
		}	
		
		#endregion Mode

		#region UI Style/Label settings

		#region DisplayHelp
		/// <summary>
		/// If True, help text for the control will be displayed Text displayed in a &lt;div&gt; 
		/// tag adjacent to the control in presentation mode. The value of ControlHelpText is used 
		/// as the help text. If no value is specified, the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(mblnDISPLAY_HELP),
		Description("If True, the help text <div> for the control will be displayed")]
		public bool DisplayHelp
		{
			set
			{
				_Display_Help = value;
			}
			get
			{
				return _Display_Help;
			}
		}	
		#endregion DisplayHelp

		#region ControlHelpText
		/// <summary>
		/// Text displayed adjacent to the control in presentation mode when DisplayHelp is "True". 
		/// If no value is specified, the 
		/// default of "Select options to filter and change presentation of returned FAQ links" 
		/// is used. 
		/// </summary>
		/// <value>String value of the help text.</value>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(mcstrHELP_TEXT),
		Description("Text displayed in a <div> tag adjacent to the control in presentation mode")]
		public string ControlHelpText
		{
			set
			{
				_Control_Help_Text = value;
			}
			get
			{
				return _Control_Help_Text;
			}
		}	
		#endregion ControlHelpText

		#region ControlHelpCss
		/// <summary>
		/// The CSS class of the control help text when viewed in CMS edit mode. 
		/// </summary>
		/// <value>String value of the CSS class.</value>
		[
		Browsable(true),
		Category("CSS"),
		DefaultValue(""),
		Description("The CSS class of the control help text when viewed in CMS edit mode.")]
		public string ControlHelpCss
		{
			set
			{
				_Control_CSS_Help = value;
			}
			get
			{
				return _Control_CSS_Help;
			}
		}	
		#endregion ControlHelpCss

		#region ControlTitleCss
		/// <summary>
		/// The CSS class of the control title text when viewed in CMS edit mode. 
		/// </summary>
		/// <value>String value of the CSS class.</value>
		[
		Browsable(true),
		Category("CSS"),
		DefaultValue(""),
		Description("The CSS class of the control title text when viewed in CMS edit mode.")]
		public string ControlTitleCss
		{
			set
			{
				_Control_CSS_title = value;
			}
			get
			{
				return _Control_CSS_title;
			}
		}	
		#endregion ControlTitleCss

		#region ControlBackgroundCss
		/// <summary>
		/// The CSS class of the HTML table built by the control when viewed in CMS edit mode.
		/// </summary>
		/// <value>String value of the CSS class.</value>
		[
		Browsable(true),
		Category("CSS"),
		DefaultValue(""),
		Description("The CSS class of the HTML table built by the control when viewed in CMS edit mode.")]
		public string ControlBackgroundCss
		{
			set
			{
				_Control_CSS_Panel = value;
			}
			get
			{
				return _Control_CSS_Panel;
			}
		}	
		#endregion ControlBackgroundCss

		#region ControlTitle
		/// <summary>
		/// The title of the control when viewed in CMS edit mode. If no value is specified, the 
		/// default of "FAQ Control" is used.
		/// </summary>
		/// <remarks>
		/// This text appears as the heading of the FAQ control in authoring mode.		
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(mcstrCONTROL_DISPLAY_NAME),
		Description("The title of the control when viewed in CMS edit mode")]
		public string ControlTitle
		{
			set
			{
				_Control_Display_Name = value;
			}
			get
			{
				return _Control_Display_Name;
			}
		}	
		#endregion ControlTitle

		#endregion UI Style/Label settings

		#region Author Enablers

		#region ChannelFilterInWBC
		/// <summary>
		/// If True, the ChannelFilter property can be set by the page author in the edit mode. 
		/// The author can enter the channel filter list via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the ChannelFilter property can be set by the page author in the edit mode")]
		public bool ChannelFilterInWBC
		{
			set
			{
				_enable_search_Channels = value;
			}
			get
			{
				return _enable_search_Channels;
			}
		}
		#endregion ChannelFilterInWBC

		#region PlaceholderFilterInWBC
		/// <summary>
		/// If True, the PlaceholderFilter property can be set by the page author in the edit mode. 
		/// The author can enter the placeholder filter list via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the PlaceholderFilter property can be set by the page author in the edit mode")]
		public bool PlaceholderFilterInWBC
		{
			set
			{
				_enable_search_Placeholders = value;
			}
			get
			{
				return _enable_search_Placeholders;
			}
		}
		#endregion PlaceholderFilterInWBC
		
		#region XsltStylesheetPathInWBC
		/// <summary>
		/// If True, the XsltStylesheetPath property can be set by the page author in the edit mode. 
		/// The author can enter the path via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the XsltStylesheetPath property can be set by the page author in the edit mode")]
		public bool XsltStylesheetPathInWBC
		{
			set
			{
				_enable_XSLT_Stylesheet = value;
			}
			get
			{
				return _enable_XSLT_Stylesheet;
			}
		}
		#endregion XsltStylesheetPathInWBC

		#region TemplateFilterInWBC
		/// <summary>
		/// If True, the TemplateFilter property can be set by the page author in the edit mode. 
		/// The author can enter the template filter list via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the TemplateFilter property can be set by the page author in the edit mode")]
		public bool TemplateFilterInWBC
		{
			set
			{
				_enable_TemplateName = value;
			}
			get
			{
				return _enable_TemplateName;
			}
		}
		#endregion TemplateFilterInWBC

		#region XmlRootNodeNameInWBC
		/// <summary>
		/// If True, the XmlRootNodeName property can be set by the page author in the edit mode. 
		/// The author can enter the name via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the XmlRootNodeName property can be set by the page author in the edit mode")]
		public bool XmlRootNodeNameInWBC
		{
			set
			{
				_enable_RootNodeName = value;
			}
			get
			{
				return _enable_RootNodeName;
			}
		}
		#endregion XmlRootNodeNameInWBC

		#region MaxReturnCountInWBC
		/// <summary>
		/// If True, the MaxReturnCount property can be set by the page author in the edit mode. 
		/// The author can enter the value via a TextBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the MaxReturnCount property can be set by the page author in the edit mode")]
		public bool MaxReturnCountInWBC
		{
			set
			{
				_enable_MaxReturn = value;
			}
			get
			{
				return _enable_MaxReturn;
			}
		}
		#endregion MaxReturnCountInWBC

		#region MaxReturnPerChannelInWBC
		/// <summary>
		/// If True, the MaxReturnPerChannel property can be set by the page author in the edit mode. 
		/// The author can enter the value via a CheckBox. If no value is specified, 
		/// the default of "False" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(false),
		Description("If True, the MaxReturnPerChannel property can be set by the page author in the edit mode")]
		public bool MaxReturnPerChannelInWBC
		{
			get
			{
				return _enable_MaxPerChannel;
			}
			set
			{
				_enable_MaxPerChannel = value;
			}
		}
		#endregion MaxReturnPerChannelInWBC

		#region VisibilitySetInWBC
		/// <summary>
		/// If True, the Visible property can be set by the page author in the edit mode. 
		/// The author can enter the value via a CheckBox. If no value is specified, 
		/// the default of "True" is used. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Values set by the author will override control property values 
		/// set in control tag in the .aspx page.
		/// </remarks>
		[
		Browsable(true),
		Category("Enable_in_WBC"),
		DefaultValue(true),
		Description("If True, the Visible property can be set by the page author in the edit mode")]
		public bool VisibilitySetInWBC
		{
			get
			{
				return _enable_active;
			}
			set
			{
				_enable_active = value;
			}
		}
		#endregion VisibilitySetInWBC

		#endregion

		#region Public Properties Pass Through

		#region ChannelFilter
		/// <summary>
		/// A comma-separated list of channel paths that are to be searched. If no value is 
		/// specified, the current channel will be used. This property is used in conjunction 
		/// with the other filter properties.
		/// </summary>
		/// <value>String value of channel paths.
		/// </value>
		/// <remarks>Each channel path should begin with a slash and be a full path to the 
		/// channel in the hierarchy, e.g. "/Channels/mySite/Sales/FAQs/". Each posting found 
		/// according to the filters is added to the XML structure as a &lt;post&gt; element.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma-separated list of channel paths that are to be searched. If no value is specified, the current channel will be used.")]
		public string ChannelFilter
		{
			set
			{
				_search_Channels = value;
			}
			get
			{
				return _search_Channels;
			}
		}
		#endregion ChannelFilter

		#region PlaceholderFilter
		/// <summary>
		/// A comma-separated list of placeholder names whose contents will be returned. If no value is 
		/// specified, no placeholder values will be returned. This property is used in conjunction 
		/// with the other filter properties.
		/// </summary>
		/// <value>String value of placeholder names.
		/// </value>
		/// <remarks>
		/// Typicallly, the heading/title placeholder of the FAQ template would be specified here as a 
		/// minimum. Each placeholder is added to the XML structure as a &lt;placeholder&gt; element.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma delimted list of placeholders to search, if left blank all placeholders will be left blank")]
		public string PlaceholderFilter
		{
			set
			{
				_search_Placeholders = value;
			}
			get
			{
				return _search_Placeholders;
			}
		}
		#endregion PlaceholderFilter
		
		#region XsltStylesheetPath
		/// <summary>
		/// The path to the XSLT file which converts the XML generated for 
		/// the FAQs to HTML to be displayed on the page. A path to a 
		/// valid XSLT file <b>must</b> be supplied in order for the control to output 
		/// HTML.
		/// </summary>
		/// <value>
		/// String value of the virtual path to the file (including filename).
		/// </value>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Custom"),
		DefaultValue(""),
		Description("Set this to change the presentation style of the control. Using XSLT")]
		public string XsltStylesheetPath
		{
			set
			{
				_XSLT_Stylesheet = value;
			}
			get
			{
				return _XSLT_Stylesheet;
			}
		}
		#endregion XsltStylesheetPath

		#region TemplateFilter
		/// <summary>
		/// A comma-separated list of template names whose contents will be returned. If no value is 
		/// specified, <b>all</b> templates will be considered to be valid. This property is used in conjunction 
		/// with the other filter properties.
		/// </summary>
		/// <value>String value of template names.
		/// </value>
		/// <remarks>
		/// Typicallly, the heading/title placeholder of the FAQ template would be specified here as a 
		/// minimum. Each placeholder is added to the XML structure as a &lt;placeholder&gt; element.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma delimted list of templates to search, if left blank all templates will be considered valid")]
		public string TemplateFilter
		{
			set
			{
				_TemplateName = value;
			}
			get
			{
				return _TemplateName;
			}
		}
		#endregion TemplateFilter

		#region XmlRootNodeName
		/// <summary>
		/// This property can be used to override the default root node name used in the XML 
		/// generated by the control. If no value is specified, the default of "root" is used.
		/// </summary>
		/// <value>String value of root node name to use.</value>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(m_cstrROOT_NODE),
		Description("This property can be used to override the default root node name used in the XML generated by the control.")]
		public string XmlRootNodeName
		{
			set
			{
				_RootNodeName = value;
			}
			get
			{
				return _RootNodeName;
			}
		}
		#endregion XmlRootNodeName

		#region MaxReturnCount
		/// <summary>
		/// This maximum number of FAQ items to be returned. If 0 is specified, all FAQ items found 
		/// in the search (according to filters set) will be returned.
		/// </summary>
		/// <value>Integer value representing maximum number of FAQs.</value>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(0),
		Description("Set this if you wish to limit the number of items returned")]
		public int MaxReturnCount
		{
			set
			{
				_MaxReturn = value;
			}
			get
			{
				return _MaxReturn;
			}
		}
		#endregion MaxReturnCount

		#region MaxReturnPerChannel
		/// <summary>
		/// If True, the value specified in MaxReturnCount is used for each channel where FAQ items 
		/// are found. For example, if MaxReturnCount = 3 and MaxReturnPerChannel = "True", a maximum 
		/// of 3 FAQs will be returned per channel. If no value is specified, the default of 
		/// "False" is used.
		/// </summary>
		/// <value>True or False.</value>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(false),
		Description("If True, the value specified in MaxReturnCount is used for each channel.")]
		public bool MaxReturnPerChannel
		{
			get
			{
				return _MaxPerChannel;
			}
			set
			{
				_MaxPerChannel = value;
			}
		}
		#endregion MaxReturnPerChannel

		#region Visible
		/// <summary>
		/// If true, the control output is displayed in presentation mode.
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>
		/// Equivalent WBC visibility property is VisibilitySetInWBC.
		/// </remarks>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(false),
		Description("If true, the control output is displayed in presentation mode.")]
		public new bool Visible
		{
			get
			{
				return _active;
			}
			set
			{
				_active = value;
			}
		}
		#endregion

		#region XslArguments
		/// <summary>
		/// This property can optionally be used to programatically supply an XsltArgumentList 
		/// object using the PreTransform event.
		/// </summary>
		/// <value>An XsltArgumentList object containing the arguments you wish to use 
		/// in the XSL transformation.</value>
		/// <remarks>To use, an event handler should be attached to the PreTransform 
		/// event which sets this property to a valid XsltArgumentList object.</remarks>
		public XsltArgumentList XslArguments
		{
			set
			{
				if(phLister != null)
				{
					phLister.XSL_Args = value;
				}
			}
		}
		#endregion XslArguments

		#endregion Public Properties Pass Through

		#endregion Public Properties

		#region Constructor
		/// <summary>
		/// Initialises a new instance of the FaqControl class.
		/// </summary>
		public FaqControl()
		{
			//
			// TODO: Add constructor logic here
			//
		}
		#endregion Constructor

		/// <summary>
		/// Called when the control is initialised.
		/// </summary>
		/// <param name="e">EventArgs class.</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense(); 
			base.OnInit (e);
		}

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing 
		private void ValidateLicense()
		{
			bool updateLicense = false;
			if (license == null)
				updateLicense = true;
			else if (lastDate != DateTime.Today)
				updateLicense = true;
			if (updateLicense)
			{
				license = (PlusPackLicense)LicenseManager.Validate(typeof(FaqControl),this);
				lastDate = DateTime.Today;
			}

			switch(license.Validity)
			{
				case MSIBLicenseValidator.LicenseState.Full:
					break;
				case MSIBLicenseValidator.LicenseState.Trial_Active:
					if (updateLicense)
					{
						TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
						int daysRemaining = span.Days + 1;
						if (daysRemaining <= 7)
							throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
								daysRemaining));
					}
					break;
				case MSIBLicenseValidator.LicenseState.Invalid:
				case MSIBLicenseValidator.LicenseState.None:
					throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
				case MSIBLicenseValidator.LicenseState.Trial_Expired:
					throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using the FAQ component please purchase the relevant license(s).");
			}           
		}

		#endregion

		#region Placeholder Overrides
		/// <summary>
		/// Creates child controls used in authoring mode. 
		/// </summary>
		/// <param name="authoringContainer">Container to which child controls are added.</param>
		/// <remarks>Adds a HTML table containing child controls.</remarks>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			authoringContainer.Controls.Add( buildInterface());
		}
		/// <summary>
		/// Creates child controls used in presentation mode. 
		/// </summary>
		/// <param name="presentationContainer">Container to which child controls are added.</param>
		protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{


			LoadFromCMS();
			if(this.Visible)
			{
				phLister = new postList();
			
				phLister.search_Channels = _search_Channels;
				phLister.search_Placeholders = _search_Placeholders;
				phLister.XSLT_Stylesheet = _XSLT_Stylesheet;
				phLister.TemplateName = _TemplateName;
				phLister.RootNodeName = _RootNodeName;
				phLister.MaxReturn = _MaxReturn;
				phLister.MaxPerChannel = _MaxPerChannel;
				// Reprocessing
				if(ReprocessReady!=null)
				{
					phLister.reprocessReady += new EventHandler(OnReprocessReady);
				}
				if(PreTransform!=null)
				{
					phLister.preTransform += new EventHandler(preTransform);
				}
				if (this.Mode != EnmMode.Production)
				{
					// populate our _outputXml variable if not done already..
					if ((_xmlOutput == null) || (_xmlOutput == ""))
					{
						XmlDocument postListerOutput = phLister.getXML();
						_xmlOutput = postListerOutput.InnerXml;
					}
					Literal litOutput = new Literal();
					litOutput.Text = displayAsHTML(_xmlOutput);
					presentationContainer.Controls.Add(litOutput);
				}
				else
				{
					presentationContainer.Controls.Add(phLister);
				}
			}
		}
		/// <summary>
		/// Loads content from the placeholder in authoring mode.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			// Load authoring content
			EnsureChildControls();
			LoadFromCMS();
			loadDataToInterface();
		}
		/// <summary>
		/// Loads content from the placeholder in presentation mode.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs class.</param>
		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			// LoadFromCMS();

		}
		/// <summary>
		/// Saves content to the underlying placeholder.
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs class.</param>
		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			// Save content back into placeholder
			EnsureChildControls();

			getFromInterface();

			string strXML = saveToXML().InnerXml;

			this.BoundXmlPlaceholder.XmlAsString = strXML;
			//
			// : TO DO SAVE DATA
			//
		}
		#endregion Placeholder Overrides

		#region LoadFromCMS()
		private void LoadFromCMS()
		{
			EnsureChildControls();
			WebAuthorContext webAuthorContext = WebAuthorContext.Current;
			try
			{
				XmlDocument xmlData = new XmlDocument();

				switch (webAuthorContext.Mode)
				{
					case WebAuthorContextMode.AuthoringNew:
						
						try
						{
							xmlData.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
						}
						catch
						{
							// Do nothing
						}
						break;
					case WebAuthorContextMode.AuthoringReedit:
						

						xmlData.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
						LoadFromXML(xmlData);

						break;

					default:

						xmlData.LoadXml(this.BoundXmlPlaceholder.XmlAsString);
						LoadFromXML(xmlData);

						break;
				}
			}
			catch (Exception exp)
			{
			
			}
		}

		#endregion LoadFromCMS()

		#region LoadFromXML(XmlDocument xmlData)
		/// <summary>
		/// Loads the Saved settings from the PH XML
		/// </summary>
		private void LoadFromXML(XmlDocument xmlData)
		{
			if(_enable_search_Channels)
			{
				_search_Channels = LoadNode(mcstrXML_NODE_SEARCH_CHANNELS, xmlData);
			}
			if(_enable_search_Placeholders)
			{
				_search_Placeholders = LoadNode(mcstrXML_NODE_SEARCH_PLACEHOLDERS, xmlData);
			}
			if(_enable_XSLT_Stylesheet)
			{
				_XSLT_Stylesheet = LoadNode(mcstrXML_NODE_XSLT_STYLESHEET, xmlData);
			}
			if(_enable_TemplateName)
			{
				_TemplateName = LoadNode(mcstrXML_NODE_TEMPLATENAME, xmlData);
			}
			if(_enable_RootNodeName)
			{
				_RootNodeName = LoadNode(mcstrXML_NODE_ROOTNODENAME, xmlData);
			}
			if(_enable_MaxReturn)
			{
				try
				{
					_MaxReturn = int.Parse(LoadNode(mcstrXML_NODE_MAXRETURN, xmlData, "0"));
				}
				catch
				{
				}
			}
			if(_enable_MaxPerChannel)
			{
				_MaxPerChannel = bool.Parse(LoadNode(mcstrXML_NODE_MAXPERCHANNEL, xmlData, "false"));
			}
			if(_enable_active)
			{
				_active = bool.Parse(LoadNode(mcstrXML_NODE_ENABLELISTER, xmlData, "false"));
			}
		}
		#endregion LoadFromXML()

		#region LoadNode(string strNode, XmlDocument xmlData, string strDefault)

		private string LoadNode(string strNode, XmlDocument xmlData)
		{
			return LoadNode(strNode,  xmlData,  null);
		}

		/// <summary>
		/// Load a TAG from the XML data set
		/// </summary>
		private string LoadNode(string strNode, XmlDocument xmlData, string strDefault)
		{
			string nodeSelect = mcstrXML_XPATH.Replace("*N*", strNode);
			string strReturn = (strDefault == null) ? "" : strDefault;

			XmlNode xmlNode = xmlData.SelectSingleNode(nodeSelect);


			if(xmlNode != null)
			{

				strReturn = xmlNode.InnerText;
			}

			return strReturn;
		}
		#endregion LoadNode


		#region saveToXML()
		/// <summary>
		/// Saves the user settings to XML
		/// </summary>
		/// <remarks>Saves all settings, though locked setting are not used, this caters for dev changes after posting</remarks>
		private XmlDocument saveToXML()
		{

			XmlDocument xmlReturn = new XmlDocument();

			XmlElement xmlRoot = xmlReturn.CreateElement("", mcstrXML_ROOT, "");

			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_SEARCH_CHANNELS, _search_Channels , xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_SEARCH_PLACEHOLDERS, _search_Placeholders, xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_XSLT_STYLESHEET, _XSLT_Stylesheet , xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_TEMPLATENAME, _TemplateName , xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_ROOTNODENAME, _RootNodeName , xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_MAXRETURN, _MaxReturn.ToString() , xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_MAXPERCHANNEL, ((_MaxPerChannel) ? "true" : "false"), xmlReturn));
			xmlRoot.AppendChild(saveNode(mcstrXML_NODE_ENABLELISTER, ((_active) ? "true" : "false") , xmlReturn));
			//xmlRoot.AppendChild(saveNode(mcstrXML_NODE_, _ , xmlReturn));
			//xmlRoot.AppendChild(saveNode(mcstrXML_NODE_, _ , xmlReturn));

			xmlReturn.AppendChild(xmlRoot);

			return xmlReturn;

		}

		private XmlElement saveNode(string strName, string strValue, XmlDocument xmlDoc)
		{
			// First of create the containing Node
			XmlElement xmlReturn = xmlDoc.CreateElement("", mcstrXML_TAG, "");

			#region Name Attribute

			XmlAttribute xmlName = xmlDoc.CreateAttribute("", mcstrXML_ATTRIBUTE_NAME, "");

			xmlName.Value = strName;

			xmlReturn.Attributes.Append(xmlName);

			#endregion Name Attribute

			xmlReturn.InnerText = strValue;

			return xmlReturn;

		}


		#endregion 


		#region buildInterface()
		/// <summary>
		/// Returns a table with controls initialised and added
		/// </summary>
		/// <returns>Table object</returns>
		private Table buildInterface()
		{
			Table tblReturn = new Table();

			tblReturn.CssClass = ControlBackgroundCss;
			tblReturn.CellPadding = 4;
			tblReturn.CellSpacing = 0;

			#region Title Row

			TableRow trTitle = new TableRow();
			TableCell tdTitle = new TableCell();

			tdTitle.CssClass = ControlTitleCss;
			tdTitle.ColumnSpan = 3;
			tdTitle.Text = ControlTitle;

			trTitle.Controls.Add(tdTitle);
			tblReturn.Controls.Add(trTitle);

			if(DisplayHelp)
			{
			
				TableRow trHelp = new TableRow();

				TableCell tdSpare = new TableCell();
				tdSpare.ColumnSpan = 2;
				trHelp.Controls.Add(tdSpare);

				TableCell tdHelp = new TableCell();
				tdHelp.CssClass = ControlHelpCss;
				tdHelp.Text = ControlHelpText;
				tdHelp.VerticalAlign = VerticalAlign.Top;
				tdHelp.RowSpan = 15;

				trHelp.Controls.Add(tdHelp);
				tblReturn.Controls.Add(trHelp);

			}

			#endregion Title Row

			if(_enable_search_Channels)
			{
				txtsearch_channels = new TextBox();
				txtsearch_channels.ID = mcstrID_SEARCH_CHANNELS;
				tblReturn.Controls.Add(addRow(mcstrLABEL_SEARCH_CHANNELS,txtsearch_channels));
			}
			if(_enable_search_Placeholders)
			{
				txtsearch_placeholders = new TextBox();
				txtsearch_placeholders.ID = mcstrID_SEARCH_PLACEHOLDERS;
				tblReturn.Controls.Add(addRow(mcstrLABEL_SEARCH_PLACEHOLDERS,txtsearch_placeholders));
			}
			if(_enable_XSLT_Stylesheet)
			{
				txtxslt_stylesheet = new TextBox();
				txtxslt_stylesheet.ID = mcstrID_XSLT_STYLESHEET;
				tblReturn.Controls.Add(addRow(mcstrLABEL_XSLT_STYLESHEET,txtxslt_stylesheet));
			}
			if(_enable_TemplateName)
			{
				txttemplatename = new TextBox();
				txttemplatename.ID = mcstrID_TEMPLATENAME;
				tblReturn.Controls.Add(addRow(mcstrLABEL_TEMPLATENAME,txttemplatename));
			}
			if(_enable_RootNodeName)
			{
				txtrootnodename = new TextBox();
				txtrootnodename.ID = mcstrID_ROOTNODENAME;
				tblReturn.Controls.Add(addRow(mcstrLABEL_ROOTNODENAME,txtrootnodename));
			}
			if(_enable_MaxReturn)
			{
				txtmaxreturn = new TextBox();
				txtmaxreturn.ID = mcstrID_MAXRETURN;
				txtmaxreturn.Width = Unit.Point(30);
				tblReturn.Controls.Add(addRow(mcstrLABEL_MAXRETURN,txtmaxreturn));
			}
			if(_enable_MaxPerChannel)
			{
				chkmaxperchannel = new CheckBox();
				chkmaxperchannel.ID = mcstrID_MAXPERCHANNEL;
				tblReturn.Controls.Add(addRow(mcstrLABEL_MAXPERCHANNEL,chkmaxperchannel));
			}
			if(_enable_active)
			{
				chkEnableLister = new CheckBox();
				chkEnableLister.ID = mcstrID_ENABLELISTER;
				tblReturn.Controls.Add(addRow(mcstrLABEL_ENABLELISTER,chkEnableLister));
			}

			return tblReturn;

		}
		#endregion buildInterface

		#region addRow(string strLabel, Control ctlControl)
		/// <summary>
		/// Returns a table row containing the passed control and a label
		/// </summary>
		/// <param name="txtControl">Control to Add to the row</param>
		/// <param name="dcTAG">associated tag object</param>
		/// <returns>TableRow object containing 2 table cells and control</returns>
		private TableRow addRow(string strLabel, Control ctlControl)
		{
			TableRow trRow = new TableRow();

			TableCell tdCell = new TableCell();
			Literal ltrLabel = new Literal();

			ltrLabel.Text = strLabel;

			tdCell.Controls.Add(ltrLabel);

			trRow.Controls.Add(tdCell);

			tdCell = new TableCell();

			tdCell.Controls.Add(ctlControl);

			trRow.Controls.Add(tdCell);

			return trRow;
		}
		#endregion addRow

		#region loadDataToInterface()
		/// <summary>
		/// Called after postback checks in order to populate data
		/// </summary>
		private void loadDataToInterface()
		{
			if(_enable_search_Channels)
			{
				txtsearch_channels.Text = _search_Channels;
			}
			if(_enable_search_Placeholders)
			{
				txtsearch_placeholders.Text = _search_Placeholders;
			}
			if(_enable_XSLT_Stylesheet)
			{
				txtxslt_stylesheet.Text = _XSLT_Stylesheet;
			}
			if(_enable_TemplateName)
			{
				txttemplatename.Text = _TemplateName;
			}
			if(_enable_RootNodeName)
			{
				txtrootnodename.Text = _RootNodeName;
			}
			if(_enable_MaxReturn)
			{
				txtmaxreturn.Text = _MaxReturn.ToString();
			}
			if(_enable_MaxPerChannel)
			{
				chkmaxperchannel.Checked = _MaxPerChannel;
			}
			if(_enable_active)
			{
				chkEnableLister.Checked = _active;
			}
		}
		#endregion loadDataToInterface

		#region getFromInterface()
		/// <summary>
		/// Called after postback checks in order to populate data
		/// </summary>
		private void getFromInterface()
		{
			if(_enable_search_Channels)
			{
				_search_Channels = txtsearch_channels.Text;
			}
			if(_enable_search_Placeholders)
			{
				_search_Placeholders = txtsearch_placeholders.Text;
			}
			if(_enable_XSLT_Stylesheet)
			{
				_XSLT_Stylesheet = txtxslt_stylesheet.Text;
			}
			if(_enable_TemplateName)
			{
				_TemplateName = txttemplatename.Text;
			}
			if(_enable_RootNodeName)
			{
				_RootNodeName = txtrootnodename.Text;
			}
			if(_enable_MaxReturn)
			{
				try
				{
					_MaxReturn = int.Parse(txtmaxreturn.Text);
				}
				catch
				{
				}
			}
			if(_enable_MaxPerChannel)
			{
				_MaxPerChannel = chkmaxperchannel.Checked;
			}
			if(_enable_active)
			{
				_active = chkEnableLister.Checked;
			}
		}
		#endregion loadDataToInterface

		#region reprocessResult(string XSL_Path)
		/// <summary>
		/// Call this function to reuse the generate XML
		/// </summary>
		/// <param name="XSL_Path">Path to new style sheet to use, if null the XML is returned as string</param>
		/// <returns>Transformed XML as string</returns>
		public string reprocessResult(string XSL_Path)
		{
			string sReturn = "";

			if(phLister != null)
			{
				
				sReturn = phLister.reprocessResult(XSL_Path);
			
			}

			return sReturn;

		}
		#endregion



		private XmlPlaceholder BoundXmlPlaceholder
		{
			get
			{
				return (XmlPlaceholder)this.BoundPlaceholder;
			}
		}
		
		#region Display As HTML

		/// <summary>
		/// Convert HTML and XML to display in a browser
		/// </summary>
		/// <param name="strDisplay">The string to convert</param>
		/// <returns>The converted string</returns>
		private string displayAsHTML(string strDisplay)
		{
			string strReturn = strDisplay;
			strReturn = strReturn.Replace("><", "&gt;*BE**BR**B*&lt;");
			strReturn = strReturn.Replace("<","*B*&lt;");
			strReturn = strReturn.Replace(">", "&gt;*BE*");
			strReturn = strReturn.Replace("*BR*", "<br>");
			strReturn = strReturn.Replace("*BE*", "</b>");
			strReturn = strReturn.Replace("*B*", "<b>");
			return strReturn;
		}

		#endregion

		/// <summary>
		/// Cleans up resources used by the control.
		/// </summary>
		public override void Dispose()
		{	
			if (license != null)
			{
				license.Dispose();
				license = null;
			}
			base.Dispose ();
		}

	}
}
