using System;
using System.Text;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.Xml;
using System.Xml.Xsl;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.WebControls;

namespace MSIBPlusPack.Utilities
{
	/// <summary>
	/// Summary description for WebCustomControl1.
	/// </summary>
	/// <remarks>
	/// The control is design to work with minimal interaction.
	/// If the control properties are NOT set the following applies
	/// 
	/// Property			|	NOT SET ACTION											|	SET ACTION
	/// ---------------------------------------------------------------------------------------------------------------------------------------------------------
	/// searchChannels		| The Control will search it's own parent channel.			| Control will attempt to aquire and search specified channels 
	/// searchPlaceHolders	| The control will get info from all placeholders			| The control will attempt to gather info from placeholders specified
	/// XSLT_Stylesheet		| The control will Transform using internal XSLT			| The control will attempt transform using provided XSLT
	/// TemplateName		| The control will query all posts within directory			| The control will query only posts using specified templates
	/// ---------------------------------------------------------------------------------------------------------------------------------------------------------
	/// </remarks>
	[DefaultProperty("Text"),
		]
	internal class postList : System.Web.UI.WebControls.WebControl
	{

		#region Event Code
		public event EventHandler reprocessReady;

		protected void OnReprocessReady()
		{
			if(reprocessReady!=null)
			{
				reprocessReady(this, new EventArgs());
			}
		}

		public event EventHandler preTransform;

		protected void PreTransform()
		{
			if(preTransform!=null)
			{
				preTransform(this, new EventArgs());
			}
		}
		#endregion

		#region Text Property
		private string text;

		[Bindable(true),
			Category("Appearance"),
			DefaultValue("")]
		public string Text
		{
			get
			{
				return text;
			}

			set
			{
				text = value;
			}
		}
		#endregion

		#region Private Member Types
		
		public enum cmsPostProperties : int
		{
			name = 1,
			displayname = 2,
			url = 3,
			guid = 4,
			number = 5
		};


		#endregion

		#region Private Member Constants

		//Root Node Name
		private const string m_cstrROOT_NODE = "root";

		// Should the control validate that Channels contain postings
		private const bool m_blnVALIDATE_CHANNEL = false;

		// Should the Xml be returned if the transform returns a blank
		private const bool m_blnRETURN_XML_IF_BLANK = false;

		// Programatically Items Are Added with the following node name
		private const string m_cstrEXTERNAL_ITEM = "externalitem";

		#region XmlNode Constants
		
		// Post Node
		private const string m_cstrXML_NODE_POST = "post";

		// Placeholder Node
		private const string m_cstrXML_NODE_PLACEHOLDER = "placeholder";
		
		// Name
		private const string m_cstrXML_NODE_NAME = "name";

		// Value
		private const string m_cstrXML_NODE_VALUE = "value";






		#endregion

		// The items that should be cleaned from values.
		private const string m_cstrREPLACE_VALUES = "<br>,&nbsp;";

		// The replacement for the items replaced.
		private const string m_cstrREPLACE_VALUE = " ";

		#endregion

		#region Private Member Variables
		
		//This is used to store the Channels the control will search (Comma Delimited)
		private string _search_Channels = "";

		/// <summary>
		/// This contains a comma delimited list of placeholders to retrieve
		/// </summary>
		private string _search_Placeholders = "";

		/// <summary>
		/// The Path to the XSLT Style sheet to use for the transform
		/// </summary>
		private string _XSLT_Stylesheet = "";

		private string _TemplateName = "";

		private string _RootNodeName = m_cstrROOT_NODE;

		//This is the number of items returned (Post Items)
		private int _ItemCount = 0; 

		//If this is left at Zero, all items found are returned
		private int _MaxReturn = 0;

		//Should the Max Return be on a per channel basis
		private bool _MaxPerChannel = false;

		//Used mainly in debug, if true the code attempts to save the resultant XML to file
		private bool _SaveXml = false;

		// This is the XML Produced, and can be used for reprocessing
		private XmlDocument _xmlResult;

		// XSL Params
		private XsltArgumentList _XSL_Args = null;

		// Must a channel contain Postings
		private bool _validatePostings = m_blnVALIDATE_CHANNEL;
		
		// Should the Xml be returned if the transform returns a blank
		private bool _ReturnXml = m_blnRETURN_XML_IF_BLANK;

		// Programatically Items Are Added with the following node name
		private string _externalItemNodeName = m_cstrEXTERNAL_ITEM;

		// Items to replace within value text
		private string _replaceList = m_cstrREPLACE_VALUES;

		// Replace those items with
		private string _replacementValue = m_cstrREPLACE_VALUE;

		#endregion

		#region Public Properties

		#region search_Channels
		/// <summary>
		/// This should be a comma delimited list of channels that are to be searched
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma delimted list of channels to search, if left blank the current channel will be used")]
		public string search_Channels
		{
			set
			{
				_search_Channels = value;
			}
			get
			{
				return _search_Channels;
			}
		}
		#endregion

		#region search_Placeholders
		/// <summary>
		/// A comma delimited list of placeholders to be interogated from each post
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma delimted list of placeholders to search, if left blank all placeholders will be left blank")]
		public string search_Placeholders
		{
			set
			{
				_search_Placeholders = value;
			}
			get
			{
				return _search_Placeholders;
			}
		}
		#endregion
		
		#region XSLT_Stylesheet
		/// <summary>
		/// A path to the XSL style sheet that will be used to present the gathered data
		/// </summary>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("Custom"),
		DefaultValue(""),
		Description("Set this to change the presentation style of the control. Using XSLT")]
		public string XSLT_Stylesheet
		{
			set
			{
				_XSLT_Stylesheet = value;
			}
			get
			{
				return _XSLT_Stylesheet;
			}
		}
		#endregion

		#region TemplateName
		/// <summary>
		/// Set this property if the directory contains templates you do not wish to return, 
		/// if set then the control will only return results from templates whose names are found within this comma delimited list.
		/// If left blank all templates will be parsed.
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(""),
		Description("A comma delimted list of templates to search, if left blank all templates will be considered valid")]
		public string TemplateName
		{
			set
			{
				_TemplateName = value;
			}
			get
			{
				return _TemplateName;
			}
		}
		#endregion

		#region RootNodeName
		/// <summary>
		/// Set this to channge the default root node name in the XML
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(m_cstrROOT_NODE),
		Description("Use this to change the name of the root XML node")]
		public string RootNodeName
		{
			set
			{
				_RootNodeName = value;
			}
			get
			{
				return _RootNodeName;
			}
		}
		#endregion

		#region SaveXml
		/// <summary>
		/// If set to true the control will attempt to save a copy of the XmlDocument 
		/// </summary>
		[
		Browsable(true),
		Category("Debug"),
		DefaultValue(false),
		Description("If set to true the control will attempt to save a copy of the XmlDocument to C:\\Temp\\sampleXml.xml")]
		public bool SaveXml
		{
			set
			{
				_SaveXml = value;
			}
			get
			{
				return _SaveXml;
			}
		}
		#endregion

		#region MaxReturn
		/// <summary>
		/// Set this if you wish to limit the number of items returned
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(0),
		Description("Set this if you wish to limit the number of items returned")]
		public int MaxReturn
		{
			set
			{
				_MaxReturn = value;
			}
			get
			{
				return _MaxReturn;
			}
		}
		#endregion

		#region MaxPerChannel
		/// <summary>
		/// If true then the Max Return is reset for each Channel
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(false),
		Description("If true then the Max Return is reset for each Channel")]
		public bool MaxPerChannel
		{
			get
			{
				return _MaxPerChannel;
			}
			set
			{
				_MaxPerChannel = value;
			}
		}
		#endregion

		#region XSL_Args
		/// <summary>
		/// Using the PreTransform event this property can be used to supply dynamic Args
		/// </summary>
		public XsltArgumentList XSL_Args
		{
			set
			{
				_XSL_Args = value;
			}
		}
		#endregion

		#region validateChannel
		/// <summary>
		/// If true then the Max Return is reset for each Channel
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(m_blnVALIDATE_CHANNEL),
		Description("If true a channel requires a posting to be rendered in production and preview mode.")]
		public bool validateChannel
		{
			get
			{
				return _validatePostings;
			}
			set
			{
				_validatePostings = value;
			}
		}
		#endregion

		#region ReturnXml
		/// <summary>
		/// If true then the Max Return is reset for each Channel
		/// </summary>
		[
		Browsable(true),
		Category("Custom"),
		DefaultValue(m_blnRETURN_XML_IF_BLANK),
		Description("If true the Xml result will be returned if the transform returns nothing.")]
		public bool ReturnXml
		{
			get
			{
				return _ReturnXml;
			}
			set
			{
				_ReturnXml = value;
			}
		}
		#endregion

		#region externalItemNodeName
		/// <summary>
		/// Programatically Items Are Added with the following node name
		/// </summary>
		[
			Category("Custom"),
			DefaultValue(m_cstrEXTERNAL_ITEM),
			Description("Programatically Items Are Added with the following node name")
		]
		public string externalItemNodeName
		{
			get
			{
				return _externalItemNodeName;
			}
			set
			{
				_externalItemNodeName = value;
			}
		}
		#endregion

		#region replaceList
		/// <summary>
		/// Comma delimited list of Items to replace within value text
		/// </summary>
		[
		Category("Custom"),
		DefaultValue(m_cstrREPLACE_VALUES),
		Description("Comma delimited list of Items to replace within value text")
		]
		public string replaceList
		{
			get
			{
				return _replaceList;
			}
			set
			{
				_replaceList = value;
			}
		}
		#endregion

		#region replacementValue
		/// <summary>
		/// The value used to replace those items featured on the replaceList.
		/// </summary>
		[
		Category("Custom"),
		DefaultValue(m_cstrREPLACE_VALUE),
		Description("The value used to replace those items featured on the replaceList.")
		]
		public string replacementValue
		{
			get
			{
				return _replacementValue;
			}
			set
			{
				_replacementValue = value;
			}
		}
		#endregion

		#endregion

		#region Custom Functions
			
		#region getXML()

		#region Overloads [getXML(string searchChannels, string searchPlaceHolders), getXML(string searchPlaceHolders)]
		/// <summary>
		/// Using the appropriate settings this function returns XML of the searched directories
		/// </summary>
		/// <param name="searchChannels">A comma delimited list of channels to be searched</param>
		/// <param name="searchPlaceHolders">A comma delimited list of placeholders to included in the resultant XML</param>
		/// <returns>XmlDocument containing posts found within the directory</returns>
		public XmlDocument getXML(string searchChannels, string searchPlaceHolders)
		{
		
			this.search_Channels = search_Channels;
			this.search_Placeholders = search_Placeholders;
			
			return getXML();
		
		}

		/// <summary>
		/// Using the appropriate settings this function returns XML of the searched directories
		/// </summary>
		/// <param name="searchPlaceHolders">A comma delimited list of placeholders to included in the resultant XML</param>
		/// <returns>XmlDocument containing posts found within the directory</returns>
		public XmlDocument getXML(string searchPlaceHolders)
		{		
			this.search_Placeholders = search_Placeholders;
			
			return getXML();	
		}
		#endregion

		/// <summary>
		/// Using the appropriate settings this function returns XML of the searched directories
		/// </summary>
		/// <returns>XmlDocument containing posts found within the directory</returns>
		public XmlDocument getXML()
		{
			//Ok create our master document
			XmlDocument xmlReturn = new XmlDocument();

			XmlElement xmlRoot = xmlReturn.CreateElement("",_RootNodeName,"");

			//If we have been given a search_Channels setting we call into searchChannels
			//If not the the rule is to search only the current channel
			XmlElement xmlChannels = (_search_Channels.Length > 0) ? searchChannels(xmlReturn) : searchChannel(xmlReturn, CmsHttpContext.Current.Channel);
			
			xmlRoot.AppendChild(xmlChannels);

			xmlReturn.AppendChild(xmlRoot);

			return xmlReturn;
		}

		#endregion

		#region searchChannels(XmlDocument xmlDoc)
		/// <summary>
		/// Loops the Array of channels supplied
		/// </summary>
		/// <param name="xmlDoc">XmlDocument - used only to create elements</param>
		/// <returns>XmlElement</returns>
		public XmlElement searchChannels(XmlDocument xmlDoc)
		{
			string[] sSearchArray = _search_Channels.Split(',');

			XmlElement xmlChannels = xmlDoc.CreateElement("", "channels", "");

			for(int iLoop = 0; iLoop < sSearchArray.Length; iLoop++)
			{

				//If we are using MaxReturn and on a Per channel Basis, then we reset _Item Count Here
				_ItemCount = ((_MaxReturn > 0) && (_MaxPerChannel == true)) ? 0 : _ItemCount;

				string sChannel = sSearchArray[iLoop];

				Channel cmsChannel;

				// If we are given a * this refers to the current channel
				if(sChannel != "*")
				{
					cmsChannel = getChannel(sChannel);
				}
				else
				{
					cmsChannel = getChannel();
				}

				if(cmsChannel != null && validateChannelForRender(cmsChannel))
				{

					xmlChannels.AppendChild(searchChannel(xmlDoc, cmsChannel));
	
				}

			}

			return xmlChannels;
		}
		#endregion

		#region ValidateChannel
		protected virtual bool validateChannelForRender(Channel cmsChannel)
		{
			bool blnReturn = true;

			if(validateChannel && (WebAuthorContext.Current.Mode == WebAuthorContextMode.PresentationPublished || WebAuthorContext.Current.Mode == WebAuthorContextMode.PresentationUnpublishedPreview || WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringPreview))
			{
				
				blnReturn = cmsChannel.Postings.Count > 0;

			}


			return blnReturn;
		}

		#endregion

		#region searchChannel(XmlDocument xmlDoc, Channel cmsChannel)
		/// <summary>
		/// Loops all posts in the given channel, calls into searchPost to retrieve data from post
		/// </summary>
		/// <param name="xmlDoc"></param>
		/// <param name="cmsChannel"></param>
		/// <returns></returns>
		protected virtual XmlElement searchChannel(XmlDocument xmlDoc, Channel cmsChannel)
		{

			XmlElement xmlChannel = getChannelData(xmlDoc, cmsChannel);
			
			//Now we loop the posts in this channel and if valid we add them
			foreach(Posting cmsPost in cmsChannel.Postings)
			{
				if(isValidPost(cmsPost) && ((_MaxReturn == 0) || (_ItemCount < _MaxReturn)))
				{
					xmlChannel.AppendChild(searchPost(xmlDoc, cmsPost));
				}
			}
			
			return xmlChannel;

		}


		/// <summary>
		/// This function is responsible for gathering channel property values, and can be overridden
		/// </summary>
		/// <param name="xmlDoc">XmlDocument boject used to create XmlElements</param>
		/// <param name="cmsPost">cms Channel object</param>
		/// <returns></returns>
		protected virtual XmlElement getChannelData(XmlDocument xmlDoc, Channel cmsChannel)
		{

			#region Create Channel Fragment, and name attribute

			XmlElement xmlReturn = xmlDoc.CreateElement("","channel","");

			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "name", cmsChannel.DisplayName.Replace("&", "&amp;")));

			// Url
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "url", cmsChannel.UrlModePublished));

			// Posting Count
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, "postcount", cmsChannel.Postings.Count.ToString()));

			#endregion	

			return xmlReturn;

		}

		#endregion

		#region searchPost(XmlDocument xmlDoc, Posting cmsPost)
		/// <summary>
		/// This function build the XML Element that represents a post
		/// Each Post element has the following attributes
		///	  
		///	name			= Posting.Name
		///	displayname		= Posting.DisplayName
		///	url				= Posting.Url
		///	guid			= Posting.GUID
		///	  
		///	It also contains a collection of Placholder Nodes
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create elements</param>
		/// <param name="cmsPost">The CMS Posting to be searched</param>
		/// <returns>XmlElement representing the posting</returns>
		protected virtual XmlElement searchPost(XmlDocument xmlDoc, Posting cmsPost)
		{
		

			//Increment the item count 
			_ItemCount++;

			XmlElement xmlReturn = getPostData(xmlDoc, cmsPost);

			if(_search_Placeholders.Length > 0)
			{

				//Now we loop the placeholders
				foreach(Placeholder cmsPH in cmsPost.Placeholders)
				{
					if(isValidPlaceHolder(cmsPH))
					{
					
						xmlReturn.AppendChild(getPlaceholderData(xmlDoc, cmsPH));

					}
				}

			}

			return xmlReturn;

		}

		/// <summary>
		/// This function is responsible for gathering post property values, and can be overridden
		/// </summary>
		/// <param name="xmlDoc">XmlDocument boject used to create XmlElements</param>
		/// <param name="cmsPost">cms Posting object</param>
		/// <returns></returns>
		protected virtual XmlElement getPostData(XmlDocument xmlDoc, Posting cmsPost)
		{

			XmlElement xmlReturn = xmlDoc.CreateElement("",m_cstrXML_NODE_POST,"");

			#region Gather the default post values and assign them as attributes
			//Posting Name
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.name.ToString(), cmsPost.Name));
			//Display Name
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.displayname.ToString(), cmsPost.DisplayName));
			//URL
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.url.ToString(), cmsPost.Url));
			//GUID
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.guid.ToString(), cmsPost.Guid));
			//Item Number
			xmlReturn.Attributes.Append(getAttribute(xmlDoc, cmsPostProperties.number.ToString(),_ItemCount.ToString() ));
			#endregion

			return xmlReturn;

		}
		#endregion

		#region getPlaceholderData(XmlDocument xmlDoc, Placeholder cmsPH)
		/// <summary>
		/// Takes the Name and raw data and creates an element
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create the Element</param>
		/// <param name="cmsPH">The placeholder to query</param>
		/// <returns>XmlElement</returns>
		protected virtual XmlElement getPlaceholderData(XmlDocument xmlDoc, Placeholder cmsPH)
		{
			XmlElement xmlPH = xmlDoc.CreateElement("", m_cstrXML_NODE_PLACEHOLDER, "");

			#region Add the attributes

			Type phType = cmsPH.GetType();

			string sValue = (phType.UnderlyingSystemType.ToString().IndexOf("Image") > 0) ? ((ImagePlaceholder)cmsPH).Src :cmsPH.Datasource.RawContent;

			xmlPH.AppendChild(getElement(xmlDoc, m_cstrXML_NODE_NAME, cmsPH.Name));
			xmlPH.AppendChild(getElement(xmlDoc, m_cstrXML_NODE_VALUE, sValue));

			#endregion

			return xmlPH;
		}
		#endregion

		#region getAttribute(XmlDocument xmlDoc, string sName, string sValue)
		/// <summary>
		/// Create and returns and XmlAttribute from the passed parameters
		/// </summary>
		/// <param name="xmlDoc">XmlDocument</param>
		/// <param name="sName">string - then name of the attribute</param>
		/// <param name="sValue">string - the value</param>
		/// <returns>XmlAttribute</returns>
		protected XmlAttribute getAttribute(XmlDocument xmlDoc, string sName, string sValue)
		{
			XmlAttribute xmlReturnAttribute = xmlDoc.CreateAttribute(sName);

			xmlReturnAttribute.Value = sValue;

			return xmlReturnAttribute;
		}
		#endregion

		#region XmlElement getElement(XmlDocument xmlDoc, string sName, string sValue)
		/// <summary>
		/// Returns an XmlElement with the node name as requested and the value. 
		/// </summary>
		/// <param name="xmlDoc">XmlDocument - used to create the element</param>
		/// <param name="sName">string - name of the element node</param>
		/// <param name="sValue">string - the node content</param>
		/// <returns>XmlElement</returns>
		protected XmlElement getElement(XmlDocument xmlDoc, string sName, string sValue)
		{
			XmlElement xmlReturn = xmlDoc.CreateElement("", sName, "");

			sValue = cleanItem(sValue);

			try
			{

				xmlReturn.InnerXml = sValue;

			}
			catch
			{
				xmlReturn.InnerText = sValue;
			}

			return xmlReturn;

		}
		#endregion

		#region Validators [isValidPost(Posting cmsPost), isValidPlaceHolder(Placeholder cmsPH)]

		/// <summary>
		/// Applies the business rules, such a template validation to the post
		/// </summary>
		/// <param name="cmsPost">Post to validate</param>
		/// <returns>bool</returns>
		protected virtual bool isValidPost(Posting cmsPost)
		{
			bool blnReturn = true;

			#region Comment on Rule
			/*
				Postings are validated by their template, if the develop has specified specific templates.
				The templates are in comma delimited list (if supplied) and to keep things simple we simply
				look for the Postings template name in the string. 
			*/
			#endregion
			
			//First off, have we been given any templates to screen against
			if(_TemplateName.Length > 0)
			{

				string sText = cmsPost.Template.Name.ToUpper();
				
				blnReturn = (_TemplateName.ToUpper().IndexOf(sText) > -1);

			}

			return blnReturn;
		}

		protected virtual bool isValidPlaceHolder(Placeholder cmsPH)
		{
			bool blnReturn = true;

			//First off, have we been given any placeholders to screen against
			if(_search_Placeholders.Length > 0)
			{

				/*
				 
				  If our list of placeholders looks like this
					
						story_start,story_precis,test 
					
				  And we test for story our result is true.
				  by adding the commas we negate this problem.
				  
				  We add the commas to the test string to cater for 
				  where only only one item exists in the list etc.
				  
				*/
				string sText = "," + cmsPH.Name.ToUpper() + ",";
				string sTest = "," + _search_Placeholders.ToUpper() + ",";
				
				blnReturn = (sTest.IndexOf(sText) > -1) || _search_Placeholders == "*";

			}

			return blnReturn;
		}


		#endregion

		#region getChannel
		/// <summary>
		/// Returns the current channel
		/// </summary>
		/// <returns>Channel object</returns>
		private Channel getChannel()
		{
			return CmsHttpContext.Current.Channel;
		}

		/// <summary>
		/// Returns the requested channel using Searches
		/// </summary>
		/// <remarks>
		/// If sChannel starts with '/' then it is considered to be pathed from Root and will have '/Channels' appended.
		/// If sChannel does not start with '/' it will be considered to follow on from current channel
		/// </remarks>
		/// <param name="sChannel">Path to channel [NOTE: prefix '/channels/' is not required</param>
		/// <returns>Channel or NULL if not found</returns>
		private Channel getChannel(string sChannel)
		{
			string sSearch;

			int iTest = sChannel.IndexOf("/");

			if(sChannel.IndexOf("..") == 0)
			{
				return getChannelRelative(sChannel);
			}
			
			// First case [Most likely] we are going from route
			if(sChannel.IndexOf("/") == 0)
			{
				if(sChannel.ToLower().IndexOf("/channels") != 0)
				{
					sSearch = "/channels" + sChannel;
				}
				else
				{
					sSearch = sChannel;
				}
			}
			else
			{
				// We are following on from current channel
				sSearch = getChannel().Path + "/" + sChannel;
			}
				
			sSearch = sSearch.Replace("//","/");

			Channel cmsReturnChannel = (Channel)CmsHttpContext.Current.Searches.GetByPath(sSearch);

			return cmsReturnChannel;

		}

		private Channel getChannelRelative(string sPath)
		{
			Channel cmsReturnChannel = (Channel)CmsHttpContext.Current.Channel.GetByRelativePath(sPath);

			return cmsReturnChannel;
		}
		#endregion

		#region cleanItem(string sValue)
		/// <summary>
		/// Using the replaceList this function cleans the passed value.
		/// </summary>
		/// <param name="sValue">string - value to be cleaned</param>
		/// <returns>string</returns>
		protected virtual string cleanItem(string sValue)
		{
			string[] arrClean = replaceList.Split(',');

			string sReturn = sValue;

			for(int iLoop=0; iLoop<arrClean.Length; iLoop++)
			{
		
				sReturn = sReturn.Replace(arrClean[iLoop], replacementValue);
				sReturn = sReturn.Replace(arrClean[iLoop].ToUpper(), replacementValue);

			}

			return sReturn;
		
		}
		#endregion

		#endregion

		#region getHTML()
		/// <summary>
		/// This is the main render function and does the magic
		/// </summary>
		/// <returns>String containing HTML, or XML in displayable format</returns>
		private string getHTML()
		{
			_xmlResult = getXML();

			#region Save the XML away to a file
			if(_SaveXml)
			{
				try
				{

					_xmlResult.Save("C:\\temp\\sampleXML.xml");

				}
				catch
				{
					//Do Nothing
				}
			}

			#endregion

			string sReturn = "";
			string sError = "";

			// B4 Transform we raise an event to allow params to be added
			PreTransform();

			//If we have been given a style sheet then use it
			if(XSLT_Stylesheet.Length > 0)
			{
				

				sReturn = transFormResult(_XSLT_Stylesheet, _xmlResult);

			}

			
			//Either we didn't have a style sheet, or there was a nasty error
			//So we display the XML instead
			if(sReturn.Length == 0 && ReturnXml)
			{
				
				sReturn = postHelper.displayAsHTML(_xmlResult.InnerXml);

			}

			// Notify XML ready for reprocessing
			OnReprocessReady();

			return sReturn + sError;

		}
		#endregion

		#region transFormResult(string strXSL_Path, XmlDocument XML_Data)
		/// <summary>
		/// Transforms the supplied XmlDocument using Xsl
		/// </summary>
		/// <param name="strXSL_Path">string - path to Xsl</param>
		/// <param name="XML_Data">XmlDocument - the data to transform</param>
		/// <returns>string - transformed result</returns>
		private string transFormResult(string strXSL_Path, XmlDocument XML_Data)
		{
			XmlTextReader xmlTransform = null; 
			string sReturn = "";

			try
			{

				string sPath = HttpContext.Current.Server.MapPath(strXSL_Path);

				xmlTransform = new XmlTextReader(sPath);

				sReturn = postHelper.transformXML(XML_Data, xmlTransform, _XSL_Args);
	
				xmlTransform.Close();

			}
			catch(Exception e)
			{
				StringBuilder sErr = new StringBuilder();

				sErr.Append("<br><b>");
				sErr.Append("<error>");

				#region Message
				sErr.Append("<message>");

				sErr.Append(e.Message);

				sErr.Append("</message>");
				#endregion

				sErr.Append("</b>");

				if(xmlTransform != null)
				{	
					xmlTransform.Close();
				}

				sReturn += sErr.ToString();

			}

			return sReturn;
		}
		#endregion

		#region reprocessResult(string XSL_Path)
		/// <summary>
		/// Call this function to reuse the generate XML
		/// </summary>
		/// <param name="XSL_Path">Path to new style sheet to use, if null the XML is returned as string</param>
		/// <returns>Transformed XML as string</returns>
		public string reprocessResult(string XSL_Path)
		{
			if(XSL_Path == null)
			{
				return _xmlResult.InnerXml;
			}
			else
			{
				return transFormResult(XSL_Path, _xmlResult);
			}

		}
		#endregion

		#region addItem(string sName, string sDisplayName, string sUrl, string sGUID, string sText, string sNodeName)
		/// <summary>
		/// Allows the external addition of items to the Postlister Items Collection.
		/// </summary>
		/// <param name="sName">string</param>
		/// <param name="sDisplayName">string</param>
		/// <param name="sUrl">string</param>
		/// <param name="sGUID">string</param>
		/// <param name="sText">string</param>
		/// <param name="sNodeName">string</param>
		/// <example>object.addItem("news", "BBC News", "http://www.bbc.co.uk", "", "Local and international news site", "");</example>
		public virtual void addItem(string sName, string sDisplayName, string sUrl, string sGUID, string sText, string sNodeName)
		{
			// Determine our node name
			string nodeName = (sNodeName == null || sNodeName == "") ? externalItemNodeName : sNodeName;

			// Create an Element to contain the information
			XmlElement xmlItem = getElement(_xmlResult, nodeName, cleanItem( sText ));//_xmlResult.CreateElement("", nodeName, "");

			// Add our attribute values. Name / Display Name / Url / Guid / Text.
			xmlItem.Attributes.Append(getAttribute(_xmlResult, cmsPostProperties.name.ToString(), sName));

			xmlItem.Attributes.Append(getAttribute(_xmlResult, cmsPostProperties.displayname.ToString(), sDisplayName));

			xmlItem.Attributes.Append(getAttribute(_xmlResult, cmsPostProperties.url.ToString(), sUrl));

			xmlItem.Attributes.Append(getAttribute(_xmlResult, cmsPostProperties.guid.ToString(), sGUID));

			xmlItem.Attributes.Append(getAttribute(_xmlResult,cmsPostProperties.number.ToString(), (++_ItemCount).ToString()));
			
			
			_xmlResult.FirstChild.AppendChild(xmlItem);

		}
		#endregion

		/// <summary>
		/// Render this control to the output parameter specified.
		/// </summary>
		/// <param name="output"> The HTML writer to write out to </param>
		protected override void Render(HtmlTextWriter output)
		{
			output.Write(getHTML());
		}
	}
}
