using System;
using System.ComponentModel;
using System.Drawing;
using System.Web.UI;
using ioko.ComponentModel.LicenseProvider;
using Microsoft.ContentManagement.Publishing;
using Image = System.Web.UI.WebControls.Image;

namespace MSIBPlusPack.Web.UI.WebControls
{
	/// <summary>
	/// The MicrositeImage class manages microsite images for MCMS Channel scripts and Template pages.
	/// It allows templates to source images from the resource library, context sensitive to the current channel. 
	/// </summary>
	/// <remarks>
	/// <para>
	/// MSIB Plus Pack Microsite components require the root of the MCMS Resource gallery to include a "Channels" sub-gallery.
	/// The Channels Resource gallery should include sub-galleries exactly matching the nomenclature and structure of the site's Channels up to the level where any microsite components are to be applied.
	/// </para>
	/// <para>
	/// Each flavour of microsite image used by a single MCMS Channel Script or Template should have identical filenames.
	/// They should be distributed throughout children of the "Channels" Resource gallery to achieve the required microsite branding over CMS Channels.
	/// </para>
	/// <para>
	/// The property <see cref="MicrositeImage.ResourceName"/> is used to specify the filename of the image to locate in the Resource gallery.
	/// MicrositeImage iterates up through the Channel hierarchy from the current location.
	/// It maps each Channel path to the equivalent Resource gallery path (with "Channels" prefix).
	/// The class iterates through parent Resource gallaries until a file of the required name or the "Channels" top-level Resource gallery is found.
	/// The Image class's ImageUrl and AlternateText properties can be used to provide default values to use when no corresponding Resource gallery item is found.
	/// </para>
	/// <para>
	/// MicrositeImage acts like the Microsoft.Web.UI.WebControls.Image from which it derives, except the image URL (HREF attribute) and ALT attribute are derived from the current site context.
	/// </para>
	/// <para>
	/// To expose resources, the website configuration must offer client browsers a minimum of read access to the MCMS virtual directory 'NR' (which exposes items in the MCMS Resource gallery). This is automatically configured when a web site is configured as an MCMS site via the Server Configuration Application.
	/// </para>
	/// </remarks>
	/// <example>
	/// The example below is an ASPX-based instance of MicrositeImage that includes a Microsite resource named Balloon.bmp:
	/// <code>
	/// &lt;%@ Register TagPrefix="cc1" Namespace="MSIBPlusPack.Web.UI.WebControls" Assembly="MSIBPlusPack.Web.UI.WebControls" %&gt;
	/// &lt;cc1:MicrositeImage id="MicrositeImage1"
	/// ResourceName="balloon.bmp" AlternateValue="DisplayDescription"
	/// ImageUrl="/MSIBPlusPack/images/balloon.bmp" AlternateText="Have your say"
	/// IncludeChannelInAlt="true" ShowHelperText="false" 
	/// ImageUrl="/Images/balloon.bmp" AlternateText="Have your say"
	/// ForeColor="#FFC0C0" Width="72" Height="52" runat="server"&gt;&lt;/cc1:MicrositeImage&gt;
	/// </code>
	/// The code below performs the same task in C#:
	/// <code>
	/// MSIBPlusPack.Web.UI.WebControls.MicrositeImage myImage = new MSIBPlusPack.Web.UI.WebControls.MicrositeImage();
	/// myImage.ID = "MicrositeImage1";
	/// myImage.ResourceName = "balloon.bmp";
	/// myImage.AlternateValue = "ResourceProperty.DisplayDescription";
	/// myImage.ImageUrl = "/MSIBPlusPack/images/balloon.bmp";
	/// myImage.AlternateText = "Have your say";
	/// myImage.IncludeChannelInAlt = true;
	/// myImage.ShowHelperText = false;
	/// myImage.ImageUrl = "/Images/balloon.bmp";
	/// myImage.AlternateText = "Have your say";
	/// myImage.ForeColor="#FFC0C0";
	/// myImage.Width="72";
	/// myImage.Height="52";
	/// </code>
	/// </example>
	[
		ToolboxData("<{0}:Micrositeimage runat=\"server\"></{0}:Micrositeimage>"),
			ParseChildren(true),
		ToolboxBitmap(typeof(MicrositeImage), "MicrositeImageIcon.BMP")
	]
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public class MicrositeImage : Image
	{
		#region Private State management fields
		private CmsHttpContext cmsHttpContext;
		private ResourceProperty alternateValue;
		private string resourceName;
		private bool includeChannelInAlt;
		private bool showHelperText;
		#endregion

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		/// <summary>
		/// This member handles mandatory component initialization. It has been sealed.
		/// </summary>
		/// <param name="e">EventArgs parameter</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();

			base.OnInit (e);
		}

		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(MicrositeImage),this);
					lastDate = DateTime.Today;
				}

				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;

					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}

						return;

					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;

					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}		
			} 
			catch {}

			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}

		#endregion

		#region Constructors
		/// <summary>
		/// The instance constructor for the MicrositeImage class.
		/// Sets
		/// <list type="bullet">
		/// <item><see cref="MicrositeImage.AlternateValue"/> property to <see cref="ResourceProperty.DisplayName"/></item>
		/// <item><see cref="MicrositeImage.IncludeChannelInAlt "/> property to False</item>
		/// <item><see cref="MicrositeImage.ShowHelperText"/> property to True</item>
		/// </list>
		/// </summary>
		public MicrositeImage()
		{		
			includeChannelInAlt = false;
			resourceName = "";
			AlternateValue = ResourceProperty.DisplayName;
			showHelperText = true;
		}
		#endregion

		#region Properties
		/// <summary>
		/// The AlternateValue property specifies which property of the item in the Resource gallery should be used as the AlternateText - i.e. HTML ALT attribute for the IMG (image) element in the rendered HTML.
		/// It can be any value of enumeration <see cref="ResourceProperty"/>.
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(ResourceProperty.DisplayName),
				Description("Which Resource property to use as the Image's AlternateText.")
		]
		public ResourceProperty AlternateValue
		{
			get { return alternateValue; }
			set { alternateValue = value; }
		}

		/// <summary>
		/// The IncludeChannelInAlt property exposes a boolean indicating whether the Image AlternateText (HTML ALT attribute) should be prefixed with the name of the current Channel. 
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(false),
				Description("Should the Image AlternateText be prefixed with the name of the current Channel?")
		]
		public bool IncludeChannelInAlt
		{
			get { return includeChannelInAlt; }
			set { includeChannelInAlt = value; }
		}

		/// <summary>
		/// The ResourceName property exposes the name of the image item to locate in the CMS Resource gallery.
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(""),
				Description("The name of the image item to locate in the CMS Resource gallery.")
		]	
		public string ResourceName
		{
			get { return resourceName; }
			set { resourceName = value; }
		}

		/// <summary>
		/// The ShowHelperText property exposes a boolean indicating whether explanatory text should be included the AlternateText (HTML ALT attribute) of the image when it is displayed in CMS Site Edit mode.
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(true),
				Description("Whether the image's AlternateText should include explanatory text in CMS Site Edit mode.")
		]
		public bool ShowHelperText
		{
			get { return showHelperText; }
			set { showHelperText = value; }
		}
		#endregion

		#region Control creation members
		/// <summary>
		/// Attempts to find the specified item in the Resource gallery from the current Channel context.
		/// If found, it updates the base class's ImageUrl and AlternateText properties accordingly.
		/// </summary>
		protected override void  CreateChildControls()
		{
			cmsHttpContext = CmsHttpContext.Current;
			Channel chaCurrent = cmsHttpContext.Channel;

			while(!chaCurrent.Equals(cmsHttpContext.RootChannel)) {
				ResourceGallery rgChannel = (ResourceGallery)cmsHttpContext.Searches.GetByPath("/Resources" + chaCurrent.Path);
				if(rgChannel != null) {
					if(rgChannel.Resources[ResourceName] != null) {
						string alternateText = "";
						Resource resChannel = rgChannel.Resources[ResourceName];
						base.ImageUrl = resChannel.Url;

						if (showHelperText && (cmsHttpContext.Mode.Equals( PublishingMode.Update) || cmsHttpContext.Mode.Equals(PublishingMode.Unpublished)))
							alternateText = String.Format(@"{0}\{1}: In order to change this image, ensure that a resource named '{1}' exists in the resource gallery '{2}' or in a parent gallery.",
											CmsHttpContext.Current.Channel.Name,
											this.ResourceName, 
											cmsHttpContext.Channel.Path);
						else
						{
							switch(AlternateValue) 
							{
								case ResourceProperty.DisplayDescription:
									alternateText = resChannel.Description;
									break;
								case ResourceProperty.DisplayName:
									alternateText = resChannel.DisplayName;
									break;
								default:
									alternateText = resChannel.Name;
									break;
							}
							if (includeChannelInAlt)
								alternateText = String.Format("{0}: {1}", rgChannel.Name, alternateText);
						}
						base.AlternateText = alternateText;
						break;
					}	
				}
				chaCurrent = chaCurrent.Parent;
			}
			base.CreateChildControls();
		}
		#endregion
	}
}
