using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Design;
using System.Web.UI;
using System.Web.UI.Design;
using ioko.ComponentModel.LicenseProvider;
using Microsoft.ContentManagement.Publishing;

namespace MSIBPlusPack.Web.UI.WebControls
{
	/// <summary>
	/// The MicrositeStylesheet class manages microsite stylsheets for MCMS Channel Scripts and Template pages.
	/// It allows templates to source stylesheets from the resource library, context sensitive to the current channel. 
	/// </summary>
	/// <remarks> 
	/// <para>
	/// MSIB Plus Pack Microsite components require the root of the MCMS Resource gallery to include a "Channels" sub-gallery.
	/// The Channels Resource gallery should include sub-galleries exactly matching the nomenclature and structure of the site's Channels up to the level where any microsite components are to be applied.
	/// </para>
	/// <para>
	/// Each flavour of microsite stylesheet used by a single MCMS Channel Script or Template should have identical filenames.
	/// They should be distributed throughout children of the "Channels" Resource gallery to achieve the required microsite branding over CMS Channels.
	/// </para>
	/// <para>
	/// The property <see cref="MicrositeStylesheet.ResourceName"/> is used to specify the filename of the stylesheet to locate in the Resource gallery.
	/// MicrositeStylesheet iterates up through the Channel hierarchy from the current location.
	/// It maps each Channel path to the equivalent Resource gallery path (with "Channels" prefix).
	/// The class iterates through parent Resource gallaries until a file of the required name or the "Channels" top-level Resource gallery is found.
	/// </para>
	/// <para>
	/// When an applicable stylesheet cannot be found in the Resource gallery:
	/// <list type="bullet">
	/// <item>The default stylesheet URL is used - see <see cref="MicrositeStylesheet.DefaultUrl"/>.</item>
	/// <item>Or this control does not render a LINK element into the HTML.</item>
	/// </list>
	/// </para>
	/// <para>
	/// When appropriate, MicrositeStylesheet writes a LINK element into HTML documents to style it with the required microsite stylesheet.
	/// To achieve HTML compliancy it should be placed within the HEAD element atop the page, to extend this to achieve XHTML compliancy the <see cref="MicrositeStylesheet.XHtmlCompliant"/> property should be set to true.
	/// </para>
	/// <para>
	/// To expose resources, the website configuration must offer client browsers a minimum of read access to the MCMS virtual directory 'NR' (which exposes items in the MCMS Resource gallery). This is automatically configured when a web site is configured as an MCMS site via the Server Configuration Application.
	/// </para>
	/// </remarks>
	/// <example>
	/// The example below is an ASPX-based instance of MicrositeImage that includes a Microsite resource named stylesheet.css:
	/// <code>
	/// &lt;%@ Register TagPrefix="cc1" Namespace="MSIBPlusPack.Web.UI.WebControls" Assembly="MSIBPlusPack.Web.UI.WebControls" %&gt;
	/// &lt;HTML&gt;
	///		&lt;HEAD&gt;
	///			&lt;cc1:MicrositeStylesheet id="MicrositeStylesheet1" ResourceName="stylesheet.css" DefaultUrl="/MSIBPlusPack/stylesheets/default.css" runat="server"&gt;&lt;/cc1:MicrositeStylesheet&gt;&lt;/P&gt;
	///		&lt;/HEAD&gt;
	///		&lt;BODY&gt;
	///			&lt;H1&gt;Microsite Page&lt;/H1&gt;
	///		&lt;/BODY&gt;
	/// &lt;/HTML&gt;
	/// </code>
	/// </example>
	[
		ToolboxData("<{0}:MicrositeStylesheet runat=\"server\"></{0}:MicrositeStylesheet>"),
			ParseChildren(true),
			ToolboxBitmap(typeof(MicrositeStylesheet), "MicrositeStylesheetIcon.BMP")
	]
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public class MicrositeStylesheet : Control
	{
		#region Private state management fields
		private string resourceName;
		private string defaultUrl;
		private bool xHtmlCompliant;
		#endregion

		#region Protected state management fields
		/// <summary>
		/// The URL of the item in the Resource gallery. Its value is determined by the <see cref="MicrositeStylesheet.CreateChildControls"/> member and consumed by the <see cref="MicrositeStylesheet.Render"/> member.
		/// </summary>
		protected string url;
		#endregion

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		/// <summary>
		/// This member handles mandatory component initialization. It has been sealed.
		/// </summary>
		/// <param name="e">EventArgs parameter</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();

			base.OnInit (e);
		}

		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(MicrositeStylesheet),this);
					lastDate = DateTime.Today;
				}

				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;

					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}

						return;

					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;

					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}		
			} 
			catch {}

			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion

		#region Constructors
		/// <summary>
		/// The instance constructor for the MicrositeStylesheet class.
		/// Sets XHTML compliancy to false (see <see cref="MicrositeStylesheet.XHtmlCompliant"/>).
		/// </summary>
		public MicrositeStylesheet()
		{
			resourceName = "";
			defaultUrl = "";
			url = "";
			xHtmlCompliant = false;
		}
		#endregion

		#region Properties
		/// <summary>
		/// The DefaultURL string property exposes the URL of the default stylesheet that should be used when an appropriate item cannot be found in the Resource gallery.
		/// </summary>
		/// <remarks>
		/// This control does not verify that a stylesheet exists at the specified URL.
		/// </remarks>
		[
			Editor(typeof(UrlEditor), typeof(UITypeEditor)),
				Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(""),
				Description("The URL of the stylesheet to use when a valid item is not found in the Resource gallery")
		]
		public string DefaultUrl
		{
			get { return defaultUrl; }
			set { defaultUrl = value; }
		}

		/// <summary>
		/// The ResourceName property exposes the name of the stylesheet to locate in the CMS Resource gallery.
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(""),
				Description("The name of the stylesheet item to locate in the CMS Resource gallery.")
		]
		public string ResourceName
		{
			get { return resourceName; }
			set { resourceName = value; }
		}

		
		/// <summary>
		/// The XHtmlCompliant property exposes a boolean indicating whether the injected HTML should be XHTML compliant.
		/// </summary>
		[
			Bindable(true),
				Category("MSIB Plus Pack - Microsite"),
				DefaultValue(false),
				Description("Should the HTML be XHTML compliant?")
		]
		public bool XHtmlCompliant
		{
			get { return xHtmlCompliant; }
			set { xHtmlCompliant = value; }
		}
		#endregion	

		#region Control creation members
		/// <summary>
		/// This member overrides the base class's CreateChildControls member to determine the URL of an appropriate item in the Resource gallery.
		/// It updates <see cref="MicrositeStylesheet.url"/> accordingly.
		/// </summary>
		/// <remarks>
		/// It can be overridden to implement bespoke functionality to determine the value of <see cref="MicrositeStylesheet.url"/>.
		/// </remarks>
		protected override void CreateChildControls()
		{
			CmsHttpContext cmsHttpContext = CmsHttpContext.Current;

			url = "";
			Channel chaCurrent = cmsHttpContext.Channel;

			while ((0 == url.Length) && !chaCurrent.Equals(cmsHttpContext.RootChannel)) 
			{
				ResourceGallery rgChannel = (ResourceGallery)cmsHttpContext.Searches.GetByPath("/Resources" + chaCurrent.Path);
				if ((null != rgChannel) && (null != rgChannel.Resources[resourceName])) 
				{
					url = rgChannel.Resources[resourceName].Url;
					break;
				}
				chaCurrent = chaCurrent.Parent;
			}
		}
		#endregion

		#region Rendering members
		/// <summary>
		/// This member overrides the base class's Render member to write a LINK element into an HTML document to style it with the appropriate stylesheet.
		/// </summary>
		/// <remarks>
		/// It can be overridden to implment bespoke functionality to inject the stylesheet <see cref="MicrositeStylesheet.url"/> as required.
		/// </remarks>
		/// <param name="writer">The HtmlTextWriter output stream.</param>
		protected override void Render(HtmlTextWriter writer)
		{
			if ((url.Length > 0) || (defaultUrl.Length > 0)) {
				writer.WriteBeginTag("LINK");
				writer.WriteAttribute("rel", "stylesheet");
				writer.WriteAttribute("type", "text/css");
				if(url.Length > 0)
					writer.WriteAttribute("href", url);
				else
					writer.WriteAttribute("href", defaultUrl);
				if (xHtmlCompliant)
					writer.Write(" /");
				writer.Write(HtmlTextWriter.TagRightChar);
			}
		}
		#endregion
	}
}
