using System;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.XPath;
using System.Xml.Xsl;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.ComponentModel;
using System.Resources;
using System.Reflection;
using Microsoft.ContentManagement.Publishing;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.Web;
using Microsoft.ContentManagement.WebAuthor;
using Microsoft.ContentManagement.WebControls;
using ioko.ComponentModel.LicenseProvider;


[

	assembly:TagPrefix("MSIBPlusPack.Web.UI.WebControls.Navigation", "MSIB")

]

namespace MSIBPlusPack.Web.UI.WebControls
{
	/// <summary>
	/// This class provides the MSIB navigation control. This component will provide customisable output using XML/XSLT for CMS website navigation.
	/// Internal XSLT stylesheets are provided as default output formatting.
	/// </summary>
	/// <remarks>
	/// There are a number of internal rules which apply in all navigation modes:-<br/><br/>
	/// <list type="number">
	/// <item><description>You can overide the DisplayName of a Channel. Simply add a custom property called 'navName' to the Channel object, and the value of this Custom Property will be used.</description></item>
	/// <item><description>Prefixing the Channel/Posting name with '_' will cause the Channel or Posting to be hidden whilst in Presentation mode.</description></item>
	/// <item><description>If the Posting name is the same as the parent Channel name, the control assumes this is the default page for the Channel and does not render the Posting link in presentation mode.</description></item>
	/// </list>
	/// <seealso cref="NavigationControl.PresentationMode"/>
	/// </remarks>
	[
		Designer(typeof(MSIBPlusPack.Web.UI.WebControls.Design.NavigationDesigner)),
		DefaultProperty("PresentationMode"),
		LicenseProviderAttribute(typeof(PlusPackLicenseProvider)),
		ToolboxData("<{0}:NavigationControl runat=server></{0}:NavigationControl>")
	]
	public class NavigationControl : System.Web.UI.WebControls.WebControl
	{
		/// <summary>
		/// Called when the control is initialised.
		/// </summary>
		/// <param name="e">EventArgs class.</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();
			base.OnInit (e);
		}

		#region Licensing 

		#region Licensing function
		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

 				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(NavigationControl),this);
					lastDate = DateTime.Today;
				}
				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;
					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}
						return;
					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;
					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}           
			} 
			catch {}
				throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion Licensing function

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#endregion Licensing 
	
		#region Enums

		#region NavigationPresentationMode
		/// <summary>
		/// A list of valid navigation types the control is capable of providing. Provides values for the 
		/// 'PresentationMode' property.
		/// </summary>
		/// <remarks>
		/// The general types of navigation provided by the control are:-<br/><br/>    
		///  
		/// 1 - Secondary [All Channels]<br/>
		///	2 - Secondary [Select Primary Channel]<br/>
		///	3 - Secondary [Ignore '_' (DEBUG USE ONLY)]<br/>
		///	4 - Primary<br/>
		///	5 - Primary [As combo]<br/>
		///	6 - Breadcrumb<br/>
		///	7 - Highlight
		///	<seealso cref="NavigationControl.PresentationMode"/>
		/// </remarks>
		public enum NavigationPresentationMode :int
		{
			/// <summary>
			/// Secondary [All Channels] - Will display entire navigation up to CMS root Channel 
			/// (or the root Channel which has been specified through property values. 
			/// The selected Posting's parent will be expanded when using the default internal stylesheet 
			/// for this mode.
			/// </summary>
			SecondaryAllChannels = 1,
			/// <summary>
			/// Secondary [Select Primary Channel] - Only renders the parent Channel of the current Posting.
			/// </summary>
			SecondarySelectedPrimary = 2,
			/// <summary>
			/// Same as Secondary [All Channels] but Channels/Postings with a name prefixed with an underscore are included.
			/// </summary>
			SecondaryIgnoreUnderScore = 3,
			/// <summary>
			/// Displays primary Channel navigation.
			/// </summary>
			/// <remarks>
			/// Primary obeys the following rules:
			///		Uses MaxReturn to determine number of Channels to return
			///		In the initial pass it looks for 'Important' channels
			///		If these are not found it falls back on channels that are not hidden
			///		The Base channel is used to define the parent of the primary channels
			/// </remarks>
			Primary = 4,
			/// <summary>
			/// Displays primary Navigation in an ASP.Net combobox (when using the default XSLT stylesheet).
			/// </summary>
			/// <remarks>
			/// Primary obeys the following rules:
			///		Uses MaxReturn to determine number of Channels to return
			///		In the initial pass it looks for 'Important' channels
			///		If these are not found it falls back on channels that are not hidden
			///		The Base channel is used to define the parent of the primary channels
			/// </remarks>
			ComboBox	= 5,
			/// <summary>
			/// Displays the navigation as a breadcrumb path.
			/// </summary>
			BreadCrumb = 6,
			/// <summary>
			/// Returns only important posts within the current channel.
			/// </summary>
			/// <remarks>
			/// Highlight obeys the following rules:
			/// Presentation Mode:
			///		The current posting is ignored
			///		Hidden Postings '_' are ignored
			///		The posting must be marked as important to be included.
			///		Does obey the 'Ignore Style' setting
			///		Lists posts up to the MaxReturn setting [Default value=6]
			///	Edit/Author Mode:
			///		The current posting is ignored
			///		Hidden postings are shown
			///		The posting must be masked as important to be included
			///		Does not obey the Ignore Style setting
			///		Ignores MaxReturn and simply returns all important posts
			/// </remarks>
			Highlight = 7,
			/// <summary>
			/// This returns a listing of Channels and all sub channels
			/// </summary>
			/// <remarks>
			/// The following properties are unique to this presentation mode
			/// 
			/// MapDepth - int - This describe how many levels from the root that the control should iterate
			/// AllChannels - bool - If set to true this ignores the Important Channel rule
			/// 
			/// </remarks>
			SiteMap = 8,
			/// <summary>
			/// This returns a listing of Channels and all sub channels and postings in alphabetical order.
			/// </summary>
			/// <remarks>
			/// The following properties are unique to this presentation mode
			/// 
			/// MapDepth - int - This describe how many levels from the root that the control should iterate
			/// 
			/// 
			/// </remarks>
			AtoZ = 9
		};
		#endregion

		#region Root Channel finder GetRootChannelFrom
		/// <summary>
		/// A list of valid locations the control can use to find the Channel it should use as the root Channel.
		/// </summary>
		public enum GetRootChannelFrom : int
		{
			/// <summary>
			/// Will look for a Custom Property of the parent Channel with the name 'navName', or that specified in the 'CustomNavigationName' property. 
			/// </summary>
			ParentChannel = 1,

			/// <summary>
			/// Will look for a Custom Property of the current Posting with the name 'navName', or that specified in the 'CustomNavigationName' property. 
			/// </summary>
			ParentPosting = 2,

			/// <summary>
			/// Will use internal property
			/// </summary>
			Property = 3

		};
		#endregion

		#region Mode
		/// <summary>
		/// A list of valid behaviour modes for the control, such as 'Production', 'Development' etc.
		/// </summary>
		public enum EnmMode : int
		{
			/// <summary>
			/// The control will output useful error messages and maintain page state.
			/// </summary>
			Development = 1,
			/// <summary>
			/// The control will output the generated XML, all debug and error messages, and page state 
			/// will not be maintained.
			/// </summary>
			DevelopmentDebugXML = 2,
			/// <summary>
			/// The control will not output any error messages. If there is a problem using the XSLT stylesheet 
			/// which has been supplied (or if none has been supplied), an internal XSLT stylesheet will be used.
			/// </summary>
			Production = 3,
			/// <summary>
			/// The control will only display in CMS unpublished mode.
			/// </summary>
			ProductionEditModeOnly = 4
		};
		#endregion

		#endregion

		#region Events Declarations
		
		/// <summary>
		/// This event is raised prior to the transformation of the XML representing the navigation. 
		/// This event can be used to add additional links to the navigation. This would be accomplished by 
		/// calling the 'AddItem' method to append XML nodes to the generated XML.
		/// </summary>
		/// <remarks>This event uses the standard EventHandler delegate.
		/// <seealso cref="NavigationControl.AddItem"/>
		/// </remarks>
		public event EventHandler PreTransform;

		#endregion

		#region Event Triggers
		
		/// <summary>
		/// Raises the PreTransform event.
		/// </summary>
		/// <param name="e">Instance of EventArgs class.</param>
		/// <remarks>
		/// <seealso cref="NavigationControl.PreTransform"/>
		/// </remarks>
		public void RaisePreTransform(EventArgs e)
		{

			// If there is a subscription
			if(PreTransform != null)
			{
				PreTransform(this, e);
			}

		}

		#endregion

		#region Constants

		private const string mcstrNOINDEX_NODE_TEXT = "noindex";
		private const string mcstrMSIB21_CHANNEL_IDENTIFIER = "%lcid%";

		// This constant is the default custom property that the nav control uses for overriding the display name
		private const string mcstrCUSTOM_PROPERTY_NAV_NAME = "navName";

		#region XmlNodeConstants
		
		private const string mcstrXML_ROOT_NODE = "nav";
		

		#endregion

		#endregion

		#region Member Types

		//Default Root Find (We default to using the standard internal property)
		private const GetRootChannelFrom m_cenmGET_ROOT_FROM = GetRootChannelFrom.Property;

		private const int mcintMAX_SECONDARY_NAV = 3;

		#endregion

		#region Public Properties

		#region PresentationMode
		/// <summary>
		/// This property is used to define which type of navigation the control should generate.
		/// </summary>
		/// <value>
		/// Only values in the 'NavigationPresentationMode' enumerator are valid. These can easily
		/// be selected using the Properties dialog in Visual Studio .Net.
		/// </value>
		/// <remarks>
		/// Property values are used differently in the various navigation modes. The usage for each
		/// mode is as follows:-<br/><br/>
		/// <u>Primary navigation</u><br/><br/>
		/// <list type="bullet">
		/// <item><description>The root Channel as specified in <code>CurrentChannel</code> property
		/// (programmatic assignment only) or (more typically) the <code>RootChannel</code> property is used to define the 
		/// parent of the primary Channels listed. If neither of these properties are specified, the CMS root Channel is used.</description></item>
		/// <item><description>The <code>MaxReturn</code> property to determine number of Channels to return</description></item>
		/// <item><description>In the initial pass the control looks for 'Important' Channels only</description></item>
		/// <item><description>If these are not found it falls back on channels that are not hidden</description></item>
		/// </list><br/><br/>
		/// <u>Secondary navigation</u><br/><br/>
		/// <list type="bullet">
		/// <item><description>The current CMS Channel is used as the root Channel, unless a Channel has been programmatically assigned
		/// using the <code>CurrentChannel</code> property. In the case where no current Channel can be established 
		/// (e.g. the current request has no CMSHttpContext), any value specified in the <code>RootChannel</code> property is
		/// used.</description></item> 
		/// </list><br/><br/>
		/// <u>Breadcrumb mode</u><br/><br/>
		/// <list type="bullet">
		/// <item><description>The current Channel is used as the last Channel displayed in the breadcrumb path.</description></item>
		/// <item><description>Any value specified in the <code>CurrentChannel</code> property(programmatic assignment 
		/// only) or (more typically) the <code>RootChannel</code> property is used to define the first Channel
		/// displayed. If no values are specified in these properties, the CMS root Channel is used.</description></item>
		/// <item><description>Alternatively the <code>CurrentPosting</code> property can be used to programmatically 
		/// assign a Posting whose location the control should use for it's breadcrumb path.</description></item>
		/// </list><br/><br/> 
		/// <u>Highlight mode</u><br/>
		/// <list type="bullet">
		/// <item><description>Only Postings marked as 'Important' are displayed.</description></item>
		/// </list>
		/// <br/>
		/// Additionally, the following are specific to the CMS mode of the user session:- 
		/// <br/><br/>
		/// Presentation mode:-
		/// <br/><br/>
		/// <list type="bullet">
		/// <item><description>Postings with names prefixed with an underscore are ignored by default.</description></item>
		/// <item><description>The number of Postings displayed can be limited by use of the <code>MaxReturn</code> 
		/// property.</description></item>
		/// <item><description>The <code>IgnoreStyle</code> property can be used to disregard rule 
		/// concerning Posting names beginning with an underscore.</description></item>
		/// </list><br/><br/> 
		/// Authoring mode:-
		/// <br/><br/>
		/// <list type="bullet">
		/// <item><description>Postings with names prefixed with an underscore are displayed.</description></item>
		/// <item><description>All Postings marked as 'Important' are shown regardless of value in <code>MaxReturn</code></description> property.</item>	
		/// </list><br/><br/> 
		/// <u>Sitemap mode</u>
		/// <br/><br/>
		/// <list type="bullet">
		/// <item><description>Returns a list of Channels according to values specified in <code>MapDepth</code> and
		/// <code>AllChannels</code> properties.</description></item>
		/// <item><description>The root Channel as specified in <code>CurrentChannel</code> property
		/// (programmatic assignment only) or (more typically) the <code>RootChannel</code> property is used to define the 
		/// starting point of the sitemap. If neither of these properties are specified, the CMS root Channel is used.</description></item>
		/// <item><description>The <code>MapDepth</code> property is used to determine how many levels in the Channel
		/// hierarchy (from the CMS root Channel) the sitemap should consist of.</description></item>
		/// <item><description>The <code>AllChannels</code> property is used to specify whether all Channels should
		/// be returned, or only Channels marked as 'Important'.</description></item>
		/// </list><br/><br/> 
		/// <u>A-Z mode</u>
		/// <br/><br/>
		/// <list type="bullet">
		/// <item><description>Returns a list of Channels and Postings sorted in alphabetical order.</description></item>
		/// <item><description>The root Channel as specified in <code>CurrentChannel</code> property
		/// (programmatic assignment only) or (more typically) the <code>RootChannel</code> property is used to define the 
		/// starting point of the sitemap. If neither of these properties are specified, the CMS root Channel is used.</description></item>
		/// <item><description>The <code>MapDepth</code> property is used to determine how many levels in the Channel
		/// hierarchy (from the root Channel which has been established) the A-Z list should consist of.</description></item>
		/// </list>
		/// <seealso cref="NavigationPresentationMode"/>
		/// </remarks>
		/// <example><code>PresentationMode="1"</code>
		/// This sets the control to render in Secondary [All Channels] mode.
		/// </example>
		[
		Category("MSIBNav"),
		DefaultValue(NavigationPresentationMode.SecondaryAllChannels),
		Description("Set this to change the presentation style of the control.")]
		public NavigationPresentationMode PresentationMode
		{
			get
			{
				return m_csPresMode;
			}
			set 
			{

				try
				{
					
					m_csPresMode = value;
	
				}
				catch
				{
					//Error handling	- Note we check our mode within the Error Report Function
					m_csPresMode = NavigationPresentationMode.SecondaryAllChannels;

					logError(mclngPRESENTATION_ERROR, "PresentationMode", mcstrPRESENTATION_ERROR_MESSAGE);

				}

			}

		}
		//Allow user to change the value of property - in design view
		bool ShouldSerializePresentationMode(){return true;}


		#endregion

		#region Dev Mode
		/// <summary>
		/// Determines the behaviour of the control. This parameter should be used carefully.
		/// </summary>
		/// <value>Only values in the Mode enumerator are valid. These can easily
	    /// be selected using the Properties dialog in Visual Studio .Net.</value>
		/// <remarks>
		/// The following table lists each mode and the corresponding behaviour.
		/// <list type="table">
		/// <listheader>
		/// <term>Enumerator value</term>
		/// <description>Behaviour</description>
		/// </listheader>
		/// <item>
		/// <term>Development</term>
		/// <description>Used in standard development, runs as in Production mode, but will display errors to the user interface.</description>
		/// </item>
		/// <item>
		/// <term>DevelopmentDebugXML</term>
		/// <description>Used when debugging XML output - control returns debug information and XML structure to the user interface.</description>
		/// </item>
		/// <item>
		/// <term>Production</term>
		/// <description>Used in production - will not return errors to the user inteface, and will resort to internal XSLT to return result if 
		/// no XSLT file is specified.</description>
		/// </item>
		/// </list>
		/// <seealso cref="NavigationControl.Mode"/>
		/// </remarks>
		/// <returns>enumerator <code>Mode </code></returns>
		[
		Category("MSIBNav"),
		DefaultValue(EnmMode.Production),
		Description("Set this to change mode of the control.")]

		public EnmMode Mode
		{
			get
			{
				return m_DevelopmentMode;
			}
			set
			{
				try
				{
					
					m_DevelopmentMode = value;

				}
				catch
				{
					//This property we default to Dev DEBUG mode
					m_DevelopmentMode = EnmMode.DevelopmentDebugXML;
				}
			}

		}

		//Allow user to change the value of property - in design view
		bool ShouldSerializeDevelopmentMode(){return true;}

		#endregion

		#region XslStyle
		/// <summary>
		/// A path to the main XSLT file used to render the navigation output of the control.
		/// </summary>
		/// <value>A string representing the virtual path to the XSLT file used to render output.</value>
		/// <remarks>
		/// In the event of the style sheet being badly formed or not existing,
		/// the code will revert to using it's own internal style sheet.
		/// </remarks>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("MSIBNav"),
		DefaultValue(""),
		Description("Set this to change the presentation style of the control using XSLT")]
		public string XslStyle
		{
			get
			{
				return mstrXSLStyleURL;
			}
			set
			{
				mstrXSLStyleURL = value;	
			}
		}

		bool ShouldSerializeXSLStyle(){return true;}

		#endregion

		#region XslErrorStyle
		/// <summary>
		/// A path to the XSL file used to render errors in debugXML mode.
		/// </summary>
		/// <value>A string representing the virtual path to the XSLT file used to render errors.</value>
		/// <remarks>
		/// In the event of the style sheet being badly formed or not existing,
		/// the code will revert to using it's own internal style sheet.
		/// <seealso cref="NavigationControl.Mode"/>
		/// </remarks>
		[
		Editor(typeof(System.Web.UI.Design.XslUrlEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("MSIBNav"),
		DefaultValue(""),
		Description("Set this to the internal Error style sheet.")]
		public string XslErrorStyle
		{
			get
			{
				return m_strError_XSL;
			}
			set
			{
				m_strError_XSL = value;	
			}
		}

		bool ShouldSerializeXSLErrorStyle(){return true;}

		#endregion

		#region MaxReturn
		/// <summary>
		/// Used in primary and highlight modes. This property determines the maximum number of Channels or Postings to be returned.
		/// </summary>
		/// <remarks>In primary mode, 'MaxReturn' refers to Channels. In highlight mode, 'MaxReturn' refers to Postings marked as 'Important'. 
		/// The default value is 6.</remarks>
		/// <value>An integer representing how many Channels or Postings should be returned.</value>
		[
		Category("MSIBNav"),
		DefaultValue(6),
		Description("In primary navigation mode this represents the maximum number of channels to return. In highlight mode it represents the maximum important posts.")]
		public int MaxReturn
		{
			get
			{
				return m_intMaxReturn;
			}
			set 
			{
				m_intMaxReturn = value;
				mintMax_Primary_Channels = m_intMaxReturn;
			}
		}

		bool ShouldSerializeMax_Return(){return true;}

		#endregion

		#region Include Current Posting
		/// <summary>
		/// If True, data for the current Posting is included when the XML is generated.
		/// </summary>
		/// <remarks>Default value is False. If True, an attribute of 'selected="true"' will be added to the XML node.</remarks>
		/// <value>True or False.</value>
		[
		Category("MSIBNav"),
		DefaultValue(false),
		Description("Should the control include the current posting within the dataset NOTE: a selected element will be added")]
		public bool IncludeCurrentPosting
		{
			get
			{
				return mblnIncludeCurrentPosting;
			}
			set
			{
				mblnIncludeCurrentPosting = value;
			}
		}

		#endregion

		#region Ignore Style
		/// <summary>
		/// If True, the rule of hiding Channels or Postings with names prefixed with an underscore is ignored.
		/// </summary>
		/// <remarks>Default value is False.</remarks>
		/// <value>True or False.</value>
		[
		Category("MSIBNav"),
		DefaultValue(false),
		Description("Should the control ignore formating rules")]
		public bool IgnoreStyle
		{
			get
			{
				return mblnIgnore_Style;
			}
			set
			{
				mblnIgnore_Style = value;
			}
		}

		bool ShouldSerializeIgnoreStyle(){return true;}

		#endregion

		#region Render Breaks
		/// <summary>
		/// If True, extra nodes are inserted into the generated XML to separate Channel elements.
		/// </summary>
		/// <remarks>Additionally, the Channel must have a custom property named 'break' with a 
		/// string value of 'true' for the XML to be generated in this way. This can be used for custom 
		/// formatting.
		/// </remarks>
		/// <value>True or False.</value>
		[
		Category("MSIBNav"),
		DefaultValue(false),
		Description("Should break elements be created against custom properties")]
		public bool RenderBreaks
		{
			get
			{
				return m_blnRender_Breaks;
			}
			set
			{
				m_blnRender_Breaks = value;
			}
		}

		bool ShouldSerializeRenderBreaks(){return true;}

		#endregion

		#region Root Channel
		/// <summary>
		/// This is the primary means of explicitly specifying what the control should consider it's 
		/// root Channel. 
		/// </summary>
		/// <value>A string value representing a CMS path to the Channel.</value>
		/// <remarks>See the 'PresentationMode' property for information on 
		/// how this value will be used, and what the effect of not specifying it is.
		/// <seealso cref="NavigationControl.PresentationMode"/>
		/// </remarks>
		/// <example>/Channels/subChannel1/myChannel</example>
		[
		Browsable(true),
		Category("MSIBNav"),
		DefaultValue(""),
		Description("What show the Nav control consider it's Root channel")]
		public string RootChannel
		{
			get
			{
				string sReturn = mstrRoot_Channel;
				
				// if the control has been told to determine its root from a method other than the control property
				if(sReturn == "" && DetermineRootChannelUsing != GetRootChannelFrom.Property)
				{
					
					sReturn = GetRootChannelName();

				}

				return sReturn;
			}
			set
			{
				//Ok is it a valid channel
				try
				{
					//This only works runtime 
					if(value != null)
					{
						mstrRoot_Channel = value;
					}
					else
					{
						mstrRoot_Channel = "";
					}

				}
				catch(Exception e)
				{

					mstrRoot_Channel = "Error";
					logError(mclngROOT_CHANNEL_ERROR, "Root Channel Property" ,mcstrROOT_CHANNEL_ERROR_MESSAGE + ' ' + e.Message);
					
				}
			}
		}
		#endregion

		#region DisplayRoot
		/// <summary>
		/// If True, data for the root Channel will be included when the XML is generated. 
		/// It should be noted that with this set to true there will always
		/// be a single channel element below the initial 'nav' element. 
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>
		/// In most cases this would be set to False.
		/// </remarks>
		[
		Browsable(true),
		Category("MSIBNav.Appearance"),
		DefaultValue(false),
		Description("Should the Root channel be included for transformation, NOTE: This will mean the root will be Parent.")]
		
		public bool DisplayRoot
		{

			get
			{
				return m_blnDisplayRoot;
			}
			set
			{
				m_blnDisplayRoot = value;

			}

		}

		bool ShouldSerializeRootChannel(){return true;}

		#endregion

		#region Search Indexing

		#region IndexControl
		/// <summary>
		/// Determines if the control should hide itself from search engines.
		/// </summary>
		/// <value>
		/// True or False.
		/// </value>
		/// <remarks>
		/// Default is True. If True, the control will be wrapped in tags with the name specified in the 'IndexNode' property.
		/// Typically, this value would be 'noindex' which would prevent most search engines indexing the control.
		/// </remarks>
		/// 
		[
		Category("MSIBNav"),
		DefaultValue(true),
		Description("Should the control shield itself from searching")]
		public bool IndexControl
		{
			get
			{
				return m_blnIndex;
			}
			set
			{
				m_blnIndex = value;
			}
		}

		bool ShouldSerializeIndexControl(){return true;}

		#endregion

		#region Index Node

		/// <summary>
		/// This allows the user to define the node name that is used to stop search engine indexing of the control.
		/// </summary>
		/// <value>A string which would be the name of the tags which are wrapped around the control. 
		/// The default value is 'noindex'.
		/// </value>
		/// <remarks>
		/// Setting this property automatically sets 'IndexControl' property to True. Typically 'noindex' would
		/// be used.
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue("noindex"),
		Description("Node name to wrap control with to stop indexing, if this is edited, IndexControl is set to true.")]
		public string IndexNode
		{
			
			get
			{
				
				return (m_strIndexNode.Length < 1) ? mcstrNOINDEX_NODE_TEXT : m_strIndexNode;
			}
			set
			{

				if(value.IndexOf(" ") < 1 && value.IndexOf(".") < 1)
				{
					m_strIndexNode = value;
				}
				IndexControl = true;
			}

		}

		bool ShouldSerializeIndexNode(){return true;}

		#endregion
		
		#endregion

		#region map depth
		/// <summary>
		/// Used in sitemap and A-Z modes. This property determines how many levels down the Channel hierarchy
		/// the control should gather Channels from.
		/// </summary>
		/// <remarks>The default value is 2.</remarks>
		/// <value>An integer representing how many levels to traverse.</value>
		[
		Category("MSIBNav"),
		DefaultValue(mcintMAP_DEPTH),
		Description("This value determines the number of Channel levels to navigate when in SiteMap or A-Z mode")]		
		public int MapDepth
		{
			set
			{
				m_intMapDepth = (value > 0) ? value : mcintMAP_DEPTH;
			}
			get
			{
				return m_intMapDepth;
			}
		}

		bool ShouldSerializemapDepth(){return true;}

		#endregion

		#region All Channels
		/// <summary>
		/// Set this to true if you wish to return all visible primary channels.
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>Note this is specific to SiteMap.</remarks>
		[
		Category("MSIBNav"),
		DefaultValue(false),
		Description("Set this to true if you wish to return all visible primary channels.")]
		public bool AllChannels
		{
			set
			{
				m_blnAllChannels = value;
			}
			get
			{
				return m_blnAllChannels;
			}
		}

		bool ShouldSerializeAllChannels(){return true;}
		#endregion

		#region Custom Properties
		/// <summary>
		/// A comma-delimted set of custom properties of a Channel or Posting object that should be included 
		/// within the XML generated by the control.
		/// </summary>
		/// <value>A string of comma-delimited custom property values which should be collected.</value>
		/// <remarks>
		/// This property allows the inclusion of Channel and Posting custom property values in the XML
		/// which is transformed into HTML. This can be used to facilitate different branding/colors 
		/// for each Channel for example.
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue(""),
		Description("A comma delimted set of custom properties that should be included within the XML.")]
		public string CustomProperties
		{
			get
			{
				return m_strCustomProperties;
			}
			set
			{
				m_strCustomProperties = value;
			}

		}

		#endregion

		#region DetermineRootChannelUsing
		/// <summary>
		/// Specfies how the control should determine the root Channel to use.
		/// </summary>
		/// <value>Only values in the GetRootChannelFrom enumerator are valid. These can easily
		/// be selected using the Properties dialog in Visual Studio .Net.
		/// </value>
		/// <remarks>
		/// For flexibility, there are several ways of specifying the root Channel for the control. Typically,
		/// the 'RootChannel' property would be used.
		/// <seealso cref="NavigationControl.GetRootChannelFrom"/>
		/// <seealso cref="NavigationControl.RootChannel"/>
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue(m_cenmGET_ROOT_FROM),
		Description("Use this property to determine where the control should get it's root Channel name from. \n This setting can be ignored if you intend on setting the base channel through the control")]
		public GetRootChannelFrom DetermineRootChannelUsing
		{
			set
			{
				m_enmGet_Root_Channel_From = value;
			}
			get
			{
				return m_enmGet_Root_Channel_From;
			}
		}
		#endregion

		#region RootChannelPropertyName
		/// <summary>
		/// When the root Channel is to be obtained from a Channel or Posting custom property value, 
		/// this property can override the default custom property name of 'root_channel'.
		/// </summary>
		/// <value>A string value representing the name of the custom property to use.
		/// </value>
		/// <remarks>
		/// The default custom property name used is 'root_channel'.
		/// <seealso cref="NavigationControl.DetermineRootChannelUsing"/>
		/// <seealso cref="NavigationControl.RootChannel"/>
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue(mcstrROOT_CHANNEL_CUSTOM_PROPERTY_NAME),
		Description("Set this if you are not using the default 'root_channel' property name.\n This setting can be ignored if you intend on setting the base channel through the control")]
		public string RootChannelPropertyName
		{
			set
			{
				m_strRoot_Channel_CustomProperty_Name = value;
			}
			get
			{
				if(m_strRoot_Channel_CustomProperty_Name.Length > 0)
				{
					return m_strRoot_Channel_CustomProperty_Name;
				}
				else
				{
					return mcstrROOT_CHANNEL_CUSTOM_PROPERTY_NAME;
				}
			}		
		}

		#endregion

		#region saveXML Path
		/// <summary>
		/// In development mode, the XML which is generated by the control is saved to this file.
		/// </summary>
		/// <value>A string value representing the absolute path of the XML file to save to.</value>
		/// <remarks>
		/// This allows the developer to easily work with and examine the XML which is generated.
		/// <seealso cref="NavigationControl.Mode"/>
		/// <seealso cref="NavigationControl.Mode"/>
		/// </remarks>
		[
		Editor(typeof(System.Windows.Forms.Design.FileNameEditor), typeof(System.Drawing.Design.UITypeEditor)),
		Category("MSIBNav"),
		DefaultValue(""),
		Description("This is where the resultant XML is saved to in development mode.")]
		public string XmlSavePath
		{
			get
			{
				return m_strSaveXmlPath;
			}
			set
			{
				m_strSaveXmlPath = value;	
			}
		}
		#endregion

		#region IgnoreCustomChannelName
		/// <summary>
		/// If True, the DisplayName property of the Channel object will be used in navigation, otherwise
		/// the value of the 'navName' Channel custom property (or an alternative specified in 'CustomNavigationName' property) will be used.
		/// </summary>
		/// <value>True or False.</value>
		/// <remarks>
		/// Default value is False.
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue(false),
		Description("Set this to true if you wish to ignore NavName property of channels.")]
		public bool IgnoreCustomChannelName
		{
			set
			{
				_IgnoreCustomChannelName = value;
			}
			get
			{
				return _IgnoreCustomChannelName;
			}
		}

		#endregion

		#region CustomNavigationName
		/// <summary>
		/// If Channel names are to be obtained from a Channel custom property value rather than 
		/// the Channel.DisplayName property (IgnoreCustomChannelName property is false), the custom property specified here will be used rather than the default custom
		/// property name of 'navName'. 
		/// </summary>
		/// <value>A string representing an alternative custom property name to use.</value>
		/// <remarks>
		/// If 'IgnoreCustomChannelName' is True, the DisplayName of a Channel is used. If False, the value of 
		/// the 'CustomNavigationName' property of each Channel is used, unless an alternative custom property name is 
		/// specified here.
		/// <seealso cref="NavigationControl.IgnoreCustomChannelName"/>
		/// </remarks>
		[
		Category("MSIBNav"),
		DefaultValue(mcstrCUSTOM_PROPERTY_NAV_NAME),
		Description("Use this to override the default property used to override the Nav Name.")]
		public string CustomNavigationName
		{
			set
			{
				_CustomNavName = value;
				if(_CustomNavName == "")
				{
					_CustomNavName = mcstrCUSTOM_PROPERTY_NAV_NAME;
				}
			}
			get
			{
				return _CustomNavName;
			}
		}

		#endregion

		#region CurrentPosting
		/// <summary>
		/// Gets or sets the current Posting programatically as used by the control in breadcrumb mode.
		/// </summary>
		/// <remarks>If not specified, the breadcrumb will lead to the current Posting.</remarks>
		/// <value>
		/// A Microsoft.ContentManagement.Publishing.Posting object which the breadcrumb path should lead to.
		/// </value>
		public Posting CurrentPosting
		{	
			set
			{
				_CurrentPosting = value;
			}
			get
			{
				if(_CurrentPosting == null)
				{
					try
					{

						return CmsHttpContext.Current.Posting;

					}
					catch
					{
						return null;
					}
				}
				
				return _CurrentPosting;
			}
		}
		#endregion

		#region CurrentChannel
		/// <summary>
		/// Gets or sets the root Channel for the control programmatically. 
		/// </summary>
		/// <remarks>If specified, this value takes precendence over any root Channel assigned by other means.</remarks>
		/// <value>
		/// A Microsoft.ContentManagement.Publishing.Channel object.
		/// </value>
		public Channel CurrentChannel
		{
			set
			{
				_CurrentChannel = value;
			}
			get
			{
				return _CurrentChannel;
			}
		}
		#endregion

		#region Member Variables

		#region Internal Default constants
		//Presentation mode
		private const int mcintPRESENTATION_MODE = (int) NavigationPresentationMode.SecondaryAllChannels;
		private const int mcintMAX_MODE = 4;
		//Dev Mode
		private const int mcintDEV_MODE = (int) EnmMode.Production;
		private const int mcintMAX_DEV = 3;
		//XSLT Style sheet 
		//Slightly different here, this is the default style sheet itself
		private const string mcstrSTYLE_PRIMARY = mcstrPRIMARY_XSL_DEFAULT; //See End of this Code for def
		private const string mcstrSTYLE_SECONDARY = mcstrSECONDARY_XSL_DEFAULT;
		//Determines whether the component obey rules about '_' and '|'
		private const bool mcblnIGNORE_STYLE = false;
		//Cache Period 
		private const long mclngCACHE_PERIOD = 0;
		//Maximum Primary Channels to Display - Only used in primary channel mode
		private const int mcintMAX_PRIMARY_CHANNELS = 6;

		//Map constants
		private const int mcintMAP_DEPTH = 2;

		//Default Root_Channel Property name
		private const string mcstrROOT_CHANNEL_CUSTOM_PROPERTY_NAME = "root_channel";

		#endregion

	

		private EnmMode m_DevelopmentMode = (EnmMode)mcintDEV_MODE;

		private NavigationPresentationMode m_csPresMode = (NavigationPresentationMode)mcintPRESENTATION_MODE;

		//This represents the maximum number of Channels/Posts to return in Primary and Highlight Mode
		private int m_intMaxReturn = 6;

		//When in secondary nav mode, should it display the given root node or not
		private bool m_blnDisplayRoot; 

		//This is whether or not the control is obeying the 'break' custom property rule
		private bool m_blnRender_Breaks;

		//This property allows the user to overide the internally supplied Error style sheet
		private string m_strError_XSL = "";

		//Ok, this is the URL to the style sheet specified
		private string mstrXSLStyleURL = "";
		//The Default internal XSL
		private string mstrDEFAULT_XSL = "";
		// Should the current posting be included within the dataset [Default is false]
		private bool mblnIncludeCurrentPosting = false;
		//Are we obeying flags
		private bool mblnIgnore_Style = mcblnIGNORE_STYLE;
		//Caching Policy [0 = No Cache]
		private long mlngCache_Period = mclngCACHE_PERIOD;
		private bool mblnCache_On; //This is simply an internal state variable
		//What root channel should we use? NOTE: "" denotes use site root
		private string mstrRoot_Channel = "";
		//Primary channels to display
		private int mintMax_Primary_Channels = mcintMAX_PRIMARY_CHANNELS;

		//Search Indexing
		private bool m_blnIndex = true;
		private string m_strIndexNode = mcstrNOINDEX_NODE_TEXT;

		//Custom property Inclusion
		private string m_strCustomProperties = "";
		private string[] m_arrCustomProperties;

		//Map Depth ie how many channels to drill down
		private int m_intMapDepth = mcintMAP_DEPTH;
		private bool m_blnAllChannels = false; 

		//Root channel custom property name
		private string m_strRoot_Channel_CustomProperty_Name = mcstrROOT_CHANNEL_CUSTOM_PROPERTY_NAME;

		//Where should the root channel name be retrieved from
		private GetRootChannelFrom m_enmGet_Root_Channel_From = m_cenmGET_ROOT_FROM;

		//Error
		private StringBuilder msbError = new StringBuilder();

		//Path to save the Nav XML to
		private string m_strSaveXmlPath = "";

		// This provides an item count attribute, used in sitemap only at present
		private int m_item = 0;

		// This allows the Nav to be drawn from another point, if set then the control uses this as the post as opposed current
		private Posting _CurrentPosting = null;

		// This if set will override the root_channel setting, and allows the Nav to provide secondary nav from anywhere
		private Channel _CurrentChannel = null;

		// If this is set to try the control will ignore the navname custom property
		private bool _IgnoreCustomChannelName = false;

		// The custom property the nav will use to override  the navname
		private string _CustomNavName = mcstrCUSTOM_PROPERTY_NAV_NAME;

		// The Navigation Xml. This is at class scope in order that other processes may add items via an AddItem interface
		private XmlDocument _xmlData = null;


		#endregion

		#region Error Numbers and Messages
		
		//Unknown
		private const long mclngNAV_ERROR_UNKOWN = 1000000;

		#region Style Sheet Errors
		private const long mclngXSL_ERROR_BASE = 9000000;
		private enum csNavErrorXSL : long
		{
			XSL_ERROR_MALFORMED = mclngXSL_ERROR_BASE + 1
		};
		
		#endregion
		
		#region Presentation Error
		private const long mclngPRESENTATION_ERROR = 2000000;
		private const string mcstrPRESENTATION_ERROR_MESSAGE = "The presentation mode requested was invalid, please check property setting, default value used.";
		#endregion

		#region Channel Path [Root Channel] Error
		private const long mclngROOT_CHANNEL_ERROR = 3000000;
		private const string mcstrROOT_CHANNEL_ERROR_MESSAGE = "The root channel provided, either does not exist, or path given is incorrect.";
		#endregion

		#endregion

		#region Error Handling

		#region Log_Error
		/// <summary>
		/// Appends an error to the string builder class
		/// </summary>
		/// <param name="errNumber"></param>
		/// <param name="Location"></param>
		/// <param name="Error_Description"></param>
		private void logError(long errNumber, string Location, string Error_Description)
		{
			//We only log errors if we are in Dev mode (We now check to see if anyone is Listening for the error event)
			if(m_DevelopmentMode != EnmMode.Production || NavigationError != null)
			{
				//Our template XML Node for an Error
				//Set the output encoding 
				ASCIIEncoding ascEnc = new ASCIIEncoding();

				//Now we open a steam onto the formatted memory
				MemoryStream xmlStream = new MemoryStream();

				//And finally we grab our XML Writer 
				XmlTextWriter xmlError = new XmlTextWriter(xmlStream, ascEnc);

				//Set formating
				xmlError.Formatting= Formatting.Indented;

				//Write out error, then we put it safely in out string builder
				xmlError.WriteStartElement("error");

				xmlError.WriteElementString("number", errNumber.ToString());
					
				xmlError.WriteElementString("location", Location);

				xmlError.WriteElementString("description", Error_Description);

				xmlError.WriteEndElement();


				xmlError.Flush();
				xmlError.Close();


				msbError.Append(ascEnc.GetString(xmlStream.ToArray()));

			}

		}
		#endregion

		#region Render Errors
		/// <summary>
		/// This function returns a transformed copy of the errors.
		/// </summary>
		/// <returns>Formatted errors collection</returns>
		/// <remarks>This function is called after testing that errors exist, and assumes that the logic test for devmode has been carried out</remarks>
		private string renderErrors()
		{

			StringBuilder sbReturn = new StringBuilder();

			//Aquire our name
			Assembly assembly = Assembly.GetExecutingAssembly();

			sbReturn.Append("<b>" + assembly.GetName().Name + "</b>");

			XmlTextReader xrdrErrorXSL;

			//attempt to load an external style sheet
			try
			{

				string strError_XSL_Path = HttpContext.Current.Server.MapPath(m_strError_XSL);

				
				XmlDocument xmlTest = new XmlDocument();
				xmlTest.Load(strError_XSL_Path);

				xrdrErrorXSL  = new XmlTextReader(strError_XSL_Path);

			}
			catch(Exception errTrans)
			{
				//Note here we only log an error here if the m_strError_XSL <> ""
				if(m_strError_XSL.Length > 0)
				{
					logError(mclngXSL_ERROR_BASE, "renderErrors()", "External stylesheet faulty. " + errTrans.InnerException);
				}
				//Style Sheet
				string strErrorXSL = accessDefaultXSL("Error.xslt");
				
				StringReader srdrErrorXSL = new StringReader(strErrorXSL);

				xrdrErrorXSL = new XmlTextReader(srdrErrorXSL);
			}

			sbReturn.Append(NavigationHelper.TransformXML("<errors>" + msbError.ToString() + "</errors>", xrdrErrorXSL));

			return sbReturn.ToString();

		}
		#endregion

		/// <summary>
		/// This event is raised when the control has generated errors. Details of the error(s) can 
		/// be accessed in XML string format through the 'Error' property. 
		/// </summary>
		/// <remarks>
		/// This event uses the standard EventHandler delegate.
		/// <seealso cref="NavigationControl.Error"/>
		/// </remarks>
		public event EventHandler NavigationError;

		/// <summary>
		/// Returns a string containing the XML of any errors generated.
		/// </summary>
		/// <remarks>
		/// An XML log of errors generated by the control is built, this property gives access to the output as string.
		/// </remarks>
		/// <value>A string in XML format.</value>
		/// <returns></returns>
		public string Error
		{

			get
			{
				return msbError.ToString();
			}
		}


		#endregion

		#endregion

		#region Internal Functions

		
		#region aquire an XML writer to string 
		/// <summary>
		/// </summary>
		/// <returns></returns>
		private XmlTextWriter getXMLTextWriter()
		{
			//Set the output encoding 
			ASCIIEncoding ascEnc = new ASCIIEncoding();

			//Now we open a steam onto the formatted memory
			MemoryStream xmlStream = new MemoryStream();

			//And finally we grab our XML Writer 
			XmlTextWriter xmlWrite = new XmlTextWriter(xmlStream, ascEnc);

			//Set formating
			xmlWrite.Formatting= Formatting.Indented;

			return xmlWrite;

		}

		#endregion

		#region Get Root Channel
		/// <summary>
		/// Private: returns the Root channel that the Nav should work from.
		/// </summary>
		/// <remarks>
		/// This call and the root channel are only relevant when in <code>PresentationMode 1</code></remarks>
		/// <returns>CMS Channel</returns>
		private Channel getRoot()
		{
			Channel cmsRoot;

			// Ok, has someone programmatically given us a root channel
			if(_CurrentChannel != null)
			{
				return _CurrentChannel;
			}

			string sRoot_Channel = this.RootChannel;
			if (IsMsib21ChannelPath(sRoot_Channel))
			{
				sRoot_Channel = ReplaceWithLocaleID(sRoot_Channel);
			}

			//Ok have we been given a root channel, via the property inteface?
			if(sRoot_Channel.Length > 0)
			{
				//The property has been validated, but just in case

				try
				{
					cmsRoot = (Channel)CmsHttpContext.Current.Searches.GetByPath(sRoot_Channel);

					if(cmsRoot == null)
					{
						//lets assume we have no current context
						CmsApplicationContext cmsApp = new CmsApplicationContext();

						cmsApp.AuthenticateAsGuest();

						cmsRoot = (Channel)cmsApp.Searches.GetByPath(sRoot_Channel);

					}
				}
				catch(Exception ech)
				{

					try
					{

						cmsRoot = CmsHttpContext.Current.RootChannel;
						logError(mclngNAV_ERROR_UNKOWN, "getRoot();", "Unable to aquire supplied channel " + ech.InnerException);

						if(cmsRoot == null)
						{
							//lets assume we have no current context
							CmsApplicationContext cmsApp = new CmsApplicationContext();

							cmsApp.AuthenticateAsGuest();

							cmsRoot = cmsApp.RootChannel;
						}
					}
					catch(Exception e)
					{
						//lets assume we have no current context
						CmsApplicationContext cmsApp = new CmsApplicationContext();

						cmsRoot = cmsApp.RootChannel;
					}
				}
			}
			else
			{

				//No, so we assume this installs Root
				switch(m_csPresMode)
				{
					case NavigationPresentationMode.SecondarySelectedPrimary:
					{
						cmsRoot = CmsHttpContext.Current.Channel;
						break;
					}
					default:
					{
						cmsRoot = CmsHttpContext.Current.RootChannel;
						break;
					}
				}
			}

			//Last bail out
			if(cmsRoot == null)
			{
				cmsRoot = CmsHttpContext.Current.Channel;
			}

			return cmsRoot;
		}
		#endregion

		#region GetRootChannelName()
		/// <summary>
		/// This code is used to determine where the Root Channel name is retrieved from
		/// </summary>
		/// <returns></returns>
		private string GetRootChannelName()
		{
			string strReturn = "";

			switch(m_enmGet_Root_Channel_From)
			{
				case GetRootChannelFrom.ParentChannel:
				case GetRootChannelFrom.ParentPosting:
				{
					try
					{
						//Get either the current posting or channel depending on user selection
						ChannelItem cmsItem = (m_enmGet_Root_Channel_From == GetRootChannelFrom.ParentChannel) ? (ChannelItem)CmsHttpContext.Current.Channel : (ChannelItem)CmsHttpContext.Current.Posting;

						//Attempt to retrieve a custom property from it
						strReturn = GetCustomProperty(cmsItem, this.RootChannelPropertyName);

						if(strReturn == "")
						{
							strReturn = mstrRoot_Channel;
						}

					}
					catch
					{
						strReturn = mstrRoot_Channel;
					}
					break;
				}
				default:
				{
					strReturn = mstrRoot_Channel;
					break;
				}
		

			}

			return strReturn;
				
		}
		#endregion

		#region Main Get XML Function
		/// <summary>
		/// This method returns an XmlDocument object which contains the XML structure representing the navigation 
		/// to be displayed. An XML string is then extracted from this document which is then transformed
		/// into HTML using XSLT.
		/// </summary>
		/// <remarks>
		/// The format and content of the XML generated will depend mainly on the mode selected in the 
		/// 'PresentationMode' property.
		/// <seealso cref="NavigationControl.PresentationMode"/>
		/// </remarks>
		/// <returns>An XmlDocument object containing the XML representation of the links to be displayed.</returns>
		public XmlDocument GetNavigationXmlAsDoc()
		{
			XmlDocument xmlNav = new XmlDocument();

            #region Depending on the selected navigation type we call an appropriate function to return XML 
			switch(m_csPresMode)
			{
				case NavigationPresentationMode.SecondaryAllChannels:
				case NavigationPresentationMode.SecondaryIgnoreUnderScore:
				case NavigationPresentationMode.SecondarySelectedPrimary:
				{
					xmlNav = getSecondaryNavXML();
					break;
				}
				case NavigationPresentationMode.ComboBox:
				case NavigationPresentationMode.Primary:
				{
					xmlNav = getPrimaryNavXML();
					break;
				}
				case NavigationPresentationMode.BreadCrumb:
				{
					xmlNav = getBreadCrumb();
					break;
				}
				case NavigationPresentationMode.Highlight:
				{
					xmlNav = getHighlight();
					break;
				}

				case NavigationPresentationMode.SiteMap:
				{
					xmlNav = getSiteMap();
					break;
				}
				case NavigationPresentationMode.AtoZ:
				{
					xmlNav = getAZNav();
					break;
				}
				default:
				{

					m_csPresMode = NavigationPresentationMode.SecondaryAllChannels;
					xmlNav = getPrimaryNavXML();
					break;
                 

				}

			}
            #endregion

			#region If in Dev Mode we save the XML
			if(m_DevelopmentMode != EnmMode.Production)
			{
				
				saveXML(xmlNav);
		
			}
			#endregion

			return xmlNav;
		}

 
		/// <summary>
		/// This method is the high-level function which returns the XML string representing the navigation 
		/// to be displayed. This XML can then be transformed into HTML using XSLT.
		/// </summary>
		/// <remarks>
		/// The format and content of the XML generated will depend mainly on the mode selected in the 
		/// 'PresentationMode' property.
		/// <seealso cref="NavigationControl.PresentationMode"/>
		/// </remarks>
		/// <returns>A string representing the XML representation of the links to be displayed.</returns>
		public string GetNavigationXmlAsString()
		{
			// If we are using display in edit mode only
			if((m_DevelopmentMode == EnmMode.ProductionEditModeOnly) && !isEditMode())
			{
				return "";
			}

			//Ok we first have to Deal with the presentation Mode
			//1-3 Are Secondary Navigation Modes
			//4 [+ in future] are Primary Nav options
			XmlDocument xmlNav;

			_xmlData = GetNavigationXmlAsDoc();

			// Raise out PreTransform Event
			RaisePreTransform(null);

			//we have our Nav XML
			//Ok this is why we are going to load the XSL into a reader first
			//If the property hasn't been set, or the XSL given is faulty
			//We want to use our internal variable XSL which is a STRING constant
			//So to ensure the transform call is standard, this is the way we work.
			XmlTextReader xrdrXSL = getStyleSheet(mstrXSLStyleURL);

			string strOutPut = "";

			#region Here we transform the resulting XML into output using XSL

			try
			{

				//And finally transform the results
				strOutPut = NavigationHelper.TransformXML(_xmlData, xrdrXSL);

			}
			catch(Exception e)
			{
				#region Error Handler [NOTE: this is where the failover occurs]
				//Now the most likely event here is that the XSL is incorrect
				//So we look at our development mode and act accordingly
			
				logError(mclngNAV_ERROR_UNKOWN, "GetNavigationXmlAsString();", "Transform with the supplied XSL failed. " + e.InnerException);

				switch(m_DevelopmentMode)
				{
					case EnmMode.Production:
					case EnmMode.Development:
					{
						
						//In production mode we attempt to use internal XSL before we completely fall over
						//To do this we call the overloaded option that returns the internal only.
						xrdrXSL = getStyleSheet();

						//We have a try here, as the internal may fail
						try
						{
							strOutPut = NavigationHelper.TransformXML(_xmlData, xrdrXSL);
						}
						catch(Exception exec)
						{
							logError(mclngNAV_ERROR_UNKOWN, "GetNavigationXmlAsString();", "Unknown Error. " + exec.InnerException);

						}

						break;
					}
						
				}
				#endregion
			}

			//Clear down the style sheet object
			//If we don't do this we experience problems with the style sheet being locked
			xrdrXSL.Close();
			xrdrXSL = null;

			#endregion

			#region Here we deal with the case of being in Dev or Dev_XML Mode
			//Dev mode may require us to spit our errors
			if(m_DevelopmentMode == EnmMode.Development || m_DevelopmentMode == EnmMode.DevelopmentDebugXML)
			{
				if(msbError.ToString().Length > 0)
				{

					strOutPut += renderErrors();

				}

				#region If we are in DEBUG_XML Mode we need to output the XML to the browser
				if(m_DevelopmentMode == EnmMode.DevelopmentDebugXML)
				{
						
					_xmlData.PreserveWhitespace = true;

					strOutPut += "<table bgcolor=\"#ffffcc\" border=\"0\" cellpadding=\"5\"><tr><td><font color=\"maroon\"><b>XML OUTPUT</b>";

					strOutPut += "</td></tr><tr><td><font color=\"navy\">";

					strOutPut += NavigationHelper.DisplayAsHTML(_xmlData.InnerXml);

					strOutPut += "</font></td><tr></table>";

				}
				#endregion

			}
			#endregion

			//This Code has been added to Support an Error Event coming from this control
			try
			{
				if(msbError.ToString().Length > 0){ NavigationError(this, new EventArgs());}
			}
			catch
			{
				//do nothing
			}

			

			_xmlData.RemoveAll(); 

			#region Add the NoIndex Tags 
			//No Index Code
			if(m_blnIndex == true)
			{
				StringBuilder strNoIndex = new StringBuilder();

				strNoIndex.Append("<");
				strNoIndex.Append(this.IndexNode);
				strNoIndex.Append(">");
				
				strNoIndex.Append(strOutPut);

				strNoIndex.Append("</");
				strNoIndex.Append(this.IndexNode);
				strNoIndex.Append(">");

				strOutPut = strNoIndex.ToString();

			}
			#endregion


			return strOutPut;

		}


		#endregion

		#region Stylesheet Retrieval functions
		/// <summary>
		/// Provided to return the internal XSL by default
		/// </summary>
		/// <returns><code>XmlTextReader</code> containing a style sheet</returns>
		private XmlTextReader getStyleSheet(){return getStyleSheet("");}
		/// <summary>
		/// Main function, this attempt to load the style sheet from the provide URL
		/// </summary>
		/// <param name="szFileURL"><code>string</code> URL to the style sheet</param>
		/// <returns><code>XmlTextReader</code> containing a style sheet</returns>
		/// <remarks>If the function is unable to load the XSL it will default to the internally stored XSL for the style.</remarks>
		private XmlTextReader getStyleSheet(string szFileURL)
		{

			XmlTextReader xrdrXSL;

			try
			{
				
				string strFilePath = HttpContext.Current.Server.MapPath(mstrXSLStyleURL);

			XmlDocument xmlTest = new XmlDocument();
				xmlTest.Load(strFilePath);

				xrdrXSL = new XmlTextReader(strFilePath);
					
			}
			catch(Exception e)
			{
				
				//Get the correct default XSL
				setDefaultXSL();
				
				StringReader srdrInternalXSL = new StringReader(mstrDEFAULT_XSL);

				xrdrXSL = new XmlTextReader(srdrInternalXSL);

				//Add an error
				logError(mclngPRESENTATION_ERROR, "GetNavigationXmlAsString();", "The XSL style sheet was incorrect or poorly formed." + e.Message);
			
			}

			return xrdrXSL;

		}


		#endregion

		#region Get Secondary Nav XML

		/// <summary>
		/// This is the primary call to retreive the Secondary Navigation Structure and handles Display logic
		/// </summary>
		/// <returns><code>string</code> containing XML ready for transformation</returns>
		/// <remarks>
		/// We have two Display modes - Contained within NavigationPresentationMode
		///  1 - Display All Channels, expand Selected Channel
		///  2 - Display Only the Immediate Parent channel and its content [Relative to selected Post]
		///
		/// NOTE : In all instances where a current channel is not found e.g. we are in an ASP page
		///		  The Root Channel will be used as the default if the Default_Channel has not been set
		/// This is the High level Call. 
		/// It dictates the scope of the NAV retreival by setting a 'Stop' Channel.
		/// E.G.
		/// If we are in Mode 2, we only want retrieve the Immediate Parent so we pass :-
		/// CmsHttpContext.Current.Posting.Parent.Name;
		///
		/// If we are in Mode 1 :-
		/// getRoot(); - Internal Function
		/// Now the function that we call is recursive
		/// It is capable of retrieving Channels and Posts (For the Selected Post Parent)
		/// And also of retriving Channels only (all others)
		/// It recursive works its way until it reaches the 'Base' Channel name given
		/// Obviously Only the first call from here should require posts to be retrived
		///</remarks>
		private XmlDocument getSecondaryNavXML()
		{
			
			XmlDocument xmlResult = new XmlDocument();

			XmlNode xmlRoot = xmlResult.CreateNode(XmlNodeType.Element, mcstrXML_ROOT_NODE, "");

			Channel cmsCurrent = (_CurrentChannel == null) ? CmsHttpContext.Current.Channel : _CurrentChannel;

			if(cmsCurrent == null)
			{
				
				cmsCurrent = getRoot();

			}
	
			xmlRoot.AppendChild(getElement(xmlResult, getRoot().Name, cmsCurrent, null, null, true));

			xmlResult.AppendChild(xmlRoot);

			return xmlResult;
		}

		#region Per Channel recursive call
		/// <summary>
		/// This call is recursive and builds an XML tree representing the CMS Navigation
		/// </summary>
		/// <param name="xmlDoc"><code>XmlDocument</code> used by the function to create and element.</param>
		/// <param name="szBaseChannel"><code>string</code> This should represent the Channel name at which rendering should stop</param>
		/// <param name="cmsRenderChannel"><code>Channel</code> the cmsChannel object to render.</param>
		/// <param name="xmlAppendElement"><code>XmlElement</code> - This elemnt will be a appended as a child of the Element being built</param>
		/// <param name="szAppendTo"><code>string</code> the <code>name</code> of the channel to append the element to.</param>
		/// <param name="displayPosts"><code>bool</code> if set to true the function will render posts as well</param>
		/// <returns>XmlElement</returns>
		private XmlElement getElement(XmlDocument xmlDoc, string szBaseChannel, Channel cmsRenderChannel, XmlElement xmlAppendElement, string szAppendTo, bool displayPosts)
		{
				
			//Create our new Channel Element
			//Each container channel has a 'children' element
			XmlElement xmlChannelElement = xmlDoc.CreateElement("", "children", "");

			// This new method will be called Natural Sorting
			foreach(ChannelItem cmsItem in cmsRenderChannel.AllChildren)
			{
				#region Channel Object
				//CHANNEL CODE
				if(cmsItem is Channel)
				{
					Channel cmsChildChannel = (Channel)cmsItem;

					// NOTE we override the render channel rule if we are displaying info contained within a hidden channel
					if((renderChannel(cmsChildChannel) || (cmsChildChannel.Name == szAppendTo)) && cmsChildChannel.Name != null)
					{
					
						//Get the Channel info
						XmlElement xmlChannel = AddChannelXmlElement(xmlDoc, cmsChildChannel);

						#region Does this channel have a divider
						try
						{
							if(cmsChildChannel.CustomProperties["break"].Value == "true" && m_blnRender_Breaks)
							{
							
								//Here we add a BREAK node
								xmlChannelElement.AppendChild(AddBreakXmlElement(xmlDoc));

							}
						}
						catch
						{
							
							//As we have no property do nothing

						}
						#endregion

						//Do we have a passed element, if so is it for this channel
						if(cmsChildChannel.Name == szAppendTo)
						{
							
							xmlChannel.AppendChild(xmlAppendElement);

						}

						// Now add it to our Parent Element
						xmlChannelElement.AppendChild(xmlChannel);

					}
				}
				#endregion

				#region Posting Object
				// POSTING CODE
				if(cmsItem is Posting && displayPosts)
				{
					Posting cmsPost = (Posting)cmsItem;

					//Check to ensure we are going to render this post
					if(renderPost(cmsRenderChannel, cmsPost))
					{

						XmlElement xmlPost = AddPostingXmlElement(xmlDoc, cmsPost);

						xmlChannelElement.AppendChild(xmlPost);

					}


				}
				#endregion
			}
	
			//Our return Element
			XmlElement xmlRoot;
			
			// Now if this channel isn't the 'Base' and has a parent we call into it
			if( (cmsRenderChannel.Name != szBaseChannel) && cmsRenderChannel.Parent != null)
			{
			
				//Pass it up
				xmlRoot = getElement(xmlDoc, szBaseChannel, cmsRenderChannel.Parent, xmlChannelElement, cmsRenderChannel.Name, false);	

			}
			else
			{

				if(m_blnDisplayRoot)
				{
					// This is the last last channel so
					// Call channel Info one last time
					xmlRoot = AddChannelXmlElement(xmlDoc, cmsRenderChannel);

					// Append everything we have to the channel info 
					xmlRoot.AppendChild(xmlChannelElement);
				}
				else
				{

					xmlRoot = xmlChannelElement;

				}
			}

			// Pass it up the tree
			return xmlRoot;

		}
		#endregion
		
		#endregion

		#region Channel and Post Render validators
		/// <summary>
		/// This function applies display rules and returns true if the channel should be rendered.
		/// </summary>
		/// <param name="cmsChannel"></param>
		/// <returns></returns>
		private bool renderChannel(Channel cmsChannel)
		{
			bool blnReturn = true;

			//Although there are not presently any hidden channel options, this is where they will go
			if((CmsHttpContext.Current.Mode == PublishingMode.Published || WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringPreview) && cmsChannel.Postings.Count == 0)
			{
				blnReturn = false;
			}

			if((WebAuthorContext.Current.Mode == WebAuthorContextMode.AuthoringPreview || WebAuthorContext.Current.Mode == WebAuthorContextMode.PresentationUnpublishedPreview) && cmsChannel.IsHiddenModePublished == true)
			{
				blnReturn = false;
			}
			

			return blnReturn;

		}


		/// <summary>
		/// This applies the display rules to a posting
		/// </summary>
		/// <param name="cmsChannel"></param>
		/// <param name="cmsPost"></param>
		/// <returns></returns>
		private bool renderPost(Channel cmsChannel, Posting cmsPost)
		{
			//Note we default to true
			bool blnReturn = true;

			// Ok here are the Rules for not Rendering [NOTE : if we are in Edit mode mode we render all regardless]
			// don't render if:
			// Posting Name = the Channel Name or Display Name 
			// Posting name starts with an '_'
			
			//So first off, only start to apply rules if we are in Published mode
			if(CmsHttpContext.Current.Mode == PublishingMode.Published && !mblnIgnore_Style)
			{
				string strPost = cmsPost.Name.ToUpper();
				string strChan = cmsChannel.Name.ToUpper();
				string strCDis = cmsChannel.DisplayName.ToUpper();
				
				blnReturn = (strChan != strPost) && (strCDis != strPost) && (strPost.IndexOf("_") != 0);

			}
			
			return blnReturn;

		}

		#endregion

		#region Element information retrieval code (Used across all nav styles)

		#region Get a single channel element
		/// <summary>
		/// Overridable. This method generates an XmlElement object containing CMS Channel information. The name 
		/// of the XML node will be 'channel'. Data captured includes the Channel's Name, 
		/// DisplayName, URL and any Custom Property values specified in the 'CustomProperties' property of the control. 
		/// </summary>
		/// <param name="xmlDoc">A valid XmlDocument object the XmlElement will be added to.</param>
		/// <param name="cmsChannel">The Channel whose details you wish to capture.</param>
		/// <returns>An XmlElement object containing the data on the passed Channel.</returns>
		/// <remarks>The name of the node returned will be 'channel'. Set the Mode property to DevelopmentDebugXML to see the XML generated by the control.</remarks>
		protected virtual XmlElement AddChannelXmlElement(XmlDocument xmlDoc, Channel cmsChannel) {return AddChannelXmlElement(xmlDoc, cmsChannel, "channel");}
		/// <summary>
		/// Overridable. This method generates an XmlElement object containing CMS Channel information. The name 
		/// of the XML node will be 'channel'. Data captured includes the Channel's Name, 
		/// DisplayName, URL and any Custom Property values specified in the 'CustomProperties' property of the control. 
		/// </summary>
		/// <param name="xmlDoc">A valid XmlDocument object the XmlElement will be added to.</param>
		/// <param name="cmsChannel">The Channel whose details you wish to capture.</param>
		/// <param name="strNodeName">A string representing the name the returned XmlElement should have.</param>
		/// <returns>An XmlElement object containing the data on the passed Channel.</returns>
		/// <remarks>The name of the node returned will be 'channel'. Set the Mode property to DevelopmentDebugXML to see the XML generated by the control.</remarks>
		protected virtual XmlElement AddChannelXmlElement(XmlDocument xmlDoc, Channel cmsChannel, string strNodeName)
		{
			// Create a channel Element
			XmlElement xmlChildChannel = xmlDoc.CreateElement("", strNodeName, "");

			//If we have been passed a null reference (NOTE This can occur, but rarely)
			if(cmsChannel == null)
			{
				return xmlChildChannel;
			}

			// Now the Properties

			// Name
			XmlElement xmlElem = xmlDoc.CreateElement("","name", "");

			xmlElem.InnerXml = cmsChannel.Name;

			xmlChildChannel.AppendChild(xmlElem);

			// Display Name
			xmlElem = xmlDoc.CreateElement("", "displayname", "");

			//Ok, The User can override the Display Name (May be to long etc.) by setting the custom property "navName"
			string strDisplayName;
			
			if(!IgnoreCustomChannelName)
			{
				try
				{
					strDisplayName = cmsChannel.CustomProperties[CustomNavigationName].Value;
				}
				catch
				{
					//If they haven't set navName
					strDisplayName = cmsChannel.DisplayName;
				}
			}
			else
			{
				strDisplayName = cmsChannel.DisplayName;
			}

			
			try
			{
				xmlElem.InnerXml = (mblnIgnore_Style==true)? strDisplayName : NavigationHelper.GetDisplayName(strDisplayName);
			}
			catch(Exception parseError)
			{
				//For speed only do this work if we are in develop mode
				if(m_DevelopmentMode != EnmMode.Production)
				{
					string strError = strDisplayName + " caused a Parser Error " + parseError.Message + " ";

					if(parseError.InnerException != null)
					{
						strError += parseError.InnerException.Message;
					}

					//We have had an XML parsee error, so log it and also add the Erroring name
					logError(mclngNAV_ERROR_UNKOWN, "AddChannelXmlElement", strError);
				}

				xmlElem.InnerText = (mblnIgnore_Style==true)? strDisplayName : NavigationHelper.GetDisplayName(strDisplayName);
				
			}	


			xmlChildChannel.AppendChild(xmlElem);
			
			// Url
			xmlElem = xmlDoc.CreateElement("", "postingurl", "");

			xmlElem.InnerText = cmsChannel.Url;


			xmlChildChannel.AppendChild(xmlElem);

			//Now we add Custom Properties if there are any
			if(m_strCustomProperties.Length > 0)
			{
				
				xmlChildChannel.AppendChild(AddCustomPropertiesXmlElement(xmlDoc, cmsChannel));

			}

			#region Primary Navigation Specific
			if(m_csPresMode == NavigationPresentationMode.Primary)
			{
				//This where we determine if the channel is the 'selected channel'
				//To do this we look at the current post and get it URL Path,
				//If our channel name is in there then presto
				try
				{

					Posting cmsPost = CmsHttpContext.Current.Posting;

					if(cmsPost.Path.ToLower().IndexOf(cmsChannel.Name.ToLower()) >= 0)
					{
						xmlElem = xmlDoc.CreateElement("", "selected", "");						
						xmlElem.InnerText = "true";
						
						xmlChildChannel.AppendChild(xmlElem);

					}


				}
				catch
				{
					//Do nothing
				}
			}
			#endregion

			return xmlChildChannel;

		}
		#endregion

		#region Get a single Post element
		/// <summary>
		/// Overridable. This method generates an XmlElement object containing CMS Posting information.
		/// The name of the XML node will be 'posting'. Data captured includes the Posting's Name, DisplayName, 
		/// URL, and any Custom Property values specified in the 'CustomProperties' property of the control.
		/// </summary>
		/// <param name="xmlDoc">A valid XmlDocument object the XmlElement will be added to.</param>
		/// <param name="cmsPost">The Posting whose details you wish to capture.</param>
		/// <remarks>The name of the XML node will be 'posting'. Set the Mode property to DevelopmentDebugXML to see the XML generated by the control.</remarks>
		/// <returns>An XmlElement object containing the data on the passed Posting.</returns>
		protected virtual XmlElement AddPostingXmlElement(XmlDocument xmlDoc, Posting cmsPost)
		{
			// Create a channel Element
			XmlElement xmlPost = xmlDoc.CreateElement("", "posting", "");

			#region Current Posting
			// If this post is the currently displayed posting then we need to added a <selected> element
			XmlAttribute xmlSelected = xmlDoc.CreateAttribute("", "selected", "");

			Posting curPost = CmsHttpContext.Current.Posting;
			
			bool blnSelected = false;

			if(curPost != null)
			{

				blnSelected = (curPost.Guid == cmsPost.Guid);

			}

			xmlSelected.Value = (blnSelected) ? bool.TrueString : bool.FalseString;

			xmlPost.Attributes.Append(xmlSelected);
			#endregion

			// Name
			XmlElement xmlElem = xmlDoc.CreateElement("","name", "");

			xmlElem.InnerText = cmsPost.Name;

			xmlPost.AppendChild(xmlElem);

			// Display Name
			xmlElem = xmlDoc.CreateElement("", "displayname", "");

			//Ok, The User can override the Display Name (May be to long etc.) by setting the custom property "navName"
			string strDisplayName;
				
			try
			{
				strDisplayName = cmsPost.CustomProperties["navName"].Value;
			}
			catch
			{
				//If they haven't set navName
				strDisplayName = cmsPost.DisplayName;
			}

			string strPostDisplay = (mblnIgnore_Style) ? strDisplayName : NavigationHelper.GetDisplayName(strDisplayName); 

			if(cmsPost.Name.IndexOf("_") == 1)
			{
					
				strPostDisplay = "_" + strPostDisplay;
				
			}

			xmlElem.InnerText = strPostDisplay;

			xmlPost.AppendChild(xmlElem);
			
			// Url
			xmlElem = xmlDoc.CreateElement("", "postingurl", "");

			xmlElem.InnerText = cmsPost.Url;

			xmlPost.AppendChild(xmlElem);

			//Now we add Custom Properties if there are any
			if(m_strCustomProperties.Length > 0)
			{
				
				xmlPost.AppendChild(AddCustomPropertiesXmlElement(xmlDoc, cmsPost));

			}


			return xmlPost;
		}
		#endregion

		#region Get Custom Properties from a 'ChannelItem'
		/// <summary>
		/// Overridable. This method captures Custom Property values specified in the 'CustomProperties' property, 
		/// and adds the name and value to the returned XmlElement.
		/// </summary>
		/// <param name="xmlDoc">An XmlDocument object into which the XmlElement will be inserted.</param>
		/// <param name="cmsItem">A CMS ChannelItem object (either a Channel or a Posting) whose Custom Properties will be collected.</param>
		/// <returns>XmlElement.</returns>
		/// <remarks>The name of the XML node will be 'custom'.</remarks>
		protected XmlElement AddCustomPropertiesXmlElement(XmlDocument xmlDoc, ChannelItem cmsItem)
		{
			// Create a channel Element
			XmlElement xmlCustom = xmlDoc.CreateElement("", "custom", "");

			// Now loop the Properties
			if(m_arrCustomProperties == null)
			{
				m_arrCustomProperties = m_strCustomProperties.Split(',');
			}

			for(int iLoop=0; iLoop < m_arrCustomProperties.Length; iLoop++)
			{

				//The custom prop may not exist in every instance
				try
				{

					// Name
					XmlElement xmlElem = xmlDoc.CreateElement("",m_arrCustomProperties[iLoop], "");

					xmlElem.InnerXml = cmsItem.CustomProperties[m_arrCustomProperties[iLoop]].Value;

					xmlCustom.AppendChild(xmlElem);

				}
				catch
				{
					//Do nothing
				}


			}
		
			return xmlCustom;

		}


		#endregion

		#region Get Custom Property
		/// <summary>
		/// This method attempts to retrieve the value of a specified Custom Property from a Channel or Posting.
		/// </summary>
		/// <param name="cmsItem">A CMS ChannelItem object (either a Channel or a Posting) whose value you wish to retrieve.</param>
		/// <param name="sProperty">The name of the Custom Property you wish to retrieve.</param>
		/// <returns>The value of the specified Custom Property if found, otherwise an empty string ("")	.</returns>
		protected virtual string GetCustomProperty(ChannelItem cmsItem, string sProperty){return GetCustomProperty(cmsItem, sProperty, "");}

		/// <summary>
		/// This method attempts to retrieve the value of a specified Custom Property from a Channel or Posting.
		/// </summary>
		/// <param name="cmsItem">A CMS ChannelItem object (either a Channel or a Posting) whose value you wish to retrieve.</param>
		/// <param name="sProperty">The name of the Custom Property you wish to retrieve.</param>
		/// <param name="sDefaultValue">An alternative to an empty string, to be returned if the Custom Property cannot be found.</param>
		/// <returns>The value of the specified Custom Property if found, otherwise an empty string ("").</returns>
		protected virtual string GetCustomProperty(ChannelItem cmsItem, string sProperty, string sDefaultValue)
		{
			string strReturn = "";

			try
			{
				strReturn = cmsItem.CustomProperties[sProperty].Value;
			}
			catch
			{
				strReturn = sDefaultValue;
			}

			return strReturn;

		}
		#endregion

		#region Get a break Element
		/// <summary>
		/// Overridable. Returns an XML element node with the name 'break'.
		/// </summary>
		/// <param name="xmlDoc">An XmlDocument object the element will be added to.</param>
		/// <returns>XmlElement.</returns>
		/// <remarks>Can be overriden if an alternative node name is required.</remarks>
		protected virtual XmlElement AddBreakXmlElement(XmlDocument xmlDoc)
		{
			// Create a channel Element
			XmlElement xmlBreak = xmlDoc.CreateElement("", "break", "");

			return xmlBreak;

		}
		#endregion

		 #endregion

		#region Add a single element (from an external source)
		/// <summary>
		/// Overridable. Adds an additional item to the Navigation XML prior to transitition and render. 
		/// This method can be used to add 'fixed' links (internal or external to CMS) to the navigation 
		/// which is displayed, and is used with the 'PreTransform' event.
		/// </summary>
		/// <remarks>Use the AddItem(string sName, string sDisplayName, string sUrl, string sNodeName) 
		/// overload and specify sNodeName as "channel" in order for the added XML node to match 
		/// the existing nodes. This avoids new templates needing to be defined in your XSLT stylesheet. 
		/// (Note: added links may not be rendered if there is no corresponding template in the 
		/// XSLT stylesheet.)
		/// <seealso cref="NavigationControl.PreTransform"/>
		/// </remarks>
		/// <param name="sUrl">Link URL.</param>
		/// <param name="sName">Link text.</param>
		public virtual void AddItem(string sName, string sUrl)
		{
			AddItem(sName, sName, sUrl, "externalitem");
		}
		/// <summary>
		/// Overridable. Adds an additional item to the Navigation XML prior to transitition and render.
		/// This method can be used to add 'fixed' links (internal or external to CMS) to the navigation 
		/// which is displayed, and is used with the 'PreTransform' event.
		/// </summary>
		/// <remarks>Use the AddItem(string sName, string sDisplayName, string sUrl, string sNodeName) 
		/// overload and specify sNodeName as "channel" in order for the added XML node to match 
		/// the existing nodes. This avoids new templates needing to be defined in your XSLT stylesheet. 
		/// (Note: added links may not be rendered if there is no corresponding template in the 
		/// XSLT stylesheet.)
		/// <seealso cref="NavigationControl.PreTransform"/>
		/// </remarks>
		/// <param name="sDisplayName">Content of the DisplayName XML node.</param>
		/// <param name="sUrl">Link URL.</param>
		/// <param name="sName">Link text.</param>
		public virtual void AddItem(string sName, string sDisplayName, string sUrl)
		{
			AddItem(sName, sDisplayName, sUrl, "externalitem");
		}
		/// <summary>
		/// Overridable. Adds an additional item to the Navigation XML prior to transitition and render.
		/// This method can be used to add 'fixed' links (internal or external to CMS) to the navigation 
		/// which is displayed, and is used with the 'PreTransform' event.
		/// </summary>
		/// <remarks>Use the AddItem(string sName, string sDisplayName, string sUrl, string sNodeName) 
		/// overload and specify sNodeName as "channel" in order for the added XML node to match 
		/// the existing nodes. This avoids new templates needing to be defined in your XSLT stylesheet. 
		/// (Note: added links may not be rendered if there is no corresponding template in the 
		/// XSLT stylesheet.)
		/// <seealso cref="NavigationControl.PreTransform"/>
		/// </remarks>
		/// <param name="sDisplayName">Content of the DisplayName XML node.</param>
		/// <param name="sUrl">Link URL.</param>
		/// <param name="sName">Link text.</param>
		/// <param name="sNodeName">Node name to use in generated XML.</param>
		public virtual void AddItem(string sName, string sDisplayName, string sUrl, string sNodeName)
		{
			// Create a channel Element
			XmlElement xmlChildChannel = _xmlData.CreateElement("", sNodeName, "");

			// Name
			XmlElement xmlElem = _xmlData.CreateElement("","name", "");

			xmlElem.InnerXml = sName;

			xmlChildChannel.AppendChild(xmlElem);

			// Display Name
			xmlElem = _xmlData.CreateElement("", "displayname", "");

			xmlElem.InnerXml = sDisplayName;

			xmlChildChannel.AppendChild(xmlElem);
			
			// Url
			xmlElem = _xmlData.CreateElement("", "postingurl", "");

			xmlElem.InnerText = sUrl;

			xmlChildChannel.AppendChild(xmlElem);

			_xmlData.FirstChild.AppendChild(xmlChildChannel);

		}
		#endregion

		#region Get Primary Nav XML
		/// <summary>
		/// Use this call, it will attemt to retrieve Important channels and fall back if required.
		/// </summary>
		/// <returns></returns>
		private XmlDocument getPrimaryNavXML() 
		{

			XmlDocument xmlPrimary;

			xmlPrimary = getPrimaryNavXML(true);

			
			int iNodeCount = xmlPrimary.ChildNodes[0].ChildNodes.Count;
			int iTest = (this.DisplayRoot) ? 1 : 0;

			if(iNodeCount == iTest)
			{
				xmlPrimary = getPrimaryNavXML(false);
			}

		
			return xmlPrimary;
		}

		private XmlDocument getPrimaryNavXML(bool blnImportantChannel)
		{
			int iChannelCount = mintMax_Primary_Channels;

			
			XmlDocument xmlPrimaryNav = new XmlDocument();

			XmlElement xmlRoot = xmlPrimaryNav.CreateElement("", mcstrXML_ROOT_NODE, "");


			//The Primary Navigation DOES obey the Root Channel Rule, it sees this as the main site
			Channel cmsRoot = getRoot();

			// If we are including the Root channel (Effectively a Home link)
			if(this.DisplayRoot)
			{
			
				xmlRoot.AppendChild(AddChannelXmlElement(xmlPrimaryNav, cmsRoot));

			}

			//Now we Loop and retrieve the Channels Directly Below the Root.
			foreach(Channel cmsChannel in cmsRoot.Channels)
			{

				//This is the Default Hidden logic
				bool blnIsHidden = (mblnIgnore_Style) ? false : cmsChannel.Name.IndexOf("_") == 0;
			
				//However if we are In 
				//		Edit Mode
				//		DEBUG Presentation Mode
				//The flag is automatically set to false
				if(CmsHttpContext.Current.Mode != PublishingMode.Published || EnmMode.DevelopmentDebugXML == m_DevelopmentMode)
				{
					blnIsHidden = false;
				}

				if((cmsChannel.IsImportant == blnImportantChannel || this.AllChannels) && (iChannelCount > 0 || this.MaxReturn == 0) && !blnIsHidden && renderChannel(cmsChannel))
				{

					//NOTE: we pass in the xml doc, so the function can create elements
					xmlRoot.AppendChild(AddChannelXmlElement(xmlPrimaryNav, cmsChannel));

					
					iChannelCount--;
				}
			}

			// Append the data to our core document
			xmlPrimaryNav.AppendChild(xmlRoot);

			return xmlPrimaryNav;
		}
		#endregion

		#region Get Highlight
		
		/// <summary>
		/// This mode returns posts 
		/// </summary>
		/// <returns>XmlDocument - containing highlight posts</returns>
		private XmlDocument getHighlight()
		{
			int iPostCount = m_intMaxReturn;

			XmlDocument xmlHighlight = new XmlDocument();

			XmlElement xmlRoot = xmlHighlight.CreateElement("", mcstrXML_ROOT_NODE, "");

			//Ok get our current channel, or the root if we don't have one
			Channel cmsRoot = CmsHttpContext.Current.Channel;

			if(cmsRoot == null)
			{

				//We don't have a channel
				cmsRoot = getRoot();

			}

			//Now we Loop and retrieve the Channels Directly Below the Root.
			foreach(Posting cmsPost in cmsRoot.Postings)
			{

				//This is the Default Hidden logic
				bool blnIsHidden = (mblnIgnore_Style) ? false : cmsPost.Name.IndexOf("_") == 0;
			
				//However if we are In 
				//		Edit Mode
				//		DEBUG Presentation Mode
				//The flag is automatically set to false
				if(CmsHttpContext.Current.Mode != PublishingMode.Published || EnmMode.DevelopmentDebugXML == m_DevelopmentMode)
				{
					//Note we also increment iPostCount (so it shows all important posts
					iPostCount++;
					blnIsHidden = false;
				}

				if((cmsPost.IsImportant == true) && (iPostCount > 0) && renderPost(cmsRoot,cmsPost) && (cmsPost != CmsHttpContext.Current.Posting))
				{

					//NOTE: we pass in the xml doc, so the function can create elements
					xmlRoot.AppendChild(AddPostingXmlElement(xmlHighlight, cmsPost));
					
					iPostCount--;
				}
			}

			// Append the data to our core document
			xmlHighlight.AppendChild(xmlRoot);
			
			return xmlHighlight;
		}
		

		#endregion

		#region Get Breadcrumb

		#region Primary Breadcrumb Call
		/// <summary>
		/// Entry point for retrieving a breadcrumbtrail XML Doc 
		/// </summary>
		/// <returns>XmlDocument contain trail up to Base channel with Post</returns>
		private XmlDocument getBreadCrumb()
		{

			#region Sample XML
			//Sample XML 
			/*
			<nav>
				<channel>
					<name>Main</name>
					<channel>
						<name>News</name>
						<post>
							<name>Me</post>
						</post>
					</channel>
				<channel>
			</nav>
			*/
			#endregion

			//Create our return Document
			XmlDocument xmlReturn = new XmlDocument();

			Posting curPost = this.CurrentPosting;

			XmlNode xmlRoot = xmlReturn.CreateNode(XmlNodeType.Element, mcstrXML_ROOT_NODE, "");

			XmlElement xmlPost = null;

			//Create an element to store our Current Post
			try
			{
				xmlPost =  AddPostingXmlElement(xmlReturn, curPost);

				//Will be passing this into our Channel Recursion Call
				XmlElement xmlResult = getTrail(xmlReturn, xmlPost, curPost.Parent, getRoot().Name);

				//Add the trail to it's root element
				xmlRoot.AppendChild(xmlResult);

			}
			catch
			{
				//Do Nothing
			}

			//And add this into a document
			xmlReturn.AppendChild(xmlRoot);

			return xmlReturn;

		}
		

		#endregion

		#region Recursive Call to Get the trail

		/// <summary>
		/// Called recursively to build to breadcrumb trail into an XML Doc
		/// </summary>
		/// <param name="xmlDoc">xmlDocument used to create elements</param>
		/// <param name="xelmChildren">Child elements to be appended to the current channel</param>
		/// <param name="curChannel">The channel to render</param>
		/// <param name="szBaseChannel">The Base channel name</param>
		/// <returns>XmlElement containing full trail</returns>
		private XmlElement getTrail(XmlDocument xmlDoc, XmlElement xelmChildren, Channel curChannel, string szBaseChannel)
		{

			//get the info for the passed channel
			XmlElement xmlChannel = AddChannelXmlElement(xmlDoc, curChannel);

			//Append children
			if(xelmChildren != null)
			{
				xmlChannel.AppendChild(xelmChildren);
			}

			//Our return Element
			XmlElement xmlRoot;
			
			// Now if this channel isn't the 'Base' and has a parent we call into it
			if( (curChannel.Name != szBaseChannel) && curChannel.Parent != null)
			{
			
				//Pass it up
				xmlRoot = getTrail(xmlDoc, xmlChannel, curChannel.Parent, szBaseChannel);	

			}
			else
			{
				// This is the last last channel so
				// Call channel Info zone last time
				if(m_blnDisplayRoot)
				{

					// Append everything we have to the channel info 
					xmlRoot = xmlChannel;

				}
				else
				{
					//If we are not displaying the root we ignore the channel we just got
					xmlRoot	= xelmChildren;

				}
			}

			// Pass it up the tree
			return xmlRoot;


		}


		#endregion

		#endregion

		#region Get SiteMap
		private XmlDocument getSiteMap()
		{

			XmlDocument xmlSiteMap = new XmlDocument();

			xmlSiteMap.AppendChild(getSiteMapChannels(xmlSiteMap, getRoot(), 0));

			return xmlSiteMap;

		}
		private XmlElement getSiteMapChannels(XmlDocument xmlDoc, Channel cmsContainerChannel, int iCurDepth)
		{

			//This is the default wrapping Node, and is used when past the first Channel set
			string strElementName = "children";
			//This is used so we can differenciate between the top channels (Topics) and the sub channels
			string strChannelNodeName = "channel";
			
			if(iCurDepth == 0)
			{
				strElementName = mcstrXML_ROOT_NODE;
				strChannelNodeName = "topic";
			}

			XmlElement xmlReturn = xmlDoc.CreateElement("", strElementName, "");

			if(PresentationMode == NavigationPresentationMode.AtoZ)
			{	
				cmsContainerChannel.Channels.SortByDisplayName();
			}

			//We loop each channel within the passed container channel
			foreach(Channel cmsChildChannel in cmsContainerChannel.Channels)
			{	
				//We only render Important Channels unless the All Channels is True
				if(cmsChildChannel.IsImportant || m_blnAllChannels)
				{

					//Get the channel info
					XmlElement xmlChannel = AddChannelXmlElement(xmlDoc, cmsChildChannel, strChannelNodeName);

					if(iCurDepth == 0)
					{
						xmlChannel.Attributes.Append(getItemAttribute(xmlDoc));
					}

					//If we haven't exceeded our depth, and there are channels
					if(iCurDepth < m_intMapDepth && cmsChildChannel.Channels.Count > 0)
					{
				
						xmlChannel.AppendChild(getSiteMapChannels(xmlDoc, cmsChildChannel, iCurDepth + 1));
	
					}
				
					//We have that channels details
					xmlReturn.AppendChild(xmlChannel);
				}

			}

			return xmlReturn;

		}
		#endregion	

		#region AtoZ
		
		#region getAZNav()
		/// <summary>
		/// Root call for retrieving the A-Z Xml
		/// </summary>
		/// <returns>XmlDocument</returns>
		private XmlDocument getAZNav()
		{
			
			#region Description
			/*************************************************************************************
								The A to Z function with make use of the Union ability within
								the channels collection interface and will make it possible
								to return an entirely alphabetised collection.
			
								Step 1: This function creates a root node
								Step 2: Function calls the root node
								Step 3: Funtion calls a recursive function to retrieve all subchannels
										to a specified depth.
										
								Step 4: Collection is sorted and passed to an element rendering function.
								
								Step 5: Xml returned to Calling function
			
			
			**************************************************************************************/
			#endregion

			XmlDocument xmlAtoZ = new XmlDocument();

			XmlElement xmlRoot = xmlAtoZ.CreateElement("", mcstrXML_ROOT_NODE, "");

			Channel cmsRootChannel = getRoot();

			ChannelItemCollection cmsChannels = getAZCollection(cmsRootChannel, 0);

			cmsChannels.SortByDisplayName();

			string sCurrentLetter = "A";

			xmlRoot.AppendChild(getAnchorElement(xmlAtoZ, sCurrentLetter, false));

			foreach(ChannelItem cmsItem in cmsChannels)
			{
				string sNodeLetter = cmsItem.DisplayName.Substring(0,1).ToUpper();

				if(sNodeLetter != sCurrentLetter)
				{
					xmlRoot.AppendChild(getAnchorElement(xmlAtoZ, sNodeLetter));

					sCurrentLetter = sNodeLetter;
				}

				if(cmsItem is Channel)
				{
					xmlRoot.AppendChild(AddChannelXmlElement(xmlAtoZ, (Channel)cmsItem));
				}
				else
				{
					xmlRoot.AppendChild(AddPostingXmlElement(xmlAtoZ, (Posting)cmsItem));
				}

			}

			xmlAtoZ.AppendChild(xmlRoot);

			return xmlAtoZ;			

		}
		#endregion

		#region getAnchorElement(XmlDocument xmlDoc, string sAnchor)
		/// <summary>
		/// Used by the AtoZ functions to return Anchor element nodes
		/// </summary>
		/// <param name="xmlDoc">XmlDocument</param>
		/// <param name="sAnchor">string</param>
		/// <returns>XmlElement</returns>
		private XmlElement getAnchorElement(XmlDocument xmlDoc, string sAnchor) {return getAnchorElement(xmlDoc, sAnchor, true);}
		/// <summary>
		/// Used by the AtoZ functions to return Anchor element nodes
		/// </summary>
		/// <param name="xmlDoc">XmlDocument</param>
		/// <param name="sAnchor">string</param>
		/// <param name="backToTop">bool - sets attribute backtotop to true or false</param>
		/// <returns>XmlElement</returns>
		private XmlElement getAnchorElement(XmlDocument xmlDoc, string sAnchor, bool backToTop)
		{
			XmlElement xmlAnchorNode = xmlDoc.CreateElement("", "anchor","");

			XmlAttribute xmlLetter = xmlDoc.CreateAttribute("", "letter", "");
			xmlLetter.Value = sAnchor.ToUpper();
			xmlAnchorNode.Attributes.Append(xmlLetter);

			XmlAttribute xmlBackToTop = xmlDoc.CreateAttribute("", "backtotop", "");
			xmlBackToTop.Value = (backToTop) ? bool.TrueString.ToLower() : bool.FalseString.ToLower();
			xmlAnchorNode.Attributes.Append(xmlBackToTop);

			return xmlAnchorNode;
		}
		#endregion

		#region getAZCollection(Channel cmsCurrentChannel, int iCurDepth)
		/// <summary>
		/// This function returns a Unioned collection of Channels ready for ordering
		/// </summary>
		/// <param name="cmsCurrentChannel">Microsoft.ContentManagement.Publishing.Channel</param>
		/// <param name="iCurDepth">int</param>
		/// <returns>Microsoft.ContentManagement.Publishing.ChannelCollection</returns>
		private ChannelItemCollection getAZCollection(Channel cmsCurrentChannel, int iCurDepth)
		{
			ChannelItemCollection cmsChannels = cmsCurrentChannel.Channels;

			cmsChannels = cmsChannels.Union(cmsCurrentChannel.Postings);

			// Have we exceeded our maximum depth, NOTE 0 = Unlimited
			if(iCurDepth < MaxReturn || MaxReturn == 0)
			{
				// Now we loop through the current available channel object within the Passed Channel
				foreach(Channel cmsChan in cmsCurrentChannel.Channels)
				{
					// If the channel has postings or we are in edit mode
					if(cmsChan.Postings.Count > 0 || (WebAuthorContext.Current.Mode != WebAuthorContextMode.PresentationPublished || WebAuthorContext.Current.Mode != WebAuthorContextMode.PresentationUnpublishedPreview))
					{
						// Retrieve this channels Channel collection (recursive so children will be returned where appropriate)
						ChannelItemCollection cmsUnion = getAZCollection(cmsChan, iCurDepth+1);

						// If we were returned a collection (The channel may have had no children) We Union it
						if(cmsUnion != null)
						{
							cmsChannels = cmsChannels.Union(cmsUnion);
						}

					}

				}

			}

			return cmsChannels;

		}
		#endregion

		#endregion

		#region isEditMode()
		/// <summary>
		/// Checks to see if the current posting is in edit mode (Page Create or Switched to edit mode)
		/// </summary>
		/// <returns>bool</returns>
		/// <remarks>On error defaults to false</remarks>
		private bool isEditMode()
		{
			bool blnReturn = false;
			
			try
			{
				WebAuthorContext cmsContext = WebAuthorContext.Current;

				// Returns false if the mode is NOT Published / Preview
				blnReturn = (cmsContext.Mode != WebAuthorContextMode.PresentationPublished && cmsContext.Mode != WebAuthorContextMode.AuthoringPreview && cmsContext.Mode != WebAuthorContextMode.PresentationUnpublishedPreview);

			}
			catch
			{
				blnReturn = false;
			}

			return blnReturn;

		}
		#endregion

		#region getItemAttribute(XmlDocument xmlDoc)
		/// <summary>
		/// Used by sitemap, to give an item number to topics
		/// </summary>
		/// <param name="xmlDoc">XmlDocument used to create attribute</param>
		/// <returns>XmlAttribute</returns>
		private XmlAttribute getItemAttribute(XmlDocument xmlDoc)
		{
			XmlAttribute xmlItem = xmlDoc.CreateAttribute("","item_number","");

			xmlItem.Value = (++m_item).ToString();

			return xmlItem;
		}
		#endregion

		#region Save XML
		/// <summary>
		/// Saves the given xmlDocument to the file system, in the event of an error it will logError 
		/// </summary>
		/// <param name="xmlDoc">XmlDocument to save</param>
		private void saveXML(XmlDocument xmlDoc)
		{
			string sFileName = m_csPresMode.ToString() + ".xml";

			if(m_strSaveXmlPath.Length > 0)
			{
				
				sFileName = m_strSaveXmlPath;

			}

			try
			{
				xmlDoc.Save(sFileName);
			}
			catch(Exception e)
			{
				logError(mclngNAV_ERROR_UNKOWN, "saveXML()", "Unable to save XML file. " + e.InnerException);	
			}

		}
		#endregion

				

		#region Internal Style Sheet Operations

		#region Default Internal Styles File Names

		private const string mcstrPRIMARY_XSL_DEFAULT = "primary.xslt";

		private const string mcstrSECONDARY_XSL_DEFAULT = "secondary.xslt";

		private const string mcstrCOMBO_XSL_DEFAULT = "combo.xslt";

		private const string mcstrBREAD_XSL_DEFAULT = "bread.xslt";

		private const string mcstrHIGHLIGHT_XSL_DEFAULT = "highlight.xslt";

		private const string mcstrSITE_MAP_XSL_DEFAULT = "sitemap.xslt";

		private const string mcstrA_TO_Z_XSL_DEFAULT = "atoz.xslt";

		#endregion

		#region setDefaultXSL()
		/// <summary>
		/// Based on the presentation mode setting, it sets the default internal XSL
		/// </summary>
		private void setDefaultXSL()
		{

			mstrDEFAULT_XSL = accessDefaultXSL(m_csPresMode);

		}
		#endregion

		#region Default Style Sheet Aquisition
		/// <summary>
		/// Returns an internal XSLT resource as string.
		/// </summary>
		/// <param name="cs_Mode">Presentation Mode</param>
		/// <returns>string</returns>
		/// <remarks>Used primarily for accessing the default internal style sheet</remarks>
		private string accessDefaultXSL(NavigationPresentationMode cs_Mode)
		{
			
			string strFile;

			switch(cs_Mode)
			{
				case NavigationPresentationMode.Primary:
				{
					
					strFile = mcstrPRIMARY_XSL_DEFAULT;
					break;

				}
				case NavigationPresentationMode.ComboBox:
				{
					
					strFile = mcstrCOMBO_XSL_DEFAULT;
					break;

				}
				case NavigationPresentationMode.BreadCrumb:
				{
					
					strFile = mcstrBREAD_XSL_DEFAULT;
					break;

				}
				case NavigationPresentationMode.Highlight:
				{
					
					strFile = mcstrHIGHLIGHT_XSL_DEFAULT;
					break;

				}
				case NavigationPresentationMode.AtoZ:
				{

					strFile = mcstrA_TO_Z_XSL_DEFAULT;
					break;

				}
				case NavigationPresentationMode.SiteMap:
				{
					
					strFile = mcstrSITE_MAP_XSL_DEFAULT;
					break;

				}
				default: //Note defaults to all secondary styles
				{
					
					strFile = mcstrSECONDARY_XSL_DEFAULT;
					break;

				}

			}

			return accessDefaultXSL(strFile);

		}
		#endregion

		#region Style Sheet Aquisition
		/// <summary>
		/// Returns a Style sheet resource from 'MSIBPlusPack.Web.UI.Navigation.' using the passed file name 
		/// </summary>
		/// <param name="strFile">string</param>
		/// <returns>string</returns>
		private string accessDefaultXSL(string strFile)
		{

			Assembly assembly = Assembly.GetExecutingAssembly();
			
			string strReturn;

			try
			{
				Stream memFile = assembly.GetManifestResourceStream("MSIBPlusPack.Web.UI.WebControls.Navigation.Style." + strFile);
		
				StreamReader stmRead = new StreamReader(memFile);

				strReturn = stmRead.ReadToEnd();

				stmRead.Close();
				memFile.Close();

			}
			catch(Exception e)
			{
				strReturn = "";

				logError(mclngXSL_ERROR_BASE, "accessDefaultXSL(string strFile)", "File name = " + strFile + " " + e.InnerException);

			}


			return strReturn;

		}
		#endregion

		#endregion

		#region Research [All in development functions will reside here]


		#endregion

		#endregion

		/// <summary>
		/// Override of the Render method of the WebControl class. Responsible for rendering the 
		/// output of this control.
		/// </summary>
		/// <param name="output">The HtmlTextWriter to write to.</param>
		protected override void Render(HtmlTextWriter output)
		{
			EnsureChildControls();
			output.Write(GetNavigationXmlAsString());
		}
	
		/// <summary>
		/// Cleans up resources associated with the control.
		/// </summary>
		public override void Dispose()
		{
			if (license != null)
			{
				license.Dispose();
				license = null;
			}
			base.Dispose ();
		}

		internal bool IsMsib21ChannelPath(string strChannelPath)
		{
			return (strChannelPath.IndexOf(mcstrMSIB21_CHANNEL_IDENTIFIER) > -1) ? 
				true : false;
		}

		internal string ReplaceWithLocaleID(string strChannelPath)
		{
			string strCurrentLocaleID = System.Threading.Thread.CurrentThread.CurrentCulture.Name;
			return (strChannelPath.ToLower().Replace(mcstrMSIB21_CHANNEL_IDENTIFIER, strCurrentLocaleID));
		}

	}
}

