using System;
using System.ComponentModel;
using System.Net;
using System.Web;
using System.Web.SessionState;
using System.Web.UI;
using MSIBPlusPack.ContentManagement.Publishing.Placeholders.MapPointService;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// This class exposes the business logic for MapPoint search operations for the MSIB Plus Pack Nearest component.
	/// </summary>
	/// <remarks>
	/// <para><see cref="NearestPlaceholder"/> exposes an instance of this class via its <see cref="NearestPlaceholder.MapPointProvider"/> property. This class cannot be instantiated directly by other assemblies.</para>
	/// <para>
	/// Nearest search parameters can be independantly specified programmatically or by the author or subscriber (site user). Search parameter state is persisted to an underlying XmlPlaceholderDefinition to which NearestPlaceholder must be bound.
	/// Search parameters include <see cref="Keywords"/> which is used to determine the location,
	/// the type of item to search for (<see cref="EntityTypeValue"/>)
	/// and the <see cref="Range"/>Range (radius) in miles to search for occurances of the item type around the specified location
	/// </para>
	/// <para>
	/// A list of available Entity Types can be accessed via the <see cref="MapEntityTypes"/> member.
	/// Which exposes an array of <see cref="NearestMapEntityType"/> objects.
	/// A NearestMapEntityType instance's
	/// <see cref="NearestMapEntityType.Value"/> property is used to specify the Entity Type search parameter.
	/// </para>
	/// <para>
	/// Search results can be accessed via the <see cref="NearestResults"/> member which exposes an array of <see cref="NearestResult"/> objects.
	/// </para>
	/// <para>
	/// This class requires session state to persist the rendered map image's Mime Type and binary data.
	/// It requires View State to persist Search parameters to allow their values to be interrogated early during postback (i.e. before placeholder content has been reloaded).
	/// </para>
	/// <para>The search parameter properties exposed by this control are serialised to the bound XmlPlaceholder. This includes <see cref="MapPointDataSource"/>, <see cref="PointsOfInterestDataSource"/>, <see cref="EntityTypeValue"/>, <see cref="Keywords"/> and<see cref="Range"/>.</para>
	/// </remarks>
	public class NearestMapPointProvider : INearestMapProvider
	{
		#region Static fields
		private static NearestMapPointGlobal mapPointGlobal;
		#endregion

		#region State management fields
		private NearestMapEntityType[] mapEntityTypes;
		private string version;
		private string entityTypeValue;
		private bool entityTypeValueUpdated;
		private string keywords;
		private bool keywordsUpdated;
		private int range;
		private bool rangeUpdated;
		private bool searchActive;
		private bool searchActiveUpdated;
		private bool mapRendered;
		private bool mapRenderedUpdated;
		private string mapPointDataSource;
		private bool mapPointDataSourceUpdated;
		private string pointsOfInterestDataSource;
		private bool pointsOfInterestDataSourceUpdated;

		private NearestResult[] nearestResults;

		private StateBag viewState;
		#endregion

		#region Static constructors
		static NearestMapPointProvider()
		{
			mapPointGlobal = new NearestMapPointGlobal();
		}
		#endregion

		#region Constructors
		/// <summary>
		/// Defaults the MapPoint data source to <b>MapPoint.EU</b>.
		/// Defaults the Points Of Interest data source to <b>NavTech.EU</b>.
		/// Defaults the range of searches to 10 miles.
		/// </summary>
		internal NearestMapPointProvider()
		{
			mapPointDataSource = "MapPoint.EU";
			pointsOfInterestDataSource = "NavTech.EU";
			version = "";
			entityTypeValue = "";
			range = 10;
			searchActive = false;
			mapRendered = false;
			mapEntityTypes = null;
			nearestResults = new NearestResult[0];

			pointsOfInterestDataSourceUpdated = false;
			mapPointDataSourceUpdated = false;
			entityTypeValueUpdated = false;
			keywordsUpdated = false;
			rangeUpdated = false;
			searchActiveUpdated = false;
			mapRenderedUpdated = false;

		}
		#endregion

		#region View state parsing members
		internal void LoadViewState(StateBag ViewState)
		{
			viewState = ViewState;

			// If values have been programmatically updated before we had a view state, store them now:
			if (pointsOfInterestDataSourceUpdated)
				PointsOfInterestDataSource = pointsOfInterestDataSource;
			if (mapPointDataSourceUpdated)
				MapPointDataSource = mapPointDataSource;;
			if (entityTypeValueUpdated)
				EntityTypeValue = entityTypeValue;
			if (keywordsUpdated)
				Keywords = keywords;
			if (rangeUpdated)
				Range = range;
			if (searchActiveUpdated)
				SearchActivePrivate = searchActive;
			if (mapRenderedUpdated)
				MapRenderedPrivate = mapRendered;

			if (null != ViewState["MapPointProvider.EntityTypeValue"])
				entityTypeValue = (string)ViewState["MapPointProvider.EntityTypeValue"];

			if (null != ViewState["MapPointProvider.Keywords"])
				keywords = (string)ViewState["MapPointProvider.Keywords"];

			if (null != ViewState["MapPointProvider.Range"])
				range = (int)ViewState["MapPointProvider.Range"];

			if (null != ViewState["MapPointProvider.SearchActive"])
				searchActive = (bool)ViewState["MapPointProvider.SearchActive"];

			if (null != ViewState["MapPointProvider.MapRendered"])
				mapRendered = (bool)ViewState["MapPointProvider.MapRendered"];
		}
		#endregion

		#region Properties
		/// <summary>
		/// This property exposes proxy configuration for communication with all MapPoint web services.
		/// </summary>
		/// <remarks>
		/// See
		/// System.Net.IWebProxy
		/// in MSDN and other publications for details about using this .Net Framework class.
		/// </remarks>
		/// <example>
		/// The example below shows how to provide proxy server authentication by specifying the credentials:
		/// <code>myNearestPlaceholder.MapPointProvider.Proxy.Credentials = CredentialsCache.DefaultCredentials;</code>
		/// </example>
		public IWebProxy WebServiceProxy
		{
			get
			{
				return mapPointGlobal.Proxy;
			}
		}

		/// <summary>
		/// <para>
		/// This property exposes the name of the points of interest data source.
		/// It defaults to <b>NavTech.EU</b>, but can be modified to any custom data source.
		/// </para>
		/// <para>See your MapPoint Web Service documentation for further information about uploading your custom points of interest data.</para>
		/// <para>MapPoint Web Services will throw an informative error if the specified data source does not exist.</para>
		/// </summary>
		/// <example>
		/// The following example sets the points of interest data source to NavTech US:
		/// <code>
		/// myNearestPlaceholder.MapPointProvider.PointsOfInterestDataSource = "NavTech.US";
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue("NavTech.EU"),
				Description("MapPoint Web Services Points Of Interest Data Source")
		]
		public string PointsOfInterestDataSource
		{
			get { return pointsOfInterestDataSource; }
			set
			{
				pointsOfInterestDataSource = value;
				if (null == viewState)
					pointsOfInterestDataSourceUpdated = true;
				else
					viewState["MapPointProvider.PointsOfInterestDataSource"] = value;
			}
		}

		/// <summary>
		/// <para>
		/// This property exposes the name of the map provider data source.
		/// It defaults to <b>MapPoint.EU</b>, but can be modified to e.g. <b>MapPoint.US</b> or <b>MapPoint.Canada</b>.
		/// </para>
		/// <para>
		/// The map provider data source is used to determine a location from the specified keywords and to render map images.
		/// </para>
		/// </summary>
		/// <example>
		/// The following example sets the map provider data source to MapPoint US:
		/// <code>
		/// myNearestPlaceholder.MapPointProvider.MapPointDataSource = "MapPoint.US";
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue("MapPoint.EU"),
				Description("MapPoint Web Services Data Source")
		]
		public string MapPointDataSource
		{
			get { return mapPointDataSource; }
			set
			{
				mapPointDataSource = value;
				if (null == viewState)
					mapPointDataSourceUpdated = true;
				else
					viewState["MapPointProvider.MapPointDataSource"] = value;
			}
		}

		/// <summary>
		/// <para>
		/// This property exposes the Value of the type of point of interest (entity type) that will be searched for.
		/// E.g. to search for Hotels in the NavTech.EU data source this should be set to <b>SIC7011</b>.
		/// </para>
		/// <para>
		/// The <see cref="MapEntityTypes"/> member returns a list of entity types including the
		/// <see cref="NearestMapEntityType.Name"/> and <see cref="NearestMapEntityType.Value"/> of each type.
		/// This list can be parsed programmatically to populate a drop-down list to allow the author or subscriber (site user) to specify an entity type.
		/// This property should be populated with the <see cref="NearestMapEntityType.Value"/> of the Entity Type to search for.
		/// </para>
		/// <para>This property is null until specified.</para>
		/// </summary>
		/// <example>
		/// The example below depicts setting the current selected value in a entity type drop down list when the placeholder content is loaded for authoring:
		/// <code>
		/// protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		/// {
		///		// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member first to initialize MapPointProvider
		///		base.LoadPlaceholderContentForAuthoring(e);
		/// 
		///		if (null != MapPointProvider.EntityTypeValue)
		///			ddlEntityType.SelectedValue = MapPointProvider.EntityTypeValue;
		/// }
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue(""),
				Description("The Value of the Entity Type to search for")
		]
		public string EntityTypeValue
		{
			get { return entityTypeValue; }
			set
			{
				entityTypeValue = value;
				if (null == viewState)
					entityTypeValueUpdated = true;
				else
					viewState["MapPointProvider.EntityTypeValue"] = value;
			}
		}

		/// <summary>
		/// <para>This property exposes the keywords that will be used to determine the location to find points of interest around.</para>
		/// <para>
		/// MapPoint will attempt to find the most likey location given to any keyword, this includes synonyms and incorrect spellings.
		/// It is advisable to set this property to a postcode / zip code.
		/// </para>
		/// <para>This property is null until specified.</para>
		/// </summary>
		/// <example>
		/// The following example depicts a Search button click event handler in an placeholder inherited from Nearest.
		/// It updates the Keywords and <see cref="Range"/> properties from a text box and drop down list.
		/// Then performs the search operation by executing <see cref="FindNearest"/>.
		/// <code>
		/// public void btnSearch_Click(object sender, System.EventArgs e)
		/// {
		/// 	MapPointProvider.Keywords = tbKeywords.Text;
		///		MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);	
		/// 	MapPointProvider.FindNearest();
		/// }
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue(""),
				Description("Keywords search parameter - postcode?")
		]
		public string Keywords
		{
			get { return keywords; }
			set
			{
				keywords = value;
				if (null == viewState)
					keywordsUpdated = true;
				else
					viewState["MapPointProvider.Keywords"] = value;
			}
		}

		/// <summary>
		/// <para>Specifies the radius (in miles) around the location that will be searched for the occurances of the specified entity type.</para>
		/// <para>MapPoint will allow a value between 1-250 for this property, however most points of interest data sources will only allow 1-80.</para>
		/// </summary>
		/// <example>
		/// The following example depicts a Search button click event handler in an placeholder inherited from Nearest.
		/// It updates the Range and <see cref="Keywords"/> properties from a text box and drop down list.
		/// Then performs the search operation by executing <see cref="FindNearest"/>.
		/// <code>
		/// public void btnSearch_Click(object sender, System.EventArgs e)
		/// {
		/// 	MapPointProvider.Keywords = tbKeywords.Text;
		///		MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);	
		/// 	MapPointProvider.FindNearest();
		/// }
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue("10"),
				Description("nearest search range from location")
		]
		public int Range
		{
			get { return range; }
			set
			{
				range = value;
				if (null == viewState)
					rangeUpdated = true;
				else
					viewState["MapPointProvider.Range"] = value;
			}
		}

		/// <summary>
		/// This read-only boolean search result property denotes whether search results have been obtained.
		/// This property is updated when a search operation is performed by <see cref="FindNearest"/>.
		/// </summary>
		/// <example>
		/// The example below performs a nearest search.
		/// It sets the visibility of a results panel (containing a list of the results obtained from <see cref="NearestResults"/>).
		/// It sets the visibility of an HTML image containing the rendered map via the <see cref="MapRendered"/> property.
		/// <code>
		/// MapPointProvider.FindNearest();
		/// pnlResults.Visible = MapPointProvider.SearchActive;
		/// imgResultsMap.Visible = MapPointProvider.MapRendered;
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue("false"),
				Description("Search Active")
		]
		public bool SearchActive
		{
			get { return searchActive; }
		}

		private bool SearchActivePrivate
		{
			set
			{
				searchActive = value;
				if (null == viewState)
					searchActiveUpdated = true;
				else
					viewState["MapPointProvider.SearchActive"] = value;
			}
		}

		/// <summary>
		/// This read-only boolean search result property denotes whether a search result map has been rendered.
		/// This property is updated when a search operation is performed by <see cref="FindNearest"/>.
		/// </summary>
		/// <remarks>
		/// This property can be <span class="code">false</span> if <see cref="SearchActive"/> is
		/// <span class="code">true</span>.
		/// It will always be <span class="code">false</span> if <see cref="SearchActive"/> is
		/// <span class="code">false</span>.
		/// </remarks>
		/// <example>
		/// The example below performs a nearest search.
		/// It sets the visibility of an HTML image containing the rendered map.
		/// It sets the visibility of a results panel via the <see cref="SearchActive"/> property.
		/// <code>
		/// MapPointProvider.FindNearest();
		/// imgResultsMap.Visible = MapPointProvider.MapRendered;
		/// pnlResults.Visible = MapPointProvider.SearchActive;
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue("false"),
				Description("Map Rendered")
		]
		public bool MapRendered
		{
			get { return mapRendered; }
		}

		private bool MapRenderedPrivate
		{
			set
			{
				mapRendered = value;
				if (null == viewState)
					mapRenderedUpdated = true;
				else
					viewState["MapPointProvider.MapRendered"] = value;
			}
		}
	
		/// <summary>
		/// This read-only property exposes version information for the MapPoint and Points of Interest data sources.
		/// It is initialized by MapPoint Web Service validation as performed by <see cref="NearestPlaceholder.CreateAuthoringChildControls"/> and <see cref="NearestPlaceholder.CreatePresentationChildControls"/> it will not be defined before such time.
		/// </summary>
		public string Version
		{
			get { return version; }
		}
		
		/// <summary>
		/// Returns a array of <see cref="NearestMapEntityType.Name"/>-<see cref="NearestMapEntityType.Value"/> pairs of Entity Types.
		/// I.e. types of points of interest that can be searched for - as denoted by the specified points of interest data source.
		/// </summary>
		/// <example>
		/// The example below depicts a placeholder inheriting the Nearest component overriding the
		/// <see cref="NearestPlaceholder.CreateAuthoringChildControls"/> to populate a drop down with a list of
		/// entities:
		/// <code>
		/// protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		/// {
		/// 	// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
		/// 	base.CreateAuthoringChildControls(authoringContainer);
		/// 	// Create entity type drop down control
		/// 	ddlEntityType = new DropDownList();
		/// 	ddlEntityType.ID = "ddlEntityType";
		/// 	foreach (NearestMapEntityType mapEntityType in MapPointProvider.MapEntityTypes)
		/// 	ddlEntityType.Items.Add(new ListItem(mapEntityType.Name, mapEntityType.Value));
		/// </code>
		/// </example>
		public NearestMapEntityType[] MapEntityTypes
		{
			get
			{
				if (null == mapEntityTypes)
					GetEntityTypes();

				return mapEntityTypes;
			}
		}

		private void GetEntityTypes()
		{
			EntityType[] et = mapPointGlobal.CommonService.GetEntityTypes(pointsOfInterestDataSource);
			mapEntityTypes = new NearestMapEntityType[et.Length];

			int index = 0;
			foreach (EntityType etype in et)
			{
				mapEntityTypes[index++] = new NearestMapEntityType(etype.DisplayName, etype.Name);
			}
		}


		/// <summary>
		/// An array of <see cref="NearestResult"/> objects detailing all points of intereset returned by the search.
		/// The list is in order of the score associated with the result by MapPoint.
		/// </summary>
		/// <example>
		/// The example below depicts a method that adds an HTML bulletted list of results to the Panel control pnlSearchResults:
		/// <code>
		/// private void PopulateNearestResults()
		/// {
		/// 	pnlSearchResults.Controls.Clear();
		/// 
		/// 	if (0 == MapPointProvider.NearestResults.Length)
		///		{
		/// 		pnlSearchResults.Controls.Add(new LiteralControl("No items were found near the specified location."));
		/// 		return;
		///		}
		/// 
		/// 	pnlSearchResults.Controls.Add(new LiteralControl("&lt;ul&gt;"));
		/// 	foreach(NearestResult nearestResult in MapPointProvider.NearestResults)
		///		{
		/// 		StringBuilder stringBuilder = new StringBuilder();
		/// 		stringBuilder.AppendFormat("&lt;li&gt; {0}, ", nearestResult.Name);
		/// 		if (nearestResult.AddressLine != null)
		/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.AddressLine);
		/// 		if (nearestResult.PrimaryCity != null)
		/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.PrimaryCity);
		/// 		if (nearestResult.SecondaryCity != null)
		/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.SecondaryCity);
		/// 		stringBuilder.AppendFormat("{0}&lt;/li&gt;", nearestResult.PostalCode);
		/// 		pnlSearchResults.Controls.Add(new LiteralControl(stringBuilder.ToString()));
		///		}
		/// 	pnlSearchResults.Controls.Add(new LiteralControl("&lt;/ul&gt;"));
		/// }
		/// </code>
		/// </example>
		public NearestResult[] NearestResults
		{
			get
			{
				if ((null != viewState) && (0 == nearestResults.Length) && (null != viewState["MapPointProvider.NearestResults"]))
					nearestResults = (NearestResult[])viewState["MapPointProvider.NearestResults"];

				return nearestResults;
			}
		}
		#endregion

		#region Map Connectivity Validation
		/// <summary>
		/// validates connectivity to the MapPoint web services data sources (see map ds prop + poi ds prop)
		/// </summary>
		internal void ValidateConnectivity()
		{
			// N.B. MapPoint.Net will throw an informative error if the specified MapPoint or Points Of Interest data source does not exist.

			string mapPointDataSourceVersion, pointsOfInterestDataSourceVersion;

			ValidateDataSource("MapPoint", mapPointDataSource, "Find Places", DataSourceCapability.CanFindPlaces, out mapPointDataSourceVersion);
			ValidateDataSource("Points Of Interest", pointsOfInterestDataSource, "Find Nearby", DataSourceCapability.CanFindNearby, out pointsOfInterestDataSourceVersion);
			
			version = String.Format("{0}.\n{1}. ", mapPointDataSourceVersion, pointsOfInterestDataSourceVersion);
		}

		private bool ValidateDataSource(string DataSourceDescription, string DataSourceName, string RequiredCapabilityDescription, DataSourceCapability RequiredCapability, out string DataSourceVersion)
		{
			bool DataSourceCapable = false;
			DataSourceVersion = "";

			// Determine whether the MapPoint Data Source specified support Find Places functionality:
			DataSource[]ds = null;
			try
			{
				ds = mapPointGlobal.CommonService.GetDataSourceInfo(new string[]{DataSourceName});
			}
			catch (WebException e)
			{
				throw(new Exception(String.Format("The {0} data source '{1}' could not be contacted. {2}",
					DataSourceDescription, DataSourceName, e.Message)));
			}
			if ((null != ds) && (ds.Length > 0))
			{
				if ((ds[0].Capability & RequiredCapability) == RequiredCapability)
					DataSourceCapable = true;

				DataSourceVersion = String.Format("{0} ({1}): {2}", ds[0].Name, ds[0].Version, ds[0].Description);
			}

			if (!DataSourceCapable)
			{
				throw(new Exception(String.Format("The {0} data source '{1}' does not support {2} search capability required by the MSIB Plus Pack Nearest component.",
					DataSourceDescription, DataSourceName, RequiredCapabilityDescription)));
			}

			return true;
		}
		#endregion

		#region Nearest search member
		/// <summary>
		/// <para>This member performs the Nearest search with the current search parameters.</para>
		/// <para>
		/// It integrates with MapPoint finder to determine the required location from the provided <see cref="Keywords"/>.
		/// If a location is determined it uses MapPoint finder to search for occurances of <see cref="EntityTypeValue"/> within a radius of <see cref="Range"/>.
		/// </para>
		/// </summary>
		/// <remarks>
		/// <para>It sets the values of search result properties <see cref="SearchActive"/> and <see cref="MapRendered"/> accordingly, and populates <see cref="NearestResults"/>.</para>
		/// <para>
		/// Finally it uses the MapPoint renderer to generate an image representation of the map, marking the location and points of interest.
		/// Map image data is stored in session and accessed via the static members <see cref="MapImageMimeType"/> and <see cref="MapImageBits"/>.
		/// </para>
		/// </remarks>
		/// <returns>
		/// Boolean indicating whether the search was successful.
		/// i.e. whether the location specified by the keywords was found, not whether any points of interest were found.
		/// </returns>
		/// <example>
		/// The following example depicts a Search button click event handler in an placeholder inherited from Nearest.
		/// It updates the <see cref="Range"/> and <see cref="Keywords"/> properties from a text box and drop down list.
		/// Then performs the search operation by executing <see cref="FindNearest"/>.
		/// It sets the visibility of a results panel via the <see cref="SearchActive"/> property.
		/// It sets the visibility of an HTML image containing the rendered map via the <see cref="MapRendered"/> property.
		/// <code>
		/// public void btnSearch_Click(object sender, System.EventArgs e)
		/// {
		/// 	MapPointProvider.Keywords = tbKeywords.Text;
		///		MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);	
		///		MapPointProvider.FindNearest();
		///		pnlResults.Visible = MapPointProvider.SearchActive;
		///		imgResultsMap.Visible = MapPointProvider.MapRendered;
		/// }
		/// </code>
		/// </example>
		public bool FindNearest()
		{
			bool returnValue = true;
			bool mapRendered = false;

			nearestResults = new NearestResult[0];

			if ((null == entityTypeValue) || (null == keywords))
				returnValue = false;

			FindResults myFindResults = null;
			if (returnValue)
			{
				//Set up the Find options
				FindOptions myFindOptions = new FindOptions();
				myFindOptions.ThresholdScore = 0;

				//Set up the Find Specification
				FindSpecification fp = new FindSpecification();
				fp.DataSourceName = mapPointDataSource;
				fp.InputPlace = keywords;
				fp.Options = myFindOptions;

				myFindResults = mapPointGlobal.FindService.Find(fp);

				if (0 == myFindResults.NumberFound)
					returnValue = false;
			}

			if (returnValue)
			{
				//Set up the FindNearby options object
				FindOptions myFindNearbyOptions = new FindOptions();
				myFindNearbyOptions.Range = new FindRange();
				myFindNearbyOptions.Range.Count = range;

				//Set up the specification object
				FindNearbySpecification findNearbySpec = new FindNearbySpecification();
				findNearbySpec.Options = myFindNearbyOptions;
				findNearbySpec.DataSourceName = pointsOfInterestDataSource;
				findNearbySpec.Distance = Convert.ToDouble(range);
				findNearbySpec.LatLong = myFindResults.Results[0].FoundLocation.LatLong; 
				findNearbySpec.Filter = new FindFilter();
				findNearbySpec.Filter.EntityTypeName = entityTypeValue;

				//Create a FindResults object to store the results of the FindNearby request
				FindResults myFindNearbyResults = mapPointGlobal.FindService.FindNearby(findNearbySpec);
         
        
				//Now use the found address and the FindNearby results to render a map.
				//Output the results of the FindNearby search as well.

				//Create an array of Location and Pushpin objects, setting upperbound via the upperbound of the FindNearby results,
				//adding one array element for the original address.  Draw the map to encompass all these points.
				Location[] myLocations = new Location[myFindNearbyResults.Results.Length + 1];
				Pushpin[] myPushPins = new Pushpin[myFindNearbyResults.Results.Length + 1];

				//Add FindNearby results to the list box and fill the pushpin and location arrays
				int i = 0;
				nearestResults = new NearestResult[myFindNearbyResults.Results.Length];
				foreach (FindResult findResult in myFindNearbyResults.Results)
				{
					myLocations[i] = new Location();
					myLocations[i].LatLong = findResult.FoundLocation.LatLong;
					myPushPins[i] = new Pushpin();
					myPushPins[i].PinID = "pin" + i.ToString();
					//KEVNOTE
					myPushPins[i].IconName = String.Format("RedCircle{0}", i + 1);
					myPushPins[i].IconDataSource = "MapPoint.Icons";
					myPushPins[i].LatLong = findResult.FoundLocation.LatLong;

					//Add the name and lat/long values to the find nearby results list
					nearestResults[i] = new NearestResult(++i, findResult.Score, findResult.FoundLocation.Entity.DisplayName,
						findResult.FoundLocation.Address.AddressLine, findResult.FoundLocation.Address.PrimaryCity, findResult.FoundLocation.Address.SecondaryCity, 
						findResult.FoundLocation.Address.PostalCode, findResult.FoundLocation.Address.CountryRegion,
						findResult.FoundLocation.LatLong.Longitude, findResult.FoundLocation.LatLong.Latitude);
				}

				try 
				{           
					// We repeat the entire myLocations/myPushPins action one more time to get a pin in for the original address.
					myLocations[i] = myFindResults.Results[0].FoundLocation;
					myPushPins[i] = new Pushpin();
					myPushPins[i].PinID = "pin" + i.ToString();
					//					myPushPins[i].Label = textAddressLine.Text;
					myPushPins[i].IconName = "0";
					myPushPins[i].IconDataSource = "MapPoint.Icons";
					myPushPins[i].LatLong = myFindResults.Results[0].FoundLocation.LatLong;
				} 
				catch {}

				//Set up the specification object using the location array
				ViewByBoundingLocations[] myViews = new ViewByBoundingLocations[1];
				myViews[0] = new ViewByBoundingLocations();
				myViews[0].Locations = myLocations;

				MapSpecification mapSpec = new MapSpecification();
				mapSpec.Pushpins = myPushPins;
				mapSpec.Views = myViews;
				mapSpec.DataSourceName = mapPointDataSource;
				mapSpec.Options = new MapOptions();
				mapSpec.Options.Format = new ImageFormat();
				mapSpec.Options.Format.Width = 500;
				mapSpec.Options.Format.Height = 500;


				//Store the specification object for use when panning and zooming
				//and then render the map
				HttpContext.Current.Session["myMapSpec"] = mapSpec;

				try 
				{
					RenderMap(0, 0, 0);
					mapRendered = true;
				} 
				catch {}
			}

			SearchActivePrivate = returnValue;
			MapRenderedPrivate = mapRendered;
			viewState["MapPointProvider.NearestResults"] = nearestResults;

			return returnValue;
		}
		#endregion

		#region Map navigation members
		/// <summary>
		/// Zooms in on the map - rerendering it with an increased magnification.
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move In button event handler increasing the map magnification:
		/// <code>
		/// public void btnMoveIn_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveIn();
		///	}
		/// </code>
		/// </example>
		public void MoveIn()
		{
			RenderMap(0,0,-.25);
		}
      
		/// <summary>
		/// Zooms out on the map - rerendering it with an decreased magnification.
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move Out button event handler decreasing the map magnification:
		/// <code>
		/// public void btnMoveOut_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveOut();
		///	}
		/// </code>
		/// </example>
		public void MoveOut()
		{
			RenderMap(0,0,.25);
		}
      
		/// <summary>
		/// Pans the map westwards
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move West button event handler panning the map westwards:
		/// <code>
		/// public void btnMoveWest_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveWest();
		///	}
		/// </code>
		/// </example>
		public void MoveWest()
		{
			RenderMap(-.25,0,0);
		}
      
		/// <summary>
		/// Pans the map eastwards
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move In button event handler panning the map eastwards:
		/// <code>
		/// public void btnMoveEast_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveEast();
		///	}
		/// </code>
		/// </example>
		public void MoveEast()
		{
			RenderMap(.25,0,0);
		}
      
		/// <summary>
		/// Pans the map northwards
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move In button event handler panning the map northwards:
		/// <code>
		/// public void btnMoveNorth_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveNorth();
		///	}
		/// </code>
		/// </example>
		public void MoveNorth()
		{
			RenderMap(0,.25,0);
		}

		/// <summary>
		/// Pans the map southwards
		/// </summary>
		/// <remarks>
		/// The HTML Image element containing the rendered map must be refreshed for results of this operation to be displayed. This can be performed by refreshing the Posting ASPX page - such as occurs inherently during postback; e.g. if these events are triggered by ASP.Net controls (e.g. buttons).
		/// </remarks>
		/// <example>
		/// The following example shows a Move In button event handler panning the map southwards:
		/// <code>
		/// public void btnMoveSouth_Click(object sender, System.EventArgs e)
		/// {
		///		MapPointProvider.MoveSouth();
		///	}
		/// </code>
		/// </example>
		public void MoveSouth()
		{
			RenderMap(0,-.25,0);
		}
		#endregion

		#region Map rendering
		private void RenderMap(double myPanHorizontalAdjustment, double myPanVerticalAdjustment, double myZoomValue) 
		{
			HttpSessionState session = HttpContext.Current.Session;

			//Get the latest map specification object
			MapSpecification mapSpec = (MapSpecification)session["myMapSpec"];

			//Pan or zoom the map as appropriate
			mapSpec.Options.PanHorizontal = mapSpec.Options.PanHorizontal + myPanHorizontalAdjustment;
			mapSpec.Options.PanVertical = mapSpec.Options.PanVertical + myPanVerticalAdjustment;

			if (!(mapSpec.Options.Zoom + myZoomValue <= 0))
				mapSpec.Options.Zoom = mapSpec.Options.Zoom + myZoomValue;

            
			//Declare the map image array and get the map
			MapImage[] myMapImages;

			try
			{
				myMapImages = mapPointGlobal.RenderService.GetMap(mapSpec);

				//Store the map in session, then render it in the Map.aspx page
				session["myMapImage"] = myMapImages[0];
					
				//Also restore the changed map specification
				session["myMapSpec"] = mapSpec;
			}
			catch(Exception myException)
			{
				throw new Exception("Error occurred attempting to get a map: " + myException.Message);
			}
		}
		#endregion

		#region Map image data properties
		/// <summary>
		/// <para>This static search result property exposes the Mime Type of the rendered map image rendered by MapPoint.</para>
		/// <para>This member will an throw error when a map has not been rendered (see <see cref="MapRendered"/>).</para>
		/// </summary>
		/// <example>
		/// The example below depicts an ASPX implemention to return the rendered map as an image:
		/// <code>
		/// // This line prevents the map from being cached locally.
		/// Response.Cache.SetCacheability(HttpCacheability.NoCache);
		/// 
		/// // Now clear the buffer and write the bits to the page.
		/// Response.Clear();
		/// Response.ContentType = NearestMapPointProvider.MapImageMimeType;
		/// Response.BinaryWrite(NearestMapPointProvider.MapImageBits);
		/// Response.End();
		/// </code>
		/// </example>
		static public string MapImageMimeType
		{
			get
			{
				MapImage mapImage = (MapImage)HttpContext.Current.Session["myMapImage"];
				return mapImage.MimeData.MimeType;
			}
		}

		/// <summary>
		/// <para>This static search result property exposes the binary data of the rendered map image rendered by MapPoint.</para>
		/// <para>This member will an throw error when a map has not been rendered (see <see cref="MapRendered"/>).</para>
		/// </summary>
		/// <example>
		/// The example below depicts an ASPX implemention to return the rendered map as an image:
		/// <code>
		/// // This line prevents the map from being cached locally.
		/// Response.Cache.SetCacheability(HttpCacheability.NoCache);
		/// 
		/// // Now clear the buffer and write the bits to the page.
		/// Response.Clear();
		/// Response.ContentType = NearestMapPointProvider.MapImageMimeType;
		/// Response.BinaryWrite(NearestMapPointProvider.MapImageBits);
		/// Response.End();
		/// </code>
		/// </example>
		static public byte[] MapImageBits
		{
			get
			{
				MapImage mapImage = (MapImage)HttpContext.Current.Session["myMapImage"];
				return mapImage.MimeData.Bits;
			}
		}
		#endregion
	}
}
