using System;
using System.ComponentModel;
using System.Drawing;
using System.Web.UI;
using ioko.ComponentModel.LicenseProvider;
using Microsoft.ContentManagement.Publishing.Extensions.Placeholders;
using Microsoft.ContentManagement.WebControls;
using Microsoft.ContentManagement.WebControls.Design;
using MSIBPlusPack.Xml.Serialization;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// The NearestPlaceholder class defines a CMS placeholder component that integrates with MapPoint Web Services to allow site users to search for local or national facilities related to posting content.
	/// NearestPlaceholder provides CMS-integrated business logic for performing MapPoint searches, it does not include a presentation layer - allowing look and feel to be controlled by the implementor.
	/// </summary>
	/// <remarks>
	/// <para>
	/// NearestPlaceholder exposes an instance of the <see cref="NearestMapPointProvider"/> class via its <see cref="MapPointProvider"/> property.
	/// This is used to specify search parameters, perform the search and interrogate results; it persists state via Session and View State.
	/// </para>
	/// <para>
	/// NearestPlaceholder is derived from the CMS class Microsoft.ContentManagement.WebControls.BasePlaceholderControl.
	/// It is defined as an abstract class and must be overridden to provide a presentation layer and the overridden class implemented in appropriate CMS Posting Templates.
	/// The <a href="MSIBPlusPack.ProgrammingTasks.Nearest.Functionality.html">Functionality</a> topic provides an overview of the presentation layer requirements.
	/// The <a href="MSIBPlusPack.ProgrammingTasks.Nearest.Examples.html">Examples</a> topic provides example presentation layers overridding this class.
	/// </para>
	/// <para>
	/// Authoring presentation layer components should be managed in overrides for
	/// <see cref="CreateAuthoringChildControls"/>, <see cref="LoadPlaceholderContentForAuthoring"/> and <see cref="SavePlaceholderContent"/>.
	/// </para>
	/// <para>
	/// Subscriber (site user) presentation layer components should be managed in overrides for
	/// <see cref="CreatePresentationChildControls"/> and <see cref="LoadPlaceholderContentForPresentation"/>.
	/// </para>
	/// <para>
	/// The <see cref="NearestPlaceholderExample"/> is derived from NearestPlaceholder as an example presentation layer.
	/// It offers the most frequently required functionality for developers, authors and subscribers.
	/// The class is provided in this assembly for immediate use in a CMS solutions.
	/// </para>
	/// <para>
	/// Presentation layer classes inheriting from NearestPlaceholder must be bound to a CMS XmlPlaceholder.
	/// The XmlPlaceholder is used to maintain state of Nearest search parameters.
	/// Nearest search parameters can be configured at Design Time, programmatically, or by CMS Authors or site subscribers.
	/// Search parameters are exposed by <see cref="NearestMapPointProvider"/> and include
	/// <see cref="NearestMapPointProvider.MapPointDataSource"/>,
	/// <see cref="NearestMapPointProvider.PointsOfInterestDataSource"/>,
	/// <see cref="NearestMapPointProvider.EntityTypeValue"/>,
	/// <see cref="NearestMapPointProvider.Keywords"/> and
	/// <see cref="NearestMapPointProvider.Range"/>,
	/// </para>
	/// <para>The NearestPlaceholder class requires implementers to purchase the relevant subscription to MapPoint Web Services.</para>
	/// <para><b>Configuration</b></para>
	/// <para>
	/// MapPoint subscription details  must be stored in the host CMS ASP.Net application's <span class="code">web.config</span> file.
	/// The <b>configuration/appSettings</b> section must include your MapPoint subscription credentials under the keys <b>MPUser</b> (user name) and <b>MPPass</b> (password).
	/// It should also include the URL of the MapPoint Finder and Render service under the keys <b>MPFindService</b> and <b>MPRenderService</b>.
	/// </para>
	/// <code>
	/// &lt;?xml version="1.0" encoding="utf-8"?>
	/// &lt;configuration&gt;
	/// &lt;appSettings&gt;
	/// &lt;add key="MPFindService" value="http://findv3.staging.mappoint.net/Find-30"/&gt;
	/// &lt;add key="MPRenderService" value="http://renderv3.staging.mappoint.net/Render-30"/&gt;
	/// &lt;add key="MPUser" value="UserID" /&gt;
	/// &lt;add key="MPPass" value="Password" /&gt;
	/// ...
	/// ...
	/// &lt;/appSettings&gt;
	/// &lt;/configuration&gt;</code>
	/// <para><span class="code">web.config</span>'s <b>MPFindService</b> and <b>MPRenderService</b> properties default to the staging MapPoint service URLs. The table below depicts the URLs for the current MapPoint service stage and live environments:</para>
	/// <table class="dtTABLE" cellSpacing="0">
	///		<tr>
	///			<th>MapPoint Service</th>
	///			<th>Find URL</th>
	///			<th>Render URL</th>
	///		</tr>
	///		<tr>
	///			<td>Live</td>
	///			<td class="code">http://findv3.mappoint.net/Find-30</td>
	///			<td class="code">http://renderv3.mappoint.net/Render-30</td>
	///		</tr>
	///		<tr>
	///			<td>Stage</td>
	///			<td class="code">http://findv3.staging.mappoint.net/Find-30</td>
	///			<td class="code">http://renderv3.staging.mappoint.net/Render-30</td>
	///		</tr>
	///	</table>
	/// </remarks>
	[
		ToolboxData("<{0}:NearestPlaceholder runat=\"server\"></{0}:NearestPlaceholder>"),
			ParseChildren(true),
			SupportedPlaceholderDefinitionType(typeof(XmlPlaceholderDefinition)),
			ToolboxBitmap(typeof(NearestPlaceholder), "NearestPlaceholderIcon.bmp")
	]
	[LicenseProvider(typeof(PlusPackLicenseProvider))]
	public abstract class NearestPlaceholder : BasePlaceholderControl
	{
		#region State management fields
		private INearestMapProvider mapProvider;
		#endregion

		#region Licensing static fields
		private static PlusPackLicense license = null;
		private static DateTime lastDate;
		#endregion

		#region Licensing Test
		/// <summary>
		/// This member handles mandatory component initialization. It has been sealed.
		/// </summary>
		/// <param name="e">EventArgs parameter</param>
		protected sealed override void OnInit(EventArgs e)
		{
			ValidateLicense();

			base.OnInit (e);
		}

		private void ValidateLicense()
		{
			try
			{
				bool updateLicense = false;

				if (license == null)
					updateLicense = true;
				else if (lastDate != DateTime.Today)
					updateLicense = true;

				if (updateLicense)
				{
					license = (PlusPackLicense)LicenseManager.Validate(typeof(NearestPlaceholder),this);
					lastDate = DateTime.Today;
				}

				switch(license.Validity)
				{
					case MSIBLicenseValidator.LicenseState.Full:
						return;

					case MSIBLicenseValidator.LicenseState.Trial_Active:
						if (updateLicense)
						{
							TimeSpan span = license.ExpiryDate.Date.Subtract(lastDate);
							int daysRemaining = span.Days + 1;
							if (daysRemaining <= 7)
								throw new Exception(String.Format("Warning: Your trial license of MSIB Plus Pack will run out in {0} days. This component will function normally for the remainder of today.",
									daysRemaining));
						}

						return;

					case MSIBLicenseValidator.LicenseState.Invalid:
					case MSIBLicenseValidator.LicenseState.None:
						break;

					case MSIBLicenseValidator.LicenseState.Trial_Expired:
						throw new Exception("Your trial MSIB Plus Pack trial license has expired. To continue using this component please purchase the relevant license(s).");
				}		
			} 
			catch {}

			throw new Exception("You need a valid MSIB Plus Pack license. Please purchase the relevant license(s).");
		}
		#endregion

		#region Constructors
		/// <summary>
		/// The instance constructor for the NearestPlaceholder class initializes the <see cref="MapPointProvider"/> object for search operations.
		/// </summary>
		public NearestPlaceholder()
		{
			mapProvider = new NearestMapPointProvider();
		}
		#endregion

		#region Properties
		/// <summary>
		/// This read-only property exposes an instance of <see cref="NearestMapPointProvider"/> used by the <see cref="NearestPlaceholder"/> to manage search specification, perform searches and return search results.
		/// </summary>
		/// <remarks>
		/// See the documentation of the <see cref="NearestMapPointProvider"/> class for information about the object exposed by this property.
		/// </remarks>
		/// <example>
		/// The example below sets search parameters against the NearestPlaceholder's MapPointProvider instance.
		/// It then performs the search operation.
		/// Finally it sets the visibility of search result controls appropriately.
		/// <code>
		/// myNearestPlaceholder.MapPointProvider.MapPointDataSource = "MapPoint.US";
		/// myNearestPlaceholder.MapPointProvider.PointsOfInterestDataSource = "NavTech.US";
		/// myNearestPlaceholder.MapPointProvider.EntityTypeValue = ddlEntityType.SelectedValue;
		/// myNearestPlaceholder.MapPointProvider.Keywords = tbKeywords.Text;
		///	myNearestPlaceholder.MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);	
		/// MapPointProvider.FindNearest();
		/// pnlResults.Visible = MapPointProvider.SearchActive;
		/// imgResultsMap.Visible = MapPointProvider.MapRendered;
		/// </code>
		/// </example>
		[
			Bindable(true),
				Category("MSIB Plus Pack - NearestPlaceholder"),
				DefaultValue(""),
				Description("")
		]
		public NearestMapPointProvider MapPointProvider
		{
			get { return (NearestMapPointProvider)mapProvider; }
		}

		/// <summary>
		/// Exposes the bound XML placeholder
		/// </summary>
		private XmlPlaceholder BoundXmlPlaceholder
		{
			get { return (XmlPlaceholder)BoundPlaceholder; }
		}
		#endregion

		#region Default content population
		/// <summary>
		/// This member overrides sets the default content of the bound XmlPlaceholder.
		/// </summary>
		/// <param name="e">PlaceholderControlCancelEventArgs event arguments object.</param>
		/// <remarks>
		/// This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's OnPopulatingDefaultContent member.
		/// This method has been sealed.
		/// </remarks>
		sealed protected override void OnPopulatingDefaultContent(PlaceholderControlCancelEventArgs e)
		{
			NearestXMLSerializer nearestXMLSerializer = new NearestXMLSerializer();
			BoundXmlPlaceholder.XmlAsString = nearestXMLSerializer.XML;
		}
		#endregion

		#region Author context control creation members
		/// <summary>
		/// <para>This member validates connectivity to MapPoint Web Services and causes <see cref="MapPointProvider"/> to update its search parameters from View State.</para>
		/// <para>
		/// It should be overridden to create any presentation layer controls that site authors should interact with.
		/// </para>
		/// </summary>
		/// <param name="authoringContainer">The container control to add controls to.</param>
		/// <remarks>
		/// <para>This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's CreateAuthoringChildControls member.</para>
		/// <para>
		/// Overrides of this member should not set the state of search parameter controls.
		/// When this member is called, placeholder content will not yet have been parsed.
		/// You can perform operations with <see cref="MapPointProvider"/> results properties, but should not interrogate search parameters such as <see cref="NearestMapPointProvider.EntityTypeValue"/> and <see cref="NearestMapPointProvider.Range"/>.
		/// </para>
		/// <para>When you override this method, you must call the (base class's) overridden member first to initialize the MapPointProvider object - N.B. this operation will throw an error if there are connectivity issues.</para>
		/// <code>
		/// protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		/// {
		///		// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
		///		base.CreateAuthoringChildControls(authoringContainer);
		/// 
		///		//
		///		// TODO: Add your code here
		///		//
		///	}
		///	</code>
		/// </remarks>
		///	<example>
		///	The example below overrides NearestPlaceholder's CreateAuthoringChildControls member to create and poulate a drop down list allowing the author to specify the Entity Type.
		///	<code>
		/// protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		/// {
		/// 	// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
		/// 	base.CreateAuthoringChildControls(authoringContainer);
		/// 
		/// 	// Create entity type drop down control
		/// 	ddlEntityType = new DropDownList();
		/// 	ddlEntityType.ID = "ddlEntityType";
		/// 	foreach (NearestMapEntityType mapEntityType in MapPointProvider.MapEntityTypes)
		/// 		ddlEntityType.Items.Add(new ListItem(mapEntityType.Name, mapEntityType.Value));
		/// 
		/// 	// Add controls to container
		/// 	authoringContainer.Controls.Add(new LiteralControl("&lt;b&gt;Nearest MapPoint Search&lt;/b&gt;" +
		/// 	"&lt;p&gt;Please specify the required entity type that site users will search for in conjunction with this Posting:&lt;/p&gt;" +
		/// 	"&lt;table&gt;&lt;tr&gt;&lt;td&gt;Entity Type:&lt;/td&gt;&lt;td&gt;"));
		/// 	authoringContainer.Controls.Add(ddlEntityType);
		/// 	authoringContainer.Controls.Add(new LiteralControl(String.Format("&lt;/td&gt;&lt;/tr&gt;&lt;/table&gt;")));
		/// }
		/// </code>
		/// </example>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			MapPointProvider.LoadViewState(ViewState);
			
			// Manadatoy call to MapPointProvider.ValidateConnectivity
			MapPointProvider.ValidateConnectivity();
			
		}
		#endregion

		#region Authoring context data serialization / deserialization members
		/// <summary>
		/// <para>This member deserializes NearestPlaceholder search configuration from the bound XML placeholder content updating the state of <see cref="MapPointProvider"/> accordingly.</para>
		/// <para>It should be overridden to set the state of any authoring presentation layer components with state information stored in the xml placeholder.</para>
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs event arguments object.</param>
		/// <remarks>
		/// <para>This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's LoadPlaceholderContentForAuthoring member.</para>
		/// <para>When you override this method, you must call the (base class's) overridden member first to allow the MapPointProvider to deserialize the XML placeholder.
		/// You can then query <see cref="MapPointProvider"/> for state information.</para>
		/// <code>
		/// protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		/// {
		/// // Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member first to initialize MapPointProvider
		/// 	base.LoadPlaceholderContentForAuthoring(e);
		/// 
		/// 	//
		/// 	// TODO: Add your code here
		/// 	//
		/// }
		/// </code>
		/// </remarks>
		/// <example> 
		/// The example below sets the currently selected item in an Entity Type drop down list to the value deserialized from the placeholder:
		/// <code>
		/// protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		/// {
		/// 	// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member first to initialize MapPointProvider
		/// 	base.LoadPlaceholderContentForAuthoring(e);
		/// 
		/// 	if (null != MapPointProvider.EntityTypeValue)
		/// 	ddlEntityType.SelectedValue = MapPointProvider.EntityTypeValue;
		/// }
		/// </code>
		/// </example>
		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			EnsureChildControls();
			try {
				NearestXMLSerializer nearestXMLSerializer = new NearestXMLSerializer(BoundXmlPlaceholder.XmlAsString);

				// Set control property values from xml data:
				try 
				{
					MapPointProvider.EntityTypeValue = nearestXMLSerializer.EntityTypeValue;
					MapPointProvider.Keywords = nearestXMLSerializer.Keywords;
					MapPointProvider.MapPointDataSource = nearestXMLSerializer.MapPointDataSource;
					MapPointProvider.PointsOfInterestDataSource = nearestXMLSerializer.PointsOfInterestDataSource;
					MapPointProvider.Range = nearestXMLSerializer.Range;
				}
				catch {}
			}
			catch {
			}
		}

		/// <summary>
		/// <para>This member serializes NearestPlaceholder search configuration to the bound XML placeholder content.</para>
		/// <para>It should be overridden to update the <see cref="MapPointProvider"/> object before state information is serialized into the XML placeholder.</para>
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs event arguments object.</param>
		/// <remarks>
		/// <para>This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's SavePlaceholderContent member.</para>
		/// <para>When you override this method, you should update the <see cref="MapPointProvider"/> object before calling the (base class's) overridden member to serialize state information into the XML placeholder.</para>
		/// <code>
		/// protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		/// {
		///		//
		/// 	// TODO: Add your code here
		/// 	//
		///
		///		// Call NearestPlaceHolder class's SavePlaceholderContent last to save the MapPointProvider state to the placeholder definition
		///		base.SavePlaceholderContent(e);
		///	}
		///	</code>
		///	</remarks>
		///	<example>
		///	The example below updates the selected Entity Type value from a drop down list.
		///	<code>
		/// protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		/// {
		/// 	if (null != ddlEntityType)
		/// 	MapPointProvider.EntityTypeValue = ddlEntityType.SelectedValue;
		/// 
		/// 	// Call NearestPlaceHolder class's SavePlaceholderContent last to save the MapPointProvider state to the placeholder definition
		/// 	base.SavePlaceholderContent(e);
		/// }
		///	</code>
		///	</example>
		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			// Create NearestPlaceholder XML serializer
			NearestXMLSerializer nearestXMLSerializer = new NearestXMLSerializer();

			// Set XML serializer properties
			nearestXMLSerializer.EntityTypeValue = MapPointProvider.EntityTypeValue;
			nearestXMLSerializer.Keywords = MapPointProvider.Keywords;
			nearestXMLSerializer.MapPointDataSource = MapPointProvider.MapPointDataSource;
			nearestXMLSerializer.PointsOfInterestDataSource = MapPointProvider.PointsOfInterestDataSource;
			nearestXMLSerializer.Range = MapPointProvider.Range;

			// Copy XML to bound Placeholder
			BoundXmlPlaceholder.XmlAsString = nearestXMLSerializer.XML;
		}
		#endregion

		#region Presentation context control creation members
		/// <summary>
		/// <para>This member validates connectivity to MapPoint Web Services and causes <see cref="MapPointProvider"/> to update its search parameters from View State.</para>
		/// <para>It should be overriden to create any presentation layer components that subscribers (site users) should interact with.</para>
		/// </summary>
		/// <param name="presentationContainer">Container control to add controls to.</param>
		/// <remarks>
		/// <para>This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's CreatePresentationChildControls member.</para>
		/// <para>
		/// Overrides of this member should not set the state of search parameter controls.
		/// When this member is called, placeholder content will not yet have been parsed.
		/// You can perform operations with <see cref="MapPointProvider"/> results properties, but should not interrogate search parameters such as <see cref="NearestMapPointProvider.EntityTypeValue"/> and <see cref="NearestMapPointProvider.Range"/>.
		/// </para>
		/// <para>
		/// When you override this method, you must call the (base class's) overridden member first to initialize the MapPointProvider object - N.B. this operation will throw an error if there are connectivity issues.
		/// As per example overrides for this member it is worth error handling the call to the base member.
		/// If an error is thrown the error handler exits without creating any controls  - consequently subscribers will not see any errors during periods of temporary MapPoint connectivity issues (the controls will simply disappear).
		/// </para>
		/// <code>
		/// protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		/// {
		///		// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
		/// 	try { base.CreatePresentationChildControls(presentationContainer); } 
		/// 	catch { return; }	// Prevent errors being reported to site subscribers - just remove the control from the page
		/// 
		///		//
		///		// TODO: Add your code here
		///		//
		/// }
		/// </code>
		/// </remarks>
		/// <example>
		/// This example creates a keywords textbox, range drop down list, search button, map image control, results table and a panel to contain the results image and table.
		/// It adds all these controls to the container.
		/// Finally it checks whether a search is active, and if so populates the results table.
		/// <code>
		/// protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		/// {
		/// 	// Call NearestPlaceHolder class's CreatePresentationChildControls member frist to initialize MapPointProvider
		/// 	try { base.CreatePresentationChildControls(presentationContainer); } 
		///		catch { return; }	// Prevent errors being reported to site subscribers - just remove the control from the page
		/// 
		///		// Create Search keywords textbox
		/// 	tbKeywords = new TextBox();
		/// 	tbKeywords.ID = "tbKeywords";
		/// 	tbKeywords.Width = 128;
		/// 	tbKeywords.MaxLength = 50;			// Give the site subscriber ample space for arbitrary keywords
		/// 	tbKeywords.ToolTip = "Please enter your Postcode here";
		/// 
		/// 	// Create Range drop down list:
		/// 	ddlRange = new DropDownList();
		/// 	ddlRange.ID = "ddlRange";
		/// 	ddlRange.Width = 128;
		/// 	ddlRange.Items.Add(new ListItem("2 miles","2"));
		/// 	ddlRange.Items.Add(new ListItem("5 miles","5"));
		/// 	ddlRange.Items.Add(new ListItem("10 miles","10"));
		/// 	ddlRange.Items.Add(new ListItem("25 miles","25"));
		/// 	ddlRange.Items.Add(new ListItem("80 miles","80"));
		/// 	ddlRange.SelectedValue = "10";
		/// 
		/// 	// Create Search button
		/// 	btnSearch = new Button();
		/// 	btnSearch.ID = "btnSearch";
		/// 	btnSearch.Text = "Search";
		/// 	btnSearch.Click += new EventHandler(btnSearch_Click);
		/// 	btnSearch.ToolTip = "Click to perform search";
		/// 
		/// 	// Create Map image
		/// 	imgMap = new System.Web.UI.WebControls.Image();
		/// 	imgMap.ImageUrl = _MapImageUrl;
		/// 	imgMap.ID = "imageMap";
		/// 	imgMap.Width = 500;
		/// 	imgMap.Height = 500;
		/// 	imgMap.BorderWidth = 1;
		/// 	imgMap.AlternateText = "Map";
		/// 	imgMap.Visible = MapPointProvider.MapRendered;
		/// 
		/// 	// Create search results table:
		/// 	tblSearchResults = new Table();
		/// 	tblSearchResults.ID = "tblSearchResults";
		/// 	tblSearchResults.GridLines = GridLines.Both;
		/// 	tblSearchResults.BorderWidth = 2;
		/// 	tblSearchResults.CellPadding = 4;
		/// 	tblSearchResults.CellSpacing = 0;
		/// 
		/// 	// Create results panel
		/// 	pnlResults = new Panel();
		/// 	pnlResults.ID = "pnlResults";
		/// 	pnlResults.Visible = MapPointProvider.SearchActive;
		/// 	pnlResults.Controls.Add(imgMap);
		/// 	pnlResults.Controls.Add(new LiteralControl("&lt;br&gt;&lt;br&gt;"));
		/// 	pnlResults.Controls.Add(tblSearchResults);
		/// 
		/// 	// Add controls to container
		/// 	presentationContainer.Controls.Add(new LiteralControl("&lt;b&gt;Search&lt;/b&gt;" +
		/// 	"&lt;table&gt;&lt;tr&gt;&lt;td&gt;Postcode/Keywords:&lt;/td&gt;&lt;td&gt;"));
		/// 	presentationContainer.Controls.Add(tbKeywords);
		/// 	presentationContainer.Controls.Add(new LiteralControl("&lt;/tr&gt;&lt;tr&gt;&lt;td&gt;Range:&lt;/td&gt;&lt;td&gt;"));
		/// 	presentationContainer.Controls.Add(ddlRange);
		/// 	presentationContainer.Controls.Add(new LiteralControl("&lt;/td&gt;&lt;/tr&gt;&lt;tr&gt;&lt;td colspan='2' align='right'&gt;"));
		/// 	presentationContainer.Controls.Add(btnSearch);
		/// 	presentationContainer.Controls.Add(new LiteralControl("&lt;/td&gt;&lt;/tr&gt;&lt;/table&gt;"));
		/// 	presentationContainer.Controls.Add(pnlResults);
		/// 	presentationContainer.Controls.Add(pnlResultsLocationError);
		/// 
		/// 	// Populate the results table with the current results
		/// 	if (MapPointProvider.SearchActive)
		/// 	PopulateNearestResults();
		/// }
		/// </code>
		/// </example>
		protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{
			MapPointProvider.LoadViewState(ViewState);

			// Manadatoy call to MapPointProvider.ValidateConnectivity
			MapPointProvider.ValidateConnectivity();
		}
		#endregion

		#region Presentation context data deserialization members
		/// <summary>
		/// This member deserializes NearestPlaceholder search configuration from the bound XML placeholder content, setting <see cref="MapPointProvider"/> search parameters accordingly.
		/// It should be overridden to set the state of any subscriber presentation layer components with state information stored in the xml placeholder.
		/// This member is often not overridden, but can be used to allow authors to specify defaults, but allow subscribers to override.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs event arguments object.</param>
		/// <remarks>
		/// <para>This member overrides Microsoft.ContentManagement.WebControls.BasePlaceholderControl's LoadPlaceholderContentForPresentation member.</para>
		/// <para>
		/// When you override this method, you must call the (base class's) overridden member first to allow the MapPointProvider to deserialize the XML placeholder.
		/// You can then query <see cref="MapPointProvider"/> object for state information.
		/// </para>
		/// <code>
		/// protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		/// {
		/// 	// Call NearestPlaceHolder class's LoadPlaceholderContentForPresentation member frist to initialize MapPointProvider
		/// 	base.LoadPlaceholderContentForPresentation(e);
		/// 
		/// 	//
		/// 	// TODO: Add your code here
		/// 	//
		/// }
		/// </code>
		/// </remarks>
		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			EnsureChildControls();
			NearestXMLSerializer nearestXMLSerializer = new NearestXMLSerializer(BoundXmlPlaceholder.XmlAsString);
			try
			{
				MapPointProvider.EntityTypeValue = nearestXMLSerializer.EntityTypeValue;
				MapPointProvider.Keywords = nearestXMLSerializer.Keywords;
				MapPointProvider.MapPointDataSource = nearestXMLSerializer.MapPointDataSource;
				MapPointProvider.PointsOfInterestDataSource = nearestXMLSerializer.PointsOfInterestDataSource;
				MapPointProvider.Range = nearestXMLSerializer.Range;
			}
			catch { }
		}
		#endregion			
	}
}



