using System;
using System.Text;
using System.Web.UI;
using System.Web.UI.WebControls;
using Microsoft.ContentManagement.WebControls;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// This is an example presentation layer for the <see cref="NearestPlaceholder"/> MSIB Plus Pack component.
	/// </summary>
	/// <remarks>
	/// <para>It is a MCMS placeholder that can be implemented verbatim on any CMS-enabled website to provide the following functionality:</para>
	/// <ul>
	///		<li>The CMS author chooses the entity type for the search</li>
	///		<li>The subscriber (site user) selects the keywords and range for the nearest search</li>
	///		<li>The search results consist of a map and table of results</li>
	///		<li>The map includes navigation controls (pan and zoom).</li>
	/// </ul>
	/// <para>This presentation layer does not include any specific styling, its look and feel can easily be manipulated by cascading stylesheets.</para>
	/// <para>The CMS template you use to implement this component can programatically manipulate any property it does not offer to the subscriber or user without risking any idiosyncratic behaviour. E.g. Explict specification of data sources:</para>
	/// <code>
	/// myNearestPlaceholderExample.MapPointProvider.MapPointDataSource = "MapPoint.EU";
	/// myNearestPlaceholderExample.MapPointProvider.PointsOfInterestDataSource = "NavTech.EU";
	/// </code>
	/// <para>
	/// To implement this class you must provide an HTTP source that returns the search resultant rendered map image.
	/// A URL source (commonly an ASPX page) must expose the image data by manipulating
	/// <see cref="NearestMapPointProvider.MapImageMimeType"/> and 
	/// <see cref="NearestMapPointProvider.MapImageBits"/> accordingly (see the
	/// <a href="MSIBPlusPack.ProgrammingTasks.Nearest.Functionality.html">Functionality</a>
	/// topic).
	/// This class exposes the  property <see cref="MapImageUrl"/>
	/// to allow the developer to specify the URL of the map image rendering source.
	/// This can be implement in the template ASPX page as:
	/// </para>
	/// <code>
	/// private void Page_Load(object sender, System.EventArgs e
	/// {
	/// 	myNearestPlaceholder.MapImageUrl = "/MSIBPlusPack/Map.aspx";
	/// }</code>
	/// </remarks>
	public class NearestPlaceholderExample : NearestPlaceholder
	{		
		#region Properties
		// Internal member for MapImageURL property (with default):
		private string _MapImageUrl = "/MSIBPlusPack/Map.aspx";

		/// <summary>
		/// Url of the ASPX page that return the image data for the rendered map.
		/// </summary>
		public string MapImageUrl
		{
			get { return _MapImageUrl; }
			set { _MapImageUrl = value; }
		}
		#endregion

		#region Author context UI widget members
		/// <summary>
		/// Author's Entity Type drop down list control member
		/// </summary>
		protected DropDownList ddlEntityType;
		#endregion

		#region Author context control creation members
		/// <summary>
		/// Overrides <see cref="NearestPlaceholder"/>'s <see cref="NearestPlaceholder.CreateAuthoringChildControls"/> member to create and poulate a drop down list allowing the author to specify the Entity Type. 
		/// </summary>
		/// <param name="authoringContainer">The container control to add controls to.</param>
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
			base.CreateAuthoringChildControls(authoringContainer);

			//
			// TODO: Add your code here
			//

			// Create entity type drop down control
			ddlEntityType = new DropDownList();
			ddlEntityType.ID = "ddlEntityType";
			foreach (NearestMapEntityType mapEntityType in MapPointProvider.MapEntityTypes)
				ddlEntityType.Items.Add(new ListItem(mapEntityType.Name, mapEntityType.Value));

			// Add controls to container
			authoringContainer.Controls.Add(new LiteralControl("<b>Nearest MapPoint Search</b>\n" +
				"<p>Please specify the required MapPoint Entity Type that site Subscribers will search for in conjunction with this Posting:</p>\n" +
				"<table><tr><td>Entity Type:</td><td>"));
			authoringContainer.Controls.Add(ddlEntityType);
			authoringContainer.Controls.Add(new LiteralControl(String.Format("</td></tr></table>\n" +
				"<p style='font-size:75%'><i>{0}</i></p>\n",
				MapPointProvider.Version.Replace("\n", "</i></p><p style='font-size:75%'><i>"))));
		}
		#endregion

		#region Authoring context data serialization / deserialization members
		/// <summary>
		/// Overrides <see cref="NearestPlaceholder"/>'s <see cref="NearestPlaceholder.LoadPlaceholderContentForAuthoring"/> member to set the currently selected item in the Entity Type drop down list to the value deserialized from the placeholder
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs event arguments object.</param>
		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member first to initialize MapPointProvider
			base.LoadPlaceholderContentForAuthoring(e);

			//
			// TODO: Add your code here
			//

			if (null != MapPointProvider.EntityTypeValue)
				ddlEntityType.SelectedValue = MapPointProvider.EntityTypeValue;
		}

		/// <summary>
		/// Overrides <see cref="NearestPlaceholder"/>'s <see cref="NearestPlaceholder.SavePlaceholderContent"/> member to update the <see cref="MapPointProvider"/>'s <see cref="MapPointProvider.EntityTypeValue"/> from a drop down list. 
		/// </summary>
		/// <param name="e">PlaceholderControlSaveEventArgs event arguments object.</param>
		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			//
			// TODO: Add your code here
			//

			if (null != ddlEntityType)
				MapPointProvider.EntityTypeValue = ddlEntityType.SelectedValue;

			// Call NearestPlaceHolder class's SavePlaceholderContent last to save the MapPointProvider state to the placeholder definition
			base.SavePlaceholderContent(e);
		}
		#endregion

		#region Presentation context UI widget members
		/// <summary>
		/// Textbox for subscriber to enter location search keywords
		/// </summary>
		protected TextBox tbKeywords;

		/// <summary>
		/// Drop down list for subscriber to select range (radius) for determining points of interest
		/// </summary>
		protected DropDownList ddlRange;

		/// <summary>
		/// Button to perform search operation
		/// </summary>
		protected Button btnSearch;

		/// <summary>
		/// Panel for results (to contain table listing results and map image)
		/// </summary>
		protected Panel pnlResults;

		/// <summary>
		/// Panel for map image
		/// </summary>
		protected Panel pnlResultsMap;

		/// <summary>
		/// Panel to display search error messages
		/// </summary>
		protected Panel pnlResultsLocationError;

		/// <summary>
		/// Button to zoom in on the map - rerendering it with an increased magnification.
		/// </summary>
		protected Button btnMoveIn;

		/// <summary>
		/// Button to zoom out on the map - rerendering it with an decreased magnification. 
		/// </summary>
		protected Button btnMoveOut;

		/// <summary>
		/// Button to pan the map westwards 
		/// </summary>
		protected Button btnMoveWest;

		/// <summary>
		/// Button to pan the map eastwards 
		/// </summary>
		protected Button btnMoveEast;

		/// <summary>
		/// Button to pan the map northwards 
		/// </summary>
		protected Button btnMoveNorth;

		/// <summary>
		/// Button to pan the map southwards 
		/// </summary>
		protected Button btnMoveSouth;

		/// <summary>
		/// Control to contain tabulated nearest search results
		/// </summary>
		protected Table tblSearchResults;
		#endregion

		#region Presentation context control creation members
		/// <summary>
		/// Overrides <see cref="NearestPlaceholder"/>'s <see cref="NearestPlaceholder.CreatePresentationChildControls"/> member to create a keywords textbox, range drop down list, search button, map image control, results table and a panel to contain the results image and table.
		/// It adds all these controls to the container.
		/// Finally it checks whether a search is active, and if so populates the results table. 
		/// </summary>
		/// <param name="presentationContainer">The container control to add controls to.</param>
		protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{
			try { base.CreatePresentationChildControls(presentationContainer); } 
			catch { return; }	// Prevent errors being reported to site subscribers - just remove the control from the page

			//
			// TODO: Add your code here
			//
	
			// Create Search keywords textbox
			tbKeywords = new TextBox();
			tbKeywords.ID = "tbKeywords";
			tbKeywords.Width = 128;
			tbKeywords.MaxLength = 50;			// Give the site subscriber ample space for arbitrary keywords
			tbKeywords.ToolTip = "Please enter your Postcode here";

			// Create Range drop down list:
			ddlRange = new DropDownList();
			ddlRange.ID = "ddlRange";
			ddlRange.Width = 128;
			ddlRange.Items.Add(new ListItem("2 miles","2"));
			ddlRange.Items.Add(new ListItem("5 miles","5"));
			ddlRange.Items.Add(new ListItem("10 miles","10"));
			ddlRange.Items.Add(new ListItem("25 miles","25"));
			ddlRange.Items.Add(new ListItem("80 miles","80"));
			ddlRange.SelectedValue = "10";

			// Create Search button
			btnSearch = new Button();
			btnSearch.ID = "btnSearch";
			btnSearch.Text = "Search";
			btnSearch.Click += new EventHandler(btnSearch_Click);
			btnSearch.ToolTip = "Click to perform search";

			// Create Map image
			Image imgMap = new Image();
			imgMap.ImageUrl = _MapImageUrl;
			imgMap.ID = "imageMap";
			imgMap.Width = 500;
			imgMap.Height = 500;
			imgMap.BorderWidth = 1;
			imgMap.AlternateText = "Map";

			// Create Move In (+) button
			btnMoveIn = new Button();
			btnMoveIn.ID = "btnMoveIn";
			btnMoveIn.Text = "+";
			btnMoveIn.Width = btnMoveIn.Height = 24;
			btnMoveIn.Click += new EventHandler(btnMoveIn_Click);
			btnMoveIn.ToolTip = "Zoom in on map";

			// Create Move Out (-) button
			btnMoveOut = new Button();
			btnMoveOut.ID = "btnMoveOut";
			btnMoveOut.Text = "-";
			btnMoveOut.Width = btnMoveOut.Height = 24;
			btnMoveOut.Click += new EventHandler(btnMoveOut_Click);
			btnMoveOut.ToolTip = "Zoom out on map";

			// Create Move West (W) button
			btnMoveWest = new Button();
			btnMoveWest.ID = "btnMoveWest";
			btnMoveWest.Text = "W";
			btnMoveWest.Width = btnMoveWest.Height = 24;
			btnMoveWest.Click += new EventHandler(btnMoveWest_Click);
			btnMoveWest.ToolTip = "Move map view westwards";
			
			// Create Move East (E) button
			btnMoveEast = new Button();
			btnMoveEast.ID = "btnMoveEast";
			btnMoveEast.Text = "E";
			btnMoveEast.Width = btnMoveEast.Height = 24;
			btnMoveEast.Click += new EventHandler(btnMoveEast_Click);
			btnMoveEast.ToolTip = "Move map view eastwards";

			// Create Move North (N) button
			btnMoveNorth = new Button();
			btnMoveNorth.ID = "btnMoveNorth";
			btnMoveNorth.Text = "N";
			btnMoveNorth.Width = btnMoveNorth.Height = 24;
			btnMoveNorth.Click += new EventHandler(btnMoveNorth_Click);
			btnMoveNorth.ToolTip = "Move map view northwards";

			// Create Move South (S) button
			btnMoveSouth = new Button();
			btnMoveSouth.ID = "btnMoveSouth";
			btnMoveSouth.Text = "S";
			btnMoveSouth.Width = btnMoveSouth.Height = 24;
			btnMoveSouth.Click += new EventHandler(btnMoveSouth_Click);
			btnMoveSouth.ToolTip = "Move map view southwards";

			// Create search results table:
			tblSearchResults = new Table();
			tblSearchResults.ID = "tblSearchResults";
			tblSearchResults.GridLines = GridLines.Both;
			tblSearchResults.BorderWidth = 2;
			tblSearchResults.CellPadding = 4;
			tblSearchResults.CellSpacing = 0;

			// Create results map and map navigation controls:
			pnlResultsMap = new Panel();
			pnlResultsMap.ID = "pnlResultsMap";
			pnlResultsMap.Visible = MapPointProvider.MapRendered;
			pnlResultsMap.Controls.Add(new LiteralControl("<table cellpadding='5' cellspacing='5'>" +
				"<tr><td rowpsan='5' align='right'>"));
			pnlResultsMap.Controls.Add(imgMap);
			pnlResultsMap.Controls.Add(new LiteralControl("</td><td>"));
			pnlResultsMap.Controls.Add(new LiteralControl("<table cellpadding='0' cellspacing='0'>" +
				"<tr><td>&nbsp;</td><td>"));
			pnlResultsMap.Controls.Add(btnMoveNorth);
			pnlResultsMap.Controls.Add(new LiteralControl("</td><td>&nbsp;</td></tr>" + 
				"<tr><td>"));
			pnlResultsMap.Controls.Add(btnMoveWest);
			pnlResultsMap.Controls.Add(new LiteralControl("</td><td>&nbsp;</td><td>"));
			pnlResultsMap.Controls.Add(btnMoveEast);
			pnlResultsMap.Controls.Add(new LiteralControl("</td></tr>" + 
				"<tr><td>&nbsp;</td><td>"));
			pnlResultsMap.Controls.Add(btnMoveSouth);
			pnlResultsMap.Controls.Add(new LiteralControl("</td><td>&nbsp;</td></tr>" +
				"<tr><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td></tr>" +
				"<tr><td>"));
			pnlResultsMap.Controls.Add(btnMoveOut);
			pnlResultsMap.Controls.Add(new LiteralControl("</td><td>&nbsp;</td><td>"));
			pnlResultsMap.Controls.Add(btnMoveIn);
			pnlResultsMap.Controls.Add(new LiteralControl("</td></tr></table>"));
			pnlResultsMap.Controls.Add(new LiteralControl("</td></tr>" +
				"</td></tr></table>"));

			// Create results panel
			pnlResults = new Panel();
			pnlResults.ID = "pnlResults";
			pnlResults.Visible = MapPointProvider.SearchActive;
			pnlResults.Controls.Add(pnlResultsMap);
			pnlResults.Controls.Add(tblSearchResults);

			// Create results error panel
			pnlResultsLocationError = new Panel();
			pnlResultsLocationError.ID = "pnlResultsLocationError";
			pnlResultsLocationError.Visible = false;
			pnlResultsLocationError.Controls.Add(new LiteralControl("<b>The specified search location could not be found, please provide an alternate postcode or keyword.</b>"));

			// Add controls to container
			presentationContainer.Controls.Add(new LiteralControl("<b>Nearest MapPoint Search</b>"));
			presentationContainer.Controls.Add(new LiteralControl("<table><tr><td>Postcode:</td><td>"));
			presentationContainer.Controls.Add(tbKeywords);
			presentationContainer.Controls.Add(new LiteralControl("</tr><tr><td>Range:</td><td>"));
			presentationContainer.Controls.Add(ddlRange);
			presentationContainer.Controls.Add(new LiteralControl("</td></tr><tr><td colspan='2' align='right'>"));
			presentationContainer.Controls.Add(btnSearch);
			presentationContainer.Controls.Add(new LiteralControl("</td></tr></table>"));
			presentationContainer.Controls.Add(pnlResults);
			presentationContainer.Controls.Add(pnlResultsLocationError);

			// Populate the results table with the current results
			if (MapPointProvider.SearchActive)
				PopulateNearestResults();
		}
		#endregion

		#region Presentation context event handler members
		/// <summary>
		/// Handles clicking of the search button.
		/// Updates the <see cref="MapPointProvider"/> object with the content of the search keywords textbox and the range selected in the drop down list.
		/// Performs the search operation, populating search results and setting the visibility of result controls accordingly.
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnSearch"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnSearch_Click(object sender, EventArgs e)
		{
			MapPointProvider.Keywords = tbKeywords.Text;
			MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);	

			MapPointProvider.FindNearest();

			pnlResults.Visible = MapPointProvider.SearchActive;
			pnlResultsLocationError.Visible = !MapPointProvider.SearchActive;
			pnlResultsMap.Visible = MapPointProvider.MapRendered;

			if (MapPointProvider.SearchActive)
				PopulateNearestResults();
		}

		/// <summary>
		/// Handles clicking of the move in (+) button.
		/// Zooms in on the map - rerendering it with an increased magnification.
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveIn"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveIn_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveIn();
		}
      
		/// <summary>
		/// Handles clicking of the (move out (-) button.
		/// Zooms out on the map - rerendering it with an decreased magnification. 
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveOut"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveOut_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveOut();
		}
      
		/// <summary>
		/// Handles clicking of the west (W) button.
		/// Pans the map westwards.
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveWest"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveWest_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveWest();
		}
      
		/// <summary>
		/// Handles clicking of the east (E) button.
		/// Pans the map eastwards.
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveEast"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveEast_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveEast();
		}
      
		/// <summary>
		/// Handles clicking of the north (N) button.
		/// Pans the map northwards.
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveNorth"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveNorth_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveNorth();
		}

		/// <summary>
		/// Handles clicking of the south (S) button.
		/// Pans the map southwards.
		/// The image control does not need updating since the page is being updated at this stage (during postback).
		/// </summary>
		/// <param name="sender">Object that fired the event (<see cref="btnMoveSouth"/>)</param>
		/// <param name="e">EventArgs event arguments object.</param>
		public void btnMoveSouth_Click(object sender, EventArgs e)
		{
			MapPointProvider.MoveSouth();
		}
		#endregion

		#region Presentation context data deserialization members
		/// <summary>
		/// Overrides <see cref="NearestPlaceholder"/>'s <see cref="NearestPlaceholder.LoadPlaceholderContentForPresentation"/> member. This override denotes required syntax, but does nothing.
		/// </summary>
		/// <param name="e">PlaceholderControlEventArgs event arguments object.</param>
		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForPresentation member frist to initialize MapPointProvider
			base.LoadPlaceholderContentForPresentation(e);

			//
			// TODO: Add your code here
			//
		}
		#endregion

		#region Presentation nearest result table population
		private void PopulateNearestResults()
		{
			// Remove any existing results (to handle postback of changed result parameters when results list has been populated by control creation member)
			tblSearchResults.Rows.Clear();

			// Create table header
			TableRow tableRow = new TableRow();
			TableHeaderCell tableHeaderCell = new TableHeaderCell();
			tableHeaderCell.Text = "No.";
			tableRow.Cells.Add(tableHeaderCell);
			tableHeaderCell = new TableHeaderCell();
			tableHeaderCell.Text = "Name";
			tableRow.Cells.Add(tableHeaderCell);
			tableHeaderCell = new TableHeaderCell();
			tableHeaderCell.Text = "Address";
			tableRow.Cells.Add(tableHeaderCell);
			tableHeaderCell = new TableHeaderCell();
			tableHeaderCell.Text = "Score";
			tableRow.Cells.Add(tableHeaderCell);
			tblSearchResults.Rows.Add(tableRow);

			// When the search returns no results, provide an informative message
			if (0 == MapPointProvider.NearestResults.Length)
			{
				tableRow = new TableRow();
				TableCell tableCell = new TableCell();
				tableCell.ColumnSpan = 4;
				tableCell.Text = "No items were found near the specified location.";
				tableRow.Cells.Add(tableCell);
				tblSearchResults.Rows.Add(tableRow);

				return;
			}

			// Populate the results table
			foreach(NearestResult nearestResult in MapPointProvider.NearestResults)
			{
				tableRow = new TableRow();

				TableCell tableCell = new TableCell();
				tableCell.Text = nearestResult.ResultCount.ToString();
				tableRow.Cells.Add(tableCell);

				tableCell = new TableCell();
				tableCell.Text = nearestResult.Name;
				tableRow.Cells.Add(tableCell);

				tableCell = new TableCell();
				// Append any non-null address field (in logical address order):
				StringBuilder stringBuilder = new StringBuilder();
				if (nearestResult.AddressLine != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.AddressLine);
				if (nearestResult.PrimaryCity != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.PrimaryCity);
				if (nearestResult.SecondaryCity != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.SecondaryCity);
				stringBuilder.Append(nearestResult.PostalCode);
				tableCell.Text = String.Format(stringBuilder.ToString());
				tableRow.Cells.Add(tableCell);

				tableCell = new TableCell();
				tableCell.Text = nearestResult.Score.ToString("0.00");
				tableRow.Cells.Add(tableCell);

				tblSearchResults.Rows.Add(tableRow);
			}
		}
		#endregion
	}
}
