using System;

namespace MSIBPlusPack.ContentManagement.Publishing.Placeholders
{
	/// <summary>
	/// The NearestResult class defines a single search result from a Nearest search.
	/// I.e. an occurance of the specified Entity Type near the specified location.
	/// </summary>
	/// <remarks>
	/// <para>The <see cref="NearestMapPointProvider"/> class exposes an array containing all Nearest search results via the property <see cref="NearestMapPointProvider.NearestResults"/>.</para>
	/// <para>This class cannot be overridden nor instantiated by external assemblies.</para>
	/// </remarks>
	/// <example>
	/// The example below depicts a method that adds an HTML bulletted list of results to the Panel control pnlSearchResults:
	/// <code>
	/// private void PopulateNearestResults()
	/// {
	/// 	pnlSearchResults.Controls.Clear();
	/// 
	/// 	if (0 == MapPointProvider.NearestResults.Length)
	///		{
	/// 		pnlSearchResults.Controls.Add(new LiteralControl("No items were found near the specified location."));
	/// 		return;
	///		}
	/// 
	/// 	pnlSearchResults.Controls.Add(new LiteralControl("&lt;ul&gt;"));
	/// 	foreach(NearestResult nearestResult in MapPointProvider.NearestResults)
	///		{
	/// 		StringBuilder stringBuilder = new StringBuilder();
	/// 		stringBuilder.AppendFormat("&lt;li&gt; {0}, ", nearestResult.Name);
	/// 		if (nearestResult.AddressLine != null)
	/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.AddressLine);
	/// 		if (nearestResult.PrimaryCity != null)
	/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.PrimaryCity);
	/// 		if (nearestResult.SecondaryCity != null)
	/// 			stringBuilder.AppendFormat("{0}, ", nearestResult.SecondaryCity);
	/// 		stringBuilder.AppendFormat("{0}&lt;/li&gt;", nearestResult.PostalCode);
	/// 		pnlSearchResults.Controls.Add(new LiteralControl(stringBuilder.ToString()));
	///		}
	/// 	pnlSearchResults.Controls.Add(new LiteralControl("&lt;/ul&gt;"));
	/// }
	/// </code>
	/// </example>
	[Serializable]
	public sealed class NearestResult
	{
		#region State management fields
		private int resultCount;
		private double score;
		private string name;
		private string addressLine;
		private string primaryCity;
		private string secondaryCity;
		private string postalCode;
		private string countryRegion;
		private double longitude;
		private double latitude;
		#endregion

		#region Constructors
		internal NearestResult(int ResultCount, double Score, string Name,
			string AddressLine, string PrimaryCity, string SecondaryCity, string PostalCode, string CountryRegion,
			double Longitude, double Latitude)
		{
			resultCount = ResultCount;
			score = Score;
			name = Name;
			addressLine = AddressLine;
			primaryCity = PrimaryCity;
			secondaryCity = SecondaryCity;
			postalCode = PostalCode;
			countryRegion = CountryRegion;
			longitude = Longitude;
			latitude = Latitude;
		}
		#endregion

		#region Properties
		/// <summary>
		/// The ResultCount property exposes the count of this result. The count starts from 1 for the result with the lowest search score (i.e. the result closest to the specified location). It increments in order of increasing search score.
		/// </summary>
		public int ResultCount
		{
			get { return resultCount; }
		}

		/// <summary>
		/// The Score property exposes the score attributed to this result by the Mapping Search engine. Lower scores are closer to the specified location.
		/// </summary>
		public double Score
		{
			get { return score; }
		}

		/// <summary>
		/// The Name property exposes the Name of this map entity.
		/// </summary>
		public string Name
		{
			get { return name; }
		}

		/// <summary>
		/// The AddressLine property exposes the 1st line of the address of this map entity.
		/// </summary>
		public string AddressLine
		{
			get { return addressLine; }
		}
		
		/// <summary>
		/// The PrimaryCity property exposes the prmary city (or town) of the address of this map entity.
		/// </summary>
		public string PrimaryCity
		{
			get { return primaryCity; }
		}
		
		/// <summary>
		/// The SecondaryCity property exposes the secondary city of the address of this map entity.
		/// </summary>
		public string SecondaryCity
		{
			get { return secondaryCity; }
		}
		
		/// <summary>
		/// The PostalCode property exposes the postal code of the address of this map entity.
		/// </summary>
		public string PostalCode
		{
			get { return postalCode; }
		}
		
		/// <summary>
		/// The CountryRegion property exposes the country (or region) of the address of this map entity.
		/// </summary>
		public string CountryRegion
		{
			get { return countryRegion; }
		}

		/// <summary>
		/// The Longitude property exposes the longitudinal position of this map entity
		/// </summary>
		public double Longitude
		{
			get { return longitude; }
		}

		/// <summary>
		/// The Latitude property exposes the latitudinal position of this map entity
		/// </summary>
		public double Latitude
		{
			get { return latitude; }
		}
		#endregion
	}
}
