using System;
using System.Text;
using System.Web.UI;
using System.Web.UI.WebControls;
using Microsoft.ContentManagement.WebControls;
using MSIBPlusPack.ContentManagement.Publishing.Placeholders;

namespace NearestExample
{
	/// <summary>
	/// This is a basic example presentation layer for the MSIB Plus Pack Nearest component (overriding NearestPlaceholder).
	/// The CMS template developer specifies the entity type for the search operation.
	/// The CMS author chooses the keywords and range for the nearest search.
	/// When the subscriber (site user) browsers the Posting a tablulated list of nearest results is included with the article.
	/// 
	/// An example implementation of this control would be for a hotel guide web site.
	/// A hotel review template would include this component allowing the author to specify the location of the reviewed hotel.
	/// The published article would include a list of hotels in the vacinity.
	/// 
	/// As previously mentioned, the template developer must specify the entity type. This could be implement in the template ASPX page as:
	/// 	private void Page_Load(object sender, System.EventArgs e
	/// 	{
	/// 		myBasicAlternate.MapPointProvider.EntityTypeValue = "SIC7011";
	/// 	}
	/// </summary>
	public class NearestPlaceholderExampleBasicAlternate : NearestPlaceholder
	{
		#region Controls
		// Authoring
		protected TextBox tbKeywords;
		protected DropDownList ddlRange;

		// Presentation
		protected Panel pnlSearchResults;
		#endregion

		#region NearestPlaceholder override members
		protected override void CreateAuthoringChildControls(BaseModeContainer authoringContainer)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member frist to initialize MapPointProvider
			base.CreateAuthoringChildControls(authoringContainer);

			// Create Search keywords textbox
			tbKeywords = new TextBox();
			tbKeywords.ID = "tbKeywords";
			tbKeywords.Width = 128;
			tbKeywords.MaxLength = 50;			// Give the site subscriber ample space for arbitrary keywords
			tbKeywords.ToolTip = "Please enter your Postcode here";

			// Create Range drop down list:
			ddlRange = new DropDownList();
			ddlRange.ID = "ddlRange";
			ddlRange.Width = 128;
			ddlRange.Items.Add(new ListItem("2 miles","2"));
			ddlRange.Items.Add(new ListItem("5 miles","5"));
			ddlRange.Items.Add(new ListItem("10 miles","10"));
			ddlRange.Items.Add(new ListItem("25 miles","25"));
			ddlRange.Items.Add(new ListItem("80 miles","80"));
			ddlRange.SelectedValue = "10";

			authoringContainer.Controls.Add(new LiteralControl("<b>Nearest MapPoint Search</b>"));
			authoringContainer.Controls.Add(new LiteralControl("<table><tr><td>Postcode:</td><td>"));
			authoringContainer.Controls.Add(tbKeywords);
			authoringContainer.Controls.Add(new LiteralControl("</tr><tr><td>Range:</td><td>"));
			authoringContainer.Controls.Add(ddlRange);
			authoringContainer.Controls.Add(new LiteralControl("</td></tr></table>"));
		}

		protected override void LoadPlaceholderContentForAuthoring(PlaceholderControlEventArgs e)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForAuthoring member first to initialize MapPointProvider
			base.LoadPlaceholderContentForAuthoring(e);

			ddlRange.SelectedValue = MapPointProvider.Range.ToString();
			tbKeywords.Text = MapPointProvider.Keywords;
		}

		protected override void SavePlaceholderContent(PlaceholderControlSaveEventArgs e)
		{
			MapPointProvider.Range = Int32.Parse(ddlRange.SelectedValue);
			MapPointProvider.Keywords = tbKeywords.Text;

			// Call NearestPlaceHolder class's SavePlaceholderContent last to save the MapPointProvider state to the placeholder definition
			base.SavePlaceholderContent(e);
		}

		protected override void CreatePresentationChildControls(BaseModeContainer presentationContainer)
		{
			// Call NearestPlaceHolder class's CreatePresentationChildControls member frist to initialize MapPointProvider
			try { base.CreatePresentationChildControls(presentationContainer); } 
			catch { return; }	// Prevent errors being reported to site subscribers - just remove the control from the page

			// Create search results table:
			pnlSearchResults = new Panel();
			pnlSearchResults.ID = "pnlSearchResults";
			pnlSearchResults.Visible = false;

			presentationContainer.Controls.Add(pnlSearchResults);
		}

		protected override void LoadPlaceholderContentForPresentation(PlaceholderControlEventArgs e)
		{
			// Call NearestPlaceHolder class's LoadPlaceholderContentForPresentation member frist to initialize MapPointProvider
			base.LoadPlaceholderContentForPresentation(e);

			if (MapPointProvider.FindNearest())
			{
				PopulateNearestResults();
				pnlSearchResults.Visible = true;
			}
		}
		#endregion

		#region Presentation nearest result table population
		private void PopulateNearestResults()
		{
			pnlSearchResults.Controls.Clear();

			if (0 == MapPointProvider.NearestResults.Length)
			{
				pnlSearchResults.Controls.Add(new LiteralControl("No items were found near the specified location."));
				return;
			}

			pnlSearchResults.Controls.Add(new LiteralControl("<ul>"));
			foreach(NearestResult nearestResult in MapPointProvider.NearestResults)
			{
				StringBuilder stringBuilder = new StringBuilder();
				stringBuilder.AppendFormat("<li> {0}, ", nearestResult.Name);
				if (nearestResult.AddressLine != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.AddressLine);
				if (nearestResult.PrimaryCity != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.PrimaryCity);
				if (nearestResult.SecondaryCity != null)
					stringBuilder.AppendFormat("{0}, ", nearestResult.SecondaryCity);
				stringBuilder.AppendFormat("{0}</li>", nearestResult.PostalCode);
				pnlSearchResults.Controls.Add(new LiteralControl(stringBuilder.ToString()));
			}
			pnlSearchResults.Controls.Add(new LiteralControl("</ul>"));
		}
		#endregion
	}
}
