//---------------------------------------------------------------------
// <copyright file="Heap.cs" company="Microsoft">
//     Copyright (c) Microsoft Corporation.  All rights reserved.
// </copyright>
// <summary>
// An implementation of a Heap as a .Net collection 
// </summary>
//---------------------------------------------------------------------

namespace Microsoft.Samples.Collections
{
    using System;
    using System.Diagnostics;
    using System.Resources;
    using System.Collections;
    using System.Runtime.Serialization;
    using System.Security;
    using System.Security.Permissions;
    using System.Globalization;

    /// <summary>
    /// A top-heavy heap implementation which keeps its members in a semi-ordered fashion and returns them in an ordered fashion
    /// </summary>
    /// <remarks>
    /// Based on a left-balanced binary tree - implemented on a simple array
    /// The heap grows and shrinks by doubling/halving its size starting with a size of 10
    /// This code was inspired by the interface to the heap implementation presented in 
    /// "Mastering Algorithms with C" by Kyle Loudon, published by O'Reilly &amp; Associates.
    /// </remarks>
    [Serializable]
    public class Heap : ICollection, ICloneable, ISerializable          
    {
        #region Constants
        /// <summary>The initial size for a heap</summary>
        private const int InitialStorageSize = 10;
        #endregion

        #region Static members

        private static ResourceManager rm = new ResourceManager("Microsoft.Samples.Collections.Resources", typeof(Heap).Assembly);

        #endregion

        #region Data members
        
        /// <summary>The elements stored in the heap</summary>
        private object[] nodeArray = new object[Heap.InitialStorageSize];
    
        /// <summary>The number of valid elements that are stored in nodeArray</summary>
        private int size;

        /// <summary>Version stamp to allow enumerators to know if they are invalid</summary>
        private int version = 0;

        /// <summary>How large should the (de)allocation be on auto-sizing operations</summary>
        /// <remarks>The default value of -1 means double or halve the current storage size as needed</remarks>
        private int incrementSize = -1;
        
        /// <summary>Object for comparing entries for ordering.</summary>
        private IComparer comparer = null;

        #endregion

        #region Constructors

        /// <summary>
        /// Default constructor
        /// </summary>
        public Heap() : this(Comparer.Default)
        {
        }
        
        /// <summary>
        /// Construct a heap specifying an object to perform comparisons between heap members
        /// </summary>
        /// <param name="comparer">Object to perform comparisons between heap members.</param>
        public Heap(IComparer comparer)
        {
            this.comparer = comparer;
        }

        /// <summary>
        /// Construct a heap specifying an object to perform comparisons between heap members
        /// and a dynamic allocation size.
        /// </summary>
        /// <remarks>When the heap runs out of storage it will allocate more in chunks sized at incrementSize multiplied by the storage size of object.</remarks>
        /// <remarks>When elements are removed from the heap, if it has too much stoarge by the amount incrementSize it will discard it.</remarks>
        /// <param name="comparer">An object to perform comparisons between heap members.</param>
        /// <param name="incrementSize">The amount to increase and decrease the heap's storage by.</param>
        public Heap(IComparer comparer, int incrementSize)
        {
            this.comparer = comparer;
            this.incrementSize = incrementSize;
        }

        #endregion

        #region Static Functions
        
        /// <summary>
        /// Returns a synchronized (thread-safe) wrapper for the Heap.
        /// </summary>
        /// <remarks>
        /// To guarantee the thread safety of the Heap, all operations must be done through this wrapper.
        /// Enumerating through a collection is intrinsically not a thread-safe procedure. Even 
        /// when a collection is synchronized, other threads could still modify the collection,
        /// which causes the enumerator to throw an exception. To guarantee thread safety during
        /// enumeration, you can either lock the collection during the entire enumeration or
        /// catch the exceptions resulting from changes made by other threads.
        /// 
        /// The following code example shows how to lock the collection using the SyncRoot during
        /// the entire enumeration:
        /// <code>
        /// Heap myCollection = new Heap();
        /// lock( myCollection.SyncRoot ) 
        /// {
        ///     foreach ( Object item in myCollection ) 
        ///     {
        ///         // Insert your code here.
        ///     }
        /// }
        /// </code>
        /// </remarks>
        /// <exception cref="ArgumentNullException">heap is a null reference (Nothing in Visual Basic).</exception>
        /// <param name="heap">The Heap to synchronize.</param>
        /// <returns>A synchronized wrapper around the Heap.</returns>
        public static Heap Synchronized(Heap heap)
        {
            if (heap == null)
            {
                throw new ArgumentNullException("heap");
            }
            return new SynchronizedHeap(heap);
        }
        #endregion

        #region Primary public interface

        /// <summary>
        /// Add an object to the heap maintaining its heap properties
        /// </summary>
        /// <param name="data">The data to add to the heap</param>
        public virtual void Add(object data)
        {

            if (this.size >= this.nodeArray.Length) 
            {
                this.GrowStorage();
            }

            this.nodeArray[this.size] = data; // Place the new value at the end then do Re-Heaping if necessary
            int dataIndex = this.size; // Keep track of where we put the new item

            bool unheaped = true;  // Assume the worst
            while (unheaped && dataIndex > 0)
            {
                int parent = this.Parent(dataIndex); 
                object parentObj = this.nodeArray[parent];

                unheaped = this.comparer.Compare(parentObj, data) < 0;

                if (unheaped)
                {
                    // Swap data with its parent.
                    this.nodeArray[parent] = data;
                    this.nodeArray[dataIndex] = parentObj;
                }
                dataIndex = parent;
            }
            this.size++;
            this.version++;
        }

        /// <summary>
        /// Extract returns the highest value object from the heap
        /// </summary>
        /// <exception cref="System.InvalidOperationException">Thrown when the Heap is empty.</exception>
        /// <remarks>
        /// The highest value is defined as the highest that would be returned by sorting using the IComparer provided at construction time or else the system default Comparer.
        /// This implementation uses the locale-sensitive default comparer.
        /// </remarks>
        /// <returns>The highest value object from the heap</returns>
        public virtual object Extract()
        {
            if (this.size == 0) 
            {
                throw new InvalidOperationException(rm.GetString("Extract_HeapEmpty", CultureInfo.InvariantCulture));
            }

            // Put the old last node at the head of the tree
            object data = this.nodeArray[0];
            this.nodeArray[0] = this.nodeArray[this.size - 1];
            this.nodeArray[this.size - 1] = null;
            this.size--;

            // If we've now a whole increment size too much storage then remove some
            if ((this.size + this.IncrementSize) < this.nodeArray.Length) 
            {
                this.FlushStorage();
            }

            // Re-heapify if necessary
            int parent = 0;
            bool unheaped = true;

            while (unheaped && !this.LeafNode(parent)) 
            {
                // If the parent is less than either child
                unheaped = this.comparer.Compare(this.nodeArray[parent], this.nodeArray[this.LeftChild(parent)]) < 0 ||
                    ((this.RightChild(parent) < this.size) && (this.comparer.Compare(this.nodeArray[parent], this.nodeArray[this.RightChild(parent)]) < 0)); 
                if (unheaped)
                {
                    // Decide which side is more unheaped
                    // If there is only a left side then we must progress that way
                    // If there are both then its the most different node.
                    object temp = this.nodeArray[parent];
                    bool leftSide = this.RightChild(parent) >= this.size;
                    leftSide = leftSide || this.comparer.Compare(this.nodeArray[this.LeftChild(parent)], this.nodeArray[this.RightChild(parent)]) > 0; 
                    if (leftSide)
                    {
                        this.nodeArray[parent] = this.nodeArray[this.LeftChild(parent)];
                        this.nodeArray[this.LeftChild(parent)] = temp;
                        parent = this.LeftChild(parent);
                    }
                    else // go right
                    {
                        this.nodeArray[parent] = this.nodeArray[this.RightChild(parent)];
                        this.nodeArray[this.RightChild(parent)] = temp;
                        parent = this.RightChild(parent);
                    }
                }
            }
            this.version++;
            return data;
        }

        /// <summary>
        /// Look at the element at the top of the heap without removing it.
        /// </summary>
        /// <exception cref="System.InvalidOperationException">Thrown when the Heap is empty.</exception>
        /// <returns>The element at the top of the heap.</returns>
        public virtual object Peek()
        {
            if (this.size == 0) 
            {
                throw new InvalidOperationException(rm.GetString("Extract_HeapEmpty", CultureInfo.InvariantCulture));
            }

            return this.nodeArray[0];
        }

        /// <summary>
        /// Test whether the heap is empty of elements.
        /// </summary>
        /// <returns>Whether the heap is empty.</returns>
        public virtual bool IsEmpty()
        {
            return this.size == 0;
        }

        /// <summary>
        /// Create a copy of the heap containing the same object references
        /// </summary>
        /// <returns>A copy of the heap</returns>
        public virtual Heap Clone()
        {
            // Start with a shallow copy and then deepcopy the array and reset the version
            Heap temp = (Heap)this.MemberwiseClone();
            temp.nodeArray = new object[this.Count];
            this.CopyTo(temp.nodeArray, 0);
            temp.version = 0;
            return temp;
        }

        /// <summary>
        /// Removes all elements from the Heap.
        /// </summary>
        /// <remarks>
        /// Count is set to zero. The capacity remains unchanged.
        /// </remarks>
        public virtual void Clear()
        {
            if (this.size > 0)
            {
                for (int i = 0; i < this.size; i++)
                {
                    this.nodeArray[i] = null;
                }
                this.size = 0;
                this.version++;
            }
        }

        /// <summary>
        /// Determines whether an element is in the Heap.
        /// </summary>
        /// <remarks>
        /// This method performs with average execution time that is linear. That is, this method is an O(n) operation, where n is Count.
        /// This method determines equality by calling Object.Equals.
        /// </remarks>
        /// <param name="obj">The Object to locate in the Heap. The value can be a null reference (Nothing in Visual Basic).</param>
        /// <returns>true if obj is found in the Heap; otherwise, false.</returns>
        public virtual bool Contains(object obj)
        {
            if (this.size == 0) 
            {
                return false;
            }

            // Start at the head of the tree
            int seek = 0;
            while (seek < this.size)
            {
                if (obj.Equals(this.nodeArray[seek]))
                {
                    return true;
                }
                seek++;
            }
            return false;
        }

        #endregion

        #region Private random access functions

        /// <summary>
        /// The parent node of the given node
        /// </summary>
        /// <param name="index">the index of the given node</param>
        /// <returns>the parent node</returns>
        private int Parent(int index)
        {
            return (int)((index - 1) / 2);
        }

        /// <summary>
        /// The left-hand child of the given node
        /// </summary>
        /// <param name="index">the index of the given node</param>
        /// <returns>the left-hand child node</returns>
        private int LeftChild(int index)
        {
            return (index * 2) + 1;
        }

        /// <summary>
        /// The right-hand child of the given node
        /// </summary>
        /// <param name="index">the index of the given node</param>
        /// <returns>the right-hand child node</returns>
        private int RightChild(int index)
        {
            return (index * 2) + 2;
        }

        /// <summary>
        /// Whether the given node is a leaf node
        /// </summary>
        /// <param name="index">the index of the given node</param>
        /// <returns>true if the given node is a leaf node</returns>
        private bool LeafNode(int index)
        {
            return this.LeftChild(index) >= this.size;
        }

        #endregion

        #region Internal storage management functions

        /// <summary>
        /// The calculated value of the increment size
        /// </summary>
        private int IncrementSize
        {
            get
            {
                if (this.incrementSize == -1)
                {
                    return this.nodeArray.Length;
                }
                return this.incrementSize;
            }
        }

        /// <summary>
        /// Add new storage
        /// </summary>
        private void GrowStorage()
        {
            object[] temp = new object[this.nodeArray.Length + this.IncrementSize];
            this.nodeArray.CopyTo(temp, 0);
            this.nodeArray = temp;
        }

        /// <summary>
        /// Remove unused storage
        /// </summary>
        private void FlushStorage()
        {
            object[] temp = new object[this.nodeArray.Length - this.IncrementSize];
            Debug.Assert(temp.Length >= this.size);
            for (int i = 0; i < this.size; i++)
            {
                temp[i] = this.nodeArray[i];
            }
            this.nodeArray = temp;
        }

        #endregion

        #region ICollection Members

        /// <summary>
        /// Gets a value indicating whether access to the heap is synchronized (thread-safe).
        /// As this is an unsynchronized heap, it will aways return false.
        /// </summary>
        /// <value>Whether this Heap is synchronized</value>
        public virtual bool IsSynchronized
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Returns a value indicating the number of objects in the heap. Read-only
        /// </summary>
        /// <value>the number of objects in the heap</value>
        public virtual int Count
        {
            get
            {
                return this.size;
            }
        }

        /// <summary>
        /// Copy all the elements stored on the current heap to the specified one-dimensional array starting at the specified index
        /// </summary>
        /// <exception cref="ArgumentException">Thrown if the array is of the wrong size or rank.</exception>
        /// <param name="array">The array to copy to</param>
        /// <param name="index">The index to start from</param>
        public virtual void CopyTo(Array array, int index)
        {
            Debug.Assert(index < this.size);
            Debug.Assert(array.Rank == 1);
            Debug.Assert(array.Length >= (this.size - index));

            if (index >= this.size) 
            {
                throw new ArgumentException(rm.GetString("ArgumentRange_Index", CultureInfo.InvariantCulture), "index");
            }

            if (array.Rank != 1)
            {
                throw new ArgumentException(rm.GetString("ArgumentRank_Array", CultureInfo.InvariantCulture), "array");
            }

            if (array.Length < (this.size - index))
            {
                throw new ArgumentException(rm.GetString("ArgumentCapacity_Array", CultureInfo.InvariantCulture), "array");
            }

            for (int i = index; i < this.size; i++)
            {
                array.SetValue(this.nodeArray[i], (array.GetLowerBound(0) + i - index));
            }
        }

        /// <summary>
        /// Gets an object that can be used to synchronize access to the heap
        /// </summary>
        /// <value>object to be used to synchronize access to the heap</value>
        public virtual object SyncRoot
        {
            get
            {
                // Its appropriate to return this as even if an external thread locks
                // our instance pointer, (say as the SyncRoot of a containing class)
                // then it would almost certainly mean they didn't want us to change.
                // An alternative would be to have a private object member specifically for this job.
                return this;
            }
        }

        /// <summary>
        /// Returns an enumerator that can iterate through the heap in an unordered fashion.
        /// </summary>
        /// <returns>an enumerator that can iterate through a heap</returns>
        public virtual IEnumerator GetEnumerator()
        {
            return new HeapEnumerator(this);
        }

        #endregion

        #region IEnumerable Members

        /// <summary>
        /// Returns an enumerator that can iterate through the heap in an unordered fashion.
        /// </summary>
        /// <returns>an enumerator that can iterate through a heap</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return ((Heap)this).GetEnumerator();
        }

        #endregion

        #region ICloneable Members

        /// <summary>
        /// Create a copy of the heap containing the same object references
        /// </summary>
        /// <returns>a copy of the heap</returns>
        object ICloneable.Clone()
        {
            return ((Heap)this).Clone();

        }

        #endregion

        #region ISerializable Members

        /// <summary>
        /// Populates a SerializationInfo with the data needed to serialize the heap.
        /// </summary>
        /// <exception cref="ArgumentNullException">Thrown if the info parameter is null.</exception>
        /// <param name="info">The SerializationInfo to populate with data.</param>
        /// <param name="context">The destination for this serialization.</param>
        [SecurityPermissionAttribute(SecurityAction.Demand,SerializationFormatter=true)]
        public virtual void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            if (info == null)
            {
                throw new ArgumentNullException("info");
 
            }

            object[] usedNodeArray;

            info.AddValue("IncrementSize", this.incrementSize);
            info.AddValue("Version", this.version);
            info.AddValue("Comparer", this.comparer, typeof(IComparer));
            info.AddValue("Size", this.size);

            usedNodeArray = new object[this.size];
            this.CopyTo(usedNodeArray, 0);

            info.AddValue("Nodes", usedNodeArray, typeof(object[]));
        } 

        /// <summary>
        /// Special constructor for serialization.
        /// </summary>
        /// <exception cref="ArgumentNullException">Thrown if the info parameter is null.</exception>
        /// <exception cref="SerializationException">Thrown if the stream being serialized is inconsistent.</exception>
        /// <param name="info">The SerializationInfo to construct a Heap from.</param>
        /// <param name="context">The source of this serialization.</param>
        [SecurityPermissionAttribute(SecurityAction.Demand,SerializationFormatter=true)]
        protected Heap(SerializationInfo info, StreamingContext context)
        {
            if (this.size != 0)
            {
                return; 
            }

            if (info == null)
            {
                throw new ArgumentNullException("info");
            }

            this.incrementSize = info.GetInt32("IncrementSize");
            this.version = info.GetInt32("Version");
            this.comparer = ((IComparer) info.GetValue("Comparer", typeof(IComparer)));
            this.size = info.GetInt32("Size");
            this.nodeArray = ((object[]) info.GetValue("Nodes", typeof(object[])));
            if (this.nodeArray == null)
            {
                throw new SerializationException(rm.GetString("Serialization_MissingNodes", CultureInfo.InvariantCulture));
            }
            if (this.nodeArray.Length != this.size)
            {
                throw new SerializationException(rm.GetString("Serialization_UnmatchedNodes", CultureInfo.InvariantCulture));
            }
        }

        #endregion

        #region Enumerator
        
        /// <summary>
        /// Simple unordered enumerator for heaps
        /// </summary>
        private class HeapEnumerator : IEnumerator, ICloneable
        {
            /// <summary>The heap being enumerated</summary>
            private Heap heap;

            /// <summary>How far through the heap we are</summary>
            private int index = -1;

            /// <summary>Version stamp to allow enumerators to know if they are invalid</summary>
            private int version;

            /// <summary>
            /// Constructor solely for the use of the GetEnumerator method on the heap
            /// </summary>
            /// <param name="heap">The heap to enumerate</param>
            internal HeapEnumerator(Heap heap)
            {
                this.heap = heap;
                this.version = heap.version;
            }

            #region IEnumerator Members

            /// <summary>
            /// Sets the enumerator to its initial position, which is before the first element in the collection.
            /// </summary>
            void IEnumerator.Reset()
            {
                if (this.version != this.heap.version)
                {
                    throw new InvalidOperationException(rm.GetString("Enumerator_HeapModified", CultureInfo.InvariantCulture));
                }
                this.index = -1;
            }

            /// <summary>
            /// Gets the current element in the heap being enumerated.
            /// </summary>
            object IEnumerator.Current
            {
                get
                {
                    if (this.index < 0 || this.index >= this.heap.Count)
                    {
                        throw new InvalidOperationException(rm.GetString("Emumerator_Range", CultureInfo.InvariantCulture));
                    }

                    return this.heap.nodeArray[this.index];
                }
            }

            /// <summary>
            /// Advances the enumerator to the next element of the heap.
            /// </summary>
            /// <returns>true if the enumerator was successfully advanced to the next element; false if the enumerator has passed the end of the heap.</returns>
            bool IEnumerator.MoveNext()
            {
                if (this.version != this.heap.version)
                {
                    throw new InvalidOperationException(rm.GetString("Enumerator_HeapModified", CultureInfo.InvariantCulture));
                }
                this.index++;
                return this.index < this.heap.Count;
            }

            #endregion

            #region ICloneable Members

            /// <summary>
            /// Create a copy of the enumerator referencing the same hashtable and position.
            /// </summary>
            /// <returns>a copy of the enumerator</returns>
            object ICloneable.Clone()
            {
                return this.MemberwiseClone();
            }

            #endregion
        }

        #endregion

        #region Synchronized Wrapper

        /// <summary>
        /// Simple class to provide synchronized versions of the Heap API 
        /// </summary>
        private class SynchronizedHeap : Heap
        {
            private Heap wrappedHeap;
            private object syncBase;

            /// <summary>
            /// Constructor
            /// </summary>
            /// <param name="toWrap">The heap to provide a synchronized wrapped for</param>
            public SynchronizedHeap(Heap toWrap) : base()
            {
                this.wrappedHeap = toWrap;
                this.syncBase = toWrap.SyncRoot;
            }

            /// <summary>
            /// Add an object to the heap maintaining its heap properties
            /// </summary>
            /// <param name="data">The data to add to the heap</param>
            public override void Add(object data)
            {
                lock (this.syncBase)
                {
                    this.wrappedHeap.Add(data);
                }
            }

            /// <summary>
            /// Extract returns the highest value object from the heap
            /// </summary>
            /// <exception cref="System.InvalidOperationException">Thrown when the Heap is empty.</exception>
            /// <remarks>
            /// The highest value is defined as the highest that would be returned by sorting using the IComparer provided at construction time or else the system default Comparer.
            /// This implementation uses the locale-sensitive default comparer.
            /// </remarks>
            /// <returns>The highest value object from the heap</returns>
            public override object Extract()
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.Extract();
                }
            }

            /// <summary>
            /// Look at the element at the top of the heap without removing it.
            /// </summary>
            /// <exception cref="System.InvalidOperationException">Thrown when the Heap is empty.</exception>
            /// <returns>The element at the top of the heap.</returns>
            public override object Peek()
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.Peek();
                }
            }

            /// <summary>
            /// Test whether the heap is empty of elements.
            /// </summary>
            /// <returns>Whether the heap is empty.</returns>
            public override bool IsEmpty()
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.IsEmpty();
                }
            }

            /// <summary>
            /// Create a copy of the heap containing the same object references
            /// </summary>
            /// <returns>A copy of the heap</returns>
            public override Heap Clone()
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.Clone();
                }
            }

            /// <summary>
            /// Removes all elements from the Heap.
            /// </summary>
            /// <remarks>
            /// Count is set to zero. The capacity remains unchanged.
            /// </remarks>
            public override void Clear()
            {
                lock (this.syncBase)
                {
                    this.wrappedHeap.Clear();
                }
            
            }

            /// <summary>
            /// Determines whether an element is in the Heap.
            /// </summary>
            /// <remarks>
            /// This method performs with average execution time that is linear. That is, this method is an O(n) operation, where n is Count.
            /// This method determines equality by calling Object.Equals.
            /// </remarks>
            /// <param name="obj">The Object to locate in the Heap. The value can be a null reference (Nothing in Visual Basic).</param>
            /// <returns>true if obj is found in the Heap; otherwise, false.</returns>
            public override bool Contains(object obj)
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.Contains(obj);
                }
            }

            /// <summary>
            /// Gets a value indicating whether access to the heap is synchronized (thread-safe).
            /// As this is the synchronization wrapper, it will aways return true.
            /// </summary>
            /// <value>Whether this Heap is synchronized</value>
            public override bool IsSynchronized
            {
                get
                {
                    return true;
                }
            }

            /// <summary>
            /// Returns a value indicating the number of objects in the heap. Read-only
            /// </summary>
            /// <value>the number of objects in the heap</value>
            public override int Count
            {
                get
                {
                    lock (this.syncBase)
                    {
                        return this.wrappedHeap.Count;
                    }
                }
            }
            
            /// <summary>
            /// Copy all the elements stored on the current heap to the specified one-dimensional array starting at the specified index
            /// </summary>
            /// <exception cref="ArgumentException">Thrown if the array is of the wrong size or rank.</exception>
            /// <param name="array">The array to copy to</param>
            /// <param name="index">The index to start from</param>
            public override void CopyTo(Array array, int index)
            {
                lock (this.syncBase)
                {
                    this.wrappedHeap.CopyTo(array, index);
                }
            }
        
            /// <summary>
            /// Gets an object that can be used to synchronize access to the heap
            /// </summary>
            /// <value>object to be used to synchronize access to the heap</value>
            public override object SyncRoot
            {
                get 
                {
                    return this.syncBase;
                }
            }

            /// <summary>
            /// Returns an enumerator that can iterate through the heap in an unordered fashion.
            /// </summary>
            /// <returns>an enumerator that can iterate through a heap</returns>
            public override IEnumerator GetEnumerator()
            {
                lock (this.syncBase)
                {
                    return this.wrappedHeap.GetEnumerator();
                }
            }
    
            /// <summary>
            /// Special constructor for serialization.
            /// </summary>
            /// <exception cref="ArgumentNullException">Thrown if the info parameter is null.</exception>
            /// <exception cref="SerializationException">Thrown if the stream being serialized is inconsistent.</exception>
            /// <param name="info">The SerializationInfo to construct a Heap from.</param>
            /// <param name="context">The source of this serialization.</param>
            [SecurityPermissionAttribute(SecurityAction.Demand,SerializationFormatter=true)]
            protected SynchronizedHeap(SerializationInfo info, StreamingContext context) : base(info, context)
            {
                this.wrappedHeap = ((Heap)info.GetValue("WrappedHeap", typeof(Heap)));
                this.syncBase = this.wrappedHeap.SyncRoot;
                if (this.wrappedHeap == null)
                {
                    throw new SerializationException(rm.GetString("Serialization_MissingNodes", CultureInfo.InvariantCulture));
                }
            }

            /// <summary>
            /// Populates a SerializationInfo with the data needed to serialize the heap.
            /// </summary>
            /// <exception cref="ArgumentNullException">Thrown if the info parameter is null.</exception>
            /// <param name="info">The SerializationInfo to populate with data.</param>
            /// <param name="context">The destination for this serialization.</param>
            [SecurityPermissionAttribute(SecurityAction.Demand,SerializationFormatter=true)]
            public override void GetObjectData(SerializationInfo info, StreamingContext context)
            {
                if (info == null)
                {
                    throw new ArgumentNullException("info");
                }
                info.AddValue("WrappedHeap", this.wrappedHeap, typeof(Heap));
            } 
        }

        #endregion
    }
}
