using System;
using System.Drawing;
using System.Windows.Forms;
using Pikhulya;

namespace Pikhulya.GotDotNet.Samples.MyZoomer
{
	/// <summary>
	/// Main application form
	/// </summary>
	public class FormMain: Form
	{
		PanelView panelView;
		Point desktopDimension;

		TrackBar trackBarZoomRate;
		GroupBox gbScale;
		Panel panelCurrentColor;
		GroupBox gbCurrentColor;
		Label lblRed;
		Label lblGreen;
		Label lblBlue;
		Button btSave;
		Label lblNameBlue;
		Label lblNameGreen;
		Label lblNameRed;
		readonly string bitmapFileName;

		public FormMain()
		{
			CreateControls();
			bitmapFileName = Environment.GetFolderPath(System.Environment.SpecialFolder.Personal) + @"\Capture.bmp";
			panelView.CurrentColorChanged += new PanelView.CurrentColorChangedEventHandler(OnCurrentColorChanged);
			panelView.CaptureStopped += new PanelView.CaptureStoppedEventHandler(OnCaptureStopped);
			Microsoft.Win32.SystemEvents.DisplaySettingsChanged += new EventHandler(OnDisplaySettingsChanged);

			OnDisplaySettingsChanged(this, EventArgs.Empty);
		}

		void OnCurrentColorChanged(object sender, PanelView.CurrentColorChangedEventArgs e)
		{
			panelCurrentColor.BackColor = e.Color;
			lblRed.Text = e.Color.R.ToString();
			lblGreen.Text = e.Color.G.ToString();
			lblBlue.Text = e.Color.B.ToString();
		}

		//called after capturing has been stopped
		void OnCaptureStopped(object sender, PanelView.CaptureStoppedEventArgs e)
		{
			SetMaxSize(e.CaptureRectangle);
		}

		//Sets maximum form size
		void SetMaxSize(Rectangle captureRectangle)
		{
			int zoomRate = trackBarZoomRate.Value;
			MaximumSize = new Size(Width + ((desktopDimension.X - 1 - captureRectangle.Right) * zoomRate), 
				Height + ((desktopDimension.Y - 1 - captureRectangle.Bottom) * zoomRate));
		}

		//called after display resolution has been changed
		void OnDisplaySettingsChanged(object sender, EventArgs e)
		{
			desktopDimension = new Point(Screen.PrimaryScreen.Bounds.Right,
				Screen.PrimaryScreen.Bounds.Bottom);
			panelView.SetDisplaySettings(desktopDimension);
		}

		void CreateControls()
		{
			panelView = new PanelView();
			trackBarZoomRate = new TrackBar();
			gbScale = new GroupBox();
			panelCurrentColor = new Panel();
			gbCurrentColor = new GroupBox();
			lblBlue = new Label();
			lblGreen = new Label();
			lblRed = new Label();
			lblNameBlue = new Label();
			lblNameGreen = new Label();
			lblNameRed = new Label();
			btSave = new Button();

			gbScale.SuspendLayout();
			gbCurrentColor.SuspendLayout();
			SuspendLayout();
			
			panelView.Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right;
			panelView.BorderStyle = BorderStyle.FixedSingle;
			panelView.Location = new Point(8, 16);
			panelView.Size = new Size(200, 200);
			panelView.TabIndex = 0;

			trackBarZoomRate.Anchor = AnchorStyles.Top | AnchorStyles.Right;
			trackBarZoomRate.LargeChange = 3;
			trackBarZoomRate.Location = new Point(5, 16);
			trackBarZoomRate.Minimum = 1;
			trackBarZoomRate.Size = new Size(104, 45);
			trackBarZoomRate.TabIndex = 0;
			trackBarZoomRate.Value = 1;
			trackBarZoomRate.ValueChanged += new EventHandler(TrackBarZoomRateValueChanged);

			gbScale.Anchor = (System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right);
			gbScale.Controls.Add(trackBarZoomRate);
			gbScale.FlatStyle = System.Windows.Forms.FlatStyle.System;
			gbScale.Location = new System.Drawing.Point(216, 8);
			gbScale.Size = new System.Drawing.Size(112, 64);
			gbScale.TabIndex = 1;
			gbScale.TabStop = false;
			gbScale.Text = "Sca&le:";

			panelCurrentColor.BorderStyle = BorderStyle.Fixed3D;
			panelCurrentColor.Location = new Point(9, 16);
			panelCurrentColor.Size = new Size(95, 24);
			panelCurrentColor.TabIndex = 3;

			gbCurrentColor.Anchor = AnchorStyles.Top | AnchorStyles.Right;
			gbCurrentColor.Controls.AddRange(new System.Windows.Forms.Control[] {
																					lblBlue, lblGreen, lblRed, lblNameBlue, lblNameGreen, lblNameRed, panelCurrentColor});

			gbCurrentColor.FlatStyle = FlatStyle.System;
			gbCurrentColor.Location = new Point(216, 80);
			gbCurrentColor.Size = new Size(112, 104);
			gbCurrentColor.TabIndex = 4;
			gbCurrentColor.TabStop = false;
			gbCurrentColor.Text = "Current color:";

			lblBlue.FlatStyle = FlatStyle.System;
			lblBlue.Location = new Point(56, 80);
			lblBlue.Size = new Size(32, 16);
			lblBlue.TabIndex = 9;
			lblBlue.Text = "0";

			lblGreen.FlatStyle = FlatStyle.System;
			lblGreen.Location = new Point(56, 64);
			lblGreen.Size = new Size(32, 16);
			lblGreen.TabIndex = 8;
			lblGreen.Text = "0";

			lblRed.FlatStyle = FlatStyle.System;
			lblRed.Location = new Point(56, 48);
			lblRed.Size = new System.Drawing.Size(32, 16);
			lblRed.TabIndex = 7;
			lblRed.Text = "0";

			lblNameBlue.FlatStyle = FlatStyle.System;
			lblNameBlue.ForeColor = Color.Blue;
			lblNameBlue.Location = new Point(8, 80);
			lblNameBlue.Size = new Size(40, 16);
			lblNameBlue.TabIndex = 6;
			lblNameBlue.Text = "Blue:";

			lblNameGreen.FlatStyle = FlatStyle.System;
			lblNameGreen.ForeColor = Color.Green;
			lblNameGreen.Location = new Point(8, 64);
			lblNameGreen.Size = new Size(40, 16);
			lblNameGreen.TabIndex = 5;
			lblNameGreen.Text = "Green:";

			lblNameRed.FlatStyle = FlatStyle.System;
			lblNameRed.ForeColor = Color.Red;
			lblNameRed.Location = new Point(8, 48);
			lblNameRed.Size = new Size(40, 16);
			lblNameRed.TabIndex = 4;
			lblNameRed.Text = "Red:";

			btSave.Anchor = AnchorStyles.Bottom | AnchorStyles.Right;
			btSave.FlatStyle = FlatStyle.System;
			btSave.Location = new Point(252, 192);
			btSave.TabIndex = 5;
			btSave.Text = "&Save";
			btSave.Click += new EventHandler(BtSaveClick);

			AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			ClientSize = new Size(336, 222);
			Controls.AddRange(new Control[] {btSave,  gbCurrentColor, gbScale, panelView});

			MaximizeBox = false;
			MinimumSize = new Size(344, 250);
			Text = "MyZoomer";

			gbScale.ResumeLayout(false);
			gbCurrentColor.ResumeLayout(false);
			ResumeLayout(false);
		}

		static void Main() 
		{
			Application.Run(new FormMain());
		}

		
		//Handling form deactivation for stopping capture
		protected override void OnDeactivate(EventArgs e)
		{
			panelView.FormDeactivated();		
			base.OnDeactivate(e);
		}

		//Zoom rate changed
		void TrackBarZoomRateValueChanged(object sender, System.EventArgs e)
		{
			SetMaxSize(panelView.CaptureRectangle);
			panelView.ZoomRate = trackBarZoomRate.Value;
		}

		void BtSaveClick(object sender, System.EventArgs e)
		{
			Graphics graphicsPanel = panelView.CreateGraphics();
			Bitmap bitmap = CreateBitmapFromGraphics(graphicsPanel, 0, 0, panelView.Width, 
				panelView.Height);
			try
			{
				bitmap.Save(bitmapFileName);
			}
			catch(Exception exception)
			{
				MessageBox.Show(exception.Message, Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			finally
			{
				bitmap.Dispose();
				graphicsPanel.Dispose();
			}
		}

		
		Bitmap CreateBitmapFromGraphics(Graphics graphicsSource, int x, int y, int width, int height)
		{
			IntPtr hDCSource = graphicsSource.GetHdc();
			IntPtr hDCDestination = Win32.CreateCompatibleDC(hDCSource);
			IntPtr hBitmap = Win32.CreateCompatibleBitmap(hDCSource, width, height);
			IntPtr hPreviousBitmap = Win32.SelectObject(hDCDestination, hBitmap);
			Win32.BitBlt(hDCDestination, 0, 0, width, height, hDCSource, x, y, Win32.SRCCOPY);
			Bitmap  bitmap = Bitmap.FromHbitmap(hBitmap);
			Win32.DeleteObject(hDCDestination);
			Win32.DeleteObject(hBitmap);
			graphicsSource.ReleaseHdc(hDCSource);
			return bitmap;
		}
		
		/// <summary>
		/// All drawing occurs in this panel
		/// </summary>
		class PanelView: Panel
		{
			public class CurrentColorChangedEventArgs: EventArgs
			{
				public CurrentColorChangedEventArgs(Color color)
				{
					this.color = color;
				}

				Color color;
				public Color Color
				{
					get
					{
						return color;
					}
				}
			}
			
			public delegate void CurrentColorChangedEventHandler(object sender, CurrentColorChangedEventArgs e);
			public event CurrentColorChangedEventHandler CurrentColorChanged;

			public class CaptureStoppedEventArgs: EventArgs
			{
				Rectangle captureRectangle;
				public Rectangle CaptureRectangle
				{
					get
					{
						return captureRectangle;
					}
				}

				public CaptureStoppedEventArgs(Rectangle captureRectangle)
				{
					this.captureRectangle = captureRectangle;
				}
			}

			public delegate void CaptureStoppedEventHandler(object sender, CaptureStoppedEventArgs e);
			public event CaptureStoppedEventHandler CaptureStopped;
			
			Color previousColor;

			bool needGrab;
			ScreenCapturer capturer;

			/// <summary>
			/// Zoom rate
			/// </summary>
			public int ZoomRate
			{
				set
				{
					capturer.ZoomRate = value;
					Invalidate();
				}
			}

			public PanelView()
			{
				SetStyle(ControlStyles.ResizeRedraw, true);
				capturer = new ScreenCapturer(this);
			}

			public Rectangle CaptureRectangle
			{
				get
				{
					return capturer.CaptureRectangle;
				}
			}
			
			protected override void OnMouseDown(MouseEventArgs e)
			{
				if (e.Button == MouseButtons.Left)
				{
					capturer.BeginCapture(new Point(e.X, e.Y));
					needGrab = true;
				}

				base.OnMouseDown(e);
			}

			protected override void OnMouseMove(MouseEventArgs e)
			{
				if (needGrab)
					capturer.CopyImage(new Point(e.X, e.Y));
				else
				{
					Graphics graphics = CreateGraphics();
					IntPtr hDC = graphics.GetHdc();
					Color color = Win32.GetPixelColor(hDC, e.X, e.Y);
					if (color != previousColor)
					{
						if (CurrentColorChanged != null)
							CurrentColorChanged(this, new CurrentColorChangedEventArgs(color));
						previousColor = color;
					}
					graphics.ReleaseHdc(hDC);
				}
				
				base.OnMouseMove(e);
			}

			protected override void OnMouseUp(MouseEventArgs e)
			{
				if (needGrab)
					if (e.Button == MouseButtons.Left)
					{
						Rectangle captureRectangle = capturer.StopCapture();
						needGrab = false;
						if (CaptureStopped != null)
							CaptureStopped(this, new CaptureStoppedEventArgs(captureRectangle));
					}
				base.OnMouseUp(e);
			}

			protected override void OnPaint(PaintEventArgs e)
			{
				capturer.PaintControl(e.Graphics);
				base.OnPaint(e);
			}

			public void FormDeactivated()
			{
				if (!needGrab)
					return;
				Rectangle captureRectangle = capturer.StopCapture();
				if (CaptureStopped != null)
					CaptureStopped(this, new CaptureStoppedEventArgs(captureRectangle));
			}

			public void SetDisplaySettings(Point desktopDimension)
			{
				capturer.DesktopDimension = desktopDimension;
			}
		}
	}

	class ScreenCapturer
	{
		Rectangle oldRectangle;
		readonly IntPtr hWndDesktop;
		int desktopRight;
		int desktopBottom;
		bool isFormDeactivated;
		bool needGrab;
		Control control;
		int zoomRate = 1;

		public int ZoomRate
		{
			set
			{
				zoomRate = value;
			}
		}

		public Rectangle CaptureRectangle
		{
			get
			{
				return oldRectangle;
			}
		}

		public ScreenCapturer(Control control)
		{
			this.control = control;
			hWndDesktop = IntPtr.Zero;
			desktopRight= Screen.PrimaryScreen.Bounds.Right - 1;
			desktopBottom = Screen.PrimaryScreen.Bounds.Bottom - 1;
		}

		public void BeginCapture(Point cursorPosition)
		{
			needGrab = true;
			Rectangle rectangle = GetZoomedRectangle(cursorPosition.X, cursorPosition.Y);
			DrawRectangle(rectangle);
			oldRectangle = rectangle;
		}

		public Rectangle StopCapture()
		{
			if (needGrab || isFormDeactivated)
			{
				if (isFormDeactivated)
					isFormDeactivated = false;

				DrawRectangle(oldRectangle);
			}
			needGrab = false;
			return oldRectangle;
		}

		public void CopyImage(Point cursorPosition)
		{
			if (!needGrab)
				return;

			Rectangle rectangle = GetZoomedRectangle(cursorPosition.X, cursorPosition.Y);
			DrawRectangle(oldRectangle);
			CopyDesktopRectangle(rectangle.Left, rectangle.Top, null);
			DrawRectangle(rectangle);
			oldRectangle = rectangle;
		}

		public void PaintControl(Graphics graphics)
		{
			CopyDesktopRectangle(oldRectangle.Left, oldRectangle.Top, graphics);
		}

		//Draws rectangle onto desktop
		void DrawRectangle(Rectangle rectangle)
		{
			IntPtr hDCDesktop = Win32.GetDC(hWndDesktop);
			Win32.SetROP2(hDCDesktop, Win32.R2_NOT);
				
			Win32.MoveTo(hDCDesktop, rectangle.Left, rectangle.Top);
			Win32.LineTo(hDCDesktop, rectangle.Right, rectangle.Top);
			Win32.LineTo(hDCDesktop, rectangle.Right, rectangle.Bottom);
			Win32.LineTo(hDCDesktop, rectangle.Left, rectangle.Bottom);
			Win32.LineTo(hDCDesktop, rectangle.Left, rectangle.Top);

			Win32.SetROP2(hDCDesktop, Win32.R2_NOP);
			Win32.ReleaseDC(hWndDesktop, hDCDesktop);
		}

		Rectangle GetZoomedRectangle(int x, int y)
		{
			Point point = control.PointToScreen(new Point(x, y));
			x = point.X;
			y = point.Y;

			int halfWidth = control.Width / zoomRate / 2;
			int halfHeight = control.Height / zoomRate / 2;

			int left = x - halfWidth;
			int top = y - halfHeight;

			if (left < 0)
				left = 0;

			if (top < 0)
				top = 0;

			int right = left + control.Width / zoomRate - 1;
			int bottom = top + control.Height / zoomRate - 1;

			if (right > desktopRight)
			{
				right = desktopRight;
				left = right - control.Width / zoomRate + 1;
			}

			if (bottom > desktopBottom)
			{
				bottom = desktopBottom;
				top = bottom - control.Height / zoomRate + 1;
			}

			return Rectangle.FromLTRB(left, top, right, bottom);
		}

		void CopyDesktopRectangle(int left, int top, Graphics graphics)
		{
			if (graphics == null)
				graphics = control.CreateGraphics();

			IntPtr hDCPanel = graphics.GetHdc();
			IntPtr hDCDesktop = Win32.GetDC(hWndDesktop);

			Win32.StretchBlt(hDCPanel, 0, 0, control.Width, control.Height,
				hDCDesktop, left, top, control.Width / zoomRate, control.Height / zoomRate, 
				Win32.SRCCOPY);
			
			graphics.ReleaseHdc(hDCPanel);
			Win32.ReleaseDC(hWndDesktop, hDCDesktop);
		}

		public Point DesktopDimension
		{
			set
			{
				desktopRight = value.X - 1;
				desktopBottom = value.Y - 1;
			}
		}
	}
}