using System;
using System.IO;
using Rebex.Net;

namespace Examples
{
	/// <summary>
	/// This sample is a simple command line FTP client.
	/// In addition to basic operations, it demonstrates the following features:
	/// - extending FtpItem and using a custom list parser to get Unix rights
	/// - using a "Site" method for changing Unix rights using chmod command
	/// - using FtpList class
	/// </summary>
	class ConsoleClient
	{
		/// <summary>
		/// This class extends FtpItem class by unix rights mask.
		/// </summary>
		public class UnixItem : FtpItem
		{
			private int _rights;

			public int Rights
			{
				get
				{
					return _rights;
				}
			}


			public UnixItem (int rights, FtpItem item) : base (item.Name, item.Size, item.Type, item.Modified, item.SymlinkPath)
			{
				_rights = rights;
			}
		}

		/// <summary>
		/// Custom parser that parses Unix LIST lines and creates UnixItem items.
		/// </summary>
		private static void UnixRightsParser (object sender, FtpItemParseEventArgs e)
		{
			// if the item was ignored, ignore it as well
			if (e.Item == null)
				return;

			// if there is not space at position 10, the item is not a Unix LIST item
			if (e.RawLine.IndexOf (' ') != 10)
				return;

			// compute the rights mask
			int mask = 256;
			int rights = 0;
			for (int i=1; i<10; i++)
			{
				if (e.RawLine[i] != '-')
					rights += mask;
				mask /= 2;
			}

			// create unix item with rights
			e.Item = new UnixItem (rights, e.Item);
		}

		public void ResponseRead (object sender, FtpResponseReadEventArgs e)
		{
			Console.WriteLine (e.Response);
		}

		private DateTime lastReport = DateTime.MinValue;

		public void TransferProgress (object sender, FtpTransferProgressEventArgs e)
		{
			if (e.State == FtpTransferState.None)
				return;

			if ((DateTime.Now - lastReport).TotalSeconds > 1)
			{
				Console.WriteLine (e.BytesTransfered + " bytes.");
				lastReport = DateTime.Now;
			}
		}

		private Ftp ftp;

		public ConsoleClient ()
		{
			// create Ftp object and set response handler
			ftp = new Ftp();
			ftp.ResponseRead += new FtpResponseReadEventHandler (ResponseRead);
			ftp.TransferProgress += new FtpTransferProgressEventHandler (TransferProgress);
		}

		private void Help ()
		{
			Console.WriteLine ("!         dir          pwd");
			Console.WriteLine ("?         disconnect   quit");
			Console.WriteLine ("ascii     exit         rmdir");
			Console.WriteLine ("active    help         user");
			Console.WriteLine ("binary    ls           chmod");
			Console.WriteLine ("bye       mkdir        get");
			Console.WriteLine ("cd        open         put");
			Console.WriteLine ("close     passive");
		}

		private void Close ()
		{
			Console.WriteLine ("Disconnecting...");
			ftp.Disconnect();
		}

		private bool Open (string host)
		{
			try
			{

				if (ftp.State != FtpState.Disconnected)
				{
					Console.WriteLine ("Already connected. Disconnect first.");
					return false;
				}

				if (host == null || host.Length == 0)
				{
					Console.Write ("User: ");
					host = Console.ReadLine();
				}

				string[] p = host.Split(' ');
				if (p.Length<0 || p.Length>2)
				{
					Console.WriteLine ("Usage: hostname [port]");
					return false;
				}

				host = p[0];
				int port;
				if (p.Length == 1)
				{
					port = 21;
				}
				else
				{
					try
					{
						port = int.Parse (p[1]);
					}
					catch
					{
						port = -1;
					}
					if (port <= 0 || port > 65535)
					{
						Console.WriteLine ("Invalid port number.");
						return false;
					}
				}

				ftp.Connect (host, port);
				return true;
			}
			catch (FtpException e)
			{
				Console.WriteLine (e.Message);
				return false;
			}
		}

		private void Login (string user)
		{
			if (user == null || user.Length == 0)
			{
				Console.Write ("User: ");
				user = Console.ReadLine();
			}

			Console.Write ("Password: ");
			string pass = Console.ReadLine();

			ftp.Login (user, pass);
		}

		private void Pwd ()
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			ftp.GetCurrentDirectory();
		}

		private void Chmod (string param)
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			if (param == null || param.Length == 0)
			{
				Console.WriteLine ("Usage: chmod mode filename");
				return;
			}

			ftp.Site ("CHMOD " + param);
		}


		private void Cd (string param)
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			ftp.ChangeDirectory (param);
		}

		private void Quit ()
		{
			if (ftp.State != FtpState.Disconnected)
				Close ();

			Console.WriteLine ();
		}

		private void Ascii ()
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			ftp.SetTransferType (FtpTransferType.Ascii);
		}

		private void Active ()
		{
			ftp.Passive = false;
			Console.WriteLine ("Passive mode turned off.");
		}

		private void Passive ()
		{
			ftp.Passive = true;
			Console.WriteLine ("Passive mode turned on.");
		}

		private void Binary ()
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			ftp.SetTransferType (FtpTransferType.Binary);
		}

		private void Dir ()
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			lastReport = DateTime.Now;
			FtpList list = ftp.GetList ();

			for (int i=0; i<list.Count; i++)
			{
				FtpItem item = list[i];
				if (item.IsSymlink)
					Console.Write ("s ");
				else if (item.IsDirectory)
					Console.Write ("d ");
				else
					Console.Write ("- ");

				UnixItem unixItem = item as UnixItem;
				if (unixItem != null)
					Console.Write (Convert.ToString (unixItem.Rights, 8).PadLeft(3,'0') + " ");
				else
					Console.Write ("??? ");

				Console.Write (item.Modified.ToString("u").Substring(0,16));
				Console.Write (item.Size.ToString().PadLeft(10,' '));
				Console.Write (" {0}", item.Name);
				if (item.IsSymlink)
					Console.Write (" -> " + item.SymlinkPath);
				Console.WriteLine ();
			}
		}

		private void Ls ()
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			lastReport = DateTime.Now;
			FtpList list = ftp.GetList ();

			for (int i=0; i<list.Count; i++)
			{
				Console.WriteLine (list[i].Name);
			}
		}

		private void Get (string remotePath)
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			if (remotePath == null || remotePath.Length == 0)
			{
				Console.WriteLine ("Usage: put remotePath");
				return;
			}

			try
			{
				lastReport = DateTime.Now;
				ftp.GetFile (remotePath, Path.GetFileName(remotePath));
			}
			catch (Exception e)
			{
				if (e is FtpException)
					throw e;

				Console.WriteLine (e.Message);
			}
		}

		/// <summary>
		/// Uploads a local directory with all files and subdirectories to an FTP server.
		/// </summary>
		/// <param name="ftp">Instance of an Ftp object.</param>
		/// <param name="localPath">The local path of the root of the directory structure to be uploaded.</param>
		/// <remarks>
		/// <p>
		/// Ftp object must be connected and should have the directory set to the desired path.
		/// </p>
		/// <p>
		/// Method reports information to console. This behaviour should be changed to reflect actual needs.
		/// </p>
		/// </remarks>
		public static void UploadDirectory (Ftp ftp, string localPath)
		{
			// strip ending slashes (GetFileName returns empty string otherwise...)
			localPath = localPath.TrimEnd ('\\','/');

			Console.WriteLine ("Uploading directory: '" + localPath + "'");

			// get the name of the directory and try to create it
			string remotePath = Path.GetFileName (localPath);
			try
			{
				ftp.CreateDirectory (remotePath);
			}
			catch (FtpException e)
			{
				// throw exception if an error other than ProtocolError occurs
				if (e.Status != FtpExceptionStatus.ProtocolError)
					throw e;

				// protocol error - directory probably already exists
				Console.WriteLine (e.Message);
			}
			
			// change the directory
			ftp.ChangeDirectory (remotePath);

			// upload all files from the localPath directory
			string[] files = Directory.GetFiles (localPath);
			for (int i=0; i<files.Length; i++)
			{
				Console.WriteLine ("Uploading file: '" + files[i] + "'");
				try
				{
					ftp.PutFile (files[i], Path.GetFileName (files[i]));
				}
				catch (FtpException e)
				{
					// throw exception if an error other than ProtocolError occurs
					if (e.Status != FtpExceptionStatus.ProtocolError)
						throw e;

					// protocol error - report it and continue with the remaining files
					Console.WriteLine (e.Message);
				}
			}

			// upload all directories from the localPath directory
			string[] dirs = Directory.GetDirectories (localPath);
			for (int i=0; i<dirs.Length; i++)
				UploadDirectory (ftp, dirs[i]);

			ftp.ChangeDirectory ("..");
		}

		private void PutDir (string localPath)
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			if (localPath == null || localPath.Length == 0)
			{
				Console.WriteLine ("Usage: putdir localPath");
				return;
			}

			if (!Directory.Exists (localPath))
			{
				Console.WriteLine ("Path does not exist.");
				return;
			}

			lastReport = DateTime.Now;
			UploadDirectory (ftp, localPath);
		}


		private void Put (string localPath)
		{
			if (ftp.State == FtpState.Disconnected)
			{
				Console.WriteLine ("Not connected.");
				return;
			}

			if (localPath == null || localPath.Length == 0)
			{
				Console.WriteLine ("Usage: put localPath");
				return;
			}

			try
			{
				lastReport = DateTime.Now;
				ftp.PutFile (localPath, Path.GetFileName(localPath));
			}
			catch (Exception e)
			{
				if (e is FtpException)
					throw e;

				Console.WriteLine (e.Message);
			}
		}



		public void Run ()
		{
			while (true)
			{
				Console.Write ("ftp> ");
				string command = Console.ReadLine().Trim();
				string param = null;
				int i = command.IndexOf (' ');
				if (i > 0)
				{
					param = command.Substring (i + 1);
					command = command.Substring (0, i);
				}

				try
				{
					switch (command.ToLower())
					{
						case "!":
							goto case "quit";
						case "?":
							goto case "help";
						case "bye":
							goto case "quit";
						case "exit":
							goto case "quit";
						case "disconnect":
							goto case "close";
						case "quit":
							Quit();
							return;
						case "close":
							Close();
							break;
						case "open":
							if (!Open (param)) break;
							param = null;
							goto case "user";
						case "user":
							Login (param);
							break;
						case "help":
							Help();
							break;
						case "ascii":
							Ascii();
							break;
						case "binary":
							Binary();
							break;
						case "active":
							Active();
							break;
						case "passive":
							Passive();
							break;
						case "cd":
							Cd (param);
							break;
						case "pwd":
							Pwd();
							break;
						case "chmod":
							Chmod (param);
							break;
						case "get":
							Get (param);
							break;
						case "put":
							Put (param);
							break;
						case "putdir":
							PutDir (param);
							break;
						case "dir":
							Dir();
							break;
						case "ls":
							if (param != null && param.StartsWith ("-l"))
								Dir();
							else
								Ls();
							break;
						default:
							Console.WriteLine ("Invalid command.");
							break;
					}
				}
				catch (FtpException e)
				{
					if (e.Status != FtpExceptionStatus.ProtocolError)
						throw e;
				}

			}
		}


		[STAThread]
		static void Main (string[] args)
		{
			FtpItem.ItemParse += new FtpItemParseEventHandler (UnixRightsParser);

			ConsoleClient client = new ConsoleClient ();

			if (args.Length > 0)
			{
				string param = args[0];
				for (int i=1; i<args.Length; i++)
					param += " " + args[i];

				if (client.Open (param))
					client.Login (null);
			}

			client.Run ();
		}
	}
}
