using System;
using System.IO;
using System.Threading;
using Rebex.Net;

namespace Examples
{
	/// <summary>
	/// This sample is a simple utility to upload/download a file to/from
	/// a FTP server using asynchronous methods of the Ftp class.
	/// </summary>
	class GetPutAsync
	{
		[STAThread]
		static void Main (string[] args)
		{
			string method = null;

			if (args.Length == 4)
			{
				switch (args[0].ToLower())
				{
					case "get": method = "get"; break;
					case "put": method = "put"; break;
				}
			}

			if (method == null)
			{
				// display syntax if wrong number of parameters or unsupported operation
				Console.WriteLine ("GetPut - uploads or downloads a file from a FTP server");
				Console.WriteLine ("Syntax: GetPut get|put hostname remotepath localpath");
				Console.WriteLine ("Example: GetPut get ftp.microsoft.com /developr/readme.txt readme.txt");
				return;
			}

			string hostname = args[1];
			string remotePath = args[2];
			string localPath = args[3];

			// create Ftp object and set event handlers
			Ftp ftp = new Ftp();
			ftp.TransferProgress += new FtpTransferProgressEventHandler (TransferProgress);
			ftp.ResponseRead += new FtpResponseReadEventHandler (ResponseRead);
			ftp.CommandSent += new FtpCommandSentEventHandler (CommandSent);

			try
			{
				// connect to the server
				ftp.Connect (hostname);

				// ask for username and password and login
				Console.Write ("Username: ");
				string username = Console.ReadLine();
				Console.Write ("Password: ");
				string password = Console.ReadLine();
				ftp.Login (username, password);

				// set transfer type to binary
				ftp.SetTransferType (FtpTransferType.Binary);

				Console.WriteLine ();

				// begin asynchronous transfer
				IAsyncResult asyncResult;
				if (method == "get")
					asyncResult = ftp.BeginGetFile (remotePath, localPath, new AsyncCallback(MyCallback), null);
				else
					asyncResult = ftp.BeginPutFile (localPath, remotePath, new AsyncCallback(MyCallback), null);

				// do something else here...

				// wait for the transfer to end
				asyncResult.AsyncWaitHandle.WaitOne();

				// get the result
				long bytes;
				if (method == "get")
					bytes = ftp.EndGetFile (asyncResult);
				else
					bytes = ftp.EndPutFile (asyncResult);

				// disconnect from the server
				ftp.Disconnect();
			}
			catch (FtpException e)
			{
				// output error description to the console
				Console.WriteLine ("'{0}' error: {1}", e.Status, e.Message);
			}
			finally
			{
				// release all resources
				ftp.Dispose();
			}
		}

		public static void MyCallback (IAsyncResult asyncResult)
		{
			Console.WriteLine ("\nTransfer finished.");
		}

		public static void TransferProgress (object sender, FtpTransferProgressEventArgs e)
		{
			if (e.State != FtpTransferState.None)
				Console.WriteLine ("\rTransfered {0} bytes...", e.BytesTransfered);
		}

		public static void ResponseRead (object sender, FtpResponseReadEventArgs e)
		{
			Console.Write (e.Response);
		}

		public static void CommandSent (object sender, FtpCommandSentEventArgs e)
		{
			Console.WriteLine ("-> " + e.Command);
		}

	}
}
