using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Resources;
using System.IO;
using Rebex.Net;
using System.Text.RegularExpressions;

namespace WinFormClient
{
	// -------------------
	// struct for one site
	// -------------------

	public struct Site
	{
		public string   Host;
		public int      Port;
		public string   Login;
		public string   Password;
			
		public string   UploadDir;
		public string   NoModifyDir;
		public int      Timeout;
		public bool	    Passive;
		
		public string   Proxy;
		public int      ProxyPort;
		public bool     ProxyEnabled;
		public string   ProxyUser;
		public string   ProxyPassword;
		public int      ProxyType;
	} 

	/// <summary>
	/// very simple ftp client
	/// </summary>
	public class MainForm : System.Windows.Forms.Form
	{
		// ----------------
		// winform thingies
		// ----------------

		#region winform list
		
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.RichTextBox tbLog;
		private System.Windows.Forms.ListView listServer;
		private System.Windows.Forms.ColumnHeader name;
		private System.Windows.Forms.ColumnHeader size;
		private System.Windows.Forms.Label lblServer;
		private System.Windows.Forms.ListView listLocal;
		private System.Windows.Forms.Label lblLocal;
		private System.Windows.Forms.Label lblServerDirectory;
		private System.Windows.Forms.ColumnHeader columnHeader1;
		private System.Windows.Forms.ColumnHeader columnHeader2;
		private System.Windows.Forms.Label lblLocalDirectory;
		private System.Windows.Forms.ProgressBar pbTransfer;
		private System.Windows.Forms.Label lblProgress;
		private System.Windows.Forms.ColumnHeader date;
		private System.Windows.Forms.ColumnHeader columnHeader3;
		private System.Windows.Forms.Button btnStop;
		private System.Windows.Forms.Button btnDelete;
		private System.Windows.Forms.Button btnUpload;
		private System.Windows.Forms.Button btnDownload;
		private System.Windows.Forms.RadioButton rbBin;
		private System.Windows.Forms.RadioButton rbAsc;
		private System.Windows.Forms.Button btnRmDir;
		private System.Windows.Forms.Button btnRefresh;
		private System.Windows.Forms.Button btnMakeDir;
		private System.Windows.Forms.ComboBox cbDrives;
		private System.Windows.Forms.Button btnConnect;
		private System.Windows.Forms.GroupBox gbControl;
		private System.Windows.Forms.GroupBox groupBox1;
		private System.Windows.Forms.PictureBox headerLeft;
		private System.Windows.Forms.PictureBox headerMiddle;
		private System.Windows.Forms.PictureBox headerRight;

		#endregion

		// ----------
		// ftp object
		// ----------

		private Ftp _ftp;
				
		// -----------
		// site params
		// -----------

		private Site _site=new Site();

		// ----------------
		// global variables
		// ----------------

		private int _serverDirs=0;				// number of dirs on server side
		private int _localDirs=0;				// number of dirs on local side
		private int _selectedDrive=-1;			// selected drive (A:, C:, ...) in combo
		private string _localPath=null;			// local path
		private string _serverPath=null;		// server path
		private long _fileSize=-1;				// file size
		private int _activePanel=1;				// 0-server 1-localhost
		private DateTime _transferTime;			// transfer launch-time

		// ----------
		// constansts
		// ----------

		const string CONFIGFILE="config.xml";		// xml file consisting of settings
		const string SYMLINK="--->";				// symlink tag
		const byte	 MAGICNUMBER=42;				// XOR value for password encoded/decode process
		const string URL="http://www.rebex.net/ftp.net/";	// url
		
		// ------
		// colors
		// ------		

		Color COLORDEFAULT=Color.Black;				// default color in text-output panel
		Color COLORCOMMAND=Color.DarkGreen;			// ftp command color
		Color COLORERROR  =Color.Red;				// color of error messages		            
		
		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// write a msg to output [tbLog]
		/// </summary>
		/// <param name="message"></param>
		public void WriteToLog(string message)
		{			
			WriteToLog(message,COLORDEFAULT,true);
		}
		
		public void WriteToLog(string message,Color color)
		{			
			WriteToLog(message,color,true);
		}

		public void WriteToLog(string message,bool crlf)
		{			
			WriteToLog(message,COLORDEFAULT,crlf);
		}

		public void WriteToLog(string message,Color color,bool crlf)
		{			
			tbLog.Focus(); 
			tbLog.AppendText("");
			tbLog.SelectionColor=color;			
			tbLog.AppendText(message+(crlf ? "\r\n":null));				
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// get filename from path (xxxxx/xxxx/yyy -> yyy)
		/// </summary>
		/// <param name="path">ftp path</param>
		/// <returns>filename</returns>
		private string GetFilenameFromPath(string path)
		{
			int idx=path.LastIndexOf("/");

			if (idx!=-1 && idx+1!=path.Length)
			{
				path=path.Substring(idx+1);				
			}
			
			return path;
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// file vs. directory detection
		/// </summary>
		/// <param name="dt">filename</param>
		/// <returns>true (is file), false (is directory)</returns>
		private bool IsFile(string filename)
		{
			filename=GetFilenameFromPath(filename);			
			Regex r=new Regex(@"(.*\..*)");
			Match m=r.Match(filename);
			return m.Success;				
		}

		private bool IsDirectory(string filename)
		{
			return (!IsFile(filename));
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// format datetime to readable form in the panels
		/// </summary>
		/// <param name="dt">datetime</param>
		/// <returns>formatted string</returns>
		private string FormatFileTime(DateTime dt)
		{
			return dt.ToString("yyyy-MM-dd hh:mm");
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		private bool IsFtpReady()
		{
			if (_ftp==null || _ftp.State!=FtpState.Ready || _fileSize!=-1) return false;

			return true;
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// is ftp-panel activated?
		/// </summary>
		/// <returns>true (=it is)</returns>
		private bool SelectedFTP()
		{
			return(_activePanel==0);
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// is local-panel activated?
		/// </summary>
		/// <returns>true (=it is)</returns>
		private bool SelectedLocal()
		{
			return(_activePanel==1);
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// activate one of the panels
		/// </summary>
		/// <param name="number">0 (=server) 1 (=local)</param>
		private void ActivatePanel(int number)
		{
			_activePanel=number;

			if (number==0)
			{
				if (!listServer.Focused) listServer.Focus();

				lblServerDirectory.BackColor=SystemColors.ActiveCaption;
				lblServerDirectory.ForeColor=SystemColors.ActiveCaptionText;
				lblLocalDirectory.BackColor=SystemColors.InactiveCaption;
				lblLocalDirectory.ForeColor=SystemColors.InactiveCaptionText;
			}
			else
			{
				if (!listLocal.Focused) listLocal.Focus();

				lblServerDirectory.BackColor=SystemColors.InactiveCaption;
				lblServerDirectory.ForeColor=SystemColors.InactiveCaptionText;
				lblLocalDirectory.BackColor=SystemColors.ActiveCaption;
				lblLocalDirectory.ForeColor=SystemColors.ActiveCaptionText;
			}				
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// encoded/normal texts transfer to normal/encoded ones
		/// </summary>
		/// <param name="password">encoded/normal text</param>
		/// <returns></returns>
		private string EncodePassword(string password)
		{
			string encodedString="";

			foreach	(char signs in password)
			{				
				char sign=(char)(signs ^ MAGICNUMBER);
				encodedString+=sign.ToString();
			}

			return encodedString;
		}
		
		// -------------------------------------------------------------------------------------
		/// <summary>
		/// load a config file and set appropritate values for site
		/// </summary>
		private void LoadConfig()
		{
			// ---------------------------
			// set default values for site
			// ---------------------------

			_site.Host=null;
			_site.Port=21;
			_site.Login="";
			_site.Password="";
			_site.UploadDir=null;
			_site.NoModifyDir="/";
			_site.Timeout=10000;
			_site.Passive=true;
			_site.Proxy=null;			
			_site.ProxyEnabled=false;
			_site.ProxyPort=3128;
			_site.ProxyUser=null;
			_site.ProxyPassword=null;
			_site.ProxyType=1;

			// ----------------
			// read config file
			// ----------------

			DataSet ds=new DataSet();			

			try
			{
				if (!File.Exists(CONFIGFILE)) return;

				ds.ReadXml(CONFIGFILE);

				_site.NoModifyDir="/";
				_site.Timeout=10000;
				_site.Proxy=null;						 										

				foreach (DataRow dr in ds.Tables[0].Rows)
				{
					string item=dr["value"].ToString();

					switch(dr["name"].ToString().ToLower())
					{
						case "host":_site.Host=item;break;
						case "login":_site.Login=item;break;
						case "port":
						{
							try
							{
								_site.Port=Int32.Parse(item);
							}

							catch
							{
								_site.Port=21; // default
							}

							break;							
						}
						case "password":_site.Password=EncodePassword(item);break;
						case "passive":_site.Passive=(item.ToLower()=="false" ? false:true);break;
						case "proxy":_site.Proxy=item;break;
						case "proxylogin":_site.ProxyUser=item;break;
						case "proxypassword":_site.ProxyPassword=EncodePassword(item);break;
						case "proxyenabled":_site.ProxyEnabled=(item.ToLower()=="false" ? false:true);break;
						case "proxyport":
						{
							try
							{
								_site.ProxyPort=Int32.Parse(item);
							}

							catch
							{
								_site.ProxyPort=3128; // default
							}

							break;						
						}
						case "proxytype":
						{
							try
							{
								_site.ProxyType=Int32.Parse(item);
							}

							catch
							{
								_site.ProxyType=1; // default
							}

							break;							
						}
					}
				}							
			}			

			catch
			{
				WriteToLog("* config file read error",COLORERROR);				
			}						
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// save site values into config file
		/// </summary>
		private void SaveConfig()
		{
			DataSet ds=new DataSet("configuration");			

			try
			{
				ds.Tables.Add("key");
				ds.Tables[0].Columns.Add("name");
				ds.Tables[0].Columns.Add("value");
				
				DataRow [] dr=new DataRow[11];
				
				dr[0]=ds.Tables[0].NewRow();
				dr[1]=ds.Tables[0].NewRow();
				dr[2]=ds.Tables[0].NewRow();
				dr[3]=ds.Tables[0].NewRow();
				dr[4]=ds.Tables[0].NewRow();
				dr[5]=ds.Tables[0].NewRow();
				dr[6]=ds.Tables[0].NewRow();
				dr[7]=ds.Tables[0].NewRow();
				dr[8]=ds.Tables[0].NewRow();
				dr[9]=ds.Tables[0].NewRow();
				dr[10]=ds.Tables[0].NewRow();
				
				ds.Tables[0].Clear();				

				dr[0]["name"]="host";
				dr[0]["value"]=_site.Host;
				dr[1]["name"]="login";
				dr[1]["value"]=_site.Login;
				dr[2]["name"]="port";
				dr[2]["value"]=_site.Port;
				dr[3]["name"]="password";
				dr[3]["value"]=EncodePassword(_site.Password);
				dr[4]["name"]="passive";
				dr[4]["value"]=_site.Passive;
				dr[5]["name"]="proxyenabled";
				dr[5]["value"]=_site.ProxyEnabled;
				dr[6]["name"]="proxy";
				dr[6]["value"]=_site.Proxy;
				dr[7]["name"]="proxyport";
				dr[7]["value"]=_site.ProxyPort;
				dr[8]["name"]="proxytype";
				dr[8]["value"]=_site.ProxyType;
				dr[9]["name"]="proxylogin";
				dr[9]["value"]=_site.ProxyUser;
				dr[10]["name"]="proxypassword";
				dr[10]["value"]=_site.ProxyPassword;

				ds.Tables[0].Rows.Add(dr[0]);				
				ds.Tables[0].Rows.Add(dr[1]);				
				ds.Tables[0].Rows.Add(dr[2]);				
				ds.Tables[0].Rows.Add(dr[3]);				
				ds.Tables[0].Rows.Add(dr[4]);				
				ds.Tables[0].Rows.Add(dr[5]);
				ds.Tables[0].Rows.Add(dr[6]);
				ds.Tables[0].Rows.Add(dr[7]);
				ds.Tables[0].Rows.Add(dr[8]);				
				ds.Tables[0].Rows.Add(dr[9]);				
				ds.Tables[0].Rows.Add(dr[10]);				

				ds.WriteXml(CONFIGFILE,XmlWriteMode.IgnoreSchema);												
			}

			catch
			{
				WriteToLog("* config file not saved",COLORERROR);				
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// create ftp object, set timeout, create EventHandlers
		/// </summary>
		/// <param name="site">site config values</param>
		/// <returns>ftp object</returns>
		public Ftp CreateFtp(Site site)
		{
			Ftp ftp=new Ftp();			
			
			ftp.ResponseRead+=new FtpResponseReadEventHandler(ResponseRead);
			ftp.CommandSent+=new FtpCommandSentEventHandler(CommandSent);			
			ftp.TransferProgress+=new FtpTransferProgressEventHandler(TransferProgress);
			ftp.StateChanged+=new FtpStateChangedEventHandler(StateChanged);

			return ftp;
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// event displaying ftp state
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void StateChanged(object sender,FtpStateChangedEventArgs e)
		{
			switch (e.NewState)
			{
				case FtpState.Disconnected:lblProgress.Text="Disconnected";break;
				case FtpState.Disposed    :lblProgress.Text="Disconnected";break;
				case FtpState.Ready       :lblProgress.Text="Ready";break;
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// event writing ftp output to log
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void TransferProgress(object sender,FtpTransferProgressEventArgs e)
		{								
			if (e.State!=FtpTransferState.None && _fileSize>0)
			{		
				decimal index=(decimal)e.BytesTransfered/(decimal)_fileSize;
				pbTransfer.Value=(int)(index*pbTransfer.Maximum);				
				lblProgress.Text=e.BytesTransfered+" bytes";
			}				
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// event writing ftp output to log
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void ResponseRead(object sender,FtpResponseReadEventArgs e)
		{			
			WriteToLog(e.Response,true);						
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// write 'commands sent' to output log
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		public void CommandSent(object sender,FtpCommandSentEventArgs e)
		{
			const string PASS="PASS";
			string command=e.Command.ToString();

			// security check - transforma PASS abc to PASS ***

			if (command!=null && command.Length>PASS.Length && command.IndexOf(PASS)==0)
			{
				command="PASS ***";
			}

			WriteToLog(command,COLORCOMMAND,true);						
		}

		// -------------------------------------------------------------------------------------		
		/// <summary>
		/// make local list + drives combo box
		/// </summary>
		public void MakeLocalList()
		{			
			int dirs=0;
			int files=0;
			long size=0;

			// -------------------------------------------
			// make a logical drives list [first run only]
			// -------------------------------------------

			if (cbDrives.Items.Count<=0)
			{	
				string [] drives=Directory.GetLogicalDrives();			

				for (int c=0;c<drives.Length;c++)
				{
					string drive=drives[c];

					cbDrives.Items.Add(drive);

					// try to set default 'c:\'

					if (_selectedDrive==-1 && drive.ToLower()==@"c:\")
					{
						cbDrives.SelectedIndex=c;
						_selectedDrive=c;
					}					
				}

				if (cbDrives.SelectedIndex<=0) cbDrives.SelectedIndex=0;
			}
			
			// ---------------
			// create the list
			// ---------------

			listLocal.Items.Clear();

			// directory up... [c:\temp\426\..\ -> c:\temp\]

			if (_localPath!=null && _localPath.Length>4 && _localPath.IndexOf(@"\..\")==_localPath.Length-4)
			{
				_localPath=_localPath.Substring(0,_localPath.Length-4);
				_localPath=_localPath.Substring(0,_localPath.LastIndexOf("\\"))+"\\";				
			}			
			
			// directory up drawing

			if (_localPath!=null && _localPath.LastIndexOf("\\")!=_localPath.IndexOf("\\",0))
			{
				listLocal.Items.Add(new ListViewItem("..",0)); // up...								
				dirs++;
			}
			else
			{
				_localPath=cbDrives.SelectedItem.ToString();
			}

			DirectoryInfo directory=new DirectoryInfo(_localPath);
			DirectoryInfo[] list=directory.GetDirectories();			

			for(int c=0;c<list.Length;c++)
			{
				string[] row=new string[3];
				row[0]=list[c].Name;
				row[1]="";
				
				// time [on read-only devices LWT is not available for dirs?] 

				try
				{
					row[2]=FormatFileTime(list[c].LastWriteTime);
				}
				
				catch
				{
					row[2]="";
				}

				listLocal.Items.Add(new ListViewItem(row,1+c));							
				dirs++;								
			}			

			_localDirs=dirs;

			// files

			FileInfo[] list2=directory.GetFiles();			

			for(int c=0;c<list2.Length;c++)
			{
				string[] row=new string[3];
				row[0]=list2[c].Name;
				row[1]=list2[c].Length.ToString();								
				row[2]=FormatFileTime(list2[c].LastWriteTime);				
				
				listLocal.Items.Add(new ListViewItem(row,1+dirs+files));									
				size+=list2[c].Length;
				files++;				
			}	

			// -----
			// stats
			// -----

			lblLocal.Text=dirs+" dir"+(dirs>1 ? "s":"")+" "+
				files+" file"+(files>1 ? "s":"")+" "+
				size/1024+" K";

			// -----------------
			// working directory
			// -----------------			

			lblLocalDirectory.Text=directory.FullName;
			_localPath=directory.FullName;			
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// make ftp list
		/// </summary>
		public void MakeFtpList()
		{
			int dirs=0;
			int files=0;
			long size=0;

			listServer.Items.Clear();

			// --------------
			// not connected?
			// --------------

			if (!IsFtpReady())
			{				
				lblServerDirectory.Text="";
				lblServer.Text="";
				_serverDirs=0;

				return;			
			}

			// -------------
			// make the list
			// -------------

			FtpList list=null;						

			try
			{
				list=_ftp.GetList();
			}
			
			catch(FtpException)
			{												
				listServer.Items.Add(new ListViewItem("..",0));
				return;
			}													

			// directories 

			listServer.Items.Add(new ListViewItem("..",0));
			dirs++;

			for (int c=0;c<list.Count;c++)
			{
				// symlink

				if (list[c].IsSymlink && IsDirectory(list[c].SymlinkPath))
				{
					string[] row=new string[3];
					row[0]=list[c].Name;
					row[1]=SYMLINK;					
					row[2]=FormatFileTime(list[c].Modified);

					listServer.Items.Add(new ListViewItem(row,1+dirs));							
					dirs++;						
				}
					
				// normal directory

				if (list[c].IsDirectory)
				{						
					string[] row=new string[3];
					row[0]=list[c].Name;
					row[1]=null;
					row[2]=FormatFileTime(list[c].Modified);

					listServer.Items.Add(new ListViewItem(row,1+dirs));							
					dirs++;				
				}
			}

			// files

			for (int c=0;c<list.Count;c++)
			{
				// symlink

				if (list[c].IsSymlink && IsFile(list[c].SymlinkPath))
				{
					string[] row=new string[3];
					row[0]=list[c].Name;
					row[1]=SYMLINK;
					row[2]=FormatFileTime(list[c].Modified);
					listServer.Items.Add(new ListViewItem(row,1+dirs+files));															
					files++;
				
				}

				// normal file
				
				if (list[c].IsFile)
				{						
					string[] row=new string[3];
					row[0]=list[c].Name;
					row[1]=list[c].Size.ToString();
					row[2]=FormatFileTime(list[c].Modified);
					listServer.Items.Add(new ListViewItem(row,1+dirs+files));									
					size+=list[c].Size;
					files++;
				}				
			}			

			// -----
			// stats
			// -----

			lblServer.Text=dirs+" dir"+(dirs>1 ? "s":"")+" "+
				files+" file"+(files>1 ? "s":"")+" "+
				size/1024+" K";

			// -----------------
			// working directory
			// -----------------			

			try
			{
				string dir=_ftp.GetCurrentDirectory();
				if (dir.Length>1 && dir[dir.Length-1].ToString()!="/") dir+="/";
				_serverPath=dir;
				lblServerDirectory.Text=_serverPath;
			}

			catch(FtpException)
			{				
			}

			_serverDirs=dirs;
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// constructor
		/// </summary>
		public MainForm()
		{
			InitializeComponent();
			Width=640;
			SetRightSizeValues();
			rbBin.Checked=true;			
			MakeLocalList();			
			ActivatePanel(1);			
			lblServer.Text="";			
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// clean up any resources being used
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				if (components!=null) 
				{
					components.Dispose();
				}
			}
			base.Dispose(disposing);
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(MainForm));
			this.listServer = new System.Windows.Forms.ListView();
			this.name = new System.Windows.Forms.ColumnHeader();
			this.size = new System.Windows.Forms.ColumnHeader();
			this.date = new System.Windows.Forms.ColumnHeader();
			this.listLocal = new System.Windows.Forms.ListView();
			this.columnHeader1 = new System.Windows.Forms.ColumnHeader();
			this.columnHeader2 = new System.Windows.Forms.ColumnHeader();
			this.columnHeader3 = new System.Windows.Forms.ColumnHeader();
			this.tbLog = new System.Windows.Forms.RichTextBox();
			this.lblServer = new System.Windows.Forms.Label();
			this.lblServerDirectory = new System.Windows.Forms.Label();
			this.lblLocal = new System.Windows.Forms.Label();
			this.lblLocalDirectory = new System.Windows.Forms.Label();
			this.pbTransfer = new System.Windows.Forms.ProgressBar();
			this.lblProgress = new System.Windows.Forms.Label();
			this.btnStop = new System.Windows.Forms.Button();
			this.btnDelete = new System.Windows.Forms.Button();
			this.btnUpload = new System.Windows.Forms.Button();
			this.btnDownload = new System.Windows.Forms.Button();
			this.rbBin = new System.Windows.Forms.RadioButton();
			this.rbAsc = new System.Windows.Forms.RadioButton();
			this.btnRmDir = new System.Windows.Forms.Button();
			this.btnRefresh = new System.Windows.Forms.Button();
			this.btnMakeDir = new System.Windows.Forms.Button();
			this.cbDrives = new System.Windows.Forms.ComboBox();
			this.btnConnect = new System.Windows.Forms.Button();
			this.gbControl = new System.Windows.Forms.GroupBox();
			this.groupBox1 = new System.Windows.Forms.GroupBox();
			this.headerLeft = new System.Windows.Forms.PictureBox();
			this.headerMiddle = new System.Windows.Forms.PictureBox();
			this.headerRight = new System.Windows.Forms.PictureBox();
			this.gbControl.SuspendLayout();
			this.groupBox1.SuspendLayout();
			this.SuspendLayout();
			// 
			// listServer
			// 
			this.listServer.Anchor = ((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left);
			this.listServer.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						 this.name,
																						 this.size,
																						 this.date});
			this.listServer.FullRowSelect = true;
			this.listServer.Location = new System.Drawing.Point(8, 80);
			this.listServer.MultiSelect = false;
			this.listServer.Name = "listServer";
			this.listServer.Size = new System.Drawing.Size(264, 313);
			this.listServer.TabIndex = 2;
			this.listServer.View = System.Windows.Forms.View.Details;
			this.listServer.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.listServer_DoubleClick);
			this.listServer.DoubleClick += new System.EventHandler(this.listServer_DoubleClick);
			this.listServer.Enter += new System.EventHandler(this.listServer_Enter);
			// 
			// name
			// 
			this.name.Text = "Name";
			this.name.Width = 120;
			// 
			// size
			// 
			this.size.Text = "Size";
			this.size.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
			// 
			// date
			// 
			this.date.Text = "Date";
			this.date.Width = 120;
			// 
			// listLocal
			// 
			this.listLocal.Anchor = (System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom);
			this.listLocal.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						this.columnHeader1,
																						this.columnHeader2,
																						this.columnHeader3});
			this.listLocal.FullRowSelect = true;
			this.listLocal.Location = new System.Drawing.Point(400, 80);
			this.listLocal.MultiSelect = false;
			this.listLocal.Name = "listLocal";
			this.listLocal.Size = new System.Drawing.Size(256, 313);
			this.listLocal.TabIndex = 25;
			this.listLocal.View = System.Windows.Forms.View.Details;
			this.listLocal.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.listLocal_DoubleClick);
			this.listLocal.DoubleClick += new System.EventHandler(this.listLocal_DoubleClick);
			this.listLocal.Enter += new System.EventHandler(this.listLocal_Enter);
			// 
			// columnHeader1
			// 
			this.columnHeader1.Text = "Name";
			this.columnHeader1.Width = 120;
			// 
			// columnHeader2
			// 
			this.columnHeader2.Text = "Size";
			this.columnHeader2.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
			// 
			// columnHeader3
			// 
			this.columnHeader3.Text = "Date";
			this.columnHeader3.Width = 120;
			// 
			// tbLog
			// 
			this.tbLog.Anchor = ((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right);
			this.tbLog.BackColor = System.Drawing.SystemColors.Control;
			this.tbLog.Font = new System.Drawing.Font("Courier New", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.tbLog.Location = new System.Drawing.Point(0, 417);
			this.tbLog.Name = "tbLog";
			this.tbLog.ReadOnly = true;
			this.tbLog.Size = new System.Drawing.Size(672, 94);
			this.tbLog.TabIndex = 8;
			this.tbLog.Text = "";
			// 
			// lblServer
			// 
			this.lblServer.Anchor = System.Windows.Forms.AnchorStyles.None;
			this.lblServer.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblServer.Location = new System.Drawing.Point(16, 393);
			this.lblServer.Name = "lblServer";
			this.lblServer.Size = new System.Drawing.Size(256, 13);
			this.lblServer.TabIndex = 9;
			this.lblServer.Text = "12345";
			// 
			// lblServerDirectory
			// 
			this.lblServerDirectory.BackColor = System.Drawing.SystemColors.InactiveCaption;
			this.lblServerDirectory.Enabled = false;
			this.lblServerDirectory.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
			this.lblServerDirectory.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblServerDirectory.ForeColor = System.Drawing.SystemColors.InactiveCaptionText;
			this.lblServerDirectory.Location = new System.Drawing.Point(8, 64);
			this.lblServerDirectory.Name = "lblServerDirectory";
			this.lblServerDirectory.Size = new System.Drawing.Size(264, 16);
			this.lblServerDirectory.TabIndex = 1;
			this.lblServerDirectory.Text = "/root";
			this.lblServerDirectory.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			this.lblServerDirectory.Click += new System.EventHandler(this.listServer_Enter);
			// 
			// lblLocal
			// 
			this.lblLocal.Anchor = System.Windows.Forms.AnchorStyles.None;
			this.lblLocal.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblLocal.Location = new System.Drawing.Point(400, 393);
			this.lblLocal.Name = "lblLocal";
			this.lblLocal.Size = new System.Drawing.Size(256, 13);
			this.lblLocal.TabIndex = 12;
			this.lblLocal.Text = "12345";
			// 
			// lblLocalDirectory
			// 
			this.lblLocalDirectory.BackColor = System.Drawing.SystemColors.InactiveCaption;
			this.lblLocalDirectory.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
			this.lblLocalDirectory.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblLocalDirectory.ForeColor = System.Drawing.SystemColors.InactiveCaptionText;
			this.lblLocalDirectory.Location = new System.Drawing.Point(400, 64);
			this.lblLocalDirectory.Name = "lblLocalDirectory";
			this.lblLocalDirectory.Size = new System.Drawing.Size(256, 16);
			this.lblLocalDirectory.TabIndex = 23;
			this.lblLocalDirectory.Text = "/root";
			this.lblLocalDirectory.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			this.lblLocalDirectory.Click += new System.EventHandler(this.listLocal_Enter);
			// 
			// pbTransfer
			// 
			this.pbTransfer.Anchor = ((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right);
			this.pbTransfer.Location = new System.Drawing.Point(0, 512);
			this.pbTransfer.Name = "pbTransfer";
			this.pbTransfer.Size = new System.Drawing.Size(560, 20);
			this.pbTransfer.Step = 1;
			this.pbTransfer.TabIndex = 20;
			// 
			// lblProgress
			// 
			this.lblProgress.BackColor = System.Drawing.Color.Transparent;
			this.lblProgress.Location = new System.Drawing.Point(568, 512);
			this.lblProgress.Name = "lblProgress";
			this.lblProgress.Size = new System.Drawing.Size(104, 20);
			this.lblProgress.TabIndex = 24;
			this.lblProgress.Text = "Welcome!";
			this.lblProgress.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// btnStop
			// 
			this.btnStop.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnStop.Location = new System.Drawing.Point(16, 232);
			this.btnStop.Name = "btnStop";
			this.btnStop.TabIndex = 17;
			this.btnStop.Text = "Abort";
			this.btnStop.Visible = false;
			this.btnStop.Click += new System.EventHandler(this.btnStop_Click);
			// 
			// btnDelete
			// 
			this.btnDelete.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnDelete.Location = new System.Drawing.Point(16, 152);
			this.btnDelete.Name = "btnDelete";
			this.btnDelete.TabIndex = 11;
			this.btnDelete.Text = "Delete file";
			this.btnDelete.Click += new System.EventHandler(this.btnDelete_Click);
			// 
			// btnUpload
			// 
			this.btnUpload.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnUpload.Location = new System.Drawing.Point(16, 208);
			this.btnUpload.Name = "btnUpload";
			this.btnUpload.TabIndex = 15;
			this.btnUpload.Text = "< Upload";
			this.btnUpload.Click += new System.EventHandler(this.btnUpload_Click);
			// 
			// btnDownload
			// 
			this.btnDownload.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnDownload.Location = new System.Drawing.Point(16, 184);
			this.btnDownload.Name = "btnDownload";
			this.btnDownload.TabIndex = 13;
			this.btnDownload.Text = "Download >";
			this.btnDownload.Click += new System.EventHandler(this.btnDownload_Click);
			// 
			// rbBin
			// 
			this.rbBin.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.rbBin.Location = new System.Drawing.Point(8, 16);
			this.rbBin.Name = "rbBin";
			this.rbBin.Size = new System.Drawing.Size(56, 20);
			this.rbBin.TabIndex = 19;
			this.rbBin.Text = "Binary";
			// 
			// rbAsc
			// 
			this.rbAsc.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.rbAsc.Location = new System.Drawing.Point(8, 34);
			this.rbAsc.Name = "rbAsc";
			this.rbAsc.Size = new System.Drawing.Size(56, 20);
			this.rbAsc.TabIndex = 21;
			this.rbAsc.Text = "ASCII";
			// 
			// btnRmDir
			// 
			this.btnRmDir.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnRmDir.Location = new System.Drawing.Point(16, 128);
			this.btnRmDir.Name = "btnRmDir";
			this.btnRmDir.TabIndex = 9;
			this.btnRmDir.Text = "Remove dir";
			this.btnRmDir.Click += new System.EventHandler(this.btnRmDir_Click);
			// 
			// btnRefresh
			// 
			this.btnRefresh.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnRefresh.Location = new System.Drawing.Point(16, 72);
			this.btnRefresh.Name = "btnRefresh";
			this.btnRefresh.TabIndex = 5;
			this.btnRefresh.Text = "Refresh";
			this.btnRefresh.Click += new System.EventHandler(this.btnRefresh_Click);
			// 
			// btnMakeDir
			// 
			this.btnMakeDir.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnMakeDir.Location = new System.Drawing.Point(16, 104);
			this.btnMakeDir.Name = "btnMakeDir";
			this.btnMakeDir.TabIndex = 7;
			this.btnMakeDir.Text = "Create dir";
			this.btnMakeDir.Click += new System.EventHandler(this.btnMakeDir_Click);
			// 
			// cbDrives
			// 
			this.cbDrives.ItemHeight = 13;
			this.cbDrives.Location = new System.Drawing.Point(16, 40);
			this.cbDrives.Name = "cbDrives";
			this.cbDrives.Size = new System.Drawing.Size(72, 21);
			this.cbDrives.TabIndex = 4;
			this.cbDrives.SelectedIndexChanged += new System.EventHandler(this.cbDrives_SelectedIndexChanged);
			// 
			// btnConnect
			// 
			this.btnConnect.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.btnConnect.Location = new System.Drawing.Point(16, 16);
			this.btnConnect.Name = "btnConnect";
			this.btnConnect.TabIndex = 3;
			this.btnConnect.Text = "Connect...";
			this.btnConnect.Click += new System.EventHandler(this.btnConnect_Click);
			// 
			// gbControl
			// 
			this.gbControl.Anchor = System.Windows.Forms.AnchorStyles.Top;
			this.gbControl.Controls.AddRange(new System.Windows.Forms.Control[] {
																					this.btnStop,
																					this.btnDelete,
																					this.btnUpload,
																					this.btnDownload,
																					this.groupBox1,
																					this.btnRmDir,
																					this.btnRefresh,
																					this.btnMakeDir,
																					this.cbDrives,
																					this.btnConnect});
			this.gbControl.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.gbControl.Location = new System.Drawing.Point(288, 64);
			this.gbControl.Name = "gbControl";
			this.gbControl.Size = new System.Drawing.Size(106, 328);
			this.gbControl.TabIndex = 25;
			this.gbControl.TabStop = false;
			this.gbControl.Text = "Controls";
			// 
			// groupBox1
			// 
			this.groupBox1.Controls.AddRange(new System.Windows.Forms.Control[] {
																					this.rbBin,
																					this.rbAsc});
			this.groupBox1.FlatStyle = System.Windows.Forms.FlatStyle.System;
			this.groupBox1.Location = new System.Drawing.Point(16, 264);
			this.groupBox1.Name = "groupBox1";
			this.groupBox1.Size = new System.Drawing.Size(72, 56);
			this.groupBox1.TabIndex = 29;
			this.groupBox1.TabStop = false;
			this.groupBox1.Text = "Transfer";
			// 
			// headerLeft
			// 
			this.headerLeft.BackColor = System.Drawing.Color.White;
			this.headerLeft.Image = ((System.Drawing.Bitmap)(resources.GetObject("headerLeft.Image")));
			this.headerLeft.Name = "headerLeft";
			this.headerLeft.Size = new System.Drawing.Size(472, 59);
			this.headerLeft.TabIndex = 26;
			this.headerLeft.TabStop = false;
			// 
			// headerMiddle
			// 
			this.headerMiddle.BackColor = System.Drawing.Color.White;
			this.headerMiddle.BackgroundImage = ((System.Drawing.Bitmap)(resources.GetObject("headerMiddle.BackgroundImage")));
			this.headerMiddle.Location = new System.Drawing.Point(472, 0);
			this.headerMiddle.Name = "headerMiddle";
			this.headerMiddle.Size = new System.Drawing.Size(64, 59);
			this.headerMiddle.TabIndex = 27;
			this.headerMiddle.TabStop = false;
			// 
			// headerRight
			// 
			this.headerRight.BackColor = System.Drawing.Color.White;
			this.headerRight.Cursor = System.Windows.Forms.Cursors.Hand;
			this.headerRight.Image = ((System.Drawing.Bitmap)(resources.GetObject("headerRight.Image")));
			this.headerRight.Location = new System.Drawing.Point(536, 0);
			this.headerRight.Name = "headerRight";
			this.headerRight.Size = new System.Drawing.Size(136, 59);
			this.headerRight.TabIndex = 28;
			this.headerRight.TabStop = false;
			this.headerRight.Click += new System.EventHandler(this.headerRight_Click);
			// 
			// MainForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(672, 533);
			this.Controls.AddRange(new System.Windows.Forms.Control[] {
																		  this.lblServerDirectory,
																		  this.headerRight,
																		  this.headerMiddle,
																		  this.headerLeft,
																		  this.gbControl,
																		  this.lblProgress,
																		  this.pbTransfer,
																		  this.lblLocalDirectory,
																		  this.lblLocal,
																		  this.lblServer,
																		  this.tbLog,
																		  this.listLocal,
																		  this.listServer});
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.KeyPreview = true;
			this.MinimumSize = new System.Drawing.Size(664, 560);
			this.Name = "MainForm";
			this.Text = "Simple WinForm FTP Client";
			this.Resize += new System.EventHandler(this.MainForm_Resize);
			this.Closing += new System.ComponentModel.CancelEventHandler(this.MainForm_Closing);
			this.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.MainForm_KeyPress);
			this.gbControl.ResumeLayout(false);
			this.groupBox1.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// very Simple FTP client starts here
		/// </summary>
		[STAThread]
		static void Main() 
		{
			Application.Run(new MainForm());
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// connect/disconnect ftp server
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnConnect_Click(object sender,System.EventArgs e)
		{
			if (_ftp==null)
			{							
				// ---------------------------------
				// create ftp object [based on site]
				// ---------------------------------

				_ftp=CreateFtp(_site);

				// -----------------
				// connection dialog
				// -----------------

				WinFormClient.Connection con=new WinFormClient.Connection();

				LoadConfig();

				// ----------------------
				// show connection dialog
				// ----------------------

				con.Host=_site.Host;
				con.Port=_site.Port;
				con.Login=_site.Login;
				con.Password=_site.Password;
				con.Passive=_site.Passive;
				con.ProxyEnabled=_site.ProxyEnabled;
				con.Proxy=_site.Proxy;
				con.ProxyPort=_site.ProxyPort;
				con.ProxyType=_site.ProxyType;
				con.ProxyLogin=_site.ProxyUser;
				con.ProxyPassword=_site.ProxyPassword;

				con.ShowDialog();

				// -------------
				// not confirmed
				// -------------

				if (!con.OK) 
				{
					_ftp=null;
					return;
				}
				
				_site.Host=con.Host;
				_site.Port=con.Port;
				_site.Login=con.Login;
				_site.Password=con.Password;							
				_site.Passive=con.Passive;	
				_site.ProxyEnabled=con.ProxyEnabled;
				_site.Proxy=con.Proxy;
 				_site.ProxyPort=con.ProxyPort;		
				_site.ProxyType=con.ProxyType;
				_site.ProxyUser=con.ProxyLogin;
				_site.ProxyPassword=con.ProxyPassword;

				SaveConfig();

				// -------
				// connect
				// -------

				try
				{
					WriteToLog("> connecting to "+_site.Host+":"+_site.Port,COLORCOMMAND);

					_ftp.Passive=_site.Passive;							
					_ftp.Timeout=_site.Timeout;					

					// -----
					// proxy
					// -----

					if (_site.ProxyEnabled)
					{
						if (_site.ProxyUser.Length>0)
						{
							if (_site.ProxyPassword.Length==0) _site.ProxyPassword=null;

							_ftp.Proxy=new FtpProxy((FtpProxyType)_site.ProxyType,_site.Proxy,_site.ProxyPort,new System.Net.NetworkCredential(_site.ProxyUser,_site.ProxyPassword));
							WriteToLog("> using proxy "+_site.Proxy+":"+_site.ProxyPort,COLORCOMMAND);
						}
						else
						{
							_ftp.Proxy=new FtpProxy((FtpProxyType)_site.ProxyType,_site.Proxy,_site.ProxyPort);
							WriteToLog("> using proxy "+_site.Proxy+":"+_site.ProxyPort,COLORCOMMAND);
						}
					}

					_ftp.Connect(_site.Host,_site.Port);
					_ftp.Login(_site.Login,_site.Password);			
					
					btnConnect.Text="disconnect";
				}

				catch(FtpException ex)
				{
					WriteToLog("* "+ex.Message,COLORERROR);
					_ftp=null;
					return;
				}

				MakeFtpList();		

				return;
			}

			// -------------
			// disconnecting
			// -------------
				
			if (IsFtpReady())
			{				
				_ftp.Disconnect();
				_ftp.Dispose();
				_ftp=null;
				btnConnect.Text="connect";
				MakeFtpList(); // clearing
				return;
			}			  		
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// disconnect on exit
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MainForm_Closing(object sender,System.ComponentModel.CancelEventArgs e)
		{
			if (IsFtpReady())
			{				
				_ftp.Disconnect();
				_ftp.Dispose();
			}
		}
		
		// -------------------------------------------------------------------------------------
		/// <summary>
		/// CWD or GET file command on server side
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ListServerClick()
		{
			ActivatePanel(0);

			// --------------
			// ftp state test
			// --------------

			if (!IsFtpReady())
			{
				WriteToLog("* ftp is not ready",COLORERROR);
				return;
			}

			int index=listServer.SelectedIndices[0];		
			ListViewItem item=listServer.SelectedItems[0];						
			string[] row=new string[2];
			
			// -----------------------
			// parse listitem to row[]
			// -----------------------

			row[0]=item.Text;			
			if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
									
			string directory=null;

			// --------
			// GET file
			// --------

			if (index>=_serverDirs)			
			{
				int size;

				try
				{
					size=Int32.Parse(row[1].ToString());
				}
				catch
				{
					size=0;
				}
								
				DownloadFile(row[0],_localPath+row[0],size);												
			}
			else
			{				
				// ---
				// CWD
				// ---				

				if (row[1]==null || row[1]=="")
				{
					if (index==0)
					{					
						directory="..";					
					}
					else 
					{															
						directory=row[0];
					}

					try
					{
						_ftp.ChangeDirectory(directory);
					}

					catch(FtpException)
					{						
						return;					
					}
				}
				
				// -----------------------
				// symlink directory click
				// -----------------------				

				if (row[1]==SYMLINK)
				{					
					try
					{						
						_ftp.ChangeDirectory(row[0]);
					}

					catch(FtpException)
					{						
						return;					
					}
				}

				MakeFtpList();
			}
		}

		private void listServer_DoubleClick(object sender,System.Windows.Forms.KeyPressEventArgs e)
		{
			if (e.KeyChar=='\r') 
			{
				ListServerClick();				
				if (listServer.Items.Count>0)
				{
					listServer.Items[0].Selected=true;
					listServer.Items[0].Focused=true;
				}				
			}
		}
		
		private void listServer_DoubleClick(object sender,System.EventArgs e)
		{			
			ListServerClick();
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// change drive letter
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cbDrives_SelectedIndexChanged(object sender, System.EventArgs e)
		{			
			try
			{
				_localPath=cbDrives.SelectedItem.ToString();
				MakeLocalList();
			}

			catch(Exception ex)
			{
				WriteToLog("* "+ex.Message,COLORERROR,false);
				listLocal.Items.Clear();
				lblLocalDirectory.Text=lblLocal.Text=null;
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// LCD or PUT command on local side
		/// </summary>				
		private void ListLocalClick()
		{
			ActivatePanel(1);

			// --------------
			// ftp state test
			// --------------			

			int index=listLocal.SelectedIndices[0];
			ListViewItem item=listLocal.SelectedItems[0];						
			string[] row=new string[2];					

			// -----------------------
			// parse listitem to row[]
			// -----------------------

			row[0]=item.Text;			
			if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
									
			string directory=null;
			
			if (index>=_localDirs)	
			{				
				if (!IsFtpReady())
				{
					WriteToLog("* ftp is not ready",COLORERROR);
					return;
				}

				UploadFile(row[0],_localPath+row[0],Int32.Parse(row[1]));				
			}
			else
			{				
				// ---
				// LCD
				// ---

				if (row[1]==null || row[1]=="")
				{
					directory=row[0];						
					_localPath+=directory+"\\";																	
				}
								
				MakeLocalList();
			}					
		}

		private void listLocal_DoubleClick(object sender,System.Windows.Forms.KeyPressEventArgs e)
		{
			if (e.KeyChar=='\r') 
			{
				ListLocalClick();				
				if (listLocal.Items.Count>0)
				{
					listLocal.Items[0].Selected=true;				
					listLocal.Items[0].Focused=true;
				}
			}
		}
		
		private void listLocal_DoubleClick(object sender,System.EventArgs e)
		{		
			ListLocalClick();
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// make dir
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnMakeDir_Click(object sender,System.EventArgs e)
		{
			MakeDir dir=new MakeDir();

			// ----------
			// FTP: MKDIR
			// ----------

			if (SelectedFTP() && IsFtpReady())
			{
				dir.ShowDialog();	
				
				if (dir.Directory!=null)
				{
					try
					{
						_ftp.CreateDirectory(dir.Directory);
						MakeFtpList();
					}

					catch(FtpException)
					{						
					}
				}
			}

			// ------------
			// LOCAL: MKDIR
			// ------------

			if (SelectedLocal())
			{
				dir.ShowDialog();	
				
				if (dir.Directory!=null)
				{
					try
					{
						Directory.CreateDirectory(_localPath+dir.Directory+"//");						
						MakeLocalList();
					}

					catch(Exception ex)
					{				
						WriteToLog("* "+ex.Message,COLORERROR);
					}
				}
			}
			
			dir=null;
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// refresh listing
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnRefresh_Click(object sender, System.EventArgs e)
		{		
			// ----------
			// FTP: RFRSH
			// ----------

			if (SelectedFTP() && IsFtpReady())
			{
				MakeFtpList();
			}

			// ------------
			// LOCAL: RFRSH
			// ------------

			if (SelectedLocal())
			{
				MakeLocalList();
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// download file
		/// </summary>
		/// <param name="remote">remote address</param>
		/// <param name="local">local address</param>
		/// <param name="size">file-size</param>
		private void DownloadFile(string remote,string local,long size)
		{
			// --------------
			// ftp state test
			// --------------			

			if (!IsFtpReady()) 
			{
				WriteToLog("* ftp is not ready ",COLORERROR);
				return;
			}

			// ---------------------
			// set asc/bin transfers
			// ---------------------

			if (rbBin.Checked)
				_ftp.SetTransferType(FtpTransferType.Binary);
			else
				_ftp.SetTransferType(FtpTransferType.Ascii);

			// --------
			// transfer
			// --------

			try
			{
				IAsyncResult asyncResult;
			
				asyncResult=_ftp.BeginGetFile(remote,local,new AsyncCallback(DownloadCallback),null);

				btnStop.Visible=true;
				_fileSize=size;						
				_transferTime=DateTime.Now;
			}

			catch
			{
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// upload file
		/// </summary>
		/// <param name="remote">remote address</param>
		/// <param name="local">local address</param>
		/// <param name="size">file-size</param>
		private void UploadFile(string remote,string local,long size)
		{
			// --------------
			// ftp state test
			// --------------

			if (!IsFtpReady()) 
			{
				WriteToLog("* ftp is not ready",COLORERROR);
				return;
			}

			// ---------------------
			// set asc/bin transfers
			// ---------------------

			if (rbBin.Checked)
				_ftp.SetTransferType(FtpTransferType.Binary);
			else
				_ftp.SetTransferType(FtpTransferType.Ascii);

			// --------
			// transfer
			// --------

			try
			{
				IAsyncResult asyncResult;
			
				asyncResult=_ftp.BeginPutFile(local,remote,new AsyncCallback(UploadCallback),null);
				btnStop.Visible=true;

				_fileSize=size;							
				_transferTime=DateTime.Now;
			}

			catch
			{
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// show transfer status: bytes, time & speed
		/// </summary>
		private void ShowTransferStatus()
		{
			// ----------------
			// aborted transfer
			// ----------------

			if (_fileSize==-1) return;

			// -------------------------
			// unknown size or null size
			// -------------------------

			if (_fileSize==0) return;

			// -----------------
			// bytes transferred
			// -----------------

			string outstring=_fileSize+" byte"+(_fileSize>1 ? "s":null)+" transferred";
						
			// ----------
			// time spent
			// ----------

			TimeSpan ts=DateTime.Now-_transferTime;

			// -----
			// speed
			// -----

			if (ts.TotalSeconds>1)
			{
				outstring+=" in"+(ts.Days>0 ? " "+ts.Days+" day"+(ts.Days>1 ? "s":null):null);
				outstring+=(ts.Hours>0 ? " "+ts.Hours+" hour"+(ts.Hours>1 ? "s":null):null);
				outstring+=(ts.Minutes>0 ? " "+ts.Minutes+" min"+(ts.Minutes>1 ? "s":null):null);
				outstring+=(ts.Seconds>0 ? " "+ts.Seconds+" sec"+(ts.Seconds>1 ? "s":null):null);
			
				outstring+=" at "+((long)_fileSize/(long)ts.TotalSeconds)/1024+" KB/s";
			}
			else
			{
				outstring+=" in "+ts.TotalSeconds+" sec";
			}

			WriteToLog("> "+outstring,COLORCOMMAND);
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// callback for downloading
		/// </summary>
		/// <param name="asyncResult"></param>
		public void DownloadCallback(IAsyncResult asyncResult)
		{					
			asyncResult.AsyncWaitHandle.WaitOne();
			
			try
			{
				long bytes=_ftp.EndGetFile(asyncResult);							
			}

			catch(FtpException)
			{				
			}
			
			ShowTransferStatus();
			_fileSize=-1;			
			pbTransfer.Value=0;			
			btnStop.Visible=false;
			
			MakeLocalList();			
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// callback for uploading
		/// </summary>
		/// <param name="asyncResult"></param>
		public void UploadCallback(IAsyncResult asyncResult)
		{					
			asyncResult.AsyncWaitHandle.WaitOne();
			
			try
			{
				long bytes=_ftp.EndPutFile(asyncResult);							
			}

			catch(FtpException e)
			{
				WriteToLog("* "+e.Message,COLORERROR);
			}
			
			ShowTransferStatus();
			_fileSize=-1;
			pbTransfer.Value=0;
			btnStop.Visible=false;

			MakeFtpList();			
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// remove dir
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnRmDir_Click(object sender,System.EventArgs e)
		{
			RemoveDir dir=new RemoveDir();

			// ----------
			// FTP: RMDIR
			// ----------

			if (SelectedFTP() && IsFtpReady())
			{
				int index=listServer.SelectedIndices[0];
				ListViewItem item=listServer.SelectedItems[0];						
				string[] row=new string[2];							
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
												
				if (index>=_serverDirs)			
				{
					WriteToLog("* "+row[0]+" is not directory --- cannot apply RMDIR command",COLORERROR);			
				}
				else
				{				
					dir.Directory=row[0];
					dir.ShowDialog();	
				
					if (dir.OK)
					{
						try
						{
							_ftp.RemoveDirectory(dir.Directory);
							MakeFtpList();
						}

						catch(FtpException)
						{						
						}
					}
				}
			}

			// ------------
			// LOCAL: RMDIR
			// ------------

			if (SelectedLocal())
			{
				int index=listLocal.SelectedIndices[0];
				ListViewItem item=listLocal.SelectedItems[0];						
				string[] row=new string[2];							
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
												
				if (index>=_localDirs)			
				{
					WriteToLog("* "+row[0]+" is not directory --- cannot apply RMDIR command",COLORERROR);			
				}
				else
				{				
					dir.Directory=row[0];
					dir.ShowDialog();	
				
					if (dir.OK)
					{
						try
						{
							Directory.Delete(_localPath+row[0]+"//");							
							MakeLocalList();
						}

						catch(Exception ex)
						{				
							WriteToLog("* "+ex.Message,COLORERROR);
						}					
					}
				}			
			}
			
			dir=null;		
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// download button
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnDownload_Click(object sender,System.EventArgs e)
		{
			if (SelectedFTP() && IsFtpReady())
			{
				int index=listServer.SelectedIndices[0];
				ListViewItem item=listServer.SelectedItems[0];						
				string[] row=new string[2];
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;

				if (index>=_serverDirs) 
					DownloadFile(row[0],_localPath+row[0],Int32.Parse(row[1]));		
				else
					WriteToLog("* directory downloading is not supported",COLORERROR);
			}
			else
			{
				if (!IsFtpReady()) WriteToLog("* ftp is not ready",COLORERROR);
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// upload button
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnUpload_Click(object sender,System.EventArgs e)
		{
			if (SelectedLocal() && IsFtpReady())
			{
				int index=listLocal.SelectedIndices[0];
				ListViewItem item=listLocal.SelectedItems[0];						
				string[] row=new string[2];
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;

				if (index>=_localDirs) 
					UploadFile(row[0],_localPath+row[0],Int32.Parse(row[1]));		
				else					
					WriteToLog("* directory uploading is not supported",COLORERROR);			
			}
			else
			{
				if (!IsFtpReady()) WriteToLog("* ftp is not ready",COLORERROR);
			}
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// delete file
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnDelete_Click(object sender, System.EventArgs e)
		{
			RemoveFile file=new RemoveFile();

			// --------
			// FTP: DEL
			// --------

			if (SelectedFTP() && IsFtpReady())
			{
				int index=listServer.SelectedIndices[0];
				ListViewItem item=listServer.SelectedItems[0];						
				string[] row=new string[2];							
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
												
				if (index>=_serverDirs)			
				{
					file.Path=row[0];
					file.ShowDialog();	
				
					if (file.OK)
					{
						try
						{							
							_ftp.DeleteFile(file.Path);
							MakeFtpList();
						}

						catch(FtpException)
						{						
						}
					}					
				}
				else
				{				
					WriteToLog("* "+row[0]+" is directory --- cannot apply DEL command",COLORERROR);			
				}
			}

			// ----------
			// LOCAL: DEL
			// ----------

			if (SelectedLocal())
			{
				int index=listLocal.SelectedIndices[0];
				ListViewItem item=listLocal.SelectedItems[0];						
				string[] row=new string[2];							
				row[0]=item.Text;			
				if (item.SubItems.Count>1) row[1]=item.SubItems[1].Text;else row[1]=null;
												
				if (index>=_localDirs)			
				{
					file.Path=_localPath+row[0];
					file.ShowDialog();	
				
					if (file.OK)
					{
						try
						{
							File.Delete(file.Path);														
							MakeLocalList();
						}

						catch(Exception ex)
						{				
							WriteToLog("* "+ex.Message,COLORERROR);
						}					
					}
				}
				else
				{	
					WriteToLog("* "+row[0]+" is directory --- cannot apply DEL command",COLORERROR);								
				}			
			}
			
			file=null;												
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// activate panel on focus event
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void listServer_Enter(object sender, System.EventArgs e)
		{
			ActivatePanel(0);
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// activate panel on focus event
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void listLocal_Enter(object sender, System.EventArgs e)
		{
			ActivatePanel(1);
		}
		
		// -------------------------------------------------------------------------------------
		/// <summary>
		/// "smart" resizing
		/// </summary>		
		private void SetRightSizeValues()
		{
			int borderWidth=10;
			int borderControl=10;

			// width of panels

			int listWidth=(this.Size.Width-gbControl.Size.Width-2*borderControl-2*borderWidth)/2;
			listLocal.Width=listWidth;
			listServer.Width=listWidth;			
			listLocal.Location=new Point(borderWidth+listWidth+borderControl+gbControl.Size.Width+borderControl,listServer.Location.Y);

			// status info for panels

			lblServer.Location=new Point(listServer.Location.X,listServer.Location.Y+listServer.Size.Height+5);			
			lblLocal.Location=new Point(listLocal.Location.X,listLocal.Location.Y+listLocal.Size.Height+5);

			// local directory

			lblLocalDirectory.Location=new Point(listLocal.Location.X,lblLocalDirectory.Location.Y);
			lblLocalDirectory.Width=listLocal.Size.Width;

			// server directory

			lblServerDirectory.Location=new Point(listServer.Location.X,lblServerDirectory.Location.Y);
			lblServerDirectory.Width=listServer.Size.Width;

			// state

			lblProgress.Location=new Point(pbTransfer.Location.X+pbTransfer.Width+8,pbTransfer.Location.Y);

			// header

			headerRight.Location=new Point(this.ClientSize.Width-headerRight.Width,headerRight.Location.Y);
			headerMiddle.Width=headerRight.Location.X-headerMiddle.Location.X;
		}

		private void MainForm_Resize(object sender,System.EventArgs e)
		{
			if (WindowState!=FormWindowState.Minimized) SetRightSizeValues();
		}

		// -------------------------------------------------------------------------------------
		/// <summary>
		/// stop transfer when something is being transferred
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnStop_Click(object sender,System.EventArgs e)
		{
			_ftp.Abort();
			_fileSize=-1;
		}

		// -------------------------------------------------------------------------
		/// <summary>
		/// go to the url
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void headerRight_Click(object sender,System.EventArgs e)
		{
			System.Diagnostics.Process.Start(URL);				
		}

		// -------------------------------------------------------------------------
		/// <summary>
		/// space bar for changing panels server <-> local
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MainForm_KeyPress(object sender,System.Windows.Forms.KeyPressEventArgs e)
		{
			if (e.KeyChar==Convert.ToChar(Keys.Space))
			{
				ActivatePanel(_activePanel==0 ? 1:0);								
			}
		}				
	}
}
