using System;
using System.IO;
using System.Configuration;
using System.Threading;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	/// <summary>
	/// Tests asynchronous methods.
	/// </summary>
	abstract public class FtpAsyncTest
	{
		public Site Site = null;

		private enum Method
		{
			CreateDirectoryError,
			ChangeDirectoryError,
			CreateDirectory,
			ChangeDirectory,
			GetCurrentDirectory,
			RemoveDirectory
		}

		private Ftp ftp;
		private Method method;
		private ManualResetEvent block;
		private string folder;
		private Exception callbackError;
		private bool success;

		public void MyCallback (IAsyncResult ar)
		{
			success = false;
			callbackError = null;
			try
			{
				Assertion.AssertEquals ("Wrong async state.", method, (Method)ar.AsyncState);

				string path = null;

				switch (method)
				{
					case Method.CreateDirectoryError:
						ftp.EndCreateDirectory (ar);
						Assertion.Fail ("The call should have failed.");
						break;

					case Method.ChangeDirectoryError:
						ftp.EndCreateDirectory (ar);
						Assertion.Fail ("The call should have failed.");
						break;

					case Method.CreateDirectory:
						path = ftp.EndCreateDirectory (ar);
						if (path != null)
							Assertion.Assert ("Wrong path returned.", path.EndsWith (folder));
						Assertion.AssertEquals ("Ftp is in wrong state.", FtpState.Ready, ftp.State);
						break;

					case Method.ChangeDirectory:
						ftp.EndChangeDirectory (ar);
						Assertion.AssertEquals ("Ftp is in wrong state.", FtpState.Ready, ftp.State);
						break;

					case Method.GetCurrentDirectory:
						path = ftp.EndGetCurrentDirectory (ar);
						Assertion.AssertEquals ("Wrong directory returned.", Site.UploadDir + "/" + folder, path);
						break;

					case Method.RemoveDirectory:
						ftp.EndRemoveDirectory (ar);
						Assertion.AssertEquals ("Ftp is in wrong state.", FtpState.Ready, ftp.State);
						break;

					default:
						Assertion.Fail ("Invalid method.");
						break;
				}

				success = true;
			}
			catch (Exception e)
			{
				success = false;
				callbackError = e;
			}
			finally
			{
				block.Set();
			}
		}

		public void MyCallback2 (IAsyncResult ar)
		{
			success = true;
			block.Set();
		}

		public void WaitFinish (int seconds)
		{
			int i = seconds * 1000;
			while (!block.WaitOne (10, false))
			{
				Utils.ConsoleFlush ();
				seconds -= 10;
				Assertion.Assert ("Asynchronous method execution timeout of " + seconds + " expired.", (seconds <= 0));
			}
			Utils.ConsoleFlush ();

			if (!success)
			{
				Utils.ConsoleWriteLine (callbackError.ToString());
				throw callbackError;
			}
		}

		[Test]
		public void TestAsyncEndInCallback()
		{
			Console.WriteLine ("-- TestAsyncEndInCallback --------------------------");

			ftp = null;
			success = false;
			try
			{
				// initialization
				ftp = Utils.ConnectFtp (Site);
				ftp.ChangeDirectory (Site.UploadDir);
				ftp.SetTransferType (FtpTransferType.Binary);
				folder = Utils.GetUniqueName(ftp, "folder", "");
				block = new ManualResetEvent (false);


				// invalid CreateDirectory
				try
				{
					block.Reset ();
					method = Method.CreateDirectory;
					ftp.BeginCreateDirectory (Site.NoModifyDir + "/test", new AsyncCallback(MyCallback), method);
					WaitFinish (10);
					Assertion.Fail ("The call should have failed.");
				}
				catch (FtpException e)
				{
					Assertion.AssertEquals ("The FtpException status is not ProtocolError.", FtpExceptionStatus.ProtocolError, ((FtpException)e).Status);
					Assertion.AssertEquals ("The FtpException group is not 5.", 5, ((FtpException)e).Response.Group);
				}


				// invalid ChangeDirectory
				try
				{
					block.Reset ();
					method = Method.ChangeDirectory;
					ftp.BeginChangeDirectory (folder, new AsyncCallback(MyCallback), method);
					WaitFinish (10);
					Assertion.Fail ("The call should have failed.");
				}
				catch (FtpException e)
				{
					Assertion.AssertEquals ("The FtpException status is not ProtocolError.", FtpExceptionStatus.ProtocolError, ((FtpException)e).Status);
					Assertion.AssertEquals ("The FtpException group is not 5.", 5, ((FtpException)e).Response.Group);
				}


				// correct CreateDirectory
				block.Reset ();
				method = Method.CreateDirectory;
				ftp.BeginCreateDirectory (folder, new AsyncCallback(MyCallback), method);
				WaitFinish (10);


				// correct ChangeDirectory
				block.Reset();
				method = Method.ChangeDirectory;
				ftp.BeginChangeDirectory (folder, new AsyncCallback(MyCallback), method);
				WaitFinish (10);


				// correct GetCurrentDirectory
				block.Reset();
				method = Method.GetCurrentDirectory;
				ftp.BeginGetCurrentDirectory (new AsyncCallback(MyCallback), method);
				WaitFinish (10);
				
				// correct RemoveDirectory
				ftp.ChangeDirectory ("..");
				block.Reset ();
				method = Method.RemoveDirectory;
				ftp.BeginRemoveDirectory (folder, new AsyncCallback(MyCallback), method);
				WaitFinish (10);


				// cleanup
				ftp.Disconnect();
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		[Test]
		public void TestAsyncEndInCaller()
		{
			Console.WriteLine ("-- TestAsyncEndInCaller --------------------------");

			ftp = null;
			success = false;
			try
			{
				// initialization
				ftp = Utils.ConnectFtp (Site);
				ftp.ChangeDirectory (Site.UploadDir);
				ftp.SetTransferType (FtpTransferType.Binary);
				folder = Utils.GetUniqueName(ftp, "folder", "");
				block = new ManualResetEvent (false);
				IAsyncResult ar;


				// invalid CreateDirectory
				try
				{
					block.Reset ();
					method = Method.CreateDirectoryError;
					ar = ftp.BeginCreateDirectory (Site.NoModifyDir + "/test", new AsyncCallback(MyCallback2), method);
					ftp.EndCreateDirectory (ar);
					Assertion.Fail ("The call should have failed.");
				}
				catch (FtpException e)
				{
					Assertion.AssertEquals ("The FtpException status is not ProtocolError.", FtpExceptionStatus.ProtocolError, ((FtpException)e).Status);
					Assertion.AssertEquals ("The FtpException group is not 5.", 5, ((FtpException)e).Response.Group);
					WaitFinish (10);
				}


				// invalid ChangeDirectory
				try
				{
					block.Reset ();
					method = Method.ChangeDirectory;
					ar = ftp.BeginChangeDirectory (folder, new AsyncCallback(MyCallback2), method);
					ftp.EndChangeDirectory (ar);
					Assertion.Fail ("The call should have failed.");
				}
				catch (FtpException e)
				{
					Assertion.AssertEquals ("The FtpException status is not ProtocolError.", FtpExceptionStatus.ProtocolError, ((FtpException)e).Status);
					Assertion.AssertEquals ("The FtpException group is not 5.", 5, ((FtpException)e).Response.Group);
					WaitFinish (10);
				}


				// correct CreateDirectory
				block.Reset ();
				method = Method.CreateDirectory;
				ar = ftp.BeginCreateDirectory (folder, new AsyncCallback(MyCallback2), method);
				ftp.EndCreateDirectory (ar);
				WaitFinish (10);


				// correct ChangeDirectory
				block.Reset ();
				method = Method.ChangeDirectory;
				ar = ftp.BeginChangeDirectory (folder, new AsyncCallback(MyCallback2), method);
				ftp.EndChangeDirectory (ar);
				WaitFinish (10);


				// correct GetCurrentDirectory
				block.Reset ();
				method = Method.GetCurrentDirectory;
				ar = ftp.BeginGetCurrentDirectory (new AsyncCallback(MyCallback2), method);
				ftp.EndGetCurrentDirectory (ar);
				WaitFinish (10);

				
				// correct RemoveDirectory
				ftp.ChangeDirectory ("..");
				block.Reset ();
				method = Method.RemoveDirectory;
				ar = ftp.BeginRemoveDirectory (folder, new AsyncCallback(MyCallback2), method);
				ftp.EndRemoveDirectory (ar);
				WaitFinish (10);


				// cleanup
				ftp.Disconnect();
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		[Test]
		public void TestAsyncSimple()
		{
			Console.WriteLine ("-- TestAsyncSimple --------------------------");

			Ftp ftp = null;
			success = false;
			try
			{
				string res;
				ftp = Utils.ConnectFtp (Site);

				// Logoff and Disconnect
				try
				{
					ftp.EndLogoff (ftp.BeginLogoff (null, null));
				}
				catch (FtpException e)
				{
					Assertion.AssertEquals (FtpExceptionStatus.ProtocolError, e.Status);
				}

				ftp.EndDisconnect (ftp.BeginDisconnect (null, null));

				// Connect and Login
				res = ftp.EndConnect (ftp.BeginConnect (Site.Host, Site.Port, null, null));
				Assertion.AssertNotNull (res);
				res = ftp.EndLogin (ftp.BeginLogin (Site.Login, Site.Password, null, null));
				Assertion.AssertNotNull (res);

				// get unique folder
				string folder = Utils.GetUniqueName(ftp, "folder", "");

				// SetTransferType
				ftp.EndSetTransferType (ftp.BeginSetTransferType (FtpTransferType.Binary, null, null));

				// CreateDirectory and ChangeDirectory
				ftp.EndCreateDirectory(ftp.BeginCreateDirectory (folder, null, null));
				ftp.EndChangeDirectory(ftp.BeginChangeDirectory (folder, null, null));

				// GetCurrentDirectory
				res = ftp.EndGetCurrentDirectory (ftp.BeginGetCurrentDirectory (null, null));
				Assertion.Assert ("Wrong path returned.", res.EndsWith (folder));

				MemoryStream data = new MemoryStream (new byte[] {1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16});

				// PutFile
				data.Position = 0;
				ftp.EndPutFile (ftp.BeginPutFile (data, "file.bin", null, null));
				data.Position = 0;
				ftp.EndPutFile (ftp.BeginPutFile (data, "file.bin", 8, 12, null, null));

				// GetFileLength
				long l = ftp.EndGetFileLength (ftp.BeginGetFileLength ("file.bin", null, null));
				Assertion.AssertEquals (20, l);

				// AppendFile
				data.Position = 0;
				ftp.EndAppendFile (ftp.BeginAppendFile (data, "file.bin", null, null));
				data.Position = 0;
				ftp.EndAppendFile (ftp.BeginAppendFile (data, "file.bin", 8, null, null));

				// GetFileDateTime
				DateTime dt = ftp.EndGetFileDateTime (ftp.BeginGetFileDateTime ("file.bin", null, null));
				Assertion.Assert (dt > DateTime.Now.AddDays (-1));

				// Rename
				ftp.EndRename (ftp.BeginRename ("file.bin", "file2.bin", null, null));

				// GetFile
				data = new MemoryStream ();
				l = ftp.EndGetFile (ftp.BeginGetFile ("file2.bin", data, null, null));
				Assertion.AssertEquals (44, l);
				Assertion.AssertEquals (44, data.Length);
				data = new MemoryStream ();
				l = ftp.EndGetFile (ftp.BeginGetFile ("file2.bin", data, 16, null, null));
				Assertion.AssertEquals (28, l);
				Assertion.AssertEquals (28, data.Length);

				// GetList, GetNameList, GetRawList
				FtpList list = ftp.EndGetList (ftp.BeginGetList (null, null));
				Assertion.AssertEquals (1, list.Count);
				Assertion.AssertEquals ("file2.bin", list[0].Name.ToLower());
				string[] nameList = ftp.EndGetNameList (ftp.BeginGetNameList (null, null));
				Assertion.AssertEquals (1, nameList.Length);
				string[] rawList = ftp.EndGetRawList (ftp.BeginGetRawList (null, null));
				Assertion.Assert (rawList.Length >= 1);

				// KeepAlive, GetSystemName, Site
				ftp.EndKeepAlive (ftp.BeginKeepAlive (null, null));
				res = ftp.EndGetSystemName (ftp.BeginGetSystemName (null, null));
				Assertion.AssertNotNull (res);
				res = ftp.EndSite (ftp.BeginSite ("HELP", null, null));
				Assertion.AssertNotNull (res);

				// DeleteFile
				ftp.EndDeleteFile (ftp.BeginDeleteFile ("file2.bin", null, null));

				ftp.EndSendCommand (ftp.BeginSendCommand ("PWD", null, null));
				FtpResponse ftpRes = ftp.EndReadResponse (ftp.BeginReadResponse (null, null));
				Assertion.AssertNotNull (ftpRes);

				// cleanup
				ftp.ChangeDirectory ("..");
				ftp.RemoveDirectory (folder);
				ftp.Disconnect();
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}
		
	}
}
