using System;
using System.IO;
using System.Configuration;
using System.Net;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	/// <summary>
	/// Tests basic FTP methods
	/// </summary>
	abstract public class FtpBasicTest
	{
		public Site Site = null;
		
		/*** TEST TEMPLATE ***/

		/* Test_Name
		 - Test_Description
		Test
		public void Test_Name()
		{
			Console.WriteLine ("---------------- Test_Name ----------------");			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
			//TEST HERE	
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}*/




		// TestCreateDir
		// - Dirs are created commonly in other tests (relatively an absolutely)
		// This test tests other situations
		// - incorrect params - empty, null, existing, strange data
		[Test]
		public void TestCreateDir()
		{
			Console.WriteLine ("---------------- TestCreateDir ----------------");			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				ftp.ChangeDirectory(Site.UploadDir);

				//empty
				Console.Write("Empty parameter ...");
				try
				{
					ftp.CreateDirectory("");
					Assertion.Fail("Method CreateDirectory - empty dir name used.");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				//null
				Console.Write("Null parameter ...");
				try
				{
					ftp.RemoveDirectory(null);
					Assertion.Fail("Removed null dir??");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				
				//existing
				Console.Write("Existing dir ...");
				ftp.CreateDirectory(uniqName);
				try
				{
					ftp.CreateDirectory(uniqName);
					Assertion.Fail("Existing dir created !!!");
				}
				catch (FtpException e)
				{
					if ((e.Status == FtpExceptionStatus.ProtocolError) || (e.Response.Code == 550))	Console.WriteLine("...ok");
					else Assertion.Fail("Unexpected error status");
				}				

				//strange string
				Console.Write("Strange string ...");
				try
				{
					ftp.CreateDirectory("@[]#{}//$^&*__+|:?></.,;=%_#$<!Npihb");
					Assertion.Fail("DIR with very strange chars created");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				//nonexisting path
				//tests really component?
				Console.Write("Nonexisting path ...");
				ftp.ChangeDirectory(uniqName);
				string uniqName2 = Utils.GetUniqueName(ftp,"","");
				try
				{
					ftp.CreateDirectory(uniqName + "/" + uniqName2 + "/bleble");
					Assertion.Fail("Path changed to nonexisting dir.");
				}
				catch (FtpException e)
				{
					if ((e.Status == FtpExceptionStatus.ProtocolError) || (e.Response.Code == 550))	Console.WriteLine("...ok");
					else Assertion.Fail("Unexpected error status");
				}
				ftp.ChangeDirectory(Site.UploadDir);
				ftp.RemoveDirectory(uniqName);

			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}
		 
		
		/// <summary>
		/// TestRemoveDir - tests RemoveDir method 
		/// Dirs are removed commonly in other tests (relatively an absolutely)
		/// This test tests other situations
		///  - incorrect params - empty, null, nonexisting, strange data
		/// </summary>
		
		[Test]
		public void TestRemoveDir()
		{
			Console.WriteLine ("---------------- TestRemoveDir ----------------");			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				ftp.ChangeDirectory(Site.UploadDir);
				//null
				Console.Write("Null parameter ...");
				try
				{
					ftp.RemoveDirectory(null);
					Assertion.Fail("NULL diretory removed.");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				//non exist dir
				//ftpexception
				Console.Write("Non exist dir ...");
				try
				{
					ftp.RemoveDirectory(uniqName);
				}
				catch (FtpException e)
				{
					if ((e.Status == FtpExceptionStatus.ProtocolError) || (e.Response.Code == 550))	Console.WriteLine("...ok");
					else Assertion.Fail("Unexpected error status");
				}				
				
				//empty string
				Console.Write("Empty string ...");
				try
				{
					ftp.RemoveDirectory("");
					Assertion.Fail("Empty named dir removed??");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				//strange string
				Console.Write("Strange string ...");
				try
				{
					ftp.RemoveDirectory("@#$^&*__+|:?></.,;=%_#$<!Npihb");
					Assertion.Fail("Dir with strange chars removed??");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}


				//dir used by me cannot remove
/* 
 * TEST IS NOT VALID, UNIX ALLOWS THIS STUFF

				Console.Write("Cannot remove dir used by me ...");
				ftp.ChangeDirectory(uniqName);
				try
				{
					ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName);
				}
				catch (FtpException e)
				{
					if ((e.Status == FtpExceptionStatus.ProtocolError) || (e.Response.Code == 550))	Console.WriteLine("...ok");
					else Assertion.Fail("Unexpected error status");
				}
				ftp.ChangeDirectory(Site.UploadDir);
*/
				// normal remove :-D
				ftp.CreateDirectory(uniqName);
				Console.Write("Normal remove ...");
				ftp.RemoveDirectory(uniqName);
				Console.Write("...ok");
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}
					
		
		/// <summary>
		/// TestAppendFile - tests AppendFile method
		/// this test tests path, offsets and send bytes to memory
		///  basic test on file -->file is written  
		/// </summary>
		[Test]
		public void TestAppendFile()
		{
			Console.WriteLine ("---------------- TestAppendFile ----------------");			
			Ftp ftp = null;
			try
			{
				//PREPARE
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");
				string text = "12345678101234567x201234567x301234567x401234567x50"
					+  "1234567x601234567x701234567x801234567x90123456x100";							
				byte[] data = System.Text.Encoding.ASCII.GetBytes(text);
				System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);

				ftp.ChangeDirectory(Site.UploadDir);

				//TEST HERE
				//DEFAULT TEST FILE --> MEMORY STREAM
				//offset is 0
				//param length = -1

				Console.WriteLine("DEFAULT TEST FILE --> MEMORY STREAM");

				//no remote file				
				Console.Write("No remote file ...");
				long nUploadedLen = ftp.AppendFile(oInput, uniqName,-1);
				long fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 100 but was " + fileLen, 100, fileLen);				
				Console.WriteLine("... ok");

				//some file exists
				oInput.Position = 0;
				Console.Write("Absolute path ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,-1);
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 200 but was " + fileLen, 200, fileLen);				
				Console.WriteLine("... ok");
				
				//absolute path
				oInput.Position = 0;
				Console.Write("Absolute path ...");
				nUploadedLen = ftp.AppendFile(oInput, Site.UploadDir + "/" + uniqName,-1);
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 300 but was " + fileLen, 300, fileLen);
				Console.WriteLine("... ok");

				//absolute path from another dir
				string uniqName2 = Utils.GetUniqueName(ftp, "", "");
				ftp.CreateDirectory(uniqName2);
				ftp.ChangeDirectory(uniqName2);
				oInput.Position = 0;
				nUploadedLen = ftp.AppendFile(oInput,Site.UploadDir + "/" + uniqName,-1);
				fileLen = ftp.GetFileLength(Site.UploadDir + "/" + uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 400 but was " + fileLen, 400, fileLen);							
				ftp.ChangeDirectory("..");
				ftp.RemoveDirectory(uniqName2);
				Console.WriteLine("... ok");
				
				//remotepath is null?? (Argument)
				oInput.Position = 0;
				Console.Write("Remote path is null ...");
				try
				{
					nUploadedLen = ftp.AppendFile(oInput, null,-1);
					Assertion.Fail("PutFile method incorrect input used (RemotePath null)");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}
		
				//STREAM OFFSET TESTS FILE --> MEMORY STREAM
				Console.WriteLine("STREAM OFFSET TESTS FILE --> MEMORY STREAM");

				ftp.DeleteFile(uniqName);
				uniqName = Utils.GetUniqueName(ftp,"","");
				//offset > 0
				oInput.Position = 0;
				Console.Write("Offset > 0 ...");
				oInput.Seek (50, SeekOrigin.Current);
				nUploadedLen = ftp.AppendFile(oInput, uniqName,-1);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 50 but was " + fileLen, 50, fileLen);
				Console.WriteLine("... ok");

				//offset < 0
				oInput.Position = 100;
				Console.Write("Offset < 0 ...");
				oInput.Seek (-50, SeekOrigin.Current);
				nUploadedLen = ftp.AppendFile(oInput, uniqName,-1);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 100 but was " + fileLen, 100, fileLen);
				Console.WriteLine("... ok");

				//PARAM LENGTH TESTS FILE --> MEMORY STREAM
				Console.WriteLine("PARAM LENGTH TESTS FILE --> MEMORY STREAM");
				//param len > Stream len
				//param len > 0
				oInput.Position = 0;
				Console.Write("Param len > Stream len ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,200);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 200 but was " + fileLen, 200, fileLen);
				Console.WriteLine("... ok");

				//param len = Stream len
				oInput.Position = 0;
				Console.Write("Param len = Stream len ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,100);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 300 but was " + fileLen, 300, fileLen);
				Console.WriteLine("... ok");

				//param len < Stream len
				oInput.Position = 0;
				Console.Write("Param len < Stream len ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,50);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 350 but was " + fileLen, 350, fileLen);
				Console.WriteLine("... ok");

				//param len > Stream len from position to end
				oInput.Position = 50;
				Console.Write("param len > Stream len from position to end ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,100);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 400 but was " + fileLen, 400, fileLen);
				Console.WriteLine("... ok");

				//param len is big
				oInput.Position = 0;
				Console.Write("Param len is big ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,999999999);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 500 but was " + fileLen, 500, fileLen);
				Console.WriteLine("... ok");

				//param len is very big
				oInput.Position = 0;
				Console.Write("Param len is very big ...");
				nUploadedLen = ftp.AppendFile(oInput, uniqName,999999999999999999);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 600 but was " + fileLen, 600, fileLen);
				Console.WriteLine("... ok");

				//param len < 0
				oInput.Position = 0;
				Console.Write("Param len < 0 ...");
				try
				{
					nUploadedLen = ftp.AppendFile(oInput, uniqName,-5);
					Assertion.Fail("PutFile method incorrect input used (Param len = -5).");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				ftp.DeleteFile(uniqName);
				
				//BASIC TEST FILE --> FILE

				//PREPARE
				string tempFilePath = Path.GetTempFileName();
				uniqName = Utils.GetUniqueName(ftp, "", "");

				//local file not exists
				Console.Write("Local file not exists ...");
				File.Delete(tempFilePath);
				Console.Write("Local file not exists ...");
				try
				{
					nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,-1);
					Assertion.Fail("Appended non existing file!!");
				}
				catch (Exception e)
				{
					if (e.Message!="Appended non existing file!!") Console.WriteLine(e.Message + "... ok");					
					else throw e;
				}

				//PREPARE MORE
				tempFilePath = Path.GetTempFileName();
				System.IO.StreamWriter streemWriter = File.AppendText(tempFilePath);
				streemWriter.Write(text);
				streemWriter.Close();

				//remote file not exists
				Console.Write("Remote file not exists ...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,-1);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 100 but was " + fileLen, 100, fileLen);
				Console.WriteLine("... ok");

				//local path is null
				Console.Write("Local path is null ...");
				string pathToNowhere = null;
				try
				{
					nUploadedLen = ftp.AppendFile(pathToNowhere, uniqName,0,-1);
					Assertion.Fail("Appended file with null path");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				//local file offset<0
				Console.Write("Local file offset<0 ...");
				try
				{
					nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,-1,-1);				
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				//local file offset>0
				Console.Write("Local file offset>0...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,50,-1);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 150 but was " + fileLen, 150, fileLen);
				Console.WriteLine("... ok");

				//local file offset=0
				Console.Write("Local file offset=0...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,-1);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 250 but was " + fileLen, 250, fileLen);
				Console.WriteLine("... ok");

				//param len > file len
				Console.Write("Param len > file len...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,200);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 350 but was " + fileLen, 350, fileLen);
				Console.WriteLine("... ok");

				//param len = file len
				Console.Write("Param len = file len...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,100);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 450 but was " + fileLen, 450, fileLen);
				Console.WriteLine("... ok");

				//param len < file len
				Console.Write("Param len < file len...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,50);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 50 but was " + nUploadedLen, 50, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 500 but was " + fileLen, 500, fileLen);
				Console.WriteLine("... ok");

				//param len < 0 and not -1
				Console.Write("Param len < 0 and not -1 ...");
				try
				{
					nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,-100);				
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				//param len is big
				Console.Write("Param len is big ...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,99999999);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 600 but was " + fileLen, 600, fileLen);
				Console.WriteLine("... ok");

				//param len is very big
				Console.Write("Param len is very big ...");
				nUploadedLen = ftp.AppendFile(tempFilePath, uniqName,0,999999999999999999);				
				fileLen = ftp.GetFileLength(uniqName);
				Assertion.AssertEquals("Uploaded length differ. Expected 100 but was " + nUploadedLen, 100, nUploadedLen);
				Assertion.AssertEquals("Remote file size differ. Expected 700 but was " + fileLen, 700, fileLen);
				Console.WriteLine("... ok");				
				
				// CLEAN IT
				ftp.DeleteFile(uniqName);

			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}


		 
		 
		/// <summary>
		/// TestGetFile - tests method GetFile
		///  - usual tests are made in almost all other tests using utils
		/// (strange path, etc, stream/string, ...)
		/// + in TestBinary, TestAsciiUploadandDownload
		/// this test tests offsets and send bytes
		/// </summary>
		
		[Test]
		public void TestGetFile()
		{
			Console.WriteLine ("---------------- TestGetFile ----------------");			
				
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				ftp.ChangeDirectory(Site.UploadDir);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				string text = "12345678101234567x201234567x301234567x401234567x50"
					+  "1234567x601234567x701234567x801234567x90123456x100";
				Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
				// upload file
				// get file
				
				//RELATIVE LOCAL OFFSET from actual(end = offset 10)
				Console.WriteLine ("RELATIVE LOCAL OFFSET from actual");
				Console.Write ("Normal ... ");
				System.IO.MemoryStream oOutput = new System.IO.MemoryStream();
				//normal
				oOutput.Position = 0;
				ftp.GetFile(uniqName, oOutput,0);
				Assertion.AssertEquals("Getfile returned less.", 100,oOutput.Length);
				Console.WriteLine ("...ok");

				//100 - 50 = 50
				//actual position is 99(/100)
				Console.Write ("100-50 ...");
				oOutput.Seek (-50, SeekOrigin.Current);
				ftp.GetFile(uniqName, oOutput,50);
				byte[] data = oOutput.ToArray();
				string textOut = System.Text.Encoding.ASCII.GetString(data);
				Assertion.AssertEquals("Text in stream is not equal.", text, textOut);
				Console.WriteLine ("...ok");

				// 100+10=110
				Console.Write ("100+10 ...");
				ftp.GetFile(uniqName, oOutput,90);
				Assertion.AssertEquals("Length differ. Expected 110 but was " + oOutput.Length, 110, oOutput.Length);
				Console.WriteLine ("...ok");
							
				//set length
				oOutput.SetLength(0);

				//midmid
				Console.Write ("Midmid ...");
				oOutput.Seek (0, SeekOrigin.Current);
				ftp.GetFile(uniqName, oOutput,50);
				data = oOutput.ToArray();
				textOut = System.Text.Encoding.ASCII.GetString(data);
				Assertion.AssertEquals("Text in stream is not equal. Expected " + text.Substring(50,50) + " but obtained " + textOut, text.Substring(50,50), textOut);
				oOutput.SetLength(0);											
				Console.WriteLine ("...ok");
				Console.Write ("Less than 0 ...");
				// less than 0
				try
				{
					ftp.GetFile(uniqName, oOutput,-100);
				}
				catch (Exception e)
				{
					e=e;Console.WriteLine (e.GetType() + "...ok");
				}

				// ABSOLUTE REMOTE OFFSET
				//let's use this  uploaded file	for next tests
				//normal already tested "upstairs" :-D
				Console.WriteLine ("ABSOLUTE REMOTE OFFSET from actual");
				Console.Write ("101 - more than file max offset [100] ... ");
				//101 - more than file max offset [100]
				oOutput.SetLength(0);
				try
				{
					oOutput.Seek (101, SeekOrigin.Current);
					ftp.GetFile(uniqName, oOutput,0);
				}
				catch (FtpException e)
				{
					if ((e.Response.Group != 4) || (e.Status != FtpExceptionStatus.ProtocolError)) throw e;
					Console.WriteLine ("...ok");
				}


				Console.Write ("-1 ... ");
				oOutput.SetLength(0);
				//-1
				try
				{
					ftp.GetFile(uniqName, oOutput,-1);
					Assertion.Fail("Absolute negative offset passed");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}


				Console.Write ("Big number ... ");
				//big number 100000
				
				data = new byte[100];

				// upload from offset 100000-100 before
				System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);				
				string uniqName2 = Utils.GetUniqueName(ftp, "", "");
				oInput.Position = 0;

				ftp.PutFile(oInput,uniqName2,99900,-1);

				oOutput.SetLength(0);
				ftp.GetFile(uniqName2,oOutput,0);
				Assertion.AssertEquals("Length differ..", 100000, oOutput.Length);
				Console.WriteLine ("...ok");				
				
				//oOutput.Close();	
				ftp.DeleteFile(uniqName);
				ftp.DeleteFile(uniqName2);

				//TODO: GETFILE local file offset
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		/// <summary>
		/// TestPutFile - tests PutFile method 
		/// - usual tests are made in almost all other tests using utils
		/// (strange path, etc, stream/string, ...)  
		///  + in TestBinary, TestAsciiUploadandDownload
		///  this test tests offsets and send bytes params
		/// </summary>

		[Test]
		public void TestPutFile()
		{
			Console.WriteLine ("---------------- TestPutFile ----------------");

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				ftp.ChangeDirectory(Site.UploadDir);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				string text = "1234567890";
				byte[] data = System.Text.Encoding.ASCII.GetBytes(text);
				System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);

				//default transfer is binary					
				//RELATIVE LOCAL OFFSET from actual(end = offset 10)

/*
 * TESTS NOT USED, RELATIVE LOCAL OFFSET IS NOT SUPPORTED
 * 
  				Console.WriteLine ("RELATIVE LOCAL OFFSET from actual");
				Console.Write ("Normal ... ");
				//normal
				long nUploadedLen = ftp.PutFile(oInput,uniqName,0,-1); 
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);					
				//clean, clear, ...
				ftp.DeleteFile(uniqName);
				Console.WriteLine ("...ok");
				//10 - 5 = 5			
				Console.Write ("10 - 5 = 5 ... ");
				uniqName = Utils.GetUniqueName(ftp, "", "");
				oInput.Position=10;
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,-1); 				
				Assertion.AssertEquals("Expected length is 5, but uploaded " + nUploadedLen +" bytes.", 5, nUploadedLen);
				ftp.DeleteFile(uniqName);
				Console.WriteLine ("...ok");				
				// 1+15=16
				Console.Write ("1+15=16 ... ");
				oInput.Position=1;
				uniqName = Utils.GetUniqueName(ftp, "", "");				
				nUploadedLen = 0;
				//16>10=last position :-D
				Console.Write ("... 16>10=last ... ");
				nUploadedLen = ftp.PutFile(oInput,uniqName,15,-1);
				Assertion.Assert("Impossible ...", nUploadedLen==0);
				Console.WriteLine ("...ok");

				// less than char. count (-11)
				Console.Write ("-11 ... ");
				uniqName = Utils.GetUniqueName(ftp, "", "");				
				oInput.Position=10;
				try
				{
					nUploadedLen = ftp.PutFile(oInput,uniqName,-11,-1);
				}
				catch (Exception e)
				{
					e=e;Console.WriteLine ("...ok");
				}
*/

				// ABSOLUTE REMOTE OFFSET
				//let's use this  uploaded file	for next tests
				//normal
				Console.Write ("Normal ... ");
				oInput.Position=0;
				long nUploadedLen = ftp.PutFile(oInput,uniqName,0,-1);
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine ("...ok");
/*
 * TEST REMOVED, UNIX BEHAVIOR DIFFER FROM WIN
 * MS FTP: inserts 00
 * wuFTPd: fails 
 * 
				//15 - more than file max offset [10]
				Console.Write ("More than max offset ... ");
				oInput.Position=0;
				nUploadedLen = ftp.PutFile(oInput,uniqName,15,-1);												
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine ("...ok");
*/
				//-1
 				Console.Write ("-1 ... ");
				try
				{					
					nUploadedLen = ftp.PutFile(oInput,uniqName,-1,-1);
					Assertion.Fail("PutFile method incorrect input used.");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}
/*
 * TEST REMOVED, UNIX BEHAVIOR DIFFER FROM WIN
 * MS FTP: inserts 00
 * wuFTPd: fails 
 * 
				//100000 - big number
				Console.Write ("Big number ... ");
				oInput.Position=0;				
				nUploadedLen = ftp.PutFile(oInput,uniqName,100000,-1);
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine ("...ok");
*/
				//ABSOLUTE: too big number
				Console.Write ("Too big number ... ");
				try
				{
					oInput.Position=0;
					oInput.Seek (999999999999999999, SeekOrigin.Current);
					nUploadedLen = ftp.PutFile(oInput,uniqName,0,-1);
					Assertion.Fail("PutFile method incorrect input used.");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				//DELETE IT FINALLY
				ftp.DeleteFile(uniqName);
				
				//PARAM LENGTH
				Console.WriteLine ("PARAM LENGTH");
				//normal -1, 10, 5
				Console.Write ("Normal ... ");
				uniqName = Utils.GetUniqueName(ftp, "", "");
				oInput.Position=0;
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,-1); 				
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine ("...ok");
				oInput.Position=0;
				//exactly same as stream size
				Console.Write ("Same as stream size ... ");
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,10); 				
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine ("...ok");
				// more than stream leftover
				Console.Write ("More than stream leftover ... ");
				oInput.Position=7;
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,4); 
				Assertion.AssertEquals("Expected length is 3, but uploaded " + nUploadedLen +" bytes.", 3, nUploadedLen);
				Console.WriteLine ("...ok");
				// less than stream leftover
				Console.Write("Less than stream leftover ... ");
				oInput.Position=0;
				oInput.Seek (3, SeekOrigin.Current);
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,5); 
				Assertion.AssertEquals("Expected length is 5, but uploaded " + nUploadedLen +" bytes.", 5, nUploadedLen);
				Console.WriteLine ("...ok");
				//delete IT FINALLY
				ftp.DeleteFile(uniqName);

				//abnormal, limits>> -10, 0, 9999999999999999
				Console.Write("-10 ... ");
				uniqName = Utils.GetUniqueName(ftp, "", "");
				oInput.Position=0;
				try
				{
					nUploadedLen = ftp.PutFile(oInput,uniqName,0,-10);
					Assertion.Fail("PutFile method incorrect input used (-10).");
				}
				catch (ArgumentException e)
				{
					e=e;
					Console.WriteLine ("...ok");
				}
				Console.Write("0 ... ");
				oInput.Position=0;
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,0); 				
				Assertion.AssertEquals("Expected length is 0, but uploaded " + nUploadedLen +" bytes.", 0, nUploadedLen);
				Console.WriteLine ("...ok");

				Console.Write("999999999999999 ... ");
				oInput.Position=0;
				nUploadedLen = ftp.PutFile(oInput,uniqName,0,999999999999999);
				Assertion.AssertEquals("Expected length is 10, but uploaded " + nUploadedLen +" bytes.", 10, nUploadedLen);
				Console.WriteLine("...ok");
				//delete IT FINALLY	
				ftp.DeleteFile(uniqName);
					
				//TODO: PUTFILE local file offset		

			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		
		/// <summary>
		/// TestDeleteFile - tests DeleteFile method
		///  - existing/non existing
		///  - in another dir
		///  - absolute/relative
		/// </summary>

		[Test]
		public void TestDeleteFile()
		{
			Console.WriteLine ("---------------- TestDeleteFile ----------------");			
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "del", "");			

				//prepare
				ftp.CreateDirectory(Site.UploadDir +"/" + uniqName);
				Utils.UploadFile(ftp,Site.UploadDir +"/" + uniqName + "/" + uniqName, "brrrgrrrr", Rebex.Net.FtpTransferType.Binary);
				//existing absolutely
				ftp.DeleteFile(Site.UploadDir +"/" + uniqName + "/" + uniqName);
				//existing - relatively
				Utils.UploadFile(ftp,Site.UploadDir +"/" + uniqName + "/" + uniqName, "brrrgrrrr", Rebex.Net.FtpTransferType.Binary);
				ftp.ChangeDirectory(Site.UploadDir +"/" + uniqName);
				ftp.DeleteFile(uniqName);
				//in another dir
				Utils.UploadFile(ftp,Site.UploadDir +"/" + uniqName + "/" + uniqName, "brrrgrrrr", Rebex.Net.FtpTransferType.Binary);
				ftp.ChangeDirectory(Site.UploadDir);
				ftp.DeleteFile(Site.UploadDir +"/" + uniqName + "/" + uniqName);
				//nonexisting 1 (trim)
				ftp.ChangeDirectory(Site.UploadDir +"/" + uniqName);				
				Utils.UploadFile(ftp,Site.UploadDir +"/" + uniqName + "/" + uniqName, "brrrgrrrr", Rebex.Net.FtpTransferType.Binary);
				try
				{
					ftp.DeleteFile(Site.UploadDir +"/" + uniqName + "/cecece" + uniqName);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}
				// clean
				ftp.DeleteFile(Site.UploadDir +"/" + uniqName + "/" + uniqName);
				
				ftp.ChangeDirectory(Site.UploadDir);		
				ftp.RemoveDirectory(Site.UploadDir +"/" + uniqName);				
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}
		

		/// <summary>
		/// Tests GetFileLength method 
		///  in binary mode
		/// </summary>
		[Test]
		public void TestGetFileLength()
		{
			Console.WriteLine ("---------------- TestGetFileLength ----------------");			
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				ftp.ChangeDirectory(Site.UploadDir);
				// prepare
				string uniqName = Utils.GetUniqueName(ftp, "", ".bin");			
				long fileLength;
			
				//nonexisting path
				try
				{
					fileLength = ftp.GetFileLength(Site.UploadDir + "/" + uniqName);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}
				//existing directory path
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName);
				try
				{
					fileLength = ftp.GetFileLength(Site.UploadDir + "/" + uniqName);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName);
			
				//some texts
				uniqName = Utils.GetUniqueName(ftp, "", ".bin");			
				string txtToTransfer = "\n";			
				long nUploadedLen = Utils.UploadFile(ftp, Site.UploadDir + "/" +uniqName, txtToTransfer, FtpTransferType.Binary);
				fileLength = ftp.GetFileLength(Site.UploadDir + "/" +uniqName);						
				Console.WriteLine( ">> " + nUploadedLen+"||"+fileLength + " <<");
				Assertion.Assert(nUploadedLen+"||"+fileLength, nUploadedLen==fileLength);
				ftp.DeleteFile(Site.UploadDir + "/" +uniqName);

				//another one
				uniqName = Utils.GetUniqueName(ftp, "", ".bin");			
				txtToTransfer = "1\n2\n\r3\r\n4\r5";
				nUploadedLen = Utils.UploadFile(ftp, Site.UploadDir + "/" +uniqName, txtToTransfer, FtpTransferType.Binary);
				fileLength = ftp.GetFileLength(Site.UploadDir + "/" +uniqName);						
				Console.WriteLine( ">> " + nUploadedLen+"||"+fileLength + " <<");
				Assertion.Assert(nUploadedLen+"||"+fileLength, nUploadedLen==fileLength);
				ftp.DeleteFile(Site.UploadDir + "/" +uniqName);

				//next
				uniqName = Utils.GetUniqueName(ftp, "", ".bin");			
				txtToTransfer = "1\n2\n\r3\r\n4\r550874=428597-987-897-wqu9sd98-23409-=9rm81-=348789===+m";
				nUploadedLen = Utils.UploadFile(ftp, Site.UploadDir + "/" +uniqName, txtToTransfer, FtpTransferType.Binary);
				fileLength = ftp.GetFileLength(Site.UploadDir + "/" +uniqName);						
				Console.WriteLine( ">> " + nUploadedLen+"||"+fileLength + " <<");
				Assertion.Assert(nUploadedLen+"||"+fileLength, nUploadedLen==fileLength);
				ftp.DeleteFile(Site.UploadDir + "/" +uniqName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}
		/// <summary>
		/// Ftp server should return Greenwich datetime, but 
		/// usualy returns distant-local. Test fails when received
		/// datetime differ more then 26 hours from local .NET time 
		/// </summary>

		[Test]
		public void TestGetFileDatetime()
		{
			Console.WriteLine ("---------------- TestGetFileDatetime ----------------");			
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				// prepare
				string uniqName = Utils.GetUniqueName(ftp, "", "");
				//upload file
				Console.WriteLine("File upload ...");
				System.DateTime dtBegin = DateTime.Now;
				Utils.UploadFile(ftp, Site.UploadDir + "/" + uniqName, "lajkabajkabalalajka", FtpTransferType.Binary);
				//get actual local datetime
				System.DateTime dtEnd = DateTime.Now;
				Console.WriteLine(" ok");						
				System.DateTime dtTest = ftp.GetFileDateTime(Site.UploadDir + "/" + uniqName);	
				
				//difference
				double hourDiff = Math.Abs((dtEnd - dtTest).TotalHours);
				Console.WriteLine("Datetimes " + dtEnd + " and " + dtTest + " differ (" + hourDiff + " hours)");
				Assertion.Assert("DateTimes" + dtEnd +" a " + dtTest + " differs more than 26" + hourDiff, hourDiff<=26);
				ftp.DeleteFile(Site.UploadDir + "/" + uniqName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}
		
		
		
		/// <summary>
		/// TestConnectGoodPort - test simply Connect and Login method
		/// with correct parameters
		/// </summary>
		
		[Test]
		public void TestConnectGoodPort()
		{
			Console.WriteLine ("---------------- TestConnectGoodPort ----------------");
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.CreateFtp (Site);
				Assertion.AssertEquals ("Invalid state.", FtpState.Disconnected, ftp.State);
			
				// should pass
				Console.WriteLine ("Connect...");
				ftp.Connect(Site.Host,Site.Port);
				Console.WriteLine ("ok");
				Assertion.Assert("False state:" + ftp.State, ftp.State==FtpState.Ready);

				Console.WriteLine ("Login...");
				ftp.Login (Site.Login,Site.Password);
				Console.WriteLine ("ok");
				Assertion.Assert("False state:" + ftp.State, ftp.State==FtpState.Ready);
				
				Console.WriteLine ("Disconnect...");
				ftp.Disconnect ();
				Console.WriteLine ("ok");
				Assertion.Assert("False state:" + ftp.State, ftp.State==FtpState.Disconnected);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}
								
		/// <summary>
		/// TestChangeDirectory - test ChangeDirectory method
		/// Absolute and relative directory change
		/// </summary>
		[Test]
		public void TestChangeDirectory()
		{
			Console.WriteLine ("---------------- TestChangeDirectory ----------------");
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				//preparation
				//folder
				string uniqName = Utils.GetUniqueName(ftp, "chdir", "");
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName);		
			
				// first absolutely
				ftp.ChangeDirectory (Site.UploadDir + "/" + uniqName);
			
				Assertion.Assert("Failed to change absolutely dir. Expected " + Site.UploadDir + "/" + uniqName + ", actual is  " + ftp.GetCurrentDirectory(), 
					Site.UploadDir + "/" + uniqName == ftp.GetCurrentDirectory());

				// relatively, now is actual new with unique name
				//let's change it
				
				ftp.ChangeDirectory ("..");							
				Assertion.Assert("Expected: " + Site.UploadDir + ", actual is  " + ftp.GetCurrentDirectory(), 
					Site.UploadDir == ftp.GetCurrentDirectory());

				ftp.ChangeDirectory (uniqName);
				Assertion.Assert("Expected: " + Site.UploadDir + "/" + uniqName + ", actual is  " + ftp.GetCurrentDirectory(), 
					Site.UploadDir + "/" + uniqName == ftp.GetCurrentDirectory()); 
				ftp.ChangeDirectory ("..");
				Console.WriteLine("Remove dir ...");
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName);
				Console.WriteLine("ok.");
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		
		}		

		/// <summary>
		/// TestFileRename - test FileRename method		 
		/// - from another dir
		/// - inside same dir
		/// - non exist file
		/// - into another dir
		/// </summary>

		[Test]
		public void TestFileRename()
		{
			Console.WriteLine ("---------------- TestFileRename ----------------");			

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				//prepare
				string uniqName = Utils.GetUniqueName(ftp, "", "");
				string uniqName2 = null;
				// test file upload
				Console.WriteLine("Test file upload ...");
				string strInput = "Hi boys ...";				
				long nUploadedLen = Utils.UploadFile(ftp,Site.UploadDir + "/" + uniqName, strInput, FtpTransferType.Binary);
				Console.WriteLine ("ok.");
			
				// Rename absolutely  in the same directory
				//ftp.ChangeDirectory
				Console.WriteLine ("Rename absolutely  in the same directory ..");
				ftp.Rename(Site.UploadDir + "/" + uniqName, Site.UploadDir + "/" + "renamed" + uniqName);
				Console.WriteLine("ok.");

				Assertion.AssertEquals("Rename failed, renamed file has another size ..", Utils.GetFileLength(ftp,Site.UploadDir + "/" + "renamed" + uniqName), nUploadedLen);

				ftp.DeleteFile(Site.UploadDir + "/" + "renamed" + uniqName);
			
				// Rename relatively in the same directory 
				// Prepare
				uniqName = Utils.GetUniqueName(ftp, "", "");
				nUploadedLen = Utils.UploadFile(ftp, Site.UploadDir +  "/" + uniqName, strInput, FtpTransferType.Binary);
				Console.WriteLine ("Rename relatively in the same directory ...");
				ftp.ChangeDirectory(Site.UploadDir);
				Console.WriteLine ("We are at .. " + ftp.GetCurrentDirectory());
				ftp.Rename(uniqName, "renamed" + uniqName);										
				Console.WriteLine("ok.");
				Assertion.AssertEquals("Relative Rename unsuccesfull ..", Utils.GetFileLength(ftp,Site.UploadDir + "/renamed" + uniqName), nUploadedLen);

				ftp.DeleteFile(Site.UploadDir + "/renamed" + uniqName);

				// Rename absolutely in another directory(ies)
				// Prepare			
				//folder				
				uniqName = Utils.GetUniqueName(ftp, "dir", "");
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName);
				uniqName2 = Utils.GetUniqueName(ftp, "", ".txt");
				nUploadedLen = Utils.UploadFile(ftp,Site.UploadDir + "/" + uniqName2, strInput, FtpTransferType.Binary);
			
				Console.WriteLine ("Rename absolutely in another directory(ies)...");
				ftp.ChangeDirectory(Site.UploadDir + "/" + uniqName);
				Console.WriteLine("Current directory is " + ftp.GetCurrentDirectory());
				ftp.Rename(Site.UploadDir + "/" + uniqName2, Site.UploadDir + "/renamed" + uniqName2);					
				Console.WriteLine("ok.");							
				Assertion.AssertEquals ("Rename failed, renamed file has another size ..", Utils.GetFileLength(ftp, Site.UploadDir + "/renamed" + uniqName2), nUploadedLen);

				ftp.DeleteFile(Site.UploadDir + "/renamed" + uniqName2);
				ftp.ChangeDirectory(Site.UploadDir);
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}


		}
				
		/// <summary>
		/// TestFileRenameNoSuchFile - tests FileRename method
		/// with non existing file
		/// </summary>


		[Test]
		public void TestFileRenameNoSuchFile()
		{
			Console.WriteLine ("---------------- TestFileRenameNoSuchFile ----------------");			

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				/*Rename of non exists file, expected FtpException code 550*/
				Console.WriteLine ("Rename of non exists file, expected FtpException code 550 ...");
				string uniqName;
				string uniqName2;

				try
				{				
					uniqName = Utils.GetUniqueName(ftp, "non", ".txt");
					uniqName2 =  Utils.GetUniqueName(ftp, "non", ".txt");
					ftp.Rename(Site.UploadDir + "/" +uniqName, Site.UploadDir + uniqName2);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code!=550)
						throw e;
				}
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}


		[Test]
		public void TestStrangeNames()
		{
			Console.WriteLine ("---------------- TestStrangeNames ----------------");			

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");
				string path = uniqName + "/te st.txt";

				ftp.ChangeDirectory (Site.UploadDir);
				ftp.CreateDirectory (uniqName);
				Utils.UploadFile (ftp, path, "txt", FtpTransferType.Binary);
				Utils.GetFileLength (ftp, path);

				ftp.DeleteFile (path);
				ftp.RemoveDirectory (uniqName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		
		/// <summary>
		/// FileRenameNoDirectory - tests FileRename
		/// - rename to nonexistdirectory
		/// expected [ftpexception] 550
		/// </summary>

		[Test]
		public void TestFileRenameNoDirectory()
		{
			Console.WriteLine ("---------------- TestFileRenameNoDirectory ----------------");			

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				// prepare
				string uniqName = Utils.GetUniqueName(ftp, "", "");

				// test file upload
				Console.WriteLine("Test file upload ...");
				string strInput = "Hi boys ...";
				long nUploadedLen = Utils.UploadFile(ftp,Site.UploadDir + "/" + uniqName, strInput, FtpTransferType.Binary);
				Console.WriteLine ("ok.");
			
				//test itself
				Console.WriteLine ("Rename to nonexistdirectory ...");				
				
				try
				{

					ftp.Rename(Site.UploadDir + "/" + uniqName, Site.UploadDir + "/" + uniqName + "/" + uniqName);
					Assertion.Fail("File renamed to nonexisting dir.");
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code==550)
						ftp.DeleteFile(Site.UploadDir + "/" + uniqName); 
					else
						throw e;
				}
			}
			finally
			{								
				Utils.CloseFtp (ftp);
			}
		}

		// Rename from/to directory where user doesn't have right to modify

		[Test]		
		public void TestFileRenameNoRightsToDirectory()
		{

			Console.WriteLine ("---------------- TestFileRenameNoRights ----------------");			

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				// prepare
				string uniqName = Utils.GetUniqueName(ftp, "", "");
				string uniqName2 = Utils.GetUniqueName(ftp, "", "");

				// test file upload
				Console.WriteLine("Test file upload ...");
				string strInput = "Hi boys ...";
				
				long nUploadedLen = Utils.UploadFile(ftp,Site.UploadDir + "/" + uniqName, strInput, FtpTransferType.Binary);
				Console.WriteLine ("ok.");

				// prepare directory
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName2);
			
				//test itself						
				//File rename no rights to dir
				Console.WriteLine (" File rename ...");

				try
				{
					ftp.Rename(Site.UploadDir + "/" + uniqName, Site.NoModifyDir + "/" + uniqName);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}

				//Directory rename no rights to dir
				try
				{
					ftp.Rename(Site.UploadDir + "/" + uniqName2, Site.NoModifyDir+ "/" + uniqName2);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}

				//Directory rename no rights from dir
				try
				{
					ftp.Rename(Site.NoModifyDir+ "/" + uniqName, Site.UploadDir + "/" + uniqName2);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}

				//File rename no rights from dir
				try
				{
					ftp.Rename(Site.NoModifyDir + "/" + uniqName + ".txt",Site.UploadDir + "/" + uniqName);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
				}			
						
				ftp.DeleteFile(Site.UploadDir + "/" + uniqName); 
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName2);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}			

		/// <summary>
		/// TestGetNameList - teste GetNameList, GetRawList methods 
		/// - empty dir		
		/// - 1 file a 1 dir
		/// </summary>


		[Test]
		public void TestGetNameRawList()
		{
			Console.WriteLine ("---------------- TestGetNameRawList ----------------");			
			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				//prepare
				string uniqName = Utils.GetUniqueName(ftp, "", "");

				// prepare directory
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName);	
				ftp.CreateDirectory(Site.UploadDir + "/" + uniqName + "/" + uniqName);	

				// test file upload
				Console.WriteLine("Test file upload ...");
				string strInput = "Hi boys ...";
				long nUploadedLen = Utils.UploadFile(ftp,Site.UploadDir + "/" + uniqName + "/" + uniqName + ".txt", strInput, FtpTransferType.Binary);
				Console.WriteLine ("ok.");				
				
				// let's change dir
				ftp.ChangeDirectory (Site.UploadDir + "/" + uniqName);
				
				// file & addr
				//WAS: BUG752
				Console.WriteLine ("Expect 1 or 2 items or 550 error  in " + Site.UploadDir + ".");
				string[] arrName;
				try
				{
					arrName = ftp.GetNameList();
					Assertion.Assert (Site.UploadDir + "/" + uniqName + " includes ..." + arrName.Length + "... items.", ((arrName.Length == 2) || (arrName.Length == 1)));
					Console.WriteLine("ok.");				
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
					Console.WriteLine("ok.");
				}

				// testing just useability
				// no results
				arrName = ftp.GetRawList();
				Assertion.Assert (Site.UploadDir + "/" + uniqName + " includes ..." + arrName.Length + "... minimally anything.", arrName.Length >= 1);
				Console.WriteLine("ok.");
			
				// empty adr
				ftp.ChangeDirectory(Site.UploadDir + "/" + uniqName + "/" + uniqName);
				
				//WAS: BUG747
				try
				{
					arrName = ftp.GetNameList();
					Assertion.Assert ("Unfortunately list includes ..." + arrName.Length + "... items, 0 is expected.", arrName.Length ==0);
					Console.WriteLine("ok.");				
				}			
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code != 550) throw e;
					Console.WriteLine("ok.");
				}				

				ftp.DeleteFile(Site.UploadDir + "/" + uniqName + "/" + uniqName + ".txt");			
				ftp.ChangeDirectory ("/");
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName + "/" + uniqName);
				ftp.RemoveDirectory(Site.UploadDir + "/" + uniqName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}		

		[Test]
		public void TestAsciiUploadAndDownload()
		{
			Console.WriteLine ("---------------- TestAsciiUploadAndDownload ----------------");

			string strInput = "jedna\r\ndva\r\ntri";
			string strOutput = strInput;//.Replace("\n","\r\n");

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
			
				Console.WriteLine ("Upload test ...");

				ftp.ChangeDirectory (Site.UploadDir);
			
				//get unique file name
				string uniqueName = Utils.GetUniqueName(ftp, "ascii", ".txt");
			
				// upload data to server
				long nUploadedLen = Utils.UploadFile (ftp, uniqueName, strInput, FtpTransferType.Ascii);
						
				Console.WriteLine ("ok");
			
				ftp.SetTransferType (FtpTransferType.Ascii);
			
				System.IO.MemoryStream oOutput = new System.IO.MemoryStream();
			
				ftp.GetFile(uniqueName, oOutput);			
				byte[] data = oOutput.ToArray();			
				string strDownload = System.Text.Encoding.ASCII.GetString(data);

				Console.WriteLine ("Upload " + strInput.Length + " bytes: '" + strInput + "'");
				Console.WriteLine ("Download " + strDownload.Length + " bytes: '" + strDownload + "'");
			
				Assertion.AssertEquals ("Uploaded and downloaded files differ.", strDownload, strOutput);

				ftp.DeleteFile (uniqueName);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		/// <summary>
		/// TestBinary - tests binary upload using method PutFile 
		/// - stream -> file
		/// - file -> file
		/// </summary>


		[Test]
		public void TestBinary()
		{
			Console.WriteLine ("---------------- TestBinary ----------------");

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);

				string strInput = "je\ndna\n\rdva\r\nt\rri";
				string strOutput = strInput;//.Replace("\n","\r\n");
				string uniqName = Utils.GetUniqueName(ftp, "binar", ".bin");
			
				ftp.SetTransferType (FtpTransferType.Binary);
				
				Console.WriteLine ("Upload test ...");
				Console.WriteLine ("  1. Memory-File ...");
			
				// prepare
				// create Memory Data
				byte[] data = System.Text.Encoding.ASCII.GetBytes(strInput);
				System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);

				// safe localy				
				string tempFilePath = Path.GetTempFileName();
				FileStream bi = File.Create(tempFilePath);
				bi.Write(data,0,data.Length);
				bi.Close();
												
				// file upload
				long nUploadedLen = ftp.PutFile(oInput,Site.UploadDir + "/" +uniqName);
			
				// test of correct size 1
				Assertion.Assert(
					"Upload failed. Expected " + data.Length + ", transferred " + nUploadedLen + " bytes", 
					nUploadedLen == data.Length);
			
				Console.WriteLine ("ok");

				// correct size 2
				ftp.SetTransferType (FtpTransferType.Binary);
				System.IO.MemoryStream oOutput = new System.IO.MemoryStream();
				ftp.GetFile(Site.UploadDir + "/" + uniqName, oOutput);
				data = oOutput.ToArray();
				string strDownload = System.Text.Encoding.ASCII.GetString(data);

				Console.WriteLine ("Upload " + strInput.Length + " bytes: '" + strInput + "'");
				Console.WriteLine ("Download " + strDownload.Length + " bytes: '" + strDownload + "'");
			
				Assertion.AssertEquals ("Uploaded and downloaded strings differ.", strDownload, strOutput);
				
				//delete
				ftp.DeleteFile(Site.UploadDir + "/" + uniqName);
			
				// file-file
				uniqName = Utils.GetUniqueName(ftp, "binar", ".bin");
							
				nUploadedLen = ftp.PutFile (tempFilePath, Site.UploadDir + "/" +uniqName);
				// test of correct size 1
				Assertion.Assert(
					"Upload failed. Expected " + data.Length + ", transferred " + nUploadedLen + " bytes", 
					nUploadedLen == data.Length);			

				Console.WriteLine ("ok");			

				// test of correct size 2			
				ftp.SetTransferType (FtpTransferType.Binary);
				
				tempFilePath = Path.GetTempFileName();
				ftp.GetFile(Site.UploadDir + "/" + uniqName, tempFilePath);
				FileStream fileOutput = new FileStream(tempFilePath, FileMode.Open);
				fileOutput.Read(data, 0, data.Length);
				string strfileout = data.ToString();

				Console.WriteLine ("Upload " + strInput.Length + " bytes: '" + strInput + "'");
				Console.WriteLine ("Download " + strOutput.Length + " bytes: '" + strOutput + "'");
				
				//delete, disconnect
				ftp.DeleteFile(Site.UploadDir + "/" +uniqName);
				Assertion.AssertEquals ("Uploaded and downloaded text are different!", strDownload, strOutput);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}


		private long m_nTransfered;

		public void TransferProgressTest (object sender, FtpTransferProgressEventArgs e)
		{

			if (e.State==FtpTransferState.None)
			{
				//				return;
				if (m_nTransfered>340874)
					throw new Exception("More transferred bytes: " + m_nTransfered);

				if (m_nTransfered<340874)
					throw new Exception("Less transferred bytes: " + m_nTransfered);

				return;
			}

			if (e.BytesTransfered<0)
				throw new Exception("Stupid call ...");

			if (e.BytesTransfered<=m_nTransfered)
				throw new Exception("Uploaded bytes came down !!!");

			m_nTransfered = e.BytesTransfered;					
		}

		[Test]
		public void ProgressTest()
		{
			Console.WriteLine ("---------------- ProgressTest ----------------");

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
			
				m_nTransfered = -1;
				string uniqName = Utils.GetUniqueName(ftp, "array", ".bin");
				ftp.TransferProgress += new FtpTransferProgressEventHandler (TransferProgressTest);

				ftp.SetTransferType (FtpTransferType.Binary);

				m_nTransfered = -1;

				Console.WriteLine ("Progress test ...");

				// create data in memory
				byte[] data = new byte[340874];
				System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);

				// upload
				long nUploadedLen = ftp.PutFile(oInput,Site.UploadDir + "/" +uniqName);

				ftp.DeleteFile(Site.UploadDir + "/" +uniqName);
				// check correct size
				Assertion.Assert(
					"Expected size " + data.Length + ", transferred " + nUploadedLen + " bytes", 
					nUploadedLen == data.Length);

				// check correct size 2
				Assertion.Assert(
					"Bad transfer data. Transferred " + data.Length + ", obtained " + m_nTransfered + " bytes", 
					m_nTransfered == data.Length);
			
				Console.WriteLine ("ok");
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}
		}

		/// <summary>
		/// TestExecuteCommand - tests SendCommand and  ReadResponse methods
		/// TestExecuteCommand is tested by
		/// 1. simple commands calling (e.g. NOOP, CWD,...)
		/// 2. implementation RFC RENAME SEQUENCE (RNTO, RNFR)		
		/// </summary>


		//[Test]
		public void TestExecuteCommand()
		{
			Console.WriteLine ("---------------- TestGetNameList ----------------");

			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				ftp.ChangeDirectory(Site.UploadDir);

				ftp.SendCommand("NOOP");
				FtpResponse response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);
				Console.WriteLine (".. ok");

				ftp.SendCommand("CDUP");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);
				
				ftp.ChangeDirectory(Site.UploadDir);

				ftp.SendCommand("CWD");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);

				ftp.SendCommand("HELP");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);

				ftp.SendCommand("REMOTEHELP");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 5, response.Group);

				ftp.SendCommand("HELP SIZE");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);

				string uniqName=Utils.GetUniqueName(ftp, "", "");
				ftp.CreateDirectory(uniqName);
				ftp.ChangeDirectory(uniqName);
				ftp.SendCommand("RMD " + uniqName);
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 5, response.Group);

				ftp.SendCommand("CDUP");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);

				ftp.SendCommand("RMD " + uniqName);
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);
		
				ftp.SendCommand("LOGOUT");
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 5, response.Group);
				
				ftp.SendCommand("CWD " + Site.UploadDir);
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);

/*									
         ABOR, ALLO, DELE, SMNT, MODE, PASV,
         QUIT, SITE, PORT, SYST, STAT, RMD, MKD, PWD, STRU, and TYPE*/


				//ftp size
				uniqName = Utils.GetUniqueName(ftp, "", "");
				string text = "1234567890";
				long nUploadedLen = Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
				string uniqName2 = Utils.GetUniqueName(ftp, "", "");
				ftp.SendCommand("SIZE " + uniqName);
				response = ftp.ReadResponse();
				Assertion.AssertEquals ("Unexpected response group.", 2, response.Group);
				Assertion.AssertEquals ("Response and length differ ..", nUploadedLen.ToString(), response.Description);
				Console.WriteLine (".. ok");

				//send command test, using rename schema
				/*RENAME SEQUENCE
					+---+   RNFR    +---+   1,2    +---+
					| B |---------->| W |--------->| E |
					+---+           +---+	   --->+---+
									| |       |
							3		| | 4,5   |
					----------------	---   |
					|					   |  |   +---+
					|               ------------->| S |
					|              |   1,3 |  |   +---+
					|             2|  --------
					|              | |     |
					V              | |     |
				+---+   RNTO    +---+ 4,5  ---->+---+
				|   |---------->| W |---------->| F |
				+---+           +---+           +---+
				*/
				ftp.SendCommand("RNFR "+uniqName);
				Assertion.AssertEquals ("Unexpected state.", FtpState.Reading, ftp.State);
				response = ftp.ReadResponse();
				if ((response.Group == 1)||(response.Group == 2)) 
				{								
					if (response.Group == 1) Assertion.AssertEquals ("Unexpected state.", FtpState.Ready, ftp.State);
					if (response.Group == 2) Assertion.AssertEquals ("Unexpected state.", FtpState.Reading, ftp.State);	
					throw new Exception ("Strange ftp response ..");					
				}

				if ((response.Group == 4)||(response.Group == 5)) 
				{
					Assertion.AssertEquals ("Unexpected state.", FtpState.Ready, ftp.State);
					Console.Write("ok");
				}

				if (response.Group == 3) 
				{
					Assertion.AssertEquals ("Unexpected state.", FtpState.Processing, ftp.State);
					ftp.SendCommand("RNTO " + uniqName2);					
					response = ftp.ReadResponse();
					if ((response.Group == 1)||(response.Group == 3)) 
					{								
						if (response.Group == 3) Assertion.AssertEquals ("Unexpected state.", FtpState.Processing, ftp.State);
						if (response.Group == 1) Assertion.AssertEquals ("Unexpected state.", FtpState.Reading, ftp.State);
						throw new Exception ("Strange ftp response ..");					
					}
					if ((response.Group == 4)||(response.Group == 5)) 
					{
						Assertion.AssertEquals ("Unexpected state.", FtpState.Ready, ftp.State);
						Assertion.Fail("Rename failed ... ");
					}

					if (response.Group == 2) 
					{
						Assertion.AssertEquals ("Unexpected state.", FtpState.Ready, ftp.State);
						Console.WriteLine("...ok");
					}
				}				
				if ((response.Group < 1) || (response.Group >= 6))
				{					
					throw new Exception ("Strange ftp response ..");					
				}
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}

		}
		/// <summary>
		/// TestExceptionStatus 
		/// Test tries to prepare situation when all kind 
		/// of FtpExceptionStatus are throwed
		/// </summary>
		[Test]
		public void TestExceptionStatus()
		{
			Console.WriteLine ("---------------- TestExceptionStatus ----------------");			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp (Site);
				//string uniqName = Utils.GetUniqueName(ftp, "", "");			
				//
				Console.Write("ProxyNameResolutionFailure...");
				FtpProxy ftpProxy = new FtpProxy(FtpProxyType.FtpOpen, "samuel", 220, "bleble");
				try
				{
					ftp.Proxy = ftpProxy;
				}
				catch (FtpException e)
				{
					if (e.Status == FtpExceptionStatus.ProxyNameResolutionFailure) Console.WriteLine("...ok");				
					else Assertion.Fail("Unexpected error status");
				}

				Utils.CloseFtp(ftp);
				//ftp.Disconnect();
				string host = Site.Host;
				try
				{
					Site.Host = "kralicektvormalinkybezbranny";
					ftp = Utils.ConnectFtp(Site);
				}
				catch (FtpException e)
				{
					if (e.Status == FtpExceptionStatus.NameResolutionFailure) Console.WriteLine("...ok");				
					else Assertion.Fail("Unexpected error status");				
				}
				finally 
				{
					Site.Host = host;				
				}

				/* TODO
				FtpExceptionStatus.Pending;	
				*/

				// WAS TESTED IN BASIC TESTS
				//FtpExceptionStatus.ProtocolError;			

				//NOT TESTABLE THIS WAY
				/* 				FtpExceptionStatus.ConnectionClosed
				 * 				FtpExceptionStatus.ConnectFailure;				
				 *  			FtpExceptionStatus.ReceiveFailure;
				 *				FtpExceptionStatus.SendFailure;
				 *				FtpExceptionStatus.SocketError;				
				 *				FtpExceptionStatus.UnclassifiableError;	
				 * 				FtpExceptionStatus.Timeout; */
			
				//TESTED IN ASYNCHRONOUS TESTS
				//FtpExceptionStatus.OperationAborted;
				//FtpExceptionStatus.AsyncError;		

				// HARDLY TO TEST, LATER MEYBE
				//FtpExceptionStatus.ServerProtocolViolation;

			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}

	}
}
