using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Configuration;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	/// <summary>
	/// Tests FtpList class and GetList method.
	/// </summary>
	abstract public class FtpGetListTest
	{
		public Site Site = null;
		
		/// <summary>		
		/// TestFtpListProperties - test properties of FtpList
		/// - count, Item properties (IsSymlink, IsFile, ..)
		/// </summary>
		[Test]
		public void TestFtpListProperties()
		{
			Console.WriteLine ("---------------- TestFtpListProperties ----------------");			
			Ftp ftp = null;
			try
			{
				ftp = Utils.ConnectFtp(Site);				
				ftp.ChangeDirectory(Site.UploadDir);

				FtpList ftpList = Utils.PrepareRemoteStructure(ftp);				

				// test count properties
				Console.WriteLine("Count properties [expected " +ftpList.Count+ "] ...");
				Assertion.AssertEquals("Expected files count is bigger! ",ftpList.Count, ftp.GetList().Count);
				Console.WriteLine("[was "+ ftp.GetList().Count + "].. ok ");				

				//Console.WriteLine("ITEM PROPERTIES TEST");
				Console.WriteLine("Item properties test ...");
				FtpList ftpListRemote=ftp.GetList();
				ftpListRemote.Sort();
				ftpList.Sort();
				for (int i=0;i<ftpList.Count;i++) 
				{
					if (ftpList[i].IsFile) Assertion.AssertEquals("Expected files size differ! ",ftpList[i].Size, ftpListRemote[i].Size);
					Assertion.AssertEquals("Expected name files differ! ",ftpList[i].Name, ftpListRemote[i].Name);
					Assertion.AssertEquals("Expected IsFile differ! ",ftpList[i].IsFile, ftpListRemote[i].IsFile);
					Assertion.AssertEquals("Expected IsSymlink differ! ",ftpList[i].IsSymlink, ftpListRemote[i].IsSymlink);
					Assertion.AssertEquals("Expected IsDirectory differ! ",ftpList[i].IsDirectory, ftpListRemote[i].IsDirectory);
				}
				Console.WriteLine(".. ok ");			

				Console.WriteLine("Item non existing position size test ...");
				try
				{
					string name = ftpListRemote[ftpList.Count].Name;
					Assertion.Fail ("Impossible!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine ("...ok");
				}

				// for each ftpItem in ftpList
				// partionally tested in DeleteRemoteStructure
				//delete	PARTIONALLY (without test root :-<)

				ftpListRemote = Utils.DeleteRemoteStructure(ftp, ftpListRemote);
				//delete root :-D
				Utils.DeleteRemoteStructure(ftp, ftpListRemote);
			}
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}
		

		/// <summary>
		/// TestFtpListAdd - test ftpList.Add method
		/// with different params
		/// </summary>


		[Test]
		public void TestFtpListAdd()
		{
			Console.WriteLine ("---------------- TestFtpListAdd ----------------");			

				FtpList ftpList = new FtpList();
				DateTime now = DateTime.Now;
				FtpItem ftpItem = new FtpItem("ble1", 20, FtpItemType.Directory, now, "cesticka");
				// normal + symlink
				ftpList.Add(ftpItem);
				// test all parameters using ftpitem(string)
				Assertion.AssertEquals("Expected files size differ! ",20,ftpList["ble1"].Size );
				Assertion.AssertEquals("Expected name files differ! ","ble1", ftpList["ble1"].Name);
				Assertion.AssertEquals("Expected IsFile differ!", false, ftpList["ble1"].IsFile);
				Assertion.AssertEquals("Expected IsSymlink differ! ",false, ftpList["ble1"].IsSymlink);
				Assertion.AssertEquals("Expected IsDirectory differ! ", true, ftpList["ble1"].IsDirectory);
				Assertion.AssertEquals("Expected IsDirectory differ! ",  now, ftpList["ble1"].Modified);
				Assertion.AssertEquals("Expected Symlink path differ! ",  "cesticka", ftpList["ble1"].SymlinkPath);				
				// - item name exists
				try
				{
					ftpList.Add(ftpItem);
					Assertion.Fail ("Impossible!!");
				}
				catch (ArgumentException e)
				{				
					e=e;Console.WriteLine("... ok");
				}

				// - null item
				try
				{
					ftpList.Add(null);
					Assertion.Fail ("Impossible!!");
				}
				catch (ArgumentException e)
				{				
					e=e;Console.WriteLine("... ok");
				}
				//symlink
				ftpItem = new FtpItem("ble2", 20, FtpItemType.Symlink, DateTime.Now,"bleble");
				ftpList.Add(ftpItem);

				//file
				ftpItem = new FtpItem("ble3", 20, FtpItemType.File);
				ftpList.Add(ftpItem);
				Assertion.AssertEquals("Expected files size differ! ",20,ftpList["ble1"].Size );
				Assertion.AssertEquals("Expected name files differ! ","ble3", ftpList["ble3"].Name);
				Assertion.AssertEquals("Expected IsFile differ!", true, ftpList["ble3"].IsFile);
				Assertion.AssertEquals("Expected IsSymlink differ! ",false, ftpList["ble3"].IsSymlink);
				Assertion.AssertEquals("Expected IsDirectory differ! ", false, ftpList["ble3"].IsDirectory);				

				//  - GetTotalSize
				long nSize = ftpList.GetTotalSize();
				Assertion.AssertEquals("Total size differ ...",20,nSize);

				//ftpList.Sort(); base test in Test FtpListProperties
		}

		/// <summary>
		/// TestFtpListGetFiles
		/// [GetFiles returns all files in list]
		/// Basic test for FtpList.GetFiles method 
		/// </summary>
		
		[Test]
		public void TestFtpListGetFiles()
		{
			Console.WriteLine ("---------------- TestFtpListGetFiles ----------------");			
			Ftp ftp = null;
			try
			{	
	
				ftp = Utils.ConnectFtp(Site);				
				ftp.ChangeDirectory(Site.UploadDir);
				string strParam = null;				
				Regex rParam = null;

				//empty list
				//empty params
				Console.WriteLine("EMPTY LIST");
				FtpList ftpList = new FtpList();
				Console.Write("String, bool param ...");								
				try
				{
					ftpList.GetFiles("",true);
					Assertion.Fail ("Empty param passed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				Console.Write("String param ...");								
				try
				{
					strParam = "";
					ftpList.GetFiles(strParam);
					Assertion.Fail ("Empty param passed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				
				//regexp, match anything
				Console.Write("Regexp ...");							
				rParam = new Regex("");				
				string[] resArr = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("Regexp in GetFiles matched diff. list count:", resArr.GetLength(0), ftpList.Count);
				Console.WriteLine("...ok");

				//string, match anything
				Console.Write("Regexp ...");							
				strParam ="*";
				resArr = ftpList.GetFiles(strParam);
				Assertion.AssertEquals("Count of returned files differ...", resArr.GetLength(0), ftpList.Count);
				Console.WriteLine("...ok");
				
				//NOT EMPTY LIST
				Console.WriteLine("NOT EMPTY LIST");
				//null params
				Console.WriteLine("NULL PARAMS");
				ftpList = Utils.PrepareRemoteStructure(ftp);
				rParam = null;
				strParam = null;
				Console.Write("String, bool param ...");								
				try
				{
					ftpList.GetFiles(strParam,true);
					Assertion.Fail ("Null param pass");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				Console.Write("String param ...");								
				try
				{				
					ftpList.GetFiles(strParam);
					Assertion.Fail ("Null param pass");
				}

				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				Console.Write("RegEx param ...");								
				try
				{
					ftpList.GetFiles(rParam);
					Assertion.Fail ("Null param pass");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				Console.WriteLine("STRING BOOLEAN");
				// string-boolean
				// - normal
				Console.Write("Normal test ...");
				strParam = "*";
				resArr = ftpList.GetFiles(strParam, true);
				Assertion.AssertEquals("GetFiles returned bad files count.", resArr.GetLength(0), ftpList.Count);
				Console.WriteLine("... ok");

				//case sensitivity yes
				Console.Write("Case sensitivity yes ...");
				strParam = "DIR*";
				resArr = ftpList.GetFiles(strParam, true);

				strParam = "dir*";
				string[] resArr2 = ftpList.GetFiles(strParam, true);
				Assertion.Assert("GetFiles returned bad files count",resArr2.GetLength(0)<resArr.GetLength(0));
				Console.WriteLine("... ok");

				//case sensitivity no
				Console.Write("Case sensitivity no ...");
				strParam = "dir*";
				resArr2 = ftpList.GetFiles(strParam, false);
				Assertion.AssertEquals("GetFiles returned bad files count",resArr.GetLength(0),resArr2.GetLength(0));
				Console.WriteLine("... ok");

				// - name not exist
				Console.Write("Name not exists ...");
				strParam = "ble*";
				resArr2 = ftpList.GetFiles(strParam, false);
				Assertion.AssertEquals("GetFiles returned something (non existing match)",0, resArr2.GetLength(0));
				Console.WriteLine("... ok");

				// - name is file
				Console.Write("Name is file ...");
				strParam = "*bum";
				resArr2 = ftpList.GetFiles(strParam, false);
				Assertion.AssertEquals("GetFiles returned bad file count",1, resArr2.GetLength(0));
				Console.WriteLine("... ok");

				// - name is symlink
				Console.Write("Name is symlink ...");
				FtpItem item = new FtpItem("ble2", 20, FtpItemType.Symlink, DateTime.Now, "hulalala");
				ftpList.Add(item);
				strParam = "ble2";
				resArr2 = ftpList.GetFiles(strParam, false);
				Assertion.Assert("GetFiles returned bad file count.",resArr2[0] == "ble2");
				Console.WriteLine("... ok");

				// - after delete
				Console.Write("After delete ...");
				ftpList.Sort();
				ftpList.Remove(strParam);
				resArr2 = ftpList.GetFiles(strParam, false);
				Assertion.AssertEquals("GetFiles returned bad file count",0, resArr2.GetLength(0));
				Console.WriteLine("... ok");					

				// regexp
				Console.WriteLine("REGEXP");
				//normal test 
				Console.WriteLine("Normal reg, all items ...");
				rParam = new Regex(".*");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", ftpList.Count, resArr2.GetLength(0));
				Console.WriteLine("... ok");

				// - normal, some types of regexp
				Console.WriteLine("Normal, some types of regexp ...");
				rParam = new Regex(@"^(?:(?:(?:0?[13578]|1[02])(\/|-|\.)31)\1|(?:(?:0?[1,3-9]|1[0-2])(\/|-|\.)(?:29|30)\2))(?:(?:1[6-9]|[2-9]\d)?\d{2})$|^(?:0?2(\/|-|\.)29\3(?:(?:(?:1[6-9]|[2-9]\d)?(?:0[48]|[2468][048]|[13579][26])|(?:(?:16|[2468][048]|[3579][26])00))))$|^(?:(?:0?[1-9])|(?:1[0-2]))(\/|-|\.)(?:0?[1-9]|1\d|2[0-8])\4(?:(?:1[6-9]|[2-9]\d)?\d{2})$");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 0,resArr2.GetLength(0));

				rParam = new Regex(@"^([a-zA-Z0-9_\-\.]+)@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.)|(([a-zA-Z0-9\-]+\.)+))([a-zA-Z]{2,4}|[0-9]{1,3})(\]?)$");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 0, resArr2.GetLength(0));

				rParam = new Regex(@"^[\w-]+(?:\.[\w-]+)*@(?:[\w-]+\.)+[a-zA-Z]{2,7}$");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 0,resArr2.GetLength(0));

				rParam = new Regex(@"[D][I][R]?");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 3, resArr2.GetLength(0));

				rParam = new Regex(@"[^D][^I][^R]?");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", ftpList.Count, resArr2.GetLength(0));

				rParam = new Regex(@"(AUX|PRN|NUL|COM\d|LPT\d)+\s*$");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 0, resArr2.GetLength(0));
				
				Console.WriteLine("... ok");

				// - regexp is empty
				Console.WriteLine("Regex match anything ...");
				rParam = new Regex("");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles empty regexp returned bad file count", ftpList.Count,resArr2.GetLength(0));
				Console.WriteLine("... ok");
				
				// - regexp match nothing
				Console.WriteLine("Regex match nothing ...");
				rParam = new Regex("[x]");
				resArr2 = ftpList.GetFiles(rParam);
				Assertion.AssertEquals("GetFiles returned bad file count", 0,resArr2.GetLength(0));
				Console.WriteLine("... ok");
				
				ftpList = Utils.DeleteRemoteStructure(ftp, ftpList);
				Utils.DeleteRemoteStructure(ftp, ftpList);

			}
				
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}

		// 

		/// <summary>
		/// TestFtpListRemove - tests ftpList.Remove method
		/// with different params
		/// </summary>
		
		[Test]
		public void TestFtpListRemove()
		{
			Console.WriteLine ("---------------- TestFtpListRemove ----------------");			
			Ftp ftp = null;
			try
			{		
				ftp = Utils.ConnectFtp(Site);				
				ftp.ChangeDirectory(Site.UploadDir);
				string strParam = null;				
				FtpItem ftpItem = new FtpItem(" ", 0, FtpItemType.Directory);
			// Remove test
				Console.WriteLine("REMOVE TEST");
				FtpList ftpList = new FtpList();
				ftpList = Utils.PrepareRemoteStructure(ftp);
				FtpList ftpList2 = new FtpList();
				ftpList2 = ftpList.Clone();	

			// string-name
				Console.WriteLine("STRING PARAM");
			// null params
				Console.Write("Null param ...");
				try
				{
					ftpList.Remove(strParam);
					Assertion.Fail ("Null param passed.");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

			// - not exist
				Console.Write("Not exists ...");
				strParam = ftpList[1].Name+"bleble";
				try
				{
					ftpList.Remove(strParam);
					Assertion.Fail ("Non existing item removed!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

			// - exists
				Console.Write("Exists ...");
				strParam = ftpList[1].Name;
				ftpList.Remove(strParam);
				Assertion.AssertNull("Removed param is in list!",ftpList[strParam]);
				Console.WriteLine("...ok");

				ftpList2 = Utils.DeleteRemoteStructure(ftp, ftpList2);
				Utils.DeleteRemoteStructure(ftp, ftpList2);
						
				// INT PARAM
				Console.WriteLine("INT PARAM");
				ftp.ChangeDirectory(Site.UploadDir);
				ftpList = Utils.PrepareRemoteStructure(ftp);
				ftpList2 = ftpList.Clone();				

				// exists and is > 0
				Console.Write("Exists and is > 0 ...");
				strParam = ftpList[4].Name;
				ftpList.Remove(4);
				Assertion.AssertNull("Removed item is in list!",ftpList[strParam]);
				Console.WriteLine("...ok");

				//after delete, same index
				Console.Write("After delete, same index ...");
				strParam = ftpList[4].Name;
				ftpList.Remove(4);
				Assertion.AssertNull("Removed param is in list!",ftpList[strParam]);
				Console.WriteLine("...ok");

				//index = 1
				Console.Write("Exists and is = 1 ...");
				strParam = ftpList[1].Name;
				ftpList.Remove(1);
				Assertion.AssertNull("Removed param is in list!",ftpList[strParam]);
				Console.WriteLine("...ok");

			// - is bigger than list count
				Console.Write("Exists and is bigger than list count ...");
				try
				{
					ftpList.Remove(6);
					Assertion.Fail ("Out of array parameter passed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

			// - is a very big number
				Console.Write("Exists and is a very big number ...");
				try
				{
					ftpList.Remove(int.MaxValue);
					Assertion.Fail ("Out of array parameter passed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
			// - is a very small number
				Console.Write("Exists and is a very small number ...");
				try
				{
					ftpList.Remove(int.MinValue);
					Assertion.Fail ("Out of array parameter passed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
			// - = 0
				Console.Write("Exists and is = 0 ...");
				strParam = ftpList[0].Name;
				ftpList.Remove(0);
				Assertion.AssertNull("Removed item exists in list!",ftpList[strParam]);
				Console.WriteLine("...ok");

			// ITEM
				Console.WriteLine("ITEM");
			// - exists
				Console.Write("Exists");
				strParam = ftpList[0].Name;
				ftpList.Remove(ftpList[0]);
				Assertion.AssertNull("Removed item exists in list!!",ftpList[strParam]);
				Console.WriteLine("...ok");

			// - not exist (another instance item)
				Console.Write("Not exist (another instance item)");
				ftpItem = new FtpItem(ftpList[0].Name, ftpList[0].Size, ftpList[0].Type);
				strParam = ftpList[0].Name;
				try
				{
					ftpList.Remove(ftpItem);
					Assertion.Fail ("Ietm not in list removed!!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

			//is null
				Console.Write("Item is null ...");
				ftpItem = null;
				strParam = ftpList[0].Name;
				try
				{
					ftpList.Remove(ftpItem);
					Assertion.Fail ("NULL FtpItem removed!");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
				ftpList2 = Utils.DeleteRemoteStructure(ftp, ftpList2);
				Utils.DeleteRemoteStructure(ftp, ftpList2);
			}
				
			finally
			{
				Utils.CloseFtp (ftp);
			}	
		}


		/*** TEST TEMPLATE ***/
		// Test_Name
		// - Test_Description
		//Test
		/*public void Test_Name()
			{
				Console.WriteLine ("---------------- Test_Name ----------------");			
				Ftp ftp = null;
				try
				{
					ftp = Utils.ConnectFtp (Site);
					string uniqName = Utils.GetUniqueName(ftp, "", "");			
				//TEST HERE	
				}
				finally
				{
					Utils.CloseFtp (ftp);
				}	
			}*/
	}
}