using System;
using System.IO;
using System.Configuration;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	[TestFixture]
	/// <summary>
	/// FtpListTest tests basic public method 
	///		- FtpItem.Parse  (with FtpItem properties)
	///		- FtpList.Add
	/// </summary>
	public class FtpListTest
	{

		//############################################################################################################
		#region CONFIGURATION
		// ----------------------------------------------------
		// Test configuration - moved to Site class
		// ----------------------------------------------------

		// array of input/outputs
		// {source string alias RAWLIST, NAME, SIZE, DT MODIFY, SYMLINK PATH, ISDIR, ISFILE, ISSYMLINK}
		/*		--------------------------------------------------------------
				Unix and others:
				-rw-r--r--  1      root     other      531    Jan 29 03:26 README
				dr-xr-xr-x  2      root     other      512    Apr  8  1994 etc
				dr-xr-xr-x  2      root        512     Apr  8  1994 etc
				lrwxrwxrwx  1      root     other        7    Jan 25 00:17 bin -> usr/bin
				----------  1      owner    group  1803128    Jul 10 10:18 ls-lR.Z
				d---------  1      owner    group        0    May  9 19:45 Softlib
				-rwxrwxrwx  1      noone    nogroup    322    Aug 19  1996 message.ftp
				d--x--x--x  2      0        999           512 Jul 31  1997 bin
				d--x--x--x  3      0        999           512 Sep 08  1997 etc
				d---------  2      0        999        	  512 Nov 22  2000 private		
		
		*/


		// okList array of supported and correct input/outputs
		// {source string alias RAWLIST, NAME, SIZE, DT MODIFY, SYMLINK PATH, ISDIR, ISFILE, ISSYMLINK}
		
		DateTime dateTime = DateTime.Now;
		object[,] okList = {
							   // LIKE UNIX ftp.scene.org
		{	//SYMLINK
			"lrwxrwxrwx   1 redhound ftpadm         27 Jan  9 15:13 assembly02 -> pub/parties/2002/assembly02",
			"assembly02", 27, Utils.correctYear(new DateTime(DateTime.Now.Year,1,9,15,13,0)), "pub/parties/2002/assembly02", false, false, true},
		
		{	//Another symlink
			"lrwxrwxrwx   1 redhound ftpadm         12 Apr  8 17:05 incoming -> pub/incoming",
			"incoming", 12, Utils.correctYear(new DateTime(DateTime.Now.Year,4,8,17,05,0)), "pub/incoming", false, false, true},

		{	//file
			"-rw-r--r--   1 redhound ftpadm    5731337 Sep 18 23:04 ls-lR",
			"ls-lR", 5731337, Utils.correctYear(new DateTime(DateTime.Now.Year,9,18,23,04,0)), null, false, true, false},

		{	//another file
			"-rw-r--r--   1 redhound ftpadm     837336 Sep 18 23:04 ls-lR.gz",
			"ls-lR.gz", 837336, Utils.correctYear(new DateTime(DateTime.Now.Year,9,18,23,04,0)), null, false, true, false},					

		{	// dir with year date
			"drwxrwsr-x   9 redhound ftpadm        312 Jun 16 2001 pub",
			"pub", 312, new DateTime(2001,6,16,0,0,0), null, true, false, false},		
		
							   //	aftp.jet2web.cz
	   {	//symlink, month before day, file name . 
		   "lrwxr-xr-x   1 root     wheel          15 Jun  2  2001 fb -> .diski1/FreeBSD",
		   "fb", 15, new DateTime(2001,6,2,0,0,0), ".diski1/FreeBSD", false, false, true},
	   {	// dir month before
		   "drwxr-xr-x   4 ftp      ftp           512 Sep 23 07:09 incoming",
		   "incoming", 512, Utils.correctYear(new DateTime(DateTime.Now.Year,9,23,7,9,0)), null, true, false, false},
	   {	//29 feb - correct
		   "-rw-r--r--   1 root     wheel         461 Feb  29  2004 login.msg",
		   "login.msg", 461, new DateTime(2004,2,29,0,0,0), null, false, true, false},
	   
							   //ftp.vse.cz
		{	//user is number
			"d--x--x--x    3 0        999           512 Apr 04 07:18 ZERAW",
			"ZERAW", 512, Utils.correctYear(new DateTime(DateTime.Now.Year,4,4,7,18,0)), null, true, false, false},
							   // other (almost) correct strings
		{//another file
			"----------  1      owner    group  1803128    Jul 10 10:18 ls-lR.Z",
			"ls-lR.Z", 1803128, Utils.correctYear(new DateTime(DateTime.Now.Year,7,10,10,18,0)), null, false, true, false},

		{//no group
			"dr-xr-xr-x  2      nogroup        512     Apr  8  1994 etc",
			"etc", 512, new DateTime(1994,4,8,0,0,0), null, true, false, false},

		{//no owner
			"-rwxrwxrwx  1          noowner    322    Aug 19  1996 message.ftp",
			"message.ftp", 322, new DateTime(1996,8,19,0,0,0), null, false, true, false},
				
		{//strange names lot of 
			"-rwxrwxrwx  1          noowner    322    Aug 19  1996 + . +",
			"+ . +", 322, new DateTime(1996,8,19,0,0,0), null, false, true, false},
		{//strange names 2
			"-rwxrwxrwx  1          noowner    322    Aug 19  1996 +,zm ]-0-2985 . + ,zm ]-0-2985",
			"+,zm ]-0-2985 . + ,zm ]-0-2985", 322, new DateTime(1996,8,19,0,0,0), null, false, true, false},							   
		{//strange names 3
			"-rwxrwxrwx  1          noowner    322    Aug 19  1996 .......",
			".......", 322, new DateTime(1996,8,19,0,0,0), null, false, true, false},

							   /*lrwxr-xr-x    1 0        999             8 Nov 15  2001 ZERAW.AT -> ZERAW/AT
								   d--x--x--x    2 0        999           512 Jul 31  1997 bin
								   d--x--x--x    3 0        999           512 Sep 08  1997 etc
								   d---------    2 0        999           512 Nov 22  2000 private
								   drwxr-xr-x   24 12345    999          1024 May 17 10:56 pub
								   drwxrwxrwx    2 0        0             512 Sep 23 09:47 upload
								   ---------------------------------------------------------------------------
								   ftp.jp.scene.org

								   total 9288
								   drwxr-xr-x   6 ftpadmin  ftp           512 Jan 16  2002 .1
								   drwx------   2 ftpadmin  ftp           512 Nov 30  2000 .ssh
								   d--x--x--x   2 root      operator      512 Nov 16  2000 bin
								   d--x--x--x   2 root      operator      512 Nov 16  2000 dev
								   d--x--x--x   2 root      operator      512 Apr 25 09:07 etc
								   -rw-r--r--   1 ftpadmin  ftp       9488439 Sep 19 09:37 ls-lR.gz
								   drwxr-xr-x  14 ftpadmin  ftp           512 May 13 01:50 pub
								   d--x--x--x   3 root      operator      512 Nov 16  2000 usr
								   -rw-r--r--   1 ftpadmin  ftp           219 Nov 28  2000 welcome.msg*/

							   //DOS-MODE  		
							   //maslo WIN XP FTP
							   //WAS BUG 833
		{//normal file
			"09-15-02  03:08PM		           4873 0command.txt",
			"0command.txt", 4873, new DateTime(2002,9,15,15,8,0), null, false, true, false},
		{//DOTS file
			"09-13-02  11:08AM                15425 .......txt",
			".......txt", 15425, new DateTime(2002,9,13,11,8,0), null, false, true, false},		
		{//  file - this century
			//WAS BUG 834
			"2-29-04  08:04PM                 7704 Appendix FTP LIST Specification s s s s s .txt		",
			"Appendix FTP LIST Specification s s s s s .txt", 7704, new DateTime(2004,2,29,20,4,0), null, false, true, false},		
		{//dir, last century
			"2-28-99  01:34PM       <DIR>          pokus		",
			"pokus", 0, new DateTime(1999,2,28,13,34,0), null, true, false, false},

							   /*  
							   09-12-02  12:30PM                13794 cert.txt
							   09-13-02  11:07AM                16906 ftp-daemons.txt
							   09-12-02  12:38PM                 1019 list_problems.txt
							   09-11-02  06:50PM                25215 rfc1350.txt
							   09-11-02  08:38PM                 8697 rfc1579.txt
							   09-11-02  08:39PM                15451 rfc2428.txt
							   09-12-02  12:39PM                18319 rfc2577.txt
							   09-11-02  08:37PM                 9935 rfc775.txt
							   09-11-02  06:50PM                23520 rfc783.txt
							   09-13-02  11:07AM                39369 rfc854.txt*/


		//NetWare:
		{//dir before
			"d [R----F-]  supervisor          512     Mar 16 18:53    login",
			"login", 512, Utils.correctYear(new DateTime(DateTime.Now.Year,3,16,18,53,0)), null, true, false, false},

 		{//dir after
		   "d [R----F-]  supervisor          512     Jan 12 18:53    login",
		   "login", 512, Utils.correctYear(new DateTime(DateTime.Now.Year,1,12,18,53,0)), null, true, false, false},
		
		{//file
			"- [R----F-]  rhesus           214059     Oct 20 15:27    cx.exe",
			"cx.exe", 214059, Utils.correctYear(new DateTime(DateTime.Now.Year,10,20,15,27,0)), null, false, true, false},				

							   //NetPresenz for the Mac:
		{//file
			"-------r--  326          1391972  1392298    Nov 22  1995 MegaPhone.sit",
			"MegaPhone.sit", 1391972, new DateTime(1995,11,22,0,0,0), null, false, true, false},

		{//another file
			"-------r--  326          9999999999  1392298    Nov 22  1995 MegaPhone.sit.qw.qw.   ",
			"MegaPhone.sit.qw.qw.", 9999999999, new DateTime(1995,11,22,0,0,0), null, false, true, false},

		{//dir
			"drwxrwxr-x  folder                2    May 10  1996 network",
			"network", 0, new DateTime(1996,5,10,0,0,0), null, true, false, false},	
	
							   //eplf (http://pobox.com/~djb/proto/eplf.txt):

/*		+/,m824255907,i!#@$%^&*(), 514
		+r,up644, This file name has spaces, commas, etc.
		+up000, secret*/


		{//dir
			"+i8388621.29609,m824255902,/,\tdev",
			"dev", 0, new DateTime(1996,2,14,0,58,22), null, true, false, false},	
		{//file
			"+i8388621.44468,m839956783,r,s10376,\trfc.txt ",
			"rfc.txt", 10376, new DateTime(1996,8,13,19,19,43), null, false, true, false},
							   //author examples		
		{//file1
			"+i8388621.48594,m825718503,r,s280,\tdjb.html",
			"djb.html", 280, new DateTime(1996,3,1,23,15,03), null, false, true, false},
		{//dir
			"+i8388621.50690,m824255907,/,\t514",
			"514", 0, new DateTime(1996,2,14,0,58,27), null, true, false, false},
		{//file2
			"+i8388621.48598,m824253270,r,s612,\t514.html",
			"514.html", 612, new DateTime(1996,2,14,0,14,30), null, false, true, false},

							   //VMS:

	   {
		   ".WWW_BROWSABLE;2            0   7-DEC-1995 16:29 [ANONYMOUS] (RWED,RE,RE,RE)",
		   ".WWW_BROWSABLE", 0,  new DateTime(1995,12,7,16,29,00),null, false, true, false},

	   {//some exam
		   "CII-MANUAL.TEX;1  213/216  29-JAN-1996 03:33:12  [ANONYMOU,ANONYMOUS]   (RWED,RWED,,)",
		   "CII-MANUAL.TEX", 216,  new DateTime(1996,1,29,3,33,12),null, false, true, false},
							   //vms.huji.ac.il
		{//dir
			"CONDIC.DIR;1                1   8-JAN-2003 18:28 [AGAY] (RWE,RWE,,RE)",
			"CONDIC", 512,  new DateTime(2003,1,8,18,28,0),null, true, false, false},
		{//file tar
			"DANI.TAR;1              99999955340  10-OCT-2002 21:50 [SYSTEM] (RWED,RWED,,RE)",
			"DANI.TAR", 512*99999955340,  new DateTime(2002,10,10,21,50,0),null, false, true, false},
		{//dir FORTRAn :-D
			"FORTRAN_.DIR;1               5   8-JAN-2003 18:31 [AGAY] (RWE,RWE,,RE)",
			"FORTRAN_", 512*5,  new DateTime(2003,1,8,18,31,0),null, true, false, false},
	   //BUG:836
		{//NO RIGHTS
			"CONC_C_G.ASC;1      %%RMS-E-PRV, insufficient privilege or file protection violation",
			"CONC_C_G.ASC", 0,  null,null, false, true, false}
		};

		/*
		 * TO VERSION 1.0 - improve Utils.correctYear algorithm
		 * {	//dir 
			"drwxrwsr-x   5 redhound ftpadm        120 Apr 11 15:27 mirrors",
			"mirrors", 120, Utils.correctYear(new DateTime(DateTime.Now.Year,4,11,15,27,0)), null, true, false, false},		*/
		
		// incorrect inputs  
		string[] badList = {
		//UNIX
		//no group no owner ?? 
		"-rwxrwxrwx  1               322    Aug 19  1996 message1.ftp",
		//augu
		"-rwxrwxrwx  1       1   1     322    Augu 19  1996 message2.ftp",
		//no month
		"-rwxrwxrwx  1          ble ble    322    19  1996 message3.ftp",
		//no month 2
		"-rwxrwxrwx  1          1 1    322    19  1996 message4.ftp",
		// no day
		//BUG837
		"-rw-r--r--   1 root     wheel         461 Feb    2004 lmessage4.ftp",			
		
		//CORRECT string "-rw-r--r--   1 root     wheel         461 Feb 29   2004 login.msg",

		};				

	   	#endregion

		
		//############################################################################################################
		#region TESTS
		
		/*** TEST TEMPLATE  ***/
		// Test_Name
		// - Test_Description
		//Test
		/*public void Test_Name()
		{
			Utils.ConsoleWriteLine ("---------------- Test_Name ----------------");			
		}*/

		/// <summary>
		/// TestOkItems 
		/// - for each item in array okList tries to parse using FtpItem class
		/// - oklist array is a list of correct ftp raw lists
		/// - test adding each item into FtpLIst
		/// </summary>
		[Test]
		public void TestOkItems()
		{
			Utils.ConsoleOpen (System.Threading.Thread.CurrentThread);
			Utils.ConsoleWriteLine ("---------------- TestOkItems ----------------");			
			try
			{
				FtpList ftpList = new FtpList();
		
				for (int i=0; i<=((Array) okList).GetUpperBound(0); i++)
				{
					//TODO:  add exception throw with correct Item position in an array					

					Utils.ConsoleWrite(">> " + okList[i,0]);
					FtpItem ftpItem = FtpItem.Parse((string)okList[i,0]);
					Assertion.AssertNotNull("Not parsed.",ftpItem);
					Utils.ConsoleWriteLine("  ...ok");
					Utils.ConsoleWrite("Tests " + ftpItem.Name + "...");
					Assertion.AssertEquals ("Name not parsed correctly",okList[i,1],ftpItem.Name);
					Assertion.AssertEquals("Size not parsed correctly",okList[i,2],ftpItem.Size);
					if (okList[i,3]!=null) Assertion.AssertEquals("Dt modify not parsed correctly",okList[i,3],ftpItem.Modified);
					Assertion.AssertEquals("SymlinkPath info not parsed correctly",okList[i,4],ftpItem.SymlinkPath);
					Assertion.AssertEquals("IsDir info not parsed correctly",okList[i,5],ftpItem.IsDirectory);
					Assertion.AssertEquals("IsFile info not parsed correctly",okList[i,6],ftpItem.IsFile);
					Assertion.AssertEquals("IsSymlink not parsed correctly",okList[i,7],ftpItem.IsSymlink);
					
					Utils.ConsoleWriteLine("ok");
					// ftp list adding
					Utils.ConsoleWrite("Ftp list adding...");
					ftpList.Add(ftpItem);
					Utils.ConsoleWriteLine("ok");
					// ftp list remove
					Utils.ConsoleWrite("Ftp list remove...");
					ftpList.Remove(ftpItem);
					Utils.ConsoleWriteLine("ok");
				}
			}
			finally
			{
				Utils.ConsoleClose ();
			}
		}
		
		/// <summary>
		/// TestBadItems 
		/// - for each item in array badList tries to parse using FtpItem class
		/// - array badList is a list of incorrect inputs
		/// </summary>
		
		[Test]
		public void TestBadItems()
		{
			Utils.ConsoleOpen (System.Threading.Thread.CurrentThread);
			Utils.ConsoleWriteLine ("---------------- TestBadItems ----------------");

			// nothing should cause exception
			foreach (string item in badList)
			{
				Utils.ConsoleWriteLine(item);
				Utils.ConsoleWrite ("test ...");
				FtpItem ftpItem = FtpItem.Parse(item);
				Assertion.AssertNull("Bad input parsed",ftpItem);
				Utils.ConsoleWriteLine("ok");
			}

			Utils.ConsoleClose ();
		}


		#endregion
	
	}
}