using System;
using System.IO;
using System.Threading;
using System.Net;
using System.Configuration;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	/// <summary>
	/// Tests FtpWebRequest and FtpWebResponse
	/// </summary>
	abstract public class FtpWebRequestTest
	{
		public Site Site = null;

		// Test some basic webrequest features 
		// ascii, binary, long file, not existing URI, incorrect password, incorrect port
		[Test]
		public void TestWebRequestBasic()
		{
			Console.WriteLine ("---------------- TestWebRequestBasic ----------------");			

			string text = null;
			string textLoc=null;
			
			//very simple test
			Console.Write("Very simple test...");
			text = "12345678101234567x201234567x301234567x401234567x50"
				+  "1234567x601234567x701234567x801234567x90123456x100";							
			textLoc = MakeWebRequest(text, FtpTransferType.Binary,0);
			Assertion.AssertEquals("Requested data differ ...",text,textLoc);
			Console.WriteLine("...ok");

			//binary
			Console.Write("Binary...");
			text = "je\ndna\n\rdva\r\nt\rri";
			textLoc = MakeWebRequest(text, FtpTransferType.Binary,0);
			Assertion.AssertEquals("Requested data differ ...",text,textLoc);
			Console.WriteLine("...ok");

			//ascii
			Console.Write("Ascii...");
			text = "je\ndna\n\rdva\r\nt\rri";
			textLoc = MakeWebRequest(text, FtpTransferType.Ascii,0);
			Assertion.Assert("Requested data differ ...",text.Length>=textLoc.Length);
			Console.WriteLine("...ok");

			//quite long
			Console.Write("Quite long test ...");
			byte[] data = new byte[340874];
			text = System.Text.Encoding.ASCII.GetString(data);
			textLoc = MakeWebRequest(text, FtpTransferType.Binary, 0);
			Assertion.AssertEquals("Requested data differ ...",text,textLoc);
			Console.WriteLine("...ok");

			// some troubles on server
			// TODO:insufficient rights

			// URI not exists 
			Console.Write("URI not exists ...");
			text = "je\ndna\n\rdva\r\nt\rri";
			try
			{
				textLoc = MakeWebRequest(text, FtpTransferType.Binary, 1);
				Assertion.Assert("Very strange",1!=1);
			}
			catch (WebException e)
			{
				if (e.Status == WebExceptionStatus.ProtocolError) Console.Write("...ok");
			}

			// Incorrect webrequest port
			Console.Write("URI not exists ...");
			text = "je\ndna\n\rdva\r\nt\rri";
			try
			{
				textLoc = MakeWebRequest(text, FtpTransferType.Binary, 3);
				Assertion.Assert("Very strange",1!=1);
			}
			catch (WebException e)
			{
				e=e; Console.Write("...ok");
			}

			// incorrect login
			Console.Write("Incorrect login ...");
			text = "je\ndna\n\rdva\r\nt\rri";
			try
			{
				textLoc = MakeWebRequest(text, FtpTransferType.Binary, 2);
				Assertion.Assert("Very strange",1!=1);
			}
			catch (WebException e)
			{
				e=e;Console.Write("...ok");
			}

			// 4..incorrect URI syntax
			// 5..long URI [1025]
			Console.Write("URI [1025] ...");
			text = "je\ndna\n\rdva\r\nt\rri";
			try
			{
				textLoc = MakeWebRequest(text, FtpTransferType.Binary, 5);
				Assertion.Assert("Very strange",1!=1);
			}
			catch (WebException e)
			{
				e=e;Console.Write("...ok");
			}
		}

		// TestWebRequestProperties
		// test setting and (not) using of all properties
		// not supported property - exception
		// not used property - is ignored

		[Test]
		public void TestWebRequestProperties()
		{
			Console.WriteLine ("---------------- TestWebRequestProperties ----------------");
			Ftp ftp = null;
			try
			{
				//PREPARE
				ftp = Utils.ConnectFtp (Site);
				NetworkCredential myCred = new NetworkCredential(Site.Login,Site.Password);
				string text = "0123456789";
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				ftp.ChangeDirectory(Site.UploadDir);
				long nUploadedLen = Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
				string uri = Utils.CreateUri(Site.Host, Site.Port, null,null,Site.UploadDir) + "/" +uniqName;
				Console.WriteLine("URI:" + uri);				
				WebRequest.RegisterPrefix ("ftp://", FtpWebRequest.Creator);


				//uri constructor

				System.Uri uriUri = new System.Uri(uri);
				WebRequest request = WebRequest.Create (uriUri);
				request.Credentials = myCred;
				WebResponse response = request.GetResponse();
				Stream stream = response.GetResponseStream();
				StreamReader local = new StreamReader(stream);
				string textLoc=null;
				textLoc = local.ReadToEnd();				
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	
				
				response.Close();
				stream.Close();
				local.Close();				

				// CONTENTLENGTH - not supported
				Console.Write("CONTENTLENGTH - not supported ...");
				request = WebRequest.Create (uriUri);
				request.Credentials = myCred;
				response = request.GetResponse();
				try
				{
					request.ContentLength = -5;
				}
				catch (NotSupportedException e)
				{
					response.Close();
					stream.Close();
					local.Close();
					e=e;Console.WriteLine("...ok");
				}

				// CONTENTTYPE PROPERTY - not used
				Console.Write("CONTENTTYPE PROPERTY - not used ...");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				request.ContentType = "application/msword";
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();			

				// CREATOR PROPERTY
				// Not test? Impossible ;-D

				// CREDENTIALS PROPERTY
				// credentials are mandatory within NOT anonymous servers test
				// correct input tested everywhere 
				// incorrect credentials are tested in anonymous tests (TODO: PREPARE THEM)

				// HeadersProperty
				Console.Write("HEADER PROPERTY - not used ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;
				WebHeaderCollection whc = new WebHeaderCollection();
				whc.Add("chardsgfsdfgdfetasdhflaksdhjgf","jhsdfjhsadfhg");
				request.Headers = whc;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();
	
				// PreAuthententicateProperty 1
				Console.Write("PREAUTHENTICATE PROPERTY 1 - not used ... ");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				request.PreAuthenticate = true;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();	

				// PreAuthententicateProperty 2
				Console.Write("PREAUTHENTICATE PROPERTY 2 - not used ...");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				request.PreAuthenticate = true;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();	


				// METHOD PROPERTY
				Console.WriteLine("METHOD PROPERTY - not used");
				// get
				Console.Write(" get ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;
				request.Method = "GET";
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();

				// ProxyProperty 1
				Console.Write("PROXY PROPERTY - ignored 1");
				
				//If this property is not set to an instance of FtpProxy, it is ignored.

				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				//System.Net.IWebProxy proxy = new System.Net.WebProxy
				System.Net.IWebProxy proxy = new System.Net.WebProxy("http://jhshspiupic.uepihhbwyge.hewgx.jhwe");
				request.Proxy = proxy;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();		

				// ProxyProperty 2
				Console.Write("PROXY PROPERTY - ignored 2");
				
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				//System.Net.IWebProxy proxy = new System.Net.WebProxy
				proxy = new System.Net.WebProxy("ftp://jhshspiupic.uepihhbwyge.hewgx.jhwe");
				request.Proxy = proxy;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();		

				// TimeOut Property
				Console.WriteLine("TIMEOUT PROPERTY");
				// 0
				// WebException with the Status property set to WebExceptionStatus.Timeout.
				Console.Write(" =0 ...");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				request.Timeout = 0;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();					


/*				try
				{
					request.Timeout = 0;
					Assertion.Fail("Strange ...");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}
*/

				// <0
				Console.Write(" <0 ...");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				try
				{
					request.Timeout = -10;
					Assertion.Fail("Strange ...");
				}
				catch (ArgumentException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				// >>0
				Console.Write(" >>0 ...");
				request = WebRequest.Create (uri);
				request.Credentials = myCred;
				request.Timeout = 60000;
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();			
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();					

				// Situation >> If it is a FtpProxy instance, tested by setting proxy settings
				// in a Sites class

				// RequestUriProperty
				Console.WriteLine("METHOD PROPERTY");

				// retr
				Console.Write(" retr ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;
				request.Method = "RETR";
				response = request.GetResponse();
				stream = response.GetResponseStream();
				local = new StreamReader(stream);
				textLoc = local.ReadToEnd();
				Assertion.AssertEquals("Requested data differ ...",text,textLoc);
				Console.WriteLine("...ok");	

				response.Close();
				stream.Close();
				local.Close();
				ftp.DeleteFile(uniqName);

				// expected InvalidOperationException
				Console.Write(" Invalid operation ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;				
				try
				{
					request.Method = "HELP";					
					Assertion.Fail("Strange ...");
				}
				catch (ArgumentException e)
				//catch (InvalidOperationException e)
				{
					e=e;Console.WriteLine("...ok");
				}

				// PUT
				text = "012345678901234567890123456789";
				uniqName = Utils.GetUniqueName(ftp, "", "");
				uri = Utils.CreateUri(Site.Host, Site.Port, null,null,Site.UploadDir) + "/" +uniqName;

				Console.Write(" put ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;
				request.Method = "PUT";
				byte[] data = System.Text.Encoding.ASCII.GetBytes(text);
				stream = request.GetRequestStream();
				stream.Write(data,0,30);
				stream.Close();	
				
				System.IO.MemoryStream oOutput = new System.IO.MemoryStream();
				ftp.GetFile(uniqName, oOutput);
				data = oOutput.ToArray();
				string textOut = System.Text.Encoding.ASCII.GetString(data);
				Assertion.AssertEquals("Requested data differ ...",text, textOut);
				Console.WriteLine("...ok");	
				ftp.DeleteFile(uniqName);
				
				// STOR
				text = text + "012345678901234567890123456789";
				uniqName = Utils.GetUniqueName(ftp, "", "");
				uri = Utils.CreateUri(Site.Host, Site.Port, null,null,Site.UploadDir) + "/" +uniqName;

				Console.Write(" stor ...");
				request = WebRequest.Create(uri);
				request.Credentials = myCred;
				request.Method = "STOR";
				data = System.Text.Encoding.ASCII.GetBytes(text);
				stream = request.GetRequestStream();
				stream.Write(data,0,60);
				stream.Close();	
				oOutput.SetLength(0);				
				ftp.GetFile(uniqName, oOutput);
				data = oOutput.ToArray();
				textOut = System.Text.Encoding.ASCII.GetString(data);
				Assertion.AssertEquals("Requested data differ ...",text, textOut);
				Console.WriteLine("...ok");	
				oOutput.Close();
				ftp.DeleteFile(uniqName);
		

				// CONTENTLENGTH - not supported
				Console.Write("CONTENTLENGTH - not supported ..."); 				
				try
				{
					request = WebRequest.Create(uri);
				}
					//TODO:specifikovat chybu lpe
				catch (WebException e)
				{
					e=e;Console.WriteLine("...ok");
				}
			}

			finally
			{
				Utils.CloseFtp (ftp);
			}			
		}



		/// <summary>
		/// Uploads file and gets it from remote server using webrequest, webresponse.
		/// This method simplifyies some tests
		/// </summary>
		/// <param name="text">Text to upload</param>
		/// <param name="ftpTransferType">FtpTransferType</param>
		/// <param name="intSwitch">Switch for some tested situations
		/// 0..normally
		/// 1..URI NOT EXISTS
		/// 2..incorrect username, password
		/// 3..strange port where nobody listen
		/// 4..incorrect URI syntax
		/// 5..long URI
		/// </param>
		/// <returns>string with requested text</returns>
		private string MakeWebRequest(string text, FtpTransferType ftpTransferType, int intSwitch)
		{
			Ftp ftp = null;
			//string helpString = null;
			int helpInt = int.MinValue;
			try
			{
				//PREPARE
				ftp = Utils.ConnectFtp (Site);
				string uniqName = Utils.GetUniqueName(ftp, "", "");			
				NetworkCredential myCred = new NetworkCredential(Site.Login,Site.Password);

				ftp.ChangeDirectory(Site.UploadDir);
				long nUploadedLen = Utils.UploadFile(ftp, uniqName, text, ftpTransferType);
				if (intSwitch == 1) uniqName = Utils.GetUniqueName(ftp, "", "");
				if (intSwitch == 2)
				{
					myCred = new NetworkCredential("incorrectusernameihopeveryoptimisticpointofview","incorrectusernameihopeveryoptimisticpointofview");
				}
				if (intSwitch == 3)
				{
					helpInt=Site.Port;
					Site.Port=int.MaxValue;
				}
				string uri = Utils.CreateUri(Site.Host, Site.Port, null, null,Site.UploadDir) + "/" +uniqName;
				byte[] data = new byte[1025];
				if (intSwitch == 5) uri= uri + System.Text.Encoding.ASCII.GetString(data);
				Console.WriteLine("URI:" + uri);				

				WebRequest.RegisterPrefix ("ftp://", FtpWebRequest.Creator);
				WebRequest request = WebRequest.Create(uri);	
				request.Credentials = myCred;
				
				//TODO: Vytvoit Utils.CreateWebRequest(??), tak aby to bylo efektivn
				// a lo nastavit proxy
				//WebRequest request = Utils.CreateWebRequest(uri)

				WebResponse response = request.GetResponse();
				Stream stream = response.GetResponseStream();
				StreamReader local = new StreamReader(stream);	


				string textLoc=null;
				textLoc = local.ReadToEnd();
				stream.Close();
				local.Close();
				ftp.DeleteFile(uniqName);
				return textLoc;


				//FOR FILE SAVE PURPOSES
				//string tempFilePath = Path.GetTempFileName();
				//Stream local = File.Create (tempFilePath);
				//MemoryStream local = new MemoryStream();
				//delka = Utils.SaveStream(stream, local);

			}

			finally
			{
				//correct values
				if (intSwitch == 3) Site.Port=helpInt;				
				Utils.CloseFtp (ftp);
			}

		}

	}
}