using System;
using System.IO;
using System.Collections;
using System.Threading;
using System.Configuration;
using Rebex.Net;
using NUnit.Framework;

namespace Rebex.Tests.FtpTest
{
	public sealed class Utils
	{
		private Utils() {}

		/// <summary>
		/// Creates ftp object, sets timeout and mode.
		/// </summary>
		/// <param name="site">site config values</param>
		/// <returns>ftp object</returns>
		public static Ftp CreateFtp (Site site)
		{
			ConsoleOpen (Thread.CurrentThread);
			Ftp ftp = new Ftp();
			ftp.Timeout = site.Timeout;
			ftp.Passive = site.Passive;
			if (site.Proxy!=null)
				ftp.Proxy = site.Proxy;

			return ftp;
		}

		/// <summary>
		/// Returns unique name in the current directory.
		/// </summary>
		/// <param name="ftp">ftp object</param>
		/// <param name="prefix">name prefix</param>
		/// <param name="postfix">name postfix</param>
		/// <returns>unique name</returns>
		public static string GetUniqueName (Ftp ftp, string prefix, string postfix)
		{
			Random rnd = new Random();
			
			while (true)
			{
				string name = prefix + (rnd.Next(899999)+100000) + postfix;

				try
				{
					ftp.GetFileDateTime(name);
				}
				catch (FtpException e)
				{
					if (e.Response==null || e.Response.Code!=550)
						throw e;

					string cur = ftp.GetCurrentDirectory();
					try
					{
						ftp.ChangeDirectory(name);
					}
					catch (FtpException ex)
					{
						if (ex.Response.Code==550) return name;
						throw ex;
					}
					finally
					{
						ftp.ChangeDirectory(cur);
					}

				}
			}
		}

		public static long GetFileLength (Ftp ftp, string path)
		{
			return ftp.GetFileLength (path);
		}

		/// <summary>
		/// Uploads string as file via ftp.PutFile
		/// </summary>
		/// <param name="path">Path to </param>
		/// <param name="text">Text to be transferred</param>
		/// <param name="transtype">Type of transfer - binary, ascii</param>
		/// <returns>Legth of uploaded file</returns>
		public static long UploadFile (Ftp ftp, string path, string text, FtpTransferType transtype)
		{
			byte[] data = System.Text.Encoding.ASCII.GetBytes(text);
			System.IO.MemoryStream oInput = new System.IO.MemoryStream(data);
			ftp.SetTransferType (transtype);

			long nUploadedLen = ftp.PutFile(oInput,path); 
			
			// test correct size
			Assertion.Assert(
				"Upload failed. Expected " + data.Length + ", transferred " + nUploadedLen + " bytes", 
				nUploadedLen == data.Length);

			return nUploadedLen;
		}
		/// <summary>
		/// <see cref="CreateFtp"/>,
		/// connect, login
		/// </summary>
		/// <param name="site">site config values</param>
		/// <returns>ftp object</returns>
		public static Ftp ConnectFtp (Site site)
		{
			Ftp ftp = CreateFtp(site);
			ftp.ResponseRead += new FtpResponseReadEventHandler (ResponseRead);
			ftp.CommandSent += new FtpCommandSentEventHandler (CommandSent);
			ftp.StateChanged += new FtpStateChangedEventHandler (StateChanged);
			ftp.TransferProgress += new FtpTransferProgressEventHandler (TransferProgress);

			ConsoleWriteLine ("Connect...");
			ftp.Connect (site.Host,site.Port);
			
			ConsoleWriteLine ("Login...");
			ftp.Login (site.Login,site.Password);
			return ftp;
		}



		/// <summary>
		/// Prepares remote structure on ftp server for list tests.
		/// Connect, login, upload file, create dirs, 
		/// creates manualy FtpList for comparison
		/// </summary>
		/// <param name="ftp">ftp object</param>
		/// <returns>FtpList object for test purposes</returns>		
		public static FtpList PrepareRemoteStructure(Ftp ftp)
		{
			string text = "1234567890";
			FtpList ftpList = new FtpList();

			string uniqName1 = Utils.GetUniqueName(ftp, "_GETLIST", "");			
			ftp.CreateDirectory(uniqName1);
			ftp.ChangeDirectory(uniqName1);			

			// 3 files
			string uniqName = Utils.GetUniqueName(ftp, "", "bim");
			long nUploadedLen = Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
			FtpItem ftpItem = new FtpItem(uniqName,nUploadedLen, FtpItemType.File);
			ftpList.Add(ftpItem);
			
			text = text + text;
			uniqName = Utils.GetUniqueName(ftp, "", "bac");
			nUploadedLen = Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
			ftpItem = new FtpItem(uniqName, nUploadedLen, FtpItemType.File);
			ftpList.Add(ftpItem);
			
			text = text + text;
			uniqName = Utils.GetUniqueName(ftp, "", "bum");
			nUploadedLen = Utils.UploadFile(ftp, uniqName, text, FtpTransferType.Binary);
			ftpItem = new FtpItem(uniqName,nUploadedLen, FtpItemType.File);
			ftpList.Add(ftpItem);

			// 3 dirs
			uniqName = Utils.GetUniqueName(ftp, "DIR", "");
			ftp.CreateDirectory(uniqName);
			ftpItem = new FtpItem(uniqName,int.MaxValue, FtpItemType.Directory);
			ftpList.Add(ftpItem);

			uniqName = Utils.GetUniqueName(ftp, "DIR", "");
			ftp.CreateDirectory(uniqName);
			ftpItem = new FtpItem(uniqName,int.MaxValue, FtpItemType.Directory);
			ftpList.Add(ftpItem);

			uniqName = Utils.GetUniqueName(ftp, "DIR", "");
			ftp.CreateDirectory(uniqName);
			ftpItem = new FtpItem(uniqName,int.MaxValue, FtpItemType.Directory);
			ftpList.Add(ftpItem);
			
			return ftpList;
		}


		/// <summary>
		/// Delete files and dirs on remote server created
		/// via <see cref="CreateRemoteStructure"/>. Needs to be
		/// called 2 times
		/// </summary>
		/// <param name="ftp">ftp object</param>
		/// <param name="ftpList">ftpList object</param>
		/// <returns>ftplList of one level up from deleted dir</returns>				
		public static FtpList DeleteRemoteStructure(Ftp ftp, FtpList ftpList)
		{
			for (int i=0;i<ftpList.Count;i++)
			{
				try
				{					
					if (ftpList[i].IsFile) ftp.DeleteFile(ftpList[i].Name);
					if (ftpList[i].IsDirectory) ftp.RemoveDirectory(ftpList[i].Name);
				}
				catch
				{
					Console.WriteLine("Delete failed [ " + ftpList[i].Name + " ]");
				 }
			}
			ftp.ChangeDirectory("..");
			ftpList = ftp.GetList();
			FtpList ftpList2 = new FtpList();
			//find getlist dirs
			ftpList.Sort();
			bool bul;
			foreach (FtpItem ftpItem in ftpList)				
			{
				try
				{
					bul = (ftpItem.Name.Substring(1,8)!="_GETLIST");
				}
				catch
				{
					bul = false;
				}

				if ( bul && ftpItem.IsDirectory) ftpList2.Add(ftpItem);
			}
			return ftpList2;
				
		}

		/// <summary>
		/// Creates string for ftpWebRequest tests
		/// </summary>
		/// <param name="host">host name</param>
		/// <param name="port">port</param>
		/// <param name="login">login</param>
		/// <param name="password">password</param>
		/// <param name="dir">remote dir</param>
		/// <returns>string for ftpWebRequest</returns>
		public static string  CreateUri (string host, int port, string login, string password, string dir)
		{
			// ex: ftp://anonymous:yesman@maslo.maslo.com:21/incoming/
			string uri=null;
			if (host == null) host="";
			if (login == null) login="";
			if (password == null) password="";
			if (dir == null) dir="";
			uri = "ftp://";
			if (login!="") uri=uri+login;
			if (password!="") uri=uri+":"+password;
			if ((password!="")||(login!="")) uri = uri + "@";
			uri= uri + host;
			if (port > 0) uri = uri + ":"+port.ToString();
			uri = uri + dir;
			return uri;
		}

		/// <summary>
		/// Transfer stream from one to another
		/// </summary>
		/// <param name="streamIn">Input stream</param>
		/// <param name="streamOut">Output stream</param>
		/// <returns>bytes transferred</returns>
		public static int SaveStream (Stream streamIn, Stream streamOut)
		{
			byte[] buffer = new byte[1024];
			int n; int i=0;
			do
			{
				n = streamIn.Read (buffer, 0, buffer.Length);
				streamOut.Write (buffer, 0, n);
				i=+i;
			} while (n > 0);
			return i;
		}

		public static void CloseFtp (Ftp ftp)
		{
			try
			{
				if (ftp != null)
				{
					ConsoleWriteLine ("Disposing...");
					ftp.Dispose ();
					ConsoleWriteLine ("ok");
				}
			}
			finally
			{
				ConsoleClose ();
			}
		}

		/// <summary>
		/// Recounts Input Datetime for list tests
		/// </summary>
		/// <param name="dateTime">Input DateTime</param>
		/// <returns>Output DateTime</returns>
		public static DateTime correctYear(DateTime dateTime)
		{			
			if (dateTime > DateTime.Now)
				dateTime = dateTime.AddYears(-1);
			return(dateTime);
		}

		private static Queue _queue = new Queue();
		private static Thread _thread = null;

		public static void ConsoleOpen (Thread thread)
		{
			lock (_queue.SyncRoot)
			{
				if (_thread != null)
					throw new Exception ("Output already initialized.");

				if (thread == null)
					throw new Exception ("Thread cannot be null reference.");

				_thread = thread;
			}
		}

		public static void ConsoleFlush ()
		{
			lock (_queue.SyncRoot)
			{
				if (_thread == null)
					throw new Exception ("Output not initialized.");

				while (_queue.Count > 0)
				{
					Console.Write (_queue.Dequeue());
				}
			}
		}

		private static void ConsoleWriteInternal (string val)
		{
			lock (_queue.SyncRoot)
			{
				if (_thread == null)
					throw new Exception ("Output not initialized.");

				val = val.Replace ("\n", "\r\n");

				if (Thread.CurrentThread == _thread)
				{
					ConsoleFlush ();
					Console.Write (val.ToString());
				}
				else
				{
					_queue.Enqueue (val.ToString());
				}
			}
		}

		public static void ConsoleWrite (object val)
		{
			ConsoleWriteInternal (val.ToString());
		}

		public static void ConsoleWrite (string format, params object[] args)
		{
			ConsoleWriteInternal (String.Format (format, args));
		}

		public static void ConsoleWriteLine (object val)
		{
			ConsoleWriteInternal (val + "\n");
		}

		public static void ConsoleWriteLine (string format, params object[] args)
		{
			ConsoleWriteInternal (String.Format (format, args) + "\n");
		}

		public static void ConsoleClose ()
		{
			lock (_queue.SyncRoot)
			{
				if (_thread == null)
					throw new Exception ("Output already finished.");

				ConsoleFlush ();

				_thread = null;
			}
		}

		public static void ResponseRead (object sender, FtpResponseReadEventArgs e)
		{
			ConsoleWriteLine ("### " + DateTime.Now.ToLongTimeString() + " <- " + e.Response);
		}

		public static void TransferProgress (object sender, FtpTransferProgressEventArgs e)
		{
			if (e.State!=FtpTransferState.None)
				ConsoleWriteLine ("### " + e.BytesTransfered + " transfered...");
		}

		public static void CommandSent (object sender, FtpCommandSentEventArgs e)
		{
			ConsoleWriteLine ("### " + DateTime.Now.ToLongTimeString() + " -> " + e.Command);
		}

		public static void StateChanged (object sender, FtpStateChangedEventArgs e)
		{
			ConsoleWriteLine ("### " + DateTime.Now.ToLongTimeString() + " State changed to " + e.NewState);
		}


	}
}
