' Directory Listbox User Control
' Written in VB.NET by 
' Mike Stanley
' Senior Applications Architect
' mstanley@apexcgi.com

' Apex Consulting Group, Inc.
' P.O. Box 636
' Wilmington, MA 01887
' Tel: 617.489.9000   Fax: 781.944.1988
' www.apexcgi.com

Imports System.IO
Imports System.Management

Namespace ApexCGI.WinForms.VB.CustomerControl
    'Uses the Treeview control to create a drive and directory listbox control similar to Windows Explorer.
    'Note that the control can be navigated using the keyboard as well as the mouse.
    'Use the up, down, left and right arrow keys.

    Public Class DirectoryListbox
        Inherits System.Windows.Forms.UserControl

        Public Event FolderSelected(ByVal strFolder As String)

        'Indexes used by the Treeview's ImageList control.
        Private Enum ImageIndexEnum
            ClosedFolder = 1
            RemovableDisk = 2
            LocalDisk = 3
            NetworkDrive = 4
            CDROM = 5
            OpenFolder = 6
        End Enum

        'Meaning of the drive type returned by the Management object.
        Private Enum DriveTypeEnum
            RemovableDisk = 2
            LocalDisk = 3
            NetworkDrive = 4
            CDROM = 5
            RAMDisk = 6
        End Enum

#Region " Windows Form Designer generated code "

        Public Sub New()
            MyBase.New()

            'This call is required by the Windows Form Designer.
            InitializeComponent()

            'Add any initialization after the InitializeComponent() call

        End Sub

        'UserControl overrides dispose to clean up the component list.
        Protected Overloads Overrides Sub Dispose(ByVal disposing As Boolean)
            If disposing Then
                If Not (components Is Nothing) Then
                    components.Dispose()
                End If
            End If
            MyBase.Dispose(disposing)
        End Sub

        'Required by the Windows Form Designer
        Private components As System.ComponentModel.IContainer

        'NOTE: The following procedure is required by the Windows Form Designer
        'It can be modified using the Windows Form Designer.  
        'Do not modify it using the code editor.
        Friend WithEvents tvwFolders As System.Windows.Forms.TreeView
        Friend WithEvents ImageList1 As System.Windows.Forms.ImageList
        <System.Diagnostics.DebuggerStepThrough()> Private Sub InitializeComponent()
            Me.components = New System.ComponentModel.Container()
            Dim resources As System.Resources.ResourceManager = New System.Resources.ResourceManager(GetType(DirectoryListbox))
            Me.tvwFolders = New System.Windows.Forms.TreeView()
            Me.ImageList1 = New System.Windows.Forms.ImageList(Me.components)
            Me.SuspendLayout()
            '
            'tvwFolders
            '
            Me.tvwFolders.AllowDrop = True
            Me.tvwFolders.Dock = System.Windows.Forms.DockStyle.Fill
            Me.tvwFolders.ImageList = Me.ImageList1
            Me.tvwFolders.Name = "tvwFolders"
            Me.tvwFolders.Size = New System.Drawing.Size(288, 240)
            Me.tvwFolders.TabIndex = 5
            '
            'ImageList1
            '
            Me.ImageList1.ColorDepth = System.Windows.Forms.ColorDepth.Depth8Bit
            Me.ImageList1.ImageSize = New System.Drawing.Size(16, 16)
            Me.ImageList1.ImageStream = CType(resources.GetObject("ImageList1.ImageStream"), System.Windows.Forms.ImageListStreamer)
            Me.ImageList1.TransparentColor = System.Drawing.Color.Transparent
            '
            'DirectoryListbox
            '
            Me.Controls.AddRange(New System.Windows.Forms.Control() {Me.tvwFolders})
            Me.Name = "DirectoryListbox"
            Me.Size = New System.Drawing.Size(288, 240)
            Me.ResumeLayout(False)

        End Sub
#End Region

        Protected Overrides Sub Finalize()
            MyBase.Finalize()
        End Sub

        Protected Overrides Sub OnResize(ByVal e As System.EventArgs)
            'Not needed - just set the treeview control's Anchor property 
            'to Top, Left and the Dock property to Fill.

            ''''When the control is dropped on a form or when the user resizes it,
            ''''this event is fired.  Make the Treeview control the same size as
            ''''the containing user control.
            '''With tvwFolders
            '''    .Left = 0
            '''    .Top = 0
            '''    .Width = Width
            '''    .Height = Height
            '''End With

        End Sub

        Private Sub tvwFolders_BeforeExpand(ByVal sender As Object, ByVal e As System.Windows.Forms.TreeViewCancelEventArgs) Handles tvwFolders.BeforeExpand
            'The user has expanded a node in the control, 
            'load any subfolders that might exist into the control.
            LoadSubfolders(e.Node)
        End Sub

        Private Sub tvwFolders_AfterSelect(ByVal sender As Object, ByVal e As System.Windows.Forms.TreeViewEventArgs) Handles tvwFolders.AfterSelect
            'The user has selected a node in the control.
            Dim strFolder As String = tvwFolders.SelectedNode.FullPath

            'Raise the FolderSelected event to the calling application, 
            'passing it the full path of the selected folder.


            RaiseEvent FolderSelected(strFolder)
        End Sub

        Private Sub tvwFolders_BeforeCollapse(ByVal sender As Object, ByVal e As System.Windows.Forms.TreeViewCancelEventArgs) Handles tvwFolders.BeforeCollapse
            'The user is collapsing a node.  
            'If the node's icon is an open folder (not a drive icon), change it to a closed folder.

            With e.Node
                If .ImageIndex = ImageIndexEnum.OpenFolder Then
                    .ImageIndex = ImageIndexEnum.ClosedFolder
                    .SelectedImageIndex = ImageIndexEnum.ClosedFolder
                End If
            End With

        End Sub

        Public Sub Open()
            'Overloaded method.
            '#1 - Initial loading of the control when no path is passed as a parameter.
            'Loads the control with all the drives available.  
            'Saves the need for a separate drive list control.
            Dim objDriveNode As TreeNode
            Dim strDrives As String()
            Dim strDrive As String

            With tvwFolders
                .BeginUpdate()  'Switch off screen updating until we are finished.
                .Nodes.Clear()  'Remove anything left over from previous processing.
            End With

            'Get a string array containing the available drive letters.
            strDrives = Directory.GetLogicalDrives()

            'Iterate through the array and add a new node for each drive.
            For Each strDrive In strDrives
                If strDrive.EndsWith("\") Then
                    strDrive = strDrive.TrimEnd("\")
                End If

                objDriveNode = New TreeNode(strDrive)
                'Set the Treeview icon the appropriate image for the drive type.
                With objDriveNode
                    .ImageIndex = GetDriveIcon(strDrive)
                    .SelectedImageIndex = .ImageIndex
                End With
                tvwFolders.Nodes.Add(objDriveNode)
                IdentifyParentFolders(objDriveNode)

                'Select the C: drive by default.
                If strDrive.StartsWith("C") Then
                    tvwFolders.SelectedNode = objDriveNode
                End If

                objDriveNode = Nothing
            Next strDrive

            tvwFolders.EndUpdate()  'Display the contents of the control.

        End Sub

        Public Sub Open(ByVal strPath As String)
            'Overloaded method.
            '#2 - Initial loading of the control when a path is passed as a parameter.
            'Loads the specified path into the control.
            Dim objFolderNode As TreeNode

            'Check that the path is valid.
            If Not Directory.Exists(strPath) Then
                Throw New DirectoryNotFoundException(strPath + " does not exist")
            End If

            With tvwFolders
                .BeginUpdate()  'Switch off screen updating until we are finished.
                .Nodes.Clear()  'Remove anything left over from previous processing.
            End With

            'If the path supplied ends with a "\" remove it.
            'The "\" is not needed and it causes other problems later with "\\" appearing in the path
            'because the Treeview control separator is also a "\".
            If strPath.EndsWith("\") Then
                strPath = strPath.TrimEnd("\")
            End If

            objFolderNode = New TreeNode(strPath)

            'Set the node icon.
            With objFolderNode
                .ImageIndex = ImageIndexEnum.ClosedFolder
                .SelectedImageIndex = .ImageIndex
            End With

            tvwFolders.Nodes.Add(objFolderNode)

            IdentifyParentFolders(objFolderNode)    'Identify whether the path contains subfolders.
            objFolderNode.Expand()                  'Expand the node to show any subfolders.
            tvwFolders.EndUpdate()                  'Display the contents of the control.
            objFolderNode = Nothing

        End Sub

        Private Sub IdentifyParentFolders(ByVal objNode As TreeNode)
            'If the folder referenced by objNode contains subfolders, add a dummy child node to the
            'objNode in order to make the "+" appear at the side of the icon in the Treeview.
            Dim strPath As String
            Dim strFolders As String()

            'Some folders cannot be accessed because of insufficient privileges or if there
            'is no disk loaded in the drive, ignore these errors and continue processing.
            'strFolders will be Nothing when those exceptions are caught.
            Try
                strPath = objNode.FullPath
                If strPath.EndsWith(":") Then
                    strPath &= "\"  'GetDirectories needs the "\" for drive letters.
                End If
                strFolders = Directory.GetDirectories(strPath)
            Catch ex As System.UnauthorizedAccessException  'Ignore
            Catch ex As System.IO.IOException               'Ignore

            End Try

            'AndAlso short-circuits the following test so that if strFolders is Nothing, 
            'strFolders.Length will not be tested, which would cause an error.
            If Not IsNothing(strFolders) AndAlso strFolders.Length > 0 Then
                objNode.Nodes.Add(New TreeNode("."))    'Dummy node
            End If
        End Sub

        Private Function GetDriveIcon(ByVal strPath As String) As ImageIndexEnum
            'Identifies the type of disk and returns the index for the icon in the 
            'ImageList Control that is appropriate for that drive type.

            'Get an object containing information about the drive.
            Dim objDiskDrive As New ManagementObject("win32_logicaldisk.deviceid=""" & strPath & """")
            objDiskDrive.Get()

            'Translate the drive type to an image index.
            Select Case CInt(objDiskDrive("drivetype").ToString())
                Case DriveTypeEnum.RemovableDisk : Return ImageIndexEnum.RemovableDisk
                Case DriveTypeEnum.LocalDisk : Return ImageIndexEnum.LocalDisk
                Case DriveTypeEnum.NetworkDrive : Return ImageIndexEnum.NetworkDrive
                Case DriveTypeEnum.CDROM : Return ImageIndexEnum.CDROM
                Case DriveTypeEnum.RAMDisk : Return ImageIndexEnum.LocalDisk
                Case Else : Return ImageIndexEnum.LocalDisk
            End Select

        End Function

        Private Sub LoadSubfolders(ByVal objParentNode As TreeNode)
            'Gets all the subfolders in the path indicated by objParentNode and loads them into the control. 
            Dim objChildNode As TreeNode
            Dim strPath As String
            Dim strPathSegments() As String
            Dim strFolders As String()
            Dim strFolder As String

            With objParentNode
                'If the current icon is a closed folder and not a drive icon,
                'change it to an open folder.
                If .ImageIndex = ImageIndexEnum.ClosedFolder Then
                    .ImageIndex = ImageIndexEnum.OpenFolder
                    .SelectedImageIndex = .ImageIndex
                End If

                'Remove dummy node.
                .Nodes.Clear()

                'Get a string array containing the full paths of all subfolders.
                strPath = .FullPath
                If strPath.EndsWith(":") Then
                    strPath &= "\"
                End If
                strFolders = Directory.GetDirectories(strPath)

                'Iterate through the array, remove all except the last part of the 
                'paths, i.e., the subfolder names and add them as nodes to the control.
                For Each strFolder In strFolders
                    strPathSegments = strFolder.Split("\")
                    objChildNode = New TreeNode(strPathSegments(strPathSegments.Length - 1))

                    With objChildNode
                        .ImageIndex = ImageIndexEnum.ClosedFolder
                        .SelectedImageIndex = .ImageIndex
                    End With

                    .Nodes.Add(objChildNode)
                    IdentifyParentFolders(objChildNode) 'Identify which subfolders have subfolders of their own.
                    objChildNode = Nothing
                Next strFolder
            End With

        End Sub

        Private Sub tvwFolders_DragDrop(ByVal sender As Object, ByVal e As System.Windows.Forms.DragEventArgs) Handles tvwFolders.DragDrop
            File.Move(e.Data.GetData(DataFormats.FileDrop, True), tvwFolders.SelectedNode.FullPath)


        End Sub

        Private Sub tvwFolders_DragEnter(ByVal sender As Object, ByVal e As System.Windows.Forms.DragEventArgs) Handles tvwFolders.DragEnter
            'check for valid file
            If (e.Data.GetDataPresent(DataFormats.FileDrop)) Then
                e.Effect = DragDropEffects.Copy
            Else
                e.Effect = DragDropEffects.None
            End If
        End Sub
    End Class

End Namespace
