///
///  This source code is freeware and is provided on an "as is" basis without warranties of any kind, 
///  whether express or implied, including without limitation warranties that the code is free of defect, 
///  fit for a particular purpose or non-infringing.  The entire risk as to the quality and performance of 
///  the code is with the end user.
///
///  Written by: Kyong Kwak ( kkwak@purposedriven.com )


using System;
using System.IO;
using System.Text;
using Microsoft.ContentManagement.Common;
using Microsoft.ContentManagement.Publishing;

namespace CmsScriptUtilities {

	/// <summary>
	/// Summary description for TemplateSwitch.
	/// </summary>
	class TemplateSwitch {

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main( string[] args ) {

			string _originalTemplateGuid = "";
			string _switchTemplateGuid = "";		
			string _startingSearchPath = "";

			switch( args.Length ) {
				case 1:
					Console.WriteLine( "Usage: file originalTemplateGuid replacementTemplateGuid startingSearchPath" );
					Console.WriteLine();
					return;
			
				case 3:
					_originalTemplateGuid = args[0];
					_switchTemplateGuid = args[1];
					_startingSearchPath = args[2];
					break;
			
				default:
					Console.WriteLine( "Please enter initial values..." );
					_originalTemplateGuid = GetUserValue( "Original Template Guid: " );
					_switchTemplateGuid = GetUserValue( "Replacement Template Guid: " );			
					_startingSearchPath = GetUserValue( "Enter Starting Search Path: /Channels/" );
					break;
			}

			CmsApplicationContext cmsContext = new CmsApplicationContext();
			// Logon as Update Mode since we need to CREATE postings
			cmsContext.AuthenticateAsCurrentUser( PublishingMode.Update );
			
			// Check that we have a valid channel
			Channel startChannel = ( Channel ) cmsContext.Searches.GetByPath( "/Channels/" + _startingSearchPath );
			if( startChannel == null ) {			
				Console.WriteLine( "Channel could not be found! Terminating..." );
				return;
			}

			// Check that we have the a valid Replacement Template
			TemplateGalleryCollection galleries = cmsContext.Searches.UserTemplateGalleries();				

			//			// Search by Name
			//			Template replacementTemplate = FindTemplateByName( galleries, _switchTemplateName );
			//			if( replacementTemplate == null ) {
			//				Console.WriteLine( "Template of that name could not be found! Terminating..." );
			//				return;
			//			}

			HierarchyItem replacementTemplate = cmsContext.Searches.GetByGuid( _switchTemplateGuid );
			if( !( replacementTemplate is Template ) ) {
				Console.WriteLine( "Template of that Guid could not be found! Terminating..." );
				return;
			}

			// We check each posting for the _originalTemplateName rather than trying to find an instance			
			try {
				ReplaceTemplates( startChannel, ( Template )replacementTemplate, _originalTemplateGuid );
				cmsContext.CommitAll();
			}
			catch( Exception exception ) {
				cmsContext.RollbackAll();
				Console.WriteLine( exception.Message );
				Console.WriteLine();
			}
		}

		/// <summary>
		/// Recursively go through ALL the Templates and find the FIRST instance that match the template name
		/// Using Depth-First Search
		/// </summary>
		/// <param name="galleries"></param>
		/// <param name="Name"></param>
		/// <returns></returns>
		private static Template FindTemplateByName( TemplateGalleryCollection galleries, string Name  ) {
			Template _ReplacementTemplate = null;

			foreach( TemplateGallery gallery in galleries ) {

				_ReplacementTemplate = FindTemplateByName( gallery.TemplateGalleries, Name );
				if( _ReplacementTemplate == null ) {
				
					TemplateCollection templates = gallery.Templates;
					for( int i = 0; i < templates.Count; i++ )
						if( templates[ i ].Name == Name ) {
							_ReplacementTemplate = templates[ i ];
							break;
						}
				}
				else break;
			}

			return _ReplacementTemplate;
		}

		private static void ReplaceTemplates(Channel currentChannel, Template ReplacementTemplate, string OriginalTemplateGuid ) {

			Console.WriteLine( "Entering Channel: {0}", currentChannel.Path );

			foreach( Channel subChannels in currentChannel.Channels )
				ReplaceTemplates( subChannels, ReplacementTemplate, OriginalTemplateGuid );
	
			System.Collections.Hashtable placeholderValuePair = new System.Collections.Hashtable( 20 );

			foreach( Posting originalPosting in currentChannel.Postings ) {
				if ( originalPosting.Template.Guid == OriginalTemplateGuid ) {

					Console.WriteLine( "Found a matching Posting: {0}", originalPosting.Name );
					Console.WriteLine( "                          {0}", originalPosting.Path );
					placeholderValuePair.Clear();

					foreach( Placeholder content in originalPosting.Placeholders )
						placeholderValuePair.Add( content.Name, content.Datasource.RawContent );

					Posting replacementPosting = currentChannel.CreatePosting( ReplacementTemplate );
					replacementPosting.Name = originalPosting.Name;
					replacementPosting.DisplayName = originalPosting.DisplayName;

					// Find any matching placeholders ( by name ) and copy data
					foreach( Placeholder content in replacementPosting.Placeholders ) {
						content.Datasource.RawContent = ( string )placeholderValuePair[ content.Name ];
						if( content.Datasource.RawContent.Length > 0 ) {
							Console.WriteLine( "Found maching placeholder: {0}", content.Name );
							Console.WriteLine( "                   copied: {0} chars", content.Datasource.RawContent.Length );
						}
					}

					Console.WriteLine( "Created a new Posting: {0}", replacementPosting.Name );
					Console.WriteLine( "                       {0}", replacementPosting.Path );

					originalPosting.ExpiryDate = DateTime.UtcNow;
					replacementPosting.Submit();
//					replacementPosting.Approve();
				}
			}
		}

		private static string GetUserValue( string Question ) {
			Console.WriteLine( Question );
			return Console.ReadLine();
		}

	}
}
